# -*- coding: utf-8 -*-
# pylint: disable=line-too-long, unused-wildcard-import
r"""|wland_gd| is still under development and is likely to change in the future.

When applying |wland_wag| on the Kiel Catchment river basins, we realised some
strengths and limitations of the `WALRUS`_ concept.  First, |wland_wag| does not
predict the high dynamics of the groundwater levels encountered at many groundwater
gauges.  This shortcoming is likely with an observed tendency to underestimate
streamflow peaks.  Second, |wland_wag| tends to run dry during summer conditions much
too fast.

.. _`WALRUS`: https://www.wur.nl/en/Research-Results/Chair-groups/Environmental-Sciences/Hydrology-and-Quantitative-Water-Management-Group/Research/WALRUS-1.htm

So far, |wland_gd| offers a physically based approach that increases groundwater
dynamics.  See the documentation on method |Calc_GF_V1| for an in-depth explanation.
The second problem seems to be related to a lack of "groundwater recharge areas" where
groundwater is less exposed to evapotranspiration.  We will address this later.

The following integration tests are identical to the ones of |wland_wag|, except that
we set the additional parameter |ThetaR| to 0.01.  Compare the
:ref:`wland_wag_base_scenario` example of |wland_wag| with the
:ref:`wland_gd_base_scenario` example of |wland_gd| to see how the differences between
both models affect groundwater dynamics.


Integration tests
=================

.. how_to_understand_integration_tests::

>>> from hydpy import IntegrationTest, Element, pub, round_
>>> pub.timegrids = "2017-02-10", "2017-04-10", "1d"
>>> from hydpy.models.wland_gd import *
>>> parameterstep("1d")
>>> land = Element("land", outlets="outlet")
>>> land.model = model
>>> at(10.0)
>>> nu(4)
>>> lt(FIELD, CONIFER, SEALED, WATER)
>>> er(False)
>>> aur(0.6 * 0.98, 0.3 * 0.98, 0.1 * 0.98, 0.02)
>>> gl(1.5)
>>> bl(0.0)
>>> cp(0.8)
>>> lai.sealed = 10.0
>>> lai.conifer = 11.0
>>> lai.field = 0.4, 0.4, 0.3, 0.7, 3.0, 5.2, 4.6, 3.1, 1.3, 0.2, 0.0, 0.0
>>> ih(0.2)
>>> tt(0.0)
>>> ti(4.0)
>>> ddf(5.0)
>>> ddt(0.0)
>>> cw(400.0)
>>> cv(0.2)
>>> cg(200000.0)
>>> cgf(1.0)
>>> cq(0.5)
>>> b(soil=SANDY_LOAM)
>>> psiae(soil=SANDY_LOAM)
>>> thetas(soil=SANDY_LOAM)
>>> thetar(0.01)
>>> zeta1(0.02)
>>> zeta2(400.0)
>>> cwe(nan)
>>> cge(nan)
>>> ac(nan)
>>> sh(1.0)
>>> st(1.0)
>>> rg(False)
>>> with model.add_petmodel_v1("evap_pet_mlc"):
...     landmonthfactor.sealed = 0.7
...     landmonthfactor.conifer = 1.3
...     landmonthfactor.field[1:4] = 0.73, 0.77, 0.95
...     landmonthfactor.water[1:4] = 1.22, 1.26, 1.28
...     dampingfactor(1.0)
...     with model.add_retmodel_v1("evap_ret_io"):
...         evapotranspirationfactor(0.9)
>>> with model.add_dischargemodel_v2("wq_walrus"):
...     crestheight(0.0)
...     bankfulldischarge(8.0)
...     dischargeexponent(1.5)
>>> test = IntegrationTest(land)
>>> test.inits = (
...     (states.ic, (-3.0, -3.0, -3.0, 0.0)),
...     (states.sp, (-3.0, -3.0, -3.0, 0.0)),
...     (states.dve, 140.0),
...     (states.dv, 140.0),
...     (states.hge, 900.0),
...     (states.dg, 1600.0),
...     (states.hq, 0.0),
...     (states.hs, -2.0),
...     (model.petmodel.sequences.logs.loggedpotentialevapotranspiration, 0.0),
... )
>>> inputs.t.series = (
...     -2.8, -1.5, -0.9, -1.6, -1.3, 1.7, 4.4, 4.5, 3.4, 4.8, 6.7, 5.8, 6.5, 5.0, 3.0,
...     3.1, 7.1, 9.4, 4.6, 3.7, 4.7, 5.9, 7.7, 6.3, 3.7, 1.6, 4.0, 5.6, 5.8, 5.7, 4.6,
...     4.2, 7.4, 6.3, 8.7, 6.4, 5.2, 5.1, 8.7, 6.2, 5.9, 5.2, 5.2, 5.9, 6.7, 7.0, 8.3,
...     9.0, 12.4, 15.0, 11.8, 9.4, 8.1, 7.9, 7.5, 7.2, 8.1, 8.6, 10.5)
>>> inputs.p.series = (
...     0.0, 0.4, 0.0, 0.0, 0.0, 0.0, 0.2, 4.5, 0.0, 3.2, 4.6, 2.3, 18.0, 19.2, 0.4,
...     8.3, 5.3, 0.7, 2.7, 1.6, 2.5, 0.6, 0.2, 1.7, 0.3, 0.0, 1.8, 8.9, 0.0, 0.0,
...     0.0, 0.9, 0.1, 0.0, 0.0, 3.9, 8.7, 26.4, 11.5, 0.9, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 1.5, 0.3, 0.2, 4.5, 0.0, 0.0, 0.0, 0.4, 0.0, 0.0, 0.0, 0.0)
>>> model.petmodel.retmodel.sequences.inputs.referenceevapotranspiration.series = (
...     0.6, 0.8, 0.7, 0.4, 0.4, 0.4, 0.4, 0.3, 0.3, 0.4, 0.3, 0.6, 0.8, 0.5, 0.8,
...     0.5, 0.4, 1.3, 0.9, 0.7, 0.7, 1.1, 1.0, 0.8, 0.6, 0.7, 0.7, 0.5, 0.8, 1.0,
...     1.2, 0.9, 0.9, 1.2, 1.4, 1.1, 1.1, 0.5, 0.6, 1.5, 2.0, 1.6, 1.6, 1.2, 1.3,
...     1.6, 1.9, 0.8, 1.5, 2.7, 1.5, 1.6, 2.0, 2.1, 1.7, 1.7, 0.8, 1.3, 2.5)
>>> inputs.fxg.series = 0.0
>>> inputs.fxs.series = 0.0
>>> test.reset_inits()
>>> conditions = model.conditions

.. _wland_gd_base_scenario:

base scenario
_____________

.. integration-test::

    >>> test("wland_gd_base_scenario",
    ...      axis1=(fluxes.pc, fluxes.fqs, fluxes.fgs, fluxes.rh),
    ...      axis2=(states.dg, states.hs))
    |                date |    t |    p | fxg | fxs | dhs |    pc |                           pe |                       pet |                                   tf |                                ei |                                   rf |                 sf |                                   pm |                                am |    ps | pve |       pv |        pq | etve |      etv |       es |       et |  gr | fxs | fxg |         cdg | fgse |       fgs |       fqs |       rh |        r |                                   ic |                                   sp | dve |         dv | hge |          dg |       hq |         hs |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 00:00:00 | -2.8 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.3942  0.702  0.378  0.6588 | 0.3942  0.702  0.378  0.0 |       0.0        0.0        0.0  0.0 |      0.0  0.000001       0.0  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  0.000117   0.000117   0.000117  0.0 |      0.0       0.0       0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 |  0.49406 | 0.000067 | 0.435763 | 0.0 | 0.0 | 0.0 |  -37.374521 |  0.0 | -0.000081 |       0.0 |      0.0 |      0.0 |      -3.0  -3.000001       -3.0  0.0 |      -3.0       -3.0       -3.0  0.0 | nan | 140.493979 | nan | 1562.625479 |      0.0 |   -2.00364 |      0.0 |
    | 2017-02-11 00:00:00 | -1.5 |  0.4 | 0.0 | 0.0 | 0.0 |  0.32 | 0.5256  0.936  0.504  0.8784 | 0.5256  0.936  0.504  0.0 |  0.000001        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  0.009388   0.009388   0.009388  0.0 |      0.0       0.0       0.0  0.0 |  0.32 | 0.0 |      0.0 |       0.0 |  0.0 | 0.658704 | 0.000199 | 0.580983 | 0.0 | 0.0 | 0.0 |  -15.520342 |  0.0 | -0.000055 |       0.0 |      0.0 |      0.0 | -2.680002  -2.680003  -2.680002  0.0 |      -3.0       -3.0       -3.0  0.0 | nan | 141.152629 | nan | 1547.105137 |      0.0 |  -1.686272 |      0.0 |
    | 2017-02-12 00:00:00 | -0.9 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.4599  0.819  0.441  0.7686 | 0.4599  0.819  0.441  0.0 |       0.0        0.0        0.0  0.0 | 0.000002  0.000004  0.000002  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  0.069591   0.069591   0.069591  0.0 |      0.0       0.0       0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.576325 |  0.00033 | 0.508328 | 0.0 | 0.0 | 0.0 |   -5.171805 |  0.0 | -0.000045 |       0.0 |      0.0 |      0.0 | -2.680004  -2.680007  -2.680004  0.0 |      -3.0       -3.0       -3.0  0.0 | nan | 141.728908 | nan | 1541.933333 |      0.0 |  -1.688597 |      0.0 |
    | 2017-02-13 00:00:00 | -1.6 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  0.006707   0.006707   0.006707  0.0 |      0.0       0.0       0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.329312 | 0.000186 | 0.290458 | 0.0 | 0.0 | 0.0 |   -0.904798 |  0.0 | -0.000043 |       0.0 |      0.0 |      0.0 | -2.680005  -2.680009  -2.680005  0.0 |      -3.0       -3.0       -3.0  0.0 | nan | 142.058178 | nan | 1541.028534 |      0.0 |  -1.690658 |      0.0 |
    | 2017-02-14 00:00:00 | -1.3 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  0.018374   0.018374   0.018374  0.0 |      0.0       0.0       0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.329299 | 0.000185 | 0.290447 | 0.0 | 0.0 | 0.0 |    0.730086 |  0.0 | -0.000043 |       0.0 |      0.0 |      0.0 | -2.680007  -2.680011  -2.680006  0.0 |      -3.0       -3.0       -3.0  0.0 | nan | 142.387434 | nan | 1541.758621 |      0.0 |  -1.692718 |      0.0 |
    | 2017-02-15 00:00:00 |  1.7 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |   8.50479    8.50479    8.50479  0.0 | 0.000009  0.000009  0.000009  0.0 |   0.0 | 0.0 | 0.000002 |  0.000007 |  0.0 | 0.329287 | 0.000183 | 0.290436 | 0.0 | 0.0 | 0.0 |      1.5035 |  0.0 | -0.000044 |  0.000003 |      0.0 |      0.0 | -2.680008  -2.680013  -2.680007  0.0 | -3.000008  -3.000009  -3.000009  0.0 | nan | 142.716675 | nan |  1543.26212 | 0.000003 |   -1.69466 |      0.0 |
    | 2017-02-16 00:00:00 |  4.4 |  0.2 | 0.0 | 0.0 | 0.0 |  0.16 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |  0.000001        0.0        0.0  0.0 | 0.000002  0.000003  0.000002  0.0 |  0.000001        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 | 22.000001  22.000001  22.000001  0.0 | 0.000023  0.000023  0.000023  0.0 |  0.16 | 0.0 | 0.000007 |  0.000017 |  0.0 | 0.329274 | 0.000268 | 0.290427 | 0.0 | 0.0 | 0.0 |    1.869514 |  0.0 | -0.000046 |  0.000013 |      0.0 |      0.0 |  -2.52001  -2.520016  -2.520009  0.0 | -3.000031  -3.000032  -3.000032  0.0 | nan | 143.045896 | nan | 1545.131634 | 0.000007 |  -1.536276 |      0.0 |
    | 2017-02-17 00:00:00 |  4.5 |  4.5 | 0.0 | 0.0 | 0.0 |   3.6 | 0.1971  0.351  0.189  0.3294 | 0.1971  0.351  0.189  0.0 |  0.716494   0.000865   0.002553  0.0 | 0.045813  0.103383  0.056188  0.0 |  0.716494   0.000865   0.002553  0.0 | 0.0  0.0  0.0  0.0 |      22.5       22.5       22.5  0.0 | 0.000023  0.000023  0.000023  0.0 |   3.6 | 0.0 | 0.135792 |  0.308222 |  0.0 | 0.182326 | 0.188386 | 0.227419 | 0.0 | 0.0 | 0.0 |  -10.351857 |  0.0 | -0.000253 |   0.08012 | 0.000248 | 0.000029 |  0.317682   0.975737    1.02125  0.0 | -3.000054  -3.000055  -3.000055  0.0 | nan | 143.092178 | nan | 1534.779777 | 0.228109 |   5.777652 | 0.000029 |
    | 2017-02-18 00:00:00 |  3.4 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.1971  0.351  0.189  0.3294 | 0.1971  0.351  0.189  0.0 |       0.0        0.0        0.0  0.0 | 0.147906  0.341646  0.186301  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 | 17.000016  17.000016  17.000016  0.0 | 0.000018  0.000018  0.000018  0.0 |   0.0 | 0.0 | 0.000005 |  0.000013 |  0.0 | 0.035705 |   0.3294 | 0.243749 | 0.0 | 0.0 | 0.0 |    6.850761 |  0.0 | -0.003016 |  0.198426 | 0.005644 | 0.000653 |  0.169777    0.63409    0.83495  0.0 | -3.000072  -3.000072  -3.000072  0.0 | nan | 143.124861 | nan | 1541.630538 | 0.029696 |  14.755903 | 0.000653 |
    | 2017-02-19 00:00:00 |  4.8 |  3.2 | 0.0 | 0.0 | 0.0 |  2.56 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |  2.023752   0.453078   0.899772  0.0 | 0.221728  0.465424  0.251476  0.0 |  2.023752   0.453078   0.899772  0.0 | 0.0  0.0  0.0  0.0 |      24.0       24.0       24.0  0.0 | 0.000025  0.000025  0.000025  0.0 |  2.56 | 0.0 | 0.425022 |  1.057656 |  0.0 | 0.028076 |   0.4392 | 0.325402 | 0.0 | 0.0 | 0.0 |   -28.76742 |  0.0 | -0.006778 |  0.568775 | 0.018355 | 0.002124 |  0.484297   2.275589   2.243702  0.0 | -3.000097  -3.000097  -3.000097  0.0 | nan | 142.721137 | nan | 1512.863118 | 0.518578 |  43.530004 | 0.002124 |
    | 2017-02-20 00:00:00 |  6.7 |  4.6 | 0.0 | 0.0 | 0.0 |  3.68 | 0.1971  0.351  0.189  0.3294 | 0.1971  0.351  0.189  0.0 |  3.340021   2.972444   3.208119  0.0 | 0.184109  0.350996  0.188997  0.0 |  3.340021   2.972444   3.208119  0.0 | 0.0  0.0  0.0  0.0 |      33.5       33.5       33.5  0.0 | 0.000035  0.000035  0.000035  0.0 |  3.68 | 0.0 | 0.903984 |  2.403007 |  0.0 | 0.008612 |   0.3294 | 0.244155 | 0.0 | 0.0 | 0.0 |  -51.065213 |  0.0 | -0.028923 |  1.792105 | 0.105384 | 0.012197 |  0.640166   2.632149   2.526586  0.0 | -3.000131  -3.000131  -3.000131  0.0 | nan | 141.796841 | nan | 1461.797905 |  1.12948 | 128.148992 | 0.012197 |
    | 2017-02-21 00:00:00 |  5.8 |  2.3 | 0.0 | 0.0 | 0.0 |  1.84 | 0.3942  0.702  0.378  0.6588 | 0.3942  0.702  0.378  0.0 |  1.642137   1.409144    1.61889  0.0 | 0.363807  0.701991  0.377995  0.0 |  1.642137   1.409144    1.61889  0.0 | 0.0  0.0  0.0  0.0 |      29.0       29.0       29.0  0.0 |  0.00003   0.00003   0.00003  0.0 |  1.84 | 0.0 |  0.43559 |  1.177914 |  0.0 | 0.020151 |   0.6588 | 0.488296 | 0.0 | 0.0 | 0.0 |    1.256242 |  0.0 | -0.106227 |  1.656185 | 0.291426 |  0.03373 |  0.474222   2.361014   2.369701  0.0 | -3.000161  -3.000161  -3.000161  0.0 | nan | 141.275176 | nan | 1463.054146 | 0.651209 | 191.227344 |  0.03373 |
    | 2017-02-22 00:00:00 |  6.5 | 18.0 | 0.0 | 0.0 | 0.0 |  14.4 | 0.5256  0.936  0.504  0.8784 | 0.5256  0.936  0.504  0.0 | 13.589441  13.048599  13.564535  0.0 | 0.504697  0.935996  0.503997  0.0 | 13.589441  13.048599  13.564535  0.0 | 0.0  0.0  0.0  0.0 |      32.5       32.5       32.5  0.0 | 0.000033  0.000033  0.000033  0.0 |  14.4 | 0.0 | 3.631273 | 10.156586 |  0.0 | 0.013861 |   0.8784 | 0.651129 | 0.0 | 0.0 | 0.0 | -194.354232 |  0.0 |  -0.22784 |   6.26682 | 0.746521 | 0.086403 |  0.780084   2.776418   2.701169  0.0 | -3.000194  -3.000195  -3.000195  0.0 | nan | 137.429923 | nan | 1268.699915 | 4.540974 | 464.449329 | 0.086403 |
    | 2017-02-23 00:00:00 |  5.0 | 19.2 | 0.0 | 0.0 | 0.0 | 15.36 | 0.3285  0.585  0.315   0.549 | 0.3285  0.585  0.315  0.0 | 14.927286  14.656218  14.931712  0.0 | 0.322035  0.584999  0.314999  0.0 | 14.927286  14.656218  14.931712  0.0 | 0.0  0.0  0.0  0.0 |      25.0       25.0       25.0  0.0 | 0.000026  0.000026  0.000026  0.0 | 15.36 | 0.0 | 3.793893 |  11.43193 |  0.0 | 0.004289 |    0.549 | 0.406979 | 0.0 | 0.0 | 0.0 |  -51.728904 |  0.0 | -1.302452 | 10.388816 | 2.284046 | 0.264357 |  0.890763   2.895201   2.814459  0.0 |  -3.00022  -3.000221  -3.000221  0.0 | nan | 132.337866 | nan | 1216.971011 | 5.584088 | 816.671885 | 0.264357 |
    | 2017-02-24 00:00:00 |  3.0 |  0.4 | 0.0 | 0.0 | 0.0 |  0.32 | 0.5256  0.936  0.504  0.8784 | 0.5256  0.936  0.504  0.0 |  0.294506   0.228504   0.294804  0.0 | 0.495651  0.935979  0.503995  0.0 |  0.294506   0.228504   0.294804  0.0 | 0.0  0.0  0.0  0.0 |  15.00006   15.00006   15.00006  0.0 | 0.000015  0.000015  0.000015  0.0 |  0.32 | 0.0 |  0.06614 |  0.215225 |  0.0 | 0.019881 |   0.8784 | 0.651116 | 0.0 | 0.0 | 0.0 |    16.46191 |  0.0 | -2.377207 |  4.955501 | 3.330921 | 0.385523 |  0.420605   2.050718   2.335659  0.0 | -3.000236  -3.000236  -3.000236  0.0 | nan |   129.9144 | nan | 1233.432921 | 0.843812 | 787.552125 | 0.385523 |
    | 2017-02-25 00:00:00 |  3.1 |  8.3 | 0.0 | 0.0 | 0.0 |  6.64 | 0.3285  0.585  0.315   0.549 | 0.3285  0.585  0.315  0.0 |   6.06746   5.421574   6.057767  0.0 | 0.308293  0.584994  0.314997  0.0 |   6.06746   5.421574   6.057767  0.0 | 0.0  0.0  0.0  0.0 | 15.500043  15.500043  15.500043  0.0 | 0.000016  0.000016  0.000016  0.0 |  6.64 | 0.0 | 1.364169 |  4.644989 |  0.0 | 0.013414 |    0.549 | 0.406946 | 0.0 | 0.0 | 0.0 |  -30.068967 |  0.0 | -1.741572 |  3.313177 | 2.825479 | 0.327023 |  0.684852    2.68415   2.602895  0.0 | -3.000251  -3.000252  -3.000252  0.0 | nan | 126.822074 | nan | 1203.363953 | 2.175624 | 737.911537 | 0.327023 |
    | 2017-02-26 00:00:00 |  7.1 |  5.3 | 0.0 | 0.0 | 0.0 |  4.24 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |  3.990587   3.797307   3.989263  0.0 | 0.251899  0.467998  0.251998  0.0 |  3.990587   3.797307   3.989263  0.0 | 0.0  0.0  0.0  0.0 |      35.5       35.5       35.5  0.0 | 0.000037  0.000037  0.000037  0.0 |  4.24 | 0.0 | 0.879972 |  3.140532 |  0.0 | 0.007238 |   0.4392 | 0.325572 | 0.0 | 0.0 | 0.0 |   -9.178465 |  0.0 | -1.569803 |  3.661652 | 2.715389 | 0.314281 |  0.682366   2.658845   2.601634  0.0 | -3.000288  -3.000288  -3.000288  0.0 | nan | 124.379537 | nan | 1194.185488 | 1.654504 |  716.13555 | 0.314281 |
    | 2017-02-27 00:00:00 |  9.4 |  0.7 | 0.0 | 0.0 | 0.0 |  0.56 | 0.8541  1.521  0.819  1.4274 | 0.8541  1.521  0.819  0.0 |  0.424048   0.206921   0.404336  0.0 | 0.695516   1.52066  0.818961  0.0 |  0.424048   0.206921   0.404336  0.0 | 0.0  0.0  0.0  0.0 |      47.0       47.0       47.0  0.0 | 0.000048  0.000048  0.000048  0.0 |  0.56 | 0.0 | 0.076779 |  0.287886 |  0.0 | 0.105418 |   1.4274 | 1.057822 | 0.0 | 0.0 | 0.0 |    20.21848 |  0.0 | -1.271379 |  1.611656 | 2.406929 |  0.27858 |  0.122801   1.491264   1.938337  0.0 | -3.000336  -3.000337  -3.000337  0.0 | nan | 123.136797 | nan | 1214.403968 | 0.330734 | 617.825045 |  0.27858 |
    | 2017-02-28 00:00:00 |  4.6 |  2.7 | 0.0 | 0.0 | 0.0 |  2.16 | 0.5913  1.053  0.567  0.9882 | 0.5913  1.053  0.567  0.0 |  1.504342   0.475474   1.340734  0.0 |  0.45371  1.052728  0.566963  0.0 |  1.504342   0.475474   1.340734  0.0 | 0.0  0.0  0.0  0.0 |      23.0       23.0       23.0  0.0 | 0.000024  0.000024  0.000024  0.0 |  2.16 | 0.0 | 0.248993 |  0.955251 |  0.0 |  0.09146 |   0.9882 | 0.732278 | 0.0 | 0.0 | 0.0 |   21.088443 |  0.0 | -0.836008 |  0.799122 | 1.867699 | 0.216169 |  0.324749   2.123061    2.19064  0.0 |  -3.00036  -3.000361  -3.000361  0.0 | nan | 122.143256 | nan | 1235.492411 | 0.486863 | 527.900925 | 0.216169 |
    | 2017-03-01 00:00:00 |  3.7 |  1.6 | 0.0 | 0.0 | 0.0 |  1.28 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |  0.950591   0.497004   0.875872  0.0 | 0.384057  0.818947  0.440979  0.0 |  0.950591   0.497004   0.875872  0.0 | 0.0  0.0  0.0  0.0 | 18.500006  18.500006  18.500006  0.0 | 0.000019  0.000019  0.000019  0.0 |  1.28 | 0.0 | 0.169296 |  0.654695 |  0.0 | 0.067123 |   0.7938 |  0.58489 | 0.0 | 0.0 | 0.0 |   29.527873 |  0.0 | -0.610859 |  0.795382 |  1.51942 | 0.175859 |  0.270102    2.08711   2.153789  0.0 | -3.000379   -3.00038   -3.00038  0.0 | nan | 121.430223 | nan | 1265.020284 | 0.346176 | 464.450935 | 0.175859 |
    | 2017-03-02 00:00:00 |  4.7 |  2.5 | 0.0 | 0.0 | 0.0 |   2.0 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |  1.532173   1.008537   1.460236  0.0 | 0.393795  0.818966  0.440983  0.0 |  1.532173   1.008537   1.460236  0.0 | 0.0  0.0  0.0  0.0 |      23.5       23.5       23.5  0.0 | 0.000024  0.000024  0.000024  0.0 |   2.0 | 0.0 | 0.284446 |  1.111912 |  0.0 | 0.060652 |   0.7938 | 0.584915 | 0.0 | 0.0 | 0.0 |   13.288412 |  0.0 | -0.475414 |  0.920397 | 1.281994 | 0.148379 |  0.344134   2.259606   2.252569  0.0 | -3.000403  -3.000404  -3.000404  0.0 | nan | 120.731015 | nan | 1278.308696 | 0.537691 |  425.69109 | 0.148379 |
    | 2017-03-03 00:00:00 |  5.9 |  0.6 | 0.0 | 0.0 | 0.0 |  0.48 | 0.7623  1.287  0.693  1.2474 | 0.7623  1.287  0.693  0.0 |  0.288966    0.08905   0.242285  0.0 | 0.505891  1.286378  0.692918  0.0 |  0.288966    0.08905   0.242285  0.0 | 0.0  0.0  0.0  0.0 |      29.5       29.5       29.5  0.0 |  0.00003   0.00003   0.00003  0.0 |  0.48 | 0.0 | 0.046247 |  0.182731 |  0.0 | 0.170509 |   1.2474 | 0.918902 | 0.0 | 0.0 | 0.0 |   27.773459 |  0.0 | -0.399918 |  0.578739 | 1.121237 | 0.129773 |  0.029276   1.364178   1.797366  0.0 | -3.000434  -3.000434  -3.000434  0.0 | nan | 120.455359 | nan | 1306.082155 | 0.141683 | 379.583656 | 0.129773 |
    | 2017-03-04 00:00:00 |  7.7 |  0.2 | 0.0 | 0.0 | 0.0 |  0.16 |  0.693   1.17   0.63   1.134 |  0.693   1.17   0.63  0.0 |  0.056032    0.00073   0.020468  0.0 |  0.28711  1.128812  0.629377  0.0 |  0.056032    0.00073   0.020468  0.0 | 0.0  0.0  0.0  0.0 |      38.5       38.5       38.5  0.0 |  0.00004   0.00004   0.00004  0.0 |  0.16 | 0.0 | 0.007808 |  0.028897 |  0.0 | 0.283266 |    1.134 | 0.834891 | 0.0 | 0.0 | 0.0 |   23.029314 |  0.0 | -0.301842 |  0.140114 | 0.912753 | 0.105643 | -0.153865   0.394636   1.307521  0.0 | -3.000473  -3.000474  -3.000474  0.0 | nan | 120.428975 | nan | 1329.111469 | 0.030467 | 326.526337 | 0.105643 |
    | 2017-03-05 00:00:00 |  6.3 |  1.7 | 0.0 | 0.0 | 0.0 |  1.36 | 0.5544  0.936  0.504  0.9072 | 0.5544  0.936  0.504  0.0 |  0.699353   0.001276   0.261852  0.0 | 0.321216  0.881369   0.50364  0.0 |  0.699353   0.001276   0.261852  0.0 | 0.0  0.0  0.0  0.0 |      31.5       31.5       31.5  0.0 | 0.000032  0.000032  0.000032  0.0 |  1.36 | 0.0 | 0.096729 |  0.359156 |  0.0 | 0.173021 |   0.9072 | 0.668103 | 0.0 | 0.0 | 0.0 |   10.406701 |  0.0 | -0.217535 |  0.212163 | 0.739748 | 0.085619 |  0.185566   0.871991   1.902029  0.0 | -3.000506  -3.000506  -3.000506  0.0 | nan | 120.287732 | nan | 1339.518171 |  0.17746 | 290.794428 | 0.085619 |
    | 2017-03-06 00:00:00 |  3.7 |  0.3 | 0.0 | 0.0 | 0.0 |  0.24 | 0.4158  0.702  0.378  0.6804 | 0.4158  0.702  0.378  0.0 |   0.13171   0.000231   0.068138  0.0 | 0.255779  0.664282  0.377897  0.0 |   0.13171   0.000231   0.068138  0.0 | 0.0  0.0  0.0  0.0 | 18.500006  18.500006  18.500006  0.0 | 0.000019  0.000019  0.000019  0.0 |  0.24 | 0.0 | 0.018186 |  0.069561 |  0.0 | 0.118812 |   0.6804 | 0.501131 | 0.0 | 0.0 | 0.0 |   13.902661 |  0.0 | -0.170176 |  0.194343 | 0.632223 | 0.073174 |  0.038077   0.447478   1.695993  0.0 | -3.000525  -3.000525  -3.000525  0.0 | nan | 120.218183 | nan | 1353.420832 | 0.052678 | 260.760967 | 0.073174 |
    | 2017-03-07 00:00:00 |  1.6 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |       0.0        0.0        0.0  0.0 | 0.203146  0.529544  0.440411  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  8.006707   8.006707   8.006707  0.0 | 0.000008  0.000008  0.000008  0.0 |   0.0 | 0.0 | 0.000002 |  0.000007 |  0.0 | 0.283401 |   0.7938 | 0.584131 | 0.0 | 0.0 | 0.0 |   11.248788 |  0.0 | -0.128403 |   0.04549 | 0.529029 |  0.06123 | -0.165069  -0.082066   1.255582  0.0 | -3.000533  -3.000533  -3.000533  0.0 | nan | 120.373179 | nan | 1364.669619 | 0.007194 | 230.082206 |  0.06123 |
    | 2017-03-08 00:00:00 |  4.0 |  1.8 | 0.0 | 0.0 | 0.0 |  1.44 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |  0.767777   0.000464   0.302021  0.0 | 0.289622  0.646586  0.440662  0.0 |  0.767777   0.000464   0.302021  0.0 | 0.0  0.0  0.0  0.0 | 20.000002  20.000002  20.000002  0.0 | 0.000021  0.000021  0.000021  0.0 |  1.44 | 0.0 | 0.106171 |  0.395474 |  0.0 | 0.187092 |   0.7938 | 0.584471 | 0.0 | 0.0 | 0.0 |    0.342097 |  0.0 | -0.094444 |  0.209072 | 0.450423 | 0.052132 |  0.217532   0.710883   1.952899  0.0 | -3.000553  -3.000554  -3.000554  0.0 | nan | 120.359656 | nan | 1365.011716 | 0.193596 | 214.286778 | 0.052132 |
    | 2017-03-09 00:00:00 |  5.6 |  8.9 | 0.0 | 0.0 | 0.0 |  7.12 | 0.3465  0.585  0.315   0.567 | 0.3465  0.585  0.315  0.0 |  6.357929   4.564264   6.159258  0.0 | 0.317365  0.584263  0.314994  0.0 |  6.357929   4.564264   6.159258  0.0 | 0.0  0.0  0.0  0.0 |      28.0       28.0       28.0  0.0 | 0.000029  0.000029  0.000029  0.0 |  7.12 | 0.0 | 1.182315 |  4.735907 |  0.0 | 0.019596 |    0.567 | 0.417878 | 0.0 | 0.0 | 0.0 |  -82.380264 |  0.0 | -0.099183 |  2.708258 | 0.555389 | 0.064281 |  0.662237   2.682356   2.598647  0.0 | -3.000582  -3.000583  -3.000583  0.0 | nan | 119.097753 | nan | 1282.631451 | 2.221245 |  321.40101 | 0.064281 |
    | 2017-03-10 00:00:00 |  5.8 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.5544  0.936  0.504  0.9072 | 0.5544  0.936  0.504  0.0 |       0.0        0.0        0.0  0.0 | 0.473399   0.93593  0.503987  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      29.0       29.0       29.0  0.0 |  0.00003   0.00003   0.00003  0.0 |   0.0 | 0.0 | 0.000006 |  0.000024 |  0.0 | 0.053829 |   0.9072 | 0.668534 | 0.0 | 0.0 | 0.0 |   31.103481 |  0.0 | -0.276773 |  1.920655 | 0.919616 | 0.106437 |  0.188839   1.746427    2.09466  0.0 | -3.000612  -3.000612  -3.000612  0.0 | nan | 118.874804 | nan | 1313.734933 | 0.300615 | 356.419422 | 0.106437 |
    | 2017-03-11 00:00:00 |  5.7 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |  0.693   1.17   0.63   1.134 |  0.693   1.17   0.63  0.0 |       0.0        0.0        0.0  0.0 | 0.342819  1.155656  0.629755  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      28.5       28.5       28.5  0.0 | 0.000029  0.000029  0.000029  0.0 |   0.0 | 0.0 | 0.000006 |  0.000024 |  0.0 | 0.237378 |    1.134 | 0.835104 | 0.0 | 0.0 | 0.0 |     16.4612 |  0.0 |  -0.26671 |  0.259929 | 0.849063 | 0.098271 |  -0.15398   0.590771   1.464905  0.0 | -3.000641  -3.000642  -3.000642  0.0 | nan | 118.845466 | nan | 1330.196133 |  0.04071 | 313.806845 | 0.098271 |
    | 2017-03-12 00:00:00 |  4.6 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.8316  1.404  0.756  1.3608 | 0.8316  1.404  0.756  0.0 |       0.0        0.0        0.0  0.0 | 0.194384  0.813489  0.748184  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      23.0       23.0       23.0  0.0 | 0.000024  0.000024  0.000024  0.0 |   0.0 | 0.0 | 0.000005 |  0.000019 |  0.0 | 0.619401 |   1.3608 | 1.000313 | 0.0 | 0.0 | 0.0 |    14.35899 |  0.0 | -0.189543 |  0.035263 | 0.687603 | 0.079584 | -0.348364  -0.222718   0.716721  0.0 | -3.000665  -3.000665  -3.000665  0.0 | nan | 119.275319 | nan | 1344.555124 | 0.005466 | 271.434962 | 0.079584 |
    | 2017-03-13 00:00:00 |  4.2 |  0.9 | 0.0 | 0.0 | 0.0 |  0.72 | 0.6237  1.053  0.567  1.0206 | 0.6237  1.053  0.567  0.0 |  0.198314   0.000023   0.002957  0.0 | 0.206879  0.452527  0.552684  0.0 |  0.198314   0.000023   0.002957  0.0 | 0.0  0.0  0.0  0.0 | 21.000001  21.000001  21.000001  0.0 | 0.000022  0.000022  0.000022  0.0 |  0.72 | 0.0 | 0.027036 |  0.094981 |  0.0 | 0.476296 |   1.0206 | 0.749357 | 0.0 | 0.0 | 0.0 |   10.433303 |  0.0 | -0.133716 |  0.052483 | 0.559937 | 0.064807 | -0.033557   0.044731    0.88108  0.0 | -3.000687  -3.000687  -3.000687  0.0 | nan | 119.590864 | nan | 1354.988427 | 0.047964 | 239.812327 | 0.064807 |
    | 2017-03-14 00:00:00 |  7.4 |  0.1 | 0.0 | 0.0 | 0.0 |  0.08 | 0.6237  1.053  0.567  1.0206 | 0.6237  1.053  0.567  0.0 |  0.023913   0.000002   0.000192  0.0 | 0.223902  0.377698  0.535186  0.0 |  0.023913   0.000002   0.000192  0.0 | 0.0  0.0  0.0  0.0 |      37.0       37.0       37.0  0.0 | 0.000038  0.000038  0.000038  0.0 |  0.08 | 0.0 | 0.003281 |  0.011453 |  0.0 |  0.48983 |   1.0206 | 0.747587 | 0.0 | 0.0 | 0.0 |    10.54758 |  0.0 | -0.098574 |  0.048575 | 0.469444 | 0.054334 | -0.201372  -0.252968   0.425702  0.0 | -3.000725  -3.000725  -3.000725  0.0 | nan | 119.978839 | nan | 1365.536007 | 0.010842 |  213.43259 | 0.054334 |
    | 2017-03-15 00:00:00 |  6.3 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.8316  1.404  0.756  1.3608 | 0.8316  1.404  0.756  0.0 |       0.0        0.0        0.0  0.0 | 0.171771  0.216652  0.491877  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      31.5       31.5       31.5  0.0 | 0.000032  0.000032  0.000032  0.0 |   0.0 | 0.0 | 0.000007 |  0.000026 |  0.0 | 0.832568 |   1.3608 | 0.974442 | 0.0 | 0.0 | 0.0 |     9.39997 |  0.0 | -0.072142 |   0.00917 | 0.393353 | 0.045527 | -0.373143   -0.46962  -0.066175  0.0 | -3.000757  -3.000757  -3.000757  0.0 | nan | 120.739258 | nan | 1374.935977 | 0.001698 | 189.671997 | 0.045527 |
    | 2017-03-16 00:00:00 |  8.7 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.9702  1.638  0.882  1.5876 | 0.9702  1.638  0.882  0.0 |       0.0        0.0        0.0  0.0 | 0.116297  0.128142  0.249606  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      43.5       43.5       43.5  0.0 | 0.000045  0.000045  0.000045  0.0 |   0.0 | 0.0 | 0.000009 |  0.000036 |  0.0 | 1.068504 |   1.5876 |  1.10469 | 0.0 | 0.0 | 0.0 |    9.198082 |  0.0 | -0.052736 |  0.001156 | 0.330516 | 0.038254 | -0.489441  -0.597762   -0.31578  0.0 | -3.000802  -3.000802  -3.000802  0.0 | nan | 121.755016 | nan | 1384.134058 | 0.000578 | 169.289574 | 0.038254 |
    | 2017-03-17 00:00:00 |  6.4 |  3.9 | 0.0 | 0.0 | 0.0 |  3.12 | 0.7623  1.287  0.693  1.2474 | 0.7623  1.287  0.693  0.0 |  1.818463   0.016206   0.250957  0.0 | 0.484994  0.989435  0.602563  0.0 |  1.818463   0.016206   0.250957  0.0 | 0.0  0.0  0.0  0.0 |      32.0       32.0       32.0  0.0 | 0.000033  0.000033  0.000033  0.0 |  3.12 | 0.0 |   0.2581 |  0.888778 |  0.0 | 0.282973 |   1.2474 | 0.909652 | 0.0 | 0.0 | 0.0 |  -12.428574 |  0.0 | -0.040442 |  0.436312 | 0.301193 |  0.03486 |  0.327102   1.516597     1.9507  0.0 | -3.000834  -3.000835  -3.000835  0.0 | nan | 121.739448 | nan | 1371.705484 | 0.453044 | 175.698347 |  0.03486 |
    | 2017-03-18 00:00:00 |  5.2 |  8.7 | 0.0 | 0.0 | 0.0 |  6.96 | 0.7623  1.287  0.693  1.2474 | 0.7623  1.287  0.693  0.0 |  6.071563   4.670998   5.753257  0.0 | 0.685968  1.286933  0.692984  0.0 |  6.071563   4.670998   5.753257  0.0 | 0.0  0.0  0.0  0.0 |      26.0       26.0       26.0  0.0 | 0.000027  0.000027  0.000027  0.0 |  6.96 | 0.0 | 1.175769 |  4.561398 |  0.0 | 0.050718 |   1.2474 | 0.919301 | 0.0 | 0.0 | 0.0 |  -76.877683 |  0.0 |  -0.06674 |  2.896153 | 0.467416 | 0.054099 |   0.52957   2.518667   2.464459  0.0 | -3.000861  -3.000862  -3.000862  0.0 | nan | 120.547657 | nan | 1294.827801 | 2.118288 | 297.008426 | 0.054099 |
    | 2017-03-19 00:00:00 |  5.1 | 26.4 | 0.0 | 0.0 | 0.0 | 21.12 | 0.3465  0.585  0.315   0.567 | 0.3465  0.585  0.315  0.0 | 20.402784  20.094611  20.410298  0.0 | 0.337475  0.584999  0.314999  0.0 | 20.402784  20.094611  20.410298  0.0 | 0.0  0.0  0.0  0.0 |      25.5       25.5       25.5  0.0 | 0.000026  0.000026  0.000026  0.0 | 21.12 | 0.0 | 4.077907 | 16.640994 |  0.0 | 0.005996 |    0.567 | 0.417923 | 0.0 | 0.0 | 0.0 | -185.215164 |  0.0 | -0.500625 | 11.201779 | 1.596338 | 0.184761 |  0.909311   2.959057   2.859162  0.0 | -3.000887  -3.000888  -3.000888  0.0 | nan | 115.975121 | nan | 1109.612637 | 7.557504 |  764.55411 | 0.184761 |
    | 2017-03-20 00:00:00 |  8.7 | 11.5 | 0.0 | 0.0 | 0.0 |   9.2 | 0.4158  0.702  0.378  0.6804 | 0.4158  0.702  0.378  0.0 |  8.931935   8.697486   8.951417  0.0 | 0.406537  0.701998  0.377999  0.0 |  8.931935   8.697486   8.951417  0.0 | 0.0  0.0  0.0  0.0 |      43.5       43.5       43.5  0.0 | 0.000045  0.000045  0.000045  0.0 |   9.2 | 0.0 | 1.660678 |  7.368982 |  0.0 | 0.006156 |   0.6804 | 0.501513 | 0.0 | 0.0 | 0.0 |  -25.509302 |  0.0 | -2.410977 | 10.723056 | 3.843916 | 0.444898 |   0.77084   2.759573   2.729747  0.0 | -3.000932  -3.000932  -3.000932  0.0 | nan | 111.909622 | nan | 1084.103335 | 4.203431 | 999.983517 | 0.444898 |
    | 2017-03-21 00:00:00 |  6.2 |  0.9 | 0.0 | 0.0 | 0.0 |  0.72 | 1.0395  1.755  0.945   1.701 | 1.0395  1.755  0.945  0.0 |  0.554126   0.276001   0.548122  0.0 | 0.842875  1.754558  0.944963  0.0 |  0.554126   0.276001   0.548122  0.0 | 0.0  0.0  0.0  0.0 |      31.0       31.0       31.0  0.0 | 0.000032  0.000032  0.000032  0.0 |  0.72 | 0.0 | 0.081904 |  0.396406 |  0.0 | 0.130834 |    1.701 | 1.253473 | 0.0 | 0.0 | 0.0 |    4.323781 |  0.0 | -2.598518 |  3.885184 | 4.067252 | 0.470747 |  0.093839   1.449014   1.956662  0.0 | -3.000964  -3.000964  -3.000964  0.0 | nan | 109.360034 | nan | 1088.427116 | 0.714654 | 871.419237 | 0.470747 |
    | 2017-03-22 00:00:00 |  5.9 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |  1.386   2.34   1.26   2.268 |  1.386   2.34   1.26  0.0 |       0.0        0.0        0.0  0.0 | 0.443211   1.68771  1.251505  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      29.5       29.5       29.5  0.0 |  0.00003   0.00003   0.00003  0.0 |   0.0 | 0.0 | 0.000005 |  0.000026 |  0.0 | 0.843462 |    2.268 | 1.668736 | 0.0 | 0.0 | 0.0 |    17.32388 |  0.0 | -1.448082 |  0.617804 | 2.979654 | 0.344867 | -0.349371  -0.238696   0.705157  0.0 | -3.000994  -3.000994  -3.000994  0.0 | nan | 108.755409 | nan | 1105.750996 | 0.096875 | 686.580539 | 0.344867 |
    | 2017-03-23 00:00:00 |  5.2 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.1088  1.872  1.008  1.8144 | 1.1088  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.139085  0.270011  0.749585  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      26.0       26.0       26.0  0.0 | 0.000027  0.000027  0.000027  0.0 |   0.0 | 0.0 | 0.000005 |  0.000023 |  0.0 | 1.176986 |   1.8144 | 1.309014 | 0.0 | 0.0 | 0.0 |   49.332488 |  0.0 | -0.754993 |  0.083772 | 2.101834 | 0.243268 | -0.488456  -0.508708  -0.044428  0.0 | -3.001021  -3.001021  -3.001021  0.0 | nan | 109.177397 | nan | 1155.083484 | 0.013125 | 550.484087 | 0.243268 |
    | 2017-03-24 00:00:00 |  5.2 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.1088  1.872  1.008  1.8144 | 1.1088  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.087831  0.124898  0.285202  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      26.0       26.0       26.0  0.0 | 0.000027  0.000027  0.000027  0.0 |   0.0 | 0.0 | 0.000005 |  0.000022 |  0.0 | 1.259234 |   1.8144 | 1.263247 | 0.0 | 0.0 | 0.0 |   49.279072 |  0.0 |  -0.44947 |  0.011353 | 1.536679 | 0.177856 | -0.576287  -0.633605  -0.329631  0.0 | -3.001047  -3.001048  -3.001048  0.0 | nan | 109.987156 | nan | 1204.362556 | 0.001795 | 452.570417 | 0.177856 |
    | 2017-03-25 00:00:00 |  5.9 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.8316  1.404  0.756  1.3608 | 0.8316  1.404  0.756  0.0 |       0.0        0.0        0.0  0.0 | 0.049333  0.062911  0.109412  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      29.5       29.5       29.5  0.0 |  0.00003   0.00003   0.00003  0.0 |   0.0 | 0.0 | 0.000005 |  0.000025 |  0.0 | 0.965599 |   1.3608 |   0.9371 | 0.0 | 0.0 | 0.0 |   38.991611 |  0.0 | -0.290317 |  0.001575 | 1.164158 | 0.134741 | -0.625621  -0.696516  -0.439043  0.0 | -3.001078  -3.001078  -3.001078  0.0 | nan | 110.662433 | nan | 1243.354167 | 0.000245 | 380.275908 | 0.134741 |
    | 2017-03-26 00:00:00 |  6.7 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.9009  1.521  0.819  1.4742 | 0.9009  1.521  0.819  0.0 |       0.0        0.0        0.0  0.0 | 0.043679  0.052843  0.081011  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      33.5       33.5       33.5  0.0 | 0.000034  0.000034  0.000034  0.0 |   0.0 | 0.0 | 0.000006 |  0.000029 |  0.0 | 1.057594 |   1.4742 |  1.01144 | 0.0 | 0.0 | 0.0 |   29.280516 |  0.0 | -0.194008 |  0.000223 | 0.907802 |  0.10507 | -0.669299  -0.749359  -0.520054  0.0 | -3.001112  -3.001112  -3.001112  0.0 | nan | 111.526012 | nan | 1272.634683 |  0.00005 | 324.866774 |  0.10507 |
    | 2017-03-27 00:00:00 |  7.0 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.1088  1.872  1.008  1.8144 | 1.1088  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 |  0.04431  0.051555  0.072303  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      35.0       35.0       35.0  0.0 | 0.000036  0.000036  0.000036  0.0 |   0.0 | 0.0 | 0.000007 |   0.00003 |  0.0 |  1.31233 |   1.8144 |  1.24206 | 0.0 | 0.0 | 0.0 |   22.837244 |  0.0 | -0.131204 |   0.00006 | 0.723672 | 0.083758 | -0.713609  -0.800914  -0.592357  0.0 | -3.001148  -3.001148  -3.001148  0.0 | nan | 112.707133 | nan | 1295.471927 |  0.00002 | 281.085639 | 0.083758 |
    | 2017-03-28 00:00:00 |  8.3 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.3167  2.223  1.197  2.1546 | 1.3167  2.223  1.197  0.0 |       0.0        0.0        0.0  0.0 |  0.04334  0.048893  0.064061  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      41.5       41.5       41.5  0.0 | 0.000043  0.000043  0.000043  0.0 |   0.0 | 0.0 | 0.000008 |  0.000035 |  0.0 | 1.568522 |   2.1546 | 1.472664 | 0.0 | 0.0 | 0.0 |   19.142318 |  0.0 | -0.089588 |  0.000038 | 0.586819 | 0.067919 | -0.756949  -0.849808  -0.656417  0.0 |  -3.00119  -3.001191  -3.001191  0.0 | nan | 114.186058 | nan | 1314.614245 | 0.000018 | 245.641069 | 0.067919 |
    | 2017-03-29 00:00:00 |  9.0 |  1.5 | 0.0 | 0.0 | 0.0 |   1.2 | 0.5544  0.936  0.504  0.9072 | 0.5544  0.936  0.504  0.0 |  0.261855   0.000022   0.000144  0.0 | 0.145067  0.238297  0.204217  0.0 |  0.261855   0.000022   0.000144  0.0 | 0.0  0.0  0.0  0.0 |      45.0       45.0       45.0  0.0 | 0.000046  0.000046  0.000046  0.0 |   1.2 | 0.0 | 0.032993 |  0.127487 |  0.0 | 0.503791 |   0.9072 | 0.637859 | 0.0 | 0.0 | 0.0 |   11.981734 |  0.0 | -0.062933 |  0.052181 | 0.488196 | 0.056504 |  0.036129   0.111873   0.339221  0.0 | -3.001236  -3.001237  -3.001237  0.0 | nan | 114.593923 | nan | 1326.595979 | 0.075324 | 221.305643 | 0.056504 |
    | 2017-03-30 00:00:00 | 12.4 |  0.3 | 0.0 | 0.0 | 0.0 |  0.24 | 1.0395  1.755  0.945   1.701 | 1.0395  1.755  0.945  0.0 |  0.076999   0.000006    0.00005  0.0 | 0.397315  0.624883  0.602032  0.0 |  0.076999   0.000006    0.00005  0.0 | 0.0  0.0  0.0  0.0 |      62.0       62.0       62.0  0.0 | 0.000064  0.000064  0.000064  0.0 |  0.24 | 0.0 | 0.009771 |  0.037475 |  0.0 | 0.802147 |    1.701 |  1.21785 | 0.0 | 0.0 | 0.0 |     11.0004 |  0.0 | -0.046009 |   0.08814 | 0.422898 | 0.048947 | -0.198185  -0.273016  -0.022861  0.0 |   -3.0013    -3.0013    -3.0013  0.0 | nan |  115.34029 | nan | 1337.596379 | 0.024659 |  200.98962 | 0.048947 |
    | 2017-03-31 00:00:00 | 15.0 |  0.2 | 0.0 | 0.0 | 0.0 |  0.16 | 1.8711  3.159  1.701  3.0618 | 1.8711  3.159  1.701  0.0 |  0.023623   0.000001   0.000007  0.0 | 0.346813  0.402176  0.468844  0.0 |  0.023623   0.000001   0.000007  0.0 | 0.0  0.0  0.0  0.0 |      75.0       75.0       75.0  0.0 | 0.000077  0.000077  0.000077  0.0 |  0.16 | 0.0 |  0.00307 |  0.011489 |  0.0 | 1.928512 |   3.0618 | 2.130296 | 0.0 | 0.0 | 0.0 |   11.865255 |  0.0 | -0.032068 |  0.029403 | 0.361638 | 0.041856 |  -0.40862  -0.515194  -0.331711  0.0 | -3.001377  -3.001377  -3.001377  0.0 | nan | 117.233664 | nan | 1349.461634 | 0.006745 | 180.032437 | 0.041856 |
    | 2017-04-01 00:00:00 | 11.8 |  4.5 | 0.0 | 0.0 | 0.0 |   3.6 | 1.2825  1.755  0.945   1.728 | 1.2825  1.755  0.945  0.0 |    1.9917   0.031322   0.415839  0.0 | 0.874859   1.42793  0.831929  0.0 |    1.9917   0.031322   0.415839  0.0 | 0.0  0.0  0.0  0.0 |      59.0       59.0       59.0  0.0 |  0.00006   0.00006   0.00006  0.0 |   3.6 | 0.0 | 0.264795 |  1.007745 |  0.0 | 0.379453 |    1.728 | 1.384995 | 0.0 | 0.0 | 0.0 |   -9.848999 |  0.0 |  -0.02363 |  0.504805 |  0.33295 | 0.038536 |  0.324821   1.625554    2.02052  0.0 | -3.001437  -3.001438  -3.001438  0.0 | nan | 117.324692 | nan | 1339.612635 | 0.509685 | 188.950355 | 0.038536 |
    | 2017-04-02 00:00:00 |  9.4 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |  1.368  1.872  1.008  1.8432 |  1.368  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.610003  1.639835  1.005964  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      47.0       47.0       47.0  0.0 | 0.000048  0.000048  0.000048  0.0 |   0.0 | 0.0 |  0.00001 |   0.00004 |  0.0 | 0.580673 |   1.8432 | 1.488396 | 0.0 | 0.0 | 0.0 |   14.746854 |  0.0 | -0.038374 |  0.440784 |  0.36696 | 0.042472 | -0.285183  -0.014281   1.014556  0.0 | -3.001485  -3.001486  -3.001486  0.0 | nan | 117.866982 | nan | 1354.359489 | 0.068941 | 188.665269 | 0.042472 |
    | 2017-04-03 00:00:00 |  8.1 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |   1.71   2.34   1.26   2.304 |   1.71   2.34   1.26  0.0 |       0.0        0.0        0.0  0.0 | 0.226994  0.478502  1.030203  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      40.5       40.5       40.5  0.0 | 0.000041  0.000041  0.000041  0.0 |   0.0 | 0.0 | 0.000008 |  0.000034 |  0.0 | 1.603398 |    2.304 |  1.83539 | 0.0 | 0.0 | 0.0 |   11.605867 |  0.0 | -0.036226 |  0.059677 | 0.332852 | 0.038524 | -0.512177  -0.492783  -0.015647  0.0 | -3.001527  -3.001527  -3.001527  0.0 | nan | 119.434146 | nan | 1365.965356 | 0.009298 | 171.045254 | 0.038524 |
    | 2017-04-04 00:00:00 |  7.9 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.7955  2.457  1.323  2.4192 | 1.7955  2.457  1.323  0.0 |       0.0        0.0        0.0  0.0 | 0.119524  0.161068  0.355469  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      39.5       39.5       39.5  0.0 |  0.00004   0.00004   0.00004  0.0 |   0.0 | 0.0 | 0.000008 |  0.000033 |  0.0 | 1.875642 |   2.4192 |  1.85517 | 0.0 | 0.0 | 0.0 |   12.526533 |  0.0 | -0.027827 |  0.008696 | 0.284399 | 0.032917 | -0.631701  -0.653851  -0.371116  0.0 | -3.001567  -3.001568  -3.001568  0.0 | nan | 121.281952 | nan | 1378.491889 | 0.000635 | 153.605043 | 0.032917 |
    | 2017-04-05 00:00:00 |  7.5 |  0.4 | 0.0 | 0.0 | 0.0 |  0.32 | 1.4535  1.989  1.071  1.9584 | 1.4535  1.989  1.071  0.0 |  0.014432   0.000001   0.000008  0.0 | 0.119617  0.145282  0.209544  0.0 |  0.014432   0.000001   0.000008  0.0 | 0.0  0.0  0.0  0.0 |      37.5       37.5       37.5  0.0 | 0.000038  0.000038  0.000038  0.0 |  0.32 | 0.0 | 0.002057 |  0.006848 |  0.0 |  1.49806 |   1.9584 |  1.49404 | 0.0 | 0.0 | 0.0 |   12.228507 |  0.0 | -0.022529 |  0.004273 | 0.243287 | 0.028158 | -0.445751  -0.479133  -0.260668  0.0 | -3.001606  -3.001606  -3.001606  0.0 | nan | 122.755427 | nan | 1390.720396 |  0.00321 | 139.018122 | 0.028158 |
    | 2017-04-06 00:00:00 |  7.2 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.4535  1.989  1.071  1.9584 | 1.4535  1.989  1.071  0.0 |       0.0        0.0        0.0  0.0 | 0.126278  0.143979  0.175695  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      36.0       36.0       36.0  0.0 | 0.000037  0.000037  0.000037  0.0 |   0.0 | 0.0 | 0.000008 |   0.00003 |  0.0 | 1.493898 |   1.9584 | 1.490585 | 0.0 | 0.0 | 0.0 |   11.557873 |  0.0 | -0.019109 |  0.003012 | 0.209661 | 0.024266 | -0.572028  -0.623112  -0.436363  0.0 | -3.001642  -3.001643  -3.001643  0.0 | nan | 124.230208 | nan | 1402.278269 | 0.000228 | 125.881588 | 0.024266 |
    | 2017-04-07 00:00:00 |  8.1 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |  0.684  0.936  0.504  0.9216 |  0.684  0.936  0.504  0.0 |       0.0        0.0        0.0  0.0 | 0.042012  0.045674  0.053504  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      40.5       40.5       40.5  0.0 | 0.000041  0.000041  0.000041  0.0 |   0.0 | 0.0 | 0.000009 |  0.000033 |  0.0 | 0.721842 |   0.9216 | 0.698471 | 0.0 | 0.0 | 0.0 |    9.523129 |  0.0 | -0.016681 |  0.000212 | 0.181972 | 0.021062 |  -0.61404  -0.668787  -0.489867  0.0 | -3.001684  -3.001684  -3.001684  0.0 | nan | 124.935359 | nan | 1411.801398 | 0.000049 | 115.136115 | 0.021062 |
    | 2017-04-08 00:00:00 |  8.6 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.1115  1.521  0.819  1.4976 | 1.1115  1.521  0.819  0.0 |       0.0        0.0        0.0  0.0 | 0.055222  0.058913  0.067492  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      43.0       43.0       43.0  0.0 | 0.000044  0.000044  0.000044  0.0 |   0.0 | 0.0 |  0.00001 |  0.000035 |  0.0 | 1.186647 |   1.4976 |  1.13298 | 0.0 | 0.0 | 0.0 |    8.230114 |  0.0 | -0.014257 |  0.000056 | 0.158917 | 0.018393 | -0.669262  -0.727699  -0.557359  0.0 | -3.001728  -3.001728  -3.001728  0.0 | nan |  126.10774 | nan | 1420.031512 | 0.000028 | 105.066701 | 0.018393 |
    | 2017-04-09 00:00:00 | 10.5 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 2.1375  2.925  1.575    2.88 | 2.1375  2.925  1.575  0.0 |       0.0        0.0        0.0  0.0 | 0.078869  0.082516  0.092006  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      52.5       52.5       52.5  0.0 | 0.000054  0.000054  0.000054  0.0 |   0.0 | 0.0 | 0.000012 |  0.000043 |  0.0 | 2.310038 |     2.88 | 2.174705 | 0.0 | 0.0 | 0.0 |   10.517272 |  0.0 | -0.012372 |  0.000048 | 0.137396 | 0.015902 | -0.748131  -0.810216  -0.649366  0.0 | -3.001781  -3.001782  -3.001782  0.0 | nan | 128.405394 | nan | 1430.548784 | 0.000022 |  94.773673 | 0.015902 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _wland_gd_seepage:

seepage
_______

>>> inputs.fxg.series = 10.0

.. integration-test::

    >>> test("wland_gd_seepage",
    ...      axis1=(fluxes.pc, fluxes.fqs, fluxes.fgs, fluxes.rh),
    ...      axis2=(states.dg, states.hs))
    |                date |    t |    p |  fxg | fxs | dhs |    pc |                           pe |                       pet |                                   tf |                                ei |                                   rf |                 sf |                                   pm |                                am |    ps | pve |       pv |        pq | etve |      etv |       es |       et |  gr | fxs |       fxg |         cdg | fgse |       fgs |       fqs |       rh |        r |                                   ic |                                   sp | dve |          dv | hge |          dg |       hq |          hs |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 00:00:00 | -2.8 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 | 0.3942  0.702  0.378  0.6588 | 0.3942  0.702  0.378  0.0 |       0.0        0.0        0.0  0.0 |      0.0  0.000001       0.0  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  0.000117   0.000117   0.000117  0.0 |      0.0       0.0       0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.494343 | 0.399555 | 0.444002 | 0.0 | 0.0 | 11.337868 | -441.312411 |  0.0 |  0.327192 |       0.0 | 0.001248 | 0.000144 |      -3.0  -3.000001       -3.0  0.0 |      -3.0       -3.0       -3.0  0.0 | nan |  129.483667 | nan | 1158.687589 |      0.0 |   11.967233 | 0.000144 |
    | 2017-02-11 00:00:00 | -1.5 |  0.4 | 10.0 | 0.0 | 0.0 |  0.32 | 0.5256  0.936  0.504  0.8784 | 0.5256  0.936  0.504  0.0 |  0.000001        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  0.009388   0.009388   0.009388  0.0 |      0.0       0.0       0.0  0.0 |  0.32 | 0.0 |      0.0 |       0.0 |  0.0 | 0.659727 |   0.8784 | 0.599448 | 0.0 | 0.0 | 11.337868 |    -75.6168 |  0.0 |  0.677048 |       0.0 | 0.018395 | 0.002129 | -2.680002  -2.680003  -2.680002  0.0 |      -3.0       -3.0       -3.0  0.0 | nan |  119.482573 | nan | 1083.070789 |      0.0 |   40.346927 | 0.002129 |
    | 2017-02-12 00:00:00 | -0.9 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 | 0.4599  0.819  0.441  0.7686 | 0.4599  0.819  0.441  0.0 |       0.0        0.0        0.0  0.0 | 0.000002  0.000004  0.000002  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  0.069591   0.069591   0.069591  0.0 |      0.0       0.0       0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.577679 |   0.7686 | 0.524888 | 0.0 | 0.0 | 11.337868 |  -69.597592 |  0.0 |  0.891566 |       0.0 | 0.060825 |  0.00704 | -2.680004  -2.680007  -2.680004  0.0 |      -3.0       -3.0       -3.0  0.0 | nan |  109.613951 | nan | 1013.473197 |      0.0 |   75.855143 |  0.00704 |
    | 2017-02-13 00:00:00 | -1.6 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  0.006707   0.006707   0.006707  0.0 |      0.0       0.0       0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.330299 |   0.4392 | 0.300109 | 0.0 | 0.0 | 11.337868 |  -70.558174 |  0.0 |  1.110199 |       0.0 | 0.131045 | 0.015167 | -2.680005  -2.680009  -2.680005  0.0 |      -3.0       -3.0       -3.0  0.0 | nan |    99.71658 | nan |  942.915023 |      0.0 |  117.823463 | 0.015167 |
    | 2017-02-14 00:00:00 | -1.3 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  0.018374   0.018374   0.018374  0.0 |      0.0       0.0       0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.330458 |   0.4392 |  0.30025 | 0.0 | 0.0 | 11.337868 |  -71.337567 |  0.0 |  1.339527 |       0.0 | 0.231318 | 0.026773 | -2.680007  -2.680011  -2.680006  0.0 |      -3.0       -3.0       -3.0  0.0 | nan |   90.048697 | nan |  871.577456 |      0.0 |  164.891537 | 0.026773 |
    | 2017-02-15 00:00:00 |  1.7 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |   8.50479    8.50479    8.50479  0.0 | 0.000009  0.000009  0.000009  0.0 |   0.0 | 0.0 | 0.000001 |  0.000008 |  0.0 | 0.330587 |   0.4392 | 0.300363 | 0.0 | 0.0 | 11.337868 |  -71.733292 |  0.0 |   1.57543 |  0.000004 | 0.361726 | 0.041866 | -2.680008  -2.680013  -2.680007  0.0 | -3.000008  -3.000009  -3.000009  0.0 | nan |   80.616844 | nan |  799.844164 | 0.000003 |  215.842703 | 0.041866 |
    | 2017-02-16 00:00:00 |  4.4 |  0.2 | 10.0 | 0.0 | 0.0 |  0.16 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |  0.000001        0.0        0.0  0.0 | 0.000002  0.000003  0.000002  0.0 |  0.000001        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 | 22.000001  22.000001  22.000001  0.0 | 0.000023  0.000023  0.000023  0.0 |  0.16 | 0.0 | 0.000002 |  0.000021 |  0.0 |  0.33069 |   0.4392 | 0.300455 | 0.0 | 0.0 | 11.337868 |  -72.274143 |  0.0 |   1.81783 |  0.000015 |  0.52107 | 0.060309 |  -2.52001  -2.520016  -2.520009  0.0 | -3.000031  -3.000032  -3.000032  0.0 | nan |   71.427494 | nan |  727.570021 |  0.00001 |  269.677035 | 0.060309 |
    | 2017-02-17 00:00:00 |  4.5 |  4.5 | 10.0 | 0.0 | 0.0 |   3.6 | 0.1971  0.351  0.189  0.3294 | 0.1971  0.351  0.189  0.0 |  0.715966   0.000866   0.002554  0.0 | 0.045788  0.103395  0.056195  0.0 |  0.715966   0.000866   0.002554  0.0 | 0.0  0.0  0.0  0.0 |      22.5       22.5       22.5  0.0 | 0.000023  0.000023  0.000023  0.0 |   3.6 | 0.0 | 0.029375 |   0.40368 |  0.0 |  0.18317 |   0.3294 | 0.230973 | 0.0 | 0.0 | 11.337868 |  -74.370198 |  0.0 |  2.062232 |  0.105045 | 0.715068 | 0.082762 |  0.318236   0.975723   1.021242  0.0 | -3.000054  -3.000055  -3.000055  0.0 | nan |   62.305652 | nan |  653.199823 | 0.298644 |  333.285905 | 0.082762 |
    | 2017-02-18 00:00:00 |  3.4 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 | 0.1971  0.351  0.189  0.3294 | 0.1971  0.351  0.189  0.0 |       0.0        0.0        0.0  0.0 |  0.14799  0.341643    0.1863  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 | 17.000016  17.000016  17.000016  0.0 | 0.000018  0.000018  0.000018  0.0 |   0.0 | 0.0 | 0.000001 |  0.000017 |  0.0 | 0.035821 |   0.3294 | 0.243901 | 0.0 | 0.0 | 11.337868 |  -76.641165 |  0.0 |    2.2887 |  0.258175 | 0.971888 | 0.112487 |  0.170246   0.634081   0.834941  0.0 | -3.000072  -3.000072  -3.000072  0.0 | nan |   53.292304 | nan |  576.558658 | 0.040486 |  397.944389 | 0.112487 |
    | 2017-02-19 00:00:00 |  4.8 |  3.2 | 10.0 | 0.0 | 0.0 |  2.56 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |  2.024047   0.452944   0.899585  0.0 | 0.221752   0.46542  0.251473  0.0 |  2.024047   0.452944   0.899585  0.0 | 0.0  0.0  0.0  0.0 |      24.0       24.0       24.0  0.0 | 0.000025  0.000025  0.000025  0.0 |  2.56 | 0.0 | 0.053722 |  1.391945 |  0.0 | 0.028199 |   0.4392 | 0.325524 | 0.0 | 0.0 | 11.337868 |  -80.483137 |  0.0 |  2.531996 |  0.752748 | 1.262122 | 0.146079 |  0.484447   2.275716   2.243883  0.0 | -3.000097  -3.000097  -3.000097  0.0 | nan |    44.46091 | nan |  496.075521 | 0.679684 |  485.504717 | 0.146079 |
    | 2017-02-20 00:00:00 |  6.7 |  4.6 | 10.0 | 0.0 | 0.0 |  3.68 | 0.1971  0.351  0.189  0.3294 | 0.1971  0.351  0.189  0.0 |  3.340109   2.972326    3.20822  0.0 | 0.184113  0.350996  0.188997  0.0 |  3.340109   2.972326    3.20822  0.0 | 0.0  0.0  0.0  0.0 |      33.5       33.5       33.5  0.0 | 0.000035  0.000035  0.000035  0.0 |  3.68 | 0.0 | 0.078869 |  3.145638 |  0.0 | 0.008653 |   0.3294 | 0.244193 | 0.0 | 0.0 | 11.337868 |  -87.800968 |  0.0 |  2.582112 |  2.345023 | 1.796926 | 0.207978 |  0.640224   2.632393   2.526666  0.0 | -3.000131  -3.000131  -3.000131  0.0 | nan |   35.634937 | nan |  408.274552 | 1.480298 |  627.786295 | 0.207978 |
    | 2017-02-21 00:00:00 |  5.8 |  2.3 | 10.0 | 0.0 | 0.0 |  1.84 | 0.3942  0.702  0.378  0.6588 | 0.3942  0.702  0.378  0.0 |  1.642161   1.408849   1.618933  0.0 |  0.36381  0.701991  0.377995  0.0 |  1.642161   1.408849   1.618933  0.0 | 0.0  0.0  0.0  0.0 |      29.0       29.0       29.0  0.0 |  0.00003   0.00003   0.00003  0.0 |  1.84 | 0.0 | 0.023727 |   1.54852 |  0.0 |  0.02025 |   0.6588 | 0.488386 | 0.0 | 0.0 | 11.337868 |  -95.251471 |  0.0 |   2.59304 |  2.171735 | 2.461991 | 0.284953 |  0.474253   2.361553   2.369738  0.0 | -3.000161  -3.000161  -3.000161  0.0 | nan |   26.886631 | nan |  313.023081 | 0.857083 |  726.636002 | 0.284953 |
    | 2017-02-22 00:00:00 |  6.5 | 18.0 | 10.0 | 0.0 | 0.0 |  14.4 | 0.5256  0.936  0.504  0.8784 | 0.5256  0.936  0.504  0.0 | 13.589478  13.049179  13.564578  0.0 | 0.504698  0.935996  0.503997  0.0 | 13.589478  13.049179  13.564578  0.0 | 0.0  0.0  0.0  0.0 |      32.5       32.5       32.5  0.0 | 0.000033  0.000033  0.000033  0.0 |  14.4 | 0.0 | 0.104481 | 13.330898 |  0.0 | 0.013929 |   0.8784 |  0.65119 | 0.0 | 0.0 | 11.337868 | -118.317035 |  0.0 |   2.29727 |  8.236518 |  3.55707 | 0.411698 |  0.780078   2.776379   2.701163  0.0 | -3.000194  -3.000195  -3.000195  0.0 | nan |    17.75548 | nan |  194.706047 | 5.951464 | 1067.203114 | 0.411698 |
    | 2017-02-23 00:00:00 |  5.0 | 19.2 | 10.0 | 0.0 | 0.0 | 15.36 | 0.3285  0.585  0.315   0.549 | 0.3285  0.585  0.315  0.0 | 14.927269  14.656171  14.931694  0.0 | 0.322035  0.584999  0.314999  0.0 | 14.927269  14.656171  14.931694  0.0 | 0.0  0.0  0.0  0.0 |      25.0       25.0       25.0  0.0 | 0.000026  0.000026  0.000026  0.0 | 15.36 | 0.0 | 0.038178 | 14.812048 |  0.0 | 0.004309 |    0.549 | 0.406996 | 0.0 | 0.0 | 11.337868 | -171.354682 |  0.0 |   0.76195 |  13.53591 | 6.280166 | 0.726871 |  0.890774   2.895209    2.81447  0.0 |  -3.00022  -3.000221  -3.000221  0.0 | nan |    7.145693 | nan |   23.351365 | 7.227602 | 1464.867412 | 0.726871 |
    | 2017-02-24 00:00:00 |  3.0 |  0.4 | 10.0 | 0.0 | 0.0 |  0.32 | 0.5256  0.936  0.504  0.8784 | 0.5256  0.936  0.504  0.0 |  0.294508   0.228504   0.294805  0.0 | 0.495654  0.935979  0.503995  0.0 |  0.294508   0.228504   0.294805  0.0 | 0.0  0.0  0.0  0.0 |  15.00006   15.00006   15.00006  0.0 | 0.000015  0.000015  0.000015  0.0 |  0.32 | 0.0 |  0.00005 |  0.274707 |  0.0 | 0.019964 |   0.8784 |  0.65119 | 0.0 | 0.0 | 11.337868 |  -25.760905 |  0.0 |  0.470714 |  6.412005 | 7.794027 | 0.902086 |  0.420612   2.050726   2.335669  0.0 | -3.000236  -3.000236  -3.000236  0.0 | nan |   -3.701548 | nan |    -2.40954 | 1.090304 | 1409.554385 | 0.902086 |
    | 2017-02-25 00:00:00 |  3.1 |  8.3 | 10.0 | 0.0 | 0.0 |  6.64 | 0.3285  0.585  0.315   0.549 | 0.3285  0.585  0.315  0.0 |  6.067423   5.421468   6.057732  0.0 | 0.308292  0.584994  0.314997  0.0 |  6.067423   5.421468   6.057732  0.0 | 0.0  0.0  0.0  0.0 | 15.500043  15.500043  15.500043  0.0 | 0.000016  0.000016  0.000016  0.0 |  6.64 | 0.0 |      0.0 |  5.872684 |  0.0 |  0.01347 |    0.549 | 0.406994 | 0.0 | 0.0 | 11.337868 |   -6.780234 |  0.0 |  4.500193 |   4.21241 | 7.327723 | 0.848116 |  0.684897   2.684264    2.60294  0.0 | -3.000251  -3.000252  -3.000252  0.0 | nan |  -10.525753 | nan |   -9.189775 | 2.750577 | 1454.125841 | 0.848116 |
    | 2017-02-26 00:00:00 |  7.1 |  5.3 | 10.0 | 0.0 | 0.0 |  4.24 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |  3.990618   3.797448   3.989293  0.0 | 0.251901  0.467998  0.251998  0.0 |  3.990618   3.797448   3.989293  0.0 | 0.0  0.0  0.0  0.0 |      35.5       35.5       35.5  0.0 | 0.000037  0.000037  0.000037  0.0 |  4.24 | 0.0 |      0.0 |  3.932571 |  0.0 | 0.007265 |   0.4392 | 0.325597 | 0.0 | 0.0 | 11.337868 |   -7.072746 |  0.0 |  4.217222 |  4.611493 |  7.77218 | 0.899558 |  0.682378   2.658817   2.601648  0.0 | -3.000288  -3.000288  -3.000288  0.0 | nan |  -17.639134 | nan |  -16.262521 | 2.071655 | 1481.260298 | 0.899558 |
    | 2017-02-27 00:00:00 |  9.4 |  0.7 | 10.0 | 0.0 | 0.0 |  0.56 | 0.8541  1.521  0.819  1.4274 | 0.8541  1.521  0.819  0.0 |  0.424053   0.206901   0.404342  0.0 | 0.695521   1.52066  0.818961  0.0 |  0.424053   0.206901   0.404342  0.0 | 0.0  0.0  0.0  0.0 |      47.0       47.0       47.0  0.0 | 0.000048  0.000048  0.000048  0.0 |  0.56 | 0.0 |      0.0 |  0.356985 |  0.0 | 0.105809 |   1.4274 |  1.05817 | 0.0 | 0.0 | 11.337868 |   -5.373284 |  0.0 |  6.507357 |  2.016054 | 7.818686 | 0.904941 |  0.122804   1.491256   1.938345  0.0 | -3.000336  -3.000337  -3.000337  0.0 | nan |  -22.363837 | nan |  -21.635805 | 0.412586 | 1475.219678 | 0.904941 |
    | 2017-02-28 00:00:00 |  4.6 |  2.7 | 10.0 | 0.0 | 0.0 |  2.16 | 0.5913  1.053  0.567  0.9882 | 0.5913  1.053  0.567  0.0 |  1.504266   0.475364   1.340625  0.0 |  0.45369  1.052728  0.566963  0.0 |  1.504266   0.475364   1.340625  0.0 | 0.0  0.0  0.0  0.0 |      23.0       23.0       23.0  0.0 | 0.000024  0.000024  0.000024  0.0 |  2.16 | 0.0 |      0.0 |  1.179255 |  0.0 | 0.091812 |   0.9882 | 0.732576 | 0.0 | 0.0 | 11.337868 |   -3.307634 |  0.0 |   7.99577 |  0.990315 |  7.84068 | 0.907486 |  0.324849   2.123164   2.190757  0.0 |  -3.00036  -3.000361  -3.000361  0.0 | nan |  -25.614123 | nan |  -24.943439 | 0.601526 | 1485.496379 | 0.907486 |
    | 2017-03-01 00:00:00 |  3.7 |  1.6 | 10.0 | 0.0 | 0.0 |  1.28 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |  0.950677   0.497096    0.87597  0.0 | 0.384085  0.818947  0.440979  0.0 |  0.950677   0.497096    0.87597  0.0 | 0.0  0.0  0.0  0.0 | 18.500006  18.500006  18.500006  0.0 | 0.000019  0.000019  0.000019  0.0 |  1.28 | 0.0 |      0.0 |  0.807151 |  0.0 | 0.067348 |   0.7938 | 0.585105 | 0.0 | 0.0 | 11.337868 |   -3.305347 |  0.0 |  8.004341 |  0.982014 | 7.909207 | 0.915417 |  0.270086    2.08712   2.153807  0.0 | -3.000379   -3.00038   -3.00038  0.0 | nan |  -28.880303 | nan |  -28.248786 | 0.426663 | 1491.632381 | 0.915417 |
    | 2017-03-02 00:00:00 |  4.7 |  2.5 | 10.0 | 0.0 | 0.0 |   2.0 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |  1.532258   1.008873    1.46039  0.0 | 0.393812  0.818966  0.440983  0.0 |  1.532258   1.008873    1.46039  0.0 | 0.0  0.0  0.0  0.0 |      23.5       23.5       23.5  0.0 | 0.000024  0.000024  0.000024  0.0 |   2.0 | 0.0 |      0.0 |   1.36808 |  0.0 | 0.060859 |   0.7938 | 0.585107 | 0.0 | 0.0 | 11.337868 |   -3.267028 |  0.0 |  7.944703 |  1.133532 | 7.969104 |  0.92235 |  0.344016   2.259281   2.252434  0.0 | -3.000403  -3.000404  -3.000404  0.0 | nan |  -32.212609 | nan |  -31.515814 | 0.661211 | 1500.287833 |  0.92235 |
    | 2017-03-03 00:00:00 |  5.9 |  0.6 | 10.0 | 0.0 | 0.0 |  0.48 | 0.7623  1.287  0.693  1.2474 | 0.7623  1.287  0.693  0.0 |  0.289004   0.088949   0.242189  0.0 | 0.505952  1.286374  0.692918  0.0 |  0.289004   0.088949   0.242189  0.0 | 0.0  0.0  0.0  0.0 |      29.5       29.5       29.5  0.0 |  0.00003   0.00003   0.00003  0.0 |  0.48 | 0.0 |      0.0 |  0.224336 |  0.0 | 0.171079 |   1.2474 | 0.919439 | 0.0 | 0.0 | 11.337868 |    -3.01492 |  0.0 |  8.352741 |  0.711968 | 8.009543 |  0.92703 |  0.029061   1.363959   1.797327  0.0 | -3.000434  -3.000434  -3.000434  0.0 | nan |  -35.026659 | nan |  -34.530733 | 0.173579 | 1502.285576 |  0.92703 |
    | 2017-03-04 00:00:00 |  7.7 |  0.2 | 10.0 | 0.0 | 0.0 |  0.16 |  0.693   1.17   0.63   1.134 |  0.693   1.17   0.63  0.0 |  0.055998   0.000737   0.020493  0.0 | 0.286944  1.128621  0.629375  0.0 |  0.055998   0.000737   0.020493  0.0 | 0.0  0.0  0.0  0.0 |      38.5       38.5       38.5  0.0 |  0.00004   0.00004   0.00004  0.0 |  0.16 | 0.0 |      0.0 |  0.035909 |  0.0 |  0.28445 |    1.134 | 0.835782 | 0.0 | 0.0 | 11.337868 |   -2.163171 |  0.0 |  8.999879 |  0.171979 | 8.029477 | 0.929338 | -0.153882     0.3946   1.307459  0.0 | -3.000473  -3.000474  -3.000474  0.0 | nan |  -37.080198 | nan |  -36.693904 | 0.037509 | 1505.159376 | 0.929338 |
    | 2017-03-05 00:00:00 |  6.3 |  1.7 | 10.0 | 0.0 | 0.0 |  1.36 | 0.5544  0.936  0.504  0.9072 | 0.5544  0.936  0.504  0.0 |  0.699272   0.001281   0.261612  0.0 | 0.321169  0.881292  0.503638  0.0 |  0.699272   0.001281   0.261612  0.0 | 0.0  0.0  0.0  0.0 |      31.5       31.5       31.5  0.0 | 0.000032  0.000032  0.000032  0.0 |  1.36 | 0.0 |      0.0 |  0.446141 |  0.0 | 0.173696 |   0.9072 | 0.668648 | 0.0 | 0.0 | 11.337868 |   -2.128034 |  0.0 |  8.949279 |  0.264427 | 8.061202 | 0.933009 |  0.185678   0.872028   1.902208  0.0 | -3.000506  -3.000506  -3.000506  0.0 | nan |  -39.295091 | nan |  -38.821938 | 0.219222 | 1510.172203 | 0.933009 |
    | 2017-03-06 00:00:00 |  3.7 |  0.3 | 10.0 | 0.0 | 0.0 |  0.24 | 0.4158  0.702  0.378  0.6804 | 0.4158  0.702  0.378  0.0 |  0.131675   0.000231   0.068176  0.0 | 0.255712  0.664055  0.377897  0.0 |  0.131675   0.000231   0.068176  0.0 | 0.0  0.0  0.0  0.0 | 18.500006  18.500006  18.500006  0.0 | 0.000019  0.000019  0.000019  0.0 |  0.24 | 0.0 |      0.0 |  0.085911 |  0.0 | 0.119356 |   0.6804 | 0.501505 | 0.0 | 0.0 | 11.337868 |   -2.285899 |  0.0 |  8.983597 |  0.239648 |  8.09241 | 0.936622 |  0.038291   0.447742   1.696135  0.0 | -3.000525  -3.000525  -3.000525  0.0 | nan |  -41.530007 | nan |  -41.107838 | 0.065486 | 1513.030665 | 0.936622 |
    | 2017-03-07 00:00:00 |  1.6 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |       0.0        0.0        0.0  0.0 | 0.203127  0.528999  0.440409  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  8.006707   8.006707   8.006707  0.0 | 0.000008  0.000008  0.000008  0.0 |   0.0 | 0.0 |      0.0 |  0.000008 |  0.0 | 0.284608 |   0.7938 | 0.585025 | 0.0 | 0.0 | 11.337868 |   -1.909937 |  0.0 |  9.215406 |  0.056508 | 8.115161 | 0.939255 | -0.164836  -0.081258   1.255726  0.0 | -3.000533  -3.000533  -3.000533  0.0 | nan |  -43.367861 | nan |  -43.017774 | 0.008986 | 1515.647073 | 0.939255 |
    | 2017-03-08 00:00:00 |  4.0 |  1.8 | 10.0 | 0.0 | 0.0 |  1.44 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |  0.768015   0.000467   0.302361  0.0 | 0.289689  0.646975   0.44066  0.0 |  0.768015   0.000467   0.302361  0.0 | 0.0  0.0  0.0  0.0 | 20.000002  20.000002  20.000002  0.0 | 0.000021  0.000021  0.000021  0.0 |  1.44 | 0.0 |      0.0 |  0.491206 |  0.0 |  0.18759 |   0.7938 | 0.585062 | 0.0 | 0.0 | 11.337868 |   -2.002462 |  0.0 |  9.033776 |  0.260782 | 8.143848 | 0.942575 |  0.217459   0.711301   1.952705  0.0 | -3.000553  -3.000554  -3.000554  0.0 | nan |  -45.484364 | nan |  -45.020237 |  0.23941 | 1520.268753 | 0.942575 |
    | 2017-03-09 00:00:00 |  5.6 |  8.9 | 10.0 | 0.0 | 0.0 |  7.12 | 0.3465  0.585  0.315   0.567 | 0.3465  0.585  0.315  0.0 |  6.357862   4.564533   6.159076  0.0 | 0.317363  0.584264  0.314994  0.0 |  6.357862   4.564533   6.159076  0.0 | 0.0  0.0  0.0  0.0 |      28.0       28.0       28.0  0.0 | 0.000029  0.000029  0.000029  0.0 |  7.12 | 0.0 |      0.0 |  5.800014 |  0.0 | 0.019668 |    0.567 | 0.417939 | 0.0 | 0.0 | 11.337868 |   -4.423548 |  0.0 |  5.966305 |  3.318541 | 8.245546 | 0.954346 |  0.662235   2.682504   2.598635  0.0 | -3.000582  -3.000583  -3.000583  0.0 | nan |   -50.83626 | nan |  -49.443785 | 2.720883 | 1540.266989 | 0.954346 |
    | 2017-03-10 00:00:00 |  5.8 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 | 0.5544  0.936  0.504  0.9072 | 0.5544  0.936  0.504  0.0 |       0.0        0.0        0.0  0.0 | 0.473407   0.93593  0.503987  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      29.0       29.0       29.0  0.0 |  0.00003   0.00003   0.00003  0.0 |   0.0 | 0.0 |      0.0 |   0.00003 |  0.0 | 0.054013 |   0.9072 | 0.668701 | 0.0 | 0.0 | 11.337868 |   -5.347612 |  0.0 |  6.689538 |  2.353198 | 8.293707 |  0.95992 |  0.188828   1.746574   2.094648  0.0 | -3.000612  -3.000612  -3.000612  0.0 | nan |  -55.430578 | nan |  -54.791397 | 0.367714 |  1534.98974 |  0.95992 |
    | 2017-03-11 00:00:00 |  5.7 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 |  0.693   1.17   0.63   1.134 |  0.693   1.17   0.63  0.0 |       0.0        0.0        0.0  0.0 | 0.342769  1.155579  0.629754  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      28.5       28.5       28.5  0.0 | 0.000029  0.000029  0.000029  0.0 |   0.0 | 0.0 |      0.0 |  0.000029 |  0.0 | 0.238268 |    1.134 | 0.835837 | 0.0 | 0.0 | 11.337868 |   -2.288398 |  0.0 |  9.101613 |  0.317906 | 8.286049 | 0.959033 | -0.153941   0.590995   1.464894  0.0 | -3.000641  -3.000642  -3.000642  0.0 | nan |  -57.428565 | nan |  -57.079795 | 0.049838 | 1536.511814 | 0.959033 |
    | 2017-03-12 00:00:00 |  4.6 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 | 0.8316  1.404  0.756  1.3608 | 0.8316  1.404  0.756  0.0 |       0.0        0.0        0.0  0.0 | 0.194386  0.813195  0.748164  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      23.0       23.0       23.0  0.0 | 0.000024  0.000024  0.000024  0.0 |   0.0 | 0.0 |      0.0 |  0.000024 |  0.0 | 0.621679 |   1.3608 | 1.002235 | 0.0 | 0.0 | 11.337868 |   -1.408605 |  0.0 |  9.427045 |  0.043089 | 8.299973 | 0.960645 | -0.348328  -0.222201    0.71673  0.0 | -3.000665  -3.000665  -3.000665  0.0 | nan |   -58.71771 | nan |  -58.488399 | 0.006772 | 1537.996418 | 0.960645 |
    | 2017-03-13 00:00:00 |  4.2 |  0.9 | 10.0 | 0.0 | 0.0 |  0.72 | 0.6237  1.053  0.567  1.0206 | 0.6237  1.053  0.567  0.0 |  0.198895   0.000023    0.00296  0.0 | 0.207398  0.453561  0.552677  0.0 |  0.198895   0.000023    0.00296  0.0 | 0.0  0.0  0.0  0.0 | 21.000001  21.000001  21.000001  0.0 | 0.000022  0.000022  0.000022  0.0 |  0.72 | 0.0 |      0.0 |  0.119662 |  0.0 | 0.477299 |   1.0206 | 0.750849 | 0.0 | 0.0 | 11.337868 |   -1.371285 |  0.0 |   9.40645 |  0.066023 | 8.314634 | 0.962342 |  -0.03462   0.044215   0.881093  0.0 | -3.000687  -3.000687  -3.000687  0.0 | nan |   -60.17183 | nan |  -59.859685 | 0.060411 | 1540.023671 | 0.962342 |
    | 2017-03-14 00:00:00 |  7.4 |  0.1 | 10.0 | 0.0 | 0.0 |  0.08 | 0.6237  1.053  0.567  1.0206 | 0.6237  1.053  0.567  0.0 |  0.023841   0.000002   0.000192  0.0 | 0.223276  0.376887  0.534972  0.0 |  0.023841   0.000002   0.000192  0.0 | 0.0  0.0  0.0  0.0 |      37.0       37.0       37.0  0.0 | 0.000038  0.000038  0.000038  0.0 |  0.08 | 0.0 |      0.0 |  0.014363 |  0.0 | 0.492272 |   1.0206 | 0.749114 | 0.0 | 0.0 | 11.337868 |   -1.451432 |  0.0 |  9.435495 |  0.060692 | 8.330075 | 0.964129 | -0.201738  -0.252674   0.425928  0.0 | -3.000725  -3.000725  -3.000725  0.0 | nan |  -61.581932 | nan |  -61.311116 | 0.014082 | 1541.658538 | 0.964129 |
    | 2017-03-15 00:00:00 |  6.3 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 | 0.8316  1.404  0.756  1.3608 | 0.8316  1.404  0.756  0.0 |       0.0        0.0        0.0  0.0 | 0.171522   0.21667    0.4912  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      31.5       31.5       31.5  0.0 | 0.000032  0.000032  0.000032  0.0 |   0.0 | 0.0 |      0.0 |  0.000032 |  0.0 | 0.835747 |   1.3608 | 0.977039 | 0.0 | 0.0 | 11.337868 |   -1.070292 |  0.0 |  9.504299 |  0.012139 | 8.342553 | 0.965573 |  -0.37326  -0.469344  -0.065272  0.0 | -3.000757  -3.000757  -3.000757  0.0 | nan |  -62.579754 | nan |  -62.381408 | 0.001975 | 1542.904461 | 0.965573 |
    | 2017-03-16 00:00:00 |  8.7 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 | 0.9702  1.638  0.882  1.5876 | 0.9702  1.638  0.882  0.0 |       0.0        0.0        0.0  0.0 | 0.116223  0.128208  0.249667  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      43.5       43.5       43.5  0.0 | 0.000045  0.000045  0.000045  0.0 |   0.0 | 0.0 |      0.0 |  0.000045 |  0.0 |  1.07248 |   1.5876 | 1.108179 | 0.0 | 0.0 | 11.337868 |   -0.788816 |  0.0 |  9.537177 |  0.001726 | 8.351801 | 0.966644 | -0.489483  -0.597552  -0.314939  0.0 | -3.000802  -3.000802  -3.000802  0.0 | nan |  -63.307966 | nan |  -63.170225 | 0.000294 |  1544.40086 | 0.966644 |
    | 2017-03-17 00:00:00 |  6.4 |  3.9 | 10.0 | 0.0 | 0.0 |  3.12 | 0.7623  1.287  0.693  1.2474 | 0.7623  1.287  0.693  0.0 |   1.81855   0.016239   0.253053  0.0 | 0.484984  0.989407  0.602651  0.0 |   1.81855   0.016239   0.253053  0.0 | 0.0  0.0  0.0  0.0 |      32.0       32.0       32.0  0.0 | 0.000033  0.000033  0.000033  0.0 |  3.12 | 0.0 |      0.0 |   1.12134 |  0.0 | 0.284049 |   1.2474 | 0.910595 | 0.0 | 0.0 | 11.337868 |   -1.724514 |  0.0 |  8.935511 |  0.551642 |  8.37268 |  0.96906 |  0.326983   1.516802   1.949358  0.0 | -3.000834  -3.000835  -3.000835  0.0 | nan |  -65.426275 | nan |  -64.894739 | 0.569992 | 1548.725948 |  0.96906 |
    | 2017-03-18 00:00:00 |  5.2 |  8.7 | 10.0 | 0.0 | 0.0 |  6.96 | 0.7623  1.287  0.693  1.2474 | 0.7623  1.287  0.693  0.0 |  6.071439   4.671048   5.751904  0.0 | 0.685957  1.286933  0.692984  0.0 |  6.071439   4.671048   5.751904  0.0 | 0.0  0.0  0.0  0.0 |      26.0       26.0       26.0  0.0 | 0.000027  0.000027  0.000027  0.0 |  6.96 | 0.0 |      0.0 |  5.619395 |  0.0 | 0.050913 |   1.2474 | 0.919467 | 0.0 | 0.0 | 11.337868 |   -4.678907 |  0.0 |  5.785493 |  3.579887 |  8.45148 | 0.978181 |  0.529587    2.51882    2.46447  0.0 | -3.000861  -3.000862  -3.000862  0.0 | nan |  -70.927737 | nan |  -69.573646 |   2.6095 | 1562.419202 | 0.978181 |
    | 2017-03-19 00:00:00 |  5.1 | 26.4 | 10.0 | 0.0 | 0.0 | 21.12 | 0.3465  0.585  0.315   0.567 | 0.3465  0.585  0.315  0.0 | 20.402804  20.094766  20.410313  0.0 | 0.337475  0.584999  0.314999  0.0 | 20.402804  20.094766  20.410313  0.0 | 0.0  0.0  0.0  0.0 |      25.5       25.5       25.5  0.0 | 0.000026  0.000026  0.000026  0.0 | 21.12 | 0.0 |      0.0 |  20.31117 |  0.0 | 0.006017 |    0.567 | 0.417941 | 0.0 | 0.0 | 11.337868 |  -13.741511 |  0.0 | -5.099846 | 13.714547 | 8.666954 |  1.00312 |  0.909308   2.959056   2.859158  0.0 | -3.000887  -3.000888  -3.000888  0.0 | nan |  -87.359435 | nan |  -83.315157 | 9.206122 | 1596.734119 |  1.00312 |
    | 2017-03-20 00:00:00 |  8.7 | 11.5 | 10.0 | 0.0 | 0.0 |   9.2 | 0.4158  0.702  0.378  0.6804 | 0.4158  0.702  0.378  0.0 |  8.931936   8.697492   8.951417  0.0 | 0.406537  0.701998  0.377999  0.0 |  8.931936   8.697492   8.951417  0.0 | 0.0  0.0  0.0  0.0 |      43.5       43.5       43.5  0.0 | 0.000045  0.000045  0.000045  0.0 |   9.2 | 0.0 |      0.0 |  8.863595 |  0.0 | 0.006176 |   0.6804 |  0.50153 | 0.0 | 0.0 | 11.337868 |  -17.221205 |  0.0 | -4.551638 | 13.006593 | 8.799881 | 1.018505 |  0.770835   2.759566   2.729743  0.0 | -3.000932  -3.000932  -3.000932  0.0 | nan | -103.242765 | nan | -100.536362 | 5.063124 | 1601.855507 | 1.018505 |
    | 2017-03-21 00:00:00 |  6.2 |  0.9 | 10.0 | 0.0 | 0.0 |  0.72 | 1.0395  1.755  0.945   1.701 | 1.0395  1.755  0.945  0.0 |  0.554136    0.27601   0.548116  0.0 | 0.842885  1.754558  0.944963  0.0 |  0.554136    0.27601   0.548116  0.0 | 0.0  0.0  0.0  0.0 |      31.0       31.0       31.0  0.0 | 0.000032  0.000032  0.000032  0.0 |  0.72 | 0.0 |      0.0 |  0.470128 |  0.0 | 0.131219 |    1.701 | 1.253818 | 0.0 | 0.0 | 11.337868 |   -8.343374 |  0.0 |  4.769683 |  4.675708 | 8.813693 | 1.020103 |  0.093815   1.448998   1.956663  0.0 | -3.000964  -3.000964  -3.000964  0.0 | nan | -109.679732 | nan | -108.879736 | 0.857544 | 1599.642563 | 1.020103 |
    | 2017-03-22 00:00:00 |  5.9 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 |  1.386   2.34   1.26   2.268 |  1.386   2.34   1.26  0.0 |       0.0        0.0        0.0  0.0 | 0.443224  1.687536  1.251484  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      29.5       29.5       29.5  0.0 |  0.00003   0.00003   0.00003  0.0 |   0.0 | 0.0 |      0.0 |   0.00003 |  0.0 | 0.845974 |    2.268 | 1.670906 | 0.0 | 0.0 | 11.337868 |   -1.958155 |  0.0 |  9.229906 |  0.741361 | 8.811762 |  1.01988 | -0.349409  -0.238537   0.705179  0.0 | -3.000994  -3.000994  -3.000994  0.0 | nan |  -110.94172 | nan | -110.837891 | 0.116214 | 1600.151973 |  1.01988 |
    | 2017-03-23 00:00:00 |  5.2 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 | 1.1088  1.872  1.008  1.8144 | 1.1088  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.139055  0.270034  0.749157  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      26.0       26.0       26.0  0.0 | 0.000027  0.000027  0.000027  0.0 |   0.0 | 0.0 |      0.0 |  0.000027 |  0.0 | 1.180443 |   1.8144 |  1.31201 | 0.0 | 0.0 | 11.337868 |   -0.312601 |  0.0 |  9.920933 |  0.100395 | 8.813482 | 1.020079 | -0.488464  -0.508571  -0.043978  0.0 | -3.001021  -3.001021  -3.001021  0.0 | nan | -111.178213 | nan | -111.150492 | 0.015846 |  1600.09592 | 1.020079 |
    | 2017-03-24 00:00:00 |  5.2 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 | 1.1088  1.872  1.008  1.8144 | 1.1088  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.087824  0.124941  0.285357  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      26.0       26.0       26.0  0.0 | 0.000027  0.000027  0.000027  0.0 |   0.0 | 0.0 |      0.0 |  0.000027 |  0.0 | 1.262958 |   1.8144 | 1.266555 | 0.0 | 0.0 | 11.337868 |   -0.075794 |  0.0 | 10.021932 |  0.013699 | 8.814087 | 1.020149 | -0.576288  -0.633511  -0.329335  0.0 | -3.001047  -3.001048  -3.001048  0.0 | nan | -111.231191 | nan | -111.226286 | 0.002174 | 1600.215605 | 1.020149 |
    | 2017-03-25 00:00:00 |  5.9 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 | 0.8316  1.404  0.756  1.3608 | 0.8316  1.404  0.756  0.0 |       0.0        0.0        0.0  0.0 | 0.049332  0.062932  0.109506  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      29.5       29.5       29.5  0.0 |  0.00003   0.00003   0.00003  0.0 |   0.0 | 0.0 |      0.0 |   0.00003 |  0.0 |   0.9685 |   1.3608 | 0.939674 | 0.0 | 0.0 | 11.337868 |   -0.280474 |  0.0 | 10.033512 |  0.001894 | 8.815369 | 1.020297 |  -0.62562  -0.696443  -0.438841  0.0 | -3.001078  -3.001078  -3.001078  0.0 | nan | -111.567048 | nan |  -111.50676 |  0.00031 | 1600.657052 | 1.020297 |
    | 2017-03-26 00:00:00 |  6.7 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 | 0.9009  1.521  0.819  1.4742 | 0.9009  1.521  0.819  0.0 |       0.0        0.0        0.0  0.0 | 0.043678  0.052857  0.081065  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      33.5       33.5       33.5  0.0 | 0.000034  0.000034  0.000034  0.0 |   0.0 | 0.0 |      0.0 |  0.000034 |  0.0 | 1.060824 |   1.4742 | 1.014298 | 0.0 | 0.0 | 11.337868 |   -0.257698 |  0.0 | 10.032795 |  0.000287 | 8.818048 | 1.020607 | -0.669298    -0.7493  -0.519906  0.0 | -3.001112  -3.001112  -3.001112  0.0 | nan | -111.811298 | nan | -111.764458 | 0.000057 | 1600.740771 | 1.020607 |
    | 2017-03-27 00:00:00 |  7.0 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 | 1.1088  1.872  1.008  1.8144 | 1.1088  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.044309  0.051566  0.072339  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      35.0       35.0       35.0  0.0 | 0.000036  0.000036  0.000036  0.0 |   0.0 | 0.0 |      0.0 |  0.000036 |  0.0 | 1.316424 |   1.8144 | 1.245678 | 0.0 | 0.0 | 11.337868 |   -0.032241 |  0.0 | 10.045225 |   0.00007 | 8.819349 | 1.020758 | -0.713608  -0.800866  -0.592246  0.0 | -3.001148  -3.001148  -3.001148  0.0 | nan | -111.787517 | nan | -111.796699 | 0.000023 | 1600.956729 | 1.020758 |
    | 2017-03-28 00:00:00 |  8.3 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 | 1.3167  2.223  1.197  2.1546 | 1.3167  2.223  1.197  0.0 |       0.0        0.0        0.0  0.0 | 0.043339  0.048902  0.064086  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      41.5       41.5       41.5  0.0 | 0.000043  0.000043  0.000043  0.0 |   0.0 | 0.0 |      0.0 |  0.000043 |  0.0 |  1.57355 |   2.1546 | 1.477104 | 0.0 | 0.0 | 11.337868 |    0.226425 |  0.0 |  10.03735 |  0.000044 | 8.818534 | 1.020664 | -0.756947  -0.849768  -0.656332  0.0 |  -3.00119  -3.001191  -3.001191  0.0 | nan | -111.514486 | nan | -111.570274 | 0.000022 | 1600.524753 | 1.020664 |
    | 2017-03-29 00:00:00 |  9.0 |  1.5 | 10.0 | 0.0 | 0.0 |   1.2 | 0.5544  0.936  0.504  0.9072 | 0.5544  0.936  0.504  0.0 |  0.262344   0.000023   0.000145  0.0 | 0.145287  0.238686  0.204268  0.0 |  0.262344   0.000023   0.000145  0.0 | 0.0  0.0  0.0  0.0 |      45.0       45.0       45.0  0.0 | 0.000046  0.000046  0.000046  0.0 |   1.2 | 0.0 |      0.0 |  0.157474 |  0.0 | 0.505162 |   0.9072 | 0.639318 | 0.0 | 0.0 | 11.337868 |   -0.618551 |  0.0 |  9.938124 |  0.064378 | 8.820236 | 1.020861 |  0.035422   0.111523   0.339255  0.0 | -3.001236  -3.001237  -3.001237  0.0 | nan | -112.409069 | nan | -112.188825 | 0.093117 | 1601.231569 | 1.020861 |
    | 2017-03-30 00:00:00 | 12.4 |  0.3 | 10.0 | 0.0 | 0.0 |  0.24 | 1.0395  1.755  0.945   1.701 | 1.0395  1.755  0.945  0.0 |  0.076871   0.000006    0.00005  0.0 | 0.396709  0.624107  0.601753  0.0 |  0.076871   0.000006    0.00005  0.0 | 0.0  0.0  0.0  0.0 |      62.0       62.0       62.0  0.0 | 0.000064  0.000064  0.000064  0.0 |  0.24 | 0.0 |      0.0 |  0.046193 |  0.0 | 0.805463 |    1.701 | 1.220163 | 0.0 | 0.0 | 11.337868 |    -0.72093 |  0.0 |  9.938832 |  0.108768 | 8.827675 | 1.021722 | -0.198158   -0.27259  -0.022548  0.0 |   -3.0013    -3.0013    -3.0013  0.0 | nan | -113.002642 | nan | -112.909755 | 0.030542 | 1602.018934 | 1.021722 |
    | 2017-03-31 00:00:00 | 15.0 |  0.2 | 10.0 | 0.0 | 0.0 |  0.16 | 1.8711  3.159  1.701  3.0618 | 1.8711  3.159  1.701  0.0 |  0.023626   0.000001   0.000007  0.0 | 0.346844  0.403413  0.470504  0.0 |  0.023626   0.000001   0.000007  0.0 | 0.0  0.0  0.0  0.0 |      75.0       75.0       75.0  0.0 | 0.000077  0.000077  0.000077  0.0 |  0.16 | 0.0 |      0.0 |  0.014253 |  0.0 | 1.934631 |   3.0618 | 2.136238 | 0.0 | 0.0 | 11.337868 |    0.390604 |  0.0 | 10.022252 |  0.035203 | 8.828682 | 1.021838 | -0.408628  -0.516004  -0.333059  0.0 | -3.001377  -3.001377  -3.001377  0.0 | nan | -112.383627 | nan |  -112.51915 | 0.009593 | 1601.389254 | 1.021838 |
    | 2017-04-01 00:00:00 | 11.8 |  4.5 | 10.0 | 0.0 | 0.0 |   3.6 | 1.2825  1.755  0.945   1.728 | 1.2825  1.755  0.945  0.0 |    1.9917   0.031385    0.41475  0.0 | 0.874826  1.427544  0.831626  0.0 |    1.9917   0.031385    0.41475  0.0 | 0.0  0.0  0.0  0.0 |      59.0       59.0       59.0  0.0 |  0.00006   0.00006   0.00006  0.0 |   3.6 | 0.0 |      0.0 |  1.245971 |  0.0 | 0.380921 |    1.728 | 1.386128 | 0.0 | 0.0 | 11.337868 |   -1.037387 |  0.0 |  9.339769 |  0.628215 | 8.834757 | 1.022541 |  0.324846   1.625066   2.020565  0.0 | -3.001437  -3.001438  -3.001438  0.0 | nan | -114.000806 | nan | -113.556537 | 0.627348 | 1604.189768 | 1.022541 |
    | 2017-04-02 00:00:00 |  9.4 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 |  1.368  1.872  1.008  1.8432 |  1.368  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.609807  1.638192   1.00595  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      47.0       47.0       47.0  0.0 | 0.000048  0.000048  0.000048  0.0 |   0.0 | 0.0 |      0.0 |  0.000048 |  0.0 | 0.583379 |   1.8432 | 1.490182 | 0.0 | 0.0 | 11.337868 |   -1.576832 |  0.0 |  9.483414 |  0.542155 | 8.848909 | 1.024179 | -0.284961  -0.013126   1.014615  0.0 | -3.001485  -3.001486  -3.001486  0.0 | nan | -115.271882 | nan |  -115.13337 | 0.085241 | 1604.685319 | 1.024179 |
    | 2017-04-03 00:00:00 |  8.1 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 |   1.71   2.34   1.26   2.304 |   1.71   2.34   1.26  0.0 |       0.0        0.0        0.0  0.0 | 0.227073  0.479042  1.028501  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      40.5       40.5       40.5  0.0 | 0.000041  0.000041  0.000041  0.0 |   0.0 | 0.0 |      0.0 |  0.000041 |  0.0 | 1.608883 |    2.304 | 1.840266 | 0.0 | 0.0 | 11.337868 |      0.0503 |  0.0 |  9.998479 |  0.073651 | 8.850708 | 1.024388 | -0.512034  -0.492168  -0.013886  0.0 | -3.001527  -3.001527  -3.001527  0.0 | nan | -115.002388 | nan | -115.083069 | 0.011632 | 1604.387696 | 1.024388 |
    | 2017-04-04 00:00:00 |  7.9 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 | 1.7955  2.457  1.323  2.4192 | 1.7955  2.457  1.323  0.0 |       0.0        0.0        0.0  0.0 | 0.119567  0.161335  0.357054  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      39.5       39.5       39.5  0.0 |  0.00004   0.00004   0.00004  0.0 |   0.0 | 0.0 |      0.0 |   0.00004 |  0.0 | 1.882447 |   2.4192 | 1.861431 | 0.0 | 0.0 | 11.337868 |    0.558379 |  0.0 | 10.063877 |  0.010068 |  8.84674 | 1.023928 | -0.631601  -0.653502   -0.37094  0.0 | -3.001567  -3.001568  -3.001568  0.0 | nan | -114.393933 | nan | -114.524691 | 0.001604 | 1603.941818 | 1.023928 |
    | 2017-04-05 00:00:00 |  7.5 |  0.4 | 10.0 | 0.0 | 0.0 |  0.32 | 1.4535  1.989  1.071  1.9584 | 1.4535  1.989  1.071  0.0 |  0.014452   0.000001   0.000008  0.0 | 0.119771  0.145598  0.209777  0.0 |  0.014452   0.000001   0.000008  0.0 | 0.0  0.0  0.0  0.0 |      37.5       37.5       37.5  0.0 | 0.000038  0.000038  0.000038  0.0 |  0.32 | 0.0 |      0.0 |  0.008711 |  0.0 | 1.503569 |   1.9584 | 1.499105 | 0.0 | 0.0 | 11.337868 |    0.303242 |  0.0 | 10.045086 |  0.005861 | 8.842571 | 1.023446 | -0.445823  -0.479102  -0.260724  0.0 | -3.001606  -3.001606  -3.001606  0.0 | nan | -114.183147 | nan | -114.221449 | 0.004454 | 1603.450355 | 1.023446 |
    | 2017-04-06 00:00:00 |  7.2 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 | 1.4535  1.989  1.071  1.9584 | 1.4535  1.989  1.071  0.0 |       0.0        0.0        0.0  0.0 | 0.126231  0.143967   0.17563  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      36.0       36.0       36.0  0.0 | 0.000037  0.000037  0.000037  0.0 |   0.0 | 0.0 |      0.0 |  0.000037 |  0.0 | 1.499805 |   1.9584 | 1.495758 | 0.0 | 0.0 | 11.337868 |    0.208968 |  0.0 | 10.057296 |  0.003867 |  8.84045 |   1.0232 | -0.572055  -0.623069  -0.436354  0.0 | -3.001642  -3.001643  -3.001643  0.0 | nan | -113.963914 | nan | -114.012481 | 0.000623 | 1603.185704 |   1.0232 |
    | 2017-04-07 00:00:00 |  8.1 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 |  0.684  0.936  0.504  0.9216 |  0.684  0.936  0.504  0.0 |       0.0        0.0        0.0  0.0 | 0.042007  0.045682  0.053505  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      40.5       40.5       40.5  0.0 | 0.000041  0.000041  0.000041  0.0 |   0.0 | 0.0 |      0.0 |  0.000041 |  0.0 | 0.724743 |   0.9216 |  0.70103 | 0.0 | 0.0 | 11.337868 |   -0.402345 |  0.0 | 10.058633 |  0.000562 | 8.840831 | 1.023244 | -0.614062  -0.668751  -0.489859  0.0 | -3.001684  -3.001684  -3.001684  0.0 | nan | -114.518407 | nan | -114.414826 | 0.000103 |  1603.83577 | 1.023244 |
    | 2017-04-08 00:00:00 |  8.6 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 | 1.1115  1.521  0.819  1.4976 | 1.1115  1.521  0.819  0.0 |       0.0        0.0        0.0  0.0 | 0.055216   0.05892  0.067492  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      43.0       43.0       43.0  0.0 | 0.000044  0.000044  0.000044  0.0 |   0.0 | 0.0 |      0.0 |  0.000044 |  0.0 | 1.191509 |   1.4976 | 1.137267 | 0.0 | 0.0 | 11.337868 |   -0.173302 |  0.0 | 10.061587 |  0.000114 |  8.84412 | 1.023625 | -0.669278   -0.72767  -0.557351  0.0 | -3.001728  -3.001728  -3.001728  0.0 | nan |  -114.60318 | nan | -114.588128 | 0.000033 | 1603.853738 | 1.023625 |
    | 2017-04-09 00:00:00 | 10.5 |  0.0 | 10.0 | 0.0 | 0.0 |   0.0 | 2.1375  2.925  1.575    2.88 | 2.1375  2.925  1.575  0.0 |       0.0        0.0        0.0  0.0 | 0.078861  0.082522  0.092005  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      52.5       52.5       52.5  0.0 | 0.000054  0.000054  0.000054  0.0 |   0.0 | 0.0 |      0.0 |  0.000054 |  0.0 | 2.319839 |     2.88 | 2.183346 | 0.0 | 0.0 | 11.337868 |    0.826183 |  0.0 |  10.06762 |  0.000059 | 8.841466 | 1.023318 |  -0.74814  -0.810193  -0.649357  0.0 | -3.001781  -3.001782  -3.001782  0.0 | nan | -113.553589 | nan | -113.761945 | 0.000028 | 1602.885403 | 1.023318 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _wland_gd_surface_water_supply:

surface water supply
____________________

>>> inputs.fxg.series =  0.0
>>> inputs.fxs.series = 10.0

.. integration-test::

    >>> test("wland_gd_surface_water_supply",
    ...      axis1=(fluxes.pc, fluxes.fqs, fluxes.fgs, fluxes.rh),
    ...      axis2=(states.dg, states.hs))
    |                date |    t |    p | fxg |  fxs | dhs |    pc |                           pe |                       pet |                                   tf |                                ei |                                   rf |                 sf |                                   pm |                                am |    ps | pve |       pv |        pq | etve |      etv |       es |       et |  gr |   fxs | fxg |         cdg | fgse |        fgs |       fqs |       rh |        r |                                   ic |                                   sp | dve |         dv | hge |          dg |       hq |          hs |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 00:00:00 | -2.8 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 | 0.3942  0.702  0.378  0.6588 | 0.3942  0.702  0.378  0.0 |       0.0        0.0        0.0  0.0 |      0.0  0.000001       0.0  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  0.000117   0.000117   0.000117  0.0 |      0.0       0.0       0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.494067 | 0.653311 | 0.448834 | 0.0 | 500.0 | 0.0 |  -72.847932 |  0.0 |  -0.427184 |       0.0 | 0.555335 | 0.064275 |      -3.0  -3.000001       -3.0  0.0 |      -3.0       -3.0       -3.0  0.0 | nan | 140.066883 | nan | 1527.152068 |      0.0 |  450.741142 | 0.064275 |
    | 2017-02-11 00:00:00 | -1.5 |  0.4 | 0.0 | 10.0 | 0.0 |  0.32 | 0.5256  0.936  0.504  0.8784 | 0.5256  0.936  0.504  0.0 |  0.000001        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  0.009388   0.009388   0.009388  0.0 |      0.0       0.0       0.0  0.0 |  0.32 | 0.0 |      0.0 |       0.0 |  0.0 | 0.658793 |   0.8784 | 0.598625 | 0.0 | 500.0 | 0.0 |  -136.55005 |  0.0 |  -1.822211 |       0.0 | 2.136492 | 0.247279 | -2.680002  -2.680003  -2.680002  0.0 |      -3.0       -3.0       -3.0  0.0 | nan | 138.903465 | nan | 1390.602018 |      0.0 |  762.998656 | 0.247279 |
    | 2017-02-12 00:00:00 | -0.9 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 | 0.4599  0.819  0.441  0.7686 | 0.4599  0.819  0.441  0.0 |       0.0        0.0        0.0  0.0 | 0.000002  0.000004  0.000002  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  0.069591   0.069591   0.069591  0.0 |      0.0       0.0       0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.576555 |   0.7686 | 0.523896 | 0.0 | 500.0 | 0.0 | -121.984012 |  0.0 |  -2.976518 |       0.0 | 3.521857 | 0.407622 | -2.680004  -2.680007  -2.680004  0.0 |      -3.0       -3.0       -3.0  0.0 | nan | 136.503501 | nan | 1268.618007 |      0.0 |  954.872782 | 0.407622 |
    | 2017-02-13 00:00:00 | -1.6 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  0.006707   0.006707   0.006707  0.0 |      0.0       0.0       0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 |  0.32956 |   0.4392 | 0.299457 | 0.0 | 500.0 | 0.0 |  -34.467875 |  0.0 |  -3.878821 |       0.0 | 4.445772 | 0.514557 | -2.680005  -2.680009  -2.680005  0.0 |      -3.0       -3.0       -3.0  0.0 | nan |  132.95424 | nan | 1234.150131 |      0.0 | 1061.088982 | 0.514557 |
    | 2017-02-14 00:00:00 | -1.3 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  0.018374   0.018374   0.018374  0.0 |      0.0       0.0       0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.329681 |   0.4392 | 0.299564 | 0.0 | 500.0 | 0.0 |   -32.62078 |  0.0 |  -4.421921 |       0.0 | 4.969671 | 0.575193 | -2.680007  -2.680011  -2.680006  0.0 |      -3.0       -3.0       -3.0  0.0 | nan |    128.862 | nan | 1201.529351 |      0.0 | 1117.159505 | 0.575193 |
    | 2017-02-15 00:00:00 |  1.7 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |   8.50479    8.50479    8.50479  0.0 | 0.000009  0.000009  0.000009  0.0 |   0.0 | 0.0 | 0.000002 |  0.000007 |  0.0 | 0.329803 |   0.4392 | 0.299672 | 0.0 | 500.0 | 0.0 |  -31.773217 |  0.0 |  -4.649601 |  0.000004 | 5.260599 | 0.608866 | -2.680008  -2.680013  -2.680007  0.0 | -3.000008  -3.000009  -3.000009  0.0 | nan |   124.5422 | nan | 1169.756134 | 0.000003 | 1148.643121 | 0.608866 |
    | 2017-02-16 00:00:00 |  4.4 |  0.2 | 0.0 | 10.0 | 0.0 |  0.16 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |  0.000001        0.0        0.0  0.0 | 0.000002  0.000003  0.000002  0.0 |  0.000001        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 | 22.000001  22.000001  22.000001  0.0 | 0.000023  0.000023  0.000023  0.0 |  0.16 | 0.0 | 0.000005 |  0.000019 |  0.0 | 0.329919 |   0.4392 | 0.299774 | 0.0 | 500.0 | 0.0 |  -31.175524 |  0.0 |  -4.714985 |  0.000013 | 5.435694 | 0.629131 |  -2.52001  -2.520016  -2.520009  0.0 | -3.000031  -3.000032  -3.000032  0.0 | nan | 120.157129 | nan | 1138.580611 | 0.000008 | 1168.649065 | 0.629131 |
    | 2017-02-17 00:00:00 |  4.5 |  4.5 | 0.0 | 10.0 | 0.0 |   3.6 | 0.1971  0.351  0.189  0.3294 | 0.1971  0.351  0.189  0.0 |  0.716522   0.000865   0.002553  0.0 | 0.045818  0.103401  0.056198  0.0 |  0.716522   0.000865   0.002553  0.0 | 0.0  0.0  0.0  0.0 |      22.5       22.5       22.5  0.0 | 0.000023  0.000023  0.000023  0.0 |   3.6 | 0.0 | 0.093016 |  0.346736 |  0.0 | 0.182733 |   0.3294 | 0.230606 | 0.0 | 500.0 | 0.0 |  -34.331941 |  0.0 |  -4.716092 |  0.090039 | 5.569049 | 0.644566 |   0.31765   0.975719    1.02124  0.0 | -3.000054  -3.000055  -3.000055  0.0 | nan | 115.530753 | nan |  1104.24867 | 0.256705 | 1189.899487 | 0.644566 |
    | 2017-02-18 00:00:00 |  3.4 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 | 0.1971  0.351  0.189  0.3294 | 0.1971  0.351  0.189  0.0 |       0.0        0.0        0.0  0.0 | 0.147887  0.341617  0.186298  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 | 17.000016  17.000016  17.000016  0.0 | 0.000018  0.000018  0.000018  0.0 |   0.0 | 0.0 | 0.000003 |  0.000015 |  0.0 | 0.035821 |   0.3294 | 0.243833 | 0.0 | 500.0 | 0.0 |   -31.54252 |  0.0 |   -4.73086 |  0.221365 | 5.722816 | 0.662363 |  0.169764   0.634101   0.834942  0.0 | -3.000072  -3.000072  -3.000072  0.0 | nan | 110.835712 | nan |  1072.70615 | 0.035355 | 1205.645261 | 0.662363 |
    | 2017-02-19 00:00:00 |  4.8 |  3.2 | 0.0 | 10.0 | 0.0 |  2.56 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |  2.023732   0.453003   0.899733  0.0 | 0.221727  0.465424  0.251476  0.0 |  2.023732   0.453003   0.899733  0.0 | 0.0  0.0  0.0  0.0 |      24.0       24.0       24.0  0.0 | 0.000025  0.000025  0.000025  0.0 |  2.56 | 0.0 | 0.254687 |   1.21092 |  0.0 | 0.028158 |   0.4392 | 0.325473 | 0.0 | 500.0 | 0.0 |  -38.682073 |  0.0 |  -4.710334 |  0.652981 | 5.862583 |  0.67854 |  0.484305   2.275674   2.243733  0.0 | -3.000097  -3.000097  -3.000097  0.0 | nan | 105.898848 | nan | 1034.024077 | 0.593294 | 1238.907257 |  0.67854 |
    | 2017-02-20 00:00:00 |  6.7 |  4.6 | 0.0 | 10.0 | 0.0 |  3.68 | 0.1971  0.351  0.189  0.3294 | 0.1971  0.351  0.189  0.0 |  3.340013   2.972315   3.208118  0.0 | 0.184109  0.350996  0.188997  0.0 |  3.340013   2.972315   3.208118  0.0 | 0.0  0.0  0.0  0.0 |      33.5       33.5       33.5  0.0 | 0.000035  0.000035  0.000035  0.0 |  3.68 | 0.0 | 0.499651 |  2.766863 |  0.0 | 0.008639 |   0.3294 | 0.244178 | 0.0 | 500.0 | 0.0 |   -45.73599 |  0.0 |  -4.991423 |  2.057405 | 6.254213 | 0.723867 |  0.640182   2.632363   2.526618  0.0 | -3.000131  -3.000131  -3.000131  0.0 | nan | 100.416412 | nan |  988.288087 | 1.302752 | 1310.238246 | 0.723867 |
    | 2017-02-21 00:00:00 |  5.8 |  2.3 | 0.0 | 10.0 | 0.0 |  1.84 | 0.3942  0.702  0.378  0.6588 | 0.3942  0.702  0.378  0.0 |  1.642124   1.408767    1.61889  0.0 | 0.363805  0.701991  0.377995  0.0 |  1.642124   1.408767    1.61889  0.0 | 0.0  0.0  0.0  0.0 |      29.0       29.0       29.0  0.0 |  0.00003   0.00003   0.00003  0.0 |  1.84 | 0.0 | 0.219347 |  1.372411 |  0.0 | 0.020219 |   0.6588 | 0.488355 | 0.0 | 500.0 | 0.0 |  -40.305953 |  0.0 |  -5.298444 |   1.91591 | 6.672962 | 0.772334 |  0.474254   2.361605   2.369733  0.0 | -3.000161  -3.000161  -3.000161  0.0 | nan |  94.918841 | nan |  947.982135 | 0.759252 | 1337.989604 | 0.772334 |
    | 2017-02-22 00:00:00 |  6.5 | 18.0 | 0.0 | 10.0 | 0.0 |  14.4 | 0.5256  0.936  0.504  0.8784 | 0.5256  0.936  0.504  0.0 |  13.58948  13.049203  13.564576  0.0 | 0.504698  0.935996  0.503997  0.0 |  13.58948  13.049203  13.564576  0.0 | 0.0  0.0  0.0  0.0 |      32.5       32.5       32.5  0.0 | 0.000033  0.000033  0.000033  0.0 |  14.4 | 0.0 | 1.641743 | 11.947372 |  0.0 | 0.013907 |   0.8784 | 0.651171 | 0.0 | 500.0 | 0.0 | -117.971599 |  0.0 |  -7.702903 |  7.360121 | 7.422974 |  0.85914 |  0.780076   2.776406    2.70116  0.0 | -3.000194  -3.000195  -3.000195  0.0 | nan |  85.588102 | nan |  830.010536 | 5.346503 | 1501.310427 |  0.85914 |
    | 2017-02-23 00:00:00 |  5.0 | 19.2 | 0.0 | 10.0 | 0.0 | 15.36 | 0.3285  0.585  0.315   0.549 | 0.3285  0.585  0.315  0.0 | 14.927268   14.65619  14.931693  0.0 | 0.322035  0.584999  0.314999  0.0 | 14.927268   14.65619  14.931693  0.0 | 0.0  0.0  0.0  0.0 |      25.0       25.0       25.0  0.0 | 0.000026  0.000026  0.000026  0.0 | 15.36 | 0.0 | 1.319604 |  13.65877 |  0.0 | 0.004304 |    0.549 | 0.406992 | 0.0 | 500.0 | 0.0 | -180.915873 |  0.0 |  -16.25481 | 12.327521 | 8.015642 | 0.927736 |  0.890773   2.895217   2.814468  0.0 |  -3.00022  -3.000221  -3.000221  0.0 | nan |  68.017992 | nan |  649.094663 | 6.677751 | 1502.550735 | 0.927736 |
    | 2017-02-24 00:00:00 |  3.0 |  0.4 | 0.0 | 10.0 | 0.0 |  0.32 | 0.5256  0.936  0.504  0.8784 | 0.5256  0.936  0.504  0.0 |  0.294507   0.228506   0.294804  0.0 | 0.495652  0.935979  0.503995  0.0 |  0.294507   0.228506   0.294804  0.0 | 0.0  0.0  0.0  0.0 |  15.00006   15.00006   15.00006  0.0 | 0.000015  0.000015  0.000015  0.0 |  0.32 | 0.0 |  0.01628 |  0.260099 |  0.0 |  0.01995 |   0.8784 | 0.651176 | 0.0 | 500.0 | 0.0 |  -15.447997 |  0.0 |  -8.917671 |  5.927614 | 8.005939 | 0.926613 |  0.420615   2.050731   2.335669  0.0 | -3.000236  -3.000236  -3.000236  0.0 | nan |   59.10399 | nan |  633.646666 | 1.010236 | 1498.879184 | 0.926613 |
    | 2017-02-25 00:00:00 |  3.1 |  8.3 | 0.0 | 10.0 | 0.0 |  6.64 | 0.3285  0.585  0.315   0.549 | 0.3285  0.585  0.315  0.0 |  6.067446   5.421533   6.057752  0.0 | 0.308293  0.584994  0.314997  0.0 |  6.067446   5.421533   6.057752  0.0 | 0.0  0.0  0.0  0.0 | 15.500043  15.500043  15.500043  0.0 | 0.000016  0.000016  0.000016  0.0 |  6.64 | 0.0 |  0.27656 |  5.623815 |  0.0 |  0.01346 |    0.549 | 0.406985 | 0.0 | 500.0 | 0.0 |      -57.21 |  0.0 |  -6.802647 |   4.00067 |  8.00239 | 0.926203 |  0.684876   2.684204    2.60292  0.0 | -3.000251  -3.000252  -3.000252  0.0 | nan |  52.038242 | nan |  576.436666 | 2.633381 | 1500.886779 | 0.926203 |
    | 2017-02-26 00:00:00 |  7.1 |  5.3 | 0.0 | 10.0 | 0.0 |  4.24 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |  3.990605   3.797404    3.98928  0.0 |   0.2519  0.467998  0.251998  0.0 |  3.990605   3.797404    3.98928  0.0 | 0.0  0.0  0.0  0.0 |      35.5       35.5       35.5  0.0 | 0.000037  0.000037  0.000037  0.0 |  4.24 | 0.0 | 0.139369 |  3.807117 |  0.0 | 0.007261 |   0.4392 | 0.325593 | 0.0 | 500.0 | 0.0 |  -64.760531 |  0.0 |  -7.279615 |  4.436518 | 8.006113 | 0.926633 |  0.682371   2.658802   2.601641  0.0 | -3.000288  -3.000288  -3.000288  0.0 | nan |   44.62652 | nan |  511.676135 |  2.00398 | 1500.740299 | 0.926633 |
    | 2017-02-27 00:00:00 |  9.4 |  0.7 | 0.0 | 10.0 | 0.0 |  0.56 | 0.8541  1.521  0.819  1.4274 | 0.8541  1.521  0.819  0.0 |  0.424023   0.206937   0.404317  0.0 | 0.695482   1.52066  0.818961  0.0 |  0.424023   0.206937   0.404317  0.0 | 0.0  0.0  0.0  0.0 |      47.0       47.0       47.0  0.0 | 0.000048  0.000048  0.000048  0.0 |  0.56 | 0.0 | 0.009705 |  0.348241 |  0.0 | 0.105777 |   1.4274 | 1.058119 | 0.0 | 500.0 | 0.0 |  -14.723343 |  0.0 |  -4.556863 |  1.951674 | 7.994099 | 0.925243 |  0.122867   1.491205   1.938363  0.0 | -3.000336  -3.000337  -3.000337  0.0 | nan |  40.165729 | nan |  496.952792 | 0.400546 | 1494.842338 | 0.925243 |
    | 2017-02-28 00:00:00 |  4.6 |  2.7 | 0.0 | 10.0 | 0.0 |  2.16 | 0.5913  1.053  0.567  0.9882 | 0.5913  1.053  0.567  0.0 |  1.504371   0.475372   1.340728  0.0 | 0.453718  1.052728  0.566963  0.0 |  1.504371   0.475372   1.340728  0.0 | 0.0  0.0  0.0  0.0 |      23.0       23.0       23.0  0.0 | 0.000024  0.000024  0.000024  0.0 |  2.16 | 0.0 | 0.026054 |  1.155882 |  0.0 | 0.091745 |   0.9882 | 0.732534 | 0.0 | 500.0 | 0.0 |  -24.398353 |  0.0 |  -3.548412 |  0.967053 | 7.921822 | 0.916877 |  0.324777   2.123105   2.190672  0.0 |  -3.00036  -3.000361  -3.000361  0.0 | nan |  36.683008 | nan |   472.55444 | 0.589375 | 1490.823671 | 0.916877 |
    | 2017-03-01 00:00:00 |  3.7 |  1.6 | 0.0 | 10.0 | 0.0 |  1.28 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |  0.950551   0.496994   0.875855  0.0 | 0.384044  0.818947  0.440979  0.0 |  0.950551   0.496994   0.875855  0.0 | 0.0  0.0  0.0  0.0 | 18.500006  18.500006  18.500006  0.0 | 0.000019  0.000019  0.000019  0.0 |  1.28 | 0.0 | 0.015043 |  0.793494 |  0.0 | 0.067343 |   0.7938 | 0.585077 | 0.0 | 500.0 | 0.0 |  -30.731545 |  0.0 |  -3.357869 |  0.963106 | 7.941661 | 0.919174 |  0.270182   2.087164   2.153838  0.0 | -3.000379   -3.00038   -3.00038  0.0 | nan |  33.377438 | nan |  441.822895 | 0.419764 | 1493.336991 | 0.919174 |
    | 2017-03-02 00:00:00 |  4.7 |  2.5 | 0.0 | 10.0 | 0.0 |   2.0 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |  1.532261   1.008675   1.460322  0.0 | 0.393813  0.818966  0.440983  0.0 |  1.532261   1.008675   1.460322  0.0 | 0.0  0.0  0.0  0.0 |      23.5       23.5       23.5  0.0 | 0.000024  0.000024  0.000024  0.0 |   2.0 | 0.0 | 0.020967 |  1.349145 |  0.0 | 0.060831 |   0.7938 | 0.585083 | 0.0 | 500.0 | 0.0 |  -38.506105 |  0.0 |  -3.405203 |  1.116525 | 7.980625 | 0.923684 |  0.344108   2.259522   2.252532  0.0 | -3.000403  -3.000404  -3.000404  0.0 | nan |  30.012098 | nan |   403.31679 | 0.652384 | 1500.052177 | 0.923684 |
    | 2017-03-03 00:00:00 |  5.9 |  0.6 | 0.0 | 10.0 | 0.0 |  0.48 | 0.7623  1.287  0.693  1.2474 | 0.7623  1.287  0.693  0.0 |  0.288959   0.089034   0.242267  0.0 |  0.50588  1.286378  0.692918  0.0 |  0.288959   0.089034   0.242267  0.0 | 0.0  0.0  0.0  0.0 |      29.5       29.5       29.5  0.0 |  0.00003   0.00003   0.00003  0.0 |  0.48 | 0.0 | 0.002808 |  0.221815 |  0.0 | 0.171053 |   1.2474 | 0.919375 | 0.0 | 500.0 | 0.0 |  -26.576924 |  0.0 |  -3.097933 |  0.702183 |  7.99349 | 0.925172 |  0.029269    1.36411   1.797347  0.0 | -3.000434  -3.000434  -3.000434  0.0 | nan |   27.08241 | nan |  376.739865 | 0.172016 |  1497.39837 | 0.925172 |
    | 2017-03-04 00:00:00 |  7.7 |  0.2 | 0.0 | 10.0 | 0.0 |  0.16 |  0.693   1.17   0.63   1.134 |  0.693   1.17   0.63  0.0 |  0.056014   0.000733   0.020497  0.0 | 0.287017  1.128526  0.629375  0.0 |  0.056014   0.000733   0.020497  0.0 | 0.0  0.0  0.0  0.0 |      38.5       38.5       38.5  0.0 |  0.00004   0.00004   0.00004  0.0 |  0.16 | 0.0 |  0.00039 |  0.035566 |  0.0 |  0.28432 |    1.134 | 0.835682 | 0.0 | 500.0 | 0.0 |  -23.262945 |  0.0 |  -2.675598 |  0.170167 | 7.946202 | 0.919699 | -0.153762   0.394851   1.307475  0.0 | -3.000473  -3.000474  -3.000474  0.0 | nan |  24.690742 | nan |   353.47692 | 0.037416 | 1489.458583 | 0.919699 |
    | 2017-03-05 00:00:00 |  6.3 |  1.7 | 0.0 | 10.0 | 0.0 |  1.36 | 0.5544  0.936  0.504  0.9072 | 0.5544  0.936  0.504  0.0 |  0.699695   0.001278   0.261984  0.0 | 0.321352  0.881396  0.503639  0.0 |  0.699695   0.001278   0.261984  0.0 | 0.0  0.0  0.0  0.0 |      31.5       31.5       31.5  0.0 | 0.000032  0.000032  0.000032  0.0 |  1.36 | 0.0 | 0.003937 |  0.442888 |  0.0 | 0.173473 |   0.9072 | 0.668589 | 0.0 | 500.0 | 0.0 |    -24.4205 |  0.0 |  -2.482883 |  0.261788 | 7.931138 | 0.917956 |  0.185191   0.872178   1.901852  0.0 | -3.000506  -3.000506  -3.000506  0.0 | nan |  22.377396 | nan |   329.05642 | 0.218516 | 1496.686974 | 0.917956 |
    | 2017-03-06 00:00:00 |  3.7 |  0.3 | 0.0 | 10.0 | 0.0 |  0.24 | 0.4158  0.702  0.378  0.6804 | 0.4158  0.702  0.378  0.0 |  0.131638   0.000231   0.068103  0.0 | 0.255659  0.664309  0.377897  0.0 |  0.131638   0.000231   0.068103  0.0 | 0.0  0.0  0.0  0.0 | 18.500006  18.500006  18.500006  0.0 | 0.000019  0.000019  0.000019  0.0 |  0.24 | 0.0 | 0.000611 |  0.085331 |  0.0 | 0.119263 |   0.6804 | 0.501467 | 0.0 | 500.0 | 0.0 |  -29.109958 |  0.0 |  -2.464377 |  0.239262 | 7.993808 | 0.925209 |  0.037894   0.447637   1.695851  0.0 | -3.000525  -3.000525  -3.000525  0.0 | nan |  20.031671 | nan |  299.946461 | 0.064585 | 1499.600994 | 0.925209 |
    | 2017-03-07 00:00:00 |  1.6 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |       0.0        0.0        0.0  0.0 |  0.20305  0.529438  0.440409  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  8.006707   8.006707   8.006707  0.0 | 0.000008  0.000008  0.000008  0.0 |   0.0 | 0.0 |      0.0 |  0.000008 |  0.0 | 0.284415 |   0.7938 | 0.584938 | 0.0 | 500.0 | 0.0 |  -26.041778 |  0.0 |  -2.311714 |  0.055803 | 7.997079 | 0.925588 | -0.165156    -0.0818   1.255442  0.0 | -3.000533  -3.000533  -3.000533  0.0 | nan |  18.004372 | nan |  273.904683 |  0.00879 | 1499.741013 | 0.925588 |
    | 2017-03-08 00:00:00 |  4.0 |  1.8 | 0.0 | 10.0 | 0.0 |  1.44 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |  0.767704   0.000465   0.301922  0.0 | 0.289598  0.646694  0.440662  0.0 |  0.767704   0.000465   0.301922  0.0 | 0.0  0.0  0.0  0.0 | 20.000002  20.000002  20.000002  0.0 | 0.000021  0.000021  0.000021  0.0 |  1.44 | 0.0 | 0.002214 |  0.488982 |  0.0 | 0.187681 |   0.7938 | 0.585007 | 0.0 | 500.0 | 0.0 |  -37.249081 |  0.0 |  -2.550243 |  0.259203 | 8.001458 | 0.926095 |  0.217542   0.711041   1.952859  0.0 | -3.000553  -3.000554  -3.000554  0.0 | nan |  15.639597 | nan |  236.655602 |  0.23857 | 1500.549574 | 0.926095 |
    | 2017-03-09 00:00:00 |  5.6 |  8.9 | 0.0 | 10.0 | 0.0 |  7.12 | 0.3465  0.585  0.315   0.567 | 0.3465  0.585  0.315  0.0 |  6.357943   4.564431    6.15922  0.0 | 0.317366  0.584263  0.314994  0.0 |  6.357943   4.564431    6.15922  0.0 | 0.0  0.0  0.0  0.0 |      28.0       28.0       28.0  0.0 | 0.000029  0.000029  0.000029  0.0 |  7.12 | 0.0 | 0.015029 |   5.78652 |  0.0 |  0.01966 |    0.567 | 0.417934 | 0.0 | 500.0 | 0.0 | -129.250919 |  0.0 |  -5.886208 |   3.30999 | 8.029164 | 0.929301 |  0.662233   2.682347   2.598645  0.0 | -3.000582  -3.000583  -3.000583  0.0 | nan |    9.75802 | nan |  107.404683 |   2.7151 | 1508.252104 | 0.929301 |
    | 2017-03-10 00:00:00 |  5.8 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 | 0.5544  0.936  0.504  0.9072 | 0.5544  0.936  0.504  0.0 |       0.0        0.0        0.0  0.0 | 0.473397   0.93593  0.503987  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      29.0       29.0       29.0  0.0 |  0.00003   0.00003   0.00003  0.0 |   0.0 | 0.0 |      0.0 |   0.00003 |  0.0 | 0.054005 |   0.9072 | 0.668688 | 0.0 | 500.0 | 0.0 |   -88.39727 |  0.0 |  -4.705385 |  2.347669 | 8.073352 | 0.934416 |  0.188836   1.746417   2.094657  0.0 | -3.000612  -3.000612  -3.000612  0.0 | nan |   5.106639 | nan |   19.007413 | 0.367461 | 1511.205616 | 0.934416 |
    | 2017-03-11 00:00:00 |  5.7 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 |  0.693   1.17   0.63   1.134 |  0.693   1.17   0.63  0.0 |       0.0        0.0        0.0  0.0 | 0.342784  1.155482  0.629754  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      28.5       28.5       28.5  0.0 | 0.000029  0.000029  0.000029  0.0 |   0.0 | 0.0 |      0.0 |  0.000029 |  0.0 |  0.23823 |    1.134 | 0.835784 | 0.0 | 500.0 | 0.0 |  -19.312172 |  0.0 |  -2.329407 |  0.317599 | 8.122932 | 0.940154 | -0.153948   0.590935   1.464903  0.0 | -3.000641  -3.000642  -3.000642  0.0 | nan |   3.015462 | nan |    -0.30476 | 0.049891 | 1516.760533 | 0.940154 |
    | 2017-03-12 00:00:00 |  4.6 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 | 0.8316  1.404  0.756  1.3608 | 0.8316  1.404  0.756  0.0 |       0.0        0.0        0.0  0.0 | 0.194399   0.81325  0.748162  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      23.0       23.0       23.0  0.0 | 0.000024  0.000024  0.000024  0.0 |   0.0 | 0.0 |      0.0 |  0.000024 |  0.0 |   0.6215 |   1.3608 | 1.002101 | 0.0 | 500.0 | 0.0 |   -0.325375 |  0.0 |  -2.140023 |  0.043137 |   8.1329 | 0.941308 | -0.348347  -0.222315   0.716742  0.0 | -3.000665  -3.000665  -3.000665  0.0 | nan |   1.496939 | nan |   -0.630134 | 0.006778 | 1516.493407 | 0.941308 |
    | 2017-03-13 00:00:00 |  4.2 |  0.9 | 0.0 | 10.0 | 0.0 |  0.72 | 0.6237  1.053  0.567  1.0206 | 0.6237  1.053  0.567  0.0 |    0.1985   0.000023   0.002958  0.0 | 0.207048  0.453034  0.552683  0.0 |    0.1985   0.000023   0.002958  0.0 | 0.0  0.0  0.0  0.0 | 21.000001  21.000001  21.000001  0.0 | 0.000022  0.000022  0.000022  0.0 |  0.72 | 0.0 | 0.000001 |  0.119424 |  0.0 | 0.477594 |   1.0206 | 0.750749 | 0.0 | 500.0 | 0.0 |   -0.250094 |  0.0 |  -2.167611 |  0.065913 | 8.134747 | 0.941522 | -0.033896   0.044627   0.881101  0.0 | -3.000687  -3.000687  -3.000687  0.0 | nan |  -0.193079 | nan |   -0.880229 | 0.060289 | 1517.093562 | 0.941522 |
    | 2017-03-14 00:00:00 |  7.4 |  0.1 | 0.0 | 10.0 | 0.0 |  0.08 | 0.6237  1.053  0.567  1.0206 | 0.6237  1.053  0.567  0.0 |  0.023892   0.000002   0.000192  0.0 | 0.223715  0.377517  0.535147  0.0 |  0.023892   0.000002   0.000192  0.0 | 0.0  0.0  0.0  0.0 |      37.0       37.0       37.0  0.0 | 0.000038  0.000038  0.000038  0.0 |  0.08 | 0.0 |      0.0 |  0.014393 |  0.0 | 0.491656 |   1.0206 | 0.749032 | 0.0 | 500.0 | 0.0 |   -0.758081 |  0.0 |  -2.160047 |  0.060766 | 8.136398 | 0.941713 | -0.201502  -0.252892   0.425762  0.0 | -3.000725  -3.000725  -3.000725  0.0 | nan |  -1.861469 | nan |   -1.638309 | 0.013915 | 1517.052573 | 0.941713 |
    | 2017-03-15 00:00:00 |  6.3 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 | 0.8316  1.404  0.756  1.3608 | 0.8316  1.404  0.756  0.0 |       0.0        0.0        0.0  0.0 | 0.171645  0.216554  0.491468  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      31.5       31.5       31.5  0.0 | 0.000032  0.000032  0.000032  0.0 |   0.0 | 0.0 |      0.0 |  0.000032 |  0.0 | 0.835519 |   1.3608 | 0.976902 | 0.0 | 500.0 | 0.0 |    -1.23221 |  0.0 |  -2.078392 |  0.012022 | 8.138485 | 0.941954 | -0.373147  -0.469446  -0.065706  0.0 | -3.000757  -3.000757  -3.000757  0.0 | nan |  -3.104343 | nan |   -2.870519 | 0.001926 | 1517.699468 | 0.941954 |
    | 2017-03-16 00:00:00 |  8.7 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 | 0.9702  1.638  0.882  1.5876 | 0.9702  1.638  0.882  0.0 |       0.0        0.0        0.0  0.0 | 0.116208   0.12809  0.249038  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      43.5       43.5       43.5  0.0 | 0.000045  0.000045  0.000045  0.0 |   0.0 | 0.0 |      0.0 |  0.000045 |  0.0 | 1.072297 |   1.5876 | 1.107912 | 0.0 | 500.0 | 0.0 |   -1.030082 |  0.0 |  -2.060898 |  0.001665 | 8.143034 | 0.942481 | -0.489355  -0.597536  -0.314744  0.0 | -3.000802  -3.000802  -3.000802  0.0 | nan |  -4.092945 | nan |   -3.900601 | 0.000305 | 1518.156135 | 0.942481 |
    | 2017-03-17 00:00:00 |  6.4 |  3.9 | 0.0 | 10.0 | 0.0 |  3.12 | 0.7623  1.287  0.693  1.2474 | 0.7623  1.287  0.693  0.0 |  1.818576   0.016236   0.252127  0.0 | 0.484984  0.989168   0.60256  0.0 |  1.818576   0.016236   0.252127  0.0 | 0.0  0.0  0.0  0.0 |      32.0       32.0       32.0  0.0 | 0.000033  0.000033  0.000033  0.0 |  3.12 | 0.0 |      0.0 |  1.121262 |  0.0 | 0.284068 |   1.2474 | 0.910533 | 0.0 | 500.0 | 0.0 |   -1.948576 |  0.0 |  -2.617969 |  0.551846 | 8.165185 | 0.945045 |  0.327084    1.51706    1.95057  0.0 | -3.000834  -3.000835  -3.000835  0.0 | nan |  -6.426846 | nan |   -5.849177 | 0.569721 | 1523.357544 | 0.945045 |
    | 2017-03-18 00:00:00 |  5.2 |  8.7 | 0.0 | 10.0 | 0.0 |  6.96 | 0.7623  1.287  0.693  1.2474 | 0.7623  1.287  0.693  0.0 |  6.071563   4.671479   5.753171  0.0 | 0.685968  1.286933  0.692984  0.0 |  6.071563   4.671479   5.753171  0.0 | 0.0  0.0  0.0  0.0 |      26.0       26.0       26.0  0.0 | 0.000027  0.000027  0.000027  0.0 |  6.96 | 0.0 |      0.0 |  5.619726 |  0.0 | 0.050896 |   1.2474 | 0.919458 | 0.0 | 500.0 | 0.0 |   -4.899013 |  0.0 |  -5.781779 |   3.58005 | 8.249952 | 0.954856 |  0.529553   2.518648   2.464415  0.0 | -3.000861  -3.000862  -3.000862  0.0 | nan | -12.157729 | nan |  -10.748191 | 2.609397 | 1537.018559 | 0.954856 |
    | 2017-03-19 00:00:00 |  5.1 | 26.4 | 0.0 | 10.0 | 0.0 | 21.12 | 0.3465  0.585  0.315   0.567 | 0.3465  0.585  0.315  0.0 | 20.402771  20.094596  20.410261  0.0 | 0.337475  0.584999  0.314999  0.0 | 20.402771  20.094596  20.410261  0.0 | 0.0  0.0  0.0  0.0 |      25.5       25.5       25.5  0.0 | 0.000026  0.000026  0.000026  0.0 | 21.12 | 0.0 |      0.0 | 20.311094 |  0.0 | 0.006016 |    0.567 | 0.417941 | 0.0 | 500.0 | 0.0 |  -14.082185 |  0.0 | -16.797206 | 13.714397 | 8.441139 | 0.976984 |  0.909307   2.959053   2.859155  0.0 | -3.000887  -3.000888  -3.000888  0.0 | nan | -28.948919 | nan |  -24.830376 | 9.206093 | 1566.763248 | 0.976984 |
    | 2017-03-20 00:00:00 |  8.7 | 11.5 | 0.0 | 10.0 | 0.0 |   9.2 | 0.4158  0.702  0.378  0.6804 | 0.4158  0.702  0.378  0.0 |  8.931946    8.69755   8.951424  0.0 | 0.406537  0.701998  0.377999  0.0 |  8.931946    8.69755   8.951424  0.0 | 0.0  0.0  0.0  0.0 |      43.5       43.5       43.5  0.0 | 0.000045  0.000045  0.000045  0.0 |   9.2 | 0.0 |      0.0 |   8.86362 |  0.0 | 0.006175 |   0.6804 | 0.501529 | 0.0 | 500.0 | 0.0 |  -17.635243 |  0.0 | -16.311367 | 13.007069 | 8.530553 | 0.987333 |  0.770823   2.759505   2.729733  0.0 | -3.000932  -3.000932  -3.000932  0.0 | nan | -45.254111 | nan |  -42.465619 | 5.062644 | 1566.770342 | 0.987333 |
    | 2017-03-21 00:00:00 |  6.2 |  0.9 | 0.0 | 10.0 | 0.0 |  0.72 | 1.0395  1.755  0.945   1.701 | 1.0395  1.755  0.945  0.0 |  0.554102   0.275967   0.548112  0.0 | 0.842841  1.754558  0.944963  0.0 |  0.554102   0.275967   0.548112  0.0 | 0.0  0.0  0.0  0.0 |      31.0       31.0       31.0  0.0 | 0.000032  0.000032  0.000032  0.0 |  0.72 | 0.0 |      0.0 |  0.470094 |  0.0 | 0.131237 |    1.701 | 1.253808 | 0.0 | 500.0 | 0.0 |   -8.813486 |  0.0 |  -7.046507 |  4.674931 | 8.500201 |  0.98382 |   0.09388   1.448981   1.956658  0.0 | -3.000964  -3.000964  -3.000964  0.0 | nan | -52.169381 | nan |  -51.279105 | 0.857807 | 1559.099922 |  0.98382 |
    | 2017-03-22 00:00:00 |  5.9 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 |  1.386   2.34   1.26   2.268 |  1.386   2.34   1.26  0.0 |       0.0        0.0        0.0  0.0 | 0.443363  1.687857  1.251508  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      29.5       29.5       29.5  0.0 |  0.00003   0.00003   0.00003  0.0 |   0.0 | 0.0 |      0.0 |   0.00003 |  0.0 | 0.845708 |    2.268 |  1.67085 | 0.0 | 500.0 | 0.0 |   -2.389655 |  0.0 |  -2.535234 |  0.742034 | 8.470464 | 0.980378 | -0.349483  -0.238877    0.70515  0.0 | -3.000994  -3.000994  -3.000994  0.0 | nan | -53.858907 | nan |  -53.668759 | 0.115804 | 1557.864538 | 0.980378 |
    | 2017-03-23 00:00:00 |  5.2 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 | 1.1088  1.872  1.008  1.8144 | 1.1088  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.139004  0.269691  0.748739  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      26.0       26.0       26.0  0.0 | 0.000027  0.000027  0.000027  0.0 |   0.0 | 0.0 |      0.0 |  0.000027 |  0.0 |   1.1805 |   1.8144 | 1.311889 | 0.0 | 500.0 | 0.0 |   -0.706265 |  0.0 |  -1.797319 |  0.099931 |  8.46822 | 0.980118 | -0.488488  -0.508568  -0.043589  0.0 | -3.001021  -3.001021  -3.001021  0.0 | nan | -54.475726 | nan |  -54.375024 |   0.0159 | 1558.273949 | 0.980118 |
    | 2017-03-24 00:00:00 |  5.2 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 | 1.1088  1.872  1.008  1.8144 | 1.1088  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.087812  0.124929  0.285484  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      26.0       26.0       26.0  0.0 | 0.000027  0.000027  0.000027  0.0 |   0.0 | 0.0 |      0.0 |  0.000027 |  0.0 | 1.262874 |   1.8144 | 1.266483 | 0.0 | 500.0 | 0.0 |   -0.457858 |  0.0 |    -1.7031 |  0.013728 | 8.470891 | 0.980427 | -0.576299  -0.633497  -0.329073  0.0 | -3.001047  -3.001048  -3.001048  0.0 | nan | -54.915952 | nan |  -54.832883 | 0.002198 | 1558.480963 | 0.980427 |
    | 2017-03-25 00:00:00 |  5.9 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 | 0.8316  1.404  0.756  1.3608 | 0.8316  1.404  0.756  0.0 |       0.0        0.0        0.0  0.0 | 0.049329  0.062934  0.109594  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      29.5       29.5       29.5  0.0 |  0.00003   0.00003   0.00003  0.0 |   0.0 | 0.0 |      0.0 |   0.00003 |  0.0 | 0.968428 |   1.3608 | 0.939618 | 0.0 | 500.0 | 0.0 |   -0.660248 |  0.0 |  -1.685812 |  0.001914 | 8.474692 | 0.980867 | -0.625629  -0.696431  -0.438667  0.0 | -3.001078  -3.001078  -3.001078  0.0 | nan | -55.633336 | nan |  -55.493131 | 0.000315 | 1559.135037 | 0.980867 |
    | 2017-03-26 00:00:00 |  6.7 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 | 0.9009  1.521  0.819  1.4742 | 0.9009  1.521  0.819  0.0 |       0.0        0.0        0.0  0.0 | 0.043675  0.052857  0.081106  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      33.5       33.5       33.5  0.0 | 0.000034  0.000034  0.000034  0.0 |   0.0 | 0.0 |      0.0 |  0.000034 |  0.0 | 1.060748 |   1.4742 | 1.014233 | 0.0 | 500.0 | 0.0 |   -0.636106 |  0.0 |  -1.675527 |   0.00029 |  8.47968 | 0.981444 | -0.669304  -0.749288  -0.519773  0.0 | -3.001112  -3.001112  -3.001112  0.0 | nan | -56.248115 | nan |  -56.129237 | 0.000059 | 1559.800307 | 0.981444 |
    | 2017-03-27 00:00:00 |  7.0 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 | 1.1088  1.872  1.008  1.8144 | 1.1088  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.044308  0.051568  0.072372  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      35.0       35.0       35.0  0.0 | 0.000036  0.000036  0.000036  0.0 |   0.0 | 0.0 |      0.0 |  0.000036 |  0.0 | 1.316329 |   1.8144 | 1.245597 | 0.0 | 500.0 | 0.0 |   -0.404731 |  0.0 |  -1.672468 |  0.000071 |  8.48371 | 0.981911 | -0.713612  -0.800856  -0.592145  0.0 | -3.001148  -3.001148  -3.001148  0.0 | nan | -56.604254 | nan |  -56.533968 | 0.000024 | 1560.048012 | 0.981911 |
    | 2017-03-28 00:00:00 |  8.3 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 | 1.3167  2.223  1.197  2.1546 | 1.3167  2.223  1.197  0.0 |       0.0        0.0        0.0  0.0 | 0.043336  0.048901  0.064103  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      41.5       41.5       41.5  0.0 | 0.000043  0.000043  0.000043  0.0 |   0.0 | 0.0 |      0.0 |  0.000043 |  0.0 | 1.573439 |   2.1546 | 1.477006 | 0.0 | 500.0 | 0.0 |    -0.14367 |  0.0 |  -1.664506 |  0.000044 | 8.485696 | 0.982141 | -0.756948  -0.849756  -0.656248  0.0 |  -3.00119  -3.001191  -3.001191  0.0 | nan |  -56.69532 | nan |  -56.677638 | 0.000022 |  1560.20608 | 0.982141 |
    | 2017-03-29 00:00:00 |  9.0 |  1.5 | 0.0 | 10.0 | 0.0 |   1.2 | 0.5544  0.936  0.504  0.9072 | 0.5544  0.936  0.504  0.0 |  0.262511   0.000023   0.000146  0.0 | 0.145361  0.238849  0.204304  0.0 |  0.262511   0.000023   0.000146  0.0 | 0.0  0.0  0.0  0.0 |      45.0       45.0       45.0  0.0 | 0.000046  0.000046  0.000046  0.0 |   1.2 | 0.0 |      0.0 |  0.157574 |  0.0 | 0.505022 |   0.9072 | 0.639289 | 0.0 | 500.0 | 0.0 |   -0.984411 |  0.0 |  -1.760531 |  0.064376 | 8.490732 | 0.982724 |   0.03518   0.111372   0.339302  0.0 | -3.001236  -3.001237  -3.001237  0.0 | nan | -57.950829 | nan |  -57.662049 |  0.09322 | 1561.477282 | 0.982724 |
    | 2017-03-30 00:00:00 | 12.4 |  0.3 | 0.0 | 10.0 | 0.0 |  0.24 | 1.0395  1.755  0.945   1.701 | 1.0395  1.755  0.945  0.0 |  0.076819   0.000006    0.00005  0.0 | 0.396454   0.62374  0.601591  0.0 |  0.076819   0.000006    0.00005  0.0 | 0.0  0.0  0.0  0.0 |      62.0       62.0       62.0  0.0 | 0.000064  0.000064  0.000064  0.0 |  0.24 | 0.0 |      0.0 |  0.046162 |  0.0 |   0.8057 |    1.701 | 1.220098 | 0.0 | 500.0 | 0.0 |   -1.088514 |  0.0 |  -1.771559 |  0.108784 | 8.500021 | 0.983799 | -0.198093  -0.272374  -0.022339  0.0 |   -3.0013    -3.0013    -3.0013  0.0 | nan | -58.916688 | nan |  -58.750562 | 0.030597 | 1562.219896 | 0.983799 |
    | 2017-03-31 00:00:00 | 15.0 |  0.2 | 0.0 | 10.0 | 0.0 |  0.16 | 1.8711  3.159  1.701  3.0618 | 1.8711  3.159  1.701  0.0 |   0.02362   0.000001   0.000007  0.0 | 0.346755  0.403415  0.470442  0.0 |   0.02362   0.000001   0.000007  0.0 | 0.0  0.0  0.0  0.0 |      75.0       75.0       75.0  0.0 | 0.000077  0.000077  0.000077  0.0 |  0.16 | 0.0 |      0.0 |   0.01425 |  0.0 | 1.934558 |   3.0618 | 2.136116 | 0.0 | 500.0 | 0.0 |    0.026639 |  0.0 |  -1.670686 |  0.035209 | 8.502756 | 0.984115 | -0.408469   -0.51579  -0.332788  0.0 | -3.001377  -3.001377  -3.001377  0.0 | nan | -58.652816 | nan |  -58.723923 | 0.009638 | 1562.228248 | 0.984115 |
    | 2017-04-01 00:00:00 | 11.8 |  4.5 | 0.0 | 10.0 | 0.0 |   3.6 | 1.2825  1.755  0.945   1.728 | 1.2825  1.755  0.945  0.0 |  1.991769   0.031289   0.414313  0.0 | 0.874718  1.427468  0.831559  0.0 |  1.991769   0.031289   0.414313  0.0 | 0.0  0.0  0.0  0.0 |      59.0       59.0       59.0  0.0 |  0.00006   0.00006   0.00006  0.0 |   3.6 | 0.0 |      0.0 |   1.24594 |  0.0 | 0.380993 |    1.728 | 1.386098 | 0.0 | 500.0 | 0.0 |   -1.376047 |  0.0 |   -2.34437 |  0.628185 | 8.515631 | 0.985605 |  0.325045   1.625453   2.021341  0.0 | -3.001437  -3.001438  -3.001438  0.0 | nan | -60.616194 | nan |   -60.09997 | 0.627393 | 1565.713055 | 0.985605 |
    | 2017-04-02 00:00:00 |  9.4 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 |  1.368  1.872  1.008  1.8432 |  1.368  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.610023  1.639364  1.005966  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      47.0       47.0       47.0  0.0 | 0.000048  0.000048  0.000048  0.0 |   0.0 | 0.0 |      0.0 |  0.000048 |  0.0 | 0.582807 |   1.8432 | 1.490151 | 0.0 | 500.0 | 0.0 |   -1.950523 |  0.0 |   -2.23164 |  0.542349 | 8.529724 | 0.987237 | -0.284978  -0.013911   1.015375  0.0 | -3.001485  -3.001486  -3.001486  0.0 | nan | -62.265027 | nan |  -62.050493 | 0.085092 | 1565.543453 | 0.987237 |
    | 2017-04-03 00:00:00 |  8.1 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 |   1.71   2.34   1.26   2.304 |   1.71   2.34   1.26  0.0 |       0.0        0.0        0.0  0.0 | 0.227067  0.478502  1.028598  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      40.5       40.5       40.5  0.0 | 0.000041  0.000041  0.000041  0.0 |   0.0 | 0.0 |      0.0 |  0.000041 |  0.0 | 1.608966 |    2.304 | 1.840186 | 0.0 | 500.0 | 0.0 |   -0.311128 |  0.0 |  -1.696484 |  0.073518 | 8.530078 | 0.987278 | -0.512045  -0.492413  -0.013223  0.0 | -3.001527  -3.001527  -3.001527  0.0 | nan | -62.352544 | nan |  -62.361621 | 0.011615 | 1565.522981 | 0.987278 |
    | 2017-04-04 00:00:00 |  7.9 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 | 1.7955  2.457  1.323  2.4192 | 1.7955  2.457  1.323  0.0 |       0.0        0.0        0.0  0.0 | 0.119557  0.161202  0.357479  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      39.5       39.5       39.5  0.0 |  0.00004   0.00004   0.00004  0.0 |   0.0 | 0.0 |      0.0 |   0.00004 |  0.0 |  1.88238 |   2.4192 | 1.861369 | 0.0 | 500.0 | 0.0 |    0.207253 |  0.0 |  -1.632475 |  0.010045 | 8.528991 | 0.987152 | -0.631602  -0.653615  -0.370703  0.0 | -3.001567  -3.001568  -3.001568  0.0 | nan | -62.102639 | nan |  -62.154368 | 0.001611 | 1565.154293 | 0.987152 |
    | 2017-04-05 00:00:00 |  7.5 |  0.4 | 0.0 | 10.0 | 0.0 |  0.32 | 1.4535  1.989  1.071  1.9584 | 1.4535  1.989  1.071  0.0 |  0.014458   0.000001   0.000008  0.0 |  0.11982  0.145604  0.209947  0.0 |  0.014458   0.000001   0.000008  0.0 | 0.0  0.0  0.0  0.0 |      37.5       37.5       37.5  0.0 | 0.000038  0.000038  0.000038  0.0 |  0.32 | 0.0 |      0.0 |  0.008714 |  0.0 |  1.50344 |   1.9584 | 1.499039 | 0.0 | 500.0 | 0.0 |   -0.046755 |  0.0 |  -1.633392 |  0.005869 | 8.528018 | 0.987039 | -0.445879   -0.47922  -0.260658  0.0 | -3.001606  -3.001606  -3.001606  0.0 | nan | -62.232591 | nan |  -62.201123 | 0.004456 | 1565.370006 | 0.987039 |
    | 2017-04-06 00:00:00 |  7.2 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 | 1.4535  1.989  1.071  1.9584 | 1.4535  1.989  1.071  0.0 |       0.0        0.0        0.0  0.0 | 0.126183  0.143877  0.175605  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      36.0       36.0       36.0  0.0 | 0.000037  0.000037  0.000037  0.0 |   0.0 | 0.0 |      0.0 |  0.000037 |  0.0 | 1.499774 |   1.9584 | 1.495674 | 0.0 | 500.0 | 0.0 |   -0.138086 |  0.0 |  -1.621338 |  0.003857 | 8.528824 | 0.987132 | -0.572062  -0.623097  -0.436263  0.0 | -3.001642  -3.001643  -3.001643  0.0 | nan | -62.354154 | nan |  -62.339209 | 0.000636 | 1565.658372 | 0.987132 |
    | 2017-04-07 00:00:00 |  8.1 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 |  0.684  0.936  0.504  0.9216 |  0.684  0.936  0.504  0.0 |       0.0        0.0        0.0  0.0 | 0.042006  0.045676  0.053523  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      40.5       40.5       40.5  0.0 | 0.000041  0.000041  0.000041  0.0 |   0.0 | 0.0 |      0.0 |  0.000041 |  0.0 | 0.724701 |   0.9216 | 0.700992 | 0.0 | 500.0 | 0.0 |   -0.744902 |  0.0 |  -1.632209 |  0.000573 | 8.532009 | 0.987501 | -0.614068  -0.668774  -0.489786  0.0 | -3.001684  -3.001684  -3.001684  0.0 | nan | -63.261662 | nan |  -63.084111 | 0.000105 | 1566.184002 | 0.987501 |
    | 2017-04-08 00:00:00 |  8.6 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 | 1.1115  1.521  0.819  1.4976 | 1.1115  1.521  0.819  0.0 |       0.0        0.0        0.0  0.0 | 0.055212  0.058911  0.067506  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      43.0       43.0       43.0  0.0 | 0.000044  0.000044  0.000044  0.0 |   0.0 | 0.0 |      0.0 |  0.000044 |  0.0 | 1.191442 |   1.4976 | 1.137205 | 0.0 | 500.0 | 0.0 |   -0.514108 |  0.0 |  -1.619655 |  0.000115 | 8.537344 | 0.988119 |  -0.66928  -0.727685  -0.557292  0.0 | -3.001728  -3.001728  -3.001728  0.0 | nan | -63.689874 | nan |  -63.598219 | 0.000033 |  1566.39807 | 0.988119 |
    | 2017-04-09 00:00:00 | 10.5 |  0.0 | 0.0 | 10.0 | 0.0 |   0.0 | 2.1375  2.925  1.575    2.88 | 2.1375  2.925  1.575  0.0 |       0.0        0.0        0.0  0.0 |  0.07886  0.082517  0.092024  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      52.5       52.5       52.5  0.0 | 0.000054  0.000054  0.000054  0.0 |   0.0 | 0.0 |      0.0 |  0.000054 |  0.0 | 2.319702 |     2.88 | 2.183225 | 0.0 | 500.0 | 0.0 |    0.487713 |  0.0 |  -1.597736 |  0.000059 | 8.537439 |  0.98813 |  -0.74814  -0.810202  -0.649315  0.0 | -3.001781  -3.001782  -3.001782  0.0 | nan | -62.967909 | nan |  -63.110506 | 0.000028 | 1566.188861 |  0.98813 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _wland_gd_elevated_regions:

elevated regions
________________

>>> inputs.fxs.series = 0.0
>>> er(conifer=True, default=False)
>>> ac(100.0)
>>> cwe(400.0)
>>> cge(200000.0)


.. integration-test::

    >>> test("wland_wag_elevated_regions",
    ...      axis1=(fluxes.pc, fluxes.fgse, fluxes.fgs),
    ...      axis2=(states.hge, states.dg))
    |                date |    t |    p | fxg | fxs | dhs |    pc |                           pe |                       pet |                                   tf |                                ei |                                   rf |                 sf |                                   pm |                                am |    ps |      pve |       pv |        pq |     etve |      etv |       es |       et |  gr | fxs | fxg |         cdg |     fgse |       fgs |       fqs |       rh |        r |                                   ic |                                   sp |        dve |         dv |        hge |          dg |       hq |          hs |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 00:00:00 | -2.8 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.3942  0.702  0.378  0.6588 | 0.3942  0.702  0.378  0.0 |       0.0        0.0        0.0  0.0 |      0.0  0.000001       0.0  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  0.000117   0.000117   0.000117  0.0 |      0.0       0.0       0.0  0.0 |   0.0 |      0.0 |      0.0 |       0.0 | 0.698121 | 0.392069 | 0.003375 | 0.435852 | 0.0 | 0.0 | 0.0 | -270.544309 | 3.434919 |  0.050701 |       0.0 |      0.0 |      0.0 |      -3.0  -3.000001       -3.0  0.0 |      -3.0       -3.0       -3.0  0.0 | 140.698121 |  138.72531 | 892.103635 | 1329.455691 |      0.0 |   -0.512767 |      0.0 |
    | 2017-02-11 00:00:00 | -1.5 |  0.4 | 0.0 | 0.0 | 0.0 |  0.32 | 0.5256  0.936  0.504  0.8784 | 0.5256  0.936  0.504  0.0 |  0.000001        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  0.009388   0.009388   0.009388  0.0 |      0.0       0.0       0.0  0.0 |  0.32 |      0.0 |      0.0 |       0.0 | 0.930742 | 0.522801 | 0.778445 | 0.596616 | 0.0 | 0.0 | 0.0 |  -38.445642 | 2.379587 |  0.197561 |       0.0 | 0.000501 | 0.000058 | -2.680002  -2.680003  -2.680002  0.0 |      -3.0       -3.0       -3.0  0.0 | 141.628863 | 138.255879 | 886.633321 | 1291.010049 |      0.0 |     4.81201 | 0.000058 |
    | 2017-02-12 00:00:00 | -0.9 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.4599  0.819  0.441  0.7686 | 0.4599  0.819  0.441  0.0 |       0.0        0.0        0.0  0.0 | 0.000002  0.000004  0.000002  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  0.069591   0.069591   0.069591  0.0 |      0.0       0.0       0.0  0.0 |   0.0 |      0.0 |      0.0 |       0.0 | 0.814317 | 0.457472 |   0.7686 | 0.523777 | 0.0 | 0.0 | 0.0 |    -2.50325 | 2.269693 |  0.213173 |       0.0 | 0.002876 | 0.000333 | -2.680004  -2.680007  -2.680004  0.0 |      -3.0       -3.0       -3.0  0.0 | 142.443181 | 137.791678 | 881.415636 | 1288.506799 |      0.0 |   10.166916 | 0.000333 |
    | 2017-02-13 00:00:00 | -1.6 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  0.006707   0.006707   0.006707  0.0 |      0.0       0.0       0.0  0.0 |   0.0 |      0.0 |      0.0 |       0.0 |  0.46529 | 0.261428 |   0.4392 |   0.2993 | 0.0 | 0.0 | 0.0 |     -3.3263 | 2.216165 |  0.213279 |       0.0 | 0.006447 | 0.000746 | -2.680005  -2.680009  -2.680005  0.0 |      -3.0       -3.0       -3.0  0.0 | 142.908471 | 137.158302 | 876.321004 | 1285.180498 |      0.0 |   15.675792 | 0.000746 |
    | 2017-02-14 00:00:00 | -1.3 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  0.018374   0.018374   0.018374  0.0 |      0.0       0.0       0.0  0.0 |   0.0 |      0.0 |      0.0 |       0.0 | 0.465265 | 0.261445 |   0.4392 | 0.299303 | 0.0 | 0.0 | 0.0 |   -3.347695 | 2.160384 |  0.214496 |       0.0 | 0.010866 | 0.001258 | -2.680007  -2.680011  -2.680006  0.0 |      -3.0       -3.0       -3.0  0.0 | 143.373735 | 136.554051 | 871.354605 | 1281.832804 |      0.0 |   20.999491 | 0.001258 |
    | 2017-02-15 00:00:00 |  1.7 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |   8.50479    8.50479    8.50479  0.0 | 0.000009  0.000009  0.000009  0.0 |   0.0 | 0.000003 | 0.000002 |  0.000007 | 0.465239 |  0.26146 |   0.4392 | 0.299304 | 0.0 | 0.0 | 0.0 |   -3.194658 | 2.106982 |  0.215594 |  0.000004 | 0.015793 | 0.001828 | -2.680008  -2.680013  -2.680007  0.0 | -3.000008  -3.000009  -3.000009  0.0 | 143.838972 | 135.977612 | 866.510968 | 1278.638146 | 0.000003 |   26.109324 | 0.001828 |
    | 2017-02-16 00:00:00 |  4.4 |  0.2 | 0.0 | 0.0 | 0.0 |  0.16 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |  0.000001        0.0        0.0  0.0 | 0.000002  0.000003  0.000002  0.0 |  0.000001        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 | 22.000001  22.000001  22.000001  0.0 | 0.000023  0.000023  0.000023  0.0 |  0.16 | 0.000007 | 0.000006 |  0.000018 | 0.465213 | 0.261475 |   0.4392 | 0.299306 | 0.0 | 0.0 | 0.0 |   -3.050563 | 2.055973 |  0.216482 |  0.000013 | 0.021141 | 0.002447 |  -2.52001  -2.520016  -2.520009  0.0 | -3.000031  -3.000032  -3.000032  0.0 | 144.304179 | 135.427576 | 861.784593 | 1275.587583 | 0.000008 |   31.138239 | 0.002447 |
    | 2017-02-17 00:00:00 |  4.5 |  4.5 | 0.0 | 0.0 | 0.0 |   3.6 | 0.1971  0.351  0.189  0.3294 | 0.1971  0.351  0.189  0.0 |  0.716527   0.000865   0.002553  0.0 | 0.045816  0.103387  0.056191  0.0 |  0.716527   0.000865   0.002553  0.0 | 0.0  0.0  0.0  0.0 |      22.5       22.5       22.5  0.0 | 0.000023  0.000023  0.000023  0.0 |   3.6 | 0.000257 | 0.182771 |  0.320715 |  0.24613 |  0.15053 |   0.3294 | 0.230304 | 0.0 | 0.0 | 0.0 |  -10.528095 |  1.99724 |  0.217994 |  0.083325 | 0.029577 | 0.003423 |  0.317647   0.975732   1.021248  0.0 | -3.000054  -3.000055  -3.000055  0.0 | 144.550052 |  134.61471 | 857.193238 | 1265.059488 | 0.237397 |   43.421954 | 0.003423 |
    | 2017-02-18 00:00:00 |  3.4 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.1971  0.351  0.189  0.3294 | 0.1971  0.351  0.189  0.0 |       0.0        0.0        0.0  0.0 | 0.147898  0.341641    0.1863  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 | 17.000016  17.000016  17.000016  0.0 | 0.000018  0.000018  0.000018  0.0 |   0.0 | 0.000005 | 0.000004 |  0.000013 | 0.009303 | 0.048962 |   0.3294 | 0.243776 | 0.0 | 0.0 | 0.0 |    1.730617 | 1.940306 |  0.209106 |  0.205188 | 0.051386 | 0.005947 |  0.169749   0.634092   0.834947  0.0 | -3.000072  -3.000072  -3.000072  0.0 |  144.55935 |  133.90262 | 852.732765 | 1266.790105 | 0.032223 |   56.725169 | 0.005947 |
    | 2017-02-19 00:00:00 |  4.8 |  3.2 | 0.0 | 0.0 | 0.0 |  2.56 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |  2.023717   0.452895   0.899659  0.0 | 0.221725  0.465424  0.251476  0.0 |  2.023717   0.452895   0.899659  0.0 | 0.0  0.0  0.0  0.0 |      24.0       24.0       24.0  0.0 | 0.000025  0.000025  0.000025  0.0 |  2.56 | 0.130837 | 0.505675 |  1.097433 |  0.00256 | 0.040877 |   0.4392 | 0.325426 | 0.0 | 0.0 | 0.0 |   -13.59945 |  1.85386 |  0.207864 |  0.592872 | 0.080551 | 0.009323 |  0.484308   2.275773   2.243812  0.0 | -3.000097  -3.000097  -3.000097  0.0 | 144.431072 | 132.718757 | 848.471018 | 1253.190655 | 0.536784 |   89.980346 | 0.009323 |
    | 2017-02-20 00:00:00 |  6.7 |  4.6 | 0.0 | 0.0 | 0.0 |  3.68 | 0.1971  0.351  0.189  0.3294 | 0.1971  0.351  0.189  0.0 |  3.340007   2.972394   3.208167  0.0 | 0.184109  0.350996  0.188997  0.0 |  3.340007   2.972394   3.208167  0.0 | 0.0  0.0  0.0  0.0 |      33.5       33.5       33.5  0.0 | 0.000035  0.000035  0.000035  0.0 |  3.68 | 0.853454 |  0.81941 |  2.468891 | 0.000004 |  0.01293 |   0.3294 | 0.244163 | 0.0 | 0.0 | 0.0 |  -13.538993 | 1.752408 |  0.157786 |  1.844316 | 0.207395 | 0.024004 |  0.640192   2.632382   2.526648  0.0 | -3.000131  -3.000131  -3.000131  0.0 | 143.577622 | 131.193859 | 844.442494 | 1239.651663 | 1.161359 |  177.971569 | 0.024004 |
    | 2017-02-21 00:00:00 |  5.8 |  2.3 | 0.0 | 0.0 | 0.0 |  1.84 | 0.3942  0.702  0.378  0.6588 | 0.3942  0.702  0.378  0.0 |  1.642152   1.408887    1.61893  0.0 | 0.363809  0.701991  0.377995  0.0 |  1.642152   1.408887    1.61893  0.0 | 0.0  0.0  0.0  0.0 |      29.0       29.0       29.0  0.0 |  0.00003   0.00003   0.00003  0.0 |  1.84 | 0.401398 | 0.395511 |  1.212154 | 0.000009 | 0.030253 |   0.6588 | 0.488316 | 0.0 | 0.0 | 0.0 |   -3.009681 | 1.694336 |  0.058918 |  1.702321 | 0.435411 | 0.050395 |  0.474231   2.361503   2.369724  0.0 | -3.000161  -3.000161  -3.000161  0.0 | 143.176232 | 130.040351 | 840.547469 | 1236.641982 | 0.671192 |  242.528133 | 0.050395 |
    | 2017-02-22 00:00:00 |  6.5 | 18.0 | 0.0 | 0.0 | 0.0 |  14.4 | 0.5256  0.936  0.504  0.8784 | 0.5256  0.936  0.504  0.0 |  13.58945  13.049105  13.564558  0.0 | 0.504697  0.935996  0.503997  0.0 |  13.58945  13.049105  13.564558  0.0 | 0.0  0.0  0.0  0.0 |      32.5       32.5       32.5  0.0 | 0.000033  0.000033  0.000033  0.0 |  14.4 | 3.624387 | 3.156856 | 10.443461 | 0.000004 | 0.020812 |   0.8784 | 0.651143 | 0.0 | 0.0 | 0.0 |  -58.056342 |  1.45024 | -0.129064 |  6.446151 | 0.945992 |  0.10949 |  0.780084   2.776403   2.701169  0.0 | -3.000194  -3.000195  -3.000195  0.0 | 139.551849 | 126.050123 | 837.213584 |  1178.58564 | 4.668503 |  520.817028 |  0.10949 |
    | 2017-02-23 00:00:00 |  5.0 | 19.2 | 0.0 | 0.0 | 0.0 | 15.36 | 0.3285  0.585  0.315   0.549 | 0.3285  0.585  0.315  0.0 | 14.927249  14.656122  14.931675  0.0 | 0.322035  0.584999  0.314999  0.0 | 14.927249  14.656122  14.931675  0.0 | 0.0  0.0  0.0  0.0 |      25.0       25.0       25.0  0.0 | 0.000026  0.000026  0.000026  0.0 | 15.36 | 3.878654 | 3.248148 | 11.733894 | 0.000001 |  0.00644 |    0.549 | 0.406983 | 0.0 | 0.0 | 0.0 |  -48.581603 | 1.206361 | -1.340882 | 10.671575 | 2.622039 | 0.303477 |  0.890801   2.895282   2.814495  0.0 |  -3.00022  -3.000221  -3.000221  0.0 | 135.673196 | 120.864352 | 834.440341 | 1130.004037 | 5.730822 |  888.011307 | 0.303477 |
    | 2017-02-24 00:00:00 |  3.0 |  0.4 | 0.0 | 0.0 | 0.0 |  0.32 | 0.5256  0.936  0.504  0.8784 | 0.5256  0.936  0.504  0.0 |  0.294511   0.228524   0.294809  0.0 | 0.495658  0.935979  0.503995  0.0 |  0.294511   0.228524   0.294809  0.0 | 0.0  0.0  0.0  0.0 |  15.00006   15.00006   15.00006  0.0 | 0.000015  0.000015  0.000015  0.0 |  0.32 |  0.05894 | 0.060077 |  0.221032 |  0.00002 | 0.029835 |   0.8784 | 0.651133 | 0.0 | 0.0 | 0.0 |    2.770764 | 1.098718 | -2.525314 |  5.085841 | 3.815803 | 0.441644 |  0.420632   2.050778   2.335691  0.0 | -3.000236  -3.000236  -3.000236  0.0 | 135.614276 | 117.759437 | 831.914553 |   1132.7748 | 0.866013 |  871.624746 | 0.441644 |
    | 2017-02-25 00:00:00 |  3.1 |  8.3 | 0.0 | 0.0 | 0.0 |  6.64 | 0.3285  0.585  0.315   0.549 | 0.3285  0.585  0.315  0.0 |  6.067458   5.421583    6.05777  0.0 | 0.308293  0.584994  0.314997  0.0 |  6.067458   5.421583    6.05777  0.0 | 0.0  0.0  0.0  0.0 | 15.500043  15.500043  15.500043  0.0 | 0.000016  0.000016  0.000016  0.0 |  6.64 | 1.384761 | 1.173121 |  4.753442 | 0.000006 | 0.020138 |    0.549 | 0.406957 | 0.0 | 0.0 | 0.0 |  -28.420639 | 0.999135 |  -1.87959 |  3.394135 | 3.317719 | 0.383995 |  0.684881   2.684202   2.602925  0.0 | -3.000251  -3.000252  -3.000252  0.0 | 134.229521 | 114.227297 |  829.61769 | 1104.354161 |  2.22532 |  822.882439 | 0.383995 |
    | 2017-02-26 00:00:00 |  7.1 |  5.3 | 0.0 | 0.0 | 0.0 |  4.24 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |  3.990607   3.797385   3.989283  0.0 |   0.2519  0.467998  0.251998  0.0 |  3.990607   3.797385   3.989283  0.0 | 0.0  0.0  0.0  0.0 |      35.5       35.5       35.5  0.0 | 0.000037  0.000037  0.000037  0.0 |  4.24 | 0.954758 | 0.732838 |  3.206414 | 0.000002 | 0.010865 |   0.4392 | 0.325578 | 0.0 | 0.0 | 0.0 |  -12.864034 | 0.913172 | -1.684664 |  3.742976 | 3.202265 | 0.370633 |  0.682374    2.65882   2.601643  0.0 | -3.000288  -3.000288  -3.000288  0.0 | 133.274765 | 111.364075 | 827.518444 | 1091.490126 | 1.688759 |  800.446674 | 0.370633 |
    | 2017-02-27 00:00:00 |  9.4 |  0.7 | 0.0 | 0.0 | 0.0 |  0.56 | 0.8541  1.521  0.819  1.4274 | 0.8541  1.521  0.819  0.0 |   0.42403   0.206939   0.404322  0.0 | 0.695492   1.52066  0.818961  0.0 |   0.42403   0.206939   0.404322  0.0 | 0.0  0.0  0.0  0.0 |      47.0       47.0       47.0  0.0 | 0.000048  0.000048  0.000048  0.0 |  0.56 | 0.051688 | 0.074997 |  0.296476 | 0.000338 | 0.158128 |   1.4274 | 1.057908 | 0.0 | 0.0 | 0.0 |    1.781618 | 0.881607 | -1.324326 |  1.646102 | 2.863027 | 0.331369 |  0.122852    1.49122    1.93836  0.0 | -3.000336  -3.000337  -3.000337  0.0 | 133.223416 | 109.682076 | 825.491762 | 1093.271745 | 0.339132 |  698.151766 | 0.331369 |
    | 2017-02-28 00:00:00 |  4.6 |  2.7 | 0.0 | 0.0 | 0.0 |  2.16 | 0.5913  1.053  0.567  0.9882 | 0.5913  1.053  0.567  0.0 |  1.504399   0.475642   1.340783  0.0 | 0.453724  1.052728  0.566963  0.0 |  1.504399   0.475642   1.340783  0.0 | 0.0  0.0  0.0  0.0 |      23.0       23.0       23.0  0.0 | 0.000024  0.000024  0.000024  0.0 |  2.16 | 0.118643 | 0.259073 |  0.988397 |  0.00027 | 0.137168 |   0.9882 | 0.732353 | 0.0 | 0.0 | 0.0 |    -5.10832 | 0.860119 | -0.769019 |  0.825206 | 2.264918 | 0.262143 |  0.324728    2.12285   2.190614  0.0 |  -3.00036  -3.000361  -3.000361  0.0 | 133.105044 | 108.361093 | 823.514477 | 1088.163425 | 0.502324 |  603.903566 | 0.262143 |
    | 2017-03-01 00:00:00 |  3.7 |  1.6 | 0.0 | 0.0 | 0.0 |  1.28 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |  0.950571    0.49684   0.875849  0.0 | 0.384051  0.818947  0.440979  0.0 |  0.950571    0.49684   0.875849  0.0 | 0.0  0.0  0.0  0.0 | 18.500006  18.500006  18.500006  0.0 | 0.000019  0.000019  0.000019  0.0 |  1.28 | 0.123725 | 0.160627 |  0.673505 | 0.000052 | 0.100757 |   0.7938 | 0.584945 | 0.0 | 0.0 | 0.0 |    2.576215 | 0.846306 | -0.454143 |  0.819453 |  1.87418 | 0.216919 |  0.270106   2.087062   2.153785  0.0 | -3.000379   -3.00038   -3.00038  0.0 | 132.981371 | 107.423928 | 821.568947 |  1090.73964 | 0.356376 |  537.482182 | 0.216919 |
    | 2017-03-02 00:00:00 |  4.7 |  2.5 | 0.0 | 0.0 | 0.0 |   2.0 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |  1.532176   1.008495   1.460233  0.0 | 0.393796  0.818966  0.440983  0.0 |  1.532176   1.008495   1.460233  0.0 | 0.0  0.0  0.0  0.0 |      23.5       23.5       23.5  0.0 | 0.000024  0.000024  0.000024  0.0 |   2.0 | 0.250495 | 0.254888 |   1.13982 | 0.000033 | 0.091045 |   0.7938 | 0.584964 | 0.0 | 0.0 | 0.0 |    2.186546 | 0.848978 | -0.272949 |  0.945147 | 1.606914 | 0.185985 |  0.344134   2.259601   2.252568  0.0 | -3.000403  -3.000404  -3.000404  0.0 |  132.73091 | 106.562646 | 819.617273 | 1092.926186 |  0.55105 |  496.630172 | 0.185985 |
    | 2017-03-03 00:00:00 |  5.9 |  0.6 | 0.0 | 0.0 | 0.0 |  0.48 | 0.7623  1.287  0.693  1.2474 | 0.7623  1.287  0.693  0.0 |  0.288967   0.089049   0.242284  0.0 | 0.505891  1.286378  0.692918  0.0 |  0.288967   0.089049   0.242284  0.0 | 0.0  0.0  0.0  0.0 |      29.5       29.5       29.5  0.0 |  0.00003   0.00003   0.00003  0.0 |  0.48 | 0.022084 |  0.04754 |  0.189204 | 0.000619 |  0.25569 |   1.2474 | 0.919041 | 0.0 | 0.0 | 0.0 |   17.725911 | 0.886548 | -0.183292 |  0.594047 | 1.423941 | 0.164808 |  0.029277   1.364175   1.797366  0.0 | -3.000434  -3.000434  -3.000434  0.0 | 132.709445 |  106.14423 | 817.579231 | 1110.652098 | 0.146207 |  448.385248 | 0.164808 |
    | 2017-03-04 00:00:00 |  7.7 |  0.2 | 0.0 | 0.0 | 0.0 |  0.16 |  0.693   1.17   0.63   1.134 |  0.693   1.17   0.63  0.0 |  0.056032    0.00073   0.020468  0.0 |  0.28711  1.128812  0.629377  0.0 |  0.056032    0.00073   0.020468  0.0 | 0.0  0.0  0.0  0.0 |      38.5       38.5       38.5  0.0 |  0.00004   0.00004   0.00004  0.0 |  0.16 | 0.000191 | 0.009176 |  0.030362 | 0.040993 | 0.404758 |    1.134 |   0.8351 | 0.0 | 0.0 | 0.0 |   14.250359 | 0.944846 | -0.081703 |  0.144905 | 1.186941 | 0.137377 | -0.153865   0.394633   1.307521  0.0 | -3.000473  -3.000474  -3.000474  0.0 | 132.750247 | 105.985686 | 815.407171 | 1124.902457 | 0.031663 |   392.76247 | 0.137377 |
    | 2017-03-05 00:00:00 |  6.3 |  1.7 | 0.0 | 0.0 | 0.0 |  1.36 | 0.5544  0.936  0.504  0.9072 | 0.5544  0.936  0.504  0.0 |   0.69935   0.001276   0.261852  0.0 | 0.321216  0.881368   0.50364  0.0 |   0.69935   0.001276   0.261852  0.0 | 0.0  0.0  0.0  0.0 |      31.5       31.5       31.5  0.0 | 0.000032  0.000032  0.000032  0.0 |  1.36 | 0.000325 | 0.113937 |  0.377751 | 0.054373 | 0.232536 |   0.9072 | 0.668215 | 0.0 | 0.0 | 0.0 |    2.782262 | 0.972307 |  0.002741 |  0.222934 | 0.987184 | 0.114257 |  0.185569   0.871989   1.902029  0.0 | -3.000506  -3.000506  -3.000506  0.0 | 132.804294 | 105.620872 | 813.171982 | 1127.684719 | 0.186479 |   354.86044 | 0.114257 |
    | 2017-03-06 00:00:00 |  3.7 |  0.3 | 0.0 | 0.0 | 0.0 |  0.24 | 0.4158  0.702  0.378  0.6804 | 0.4158  0.702  0.378  0.0 |  0.131711   0.000231   0.068138  0.0 | 0.255779  0.664281  0.377897  0.0 |  0.131711   0.000231   0.068138  0.0 | 0.0  0.0  0.0  0.0 | 18.500006  18.500006  18.500006  0.0 | 0.000019  0.000019  0.000019  0.0 |  0.24 | 0.000062 | 0.021334 |   0.07311 | 0.037539 | 0.159579 |   0.6804 | 0.501208 | 0.0 | 0.0 | 0.0 |    9.319057 | 0.989578 |   0.05111 |  0.204364 | 0.860821 | 0.099632 |  0.038079   0.447477   1.695993  0.0 | -3.000525  -3.000525  -3.000525  0.0 | 132.841772 | 105.315439 |  810.89709 | 1137.003776 | 0.055225 |   322.89545 | 0.099632 |
    | 2017-03-07 00:00:00 |  1.6 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |       0.0        0.0        0.0  0.0 | 0.203149  0.529562  0.440411  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |  8.006707   8.006707   8.006707  0.0 | 0.000008  0.000008  0.000008  0.0 |   0.0 | 0.000002 | 0.000001 |  0.000007 | 0.288057 | 0.281177 |   0.7938 |   0.5842 | 0.0 | 0.0 | 0.0 |    7.008302 | 1.014673 |  0.094936 |  0.047764 | 0.738683 | 0.085496 |  -0.16507  -0.082085   1.255582  0.0 | -3.000533  -3.000533  -3.000533  0.0 | 133.129827 | 105.184214 | 808.564507 | 1144.012078 | 0.007468 |   290.29904 | 0.085496 |
    | 2017-03-08 00:00:00 |  4.0 |  1.8 | 0.0 | 0.0 | 0.0 |  1.44 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |  0.767794   0.000465   0.302127  0.0 | 0.289627  0.646563  0.440662  0.0 |  0.767794   0.000465   0.302127  0.0 | 0.0  0.0  0.0  0.0 | 20.000002  20.000002  20.000002  0.0 | 0.000021  0.000021  0.000021  0.0 |  1.44 | 0.000121 | 0.123392 |  0.416978 | 0.171611 | 0.194938 |   0.7938 | 0.584528 | 0.0 | 0.0 | 0.0 |   -3.016441 | 1.012892 |  0.138254 |   0.22059 | 0.645061 |  0.07466 |   0.21751   0.710887   1.952794  0.0 | -3.000553  -3.000554  -3.000554  0.0 | 133.301316 | 104.887569 | 806.236021 | 1140.995637 | 0.203855 |  273.565805 |  0.07466 |
    | 2017-03-09 00:00:00 |  5.6 |  8.9 | 0.0 | 0.0 | 0.0 |  7.12 | 0.3465  0.585  0.315   0.567 | 0.3465  0.585  0.315  0.0 |  6.357888   4.564072   6.159136  0.0 | 0.317364  0.584263  0.314994  0.0 |  6.357888   4.564072   6.159136  0.0 | 0.0  0.0  0.0  0.0 |      28.0       28.0       28.0  0.0 | 0.000029  0.000029  0.000029  0.0 |  7.12 | 1.131749 | 1.007433 |  4.855912 | 0.000734 | 0.029057 |    0.567 | 0.417894 | 0.0 | 0.0 | 0.0 |  -57.538524 | 0.862359 |  0.150193 |   2.78857 |  0.76444 | 0.088477 |  0.662257   2.682553   2.598664  0.0 | -3.000582  -3.000583  -3.000583  0.0 | 132.170301 | 103.628206 | 804.253585 | 1083.457113 | 2.271197 |  382.952397 | 0.088477 |
    | 2017-03-10 00:00:00 |  5.8 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.5544  0.936  0.504  0.9072 | 0.5544  0.936  0.504  0.0 |       0.0        0.0        0.0  0.0 | 0.473404   0.93593  0.503987  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      29.0       29.0       29.0  0.0 |  0.00003   0.00003   0.00003  0.0 |   0.0 | 0.000007 | 0.000005 |  0.000025 |  0.00007 | 0.080782 |   0.9072 |  0.66858 | 0.0 | 0.0 | 0.0 |   29.521033 | 0.820808 | -0.038966 |  1.963847 | 1.171296 | 0.135567 |  0.188854   1.746623   2.094677  0.0 | -3.000612  -3.000612  -3.000612  0.0 | 132.170363 | 103.259614 |  802.36667 | 1112.978146 | 0.307375 |  418.563293 | 0.135567 |
    | 2017-03-11 00:00:00 |  5.7 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |  0.693   1.17   0.63   1.134 |  0.693   1.17   0.63  0.0 |       0.0        0.0        0.0  0.0 | 0.342827  1.155668  0.629755  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      28.5       28.5       28.5  0.0 | 0.000029  0.000029  0.000029  0.0 |   0.0 | 0.000007 | 0.000005 |  0.000024 | 0.014265 | 0.349251 |    1.134 | 0.835298 | 0.0 | 0.0 | 0.0 |    11.20131 | 0.883388 |  -0.03394 |  0.265774 | 1.092949 | 0.126499 | -0.153973   0.590955   1.464922  0.0 | -3.000641  -3.000642  -3.000642  0.0 | 132.184621 | 103.133226 | 800.335894 | 1124.179457 | 0.041625 |  374.806892 | 0.126499 |
    | 2017-03-12 00:00:00 |  4.6 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.8316  1.404  0.756  1.3608 | 0.8316  1.404  0.756  0.0 |       0.0        0.0        0.0  0.0 | 0.194387  0.813372  0.748184  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      23.0       23.0       23.0  0.0 | 0.000024  0.000024  0.000024  0.0 |   0.0 | 0.000006 | 0.000004 |   0.00002 | 0.587838 | 0.635533 |   1.3608 | 1.000487 | 0.0 | 0.0 | 0.0 |    8.813984 |  0.91579 |  0.035434 |  0.036022 | 0.910527 | 0.105385 |  -0.34836  -0.222417   0.716738  0.0 | -3.000665  -3.000665  -3.000665  0.0 | 132.772453 | 103.346294 |  798.23063 | 1132.993441 | 0.005623 |  330.726594 | 0.105385 |
    | 2017-03-13 00:00:00 |  4.2 |  0.9 | 0.0 | 0.0 | 0.0 |  0.72 | 0.6237  1.053  0.567  1.0206 | 0.6237  1.053  0.567  0.0 |  0.198331   0.000023   0.002957  0.0 | 0.206897  0.452756  0.552685  0.0 |  0.198331   0.000023   0.002957  0.0 | 0.0  0.0  0.0  0.0 | 21.000001  21.000001  21.000001  0.0 | 0.000022  0.000022  0.000022  0.0 |  0.72 | 0.000011 | 0.030926 |  0.100764 | 0.597375 | 0.415701 |   1.0206 | 0.749401 | 0.0 | 0.0 | 0.0 |    5.259473 | 0.937515 |  0.092674 |   0.05551 | 0.763574 | 0.088377 | -0.033588   0.044803   0.881095  0.0 | -3.000687  -3.000687  -3.000687  0.0 | 133.369817 | 103.354985 | 796.075424 | 1138.252914 | 0.050876 |  297.691936 | 0.088377 |
    | 2017-03-14 00:00:00 |  7.4 |  0.1 | 0.0 | 0.0 | 0.0 |  0.08 | 0.6237  1.053  0.567  1.0206 | 0.6237  1.053  0.567  0.0 |  0.023912   0.000002   0.000192  0.0 | 0.223888  0.377752  0.535188  0.0 |  0.023912   0.000002   0.000192  0.0 | 0.0  0.0  0.0  0.0 |      37.0       37.0       37.0  0.0 | 0.000038  0.000038  0.000038  0.0 |  0.08 |  0.00001 | 0.003734 |  0.012161 | 0.671979 | 0.398754 |   1.0206 | 0.747595 | 0.0 | 0.0 | 0.0 |    6.588323 | 0.953614 |  0.133348 |  0.051293 | 0.658545 |  0.07622 | -0.201388  -0.252951   0.425716  0.0 | -3.000725  -3.000725  -3.000725  0.0 | 134.041786 | 103.406546 | 793.883207 | 1144.841237 | 0.011744 |   270.25787 |  0.07622 |
    | 2017-03-15 00:00:00 |  6.3 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.8316  1.404  0.756  1.3608 | 0.8316  1.404  0.756  0.0 |       0.0        0.0        0.0  0.0 | 0.171765  0.216666  0.491846  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      31.5       31.5       31.5  0.0 | 0.000032  0.000032  0.000032  0.0 |   0.0 | 0.000008 | 0.000005 |  0.000027 | 1.181479 | 0.658083 |   1.3608 | 0.974422 | 0.0 | 0.0 | 0.0 |    5.664674 | 0.970139 |  0.166575 |  0.010231 |  0.56981 |  0.06595 | -0.373152  -0.469617   -0.06613  0.0 | -3.000757  -3.000757  -3.000757  0.0 | 135.223257 | 103.746129 | 791.653002 | 1150.505911 |  0.00154 |  245.805186 |  0.06595 |
    | 2017-03-16 00:00:00 |  8.7 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.9702  1.638  0.882  1.5876 | 0.9702  1.638  0.882  0.0 |       0.0        0.0        0.0  0.0 | 0.116292  0.128134  0.249352  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      43.5       43.5       43.5  0.0 | 0.000045  0.000045  0.000045  0.0 |   0.0 | 0.000012 | 0.000007 |  0.000037 |  1.50222 |  0.85162 |   1.5876 | 1.104645 | 0.0 | 0.0 | 0.0 |    5.489598 | 0.984683 |  0.193453 |  0.001457 | 0.496675 | 0.057486 | -0.489445  -0.597751  -0.315482  0.0 | -3.000802  -3.000802  -3.000802  0.0 | 136.725465 | 104.298853 | 789.389363 | 1155.995509 |  0.00012 |  225.142744 | 0.057486 |
    | 2017-03-17 00:00:00 |  6.4 |  3.9 | 0.0 | 0.0 | 0.0 |  3.12 | 0.7623  1.287  0.693  1.2474 | 0.7623  1.287  0.693  0.0 |  1.818542   0.016213   0.251692  0.0 |  0.48501  0.989328  0.602596  0.0 |  1.818542   0.016213   0.251692  0.0 | 0.0  0.0  0.0  0.0 |      32.0       32.0       32.0  0.0 | 0.000033  0.000033  0.000033  0.0 |  3.12 | 0.004267 | 0.287879 |  0.947184 | 0.296137 | 0.276543 |   1.2474 | 0.909723 | 0.0 | 0.0 | 0.0 |  -14.689083 | 0.959143 |  0.219903 |  0.465019 | 0.464206 | 0.053728 |  0.327003   1.516708    1.95023  0.0 | -3.000834  -3.000835  -3.000835  0.0 | 137.017336 | 104.027849 | 787.184437 | 1141.306426 | 0.482285 |    233.0561 | 0.053728 |
    | 2017-03-18 00:00:00 |  5.2 |  8.7 | 0.0 | 0.0 | 0.0 |  6.96 | 0.7623  1.287  0.693  1.2474 | 0.7623  1.287  0.693  0.0 |  6.071488   4.671028   5.752821  0.0 | 0.685962  1.286933  0.692984  0.0 |  6.071488   4.671028   5.752821  0.0 | 0.0  0.0  0.0  0.0 |      26.0       26.0       26.0  0.0 | 0.000027  0.000027  0.000027  0.0 |  6.96 | 1.216937 | 0.948827 |  4.685133 | 0.000067 | 0.076136 |   1.2474 | 0.919352 | 0.0 | 0.0 | 0.0 |  -48.750551 | 0.805216 |  0.193898 |   2.99428 | 0.662266 | 0.076651 |  0.529554   2.518748   2.464426  0.0 | -3.000861  -3.000862  -3.000862  0.0 | 135.800466 | 102.946448 | 785.333366 | 1092.555875 | 2.173137 |  358.075681 | 0.076651 |
    | 2017-03-19 00:00:00 |  5.1 | 26.4 | 0.0 | 0.0 | 0.0 | 21.12 | 0.3465  0.585  0.315   0.567 | 0.3465  0.585  0.315  0.0 | 20.402772  20.094679  20.410269  0.0 | 0.337475  0.584999  0.314999  0.0 | 20.402772  20.094679  20.410269  0.0 | 0.0  0.0  0.0  0.0 |      25.5       25.5       25.5  0.0 | 0.000026  0.000026  0.000026  0.0 | 21.12 | 5.019148 | 3.057611 | 16.970809 | 0.000001 | 0.009002 |    0.567 | 0.417928 | 0.0 | 0.0 | 0.0 |  -98.550109 | 0.488076 |  -0.34756 |  11.43817 | 1.901226 | 0.220049 |  0.909307   2.959069   2.859158  0.0 | -3.000887  -3.000888  -3.000888  0.0 | 130.781319 |  99.306241 | 784.211352 |  994.005766 | 7.705777 |  833.819419 | 0.220049 |
    | 2017-03-20 00:00:00 |  8.7 | 11.5 | 0.0 | 0.0 | 0.0 |   9.2 | 0.4158  0.702  0.378  0.6804 | 0.4158  0.702  0.378  0.0 |  8.931932   8.697497   8.951414  0.0 | 0.406537  0.701998  0.377999  0.0 |  8.931932   8.697497   8.951414  0.0 | 0.0  0.0  0.0  0.0 |      43.5       43.5       43.5  0.0 | 0.000045  0.000045  0.000045  0.0 |   9.2 | 2.069054 | 1.247058 |  7.494644 | 0.000002 | 0.009241 |   0.6804 | 0.501518 | 0.0 | 0.0 | 0.0 |  -30.236821 | 0.355464 | -2.421531 | 10.924277 | 4.350738 | 0.503558 |  0.770839   2.759574   2.729745  0.0 | -3.000932  -3.000932  -3.000932  0.0 | 128.712266 |  95.469162 | 783.394194 |  963.768945 | 4.276143 | 1088.898659 | 0.503558 |
    | 2017-03-21 00:00:00 |  6.2 |  0.9 | 0.0 | 0.0 | 0.0 |  0.72 | 1.0395  1.755  0.945   1.701 | 1.0395  1.755  0.945  0.0 |   0.55413   0.275985   0.548126  0.0 |  0.84288  1.754558  0.944963  0.0 |   0.55413   0.275985   0.548126  0.0 | 0.0  0.0  0.0  0.0 |      31.0       31.0       31.0  0.0 | 0.000032  0.000032  0.000032  0.0 |  0.72 | 0.064683 | 0.072325 |  0.407318 |  0.00044 | 0.196193 |    1.701 | 1.253571 | 0.0 | 0.0 | 0.0 |   -3.156627 | 0.300038 | -2.692614 |  3.954163 | 4.685246 | 0.542274 |  0.093828   1.449031   1.956656  0.0 | -3.000964  -3.000964  -3.000964  0.0 | 128.648023 |  92.750396 | 782.704453 |  960.612318 | 0.729299 |  968.246465 | 0.542274 |
    | 2017-03-22 00:00:00 |  5.9 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |  1.386   2.34   1.26   2.268 |  1.386   2.34   1.26  0.0 |       0.0        0.0        0.0  0.0 | 0.443274  1.688002  1.251506  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      29.5       29.5       29.5  0.0 |  0.00003   0.00003   0.00003  0.0 |   0.0 | 0.000007 | 0.000004 |  0.000026 | 0.649126 |  0.94073 |    2.268 | 1.668917 | 0.0 | 0.0 | 0.0 |   11.357409 | 0.307513 | -1.472671 |  0.630613 | 3.537274 | 0.409407 | -0.349446  -0.238971    0.70515  0.0 | -3.000994  -3.000994  -3.000994  0.0 | 129.297142 |  92.064695 | 781.997526 |  971.969727 | 0.098712 |   776.71825 | 0.409407 |
    | 2017-03-23 00:00:00 |  5.2 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.1088  1.872  1.008  1.8144 | 1.1088  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.139053  0.269832  0.749581  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      26.0       26.0       26.0  0.0 | 0.000027  0.000027  0.000027  0.0 |   0.0 | 0.000006 | 0.000003 |  0.000023 |  1.59495 |   0.9677 |   1.8144 | 1.308763 | 0.0 | 0.0 | 0.0 |   30.885691 | 0.356562 | -0.657095 |   0.08536 | 2.555959 | 0.295829 | -0.488499  -0.508804  -0.044431  0.0 | -3.001021  -3.001021  -3.001021  0.0 | 130.892085 |  92.197016 | 781.177843 | 1002.855418 | 0.013375 |  631.969957 | 0.295829 |
    | 2017-03-24 00:00:00 |  5.2 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.1088  1.872  1.008  1.8144 | 1.1088  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.087817  0.124854  0.285164  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      26.0       26.0       26.0  0.0 | 0.000027  0.000027  0.000027  0.0 |   0.0 | 0.000007 | 0.000003 |  0.000023 |  1.73901 | 1.018813 |   1.8144 | 1.262908 | 0.0 | 0.0 | 0.0 |   36.803123 | 0.458008 | -0.287039 |  0.011566 | 1.909116 | 0.220962 | -0.576315  -0.633658  -0.329595  0.0 | -3.001047  -3.001048  -3.001048  0.0 | 132.631089 |  92.699781 | 780.124951 | 1039.658541 | 0.001832 |  526.827564 | 0.220962 |
    | 2017-03-25 00:00:00 |  5.9 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.8316  1.404  0.756  1.3608 | 0.8316  1.404  0.756  0.0 |       0.0        0.0        0.0  0.0 | 0.049328  0.062899  0.109425  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      29.5       29.5       29.5  0.0 |  0.00003   0.00003   0.00003  0.0 |   0.0 | 0.000008 | 0.000004 |  0.000026 | 1.334662 | 0.780594 |   1.3608 | 0.936816 | 0.0 | 0.0 | 0.0 |   27.510361 | 0.557305 | -0.099504 |  0.001596 |  1.47659 | 0.170902 | -0.625643  -0.696556   -0.43902  0.0 | -3.001078  -3.001078  -3.001078  0.0 | 133.965743 |  93.102215 |  778.84379 | 1067.168902 | 0.000261 |   448.79009 | 0.170902 |
    | 2017-03-26 00:00:00 |  6.7 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.9009  1.521  0.819  1.4742 | 0.9009  1.521  0.819  0.0 |       0.0        0.0        0.0  0.0 | 0.043675  0.052835  0.081017  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      33.5       33.5       33.5  0.0 | 0.000034  0.000034  0.000034  0.0 |   0.0 | 0.000009 | 0.000004 |  0.000029 | 1.460917 | 0.855368 |   1.4742 | 1.011104 | 0.0 | 0.0 | 0.0 |    18.90369 | 0.631377 |  0.010637 |  0.000244 | 1.175726 | 0.136079 | -0.669318  -0.749391  -0.520037  0.0 | -3.001112  -3.001112  -3.001112  0.0 | 135.426651 |  93.652527 | 777.392349 | 1086.072591 | 0.000047 |  388.854232 | 0.136079 |
    | 2017-03-27 00:00:00 |  7.0 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.1088  1.872  1.008  1.8144 | 1.1088  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.044307  0.051549  0.072308  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      35.0       35.0       35.0  0.0 | 0.000036  0.000036  0.000036  0.0 |   0.0 | 0.000009 | 0.000005 |   0.00003 | 1.811166 | 1.062156 |   1.8144 | 1.241612 | 0.0 | 0.0 | 0.0 |   14.078475 | 0.684547 |  0.085794 |  0.000058 | 0.958106 | 0.110892 | -0.713625  -0.800939  -0.592344  0.0 | -3.001148  -3.001148  -3.001148  0.0 | 137.237808 |  94.458199 | 775.818677 | 1100.151067 | 0.000019 |  341.659712 | 0.110892 |
    | 2017-03-28 00:00:00 |  8.3 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.3167  2.223  1.197  2.1546 | 1.3167  2.223  1.197  0.0 |       0.0        0.0        0.0  0.0 | 0.043337  0.048888  0.064064  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      41.5       41.5       41.5  0.0 | 0.000043  0.000043  0.000043  0.0 |   0.0 | 0.000011 | 0.000006 |  0.000036 | 2.162575 | 1.270515 |   2.1546 | 1.472085 | 0.0 | 0.0 | 0.0 |   11.782226 | 0.726474 |  0.141201 |  0.000037 | 0.796223 | 0.092155 | -0.756962  -0.849828  -0.656408  0.0 |  -3.00119  -3.001191  -3.001191  0.0 | 139.400372 |  95.506671 | 774.148621 | 1111.933293 | 0.000018 |  303.847088 | 0.092155 |
    | 2017-03-29 00:00:00 |  9.0 |  1.5 | 0.0 | 0.0 | 0.0 |   1.2 | 0.5544  0.936  0.504  0.9072 | 0.5544  0.936  0.504  0.0 |  0.261849   0.000022   0.000144  0.0 | 0.145064  0.238287   0.20422  0.0 |  0.261849   0.000022   0.000144  0.0 | 0.0  0.0  0.0  0.0 |      45.0       45.0       45.0  0.0 | 0.000046  0.000046  0.000046  0.0 |   1.2 | 0.000019 | 0.035251 |  0.136021 | 0.693904 | 0.408409 |   0.9072 | 0.637664 | 0.0 | 0.0 | 0.0 |    6.168213 | 0.756936 |  0.181113 |  0.055666 | 0.679853 | 0.078687 |  0.036125   0.111863   0.339228  0.0 | -3.001236  -3.001237  -3.001237  0.0 | 140.094257 |  95.682474 | 772.408538 | 1118.101506 | 0.080372 |  278.199636 | 0.078687 |
    | 2017-03-30 00:00:00 | 12.4 |  0.3 | 0.0 | 0.0 | 0.0 |  0.24 | 1.0395  1.755  0.945   1.701 | 1.0395  1.755  0.945  0.0 |  0.076998   0.000006    0.00005  0.0 | 0.397313  0.624875  0.602036  0.0 |  0.076998   0.000006    0.00005  0.0 | 0.0  0.0  0.0  0.0 |      62.0       62.0       62.0  0.0 | 0.000064  0.000064  0.000064  0.0 |  0.24 | 0.000019 | 0.010414 |  0.040015 | 1.123843 | 0.640724 |    1.701 | 1.217508 | 0.0 | 0.0 | 0.0 |    7.140775 | 0.773507 |  0.208695 |  0.094062 | 0.603683 | 0.069871 | -0.198186  -0.273018  -0.022858  0.0 |   -3.0013    -3.0013    -3.0013  0.0 | 141.218082 |  96.134726 | 770.630361 | 1125.242281 | 0.026326 |  257.299118 | 0.069871 |
    | 2017-03-31 00:00:00 | 15.0 |  0.2 | 0.0 | 0.0 | 0.0 |  0.16 | 1.8711  3.159  1.701  3.0618 | 1.8711  3.159  1.701  0.0 |  0.023623   0.000001   0.000007  0.0 | 0.346812  0.402175  0.468846  0.0 |  0.023623   0.000001   0.000007  0.0 | 0.0  0.0  0.0  0.0 |      75.0       75.0       75.0  0.0 | 0.000077  0.000077  0.000077  0.0 |  0.16 | 0.000022 | 0.003256 |  0.012291 |   2.7409 | 1.520752 |   3.0618 | 2.129374 | 0.0 | 0.0 | 0.0 |    7.754985 | 0.794848 |  0.231049 |  0.031403 | 0.532877 | 0.061676 | -0.408621  -0.515194   -0.33171  0.0 | -3.001377  -3.001377  -3.001377  0.0 | 143.958959 |  97.485846 | 768.803125 | 1132.997267 | 0.007214 |  236.085036 | 0.061676 |
    | 2017-04-01 00:00:00 | 11.8 |  4.5 | 0.0 | 0.0 | 0.0 |   3.6 | 1.2825  1.755  0.945   1.728 | 1.2825  1.755  0.945  0.0 |  1.991617   0.031278   0.413917  0.0 | 0.874837  1.427861   0.83191  0.0 |  1.991617   0.031278   0.413917  0.0 | 0.0  0.0  0.0  0.0 |      59.0       59.0       59.0  0.0 |  0.00006   0.00006   0.00006  0.0 |   3.6 | 0.009029 | 0.278279 |  1.076129 | 0.325191 | 0.406703 |    1.728 |  1.38503 | 0.0 | 0.0 | 0.0 |  -12.015036 | 0.783506 |  0.251729 |  0.540592 | 0.501695 | 0.058067 |  0.324926   1.625667   2.022463  0.0 | -3.001437  -3.001438  -3.001438  0.0 | 144.275122 |  97.474246 | 767.001961 |  1120.98223 | 0.542752 |  246.762096 | 0.058067 |
    | 2017-04-02 00:00:00 |  9.4 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |  1.368  1.872  1.008  1.8432 |  1.368  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.610077  1.639882  1.005982  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      47.0       47.0       47.0  0.0 | 0.000048  0.000048  0.000048  0.0 |   0.0 | 0.000014 | 0.000007 |   0.00004 | 0.230728 | 0.756131 |   1.8432 |  1.48874 | 0.0 | 0.0 | 0.0 |   13.527818 | 0.779947 |  0.223788 |  0.469314 | 0.546386 | 0.063239 |  -0.28515  -0.014215    1.01648  0.0 | -3.001485  -3.001486  -3.001486  0.0 | 144.505836 |  98.064185 |  765.20898 | 1134.510048 | 0.073478 |  247.175308 | 0.063239 |
    | 2017-04-03 00:00:00 |  8.1 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |   1.71   2.34   1.26   2.304 |   1.71   2.34   1.26  0.0 |       0.0        0.0        0.0  0.0 | 0.227014  0.478555  1.031196  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      40.5       40.5       40.5  0.0 | 0.000041  0.000041  0.000041  0.0 |   0.0 | 0.000012 | 0.000006 |  0.000034 | 1.850067 | 1.479413 |    2.304 | 1.835131 | 0.0 | 0.0 | 0.0 |    8.492247 | 0.812372 |  0.221785 |  0.063603 | 0.506921 | 0.058671 | -0.512164   -0.49277  -0.014716  0.0 | -3.001527  -3.001527  -3.001527  0.0 | 146.355891 |  99.359191 | 763.341458 | 1143.002295 | 0.009909 |  229.162291 | 0.058671 |
    | 2017-04-04 00:00:00 |  7.9 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.7955  2.457  1.323  2.4192 | 1.7955  2.457  1.323  0.0 |       0.0        0.0        0.0  0.0 | 0.119522  0.161049   0.35667  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      39.5       39.5       39.5  0.0 |  0.00004   0.00004   0.00004  0.0 |   0.0 | 0.000012 | 0.000006 |  0.000033 | 2.281329 | 1.671825 |   2.4192 | 1.854709 | 0.0 | 0.0 | 0.0 |    8.489756 |  0.83894 |  0.233856 |  0.008642 | 0.449949 | 0.052077 | -0.631686  -0.653819  -0.371386  0.0 | -3.001567  -3.001568  -3.001568  0.0 | 148.637208 | 100.845396 | 761.412861 | 1151.492051 |   0.0013 |  211.544452 | 0.052077 |
    | 2017-04-05 00:00:00 |  7.5 |  0.4 | 0.0 | 0.0 | 0.0 |  0.32 | 1.4535  1.989  1.071  1.9584 | 1.4535  1.989  1.071  0.0 |  0.014442   0.000001   0.000008  0.0 | 0.119692  0.145401  0.209505  0.0 |  0.014442   0.000001   0.000008  0.0 | 0.0  0.0  0.0  0.0 |      37.5       37.5       37.5  0.0 | 0.000038  0.000038  0.000038  0.0 |  0.32 | 0.000012 | 0.002181 |  0.007392 | 1.831368 | 1.330415 |   1.9584 | 1.493533 | 0.0 | 0.0 | 0.0 |    7.682379 | 0.864745 |  0.241868 |  0.004842 | 0.401735 | 0.046497 |  -0.44582  -0.479221  -0.260899  0.0 | -3.001606  -3.001606  -3.001606  0.0 | 150.468564 | 101.983126 | 759.424941 |  1159.17443 |  0.00385 |  197.167521 | 0.046497 |
    | 2017-04-06 00:00:00 |  7.2 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.4535  1.989  1.071  1.9584 | 1.4535  1.989  1.071  0.0 |       0.0        0.0        0.0  0.0 | 0.126245  0.143931  0.175573  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      36.0       36.0       36.0  0.0 | 0.000037  0.000037  0.000037  0.0 |   0.0 | 0.000012 | 0.000006 |   0.00003 | 1.832375 | 1.323802 |   1.9584 | 1.490036 | 0.0 | 0.0 | 0.0 |    6.989918 | 0.886777 |  0.247163 |  0.003269 |   0.3627 | 0.041979 | -0.572066  -0.623152  -0.436472  0.0 | -3.001642  -3.001643  -3.001643  0.0 | 152.300928 | 103.110696 | 757.386373 | 1166.164347 | 0.000612 |   184.50089 | 0.041979 |
    | 2017-04-07 00:00:00 |  8.1 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |  0.684  0.936  0.504  0.9216 |  0.684  0.936  0.504  0.0 |       0.0        0.0        0.0  0.0 | 0.042006  0.045667  0.053483  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      40.5       40.5       40.5  0.0 | 0.000041  0.000041  0.000041  0.0 |   0.0 | 0.000013 | 0.000006 |  0.000034 |  0.88404 | 0.640298 |   0.9216 | 0.698202 | 0.0 | 0.0 | 0.0 |    5.324699 | 0.904687 |  0.250927 |  0.000551 | 0.330789 | 0.038286 | -0.614071  -0.668819  -0.489955  0.0 | -3.001684  -3.001684  -3.001684  0.0 | 153.184955 | 103.549572 | 755.306633 | 1171.489046 | 0.000094 |  174.444123 | 0.038286 |
    | 2017-04-08 00:00:00 |  8.6 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.1115  1.521  0.819  1.4976 | 1.1115  1.521  0.819  0.0 |       0.0        0.0        0.0  0.0 | 0.055215  0.058905   0.06747  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      43.0       43.0       43.0  0.0 | 0.000044  0.000044  0.000044  0.0 |   0.0 | 0.000014 | 0.000007 |  0.000036 | 1.451518 | 1.053457 |   1.4976 | 1.132528 | 0.0 | 0.0 | 0.0 |    4.657658 | 0.916087 |  0.255244 |    0.0001 | 0.304517 | 0.035245 | -0.669287  -0.727725  -0.557426  0.0 | -3.001728  -3.001728  -3.001728  0.0 | 154.636459 | 104.400223 | 753.200687 | 1176.146705 |  0.00003 |  165.229742 | 0.035245 |
    | 2017-04-09 00:00:00 | 10.5 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 2.1375  2.925  1.575    2.88 | 2.1375  2.925  1.575  0.0 |       0.0        0.0        0.0  0.0 | 0.078861  0.082509  0.091985  0.0 |       0.0        0.0        0.0  0.0 | 0.0  0.0  0.0  0.0 |      52.5       52.5       52.5  0.0 | 0.000054  0.000054  0.000054  0.0 |   0.0 | 0.000018 | 0.000009 |  0.000043 | 2.821037 | 2.052978 |     2.88 | 2.173778 | 0.0 | 0.0 | 0.0 |    6.554298 | 0.930233 |  0.257171 |   0.00005 | 0.279964 | 0.032403 | -0.748148  -0.810233   -0.64941  0.0 | -3.001781  -3.001782  -3.001782  0.0 | 157.457478 | 106.245246 |  751.06222 | 1182.701003 | 0.000023 |  155.914799 | 0.032403 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _wland_gd_snowfall:

snowfall
________

>>> er(False)
>>> inputs.t.series -= 8.0

.. integration-test::

    >>> test("wland_gd_snowfall", axis1=(fluxes.pc, states.sp), axis2=(inputs.t,))
    |                date |     t |    p | fxg | fxs | dhs |    pc |                           pe |                       pet |                                   tf |                                ei |                                rf |                                   sf |                                   pm |                                   am |    ps | pve |       pv |        pq | etve |      etv |       es |       et |  gr | fxs | fxg |         cdg | fgse |       fgs |       fqs |       rh |        r |                                   ic |                                   sp | dve |         dv | hge |          dg |        hq |          hs |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 00:00:00 | -10.8 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.3942  0.702  0.378  0.6588 | 0.3942  0.702  0.378  0.0 |       0.0        0.0        0.0  0.0 |      0.0  0.000001       0.0  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 |  0.49406 | 0.000067 | 0.435763 | 0.0 | 0.0 | 0.0 |  -37.374521 |  0.0 | -0.000081 |       0.0 |      0.0 |      0.0 |      -3.0  -3.000001       -3.0  0.0 |      -3.0       -3.0       -3.0  0.0 | nan | 140.493979 | nan | 1562.625479 |       0.0 |    -2.00364 |      0.0 |
    | 2017-02-11 00:00:00 |  -9.5 |  0.4 | 0.0 | 0.0 | 0.0 |  0.32 | 0.5256  0.936  0.504  0.8784 | 0.5256  0.936  0.504  0.0 |  0.000001        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |      0.0       0.0       0.0  0.0 |  0.000001        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.32 | 0.0 |      0.0 |       0.0 |  0.0 | 0.658704 | 0.000199 | 0.580983 | 0.0 | 0.0 | 0.0 |  -15.520341 |  0.0 | -0.000055 |       0.0 |      0.0 |      0.0 | -2.680002  -2.680003  -2.680002  0.0 | -2.999999       -3.0       -3.0  0.0 | nan | 141.152629 | nan | 1547.105138 |       0.0 |   -1.686273 |      0.0 |
    | 2017-02-12 00:00:00 |  -8.9 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.4599  0.819  0.441  0.7686 | 0.4599  0.819  0.441  0.0 |       0.0        0.0        0.0  0.0 | 0.000002  0.000004  0.000002  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.576325 |  0.00033 | 0.508328 | 0.0 | 0.0 | 0.0 |   -5.171804 |  0.0 | -0.000045 |       0.0 |      0.0 |      0.0 | -2.680004  -2.680007  -2.680004  0.0 | -2.999999       -3.0       -3.0  0.0 | nan | 141.728908 | nan | 1541.933335 |       0.0 |     -1.6886 |      0.0 |
    | 2017-02-13 00:00:00 |  -9.6 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.329312 | 0.000186 | 0.290458 | 0.0 | 0.0 | 0.0 |   -0.904799 |  0.0 | -0.000043 |       0.0 |      0.0 |      0.0 | -2.680005  -2.680009  -2.680005  0.0 | -2.999999       -3.0       -3.0  0.0 | nan | 142.058178 | nan | 1541.028536 |       0.0 |   -1.690662 |      0.0 |
    | 2017-02-14 00:00:00 |  -9.3 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.329299 | 0.000185 | 0.290447 | 0.0 | 0.0 | 0.0 |    0.730086 |  0.0 | -0.000043 |       0.0 |      0.0 |      0.0 | -2.680007  -2.680011  -2.680006  0.0 | -2.999999       -3.0       -3.0  0.0 | nan | 142.387434 | nan | 1541.758622 |       0.0 |   -1.692723 |      0.0 |
    | 2017-02-15 00:00:00 |  -6.3 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.329287 | 0.000183 | 0.290436 | 0.0 | 0.0 | 0.0 |    1.503678 |  0.0 | -0.000044 |       0.0 |      0.0 |      0.0 | -2.680008  -2.680013  -2.680007  0.0 | -2.999999       -3.0       -3.0  0.0 | nan | 142.716677 | nan |   1543.2623 |       0.0 |   -1.694836 |      0.0 |
    | 2017-02-16 00:00:00 |  -3.6 |  0.2 | 0.0 | 0.0 | 0.0 |  0.16 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |  0.000001        0.0        0.0  0.0 | 0.000002  0.000003  0.000002  0.0 |      0.0       0.0       0.0  0.0 |  0.000001        0.0        0.0  0.0 |  0.000008   0.000008   0.000008  0.0 |       0.0        0.0        0.0  0.0 |  0.16 | 0.0 |      0.0 |       0.0 |  0.0 | 0.329274 | 0.000267 | 0.290427 | 0.0 | 0.0 | 0.0 |    1.869901 |  0.0 | -0.000046 |       0.0 |      0.0 |      0.0 |  -2.52001  -2.520016  -2.520009  0.0 | -2.999999       -3.0       -3.0  0.0 | nan | 143.045906 | nan | 1545.132201 |       0.0 |   -1.537112 |      0.0 |
    | 2017-02-17 00:00:00 |  -3.5 |  4.5 | 0.0 | 0.0 | 0.0 |   3.6 | 0.1971  0.351  0.189  0.3294 | 0.1971  0.351  0.189  0.0 |  0.715908   0.000866   0.002554  0.0 | 0.045779  0.103421  0.056212  0.0 |      0.0       0.0       0.0  0.0 |  0.715908   0.000866   0.002554  0.0 |  0.000011   0.000011   0.000011  0.0 |       0.0        0.0        0.0  0.0 |   3.6 | 0.0 |      0.0 |       0.0 |  0.0 | 0.182336 | 0.188579 | 0.227425 | 0.0 | 0.0 | 0.0 |    1.816816 |  0.0 |  -0.00016 |       0.0 | 0.000084 |  0.00001 |  0.318303   0.975698   1.021225  0.0 | -2.284091  -2.999134  -2.997446  0.0 | nan | 143.228082 | nan | 1546.949018 |       0.0 |    1.863072 |  0.00001 |
    | 2017-02-18 00:00:00 |  -4.6 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.1971  0.351  0.189  0.3294 | 0.1971  0.351  0.189  0.0 |       0.0        0.0        0.0  0.0 | 0.148002  0.341645    0.1863  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.035641 | 0.329237 | 0.243746 | 0.0 | 0.0 | 0.0 |     1.10961 |  0.0 | -0.000414 |       0.0 | 0.000301 | 0.000035 |  0.170301   0.634053   0.834925  0.0 | -2.284091  -2.999134  -2.997446  0.0 | nan | 143.263308 | nan | 1548.058628 |       0.0 |    1.500521 | 0.000035 |
    | 2017-02-19 00:00:00 |  -3.2 |  3.2 | 0.0 | 0.0 | 0.0 |  2.56 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |  2.024091   0.452925   0.899571  0.0 | 0.221756   0.46542  0.251473  0.0 |      0.0       0.0       0.0  0.0 |  2.024091   0.452925   0.899571  0.0 |   0.00003    0.00003    0.00003  0.0 |  0.000001        0.0        0.0  0.0 |  2.56 | 0.0 |      0.0 |       0.0 |  0.0 | 0.028057 | 0.439154 |   0.3254 | 0.0 | 0.0 | 0.0 |     0.62853 |  0.0 | -0.000649 |       0.0 | 0.000568 | 0.000066 |  0.484454   2.275708    2.24388  0.0 |     -0.26   -2.54621  -2.097875  0.0 | nan | 143.290717 | nan | 1548.687158 |       0.0 |     3.56435 | 0.000066 |
    | 2017-02-20 00:00:00 |  -1.3 |  4.6 | 0.0 | 0.0 | 0.0 |  3.68 | 0.1971  0.351  0.189  0.3294 | 0.1971  0.351  0.189  0.0 |  3.340123   2.972407   3.208228  0.0 | 0.184113  0.350996  0.188997  0.0 | 0.584521  0.520171   0.56144  0.0 |  2.755601   2.452236   2.646788  0.0 |  0.018374   0.018374   0.018374  0.0 |  0.016188   0.000752   0.003746  0.0 |  3.68 | 0.0 | 0.163231 |  0.426313 |  0.0 | 0.008608 |   0.3294 | 0.244154 | 0.0 | 0.0 | 0.0 |  -11.698319 |  0.0 | -0.002484 |  0.238152 | 0.004347 | 0.000503 |  0.640218   2.632305   2.526656  0.0 |  2.479413  -0.094726   0.545167  0.0 | nan |  143.13361 | nan | 1536.988839 |  0.188162 |   18.257497 | 0.000503 |
    | 2017-02-21 00:00:00 |  -2.2 |  2.3 | 0.0 | 0.0 | 0.0 |  1.84 | 0.3942  0.702  0.378  0.6588 | 0.3942  0.702  0.378  0.0 |  1.642164   1.409265   1.618931  0.0 | 0.363811  0.701991  0.377995  0.0 |      0.0       0.0       0.0  0.0 |  1.642164   1.409265   1.618931  0.0 |  0.000888   0.000888   0.000888  0.0 |  0.000888   0.000771   0.000879  0.0 |  1.84 | 0.0 | 0.000241 |  0.000635 |  0.0 | 0.020144 |   0.6588 | 0.488293 | 0.0 | 0.0 | 0.0 |    5.455631 |  0.0 | -0.007533 |  0.164025 | 0.015788 | 0.001827 |  0.474243   2.361049    2.36973  0.0 |  4.120689   1.313768   2.163219  0.0 | nan |  143.14598 | nan |  1542.44447 |  0.024772 |   26.354278 | 0.001827 |
    | 2017-02-22 00:00:00 |  -1.5 | 18.0 | 0.0 | 0.0 | 0.0 |  14.4 | 0.5256  0.936  0.504  0.8784 | 0.5256  0.936  0.504  0.0 | 13.589457  13.048462  13.564559  0.0 | 0.504697  0.935996  0.503997  0.0 | 1.698682  1.631058   1.69557  0.0 | 11.890775  11.417404  11.868989  0.0 |  0.009388   0.009388   0.009388  0.0 |  0.009388   0.009388   0.009388  0.0 |  14.4 | 0.0 | 0.477411 |  1.257802 |  0.0 | 0.013856 |   0.8784 | 0.651125 | 0.0 | 0.0 | 0.0 |  -32.795894 |  0.0 | -0.016422 |  0.729041 | 0.044053 | 0.005099 |  0.780089   2.776592   2.701174  0.0 | 16.002075  12.721784   14.02282  0.0 | nan | 142.666003 | nan | 1509.648576 |  0.553533 |   72.672054 | 0.005099 |
    | 2017-02-23 00:00:00 |  -3.0 | 19.2 | 0.0 | 0.0 | 0.0 | 15.36 | 0.3285  0.585  0.315   0.549 | 0.3285  0.585  0.315  0.0 | 14.927277  14.656358  14.931703  0.0 | 0.322035  0.584999  0.314999  0.0 |      0.0       0.0       0.0  0.0 | 14.927277  14.656358  14.931703  0.0 |   0.00006    0.00006    0.00006  0.0 |   0.00006    0.00006    0.00006  0.0 | 15.36 | 0.0 | 0.000017 |  0.000045 |  0.0 | 0.004285 |    0.549 | 0.406976 | 0.0 | 0.0 | 0.0 |   14.917354 |  0.0 | -0.050959 |  0.478617 | 0.121267 | 0.014036 |  0.890776   2.895235   2.814472  0.0 | 30.929293  27.378082  28.954463  0.0 | nan | 142.619312 | nan |  1524.56593 |  0.074961 |   102.62462 | 0.014036 |
    | 2017-02-24 00:00:00 |  -5.0 |  0.4 | 0.0 | 0.0 | 0.0 |  0.32 | 0.5256  0.936  0.504  0.8784 | 0.5256  0.936  0.504  0.0 |  0.294509   0.228516   0.294807  0.0 | 0.495655  0.935979  0.503995  0.0 |      0.0       0.0       0.0  0.0 |  0.294509   0.228516   0.294807  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.32 | 0.0 |      0.0 |       0.0 |  0.0 | 0.019855 |   0.8784 | 0.651095 | 0.0 | 0.0 | 0.0 |    6.186144 |  0.0 | -0.063465 |  0.065204 |  0.13674 | 0.015826 |  0.420612   2.050739    2.33567  0.0 | 31.223802  27.606598  29.249269  0.0 | nan | 142.575702 | nan | 1530.752073 |  0.009757 |   95.625403 | 0.015826 |
    | 2017-02-25 00:00:00 |  -4.9 |  8.3 | 0.0 | 0.0 | 0.0 |  6.64 | 0.3285  0.585  0.315   0.549 | 0.3285  0.585  0.315  0.0 |  6.067393   5.421308   6.057702  0.0 | 0.308291  0.584994  0.314997  0.0 |      0.0       0.0       0.0  0.0 |  6.067393   5.421308   6.057702  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |  6.64 | 0.0 |      0.0 |       0.0 |  0.0 | 0.013397 |    0.549 | 0.406929 | 0.0 | 0.0 | 0.0 |     3.01934 |  0.0 | -0.059915 |  0.008441 | 0.126377 | 0.014627 |  0.684928   2.684437   2.602971  0.0 | 37.291195  33.027906  35.306971  0.0 | nan | 142.529184 | nan | 1533.771413 |  0.001316 |   93.168885 | 0.014627 |
    | 2017-02-26 00:00:00 |  -0.9 |  5.3 | 0.0 | 0.0 | 0.0 |  4.24 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |  3.990639   3.797605   3.989314  0.0 | 0.251901  0.467998  0.251998  0.0 | 1.097426  1.044342  1.097061  0.0 |  2.893213   2.753264   2.892253  0.0 |  0.069591   0.069591   0.069591  0.0 |  0.069591   0.069591   0.069591  0.0 |  4.24 | 0.0 | 0.323231 |  0.860146 |  0.0 | 0.007225 |   0.4392 | 0.325561 | 0.0 | 0.0 | 0.0 |  -22.238586 |  0.0 | -0.060378 |  0.489408 | 0.137194 | 0.015879 |  0.682388   2.658834   2.601659  0.0 | 40.114817  35.711579  38.129633  0.0 | nan | 142.152799 | nan | 1511.532827 |  0.372055 |  111.428298 | 0.015879 |
    | 2017-02-27 00:00:00 |   1.4 |  0.7 | 0.0 | 0.0 | 0.0 |  0.56 | 0.8541  1.521  0.819  1.4274 | 0.8541  1.521  0.819  0.0 |  0.424043   0.206924    0.40434  0.0 | 0.695509   1.52066  0.818961  0.0 | 0.360436  0.175885  0.343689  0.0 |  0.063606   0.031039   0.060651  0.0 |  7.013137   7.013137   7.013137  0.0 |  7.013137   7.013137   7.013137  0.0 |  0.56 | 0.0 | 2.025329 |  5.493737 |  0.0 |  0.10525 |   1.4274 |  1.05767 | 0.0 | 0.0 | 0.0 | -138.098269 |  0.0 | -0.093476 |  3.451178 | 0.318804 | 0.036899 |  0.122836    1.49125   1.938358  0.0 | 33.165286  28.729481  31.177147  0.0 | nan | 140.139244 | nan | 1373.434557 |  2.414614 |  259.606107 | 0.036899 |
    | 2017-02-28 00:00:00 |  -3.4 |  2.7 | 0.0 | 0.0 | 0.0 |  2.16 | 0.5913  1.053  0.567  0.9882 | 0.5913  1.053  0.567  0.0 |  1.504354   0.475414   1.340729  0.0 | 0.453714  1.052728  0.566963  0.0 |      0.0       0.0       0.0  0.0 |  1.504354   0.475414   1.340729  0.0 |  0.000016   0.000016   0.000016  0.0 |  0.000016   0.000016   0.000016  0.0 |  2.16 | 0.0 | 0.000004 |  0.000012 |  0.0 |  0.09131 |   0.9882 | 0.732148 | 0.0 | 0.0 | 0.0 |   56.694204 |  0.0 | -0.321328 |  2.087848 |  0.72861 |  0.08433 |  0.324768   2.123108   2.190666  0.0 | 34.669625  29.204879   32.51786  0.0 | nan | 139.909222 | nan | 1430.128761 |  0.326778 |   312.48138 |  0.08433 |
    | 2017-03-01 00:00:00 |  -4.3 |  1.6 | 0.0 | 0.0 | 0.0 |  1.28 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |  0.950596   0.497035   0.875887  0.0 | 0.384059  0.818947  0.440979  0.0 |      0.0       0.0       0.0  0.0 |  0.950596   0.497035   0.875887  0.0 |  0.000001   0.000001   0.000001  0.0 |  0.000001   0.000001   0.000001  0.0 |  1.28 | 0.0 |      0.0 |  0.000001 |  0.0 | 0.067011 |   0.7938 | 0.584792 | 0.0 | 0.0 | 0.0 |   25.124689 |  0.0 | -0.354315 |  0.282537 | 0.700583 | 0.081086 |  0.270113   2.087125   2.153799  0.0 |  35.62022  29.701913  33.393747  0.0 | nan | 139.621917 | nan |  1455.25345 |  0.044242 |  276.157404 | 0.081086 |
    | 2017-03-02 00:00:00 |  -3.3 |  2.5 | 0.0 | 0.0 | 0.0 |   2.0 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |  1.532198   1.008874   1.460256  0.0 |   0.3938  0.818966  0.440983  0.0 |      0.0       0.0       0.0  0.0 |  1.532198   1.008874   1.460256  0.0 |  0.000022   0.000022   0.000022  0.0 |  0.000022   0.000022   0.000022  0.0 |   2.0 | 0.0 | 0.000006 |  0.000016 |  0.0 | 0.060547 |   0.7938 | 0.584825 | 0.0 | 0.0 | 0.0 |   16.230148 |  0.0 | -0.283648 |  0.038212 | 0.567025 | 0.065628 |  0.344114   2.259285    2.25256  0.0 | 37.152396  30.710765  34.853981  0.0 | nan |  139.39881 | nan | 1471.483598 |  0.006046 |  238.375872 | 0.065628 |
    | 2017-03-03 00:00:00 |  -2.1 |  0.6 | 0.0 | 0.0 | 0.0 |  0.48 | 0.7623  1.287  0.693  1.2474 | 0.7623  1.287  0.693  0.0 |  0.288984   0.088968    0.24228  0.0 |  0.50592  1.286377  0.692918  0.0 |      0.0       0.0       0.0  0.0 |  0.288984   0.088968    0.24228  0.0 |  0.001244   0.001244   0.001244  0.0 |  0.001244   0.001244   0.001244  0.0 |  0.48 | 0.0 | 0.000337 |  0.000941 |  0.0 | 0.170201 |   1.2474 | 0.918647 | 0.0 | 0.0 | 0.0 |   11.678215 |  0.0 | -0.220439 |  0.005791 | 0.453767 | 0.052519 |   0.02921   1.363939   1.797362  0.0 | 37.440136  30.798489  35.095016  0.0 | nan | 139.348234 | nan | 1483.161814 |  0.001196 |  205.482507 | 0.052519 |
    | 2017-03-04 00:00:00 |  -0.3 |  0.2 | 0.0 | 0.0 | 0.0 |  0.16 |  0.693   1.17   0.63   1.134 |  0.693   1.17   0.63  0.0 |  0.056025   0.000729   0.020468  0.0 | 0.287077  1.128774  0.629377  0.0 | 0.023811   0.00031  0.008699  0.0 |  0.032214   0.000419   0.011769  0.0 |   0.45975    0.45975    0.45975  0.0 |   0.45975    0.45975    0.45975  0.0 |  0.16 | 0.0 | 0.128752 |  0.359122 |  0.0 | 0.282818 |    1.134 | 0.834465 | 0.0 | 0.0 | 0.0 |   -0.408938 |  0.0 | -0.173899 |  0.205385 | 0.375927 |  0.04351 | -0.153891   0.394436   1.307517  0.0 | 37.012601  30.339159  34.647035  0.0 | nan | 139.328401 | nan | 1482.752875 |  0.154934 |  188.107046 |  0.04351 |
    | 2017-03-05 00:00:00 |  -1.7 |  1.7 | 0.0 | 0.0 | 0.0 |  1.36 | 0.5544  0.936  0.504  0.9072 | 0.5544  0.936  0.504  0.0 |  0.699103   0.001275   0.261794  0.0 | 0.321123  0.881327   0.50364  0.0 | 0.052433  0.000096  0.019635  0.0 |  0.646671    0.00118   0.242159  0.0 |   0.00479    0.00479    0.00479  0.0 |   0.00479    0.00479    0.00479  0.0 |  1.36 | 0.0 | 0.010768 |   0.02855 |  0.0 | 0.172802 |   0.9072 | 0.667842 | 0.0 | 0.0 | 0.0 |    9.626413 |  0.0 | -0.152844 |  0.148762 | 0.335119 | 0.038787 |  0.185883   0.871833   1.902084  0.0 | 37.654481  30.335549  34.884404  0.0 | nan | 139.337591 | nan | 1492.379288 |  0.034721 |  172.352854 | 0.038787 |
    | 2017-03-06 00:00:00 |  -4.3 |  0.3 | 0.0 | 0.0 | 0.0 |  0.24 | 0.4158  0.702  0.378  0.6804 | 0.4158  0.702  0.378  0.0 |  0.131772    0.00023   0.068149  0.0 |  0.25588  0.664259  0.377898  0.0 |      0.0       0.0       0.0  0.0 |  0.131772    0.00023   0.068149  0.0 |  0.000001   0.000001   0.000001  0.0 |  0.000001   0.000001   0.000001  0.0 |  0.24 | 0.0 |      0.0 |  0.000001 |  0.0 | 0.118549 |   0.6804 | 0.500951 | 0.0 | 0.0 | 0.0 |    6.984818 |  0.0 | -0.129413 |  0.030202 | 0.286129 | 0.033117 |  0.038231   0.447344   1.696037  0.0 | 37.786253  30.335778  34.952553  0.0 | nan | 139.326726 | nan | 1499.364106 |   0.00452 |  153.378825 | 0.033117 |
    | 2017-03-07 00:00:00 |  -6.4 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |       0.0        0.0        0.0  0.0 |  0.20321  0.529519  0.440411  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.282879 |   0.7938 | 0.583701 | 0.0 | 0.0 | 0.0 |    5.255398 |  0.0 | -0.106024 |  0.003817 |  0.23938 | 0.027706 | -0.164979  -0.082174   1.255626  0.0 | 37.786253  30.335778  34.952553  0.0 | nan |  139.50358 | nan | 1504.619504 |  0.000703 |  136.127347 | 0.027706 |
    | 2017-03-08 00:00:00 |  -4.0 |  1.8 | 0.0 | 0.0 | 0.0 |  1.44 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |  0.767711   0.000464    0.30128  0.0 | 0.289599  0.646696   0.44066  0.0 |      0.0       0.0       0.0  0.0 |  0.767711   0.000464    0.30128  0.0 |  0.000002   0.000002   0.000002  0.0 |  0.000002   0.000002   0.000002  0.0 |  1.44 | 0.0 | 0.000001 |  0.000002 |  0.0 | 0.186747 |   0.7938 | 0.584185 | 0.0 | 0.0 | 0.0 |    4.348392 |  0.0 | -0.088079 |  0.000609 | 0.202486 | 0.023436 |  0.217711   0.710666   1.953686  0.0 | 38.553961  30.336241  35.253831  0.0 | nan | 139.602248 | nan | 1508.967896 |  0.000095 |  122.794788 | 0.023436 |
    | 2017-03-09 00:00:00 |  -2.4 |  8.9 | 0.0 | 0.0 | 0.0 |  7.12 | 0.3465  0.585  0.315   0.567 | 0.3465  0.585  0.315  0.0 |  6.357962   4.562585   6.159877  0.0 | 0.317367  0.584262  0.314994  0.0 |      0.0       0.0       0.0  0.0 |  6.357962   4.562585   6.159877  0.0 |  0.000453   0.000453   0.000453  0.0 |  0.000453   0.000453   0.000453  0.0 |  7.12 | 0.0 | 0.000123 |  0.000342 |  0.0 | 0.019561 |    0.567 | 0.417847 | 0.0 | 0.0 | 0.0 |    2.704937 |  0.0 | -0.077996 |  0.000277 | 0.180509 | 0.020892 |  0.662382   2.683819   2.598816  0.0 | 44.911471  34.898373  41.413255  0.0 | nan |  139.54369 | nan | 1511.672833 |  0.000161 |  116.896253 | 0.020892 |
    | 2017-03-10 00:00:00 |  -2.2 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.5544  0.936  0.504  0.9072 | 0.5544  0.936  0.504  0.0 |       0.0        0.0        0.0  0.0 | 0.472491  0.935923  0.503987  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.000888   0.000888   0.000888  0.0 |  0.000888   0.000888   0.000888  0.0 |   0.0 | 0.0 | 0.000241 |  0.000671 |  0.0 | 0.054335 |   0.9072 | 0.668444 | 0.0 | 0.0 | 0.0 |     1.82595 |  0.0 | -0.068376 |  0.000521 | 0.160593 | 0.018587 |  0.189891   1.747896   2.094829  0.0 | 44.910583  34.897485  41.412366  0.0 | nan | 139.529407 | nan | 1513.498782 |  0.000311 |  104.969567 | 0.018587 |
    | 2017-03-11 00:00:00 |  -2.3 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |  0.693   1.17   0.63   1.134 |  0.693   1.17   0.63  0.0 |       0.0        0.0        0.0  0.0 | 0.343246  1.155778  0.629754  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.000634   0.000634   0.000634  0.0 |  0.000634   0.000634   0.000634  0.0 |   0.0 | 0.0 | 0.000172 |  0.000479 |  0.0 | 0.236615 |    1.134 | 0.834717 | 0.0 | 0.0 | 0.0 |    1.984874 |  0.0 | -0.056846 |  0.000541 | 0.136949 | 0.015851 | -0.153355   0.592118   1.465074  0.0 | 44.909949  34.896851  41.411732  0.0 | nan | 139.709004 | nan | 1515.483657 |  0.000249 |   94.507768 | 0.015851 |
    | 2017-03-12 00:00:00 |  -3.4 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.8316  1.404  0.756  1.3608 | 0.8316  1.404  0.756  0.0 |       0.0        0.0        0.0  0.0 | 0.194698  0.817474  0.748186  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.000016   0.000016   0.000016  0.0 |  0.000016   0.000016   0.000016  0.0 |   0.0 | 0.0 | 0.000004 |  0.000012 |  0.0 | 0.616708 |   1.3608 | 0.999295 | 0.0 | 0.0 | 0.0 |    2.919494 |  0.0 | -0.047842 |  0.000217 | 0.117119 | 0.013555 | -0.348054  -0.225356   0.716888  0.0 | 44.909933  34.896836  41.411717  0.0 | nan | 140.277866 | nan | 1518.403151 |  0.000044 |   85.191808 | 0.013555 |
    | 2017-03-13 00:00:00 |  -3.8 |  0.9 | 0.0 | 0.0 | 0.0 |  0.72 | 0.6237  1.053  0.567  1.0206 | 0.6237  1.053  0.567  0.0 |  0.198569   0.000023   0.002961  0.0 | 0.207131  0.451043  0.552689  0.0 |      0.0       0.0       0.0  0.0 |  0.198569   0.000023   0.002961  0.0 |  0.000004   0.000004   0.000004  0.0 |  0.000004   0.000004   0.000004  0.0 |  0.72 | 0.0 | 0.000001 |  0.000003 |  0.0 | 0.475714 |   1.0206 | 0.748555 | 0.0 | 0.0 | 0.0 |    3.744463 |  0.0 | -0.041505 |  0.000039 | 0.101376 | 0.011733 | -0.033754   0.043578   0.881239  0.0 | 45.108498  34.896855  41.414673  0.0 | nan | 140.712075 | nan | 1522.147614 |  0.000007 |   77.993978 | 0.011733 |
    | 2017-03-14 00:00:00 |  -0.6 |  0.1 | 0.0 | 0.0 | 0.0 |  0.08 | 0.6237  1.053  0.567  1.0206 | 0.6237  1.053  0.567  0.0 |  0.023902   0.000002   0.000192  0.0 | 0.223811  0.376938  0.535207  0.0 | 0.008366  0.000001  0.000067  0.0 |  0.015537   0.000001   0.000125  0.0 |  0.184058   0.184058   0.184058  0.0 |  0.184058   0.184058   0.184058  0.0 |  0.08 | 0.0 | 0.052367 |  0.141954 |  0.0 | 0.489198 |   1.0206 | 0.746756 | 0.0 | 0.0 | 0.0 |   -0.266525 |  0.0 | -0.037097 |  0.080359 | 0.091091 | 0.010543 | -0.201467  -0.253362   0.425841  0.0 | 44.939977  34.712798   41.23074  0.0 | nan | 141.111809 | nan | 1521.881089 |  0.061602 |   74.800445 | 0.010543 |
    | 2017-03-15 00:00:00 |  -1.7 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.8316  1.404  0.756  1.3608 | 0.8316  1.404  0.756  0.0 |       0.0        0.0        0.0  0.0 | 0.171721  0.216415  0.491956  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.00479    0.00479    0.00479  0.0 |   0.00479    0.00479    0.00479  0.0 |   0.0 | 0.0 | 0.001333 |   0.00359 |  0.0 | 0.831056 |   1.3608 | 0.973017 | 0.0 | 0.0 | 0.0 |     5.77395 |  0.0 | -0.035603 |  0.055331 | 0.085726 | 0.009922 | -0.373188  -0.469777  -0.066116  0.0 | 44.935187  34.708008   41.22595  0.0 | nan | 141.905929 | nan | 1527.655039 |  0.009861 |   70.294486 | 0.009922 |
    | 2017-03-16 00:00:00 |   0.7 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.9702  1.638  0.882  1.5876 | 0.9702  1.638  0.882  0.0 |       0.0        0.0        0.0  0.0 | 0.116277  0.128063  0.249379  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  3.633659   3.633659   3.633659  0.0 |  3.633659   3.633659   3.633659  0.0 |   0.0 | 0.0 | 1.016363 |  2.718932 |  0.0 | 1.066482 |   1.5876 |  1.10285 | 0.0 | 0.0 | 0.0 |  -68.145644 |  0.0 | -0.040536 |  1.551856 | 0.130012 | 0.015048 | -0.489465   -0.59784  -0.315494  0.0 | 41.301528  31.074349  37.592291  0.0 | nan | 141.915512 | nan | 1459.509394 |  1.176937 |  136.459582 | 0.015048 |
    | 2017-03-17 00:00:00 |  -1.6 |  3.9 | 0.0 | 0.0 | 0.0 |  3.12 | 0.7623  1.287  0.693  1.2474 | 0.7623  1.287  0.693  0.0 |  1.818476   0.016201   0.251089  0.0 | 0.484996  0.989423  0.602617  0.0 | 0.181848   0.00162  0.025109  0.0 |  1.636628   0.014581    0.22598  0.0 |  0.006707   0.006707   0.006707  0.0 |  0.006707   0.006707   0.006707  0.0 |  3.12 | 0.0 | 0.036006 |  0.086407 |  0.0 |  0.28244 |   1.2474 | 0.909184 | 0.0 | 0.0 | 0.0 |   31.776076 |  0.0 | -0.114108 |   1.06036 | 0.283743 | 0.032841 |  0.327064   1.516536     1.9508  0.0 | 42.931449  31.082224  37.811565  0.0 | nan | 142.047838 | nan | 1491.285471 |  0.202984 |  171.070504 | 0.032841 |
    | 2017-03-18 00:00:00 |  -2.8 |  8.7 | 0.0 | 0.0 | 0.0 |  6.96 | 0.7623  1.287  0.693  1.2474 | 0.7623  1.287  0.693  0.0 |  6.071649   4.671871   5.753559  0.0 | 0.685976  1.286933  0.692984  0.0 |      0.0       0.0       0.0  0.0 |  6.071649   4.671871   5.753559  0.0 |  0.000117   0.000117   0.000117  0.0 |  0.000117   0.000117   0.000117  0.0 |  6.96 | 0.0 | 0.000033 |  0.000088 |  0.0 | 0.050614 |   1.2474 | 0.919214 | 0.0 | 0.0 | 0.0 |   15.954528 |  0.0 |  -0.14267 |  0.175531 | 0.302009 | 0.034955 |  0.529438   2.517732   2.464257  0.0 | 49.002981  35.753978  43.565006  0.0 | nan | 141.955749 | nan | 1507.239999 |   0.02754 |  163.991895 | 0.034955 |
    | 2017-03-19 00:00:00 |  -2.9 | 26.4 | 0.0 | 0.0 | 0.0 | 21.12 | 0.3465  0.585  0.315   0.567 | 0.3465  0.585  0.315  0.0 | 20.402847  20.094414   20.41026  0.0 | 0.337476  0.584999  0.314999  0.0 |      0.0       0.0       0.0  0.0 | 20.402847  20.094414   20.41026  0.0 |  0.000084   0.000084   0.000084  0.0 |  0.000084   0.000084   0.000084  0.0 | 21.12 | 0.0 | 0.000023 |  0.000063 |  0.0 | 0.005982 |    0.567 | 0.417912 | 0.0 | 0.0 | 0.0 |    6.992955 |  0.0 | -0.144763 |  0.023825 |  0.29082 |  0.03366 |  0.909116   2.958319   2.858998  0.0 | 69.405744  55.848308  63.975182  0.0 | nan | 141.816945 | nan | 1514.232953 |  0.003778 |  164.787268 |  0.03366 |
    | 2017-03-20 00:00:00 |   0.7 | 11.5 | 0.0 | 0.0 | 0.0 |   9.2 | 0.4158  0.702  0.378  0.6804 | 0.4158  0.702  0.378  0.0 |  8.931806   8.696853   8.951311  0.0 | 0.406533  0.701998  0.377999  0.0 | 6.028969  5.870376  6.042135  0.0 |  2.902837   2.826477   2.909176  0.0 |  3.633659   3.633659   3.633659  0.0 |  3.633659   3.633659   3.633659  0.0 |   9.2 | 0.0 | 2.637776 |  7.242368 |  0.0 | 0.006145 |   0.6804 |   0.5015 | 0.0 | 0.0 | 0.0 | -191.121106 |  0.0 | -0.165233 |  4.116209 |   0.4995 | 0.057812 |  0.770777   2.759468   2.729688  0.0 | 68.674922  55.041126    63.2507  0.0 | nan | 139.020081 | nan | 1323.111847 |  3.129938 |  342.739342 | 0.057812 |
    | 2017-03-21 00:00:00 |  -1.8 |  0.9 | 0.0 | 0.0 | 0.0 |  0.72 | 1.0395  1.755  0.945   1.701 | 1.0395  1.755  0.945  0.0 |  0.554111   0.275961     0.5481  0.0 | 0.842857  1.754558  0.944963  0.0 | 0.027706  0.013798  0.027405  0.0 |  0.526406   0.262163   0.520695  0.0 |   0.00342    0.00342    0.00342  0.0 |   0.00342    0.00342    0.00342  0.0 |  0.72 | 0.0 | 0.007126 |   0.02051 |  0.0 |  0.13054 |    1.701 | 1.253203 | 0.0 | 0.0 | 0.0 |   64.346367 |  0.0 | -0.503555 |  2.719201 | 1.075034 | 0.124425 |  0.093809   1.448949   1.956626  0.0 | 69.197908  55.299869  63.767974  0.0 | nan |  138.63994 | nan | 1387.458214 |  0.431248 |  399.040698 | 0.124425 |
    | 2017-03-22 00:00:00 |  -2.1 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |  1.386   2.34   1.26   2.268 |  1.386   2.34   1.26  0.0 |       0.0        0.0        0.0  0.0 | 0.443267  1.687835  1.251504  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.001244   0.001244   0.001244  0.0 |  0.001244   0.001244   0.001244  0.0 |   0.0 | 0.0 | 0.000334 |  0.000944 |  0.0 | 0.841351 |    2.268 | 1.666944 | 0.0 | 0.0 | 0.0 |   32.300776 |  0.0 | -0.515815 |  0.373351 | 0.990226 | 0.114609 | -0.349458  -0.238886   0.705122  0.0 | 69.196663  55.298624  63.766729  0.0 | nan | 138.965143 | nan |  1419.75899 |  0.058841 |  342.808148 | 0.114609 |
    | 2017-03-23 00:00:00 |  -2.8 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.1088  1.872  1.008  1.8144 | 1.1088  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.139047  0.269888  0.749587  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.000117   0.000117   0.000117  0.0 |  0.000117   0.000117   0.000117  0.0 |   0.0 | 0.0 | 0.000032 |  0.000089 |  0.0 | 1.174146 |   1.8144 | 1.306451 | 0.0 | 0.0 | 0.0 |     26.8524 |  0.0 | -0.390584 |  0.050859 | 0.768623 | 0.088961 | -0.488506  -0.508774  -0.044465  0.0 | 69.196546  55.298507  63.766612  0.0 | nan | 139.748673 | nan |  1446.61139 |  0.008071 |  287.829941 | 0.088961 |
    | 2017-03-24 00:00:00 |  -2.8 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.1088  1.872  1.008  1.8144 | 1.1088  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.087816   0.12487  0.285174  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.000117   0.000117   0.000117  0.0 |  0.000117   0.000117   0.000117  0.0 |   0.0 | 0.0 | 0.000032 |  0.000089 |  0.0 | 1.256073 |   1.8144 | 1.260439 | 0.0 | 0.0 | 0.0 |   22.666764 |  0.0 | -0.295909 |   0.00707 | 0.594107 | 0.068762 | -0.576322  -0.633644  -0.329639  0.0 | 69.196428  55.298389  63.766495  0.0 | nan | 140.708804 | nan | 1469.278155 |  0.001089 |  243.607007 | 0.068762 |
    | 2017-03-25 00:00:00 |  -2.1 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.8316  1.404  0.756  1.3608 | 0.8316  1.404  0.756  0.0 |       0.0        0.0        0.0  0.0 | 0.049327  0.062902   0.10941  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.001244   0.001244   0.001244  0.0 |  0.001244   0.001244   0.001244  0.0 |   0.0 | 0.0 | 0.000344 |  0.000934 |  0.0 |  0.96312 |   1.3608 | 0.934907 | 0.0 | 0.0 | 0.0 |   18.317582 |  0.0 | -0.230722 |   0.00146 | 0.466841 | 0.054033 | -0.625648  -0.696546  -0.439048  0.0 | 69.195184  55.297145   63.76525  0.0 | nan | 141.440858 | nan | 1487.595737 |  0.000564 |  208.800823 | 0.054033 |
    | 2017-03-26 00:00:00 |  -1.3 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.9009  1.521  0.819  1.4742 | 0.9009  1.521  0.819  0.0 |       0.0        0.0        0.0  0.0 | 0.043674  0.052837  0.081009  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.018374   0.018374   0.018374  0.0 |  0.018374   0.018374   0.018374  0.0 |   0.0 | 0.0 | 0.005137 |  0.013751 |  0.0 | 1.054831 |   1.4742 | 1.008998 | 0.0 | 0.0 | 0.0 |   14.489615 |  0.0 | -0.184226 |  0.007862 | 0.373381 | 0.043215 | -0.669322  -0.749383  -0.520058  0.0 |  69.17681  55.278771  63.746876  0.0 | nan | 142.306326 | nan | 1502.085352 |  0.006453 |  180.918456 | 0.043215 |
    | 2017-03-27 00:00:00 |  -1.0 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.1088  1.872  1.008  1.8144 | 1.1088  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.044306   0.05155  0.072302  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |      0.05       0.05       0.05  0.0 |      0.05       0.05       0.05  0.0 |   0.0 | 0.0 | 0.014156 |  0.037259 |  0.0 | 1.308831 |   1.8144 | 1.238971 | 0.0 | 0.0 | 0.0 |    12.21463 |  0.0 | -0.150412 |  0.026991 | 0.303562 | 0.035135 | -0.713628  -0.800933   -0.59236  0.0 |  69.12681  55.228771  63.696876  0.0 | nan |  143.45059 | nan | 1514.299982 |  0.016721 |  158.615327 | 0.035135 |
    | 2017-03-28 00:00:00 |   0.3 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.3167  2.223  1.197  2.1546 | 1.3167  2.223  1.197  0.0 |       0.0        0.0        0.0  0.0 | 0.043336   0.04889   0.06406  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |   1.95975    1.95975    1.95975  0.0 |   1.95975    1.95975    1.95975  0.0 |   0.0 | 0.0 | 0.561874 |  1.454063 |  0.0 | 1.564285 |   2.1546 | 1.468925 | 0.0 | 0.0 | 0.0 |  -28.267984 |  0.0 | -0.132287 |   0.84029 | 0.288226 | 0.033359 | -0.756964  -0.849823   -0.65642  0.0 |  67.16706  53.269021  61.737126  0.0 | nan | 144.320714 | nan | 1486.031998 |  0.630495 |  177.389779 | 0.033359 |
    | 2017-03-29 00:00:00 |   1.0 |  1.5 | 0.0 | 0.0 | 0.0 |   1.2 | 0.5544  0.936  0.504  0.9072 | 0.5544  0.936  0.504  0.0 |  0.261946   0.000022   0.000144  0.0 | 0.145107   0.23828  0.204215  0.0 |  0.19646  0.000017  0.000108  0.0 |  0.065487   0.000006   0.000036  0.0 |      5.05       5.05       5.05  0.0 |      5.05       5.05       5.05  0.0 |   1.2 | 0.0 | 1.483532 |  3.832713 |  0.0 | 0.502441 |   0.9072 | 0.636687 | 0.0 | 0.0 | 0.0 |  -83.805574 |  0.0 |  -0.18111 |  2.706238 | 0.457226 |  0.05292 |  0.035982   0.111875   0.339221  0.0 | 62.182546  48.219026  56.687162  0.0 | nan | 143.158513 | nan | 1402.226423 |  1.756969 |  279.440011 |  0.05292 |
    | 2017-03-30 00:00:00 |   4.4 |  0.3 | 0.0 | 0.0 | 0.0 |  0.24 | 1.0395  1.755  0.945   1.701 | 1.0395  1.755  0.945  0.0 |  0.076968   0.000006    0.00005  0.0 | 0.397174  0.624732  0.601928  0.0 | 0.076968  0.000006   0.00005  0.0 |       0.0        0.0        0.0  0.0 | 22.000001  22.000001  22.000001  0.0 | 22.000001  22.000001  22.000001  0.0 |  0.24 | 0.0 | 6.040842 |  16.60943 |  0.0 | 0.800552 |    1.701 | 1.216305 | 0.0 | 0.0 | 0.0 | -164.873342 |  0.0 |  -0.64489 |  10.92191 | 1.428218 | 0.165303 | -0.198159  -0.272863  -0.022757  0.0 | 40.182546  26.219026  34.687162  0.0 | nan | 137.273333 | nan | 1237.353082 |  7.444489 |  713.302066 | 0.165303 |
    | 2017-03-31 00:00:00 |   7.0 |  0.2 | 0.0 | 0.0 | 0.0 |  0.16 | 1.8711  3.159  1.701  3.0618 | 1.8711  3.159  1.701  0.0 |  0.023632   0.000001   0.000007  0.0 | 0.346942  0.403332   0.47052  0.0 | 0.023632  0.000001  0.000007  0.0 |       0.0        0.0        0.0  0.0 |      35.0       35.0       35.0  0.0 |      35.0  27.003244  34.732152  0.0 |  0.16 | 0.0 | 8.019377 | 25.370929 |  0.0 |  1.92555 |   3.0618 | 2.128264 | 0.0 | 0.0 | 0.0 |  -110.53264 |  0.0 | -3.867914 | 21.446484 | 4.674203 | 0.540996 | -0.408734  -0.516195  -0.333283  0.0 |  5.182546  -0.784218   -0.04499  0.0 | nan | 127.311593 | nan | 1126.820441 | 11.368934 | 1356.992828 | 0.540996 |
    | 2017-04-01 00:00:00 |   3.8 |  4.5 | 0.0 | 0.0 | 0.0 |   3.6 | 1.2825  1.755  0.945   1.728 | 1.2825  1.755  0.945  0.0 |  1.991548   0.031215   0.413719  0.0 | 0.874807  1.427486  0.831576  0.0 | 1.991548  0.031215  0.413719  0.0 |       0.0        0.0        0.0  0.0 | 19.000004  19.000004  19.000004  0.0 |  6.071372   0.262718   0.920078  0.0 |   3.6 | 0.0 | 1.213312 |  3.967332 |  0.0 | 0.379423 |    1.728 | 1.384773 | 0.0 | 0.0 | 0.0 |  -21.219014 |  0.0 | -7.094546 | 12.663221 | 7.209938 | 0.834484 |  0.324911   1.625103   2.021421  0.0 | -0.888827  -1.046935  -0.965068  0.0 | nan | 119.383158 | nan | 1105.601427 |  2.673044 | 1305.996329 | 0.834484 |
    | 2017-04-02 00:00:00 |   1.4 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |  1.368  1.872  1.008  1.8432 |  1.368  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.610094  1.639722  1.005972  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  7.013137   7.013137   7.013137  0.0 |  0.093304   0.050378   0.069838  0.0 |   0.0 | 0.0 | 0.015615 |  0.064026 |  0.0 | 0.580684 |   1.8432 | 1.488426 | 0.0 | 0.0 | 0.0 |   12.107009 |  0.0 |  -4.25785 |  2.349444 | 5.267185 | 0.609628 | -0.285183  -0.014619   1.015449  0.0 | -0.982131  -1.097314  -1.034907  0.0 | nan | 115.690377 | nan | 1117.708436 |  0.387627 |  968.145413 | 0.609628 |
    | 2017-04-03 00:00:00 |   0.1 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |   1.71   2.34   1.26   2.304 |   1.71   2.34   1.26  0.0 |       0.0        0.0        0.0  0.0 | 0.226983  0.478112  1.029561  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  1.298601   1.298601   1.298601  0.0 |  0.013653   0.008179   0.010807  0.0 |   0.0 | 0.0 | 0.002265 |  0.009687 |  0.0 | 1.603905 |    2.304 | 1.835652 | 0.0 | 0.0 | 0.0 |   21.791906 |  0.0 | -1.982545 |  0.340631 | 3.354194 | 0.388217 | -0.512166  -0.492731  -0.014112  0.0 | -0.995783  -1.105493  -1.045714  0.0 | nan | 115.309472 | nan | 1139.500342 |  0.056683 |  727.392405 | 0.388217 |
    | 2017-04-04 00:00:00 |  -0.1 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.7955  2.457  1.323  2.4192 | 1.7955  2.457  1.323  0.0 |       0.0        0.0        0.0  0.0 | 0.119526  0.161082  0.357094  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.798601   0.798601   0.798601  0.0 |  0.007994   0.004882   0.006391  0.0 |   0.0 | 0.0 |  0.00134 |  0.005694 |  0.0 | 1.876255 |   2.4192 | 1.855876 | 0.0 | 0.0 | 0.0 |   42.113913 |  0.0 | -0.979657 |  0.052255 |  2.25581 | 0.261089 | -0.631692  -0.653813  -0.371206  0.0 | -1.003777  -1.110375  -1.052105  0.0 | nan |  116.20473 | nan | 1181.614255 |  0.010123 |  571.540276 | 0.261089 |
    | 2017-04-05 00:00:00 |  -0.5 |  0.4 | 0.0 | 0.0 | 0.0 |  0.32 | 1.4535  1.989  1.071  1.9584 | 1.4535  1.989  1.071  0.0 |  0.014438   0.000001   0.000008  0.0 | 0.119666  0.145372  0.209579  0.0 | 0.005414       0.0  0.000003  0.0 |  0.009024   0.000001   0.000005  0.0 |  0.251908   0.251908   0.251908  0.0 |  0.002507   0.001518   0.001978  0.0 |  0.32 | 0.0 | 0.001132 |  0.004387 |  0.0 | 1.498583 |   1.9584 |  1.49456 | 0.0 | 0.0 | 0.0 |   52.858549 |  0.0 | -0.577827 |  0.011108 | 1.611453 | 0.186511 | -0.445796  -0.479186  -0.260793  0.0 | -0.997259  -1.111892  -1.054078  0.0 | nan | 117.124354 | nan | 1234.472804 |  0.003402 |  464.391323 | 0.186511 |
    | 2017-04-06 00:00:00 |  -0.8 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.4535  1.989  1.071  1.9584 | 1.4535  1.989  1.071  0.0 |       0.0        0.0        0.0  0.0 | 0.126256  0.143948  0.175627  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.096611   0.096611   0.096611  0.0 |  0.000976   0.000579   0.000754  0.0 |   0.0 | 0.0 | 0.000168 |  0.000684 |  0.0 | 1.494572 |   1.9584 | 1.491151 | 0.0 | 0.0 | 0.0 |   44.104976 |  0.0 | -0.383101 |  0.003346 | 1.199935 | 0.138881 | -0.572052  -0.623134   -0.43642  0.0 | -0.998235  -1.112471  -1.054832  0.0 | nan | 118.235657 | nan |  1278.57778 |   0.00074 |  385.705399 | 0.138881 |
    | 2017-04-07 00:00:00 |   0.1 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |  0.684  0.936  0.504  0.9216 |  0.684  0.936  0.504  0.0 |       0.0        0.0        0.0  0.0 | 0.042008   0.04567  0.053493  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  1.298601   1.298601   1.298601  0.0 |  0.012716   0.007641    0.00989  0.0 |   0.0 | 0.0 | 0.002218 |  0.008914 |  0.0 | 0.722182 |   0.9216 | 0.698767 | 0.0 | 0.0 | 0.0 |   32.640343 |  0.0 | -0.268641 |  0.005812 | 0.922239 | 0.106741 |  -0.61406  -0.668804  -0.489913  0.0 | -1.010951  -1.120113  -1.064722  0.0 | nan |  118.68698 | nan | 1311.218123 |  0.003842 |  327.109592 | 0.106741 |
    | 2017-04-08 00:00:00 |   0.6 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.1115  1.521  0.819  1.4976 | 1.1115  1.521  0.819  0.0 |       0.0        0.0        0.0  0.0 | 0.055218  0.058909  0.067481  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  3.184058   3.184058   3.184058  0.0 |  0.028378   0.017679   0.022514  0.0 |   0.0 | 0.0 | 0.005048 |  0.020038 |  0.0 | 1.187233 |   1.4976 | 1.133492 | 0.0 | 0.0 | 0.0 |   24.007533 |  0.0 | -0.193035 |  0.015007 | 0.727948 | 0.084253 | -0.669278  -0.727713  -0.557394  0.0 | -1.039329  -1.137792  -1.087236  0.0 | nan | 119.676131 | nan | 1335.225656 |  0.008873 |  281.437106 | 0.084253 |
    | 2017-04-09 00:00:00 |   2.5 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 2.1375  2.925  1.575    2.88 | 2.1375  2.925  1.575  0.0 |       0.0        0.0        0.0  0.0 | 0.078864  0.082512  0.091995  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 | 12.500323  12.500323  12.500323  0.0 |   0.08546   0.058198   0.071096  0.0 |   0.0 | 0.0 | 0.015907 |  0.061528 |  0.0 | 2.311243 |     2.88 | 2.175763 | 0.0 | 0.0 | 0.0 |   20.526848 |  0.0 | -0.141938 |    0.0439 | 0.586084 | 0.067834 | -0.748142  -0.810225  -0.649389  0.0 | -1.124788  -1.195991  -1.158332  0.0 | nan | 121.829529 | nan | 1355.752504 |  0.026501 |  245.144554 | 0.067834 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _wland_gd_backwater_effects:

backwater effects
_________________

>>> del model.dischargemodel
>>> with model.add_waterlevelmodel_v1("exch_waterlevel"):
...     pass
>>> from hydpy import Node
>>> waterlevel = Node("waterlevel", variable="WaterLevel")
>>> land = Element("land", outlets="outlet", receivers=waterlevel)
>>> land.model.connect()
>>> test.hydpy.update_devices(nodes=("outlet", waterlevel), elements=land)
>>> waterlevel.prepare_simseries()
>>> waterlevel.deploymode = "oldsim"
>>> waterlevel.sequences.sim.series = [
...     -1.955539, -1.593272, -1.554112, -1.514237, -1.474332, -1.434671, -1.237946,
...     2.127124, 1.721998, 3.728419, 18.207013, 25.802253, 71.102022, 99.231277,
...     90.580027, 87.096102, 104.142108, 247.067521, 291.608204, 251.76909,
...     213.036118, 180.529412, 163.8984, 149.153668, 131.288692, 115.204072,
...     102.991454, 98.139161, 87.168564, 77.587424, 69.085831, 62.643619, 60.153237,
...     56.181454, 122.382442, 155.691919, 147.940892, 148.543172, 321.621321,
...     368.151976, 308.494631, 253.669569, 211.316972, 178.790605, 153.349539,
...     133.098674, 153.5539, 254.793577, 673.736602, 1281.108691, 1200.030141,
...     859.04159, 626.665696, 481.755443, 387.029438, 320.183349, 271.859287,
...     234.668462, 205.397587]
>>> waterlevel.sequences.sim.series /= 1000.0

.. integration-test::

    >>> test("wland_gd_backwater_effects",
    ...      axis1=(fluxes.pc, states.sp), axis2=(inputs.t,))
    |                date |     t |    p | fxg | fxs |         dhs |    pc |                           pe |                       pet |                                   tf |                                ei |                                rf |                                   sf |                                   pm |                                   am |    ps | pve |       pv |        pq | etve |      etv |       es |       et |  gr | fxs | fxg |         cdg | fgse |       fgs |       fqs |        rh |         r |                                   ic |                                   sp | dve |         dv | hge |          dg |        hq |          hs |    outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 00:00:00 | -10.8 |  0.0 | 0.0 | 0.0 |    0.044461 |   0.0 | 0.3942  0.702  0.378  0.6588 | 0.3942  0.702  0.378  0.0 |       0.0        0.0        0.0  0.0 |      0.0  0.000001       0.0  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 |  0.49406 | 0.000067 | 0.435763 | 0.0 | 0.0 | 0.0 |  -37.374521 |  0.0 | -0.000081 |       0.0 | -0.000073 | -0.000008 |      -3.0  -3.000001       -3.0  0.0 |      -3.0       -3.0       -3.0  0.0 | nan | 140.493979 | nan | 1562.625479 |       0.0 |   -1.955539 | -0.000008 |
    | 2017-02-11 00:00:00 |  -9.5 |  0.4 | 0.0 | 0.0 |    0.362267 |  0.32 | 0.5256  0.936  0.504  0.8784 | 0.5256  0.936  0.504  0.0 |  0.000001        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |      0.0       0.0       0.0  0.0 |  0.000001        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.32 | 0.0 |      0.0 |       0.0 |  0.0 | 0.658704 |  0.00011 | 0.580981 | 0.0 | 0.0 | 0.0 |  -15.520339 |  0.0 | -0.000055 |       0.0 |  0.006349 |  0.000735 | -2.680002  -2.680003  -2.680002  0.0 | -2.999999       -3.0       -3.0  0.0 | nan | 141.152629 | nan | 1547.105141 |       0.0 |   -1.593272 |  0.000735 |
    | 2017-02-12 00:00:00 |  -8.9 |  0.0 | 0.0 | 0.0 |     0.03916 |   0.0 | 0.4599  0.819  0.441  0.7686 | 0.4599  0.819  0.441  0.0 |       0.0        0.0        0.0  0.0 | 0.000002  0.000004  0.000002  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.576325 | 0.000508 | 0.508331 | 0.0 | 0.0 | 0.0 |   -5.171808 |  0.0 | -0.000045 |       0.0 |  -0.00005 | -0.000006 | -2.680004  -2.680007  -2.680004  0.0 | -2.999999       -3.0       -3.0  0.0 | nan | 141.728908 | nan | 1541.933333 |       0.0 |   -1.554112 | -0.000006 |
    | 2017-02-13 00:00:00 |  -9.6 |  0.0 | 0.0 | 0.0 |    0.039875 |   0.0 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.329312 | 0.000347 | 0.290461 | 0.0 | 0.0 | 0.0 |   -0.904803 |  0.0 | -0.000043 |       0.0 | -0.000045 | -0.000005 | -2.680005  -2.680009  -2.680005  0.0 | -2.999999       -3.0       -3.0  0.0 | nan | 142.058177 | nan |  1541.02853 |       0.0 |   -1.514237 | -0.000005 |
    | 2017-02-14 00:00:00 |  -9.3 |  0.0 | 0.0 | 0.0 |    0.039905 |   0.0 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.329299 | 0.000417 | 0.290452 | 0.0 | 0.0 | 0.0 |    0.730083 |  0.0 | -0.000043 |       0.0 | -0.000046 | -0.000005 | -2.680007  -2.680011  -2.680006  0.0 | -2.999999       -3.0       -3.0  0.0 | nan | 142.387434 | nan | 1541.758613 |       0.0 |   -1.474332 | -0.000005 |
    | 2017-02-15 00:00:00 |  -6.3 |  0.0 | 0.0 | 0.0 |    0.039661 |   0.0 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000002  0.000001  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.329287 | 0.000501 | 0.290442 | 0.0 | 0.0 | 0.0 |    1.503674 |  0.0 | -0.000044 |       0.0 | -0.000049 | -0.000006 | -2.680008  -2.680013  -2.680007  0.0 | -2.999999       -3.0       -3.0  0.0 | nan | 142.716677 | nan | 1543.262287 |       0.0 |   -1.434671 | -0.000006 |
    | 2017-02-16 00:00:00 |  -3.6 |  0.2 | 0.0 | 0.0 |    0.196725 |  0.16 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |  0.000001        0.0        0.0  0.0 | 0.000002  0.000003  0.000002  0.0 |      0.0       0.0       0.0  0.0 |  0.000001        0.0        0.0  0.0 |  0.000008   0.000008   0.000008  0.0 |       0.0        0.0        0.0  0.0 |  0.16 | 0.0 |      0.0 |       0.0 |  0.0 | 0.329274 | 0.000601 | 0.290433 | 0.0 | 0.0 | 0.0 |    1.869899 |  0.0 | -0.000046 |       0.0 |  0.003148 |  0.000364 |  -2.52001  -2.520016  -2.520009  0.0 | -2.999999       -3.0       -3.0  0.0 | nan | 143.045905 | nan | 1545.132186 |       0.0 |   -1.237946 |  0.000364 |
    | 2017-02-17 00:00:00 |  -3.5 |  4.5 | 0.0 | 0.0 |     3.36507 |   3.6 | 0.1971  0.351  0.189  0.3294 | 0.1971  0.351  0.189  0.0 |  0.715908   0.000866   0.002554  0.0 | 0.045779  0.103421  0.056212  0.0 |      0.0       0.0       0.0  0.0 |  0.715908   0.000866   0.002554  0.0 |  0.000011   0.000011   0.000011  0.0 |       0.0        0.0        0.0  0.0 |   3.6 | 0.0 |      0.0 |       0.0 |  0.0 | 0.182336 | 0.001111 | 0.223675 | 0.0 | 0.0 | 0.0 |    1.826613 |  0.0 | -0.000048 |       0.0 |  0.071936 |  0.008326 |  0.318303   0.975698   1.021225  0.0 | -2.284091  -2.999134  -2.997446  0.0 | nan | 143.228193 | nan | 1546.958799 |       0.0 |    2.127124 |  0.008326 |
    | 2017-02-18 00:00:00 |  -4.6 |  0.0 | 0.0 | 0.0 |   -0.405126 |   0.0 | 0.1971  0.351  0.189  0.3294 | 0.1971  0.351  0.189  0.0 |       0.0        0.0        0.0  0.0 | 0.148002  0.341645    0.1863  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.035641 | 0.329381 | 0.243749 | 0.0 | 0.0 | 0.0 |    1.096193 |  0.0 | -0.000529 |       0.0 | -0.007054 | -0.000816 |  0.170301   0.634053   0.834925  0.0 | -2.284091  -2.999134  -2.997446  0.0 | nan | 143.263305 | nan | 1548.054993 |       0.0 |    1.721998 | -0.000816 |
    | 2017-02-19 00:00:00 |  -3.2 |  3.2 | 0.0 | 0.0 |    2.006421 |  2.56 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |  2.024091   0.452925   0.899571  0.0 | 0.221756   0.46542  0.251473  0.0 |      0.0       0.0       0.0  0.0 |  2.024091   0.452925   0.899571  0.0 |   0.00003    0.00003    0.00003  0.0 |  0.000001        0.0        0.0  0.0 |  2.56 | 0.0 |      0.0 |       0.0 |  0.0 | 0.028057 | 0.439039 | 0.325398 | 0.0 | 0.0 | 0.0 |    0.648358 |  0.0 | -0.000432 |       0.0 |  0.042038 |  0.004866 |  0.484454   2.275708    2.24388  0.0 |     -0.26   -2.54621  -2.097875  0.0 | nan | 143.290931 | nan | 1548.703351 |       0.0 |    3.728419 |  0.004866 |
    | 2017-02-20 00:00:00 |  -1.3 |  4.6 | 0.0 | 0.0 |   14.478594 |  3.68 | 0.1971  0.351  0.189  0.3294 | 0.1971  0.351  0.189  0.0 |  3.340123   2.972407   3.208228  0.0 | 0.184113  0.350996  0.188997  0.0 | 0.584521  0.520171   0.56144  0.0 |  2.755601   2.452236   2.646788  0.0 |  0.018374   0.018374   0.018374  0.0 |  0.016188   0.000752   0.003746  0.0 |  3.68 | 0.0 | 0.163232 |  0.426312 |  0.0 | 0.008608 |   0.3294 | 0.244154 | 0.0 | 0.0 | 0.0 |  -11.570748 |  0.0 | -0.000861 |  0.238151 |  0.299641 |  0.034681 |  0.640218   2.632305   2.526656  0.0 |  2.479413  -0.094726   0.545167  0.0 | nan | 143.135446 | nan | 1537.132602 |  0.188161 |   18.207013 |  0.034681 |
    | 2017-02-21 00:00:00 |  -2.2 |  2.3 | 0.0 | 0.0 |     7.59524 |  1.84 | 0.3942  0.702  0.378  0.6588 | 0.3942  0.702  0.378  0.0 |  1.642164   1.409265   1.618931  0.0 | 0.363811  0.701991  0.377995  0.0 |      0.0       0.0       0.0  0.0 |  1.642164   1.409265   1.618931  0.0 |  0.000888   0.000888   0.000888  0.0 |  0.000888   0.000771   0.000879  0.0 |  1.84 | 0.0 | 0.000241 |  0.000635 |  0.0 | 0.020144 |   0.6588 | 0.488293 | 0.0 | 0.0 | 0.0 |    5.558673 |  0.0 | -0.005322 |  0.164024 |  0.179673 |  0.020796 |  0.474243   2.361049    2.36973  0.0 |  4.120689   1.313768   2.163219  0.0 | nan | 143.150026 | nan | 1542.691275 |  0.024772 |   25.802253 |  0.020796 |
    | 2017-02-22 00:00:00 |  -1.5 | 18.0 | 0.0 | 0.0 |   45.299769 |  14.4 | 0.5256  0.936  0.504  0.8784 | 0.5256  0.936  0.504  0.0 | 13.589557  13.049009  13.564657  0.0 |   0.5047  0.935996  0.503997  0.0 | 1.698695  1.631126  1.695582  0.0 | 11.890862  11.417883  11.869075  0.0 |  0.009388   0.009388   0.009388  0.0 |  0.009388   0.009388   0.009388  0.0 |  14.4 | 0.0 | 0.477461 |  1.257786 |  0.0 | 0.013854 |   0.8784 | 0.651125 | 0.0 | 0.0 | 0.0 |  -32.098499 |  0.0 | -0.006571 |  0.729071 |  0.979127 |  0.113325 |  0.779986   2.776044   2.701076  0.0 | 16.002163  12.722263  14.022905  0.0 | nan | 142.679848 | nan | 1510.592776 |  0.553487 |   71.102022 |  0.113325 |
    | 2017-02-23 00:00:00 |  -3.0 | 19.2 | 0.0 | 0.0 |   28.129255 | 15.36 | 0.3285  0.585  0.315   0.549 | 0.3285  0.585  0.315  0.0 | 14.927197  14.657545  14.931625  0.0 | 0.322034  0.584999  0.314999  0.0 |      0.0       0.0       0.0  0.0 | 14.927197  14.657545  14.931625  0.0 |   0.00006    0.00006    0.00006  0.0 |   0.00006    0.00006    0.00006  0.0 | 15.36 | 0.0 | 0.000017 |  0.000045 |  0.0 | 0.004286 |    0.549 | 0.406976 | 0.0 | 0.0 | 0.0 |    15.94234 |  0.0 | -0.032246 |  0.478738 |  0.736942 |  0.085294 |  0.890755     2.8935   2.814452  0.0 |   30.9293  27.379748   28.95447  0.0 | nan | 142.651871 | nan | 1526.535115 |  0.074794 |   99.231277 |  0.085294 |
    | 2017-02-24 00:00:00 |  -5.0 |  0.4 | 0.0 | 0.0 |    -8.65125 |  0.32 | 0.5256  0.936  0.504  0.8784 | 0.5256  0.936  0.504  0.0 |  0.294507   0.228133   0.294805  0.0 | 0.495652  0.935979  0.503995  0.0 |      0.0       0.0       0.0  0.0 |  0.294507   0.228133   0.294805  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.32 | 0.0 |      0.0 |       0.0 |  0.0 | 0.019857 |   0.8784 | 0.651095 | 0.0 | 0.0 | 0.0 |    5.202262 |  0.0 |  -0.06387 |  0.063159 | -0.005605 | -0.000649 |  0.420596   2.049388   2.335652  0.0 | 31.223807  27.607881  29.249275  0.0 | nan | 142.607858 | nan | 1531.737377 |  0.011635 |   90.580027 | -0.000649 |
    | 2017-02-25 00:00:00 |  -4.9 |  8.3 | 0.0 | 0.0 |   -3.483925 |  6.64 | 0.3285  0.585  0.315   0.549 | 0.3285  0.585  0.315  0.0 |  6.067379   5.419977   6.057686  0.0 |  0.30829  0.584994  0.314997  0.0 |      0.0       0.0       0.0  0.0 |  6.067379   5.419977   6.057686  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |  6.64 | 0.0 |      0.0 |       0.0 |  0.0 | 0.013397 |    0.549 | 0.406929 | 0.0 | 0.0 | 0.0 |    2.876218 |  0.0 | -0.056142 |  0.010065 |  0.082167 |   0.00951 |  0.684927   2.684417    2.60297  0.0 | 37.291186  33.027858   35.30696  0.0 | nan | 142.565114 | nan | 1534.613595 |  0.001569 |   87.096102 |   0.00951 |
    | 2017-02-26 00:00:00 |  -0.9 |  5.3 | 0.0 | 0.0 |   17.046006 |  4.24 | 0.2628  0.468  0.252  0.4392 | 0.2628  0.468  0.252  0.0 |  3.990638   3.797561   3.989313  0.0 | 0.251901  0.467998  0.251998  0.0 | 1.097425  1.044329  1.097061  0.0 |  2.893212   2.753232   2.892252  0.0 |  0.069591   0.069591   0.069591  0.0 |  0.069591   0.069591   0.069591  0.0 |  4.24 | 0.0 | 0.323394 |  0.859996 |  0.0 | 0.007225 |   0.4392 | 0.325561 | 0.0 | 0.0 | 0.0 |  -21.568451 |  0.0 |  -0.04775 |  0.490165 |  0.514263 |  0.059521 |  0.682388   2.658858   2.601658  0.0 | 40.114807  35.711499  38.129622  0.0 | nan | 142.201194 | nan | 1513.045144 |    0.3714 |  104.142108 |  0.059521 |
    | 2017-02-27 00:00:00 |   1.4 |  0.7 | 0.0 | 0.0 |  142.925413 |  0.56 | 0.8541  1.521  0.819  1.4274 | 0.8541  1.521  0.819  0.0 |   0.42404   0.206933    0.40434  0.0 | 0.695506   1.52066  0.818961  0.0 | 0.360434  0.175893  0.343689  0.0 |  0.063606    0.03104   0.060651  0.0 |  7.013137   7.013137   7.013137  0.0 |  7.013137   7.013137   7.013137  0.0 |  0.56 | 0.0 | 2.027096 |  5.492149 |  0.0 | 0.105251 |   1.4274 |  1.05767 | 0.0 | 0.0 | 0.0 | -132.888694 |  0.0 | -0.021757 |  3.449265 |  3.343742 |  0.387007 |  0.122842   1.491265   1.938357  0.0 | 33.165276  28.729402  31.177135  0.0 | nan | 140.257593 | nan | 1380.156451 |  2.414284 |  247.067521 |  0.387007 |
    | 2017-02-28 00:00:00 |  -3.4 |  2.7 | 0.0 | 0.0 |   44.540683 |  2.16 | 0.5913  1.053  0.567  0.9882 | 0.5913  1.053  0.567  0.0 |  1.504649   0.475473   1.340867  0.0 |  0.45378  1.052728  0.566963  0.0 |      0.0       0.0       0.0  0.0 |  1.504649   0.475473   1.340867  0.0 |  0.000016   0.000016   0.000016  0.0 |  0.000016   0.000016   0.000016  0.0 |  2.16 | 0.0 | 0.000004 |  0.000012 |  0.0 | 0.091265 |   0.9882 | 0.732147 | 0.0 | 0.0 | 0.0 |   62.859052 |  0.0 | -0.201471 |  2.088128 |  1.892104 |  0.218994 |  0.324413   2.123064   2.190527  0.0 |  34.66991  29.204859  32.517986  0.0 | nan | 140.147383 | nan | 1443.015503 |  0.326168 |  291.608204 |  0.218994 |
    | 2017-03-01 00:00:00 |  -4.3 |  1.6 | 0.0 | 0.0 |  -39.839114 |  1.28 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |  0.950428   0.497016   0.875797  0.0 | 0.384004  0.818947  0.440979  0.0 |      0.0       0.0       0.0  0.0 |  0.950428   0.497016   0.875797  0.0 |  0.000001   0.000001   0.000001  0.0 |  0.000001   0.000001   0.000001  0.0 |  1.28 | 0.0 |      0.0 |  0.000001 |  0.0 | 0.067045 |   0.7938 | 0.584791 | 0.0 | 0.0 | 0.0 |   18.669325 |  0.0 | -0.357842 |  0.283714 | -0.027852 | -0.003224 |  0.269981     2.0871   2.153751  0.0 | 35.620337  29.701874  33.393783  0.0 | nan | 139.856586 | nan | 1461.684827 |  0.042454 |   251.76909 | -0.003224 |
    | 2017-03-02 00:00:00 |  -3.3 |  2.5 | 0.0 | 0.0 |  -38.732972 |   2.0 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |  1.531956   1.007064   1.460107  0.0 |  0.39375  0.818966  0.440983  0.0 |      0.0       0.0       0.0  0.0 |  1.531956   1.007064   1.460107  0.0 |  0.000022   0.000022   0.000022  0.0 |  0.000022   0.000022   0.000022  0.0 |   2.0 | 0.0 | 0.000006 |  0.000016 |  0.0 | 0.060578 |   0.7938 | 0.584824 | 0.0 | 0.0 | 0.0 |   13.729659 |  0.0 | -0.278686 |  0.036937 | -0.185479 | -0.021467 |  0.344275    2.26107   2.252661  0.0 | 37.152271  30.708916  34.853868  0.0 | nan | 139.638472 | nan | 1475.414486 |  0.005533 |  213.036118 | -0.021467 |
    | 2017-03-03 00:00:00 |  -2.1 |  0.6 | 0.0 | 0.0 |  -32.506706 |  0.48 | 0.7623  1.287  0.693  1.2474 | 0.7623  1.287  0.693  0.0 |  0.289031   0.089403    0.24232  0.0 | 0.505988  1.286381  0.692918  0.0 |      0.0       0.0       0.0  0.0 |  0.289031   0.089403    0.24232  0.0 |  0.001244   0.001244   0.001244  0.0 |  0.001244   0.001244   0.001244  0.0 |  0.48 | 0.0 | 0.000338 |   0.00094 |  0.0 |  0.17015 |   1.2474 | 0.918643 | 0.0 | 0.0 | 0.0 |   11.143584 |  0.0 | -0.207509 |  0.005344 | -0.193134 | -0.022353 |  0.029256   1.365286   1.797423  0.0 | 37.440057  30.797075  35.094943  0.0 | nan | 139.600775 | nan |  1486.55807 |  0.001129 |  180.529412 | -0.022353 |
    | 2017-03-04 00:00:00 |  -0.3 |  0.2 | 0.0 | 0.0 |  -16.631012 |  0.16 |  0.693   1.17   0.63   1.134 |  0.693   1.17   0.63  0.0 |  0.056049    0.00074   0.020472  0.0 | 0.287189  1.129305  0.629377  0.0 | 0.023821  0.000314  0.008701  0.0 |  0.032228   0.000425   0.011772  0.0 |   0.45975    0.45975    0.45975  0.0 |   0.45975    0.45975    0.45975  0.0 |  0.16 | 0.0 | 0.129197 |  0.358729 |  0.0 | 0.282559 |    1.134 | 0.834459 | 0.0 | 0.0 | 0.0 |    0.224083 |  0.0 | -0.150916 |  0.204087 |  0.047418 |  0.005488 | -0.153983   0.395241   1.307574  0.0 | 37.012536  30.337751  34.646965  0.0 | nan | 139.603221 | nan | 1486.782154 |  0.155771 |    163.8984 |  0.005488 |
    | 2017-03-05 00:00:00 |  -1.7 |  1.7 | 0.0 | 0.0 |  -14.744732 |  1.36 | 0.5544  0.936  0.504  0.9072 | 0.5544  0.936  0.504  0.0 |  0.699849   0.001279   0.261779  0.0 | 0.321401  0.881493   0.50364  0.0 | 0.052489  0.000096  0.019633  0.0 |  0.647361   0.001183   0.242145  0.0 |   0.00479    0.00479    0.00479  0.0 |   0.00479    0.00479    0.00479  0.0 |  1.36 | 0.0 | 0.010819 |  0.028538 |  0.0 | 0.172557 |   0.9072 | 0.667839 | 0.0 | 0.0 | 0.0 |   10.162666 |  0.0 | -0.128245 |  0.150361 |  0.043298 |  0.005011 |  0.184767   0.872469   1.902155  0.0 | 37.655106  30.334144   34.88432  0.0 | nan | 139.636714 | nan |  1496.94482 |  0.033948 |  149.153668 |  0.005011 |
    | 2017-03-06 00:00:00 |  -4.3 |  0.3 | 0.0 | 0.0 |  -17.864976 |  0.24 | 0.4158  0.702  0.378  0.6804 | 0.4158  0.702  0.378  0.0 |  0.131554   0.000231   0.068163  0.0 | 0.255521  0.664353  0.377898  0.0 |      0.0       0.0       0.0  0.0 |  0.131554   0.000231   0.068163  0.0 |  0.000001   0.000001   0.000001  0.0 |  0.000001   0.000001   0.000001  0.0 |  0.24 | 0.0 |      0.0 |  0.000001 |  0.0 | 0.118751 |   0.6804 | 0.500947 | 0.0 | 0.0 | 0.0 |    6.805808 |  0.0 | -0.111833 |  0.028668 | -0.079351 | -0.009184 |  0.037692   0.447885   1.696094  0.0 |  37.78666  30.334374  34.952483  0.0 | nan | 139.643632 | nan | 1503.750627 |  0.005281 |  131.288692 | -0.009184 |
    | 2017-03-07 00:00:00 |  -6.4 |  0.0 | 0.0 | 0.0 |   -16.08462 |   0.0 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |       0.0        0.0        0.0  0.0 | 0.202987  0.529811  0.440411  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.0 | 0.0 |      0.0 |       0.0 |  0.0 | 0.282919 |   0.7938 | 0.583692 | 0.0 | 0.0 | 0.0 |    5.103672 |  0.0 | -0.090492 |   0.00446 | -0.091319 | -0.010569 | -0.165295  -0.081926   1.255683  0.0 |  37.78666  30.334374  34.952483  0.0 | nan |  139.83606 | nan |   1508.8543 |  0.000822 |  115.204072 | -0.010569 |
    | 2017-03-08 00:00:00 |  -4.0 |  1.8 | 0.0 | 0.0 |  -12.212618 |  1.44 | 0.4851  0.819  0.441  0.7938 | 0.4851  0.819  0.441  0.0 |  0.768686   0.000464   0.301877  0.0 | 0.289895  0.646652  0.440662  0.0 |      0.0       0.0       0.0  0.0 |  0.768686   0.000464   0.301877  0.0 |  0.000002   0.000002   0.000002  0.0 |  0.000002   0.000002   0.000002  0.0 |  1.44 | 0.0 | 0.000001 |  0.000002 |  0.0 | 0.186559 |   0.7938 |  0.58418 | 0.0 | 0.0 | 0.0 |    4.336106 |  0.0 |  -0.07285 |  0.000715 | -0.050628 |  -0.00586 |  0.216125   0.710958   1.953144  0.0 | 38.555343  30.334836  35.254358  0.0 | nan | 139.949769 | nan | 1513.190406 |  0.000108 |  102.991454 |  -0.00586 |
    | 2017-03-09 00:00:00 |  -2.4 |  8.9 | 0.0 | 0.0 |   -4.852293 |  7.12 | 0.3465  0.585  0.315   0.567 | 0.3465  0.585  0.315  0.0 |  6.356504   4.562872   6.159351  0.0 | 0.317308  0.584263  0.314994  0.0 |      0.0       0.0       0.0  0.0 |  6.356504   4.562872   6.159351  0.0 |  0.000453   0.000453   0.000453  0.0 |  0.000453   0.000453   0.000453  0.0 |  7.12 | 0.0 | 0.000123 |  0.000341 |  0.0 | 0.019599 |    0.567 | 0.417846 | 0.0 | 0.0 | 0.0 |    2.940417 |  0.0 | -0.060694 |  0.000287 |  0.077809 |  0.009006 |  0.662313   2.683823   2.598799  0.0 | 44.911395  34.897256  41.413257  0.0 | nan |  139.90855 | nan | 1516.130823 |  0.000162 |   98.139161 |  0.009006 |
    | 2017-03-10 00:00:00 |  -2.2 |  0.0 | 0.0 | 0.0 |  -10.970597 |   0.0 | 0.5544  0.936  0.504  0.9072 | 0.5544  0.936  0.504  0.0 |       0.0        0.0        0.0  0.0 | 0.473416   0.93593  0.503987  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.000888   0.000888   0.000888  0.0 |  0.000888   0.000888   0.000888  0.0 |   0.0 | 0.0 | 0.000242 |   0.00067 |  0.0 | 0.053717 |   0.9072 | 0.668445 | 0.0 | 0.0 | 0.0 |    1.579078 |  0.0 | -0.056511 |  0.000524 | -0.067473 | -0.007809 |  0.188897   1.747893   2.094812  0.0 | 44.910507  34.896368  41.412368  0.0 | nan | 139.905514 | nan |   1517.7099 |  0.000308 |   87.168564 | -0.007809 |
    | 2017-03-11 00:00:00 |  -2.3 |  0.0 | 0.0 | 0.0 |    -9.58114 |   0.0 |  0.693   1.17   0.63   1.134 |  0.693   1.17   0.63  0.0 |       0.0        0.0        0.0  0.0 |  0.34233  1.153525  0.629746  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.000634   0.000634   0.000634  0.0 |  0.000634   0.000634   0.000634  0.0 |   0.0 | 0.0 | 0.000173 |  0.000478 |  0.0 | 0.237959 |    1.134 | 0.834701 | 0.0 | 0.0 | 0.0 |    1.826828 |  0.0 | -0.046127 |  0.000537 | -0.062837 | -0.007273 | -0.153432   0.594368   1.465066  0.0 | 44.909873  34.895734  41.411734  0.0 | nan | 140.097173 | nan | 1519.536728 |  0.000249 |   77.587424 | -0.007273 |
    | 2017-03-12 00:00:00 |  -3.4 |  0.0 | 0.0 | 0.0 |   -8.501593 |   0.0 | 0.8316  1.404  0.756  1.3608 | 0.8316  1.404  0.756  0.0 |       0.0        0.0        0.0  0.0 | 0.194653  0.815818  0.748189  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.000016   0.000016   0.000016  0.0 |  0.000016   0.000016   0.000016  0.0 |   0.0 | 0.0 | 0.000004 |  0.000012 |  0.0 | 0.617261 |   1.3608 | 0.999269 | 0.0 | 0.0 | 0.0 |    2.814985 |  0.0 |   -0.0382 |  0.000222 | -0.060691 | -0.007024 | -0.348085   -0.22145   0.716877  0.0 | 44.909857  34.895718  41.411719  0.0 | nan | 140.676229 | nan | 1522.351712 |  0.000039 |   69.085831 | -0.007024 |
    | 2017-03-13 00:00:00 |  -3.8 |  0.9 | 0.0 | 0.0 |   -6.442212 |  0.72 | 0.6237  1.053  0.567  1.0206 | 0.6237  1.053  0.567  0.0 |  0.198557   0.000023    0.00296  0.0 |  0.20712  0.453712  0.552688  0.0 |      0.0       0.0       0.0  0.0 |  0.198557   0.000023    0.00296  0.0 |  0.000004   0.000004   0.000004  0.0 |  0.000004   0.000004   0.000004  0.0 |  0.72 | 0.0 | 0.000001 |  0.000003 |  0.0 | 0.474816 |   1.0206 | 0.748541 | 0.0 | 0.0 | 0.0 |    3.715476 |  0.0 | -0.032248 |  0.000035 |  -0.03442 | -0.003984 | -0.033762   0.044815   0.881228  0.0 | 45.108411  34.895737  41.414675  0.0 | nan | 141.118796 | nan | 1526.067188 |  0.000007 |   62.643619 | -0.003984 |
    | 2017-03-14 00:00:00 |  -0.6 |  0.1 | 0.0 | 0.0 |   -2.490382 |  0.08 | 0.6237  1.053  0.567  1.0206 | 0.6237  1.053  0.567  0.0 |  0.023902   0.000002   0.000192  0.0 | 0.223804  0.377766  0.535205  0.0 | 0.008366  0.000001  0.000067  0.0 |  0.015536   0.000001   0.000125  0.0 |  0.184058   0.184058   0.184058  0.0 |  0.184058   0.184058   0.184058  0.0 |  0.08 | 0.0 | 0.052641 |  0.141707 |  0.0 | 0.488906 |   1.0206 | 0.746737 | 0.0 | 0.0 | 0.0 |   -0.242344 |  0.0 | -0.027688 |  0.082052 |  0.037178 |  0.004303 | -0.201468  -0.252953   0.425831  0.0 | 44.939889  34.711681  41.230742  0.0 | nan | 141.527373 | nan | 1525.824844 |  0.059662 |   60.153237 |  0.004303 |
    | 2017-03-15 00:00:00 |  -1.7 |  0.0 | 0.0 | 0.0 |   -3.971783 |   0.0 | 0.8316  1.404  0.756  1.3608 | 0.8316  1.404  0.756  0.0 |       0.0        0.0        0.0  0.0 | 0.171727  0.216661  0.491944  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.00479    0.00479    0.00479  0.0 |   0.00479    0.00479    0.00479  0.0 |   0.0 | 0.0 |  0.00134 |  0.003584 |  0.0 | 0.830931 |   1.3608 | 0.972981 | 0.0 | 0.0 | 0.0 |     5.78926 |  0.0 | -0.026741 |  0.052453 |  0.000601 |   0.00007 | -0.373195  -0.469614  -0.066112  0.0 | 44.935099  34.706891  41.225952  0.0 | nan | 142.330223 | nan | 1531.614104 |  0.010793 |   56.181454 |   0.00007 |
    | 2017-03-16 00:00:00 |   0.7 |  0.0 | 0.0 | 0.0 |   66.200988 |   0.0 | 0.9702  1.638  0.882  1.5876 | 0.9702  1.638  0.882  0.0 |       0.0        0.0        0.0  0.0 | 0.116275  0.128131  0.249381  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  3.633659   3.633659   3.633659  0.0 |  3.633659   3.633659   3.633659  0.0 |   0.0 | 0.0 | 1.021882 |  2.713965 |  0.0 | 1.066408 |   1.5876 | 1.102804 | 0.0 | 0.0 | 0.0 |  -66.980905 |  0.0 | -0.014059 |  1.551568 |  1.476384 |  0.170878 | -0.489469  -0.597745  -0.315493  0.0 |  41.30144  31.073232  37.592293  0.0 | nan | 142.360691 | nan | 1464.633199 |   1.17319 |  122.382442 |  0.170878 |
    | 2017-03-17 00:00:00 |  -1.6 |  3.9 | 0.0 | 0.0 |   33.309477 |  3.12 | 0.7623  1.287  0.693  1.2474 | 0.7623  1.287  0.693  0.0 |   1.81825   0.016204   0.251088  0.0 | 0.484947  0.989456  0.602616  0.0 | 0.181825   0.00162  0.025109  0.0 |  1.636425   0.014584   0.225979  0.0 |  0.006707   0.006707   0.006707  0.0 |  0.006707   0.006707   0.006707  0.0 |  3.12 | 0.0 | 0.036215 |  0.086205 |  0.0 | 0.282446 |   1.2474 | 0.909171 | 0.0 | 0.0 | 0.0 |   34.399536 |  0.0 | -0.065318 |  1.057165 |  1.015864 |  0.117577 |  0.327334   1.516595   1.950803  0.0 | 42.931159  31.081109  37.811566  0.0 | nan | 142.541605 | nan | 1499.032734 |  0.202231 |  155.691919 |  0.117577 |
    | 2017-03-18 00:00:00 |  -2.8 |  8.7 | 0.0 | 0.0 |   -7.751027 |  6.96 | 0.7623  1.287  0.693  1.2474 | 0.7623  1.287  0.693  0.0 |  6.071892    4.67193   5.753562  0.0 | 0.685998  1.286933  0.692984  0.0 |      0.0       0.0       0.0  0.0 |  6.071892    4.67193   5.753562  0.0 |  0.000117   0.000117   0.000117  0.0 |  0.000117   0.000117   0.000117  0.0 |  6.96 | 0.0 | 0.000033 |  0.000088 |  0.0 | 0.050597 |   1.2474 | 0.919212 | 0.0 | 0.0 | 0.0 |   14.730637 |  0.0 | -0.126982 |   0.17488 |  0.173636 |  0.020097 |  0.529444   2.517732   2.464257  0.0 | 49.002933  35.752922   43.56501  0.0 | nan | 142.465187 | nan | 1513.763372 |  0.027438 |  147.940892 |  0.020097 |
    | 2017-03-19 00:00:00 |  -2.9 | 26.4 | 0.0 | 0.0 |     0.60228 | 21.12 | 0.3465  0.585  0.315   0.567 | 0.3465  0.585  0.315  0.0 | 20.402852  20.094414   20.41026  0.0 | 0.337476  0.584999  0.314999  0.0 |      0.0       0.0       0.0  0.0 | 20.402852  20.094414   20.41026  0.0 |  0.000084   0.000084   0.000084  0.0 |  0.000084   0.000084   0.000084  0.0 | 21.12 | 0.0 | 0.000024 |  0.000063 |  0.0 | 0.005982 |    0.567 | 0.417911 | 0.0 | 0.0 | 0.0 |    6.980375 |  0.0 |  -0.12258 |  0.023737 |  0.326206 |  0.037755 |  0.909116   2.958319   2.858998  0.0 | 69.405702  55.847252  63.975186  0.0 | nan | 142.348566 | nan | 1520.743747 |  0.003764 |  148.543172 |  0.037755 |
    | 2017-03-20 00:00:00 |   0.7 | 11.5 | 0.0 | 0.0 |  173.078149 |   9.2 | 0.4158  0.702  0.378  0.6804 | 0.4158  0.702  0.378  0.0 |  8.931806   8.696853   8.951311  0.0 | 0.406533  0.701998  0.377999  0.0 | 6.028969  5.870376  6.042135  0.0 |  2.902837   2.826477   2.909176  0.0 |  3.633659   3.633659   3.633659  0.0 |  3.633659   3.633659   3.633659  0.0 |   9.2 | 0.0 | 2.656571 |  7.225453 |  0.0 | 0.006144 |   0.6804 |   0.5015 | 0.0 | 0.0 | 0.0 | -184.249081 |  0.0 | -0.048245 |  4.106767 |  4.152472 |   0.48061 |  0.770777   2.759468   2.729688  0.0 |  68.67488   55.04007  63.250704  0.0 | nan | 139.649894 | nan | 1336.494666 |   3.12245 |  321.621321 |   0.48061 |
    | 2017-03-21 00:00:00 |  -1.8 |  0.9 | 0.0 | 0.0 |   46.530655 |  0.72 | 1.0395  1.755  0.945   1.701 | 1.0395  1.755  0.945  0.0 |    0.5541   0.275967     0.5481  0.0 | 0.842841  1.754558  0.944963  0.0 | 0.027705  0.013798  0.027405  0.0 |  0.526395   0.262169   0.520695  0.0 |   0.00342    0.00342    0.00342  0.0 |   0.00342    0.00342    0.00342  0.0 |  0.72 | 0.0 |  0.00719 |  0.020452 |  0.0 |  0.13054 |    1.701 | 1.253194 | 0.0 | 0.0 | 0.0 |   73.614408 |  0.0 |  -0.32205 |  2.713382 |  2.355446 |  0.272621 |  0.093837   1.448942   1.956625  0.0 | 69.197854  55.298819  63.767978  0.0 | nan | 139.451194 | nan | 1410.109074 |  0.429521 |  368.151976 |  0.272621 |
    | 2017-03-22 00:00:00 |  -2.1 |  0.0 | 0.0 | 0.0 |  -59.657345 |   0.0 |  1.386   2.34   1.26   2.268 |  1.386   2.34   1.26  0.0 |       0.0        0.0        0.0  0.0 | 0.443283  1.689398  1.251505  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.001244   0.001244   0.001244  0.0 |  0.001244   0.001244   0.001244  0.0 |   0.0 | 0.0 | 0.000338 |  0.000941 |  0.0 | 0.840749 |    2.268 | 1.666881 | 0.0 | 0.0 | 0.0 |   21.481617 |  0.0 | -0.534702 |  0.372028 |  -0.15238 | -0.017637 | -0.349446  -0.240456    0.70512  0.0 |  69.19661  55.297575  63.766734  0.0 | nan | 139.756903 | nan | 1431.590691 |  0.058434 |  308.494631 | -0.017637 |
    | 2017-03-23 00:00:00 |  -2.8 |  0.0 | 0.0 | 0.0 |  -54.825062 |   0.0 | 1.1088  1.872  1.008  1.8144 | 1.1088  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.139053  0.268868   0.74957  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.000117   0.000117   0.000117  0.0 |  0.000117   0.000117   0.000117  0.0 |   0.0 | 0.0 | 0.000032 |  0.000089 |  0.0 | 1.174378 |   1.8144 | 1.306357 | 0.0 | 0.0 | 0.0 |   22.985039 |  0.0 | -0.390131 |  0.050878 | -0.330523 | -0.038255 | -0.488499  -0.509324  -0.044449  0.0 | 69.196492  55.297457  63.766616  0.0 | nan | 140.541118 | nan |  1454.57573 |  0.007644 |  253.669569 | -0.038255 |
    | 2017-03-24 00:00:00 |  -2.8 |  0.0 | 0.0 | 0.0 |  -42.352597 |   0.0 | 1.1088  1.872  1.008  1.8144 | 1.1088  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.087818  0.124642  0.285183  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.000117   0.000117   0.000117  0.0 |  0.000117   0.000117   0.000117  0.0 |   0.0 | 0.0 | 0.000032 |  0.000088 |  0.0 | 1.256034 |   1.8144 |  1.26034 | 0.0 | 0.0 | 0.0 |   22.096452 |  0.0 | -0.279618 |  0.006699 | -0.276346 | -0.031984 | -0.576317  -0.633967  -0.329632  0.0 | 69.196375   55.29734  63.766499  0.0 | nan | 141.517501 | nan | 1476.672182 |  0.001033 |  211.316972 | -0.031984 |
    | 2017-03-25 00:00:00 |  -2.1 |  0.0 | 0.0 | 0.0 |  -32.526367 |   0.0 | 0.8316  1.404  0.756  1.3608 | 0.8316  1.404  0.756  0.0 |       0.0        0.0        0.0  0.0 | 0.049328  0.062824  0.109412  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.001244   0.001244   0.001244  0.0 |  0.001244   0.001244   0.001244  0.0 |   0.0 | 0.0 | 0.000348 |  0.000931 |  0.0 | 0.963055 |   1.3608 | 0.934828 | 0.0 | 0.0 | 0.0 |   18.567243 |  0.0 | -0.209472 |  0.001425 | -0.210574 | -0.024372 | -0.625644  -0.696791  -0.439044  0.0 |  69.19513  55.296095  63.765254  0.0 | nan | 142.270737 | nan | 1495.239425 |  0.000539 |  178.790605 | -0.024372 |
    | 2017-03-26 00:00:00 |  -1.3 |  0.0 | 0.0 | 0.0 |  -25.441066 |   0.0 | 0.9009  1.521  0.819  1.4742 | 0.9009  1.521  0.819  0.0 |       0.0        0.0        0.0  0.0 | 0.043675  0.052786   0.08101  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.018374   0.018374   0.018374  0.0 |  0.018374   0.018374   0.018374  0.0 |   0.0 | 0.0 | 0.005192 |  0.013702 |  0.0 | 1.054745 |   1.4742 | 1.008908 | 0.0 | 0.0 | 0.0 |   14.761623 |  0.0 | -0.162752 |  0.008376 | -0.164823 | -0.019077 | -0.669319  -0.749576  -0.520055  0.0 | 69.176756  55.277721   63.74688  0.0 | nan | 143.157538 | nan | 1510.001048 |  0.005865 |  153.349539 | -0.019077 |
    | 2017-03-27 00:00:00 |  -1.0 |  0.0 | 0.0 | 0.0 |  -20.250865 |   0.0 | 1.1088  1.872  1.008  1.8144 | 1.1088  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.044306  0.051511  0.072303  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |      0.05       0.05       0.05  0.0 |      0.05       0.05       0.05  0.0 |   0.0 | 0.0 | 0.014309 |  0.037122 |  0.0 | 1.308712 |   1.8144 | 1.238854 | 0.0 | 0.0 | 0.0 |   12.359272 |  0.0 | -0.130262 |  0.026415 | -0.125293 | -0.014501 | -0.713626  -0.801087  -0.592357  0.0 | 69.126756  55.227721   63.69688  0.0 | nan | 144.321678 | nan |  1522.36032 |  0.016571 |  133.098674 | -0.014501 |
    | 2017-03-28 00:00:00 |   0.3 |  0.0 | 0.0 | 0.0 |   20.455226 |   0.0 | 1.3167  2.223  1.197  2.1546 | 1.3167  2.223  1.197  0.0 |       0.0        0.0        0.0  0.0 | 0.043337  0.048859   0.06406  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |   1.95975    1.95975    1.95975  0.0 |   1.95975    1.95975    1.95975  0.0 |   0.0 | 0.0 | 0.568045 |  1.448509 |  0.0 |  1.56413 |   2.1546 | 1.468779 | 0.0 | 0.0 | 0.0 |  -26.896987 |  0.0 | -0.093099 |  0.837818 |  0.695856 |  0.080539 | -0.756962  -0.849946  -0.656418  0.0 | 67.167006  53.267971   61.73713  0.0 | nan | 145.224663 | nan | 1495.463333 |  0.627262 |    153.5539 |  0.080539 |
    | 2017-03-29 00:00:00 |   1.0 |  1.5 | 0.0 | 0.0 |  101.239677 |   1.2 | 0.5544  0.936  0.504  0.9072 | 0.5544  0.936  0.504  0.0 |  0.261788   0.000022   0.000144  0.0 | 0.145029  0.238176  0.204206  0.0 | 0.196341  0.000017  0.000108  0.0 |  0.065447   0.000006   0.000036  0.0 |      5.05       5.05       5.05  0.0 |      5.05       5.05       5.05  0.0 |   1.2 | 0.0 | 1.500674 |  3.817214 |  0.0 | 0.502471 |   0.9072 | 0.636636 | 0.0 | 0.0 | 0.0 |   -78.73094 |  0.0 | -0.080928 |  2.694684 |  2.575268 |  0.298063 |  0.036221   0.111856   0.339232  0.0 | 62.182453  48.217977  56.687166  0.0 | nan | 144.145533 | nan | 1416.732392 |  1.749792 |  254.793577 |  0.298063 |
    | 2017-03-30 00:00:00 |   4.4 |  0.3 | 0.0 | 0.0 |  418.943025 |  0.24 | 1.0395  1.755  0.945   1.701 | 1.0395  1.755  0.945  0.0 |  0.076999   0.000006    0.00005  0.0 | 0.397318  0.624716  0.601935  0.0 | 0.076999  0.000006   0.00005  0.0 |       0.0        0.0        0.0  0.0 | 22.000001  22.000001  22.000001  0.0 | 22.000001  22.000001  22.000001  0.0 |  0.24 | 0.0 | 6.126546 | 16.532316 |  0.0 | 0.800362 |    1.701 | 1.216219 | 0.0 | 0.0 | 0.0 | -161.544409 |  0.0 | -0.062658 | 10.874162 | 10.572195 |  1.223634 | -0.198097  -0.272866  -0.022753  0.0 | 40.182453  26.217977  34.687166  0.0 | nan | 138.756691 | nan | 1255.187983 |  7.407946 |  673.736602 |  1.223634 |
    | 2017-03-31 00:00:00 |   7.0 |  0.2 | 0.0 | 0.0 |  607.372089 |  0.16 | 1.8711  3.159  1.701  3.0618 | 1.8711  3.159  1.701  0.0 |  0.023635   0.000001   0.000007  0.0 | 0.346987  0.403331  0.470525  0.0 | 0.023635  0.000001  0.000007  0.0 |       0.0        0.0        0.0  0.0 |      35.0       35.0       35.0  0.0 |      35.0  27.000525   34.73317  0.0 |  0.16 | 0.0 |  8.25291 | 25.160038 |  0.0 | 1.925111 |   3.0618 | 2.127903 | 0.0 | 0.0 | 0.0 |  -76.520325 |  0.0 | -1.261609 | 21.302812 | 19.705981 |  2.280785 | -0.408719  -0.516198  -0.333284  0.0 |  5.182453  -0.782549  -0.046004  0.0 | nan | 131.167284 | nan | 1178.667658 | 11.265171 | 1281.108691 |  2.280785 |
    | 2017-04-01 00:00:00 |   3.8 |  4.5 | 0.0 | 0.0 |   -81.07855 |   3.6 | 1.2825  1.755  0.945   1.728 | 1.2825  1.755  0.945  0.0 |  1.991552   0.031205    0.41364  0.0 | 0.874812  1.427509  0.831579  0.0 | 1.991552  0.031205   0.41364  0.0 |       0.0        0.0        0.0  0.0 | 19.000004  19.000004  19.000004  0.0 |  6.071338    0.26388   0.919082  0.0 |   3.6 | 0.0 | 1.288171 |  3.900178 |  0.0 | 0.379279 |    1.728 | 1.384656 | 0.0 | 0.0 | 0.0 |  -30.562687 |  0.0 |   -6.0275 | 12.526006 |   6.99667 |    0.8098 |  0.324917   1.625088   2.021496  0.0 | -0.888886  -1.046428  -0.965086  0.0 | nan | 124.230892 | nan | 1148.104971 |  2.639344 | 1200.030141 |    0.8098 |
    | 2017-04-02 00:00:00 |   1.4 |  0.0 | 0.0 | 0.0 | -340.988551 |   0.0 |  1.368  1.872  1.008  1.8432 |  1.368  1.872  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.610074  1.639535  1.005973  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  7.013137   7.013137   7.013137  0.0 |  0.093284   0.050482   0.069833  0.0 |   0.0 | 0.0 | 0.016804 |  0.062975 |  0.0 | 0.580566 |   1.8432 | 1.488255 | 0.0 | 0.0 | 0.0 |  -17.154119 |  0.0 | -5.054414 |  2.319369 | -2.221876 | -0.257162 | -0.285157  -0.014446   1.015522  0.0 | -0.982169  -1.096911   -1.03492  0.0 | nan | 119.740239 | nan | 1130.950852 |   0.38295 |   859.04159 | -0.257162 |
    | 2017-04-03 00:00:00 |   0.1 |  0.0 | 0.0 | 0.0 | -232.375894 |   0.0 |   1.71   2.34   1.26   2.304 |   1.71   2.34   1.26  0.0 |       0.0        0.0        0.0  0.0 | 0.227008  0.478293  1.029736  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  1.298601   1.298601   1.298601  0.0 |   0.01365   0.008194   0.010807  0.0 |   0.0 | 0.0 | 0.002417 |  0.009554 |  0.0 | 1.603371 |    2.304 | 1.835266 | 0.0 | 0.0 | 0.0 |   26.084835 |  0.0 | -2.193309 |  0.336581 | -1.650729 | -0.191057 | -0.512165  -0.492739  -0.014214  0.0 |  -0.99582  -1.105105  -1.045727  0.0 | nan | 119.147884 | nan | 1157.035688 |  0.055922 |  626.665696 | -0.191057 |
    | 2017-04-04 00:00:00 |  -0.1 |  0.0 | 0.0 | 0.0 | -144.910253 |   0.0 | 1.7955  2.457  1.323  2.4192 | 1.7955  2.457  1.323  0.0 |       0.0        0.0        0.0  0.0 | 0.119526  0.161077  0.357025  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.798601   0.798601   0.798601  0.0 |  0.007992   0.004891   0.006391  0.0 |   0.0 | 0.0 | 0.001425 |  0.005619 |  0.0 | 1.875742 |   2.4192 | 1.855415 | 0.0 | 0.0 | 0.0 |      46.208 |  0.0 | -0.974164 |  0.051553 | -0.857075 | -0.099198 | -0.631691  -0.653817  -0.371239  0.0 | -1.003812  -1.109996  -1.052117  0.0 | nan | 120.048036 | nan | 1203.243688 |  0.009988 |  481.755443 | -0.099198 |
    | 2017-04-05 00:00:00 |  -0.5 |  0.4 | 0.0 | 0.0 |  -94.726005 |  0.32 | 1.4535  1.989  1.071  1.9584 | 1.4535  1.989  1.071  0.0 |  0.014441   0.000001   0.000008  0.0 | 0.119686  0.145394  0.209574  0.0 | 0.005415       0.0  0.000003  0.0 |  0.009026   0.000001   0.000005  0.0 |  0.251908   0.251908   0.251908  0.0 |  0.002506    0.00152   0.001978  0.0 |  0.32 | 0.0 | 0.001204 |  0.004324 |  0.0 | 1.498144 |   1.9584 |  1.49419 | 0.0 | 0.0 | 0.0 |   58.485775 |  0.0 | -0.525816 |  0.011002 | -0.485755 | -0.056222 | -0.445818  -0.479212  -0.260821  0.0 | -0.997292  -1.111515   -1.05409  0.0 | nan |  121.01916 | nan | 1261.729463 |  0.003309 |  387.029438 | -0.056222 |
    | 2017-04-06 00:00:00 |  -0.8 |  0.0 | 0.0 | 0.0 |  -66.846089 |   0.0 | 1.4535  1.989  1.071  1.9584 | 1.4535  1.989  1.071  0.0 |       0.0        0.0        0.0  0.0 | 0.126247  0.143936  0.175613  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.096611   0.096611   0.096611  0.0 |  0.000976    0.00058   0.000754  0.0 |   0.0 | 0.0 | 0.000178 |  0.000675 |  0.0 | 1.494152 |   1.9584 |  1.49077 | 0.0 | 0.0 | 0.0 |   46.902684 |  0.0 | -0.338973 |   0.00326 | -0.334947 | -0.038767 | -0.572065  -0.623148  -0.436434  0.0 | -0.998268  -1.112096  -1.054844  0.0 | nan | 122.174161 | nan | 1308.632147 |  0.000724 |  320.183349 | -0.038767 |
    | 2017-04-07 00:00:00 |   0.1 |  0.0 | 0.0 | 0.0 |  -48.324062 |   0.0 |  0.684  0.936  0.504  0.9216 |  0.684  0.936  0.504  0.0 |       0.0        0.0        0.0  0.0 | 0.042006  0.045668   0.05349  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  1.298601   1.298601   1.298601  0.0 |  0.012714   0.007654   0.009889  0.0 |   0.0 | 0.0 | 0.002358 |  0.008791 |  0.0 | 0.721971 |   0.9216 | 0.698579 | 0.0 | 0.0 | 0.0 |    33.44236 |  0.0 | -0.236485 |  0.005628 | -0.221497 | -0.025636 |  -0.61407  -0.668816  -0.489924  0.0 | -1.010982   -1.11975  -1.064734  0.0 | nan | 122.657289 | nan | 1342.074507 |  0.003887 |  271.859287 | -0.025636 |
    | 2017-04-08 00:00:00 |   0.6 |  0.0 | 0.0 | 0.0 |  -37.190825 |   0.0 | 1.1115  1.521  0.819  1.4976 | 1.1115  1.521  0.819  0.0 |       0.0        0.0        0.0  0.0 | 0.055216  0.058906  0.067478  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  3.184058   3.184058   3.184058  0.0 |  0.028374   0.017708   0.022512  0.0 |   0.0 | 0.0 | 0.005366 |  0.019758 |  0.0 | 1.186879 |   1.4976 | 1.133177 | 0.0 | 0.0 | 0.0 |   23.687595 |  0.0 | -0.172645 |  0.014699 |  -0.16782 | -0.019424 | -0.669286  -0.727722  -0.557402  0.0 | -1.039356  -1.137457  -1.087246  0.0 | nan | 123.666156 | nan | 1365.762102 |  0.008947 |  234.668462 | -0.019424 |
    | 2017-04-09 00:00:00 |   2.5 |  0.0 | 0.0 | 0.0 |  -29.270875 |   0.0 | 2.1375  2.925  1.575    2.88 | 2.1375  2.925  1.575  0.0 |       0.0        0.0        0.0  0.0 | 0.078862  0.082509  0.091992  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 | 12.500323  12.500323  12.500323  0.0 |  0.085451   0.058277   0.071093  0.0 |   0.0 | 0.0 | 0.016897 |  0.060656 |  0.0 | 2.310526 |     2.88 | 2.175128 | 0.0 | 0.0 | 0.0 |   19.788729 |  0.0 | -0.129945 |  0.044276 | -0.128822 |  -0.01491 | -0.748148  -0.810231  -0.649395  0.0 | -1.124807  -1.195734  -1.158339  0.0 | nan |  125.82984 | nan | 1385.550831 |  0.025327 |  205.397587 |  -0.01491 |

>>> round_(model.check_waterbalance(conditions))
0.0

>>> del model.waterlevelmodel
>>> with model.add_dischargemodel_v2("wq_walrus"):
...     crestheight(0.0)
...     bankfulldischarge(8.0)
...     dischargeexponent(1.5)

.. _wland_gd_no_vadose_zone:

no vadose zone
______________

>>> lt(SEALED, SEALED, SEALED, WATER)
>>> assert lt.value[0] != model.petmodel.parameters.control.hrutype.values[0]  # ToDo
>>> model.petmodel.prepare_zonetypes(lt.values)

.. integration-test::

    >>> test("wland_gd_no_vadose_zone",
    ...      axis1=(fluxes.pc, fluxes.fqs, fluxes.fgs, fluxes.rh),
    ...      axis2=(states.dg, states.hs))
    |                date |     t |    p | fxg | fxs | dhs |    pc |                          pe |                      pet |                                   tf |                                ei |                                rf |                                   sf |                                   pm |                                   am |    ps | pve |  pv |        pq | etve | etv |       es |       et |  gr | fxs | fxg | cdg | fgse | fgs |       fqs |        rh |        r |                                   ic |                                   sp | dve |  dv | hge |  dg |        hq |          hs |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 00:00:00 | -10.8 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.378  0.378  0.378  0.6588 | 0.378  0.378  0.378  0.0 |       0.0        0.0        0.0  0.0 |      0.0       0.0       0.0  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.0 | 0.0 | 0.0 |       0.0 |  0.0 | 0.0 | 0.000067 | 0.000002 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |       0.0 |       0.0 |      0.0 |      -3.0       -3.0       -3.0  0.0 |      -3.0       -3.0       -3.0  0.0 | nan | nan | nan | nan |       0.0 |   -2.000067 |      0.0 |
    | 2017-02-11 00:00:00 |  -9.5 |  0.4 | 0.0 | 0.0 | 0.0 |  0.32 | 0.504  0.504  0.504  0.8784 | 0.504  0.504  0.504  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000001  0.000001  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.32 | 0.0 | 0.0 |       0.0 |  0.0 | 0.0 |  0.00024 | 0.000006 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |       0.0 |       0.0 |      0.0 | -2.680002  -2.680002  -2.680002  0.0 |      -3.0       -3.0       -3.0  0.0 | nan | nan | nan | nan |       0.0 |   -1.680307 |      0.0 |
    | 2017-02-12 00:00:00 |  -8.9 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.441  0.441  0.441  0.7686 | 0.441  0.441  0.441  0.0 |       0.0        0.0        0.0  0.0 | 0.000002  0.000002  0.000002  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.0 | 0.0 | 0.0 |       0.0 |  0.0 | 0.0 |  0.00034 | 0.000009 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |       0.0 |       0.0 |      0.0 | -2.680004  -2.680004  -2.680004  0.0 |      -3.0       -3.0       -3.0  0.0 | nan | nan | nan | nan |       0.0 |   -1.680647 |      0.0 |
    | 2017-02-13 00:00:00 |  -9.6 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.252  0.252  0.252  0.4392 | 0.252  0.252  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000001  0.000001  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.0 | 0.0 | 0.0 |       0.0 |  0.0 | 0.0 | 0.000194 | 0.000005 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |       0.0 |       0.0 |      0.0 | -2.680005  -2.680005  -2.680005  0.0 |      -3.0       -3.0       -3.0  0.0 | nan | nan | nan | nan |       0.0 |   -1.680842 |      0.0 |
    | 2017-02-14 00:00:00 |  -9.3 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.252  0.252  0.252  0.4392 | 0.252  0.252  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000001  0.000001  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.0 | 0.0 | 0.0 |       0.0 |  0.0 | 0.0 | 0.000194 | 0.000005 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |       0.0 |       0.0 |      0.0 | -2.680006  -2.680006  -2.680006  0.0 |      -3.0       -3.0       -3.0  0.0 | nan | nan | nan | nan |       0.0 |   -1.681036 |      0.0 |
    | 2017-02-15 00:00:00 |  -6.3 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.252  0.252  0.252  0.4392 | 0.252  0.252  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000001  0.000001  0.000001  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.0 | 0.0 | 0.0 |       0.0 |  0.0 | 0.0 | 0.000194 | 0.000005 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |       0.0 |       0.0 |      0.0 | -2.680007  -2.680007  -2.680007  0.0 |      -3.0       -3.0       -3.0  0.0 | nan | nan | nan | nan |       0.0 |    -1.68123 |      0.0 |
    | 2017-02-16 00:00:00 |  -3.6 |  0.2 | 0.0 | 0.0 | 0.0 |  0.16 | 0.252  0.252  0.252  0.4392 | 0.252  0.252  0.252  0.0 |       0.0        0.0        0.0  0.0 | 0.000002  0.000002  0.000002  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.000008   0.000008   0.000008  0.0 |       0.0        0.0        0.0  0.0 |  0.16 | 0.0 | 0.0 |       0.0 |  0.0 | 0.0 | 0.000299 | 0.000008 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |       0.0 |       0.0 |      0.0 | -2.520009  -2.520009  -2.520009  0.0 |      -3.0       -3.0       -3.0  0.0 | nan | nan | nan | nan |       0.0 |   -1.521529 |      0.0 |
    | 2017-02-17 00:00:00 |  -3.5 |  4.5 | 0.0 | 0.0 | 0.0 |   3.6 | 0.189  0.189  0.189  0.3294 | 0.189  0.189  0.189  0.0 |  0.002575   0.002575   0.002575  0.0 |  0.05696   0.05696   0.05696  0.0 |      0.0       0.0       0.0  0.0 |  0.002575   0.002575   0.002575  0.0 |  0.000011   0.000011   0.000011  0.0 |       0.0        0.0        0.0  0.0 |   3.6 | 0.0 | 0.0 |       0.0 |  0.0 | 0.0 | 0.190061 | 0.059622 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |       0.0 |  0.000086 |  0.00001 |  1.020457   1.020457   1.020457  0.0 | -2.997425  -2.997425  -2.997425  0.0 | nan | nan | nan | nan |       0.0 |    1.884092 |  0.00001 |
    | 2017-02-18 00:00:00 |  -4.6 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.189  0.189  0.189  0.3294 | 0.189  0.189  0.189  0.0 |       0.0        0.0        0.0  0.0 | 0.186128  0.186128  0.186128  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.0 | 0.0 | 0.0 |       0.0 |  0.0 | 0.0 |  0.32923 |  0.18899 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |       0.0 |   0.00031 | 0.000036 |  0.834329   0.834329   0.834329  0.0 | -2.997425  -2.997425  -2.997425  0.0 | nan | nan | nan | nan |       0.0 |    1.539384 | 0.000036 |
    | 2017-02-19 00:00:00 |  -3.2 |  3.2 | 0.0 | 0.0 | 0.0 |  2.56 | 0.252  0.252  0.252  0.4392 | 0.252  0.252  0.252  0.0 |  0.899276   0.899276   0.899276  0.0 | 0.251443  0.251443  0.251443  0.0 |      0.0       0.0       0.0  0.0 |  0.899276   0.899276   0.899276  0.0 |   0.00003    0.00003    0.00003  0.0 |       0.0        0.0        0.0  0.0 |  2.56 | 0.0 | 0.0 |       0.0 |  0.0 | 0.0 |  0.43916 | 0.255198 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |       0.0 |  0.000585 | 0.000068 |  2.243609   2.243609   2.243609  0.0 | -2.098149  -2.098149  -2.098149  0.0 | nan | nan | nan | nan |       0.0 |    3.630963 | 0.000068 |
    | 2017-02-20 00:00:00 |  -1.3 |  4.6 | 0.0 | 0.0 | 0.0 |  3.68 | 0.189  0.189  0.189  0.3294 | 0.189  0.189  0.189  0.0 |  3.208267   3.208267   3.208267  0.0 | 0.188997  0.188997  0.188997  0.0 | 0.561447  0.561447  0.561447  0.0 |   2.64682    2.64682    2.64682  0.0 |  0.018374   0.018374   0.018374  0.0 |  0.003734   0.003734   0.003734  0.0 |  3.68 | 0.0 | 0.0 |  0.565181 |  0.0 | 0.0 |   0.3294 | 0.191805 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.312595 |  0.005503 | 0.000637 |  2.526345   2.526345   2.526345  0.0 |  0.544937   0.544937   0.544937  0.0 | nan | nan | nan | nan |  0.252586 |   22.023581 | 0.000637 |
    | 2017-02-21 00:00:00 |  -2.2 |  2.3 | 0.0 | 0.0 | 0.0 |  1.84 | 0.378  0.378  0.378  0.6588 | 0.378  0.378  0.378  0.0 |  1.618792   1.618792   1.618792  0.0 | 0.377995  0.377995  0.377995  0.0 |      0.0       0.0       0.0  0.0 |  1.618792   1.618792   1.618792  0.0 |  0.000888   0.000888   0.000888  0.0 |  0.000879   0.000879   0.000879  0.0 |  1.84 | 0.0 | 0.0 |  0.000879 |  0.0 | 0.0 |   0.6588 | 0.383611 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.219014 |   0.02176 | 0.002519 |  2.369558   2.369558   2.369558  0.0 |   2.16285    2.16285    2.16285  0.0 | nan | nan | nan | nan |  0.034451 |    32.84847 | 0.002519 |
    | 2017-02-22 00:00:00 |  -1.5 | 18.0 | 0.0 | 0.0 | 0.0 |  14.4 | 0.504  0.504  0.504  0.8784 | 0.504  0.504  0.504  0.0 | 13.564241  13.564241  13.564241  0.0 | 0.503997  0.503997  0.503997  0.0 |  1.69553   1.69553   1.69553  0.0 | 11.868711  11.868711  11.868711  0.0 |  0.009388   0.009388   0.009388  0.0 |  0.009388   0.009388   0.009388  0.0 |  14.4 | 0.0 | 0.0 |  1.704919 |  0.0 | 0.0 |   0.8784 | 0.511485 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.989471 |  0.061663 | 0.007137 |   2.70132    2.70132    2.70132  0.0 | 14.022173  14.022173  14.022173  0.0 | nan | nan | nan | nan |  0.749898 |   91.771019 | 0.007137 |
    | 2017-02-23 00:00:00 |  -3.0 | 19.2 | 0.0 | 0.0 | 0.0 | 15.36 | 0.315  0.315  0.315   0.549 | 0.315  0.315  0.315  0.0 | 14.931774  14.931774  14.931774  0.0 | 0.314999  0.314999  0.314999  0.0 |      0.0       0.0       0.0  0.0 | 14.931774  14.931774  14.931774  0.0 |   0.00006    0.00006    0.00006  0.0 |   0.00006    0.00006    0.00006  0.0 | 15.36 | 0.0 | 0.0 |   0.00006 |  0.0 | 0.0 |    0.549 | 0.319679 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.648952 |  0.172668 | 0.019985 |  2.814547   2.814547   2.814547  0.0 | 28.953887  28.953887  28.953887  0.0 | nan | nan | nan | nan |  0.101005 |  129.747281 | 0.019985 |
    | 2017-02-24 00:00:00 |  -5.0 |  0.4 | 0.0 | 0.0 | 0.0 |  0.32 | 0.504  0.504  0.504  0.8784 | 0.504  0.504  0.504  0.0 |  0.294814   0.294814   0.294814  0.0 | 0.503995  0.503995  0.503995  0.0 |      0.0       0.0       0.0  0.0 |  0.294814   0.294814   0.294814  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.32 | 0.0 | 0.0 |       0.0 |  0.0 | 0.0 |   0.8784 | 0.511484 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.085293 |  0.197859 |   0.0229 |  2.335738   2.335738   2.335738  0.0 | 29.248701  29.248701  29.248701  0.0 | nan | nan | nan | nan |  0.015712 |  123.475311 |   0.0229 |
    | 2017-02-25 00:00:00 |  -4.9 |  8.3 | 0.0 | 0.0 | 0.0 |  6.64 | 0.315  0.315  0.315   0.549 | 0.315  0.315  0.315  0.0 |  6.056957   6.056957   6.056957  0.0 | 0.314997  0.314997  0.314997  0.0 |      0.0       0.0       0.0  0.0 |  6.056957   6.056957   6.056957  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |  6.64 | 0.0 | 0.0 |       0.0 |  0.0 | 0.0 |    0.549 | 0.319677 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.013676 |  0.186196 |  0.02155 |  2.603784   2.603784   2.603784  0.0 | 35.305657  35.305657  35.305657  0.0 | nan | nan | nan | nan |  0.002036 |   120.92665 |  0.02155 |
    | 2017-02-26 00:00:00 |  -0.9 |  5.3 | 0.0 | 0.0 | 0.0 |  4.24 | 0.252  0.252  0.252  0.4392 | 0.252  0.252  0.252  0.0 |  3.989852   3.989852   3.989852  0.0 | 0.251998  0.251998  0.251998  0.0 | 1.097209  1.097209  1.097209  0.0 |  2.892643   2.892643   2.892643  0.0 |  0.069591   0.069591   0.069591  0.0 |  0.069591   0.069591   0.069591  0.0 |  4.24 | 0.0 | 0.0 |    1.1668 |  0.0 | 0.0 |   0.4392 | 0.255742 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.663756 |  0.205341 | 0.023766 |  2.601934   2.601934   2.601934  0.0 | 38.128709  38.128709  38.128709  0.0 | nan | nan | nan | nan |   0.50508 |  146.984456 | 0.023766 |
    | 2017-02-27 00:00:00 |   1.4 |  0.7 | 0.0 | 0.0 | 0.0 |  0.56 | 0.819  0.819  0.819  1.4274 | 0.819  0.819  0.819  0.0 |  0.404441   0.404441   0.404441  0.0 | 0.818961  0.818961  0.818961  0.0 | 0.343774  0.343774  0.343774  0.0 |  0.060666   0.060666   0.060666  0.0 |  7.013137   7.013137   7.013137  0.0 |  7.013137   7.013137   7.013137  0.0 |  0.56 | 0.0 | 0.0 |  7.356912 |  0.0 | 0.0 |   1.4274 |  0.83113 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  4.631053 |   0.49186 | 0.056928 |  1.938532   1.938532   1.938532  0.0 | 31.176238  31.176238  31.176238  0.0 | nan | nan | nan | nan |  3.230939 |  348.445657 | 0.056928 |
    | 2017-02-28 00:00:00 |  -3.4 |  2.7 | 0.0 | 0.0 | 0.0 |  2.16 | 0.567  0.567  0.567  0.9882 | 0.567  0.567  0.567  0.0 |   1.34089    1.34089    1.34089  0.0 | 0.566963  0.566963  0.566963  0.0 |      0.0       0.0       0.0  0.0 |   1.34089    1.34089    1.34089  0.0 |  0.000016   0.000016   0.000016  0.0 |  0.000016   0.000016   0.000016  0.0 |  2.16 | 0.0 | 0.0 |  0.000016 |  0.0 | 0.0 |   0.9882 | 0.575388 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  2.793519 |  1.150341 | 0.133141 |  2.190679   2.190679   2.190679  0.0 | 32.517112  32.517112  32.517112  0.0 | nan | nan | nan | nan |  0.437436 |  428.982847 | 0.133141 |
    | 2017-03-01 00:00:00 |  -4.3 |  1.6 | 0.0 | 0.0 | 0.0 |  1.28 | 0.441  0.441  0.441  0.7938 | 0.441  0.441  0.441  0.0 |  0.875908   0.875908   0.875908  0.0 | 0.440979  0.440979  0.440979  0.0 |      0.0       0.0       0.0  0.0 |  0.875908   0.875908   0.875908  0.0 |  0.000001   0.000001   0.000001  0.0 |  0.000001   0.000001   0.000001  0.0 |  1.28 | 0.0 | 0.0 |  0.000001 |  0.0 | 0.0 |   0.7938 | 0.448036 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.378436 |   1.15153 | 0.133279 |  2.153792   2.153792   2.153792  0.0 | 33.393019  33.393019  33.393019  0.0 | nan | nan | nan | nan |     0.059 |  390.435891 | 0.133279 |
    | 2017-03-02 00:00:00 |  -3.3 |  2.5 | 0.0 | 0.0 | 0.0 |   2.0 | 0.441  0.441  0.441  0.7938 | 0.441  0.441  0.441  0.0 |  1.460731   1.460731   1.460731  0.0 | 0.440983  0.440983  0.440983  0.0 |      0.0       0.0       0.0  0.0 |  1.460731   1.460731   1.460731  0.0 |  0.000022   0.000022   0.000022  0.0 |  0.000022   0.000022   0.000022  0.0 |   2.0 | 0.0 | 0.0 |  0.000022 |  0.0 | 0.0 |   0.7938 |  0.44804 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.049835 |  0.971353 | 0.112425 |  2.252077   2.252077   2.252077  0.0 | 34.853729  34.853729  34.853729  0.0 | nan | nan | nan | nan |  0.009187 |  345.516369 | 0.112425 |
    | 2017-03-03 00:00:00 |  -2.1 |  0.6 | 0.0 | 0.0 | 0.0 |  0.48 | 0.693  0.693  0.693  1.2474 | 0.693  0.693  0.693  0.0 |  0.242299   0.242299   0.242299  0.0 | 0.692918  0.692918  0.692918  0.0 |      0.0       0.0       0.0  0.0 |  0.242299   0.242299   0.242299  0.0 |  0.001244   0.001244   0.001244  0.0 |  0.001244   0.001244   0.001244  0.0 |  0.48 | 0.0 | 0.0 |  0.001244 |  0.0 | 0.0 |   1.2474 | 0.704008 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.006954 |  0.805985 | 0.093285 |   1.79686    1.79686    1.79686  0.0 | 35.094783  35.094783  35.094783  0.0 | nan | nan | nan | nan |  0.003477 |   304.79046 | 0.093285 |
    | 2017-03-04 00:00:00 |  -0.3 |  0.2 | 0.0 | 0.0 | 0.0 |  0.16 |  0.63   0.63   0.63   1.134 |  0.63   0.63   0.63  0.0 |  0.020512   0.020512   0.020512  0.0 | 0.629363  0.629363  0.629363  0.0 | 0.008717  0.008717  0.008717  0.0 |  0.011794   0.011794   0.011794  0.0 |   0.45975    0.45975    0.45975  0.0 |   0.45975    0.45975    0.45975  0.0 |  0.16 | 0.0 | 0.0 |  0.468467 |  0.0 | 0.0 |    1.134 | 0.639455 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.271901 |  0.686722 | 0.079482 |  1.306986   1.306986   1.306986  0.0 | 34.646828  34.646828  34.646828  0.0 | nan | nan | nan | nan |  0.200043 |  282.803486 | 0.079482 |
    | 2017-03-05 00:00:00 |  -1.7 |  1.7 | 0.0 | 0.0 | 0.0 |  1.36 | 0.504  0.504  0.504  0.9072 | 0.504  0.504  0.504  0.0 |  0.260975   0.260975   0.260975  0.0 | 0.503638  0.503638  0.503638  0.0 | 0.019573  0.019573  0.019573  0.0 |  0.241402   0.241402   0.241402  0.0 |   0.00479    0.00479    0.00479  0.0 |   0.00479    0.00479    0.00479  0.0 |  1.36 | 0.0 | 0.0 |  0.024363 |  0.0 | 0.0 |   0.9072 | 0.511709 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.184323 |  0.621006 | 0.071876 |  1.902373   1.902373   1.902373  0.0 | 34.883439  34.883439  34.883439  0.0 | nan | nan | nan | nan |  0.040083 |  261.237836 | 0.071876 |
    | 2017-03-06 00:00:00 |  -4.3 |  0.3 | 0.0 | 0.0 | 0.0 |  0.24 | 0.378  0.378  0.378  0.6804 | 0.378  0.378  0.378  0.0 |  0.068207   0.068207   0.068207  0.0 | 0.377898  0.377898  0.377898  0.0 |      0.0       0.0       0.0  0.0 |  0.068207   0.068207   0.068207  0.0 |  0.000001   0.000001   0.000001  0.0 |  0.000001   0.000001   0.000001  0.0 |  0.24 | 0.0 | 0.0 |  0.000001 |  0.0 | 0.0 |   0.6804 | 0.383948 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.033848 |  0.539065 | 0.062392 |  1.696269   1.696269   1.696269  0.0 | 34.951645  34.951645  34.951645  0.0 | nan | nan | nan | nan |  0.006235 |  235.502773 | 0.062392 |
    | 2017-03-07 00:00:00 |  -6.4 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.441  0.441  0.441  0.7938 | 0.441  0.441  0.441  0.0 |       0.0        0.0        0.0  0.0 | 0.440409  0.440409  0.440409  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.0 | 0.0 | 0.0 |       0.0 |  0.0 | 0.0 |   0.7938 | 0.447476 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.008314 |  0.460003 | 0.053241 |   1.25586    1.25586    1.25586  0.0 | 34.951645  34.951645  34.951645  0.0 | nan | nan | nan | nan | -0.002078 |  212.116224 | 0.053241 |
    | 2017-03-08 00:00:00 |  -4.0 |  1.8 | 0.0 | 0.0 | 0.0 |  1.44 | 0.441  0.441  0.441  0.7938 | 0.441  0.441  0.441  0.0 |  0.301973   0.301973   0.301973  0.0 | 0.440654  0.440654  0.440654  0.0 |      0.0       0.0       0.0  0.0 |  0.301973   0.301973   0.301973  0.0 |  0.000002   0.000002   0.000002  0.0 |  0.000002   0.000002   0.000002  0.0 |  1.44 | 0.0 | 0.0 |  0.000002 |  0.0 | 0.0 |   0.7938 | 0.447717 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | -0.001793 |   0.39616 | 0.045852 |  1.953233   1.953233   1.953233  0.0 | 35.253616  35.253616  35.253616  0.0 | nan | nan | nan | nan | -0.000284 |  192.866568 | 0.045852 |
    | 2017-03-09 00:00:00 |  -2.4 |  8.9 | 0.0 | 0.0 | 0.0 |  7.12 | 0.315  0.315  0.315   0.567 | 0.315  0.315  0.315  0.0 |  6.159253   6.159253   6.159253  0.0 | 0.314994  0.314994  0.314994  0.0 |      0.0       0.0       0.0  0.0 |  6.159253   6.159253   6.159253  0.0 |  0.000453   0.000453   0.000453  0.0 |  0.000453   0.000453   0.000453  0.0 |  7.12 | 0.0 | 0.0 |  0.000453 |  0.0 | 0.0 |    0.567 | 0.320034 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.000011 |   0.35272 | 0.040824 |  2.598986   2.598986   2.598986  0.0 | 41.412416  41.412416  41.412416  0.0 | nan | nan | nan | nan |  0.000157 |  181.784116 | 0.040824 |
    | 2017-03-10 00:00:00 |  -2.2 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.504  0.504  0.504  0.9072 | 0.504  0.504  0.504  0.0 |       0.0        0.0        0.0  0.0 | 0.503987  0.503987  0.503987  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.000888   0.000888   0.000888  0.0 |  0.000888   0.000888   0.000888  0.0 |   0.0 | 0.0 | 0.0 |  0.000888 |  0.0 | 0.0 |   0.9072 | 0.512051 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.000506 |  0.314282 | 0.036375 |  2.094999   2.094999   2.094999  0.0 | 41.411528  41.411528  41.411528  0.0 | nan | nan | nan | nan |   0.00054 |  165.187617 | 0.036375 |
    | 2017-03-11 00:00:00 |  -2.3 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |  0.63   0.63   0.63   1.134 |  0.63   0.63   0.63  0.0 |       0.0        0.0        0.0  0.0 | 0.629751  0.629751  0.629751  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.000634   0.000634   0.000634  0.0 |  0.000634   0.000634   0.000634  0.0 |   0.0 | 0.0 | 0.0 |  0.000634 |  0.0 | 0.0 |    1.134 | 0.639836 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.000931 |  0.272705 | 0.031563 |  1.465249   1.465249   1.465249  0.0 | 41.410894  41.410894  41.410894  0.0 | nan | nan | nan | nan |  0.000243 |  150.463971 | 0.031563 |
    | 2017-03-12 00:00:00 |  -3.4 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.756  0.756  0.756  1.3608 | 0.756  0.756  0.756  0.0 |       0.0        0.0        0.0  0.0 | 0.748148  0.748148  0.748148  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.000016   0.000016   0.000016  0.0 |  0.000016   0.000016   0.000016  0.0 |   0.0 | 0.0 | 0.0 |  0.000016 |  0.0 | 0.0 |   1.3608 | 0.760401 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.000329 |  0.237333 | 0.027469 |  0.717101   0.717101   0.717101  0.0 | 41.410879  41.410879  41.410879  0.0 | nan | nan | nan | nan | -0.000071 |   137.25262 | 0.027469 |
    | 2017-03-13 00:00:00 |  -3.8 |  0.9 | 0.0 | 0.0 | 0.0 |  0.72 | 0.567  0.567  0.567  1.0206 | 0.567  0.567  0.567  0.0 |  0.002965   0.002965   0.002965  0.0 | 0.552718  0.552718  0.552718  0.0 |      0.0       0.0       0.0  0.0 |  0.002965   0.002965   0.002965  0.0 |  0.000004   0.000004   0.000004  0.0 |  0.000004   0.000004   0.000004  0.0 |  0.72 | 0.0 | 0.0 |  0.000004 |  0.0 | 0.0 |   1.0206 | 0.562076 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | -0.000093 |  0.208385 | 0.024119 |  0.881418   0.881418   0.881418  0.0 |  41.41384   41.41384   41.41384  0.0 | nan | nan | nan | nan |  0.000026 |  126.528211 | 0.024119 |
    | 2017-03-14 00:00:00 |  -0.6 |  0.1 | 0.0 | 0.0 | 0.0 |  0.08 | 0.567  0.567  0.567  1.0206 | 0.567  0.567  0.567  0.0 |  0.000192   0.000192   0.000192  0.0 | 0.535229  0.535229  0.535229  0.0 | 0.000067  0.000067  0.000067  0.0 |  0.000125   0.000125   0.000125  0.0 |  0.184058   0.184058   0.184058  0.0 |  0.184058   0.184058   0.184058  0.0 |  0.08 | 0.0 | 0.0 |  0.184125 |  0.0 | 0.0 |   1.0206 | 0.544936 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.106415 |  0.188324 | 0.021797 |  0.425997   0.425997   0.425997  0.0 | 41.229907  41.229907  41.229907  0.0 | nan | nan | nan | nan |  0.077736 |  121.385715 | 0.021797 |
    | 2017-03-15 00:00:00 |  -1.7 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.756  0.756  0.756  1.3608 | 0.756  0.756  0.756  0.0 |       0.0        0.0        0.0  0.0 | 0.492029  0.492029  0.492029  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |   0.00479    0.00479    0.00479  0.0 |   0.00479    0.00479    0.00479  0.0 |   0.0 | 0.0 | 0.0 |   0.00479 |  0.0 | 0.0 |   1.3608 | 0.509404 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.068412 |  0.177555 |  0.02055 | -0.066032  -0.066032  -0.066032  0.0 | 41.225117  41.225117  41.225117  0.0 | nan | nan | nan | nan |  0.014115 |  114.499349 |  0.02055 |
    | 2017-03-16 00:00:00 |   0.7 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.882  0.882  0.882  1.5876 | 0.882  0.882  0.882  0.0 |       0.0        0.0        0.0  0.0 | 0.249425  0.249425  0.249425  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  3.633659   3.633659   3.633659  0.0 |  3.633659   3.633659   3.633659  0.0 |   0.0 | 0.0 | 0.0 |  3.633659 |  0.0 | 0.0 |   1.5876 | 0.276189 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  2.075009 |   0.24936 | 0.028861 | -0.315457  -0.315457  -0.315457  0.0 | 37.591458  37.591458  37.591458  0.0 | nan | nan | nan | nan |  1.572764 |   202.11922 | 0.028861 |
    | 2017-03-17 00:00:00 |  -1.6 |  3.9 | 0.0 | 0.0 | 0.0 |  3.12 | 0.693  0.693  0.693  1.2474 | 0.693  0.693  0.693  0.0 |  0.251138   0.251138   0.251138  0.0 | 0.602565  0.602565  0.602565  0.0 | 0.025114  0.025114  0.025114  0.0 |  0.226024   0.226024   0.226024  0.0 |  0.006707   0.006707   0.006707  0.0 |  0.006707   0.006707   0.006707  0.0 |  3.12 | 0.0 | 0.0 |  0.031821 |  0.0 | 0.0 |   1.2474 | 0.615461 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  1.369598 |  0.496609 | 0.057478 |  1.950841   1.950841   1.950841  0.0 | 37.810775  37.810775  37.810775  0.0 | nan | nan | nan | nan |  0.234987 |  246.271666 | 0.057478 |
    | 2017-03-18 00:00:00 |  -2.8 |  8.7 | 0.0 | 0.0 | 0.0 |  6.96 | 0.693  0.693  0.693  1.2474 | 0.693  0.693  0.693  0.0 |  5.752427   5.752427   5.752427  0.0 | 0.692984  0.692984  0.692984  0.0 |      0.0       0.0       0.0  0.0 |  5.752427   5.752427   5.752427  0.0 |  0.000117   0.000117   0.000117  0.0 |  0.000117   0.000117   0.000117  0.0 |  6.96 | 0.0 | 0.0 |  0.000117 |  0.0 | 0.0 |   1.2474 | 0.704072 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   0.20355 |  0.520082 | 0.060195 |   2.46543    2.46543    2.46543  0.0 | 43.563084  43.563084  43.563084  0.0 | nan | nan | nan | nan |  0.031555 |  235.954109 | 0.060195 |
    | 2017-03-19 00:00:00 |  -2.9 | 26.4 | 0.0 | 0.0 | 0.0 | 21.12 | 0.315  0.315  0.315   0.567 | 0.315  0.315  0.315  0.0 | 20.411176  20.411176  20.411176  0.0 | 0.314999  0.314999  0.314999  0.0 |      0.0       0.0       0.0  0.0 | 20.411176  20.411176  20.411176  0.0 |  0.000084   0.000084   0.000084  0.0 |  0.000084   0.000084   0.000084  0.0 | 21.12 | 0.0 | 0.0 |  0.000084 |  0.0 | 0.0 |    0.567 | 0.320039 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.027353 |  0.495111 | 0.057305 |  2.859255   2.859255   2.859255  0.0 | 63.974177  63.974177  63.974177  0.0 | nan | nan | nan | nan |  0.004285 |  233.091874 | 0.057305 |
    | 2017-03-20 00:00:00 |   0.7 | 11.5 | 0.0 | 0.0 | 0.0 |   9.2 | 0.378  0.378  0.378  0.6804 | 0.378  0.378  0.378  0.0 |  8.951477   8.951477   8.951477  0.0 | 0.377999  0.377999  0.377999  0.0 | 6.042247  6.042247  6.042247  0.0 |   2.90923    2.90923    2.90923  0.0 |  3.633659   3.633659   3.633659  0.0 |  3.633659   3.633659   3.633659  0.0 |   9.2 | 0.0 | 0.0 |  9.675906 |  0.0 | 0.0 |   0.6804 | 0.384047 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  5.502898 |  0.817301 | 0.094595 |   2.72978    2.72978    2.72978  0.0 | 63.249748  63.249748  63.249748  0.0 | nan | nan | nan | nan |  4.177293 |  470.388437 | 0.094595 |
    | 2017-03-21 00:00:00 |  -1.8 |  0.9 | 0.0 | 0.0 | 0.0 |  0.72 | 0.945  0.945  0.945   1.701 | 0.945  0.945  0.945  0.0 |  0.548136   0.548136   0.548136  0.0 | 0.944963  0.944963  0.944963  0.0 | 0.027407  0.027407  0.027407  0.0 |  0.520729   0.520729   0.520729  0.0 |   0.00342    0.00342    0.00342  0.0 |   0.00342    0.00342    0.00342  0.0 |  0.72 | 0.0 | 0.0 |  0.030827 |  0.0 | 0.0 |    1.701 | 0.960084 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  3.630626 |  1.749968 | 0.202543 |  1.956681   1.956681   1.956681  0.0 | 63.767056  63.767056  63.767056  0.0 | nan | nan | nan | nan |  0.577494 |  559.809702 | 0.202543 |
    | 2017-03-22 00:00:00 |  -2.1 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |  1.26   1.26   1.26   2.268 |  1.26   1.26   1.26  0.0 |       0.0        0.0        0.0  0.0 | 1.251526  1.251526  1.251526  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.001244   0.001244   0.001244  0.0 |  0.001244   0.001244   0.001244  0.0 |   0.0 | 0.0 | 0.0 |  0.001244 |  0.0 | 0.0 |    2.268 | 1.271856 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.500436 |  1.687171 | 0.195274 |  0.705154   0.705154   0.705154  0.0 | 63.765812  63.765812  63.765812  0.0 | nan | nan | nan | nan |  0.078302 |  497.704544 | 0.195274 |
    | 2017-03-23 00:00:00 |  -2.8 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.008  1.008  1.008  1.8144 | 1.008  1.008  1.008  0.0 |       0.0        0.0        0.0  0.0 |  0.74934   0.74934   0.74934  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.000117   0.000117   0.000117  0.0 |  0.000117   0.000117   0.000117  0.0 |   0.0 | 0.0 | 0.0 |  0.000117 |  0.0 | 0.0 |   1.8144 | 0.770641 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   0.06619 |  1.374635 | 0.159101 | -0.044186  -0.044186  -0.044186  0.0 | 63.765695  63.765695  63.765695  0.0 | nan | nan | nan | nan |   0.01223 |  430.401718 | 0.159101 |
    | 2017-03-24 00:00:00 |  -2.8 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.008  1.008  1.008  1.8144 | 1.008  1.008  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.286063  0.286063  0.286063  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.000117   0.000117   0.000117  0.0 |  0.000117   0.000117   0.000117  0.0 |   0.0 | 0.0 | 0.0 |  0.000117 |  0.0 | 0.0 |   1.8144 |  0.31663 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.008232 |  1.107468 | 0.128179 | -0.330248  -0.330248  -0.330248  0.0 | 63.765577  63.765577  63.765577  0.0 | nan | nan | nan | nan |  0.004116 |  373.617276 | 0.128179 |
    | 2017-03-25 00:00:00 |  -2.1 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.756  0.756  0.756  1.3608 | 0.756  0.756  0.756  0.0 |       0.0        0.0        0.0  0.0 | 0.109167  0.109167  0.109167  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.001244   0.001244   0.001244  0.0 |  0.001244   0.001244   0.001244  0.0 |   0.0 | 0.0 | 0.0 |  0.001244 |  0.0 | 0.0 |   1.3608 | 0.134199 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.005903 |  0.901366 | 0.104325 | -0.439415  -0.439415  -0.439415  0.0 | 63.764333  63.764333  63.764333  0.0 | nan | nan | nan | nan | -0.000542 |  327.477421 | 0.104325 |
    | 2017-03-26 00:00:00 |  -1.3 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.819  0.819  0.819  1.4742 | 0.819  0.819  0.819  0.0 |       0.0        0.0        0.0  0.0 | 0.080905  0.080905  0.080905  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.018374   0.018374   0.018374  0.0 |  0.018374   0.018374   0.018374  0.0 |   0.0 | 0.0 | 0.0 |  0.018374 |  0.0 | 0.0 |   1.4742 | 0.108771 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.011888 |  0.743612 | 0.086066 |  -0.52032   -0.52032   -0.52032  0.0 | 63.745958  63.745958  63.745958  0.0 | nan | nan | nan | nan |  0.005944 |  289.405125 | 0.086066 |
    | 2017-03-27 00:00:00 |  -1.0 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.008  1.008  1.008  1.8144 | 1.008  1.008  1.008  0.0 |       0.0        0.0        0.0  0.0 | 0.072233  0.072233  0.072233  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |      0.05       0.05       0.05  0.0 |      0.05       0.05       0.05  0.0 |   0.0 | 0.0 | 0.0 |      0.05 |  0.0 | 0.0 |   1.8144 | 0.107076 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.032214 |  0.622194 | 0.072013 | -0.592552  -0.592552  -0.592552  0.0 | 63.695958  63.695958  63.695958  0.0 | nan | nan | nan | nan |   0.02373 |  258.059532 | 0.072013 |
    | 2017-03-28 00:00:00 |   0.3 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.197  1.197  1.197  2.1546 | 1.197  1.197  1.197  0.0 |       0.0        0.0        0.0  0.0 | 0.064013  0.064013  0.064013  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |   1.95975    1.95975    1.95975  0.0 |   1.95975    1.95975    1.95975  0.0 |   0.0 | 0.0 | 0.0 |   1.95975 |  0.0 | 0.0 |   2.1546 | 0.105825 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  1.132357 |  0.590953 | 0.068397 | -0.656565  -0.656565  -0.656565  0.0 | 61.736209  61.736209  61.736209  0.0 | nan | nan | nan | nan |  0.851123 |  281.842756 | 0.068397 |
    | 2017-03-29 00:00:00 |   1.0 |  1.5 | 0.0 | 0.0 | 0.0 |   1.2 | 0.504  0.504  0.504  0.9072 | 0.504  0.504  0.504  0.0 |  0.000144   0.000144   0.000144  0.0 | 0.204166  0.204166  0.204166  0.0 | 0.000108  0.000108  0.000108  0.0 |  0.000036   0.000036   0.000036  0.0 |      5.05       5.05       5.05  0.0 |      5.05       5.05       5.05  0.0 |   1.2 | 0.0 | 0.0 |  5.050108 |  0.0 | 0.0 |   0.9072 | 0.218227 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  3.602778 |  0.866616 | 0.100303 |  0.339124   0.339124   0.339124  0.0 | 56.686245  56.686245  56.686245  0.0 | nan | nan | nan | nan |  2.298453 |  415.340864 | 0.100303 |
    | 2017-03-30 00:00:00 |   4.4 |  0.3 | 0.0 | 0.0 | 0.0 |  0.24 | 0.945  0.945  0.945   1.701 | 0.945  0.945  0.945  0.0 |   0.00005    0.00005    0.00005  0.0 | 0.601868  0.601868  0.601868  0.0 |  0.00005   0.00005   0.00005  0.0 |       0.0        0.0        0.0  0.0 | 22.000001  22.000001  22.000001  0.0 | 22.000001  22.000001  22.000001  0.0 |  0.24 | 0.0 | 0.0 | 22.000051 |  0.0 | 0.0 |    1.701 |  0.62385 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 14.476228 |  2.422961 | 0.280435 | -0.022794  -0.022794  -0.022794  0.0 | 34.686244  34.686244  34.686244  0.0 | nan | nan | nan | nan |  9.822275 | 1002.066976 | 0.280435 |
    | 2017-03-31 00:00:00 |   7.0 |  0.2 | 0.0 | 0.0 | 0.0 |  0.16 | 1.701  1.701  1.701  3.0618 | 1.701  1.701  1.701  0.0 |  0.000007   0.000007   0.000007  0.0 | 0.470491  0.470491  0.470491  0.0 | 0.000007  0.000007  0.000007  0.0 |       0.0        0.0        0.0  0.0 |      35.0       35.0       35.0  0.0 | 34.731609  34.731609  34.731609  0.0 |  0.16 | 0.0 | 0.0 | 34.731616 |  0.0 | 0.0 |   3.0618 | 0.522317 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 28.356183 |  7.950003 | 0.920139 | -0.333291  -0.333291  -0.333291  0.0 | -0.045365  -0.045365  -0.045365  0.0 | nan | nan | nan | nan | 16.197708 | 1991.118031 | 0.920139 |
    | 2017-04-01 00:00:00 |   3.8 |  4.5 | 0.0 | 0.0 | 0.0 |   3.6 | 0.945  0.945  0.945   1.728 | 0.945  0.945  0.945  0.0 |  0.413728   0.413728   0.413728  0.0 | 0.831573  0.831573  0.831573  0.0 | 0.413728  0.413728  0.413728  0.0 |       0.0        0.0        0.0  0.0 | 19.000004  19.000004  19.000004  0.0 |  0.919718   0.919718   0.919718  0.0 |   3.6 | 0.0 | 0.0 |  1.333446 |  0.0 | 0.0 |    1.728 | 0.849501 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 14.790422 | 13.481342 |  1.56034 |  2.021409   2.021409   2.021409  0.0 | -0.965084  -0.965084  -0.965084  0.0 | nan | nan | nan | nan |  2.740732 | 2043.653612 |  1.56034 |
    | 2017-04-02 00:00:00 |   1.4 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.008  1.008  1.008  1.8432 | 1.008  1.008  1.008  0.0 |       0.0        0.0        0.0  0.0 | 1.005972  1.005972  1.005972  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  7.013137   7.013137   7.013137  0.0 |  0.069834   0.069834   0.069834  0.0 |   0.0 | 0.0 | 0.0 |  0.069834 |  0.0 | 0.0 |   1.8432 | 1.022717 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  2.410784 | 10.827169 | 1.253145 |  1.015436   1.015436   1.015436  0.0 | -1.034918  -1.034918  -1.034918  0.0 | nan | nan | nan | nan |  0.399782 | 1618.580384 | 1.253145 |
    | 2017-04-03 00:00:00 |   0.1 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |  1.26   1.26   1.26   2.304 |  1.26   1.26   1.26  0.0 |       0.0        0.0        0.0  0.0 | 1.029652  1.029652  1.029652  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  1.298601   1.298601   1.298601  0.0 |  0.010807   0.010807   0.010807  0.0 |   0.0 | 0.0 | 0.0 |  0.010807 |  0.0 | 0.0 |    2.304 | 1.055139 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.351386 |  7.464324 | 0.863926 | -0.014216  -0.014216  -0.014216  0.0 | -1.045725  -1.045725  -1.045725  0.0 | nan | nan | nan | nan |  0.059203 | 1260.278116 | 0.863926 |
    | 2017-04-04 00:00:00 |  -0.1 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.323  1.323  1.323  2.4192 | 1.323  1.323  1.323  0.0 |       0.0        0.0        0.0  0.0 |  0.35708   0.35708   0.35708  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.798601   0.798601   0.798601  0.0 |  0.006391   0.006391   0.006391  0.0 |   0.0 | 0.0 | 0.0 |  0.006391 |  0.0 | 0.0 |   2.4192 | 0.398323 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.053701 |  5.194291 | 0.601191 | -0.371296  -0.371296  -0.371296  0.0 | -1.052115  -1.052115  -1.052115  0.0 | nan | nan | nan | nan |  0.011893 | 1000.775704 | 0.601191 |
    | 2017-04-05 00:00:00 |  -0.5 |  0.4 | 0.0 | 0.0 | 0.0 |  0.32 | 1.071  1.071  1.071  1.9584 | 1.071  1.071  1.071  0.0 |  0.000008   0.000008   0.000008  0.0 | 0.210309  0.210309  0.210309  0.0 | 0.000003  0.000003  0.000003  0.0 |  0.000005   0.000005   0.000005  0.0 |  0.251908   0.251908   0.251908  0.0 |  0.001978   0.001978   0.001978  0.0 |  0.32 | 0.0 | 0.0 |  0.001981 |  0.0 | 0.0 |   1.9584 |  0.24527 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.009252 |  3.736082 | 0.432417 | -0.261613  -0.261613  -0.261613  0.0 | -1.054088  -1.054088  -1.054088  0.0 | nan | nan | nan | nan |  0.004622 |   812.78656 | 0.432417 |
    | 2017-04-06 00:00:00 |  -0.8 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.071  1.071  1.071  1.9584 | 1.071  1.071  1.071  0.0 |       0.0        0.0        0.0  0.0 | 0.175292  0.175292  0.175292  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  0.096611   0.096611   0.096611  0.0 |  0.000754   0.000754   0.000754  0.0 |   0.0 | 0.0 | 0.0 |  0.000754 |  0.0 | 0.0 |   1.9584 | 0.210954 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.003584 |  2.771673 | 0.320795 | -0.436905  -0.436905  -0.436905  0.0 | -1.054842  -1.054842  -1.054842  0.0 | nan | nan | nan | nan |  0.001792 |  672.420148 | 0.320795 |
    | 2017-04-07 00:00:00 |   0.1 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.504  0.504  0.504  0.9216 | 0.504  0.504  0.504  0.0 |       0.0        0.0        0.0  0.0 | 0.053397  0.053397  0.053397  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  1.298601   1.298601   1.298601  0.0 |  0.009889   0.009889   0.009889  0.0 |   0.0 | 0.0 | 0.0 |  0.009889 |  0.0 | 0.0 |   0.9216 | 0.070761 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.007862 |  2.112451 | 0.244497 | -0.490302  -0.490302  -0.490302  0.0 | -1.064732  -1.064732  -1.064732  0.0 | nan | nan | nan | nan |  0.003819 |  566.261243 | 0.244497 |
    | 2017-04-08 00:00:00 |   0.6 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.819  0.819  0.819  1.4976 | 0.819  0.819  0.819  0.0 |       0.0        0.0        0.0  0.0 | 0.067384  0.067384  0.067384  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 |  3.184058   3.184058   3.184058  0.0 |  0.022513   0.022513   0.022513  0.0 |   0.0 | 0.0 | 0.0 |  0.022513 |  0.0 | 0.0 |   1.4976 | 0.095989 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.017946 |  1.648182 | 0.190762 | -0.557687  -0.557687  -0.557687  0.0 | -1.087244  -1.087244  -1.087244  0.0 | nan | nan | nan | nan |  0.008385 |  483.233915 | 0.190762 |
    | 2017-04-09 00:00:00 |   2.5 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.575  1.575  1.575    2.88 | 1.575  1.575  1.575  0.0 |       0.0        0.0        0.0  0.0 | 0.091899  0.091899  0.091899  0.0 |      0.0       0.0       0.0  0.0 |       0.0        0.0        0.0  0.0 | 12.500323  12.500323  12.500323  0.0 |  0.071094   0.071094   0.071094  0.0 |   0.0 | 0.0 | 0.0 |  0.071094 |  0.0 | 0.0 |     2.88 | 0.147661 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.049923 |  1.310392 | 0.151666 | -0.649586  -0.649586  -0.649586  0.0 | -1.158338  -1.158338  -1.158338  0.0 | nan | nan | nan | nan |  0.029557 |  417.280525 | 0.151666 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _wland_gd_no_land_area:

no land area
____________

>>> nu(1)
>>> aur(1.0)
>>> lt(WATER)
>>> er(False)
>>> model.update_parameters()
>>> model.petmodel.parameters.control.dampingfactor(1.0)
>>> model.petmodel.retmodel.parameters.control.evapotranspirationfactor(0.9)
>>> test.inits.ic = 0.0
>>> test.inits.sp = 0.0
>>> test.reset_inits()
>>> conditions = model.conditions

.. integration-test::

    >>> test("wland_gd_no_land_area",
    ...      axis1=(fluxes.pc, fluxes.et, fluxes.rh),
    ...      axis2=(states.hs,))
    |                date |     t |    p | fxg | fxs | dhs |    pc |     pe | pet |  tf |  ei |  rf |  sf |  pm |  am |    ps | pve |  pv |  pq | etve | etv |       es |       et |  gr | fxs | fxg | cdg | fgse | fgs | fqs |       rh |        r |  ic |  sp | dve |  dv | hge |  dg |  hq |        hs |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 00:00:00 | -10.8 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.6588 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.000067 | 0.000067 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 |      0.0 |      0.0 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | -2.000067 |      0.0 |
    | 2017-02-11 00:00:00 |  -9.5 |  0.4 | 0.0 | 0.0 | 0.0 |  0.32 | 0.8784 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.32 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.00024 |  0.00024 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 |      0.0 |      0.0 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | -1.680307 |      0.0 |
    | 2017-02-12 00:00:00 |  -8.9 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.7686 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.00034 |  0.00034 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 |      0.0 |      0.0 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | -1.680647 |      0.0 |
    | 2017-02-13 00:00:00 |  -9.6 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.4392 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.000194 | 0.000194 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 |      0.0 |      0.0 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | -1.680841 |      0.0 |
    | 2017-02-14 00:00:00 |  -9.3 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.4392 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.000194 | 0.000194 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 |      0.0 |      0.0 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | -1.681035 |      0.0 |
    | 2017-02-15 00:00:00 |  -6.3 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.4392 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.000194 | 0.000194 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 |      0.0 |      0.0 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | -1.681229 |      0.0 |
    | 2017-02-16 00:00:00 |  -3.6 |  0.2 | 0.0 | 0.0 | 0.0 |  0.16 | 0.4392 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.16 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.000299 | 0.000299 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 |      0.0 |      0.0 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | -1.521528 |      0.0 |
    | 2017-02-17 00:00:00 |  -3.5 |  4.5 | 0.0 | 0.0 | 0.0 |   3.6 | 0.3294 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   3.6 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.188582 | 0.188582 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.000088 |  0.00001 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 |  1.889802 |  0.00001 |
    | 2017-02-18 00:00:00 |  -4.6 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.3294 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.329245 | 0.329245 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.000313 | 0.000036 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 |  1.560243 | 0.000036 |
    | 2017-02-19 00:00:00 |  -3.2 |  3.2 | 0.0 | 0.0 | 0.0 |  2.56 | 0.4392 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  2.56 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.439031 | 0.439031 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.000621 | 0.000072 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 |  3.680592 | 0.000072 |
    | 2017-02-20 00:00:00 |  -1.3 |  4.6 | 0.0 | 0.0 | 0.0 |  3.68 | 0.3294 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  3.68 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   0.3294 |   0.3294 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 |  0.00177 | 0.000205 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 |  7.029422 | 0.000205 |
    | 2017-02-21 00:00:00 |  -2.2 |  2.3 | 0.0 | 0.0 | 0.0 |  1.84 | 0.6588 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  1.84 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   0.6588 |   0.6588 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.002902 | 0.000336 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 |   8.20772 | 0.000336 |
    | 2017-02-22 00:00:00 |  -1.5 | 18.0 | 0.0 | 0.0 | 0.0 |  14.4 | 0.8784 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  14.4 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   0.8784 |   0.8784 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.008592 | 0.000994 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 21.720728 | 0.000994 |
    | 2017-02-23 00:00:00 |  -3.0 | 19.2 | 0.0 | 0.0 | 0.0 | 15.36 |  0.549 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 15.36 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |    0.549 |    0.549 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.022164 | 0.002565 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 36.509564 | 0.002565 |
    | 2017-02-24 00:00:00 |  -5.0 |  0.4 | 0.0 | 0.0 | 0.0 |  0.32 | 0.8784 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.32 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   0.8784 |   0.8784 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.030012 | 0.003474 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 35.921151 | 0.003474 |
    | 2017-02-25 00:00:00 |  -4.9 |  8.3 | 0.0 | 0.0 | 0.0 |  6.64 |  0.549 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  6.64 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |    0.549 |    0.549 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.033553 | 0.003883 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 41.978598 | 0.003883 |
    | 2017-02-26 00:00:00 |  -0.9 |  5.3 | 0.0 | 0.0 | 0.0 |  4.24 | 0.4392 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  4.24 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   0.4392 |   0.4392 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.040028 | 0.004633 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 45.739371 | 0.004633 |
    | 2017-02-27 00:00:00 |   1.4 |  0.7 | 0.0 | 0.0 | 0.0 |  0.56 | 1.4274 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.56 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   1.4274 |   1.4274 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.041965 | 0.004857 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 44.830005 | 0.004857 |
    | 2017-02-28 00:00:00 |  -3.4 |  2.7 | 0.0 | 0.0 | 0.0 |  2.16 | 0.9882 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  2.16 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   0.9882 |   0.9882 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.042121 | 0.004875 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 45.959685 | 0.004875 |
    | 2017-03-01 00:00:00 |  -4.3 |  1.6 | 0.0 | 0.0 | 0.0 |  1.28 | 0.7938 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  1.28 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   0.7938 |   0.7938 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.043217 | 0.005002 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 46.402668 | 0.005002 |
    | 2017-03-02 00:00:00 |  -3.3 |  2.5 | 0.0 | 0.0 | 0.0 |   2.0 | 0.7938 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   2.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   0.7938 |   0.7938 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.044351 | 0.005133 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 47.564517 | 0.005133 |
    | 2017-03-03 00:00:00 |  -2.1 |  0.6 | 0.0 | 0.0 | 0.0 |  0.48 | 1.2474 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.48 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   1.2474 |   1.2474 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.044597 | 0.005162 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 |  46.75252 | 0.005162 |
    | 2017-03-04 00:00:00 |  -0.3 |  0.2 | 0.0 | 0.0 | 0.0 |  0.16 |  1.134 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.16 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |    1.134 |    1.134 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.043306 | 0.005012 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 45.735214 | 0.005012 |
    | 2017-03-05 00:00:00 |  -1.7 |  1.7 | 0.0 | 0.0 | 0.0 |  1.36 | 0.9072 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  1.36 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   0.9072 |   0.9072 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.042879 | 0.004963 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 46.145135 | 0.004963 |
    | 2017-03-06 00:00:00 |  -4.3 |  0.3 | 0.0 | 0.0 | 0.0 |  0.24 | 0.6804 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.24 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   0.6804 |   0.6804 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.042828 | 0.004957 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 45.661907 | 0.004957 |
    | 2017-03-07 00:00:00 |  -6.4 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.7938 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   0.7938 |   0.7938 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.041909 | 0.004851 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 44.826198 | 0.004851 |
    | 2017-03-08 00:00:00 |  -4.0 |  1.8 | 0.0 | 0.0 | 0.0 |  1.44 | 0.7938 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  1.44 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   0.7938 |   0.7938 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.041748 | 0.004832 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 |  45.43065 | 0.004832 |
    | 2017-03-09 00:00:00 |  -2.4 |  8.9 | 0.0 | 0.0 | 0.0 |  7.12 |  0.567 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  7.12 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |    0.567 |    0.567 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.046858 | 0.005423 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 51.936792 | 0.005423 |
    | 2017-03-10 00:00:00 |  -2.2 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.9072 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   0.9072 |   0.9072 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.050832 | 0.005883 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 50.978759 | 0.005883 |
    | 2017-03-11 00:00:00 |  -2.3 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |  1.134 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |    1.134 |    1.134 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.049255 | 0.005701 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 49.795504 | 0.005701 |
    | 2017-03-12 00:00:00 |  -3.4 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.3608 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   1.3608 |   1.3608 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.047368 | 0.005482 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 48.387336 | 0.005482 |
    | 2017-03-13 00:00:00 |  -3.8 |  0.9 | 0.0 | 0.0 | 0.0 |  0.72 | 1.0206 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.72 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   1.0206 |   1.0206 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.046101 | 0.005336 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 48.040635 | 0.005336 |
    | 2017-03-14 00:00:00 |  -0.6 |  0.1 | 0.0 | 0.0 | 0.0 |  0.08 | 1.0206 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.08 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   1.0206 |   1.0206 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 |  0.04515 | 0.005226 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 47.054884 | 0.005226 |
    | 2017-03-15 00:00:00 |  -1.7 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.3608 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   1.3608 |   1.3608 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.043461 |  0.00503 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 45.650624 |  0.00503 |
    | 2017-03-16 00:00:00 |   0.7 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.5876 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   1.5876 |   1.5876 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.041347 | 0.004786 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 44.021677 | 0.004786 |
    | 2017-03-17 00:00:00 |  -1.6 |  3.9 | 0.0 | 0.0 | 0.0 |  3.12 | 1.2474 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  3.12 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   1.2474 |   1.2474 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 |  0.04149 | 0.004802 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 45.852787 | 0.004802 |
    | 2017-03-18 00:00:00 |  -2.8 |  8.7 | 0.0 | 0.0 | 0.0 |  6.96 | 1.2474 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  6.96 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   1.2474 |   1.2474 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.046842 | 0.005422 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 51.518546 | 0.005422 |
    | 2017-03-19 00:00:00 |  -2.9 | 26.4 | 0.0 | 0.0 | 0.0 | 21.12 |  0.567 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 21.12 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |    0.567 |    0.567 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.067073 | 0.007763 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 72.004472 | 0.007763 |
    | 2017-03-20 00:00:00 |   0.7 | 11.5 | 0.0 | 0.0 | 0.0 |   9.2 | 0.6804 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.2 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   0.6804 |   0.6804 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.091743 | 0.010618 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 80.432329 | 0.010618 |
    | 2017-03-21 00:00:00 |  -1.8 |  0.9 | 0.0 | 0.0 | 0.0 |  0.72 |  1.701 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.72 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |    1.701 |    1.701 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.098337 | 0.011382 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 79.352992 | 0.011382 |
    | 2017-03-22 00:00:00 |  -2.1 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |  2.268 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |    2.268 |    2.268 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.095182 | 0.011016 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 |  76.98981 | 0.011016 |
    | 2017-03-23 00:00:00 |  -2.8 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.8144 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   1.8144 |   1.8144 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.091308 | 0.010568 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 75.084102 | 0.010568 |
    | 2017-03-24 00:00:00 |  -2.8 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.8144 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   1.8144 |   1.8144 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 |   0.0879 | 0.010174 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 73.181802 | 0.010174 |
    | 2017-03-25 00:00:00 |  -2.1 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.3608 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   1.3608 |   1.3608 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.084938 | 0.009831 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 71.736064 | 0.009831 |
    | 2017-03-26 00:00:00 |  -1.3 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.4742 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   1.4742 |   1.4742 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.082313 | 0.009527 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 70.179552 | 0.009527 |
    | 2017-03-27 00:00:00 |  -1.0 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.8144 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   1.8144 |   1.8144 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.079331 | 0.009182 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 68.285821 | 0.009182 |
    | 2017-03-28 00:00:00 |   0.3 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 2.1546 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   2.1546 |   2.1546 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.075815 | 0.008775 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 66.055405 | 0.008775 |
    | 2017-03-29 00:00:00 |   1.0 |  1.5 | 0.0 | 0.0 | 0.0 |   1.2 | 0.9072 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   1.2 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   0.9072 |   0.9072 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.074113 | 0.008578 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 66.274092 | 0.008578 |
    | 2017-03-30 00:00:00 |   4.4 |  0.3 | 0.0 | 0.0 | 0.0 |  0.24 |  1.701 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.24 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |    1.701 |    1.701 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.073013 | 0.008451 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 64.740079 | 0.008451 |
    | 2017-03-31 00:00:00 |   7.0 |  0.2 | 0.0 | 0.0 | 0.0 |  0.16 | 3.0618 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.16 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   3.0618 |   3.0618 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 |  0.06929 |  0.00802 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 |  61.76899 |  0.00802 |
    | 2017-04-01 00:00:00 |   3.8 |  4.5 | 0.0 | 0.0 | 0.0 |   3.6 |  1.728 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   3.6 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |    1.728 |    1.728 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.068327 | 0.007908 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 63.572662 | 0.007908 |
    | 2017-04-02 00:00:00 |   1.4 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.8432 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   1.8432 |   1.8432 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.068237 | 0.007898 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 61.661225 | 0.007898 |
    | 2017-04-03 00:00:00 |   0.1 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |  2.304 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |    2.304 |    2.304 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.064772 | 0.007497 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 59.292453 | 0.007497 |
    | 2017-04-04 00:00:00 |  -0.1 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 2.4192 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   2.4192 |   2.4192 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.060918 | 0.007051 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 56.812335 | 0.007051 |
    | 2017-04-05 00:00:00 |  -0.5 |  0.4 | 0.0 | 0.0 | 0.0 |  0.32 | 1.9584 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.32 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   1.9584 |   1.9584 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.057657 | 0.006673 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 55.116278 | 0.006673 |
    | 2017-04-06 00:00:00 |  -0.8 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.9584 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   1.9584 |   1.9584 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.054817 | 0.006345 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 53.103062 | 0.006345 |
    | 2017-04-07 00:00:00 |   0.1 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.9216 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   0.9216 |   0.9216 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.052558 | 0.006083 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 52.128904 | 0.006083 |
    | 2017-04-08 00:00:00 |   0.6 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 1.4976 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |   1.4976 |   1.4976 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.050682 | 0.005866 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 50.580622 | 0.005866 |
    | 2017-04-09 00:00:00 |   2.5 |  0.0 | 0.0 | 0.0 | 0.0 |   0.0 |   2.88 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 |     2.88 |     2.88 | 0.0 | 0.0 | 0.0 | 0.0 |  0.0 | 0.0 | 0.0 | 0.047417 | 0.005488 | 0.0 | 0.0 | nan | nan | nan | nan | 0.0 | 47.653205 | 0.005488 |

>>> round_(model.check_waterbalance(conditions))
0.0
"""
# import...
# ...from HydPy
from hydpy.exe.modelimports import *
from hydpy.core import modeltools
from hydpy.core.typingtools import *
from hydpy.interfaces import dischargeinterfaces
from hydpy.interfaces import petinterfaces
from hydpy.interfaces import stateinterfaces

# ...from lland
from hydpy.models.wland import wland_model
from hydpy.models.wland import wland_solver
from hydpy.models.wland.wland_constants import *


class Model(
    wland_model.BaseModel,
    wland_model.Main_PETModel_V1,
    wland_model.Main_PETModel_V2,
    wland_model.Main_DischargeModel_V2,
    wland_model.Main_WaterLevelModel_V1,
    wland_model.Sub_TempModel_V1,
    wland_model.Sub_PrecipModel_V1,
    wland_model.Sub_SnowCoverModel_V1,
):
    """|wland_gd.DOCNAME.complete|"""

    DOCNAME = modeltools.DocName(
        short="W-GD",
        description=(
            "extended version of the WALRUS model with modified groundwater dynamics"
        ),
    )
    __HYDPY_ROOTMODEL__ = True

    SOLVERPARAMETERS = (
        wland_solver.AbsErrorMax,
        wland_solver.RelErrorMax,
        wland_solver.RelDTMin,
        wland_solver.RelDTMax,
    )
    SOLVERSEQUENCES = ()
    INLET_METHODS = (
        wland_model.Calc_PE_PET_V1,
        wland_model.Calc_FR_V1,
        wland_model.Calc_PM_V1,
    )
    RECEIVER_METHODS = (wland_model.Pick_HS_V1,)
    INTERFACE_METHODS = (
        wland_model.Get_Temperature_V1,
        wland_model.Get_MeanTemperature_V1,
        wland_model.Get_Precipitation_V1,
        wland_model.Get_SnowCover_V1,
    )
    ADD_METHODS = (
        wland_model.Calc_PE_PET_PETModel_V1,
        wland_model.Calc_PE_PET_PETModel_V2,
        wland_model.Return_DVH_V2,
        wland_model.Return_ErrorDV_V1,
    )
    PART_ODE_METHODS = (
        wland_model.Calc_FXS_V1,
        wland_model.Calc_FXG_V1,
        wland_model.Calc_PC_V1,
        wland_model.Calc_TF_V1,
        wland_model.Calc_EI_V1,
        wland_model.Calc_RF_V1,
        wland_model.Calc_SF_V1,
        wland_model.Calc_AM_V1,
        wland_model.Calc_PS_V1,
        wland_model.Calc_WE_W_V1,
        wland_model.Calc_PVE_PV_V1,
        wland_model.Calc_PQ_V1,
        wland_model.Calc_BetaE_Beta_V1,
        wland_model.Calc_ETVE_ETV_V1,
        wland_model.Calc_ES_V1,
        wland_model.Calc_GR_V1,
        wland_model.Calc_FQS_V1,
        wland_model.Calc_FGSE_V1,
        wland_model.Calc_FGS_V1,
        wland_model.Calc_RH_V1,
        wland_model.Calc_DVEq_V3,
        wland_model.Calc_DGEq_V1,
        wland_model.Calc_GF_V1,
        wland_model.Calc_CDG_V2,
    )
    FULL_ODE_METHODS = (
        wland_model.Update_IC_V1,
        wland_model.Update_SP_V1,
        wland_model.Update_DVE_V1,
        wland_model.Update_DV_V1,
        wland_model.Update_HGE_V1,
        wland_model.Update_DG_V1,
        wland_model.Update_HQ_V1,
        wland_model.Update_HS_V1,
    )
    OUTLET_METHODS = (
        wland_model.Calc_ET_V1,
        wland_model.Calc_R_V1,
        wland_model.Pass_R_V1,
    )
    SENDER_METHODS = ()
    SUBMODELINTERFACES = (
        petinterfaces.PETModel_V1,
        petinterfaces.PETModel_V2,
        dischargeinterfaces.DischargeModel_V2,
        stateinterfaces.WaterLevelModel_V1,
    )
    SUBMODELS = (wland_model.PegasusDGEq,)

    petmodel = modeltools.SubmodelProperty(
        petinterfaces.PETModel_V1, petinterfaces.PETModel_V2
    )
    dischargemodel = modeltools.SubmodelProperty(dischargeinterfaces.DischargeModel_V2)
    waterlevelmodel = modeltools.SubmodelProperty(stateinterfaces.WaterLevelModel_V1)


tester = Tester()
cythonizer = Cythonizer()
