"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.matchTemplate = exports.beASupersetOfTemplate = exports.exactlyMatchTemplate = exports.MatchStyle = void 0;
const cfnDiff = require("@aws-cdk/cloudformation-diff");
const assertion_1 = require("../assertion");
var MatchStyle;
(function (MatchStyle) {
    /** Requires an exact match */
    MatchStyle["EXACT"] = "exactly";
    /** Allows any change that does not cause a resource replacement */
    MatchStyle["NO_REPLACES"] = "no replaces";
    /** Allows additions, but no updates */
    MatchStyle["SUPERSET"] = "superset";
})(MatchStyle = exports.MatchStyle || (exports.MatchStyle = {}));
function exactlyMatchTemplate(template) {
    return matchTemplate(template, MatchStyle.EXACT);
}
exports.exactlyMatchTemplate = exactlyMatchTemplate;
function beASupersetOfTemplate(template) {
    return matchTemplate(template, MatchStyle.SUPERSET);
}
exports.beASupersetOfTemplate = beASupersetOfTemplate;
function matchTemplate(template, matchStyle = MatchStyle.EXACT) {
    return new StackMatchesTemplateAssertion(template, matchStyle);
}
exports.matchTemplate = matchTemplate;
class StackMatchesTemplateAssertion extends assertion_1.Assertion {
    constructor(template, matchStyle) {
        super();
        this.template = template;
        this.matchStyle = matchStyle;
    }
    assertOrThrow(inspector) {
        if (!this.assertUsing(inspector)) {
            // The details have already been printed, so don't generate a huge error message
            throw new Error('Template comparison produced unacceptable match');
        }
    }
    assertUsing(inspector) {
        const diff = cfnDiff.diffTemplate(this.template, inspector.value);
        const acceptable = this.isDiffAcceptable(diff);
        if (!acceptable) {
            // Print the diff
            cfnDiff.formatDifferences(process.stderr, diff);
            // Print the actual template
            process.stdout.write('--------------------------------------------------------------------------------------\n');
            process.stdout.write(JSON.stringify(inspector.value, undefined, 2) + '\n');
        }
        return acceptable;
    }
    isDiffAcceptable(diff) {
        switch (this.matchStyle) {
            case MatchStyle.EXACT:
                return diff.differenceCount === 0;
            case MatchStyle.NO_REPLACES:
                for (const change of Object.values(diff.resources.changes)) {
                    if (change.changeImpact === cfnDiff.ResourceImpact.MAY_REPLACE) {
                        return false;
                    }
                    if (change.changeImpact === cfnDiff.ResourceImpact.WILL_REPLACE) {
                        return false;
                    }
                }
                for (const change of Object.values(diff.parameters.changes)) {
                    if (change.isUpdate) {
                        return false;
                    }
                }
                for (const change of Object.values(diff.outputs.changes)) {
                    if (change.isUpdate) {
                        return false;
                    }
                }
                return true;
            case MatchStyle.SUPERSET:
                for (const change of Object.values(diff.resources.changes)) {
                    if (change.changeImpact !== cfnDiff.ResourceImpact.WILL_CREATE) {
                        return false;
                    }
                }
                for (const change of Object.values(diff.parameters.changes)) {
                    if (change.isAddition) {
                        return false;
                    }
                }
                for (const change of Object.values(diff.outputs.changes)) {
                    if (change.isAddition || change.isUpdate) {
                        return false;
                    }
                }
                return true;
        }
        throw new Error(`Unsupported match style: ${this.matchStyle}`);
    }
    get description() {
        return `template (${this.matchStyle}): ${JSON.stringify(this.template, null, 2)}`;
    }
}
//# sourceMappingURL=data:application/json;base64,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