"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SynthUtils = void 0;
const fs = require("fs");
const path = require("path");
const core = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
class SynthUtils {
    /**
     * Returns the cloud assembly template artifact for a stack.
     */
    static synthesize(stack, options = {}) {
        // always synthesize against the root (be it an App or whatever) so all artifacts will be included
        const assembly = synthesizeApp(stack, options);
        return assembly.getStackArtifact(stack.artifactId);
    }
    /**
     * Synthesizes the stack and returns the resulting CloudFormation template.
     */
    static toCloudFormation(stack, options = {}) {
        const synth = this._synthesizeWithNested(stack, options);
        if (synth instanceof cxapi.CloudFormationStackArtifact) {
            return synth.template;
        }
        else {
            return synth;
        }
    }
    /**
     * @returns Returns a subset of the synthesized CloudFormation template (only specific resource types).
     */
    static subset(stack, options) {
        const template = this.toCloudFormation(stack);
        if (template.Resources) {
            for (const [key, resource] of Object.entries(template.Resources)) {
                if (options.resourceTypes && !options.resourceTypes.includes(resource.Type)) {
                    delete template.Resources[key];
                }
            }
        }
        return template;
    }
    /**
     * Synthesizes the stack and returns a `CloudFormationStackArtifact` which can be inspected.
     * Supports nested stacks as well as normal stacks.
     *
     * @return CloudFormationStackArtifact for normal stacks or the actual template for nested stacks
     * @internal
     */
    static _synthesizeWithNested(stack, options = {}) {
        // always synthesize against the root (be it an App or whatever) so all artifacts will be included
        const assembly = synthesizeApp(stack, options);
        // if this is a nested stack (it has a parent), then just read the template as a string
        if (stack.nestedStackParent) {
            return JSON.parse(fs.readFileSync(path.join(assembly.directory, stack.templateFile)).toString('utf-8'));
        }
        return assembly.getStackArtifact(stack.artifactId);
    }
}
exports.SynthUtils = SynthUtils;
/**
 * Synthesizes the app in which a stack resides and returns the cloud assembly object.
 */
function synthesizeApp(stack, options) {
    const root = stack.node.root;
    if (!core.Stage.isStage(root)) {
        throw new Error('unexpected: all stacks must be part of a Stage or an App');
    }
    // to support incremental assertions (i.e. "expect(stack).toNotContainSomething(); doSomething(); expect(stack).toContainSomthing()")
    const force = true;
    return root.synth({
        force,
        ...options,
    });
}
//# sourceMappingURL=data:application/json;base64,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