"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isTagPropertyStringMap = exports.isTagPropertyJson = exports.isTagPropertyAutoScalingGroup = exports.isTagPropertyStandard = exports.isTagProperty = exports.isTagPropertyName = exports.isPropertyScrutinyType = exports.PropertyScrutinyType = exports.isUnionProperty = exports.isMapOfListsOfPrimitivesProperty = exports.isMapOfStructsProperty = exports.isPrimitiveMapProperty = exports.isMapProperty = exports.isComplexListProperty = exports.isPrimitiveListProperty = exports.isListProperty = exports.isCollectionProperty = exports.isComplexProperty = exports.isPrimitiveProperty = exports.isScalarProperty = exports.isUpdateType = exports.UpdateType = void 0;
const base_types_1 = require("./base-types");
var UpdateType;
(function (UpdateType) {
    UpdateType["Conditional"] = "Conditional";
    UpdateType["Immutable"] = "Immutable";
    UpdateType["Mutable"] = "Mutable";
})(UpdateType = exports.UpdateType || (exports.UpdateType = {}));
function isUpdateType(str) {
    switch (str) {
        case UpdateType.Conditional:
        case UpdateType.Immutable:
        case UpdateType.Mutable:
            return true;
        default:
            return false;
    }
}
exports.isUpdateType = isUpdateType;
function isScalarProperty(prop) {
    return isPrimitiveProperty(prop)
        || isComplexProperty(prop)
        // A UnionProperty is only Scalar if it defines Types or PrimitiveTypes
        || (isUnionProperty(prop) && !!(prop.Types || prop.PrimitiveTypes));
}
exports.isScalarProperty = isScalarProperty;
function isPrimitiveProperty(prop) {
    return !!prop.PrimitiveType;
}
exports.isPrimitiveProperty = isPrimitiveProperty;
function isComplexProperty(prop) {
    const propType = prop.Type;
    return propType != null && propType !== 'Map' && propType !== 'List';
}
exports.isComplexProperty = isComplexProperty;
function isCollectionProperty(prop) {
    return isListProperty(prop)
        || isMapProperty(prop)
        // A UnionProperty is only Collection if it defines ItemTypes or PrimitiveItemTypes
        || (isUnionProperty(prop) && !!(prop.ItemTypes || prop.PrimitiveItemTypes));
}
exports.isCollectionProperty = isCollectionProperty;
function isListProperty(prop) {
    return prop.Type === 'List';
}
exports.isListProperty = isListProperty;
function isPrimitiveListProperty(prop) {
    return isListProperty(prop) && !!prop.PrimitiveItemType;
}
exports.isPrimitiveListProperty = isPrimitiveListProperty;
function isComplexListProperty(prop) {
    return isListProperty(prop) && !!prop.ItemType;
}
exports.isComplexListProperty = isComplexListProperty;
function isMapProperty(prop) {
    return prop.Type === 'Map';
}
exports.isMapProperty = isMapProperty;
function isPrimitiveMapProperty(prop) {
    return isMapProperty(prop) && !!prop.PrimitiveItemType;
}
exports.isPrimitiveMapProperty = isPrimitiveMapProperty;
function isMapOfStructsProperty(prop) {
    return isMapProperty(prop) &&
        !isPrimitiveMapProperty(prop) &&
        !isMapOfListsOfPrimitivesProperty(prop);
}
exports.isMapOfStructsProperty = isMapOfStructsProperty;
// note: this (and the MapOfListsOfPrimitives type) are not actually valid in the CFN spec!
// they are only here to support our patch of the CFN spec
// to alleviate https://github.com/aws/aws-cdk/issues/3092
function isMapOfListsOfPrimitivesProperty(prop) {
    return isMapProperty(prop) && prop.ItemType === 'List';
}
exports.isMapOfListsOfPrimitivesProperty = isMapOfListsOfPrimitivesProperty;
function isUnionProperty(prop) {
    const castProp = prop;
    return !!(castProp.ItemTypes || castProp.PrimitiveTypes || castProp.Types);
}
exports.isUnionProperty = isUnionProperty;
var PropertyScrutinyType;
(function (PropertyScrutinyType) {
    /**
     * No additional scrutiny
     */
    PropertyScrutinyType["None"] = "None";
    /**
     * This is an IAM policy directly on a resource
     */
    PropertyScrutinyType["InlineResourcePolicy"] = "InlineResourcePolicy";
    /**
     * Either an AssumeRolePolicyDocument or a dictionary of policy documents
     */
    PropertyScrutinyType["InlineIdentityPolicies"] = "InlineIdentityPolicies";
    /**
     * A list of managed policies (on an identity resource)
     */
    PropertyScrutinyType["ManagedPolicies"] = "ManagedPolicies";
    /**
     * A set of ingress rules (on a security group)
     */
    PropertyScrutinyType["IngressRules"] = "IngressRules";
    /**
     * A set of egress rules (on a security group)
     */
    PropertyScrutinyType["EgressRules"] = "EgressRules";
})(PropertyScrutinyType = exports.PropertyScrutinyType || (exports.PropertyScrutinyType = {}));
function isPropertyScrutinyType(str) {
    return PropertyScrutinyType[str] !== undefined;
}
exports.isPropertyScrutinyType = isPropertyScrutinyType;
const tagPropertyNames = {
    FileSystemTags: '',
    HostedZoneTags: '',
    Tags: '',
    UserPoolTags: '',
};
function isTagPropertyName(name) {
    if (undefined === name) {
        return false;
    }
    return tagPropertyNames.hasOwnProperty(name);
}
exports.isTagPropertyName = isTagPropertyName;
/**
 * This function validates that the property **can** be a Tag Property
 *
 * The property is only a Tag if the name of this property is Tags, which is
 * validated using `ResourceType.isTaggable(resource)`.
 */
function isTagProperty(prop) {
    return (isTagPropertyStandard(prop) ||
        isTagPropertyAutoScalingGroup(prop) ||
        isTagPropertyJson(prop) ||
        isTagPropertyStringMap(prop));
}
exports.isTagProperty = isTagProperty;
function isTagPropertyStandard(prop) {
    return (prop.ItemType === 'Tag' ||
        prop.ItemType === 'TagsEntry' ||
        prop.Type === 'Tags' ||
        prop.ItemType === 'TagRef' ||
        prop.ItemType === 'ElasticFileSystemTag' ||
        prop.ItemType === 'HostedZoneTag');
}
exports.isTagPropertyStandard = isTagPropertyStandard;
function isTagPropertyAutoScalingGroup(prop) {
    return prop.ItemType === 'TagProperty';
}
exports.isTagPropertyAutoScalingGroup = isTagPropertyAutoScalingGroup;
function isTagPropertyJson(prop) {
    return prop.PrimitiveType === base_types_1.PrimitiveType.Json;
}
exports.isTagPropertyJson = isTagPropertyJson;
function isTagPropertyStringMap(prop) {
    return prop.PrimitiveItemType === 'String';
}
exports.isTagPropertyStringMap = isTagPropertyStringMap;
//# sourceMappingURL=data:application/json;base64,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