import { TagType } from './cfn-resource';
/**
 * Interface to implement tags.
 *
 * @stability stable
 */
export interface ITaggable {
    /**
     * TagManager to set, remove and format tags.
     *
     * @stability stable
     */
    readonly tags: TagManager;
}
/**
 * Options to configure TagManager behavior.
 *
 * @stability stable
 */
export interface TagManagerOptions {
    /**
     * The name of the property in CloudFormation for these tags.
     *
     * Normally this is `tags`, but Cognito UserPool uses UserPoolTags
     *
     * @default "tags"
     * @stability stable
     */
    readonly tagPropertyName?: string;
}
/**
 * TagManager facilitates a common implementation of tagging for Constructs.
 *
 * @stability stable
 */
export declare class TagManager {
    /**
     * Check whether the given construct is Taggable.
     *
     * @stability stable
     */
    static isTaggable(construct: any): construct is ITaggable;
    /**
     * The property name for tag values.
     *
     * Normally this is `tags` but some resources choose a different name. Cognito
     * UserPool uses UserPoolTags
     *
     * @stability stable
     */
    readonly tagPropertyName: string;
    private readonly tags;
    private readonly priorities;
    private readonly tagFormatter;
    private readonly resourceTypeName;
    private readonly initialTagPriority;
    /**
     * @stability stable
     */
    constructor(tagType: TagType, resourceTypeName: string, tagStructure?: any, options?: TagManagerOptions);
    /**
     * Adds the specified tag to the array of tags.
     *
     * @stability stable
     */
    setTag(key: string, value: string, priority?: number, applyToLaunchedInstances?: boolean): void;
    /**
     * Removes the specified tag from the array if it exists.
     *
     * @param key The tag to remove.
     * @param priority The priority of the remove operation.
     * @stability stable
     */
    removeTag(key: string, priority: number): void;
    /**
     * Renders tags into the proper format based on TagType.
     *
     * @stability stable
     */
    renderTags(): any;
    /**
     * Render the tags in a readable format.
     *
     * @stability stable
     */
    tagValues(): Record<string, string>;
    /**
     * Determine if the aspect applies here.
     *
     * Looks at the include and exclude resourceTypeName arrays to determine if
     * the aspect applies here
     *
     * @stability stable
     */
    applyTagAspectHere(include?: string[], exclude?: string[]): boolean;
    /**
     * Returns true if there are any tags defined.
     *
     * @stability stable
     */
    hasTags(): boolean;
    private _setTag;
    private get sortedTags();
}
