#!/usr/bin/env node
"use strict";
/**
 * automatically creates a module for any CloudFormation namespaces that do not
 * have an AWS construct library.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const fs = require("fs-extra");
const cfnspec = require("../lib");
// don't be a prude:
/* eslint-disable no-console */
/* eslint-disable quote-props */
async function main() {
    const root = path.join(__dirname, '..', '..');
    if (path.basename(root) !== '@aws-cdk') {
        throw new Error(`Something went wrong. We expected ${root} to be the "packages/@aws-cdk" directory. Did you move me?`);
    }
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const version = require('../package.json').version;
    // iterate over all cloudformation namespaces
    for (const namespace of cfnspec.namespaces()) {
        const [moduleFamily, moduleBaseName] = (namespace === 'AWS::Serverless' ? 'AWS::SAM' : namespace).split('::');
        const moduleName = `${moduleFamily}-${moduleBaseName.replace(/V\d+$/, '')}`.toLocaleLowerCase();
        const packagePath = path.join(root, moduleName);
        const lowcaseModuleName = moduleBaseName.toLocaleLowerCase();
        const packageName = `@aws-cdk/${moduleName}`;
        // we already have a module for this namesapce, move on.
        if (await fs.pathExists(packagePath)) {
            const packageJsonPath = path.join(packagePath, 'package.json');
            // eslint-disable-next-line @typescript-eslint/no-require-imports
            const packageJson = require(packageJsonPath);
            let scopes = packageJson['cdk-build'].cloudformation;
            if (typeof scopes === 'string') {
                scopes = [scopes];
            }
            if (scopes.indexOf(namespace) !== -1) {
                // V2-style module is already modeled in the root package, nothing to be done!
                continue;
            }
            else if (await fs.pathExists(path.join(root, `${moduleFamily}-${moduleBaseName}`.toLocaleLowerCase()))) {
                // V2-style package already has it's own package (legacy behavior), nothing to be done!
                continue;
            }
            else {
                // V2-style package needs to be added to it's "V1" package... Get down to business!
                console.error(`Adding ${namespace} to ${packageName}`);
                scopes.push(namespace);
                packageJson['cdk-build'].cloudformation = scopes;
                await fs.writeJson(packageJsonPath, packageJson, { encoding: 'utf-8', spaces: 2 });
                const indexTsPath = path.join(packagePath, 'lib', 'index.ts');
                const indexTs = [
                    (await fs.readFile(indexTsPath, { encoding: 'utf8' })).trimRight(),
                    `// ${namespace} CloudFormation Resources:`,
                    `export * from './${lowcaseModuleName}.generated';`,
                ].join('\n');
                await fs.writeFile(indexTsPath, indexTs, { encoding: 'utf8' });
                continue;
            }
        }
        // dotnet names
        const dotnetPackage = `Amazon.CDK.${moduleFamily}.${moduleBaseName}`;
        // java names
        const javaGroupId = 'software.amazon.awscdk';
        const javaPackage = moduleFamily === 'AWS'
            ? `services.${lowcaseModuleName}`
            : `${moduleFamily.toLocaleLowerCase()}.${lowcaseModuleName}`;
        const javaArtifactId = moduleFamily === 'AWS'
            ? lowcaseModuleName
            : `${moduleFamily.toLocaleLowerCase()}-${lowcaseModuleName}`;
        // python names
        const pythonDistName = `aws-cdk.${moduleName}`;
        const pythonModuleName = pythonDistName.replace(/-/g, '_');
        async function write(relativePath, contents) {
            const fullPath = path.join(packagePath, relativePath);
            const dir = path.dirname(fullPath);
            await fs.mkdirp(dir);
            let data;
            if (typeof contents === 'string') {
                data = contents.trimLeft(); // trim first newline
            }
            else if (Array.isArray(contents)) {
                data = contents.join('\n');
            }
            else if (typeof contents === 'object') {
                data = JSON.stringify(contents, undefined, 2);
            }
            else {
                throw new Error('Invalid type of contents: ' + contents);
            }
            await fs.writeFile(fullPath, data + '\n');
        }
        console.log(`generating module for ${packageName}...`);
        await write('package.json', {
            name: packageName,
            version,
            description: `The CDK Construct Library for ${namespace}`,
            main: 'lib/index.js',
            types: 'lib/index.d.ts',
            jsii: {
                outdir: 'dist',
                projectReferences: true,
                targets: {
                    dotnet: {
                        namespace: dotnetPackage,
                        packageId: dotnetPackage,
                        signAssembly: true,
                        assemblyOriginatorKeyFile: '../../key.snk',
                        iconUrl: 'https://raw.githubusercontent.com/aws/aws-cdk/master/logo/default-256-dark.png',
                    },
                    java: {
                        package: `${javaGroupId}.${javaPackage}`,
                        maven: {
                            groupId: javaGroupId,
                            artifactId: javaArtifactId,
                        },
                    },
                    python: {
                        classifiers: [
                            'Framework :: AWS CDK',
                            'Framework :: AWS CDK :: 1',
                        ],
                        distName: pythonDistName,
                        module: pythonModuleName,
                    },
                },
            },
            repository: {
                type: 'git',
                url: 'https://github.com/aws/aws-cdk.git',
                directory: `packages/${packageName}`,
            },
            homepage: 'https://github.com/aws/aws-cdk',
            scripts: {
                build: 'cdk-build',
                watch: 'cdk-watch',
                lint: 'cdk-lint',
                test: 'cdk-test',
                integ: 'cdk-integ',
                pkglint: 'pkglint -f',
                package: 'cdk-package',
                awslint: 'cdk-awslint',
                cfn2ts: 'cfn2ts',
                'build+test': 'yarn build && yarn test',
                'build+test+package': 'yarn build+test && yarn package',
                compat: 'cdk-compat',
                gen: 'cfn2ts',
                'rosetta:extract': 'yarn --silent jsii-rosetta extract',
            },
            'cdk-build': {
                cloudformation: namespace,
                jest: true,
                env: {
                    AWSLINT_BASE_CONSTRUCT: 'true',
                },
            },
            keywords: [
                'aws',
                'cdk',
                'constructs',
                namespace,
                moduleName,
            ],
            author: {
                name: 'Amazon Web Services',
                url: 'https://aws.amazon.com',
                organization: true,
            },
            license: 'Apache-2.0',
            devDependencies: {
                '@aws-cdk/assert': version,
                'cdk-build-tools': version,
                'cfn2ts': version,
                'pkglint': version,
            },
            dependencies: {
                '@aws-cdk/core': version,
            },
            peerDependencies: {
                '@aws-cdk/core': version,
            },
            engines: {
                node: '>= 10.13.0 <13 || >=13.7.0',
            },
            stability: 'experimental',
            maturity: 'cfn-only',
            awscdkio: {
                announce: false,
            },
            publishConfig: {
                tag: 'latest',
            },
        });
        await write('.gitignore', [
            '*.js',
            '*.js.map',
            '*.d.ts',
            'tsconfig.json',
            'node_modules',
            '*.generated.ts',
            'dist',
            '.jsii',
            '',
            '.LAST_BUILD',
            '.nyc_output',
            'coverage',
            '.nycrc',
            '.LAST_PACKAGE',
            '*.snk',
            'nyc.config.js',
            '!.eslintrc.js',
            '!jest.config.js',
            'junit.xml',
        ]);
        await write('.npmignore', [
            '# Don\'t include original .ts files when doing `npm pack`',
            '*.ts',
            '!*.d.ts',
            'coverage',
            '.nyc_output',
            '*.tgz',
            '',
            'dist',
            '.LAST_PACKAGE',
            '.LAST_BUILD',
            '!*.js',
            '',
            '# Include .jsii',
            '!.jsii',
            '',
            '*.snk',
            '',
            '*.tsbuildinfo',
            '',
            'tsconfig.json',
            '',
            '.eslintrc.js',
            'jest.config.js',
            '',
            '# exclude cdk artifacts',
            '**/cdk.out',
            'junit.xml',
            'test/',
        ]);
        await write('lib/index.ts', [
            `// ${namespace} CloudFormation Resources:`,
            `export * from './${lowcaseModuleName}.generated';`,
        ]);
        await write(`test/${lowcaseModuleName}.test.ts`, [
            "import '@aws-cdk/assert/jest';",
            "import {} from '../lib';",
            '',
            "test('No tests are specified for this package', () => {",
            '  expect(true).toBe(true);',
            '});',
        ]);
        await write('README.md', [
            `# ${namespace} Construct Library`,
            '<!--BEGIN STABILITY BANNER-->',
            '',
            '---',
            '',
            '![cfn-resources: Stable](https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge)',
            '',
            '> All classes with the `Cfn` prefix in this module ([CFN Resources]) are always stable and safe to use.',
            '>',
            '> [CFN Resources]: https://docs.aws.amazon.com/cdk/latest/guide/constructs.html#constructs_lib',
            '',
            '---',
            '',
            '<!--END STABILITY BANNER-->',
            '',
            'This module is part of the [AWS Cloud Development Kit](https://github.com/aws/aws-cdk) project.',
            '',
            '```ts',
            `import ${lowcaseModuleName} = require('${packageName}');`,
            '```',
        ]);
        await write('.eslintrc.js', [
            "const baseConfig = require('cdk-build-tools/config/eslintrc');",
            "baseConfig.parserOptions.project = __dirname + '/tsconfig.json';",
            'module.exports = baseConfig;',
        ]);
        await write('jest.config.js', [
            "const baseConfig = require('cdk-build-tools/config/jest.config');",
            'module.exports = baseConfig;',
        ]);
        const templateDir = path.join(__dirname, 'template');
        for (const file of await fs.readdir(templateDir)) {
            await fs.copy(path.join(templateDir, file), path.join(packagePath, file));
        }
        await addDependencyToMegaPackage(path.join('@aws-cdk', 'cloudformation-include'), packageName, version, ['dependencies', 'peerDependencies']);
        await addDependencyToMegaPackage('aws-cdk-lib', packageName, version, ['devDependencies']);
        await addDependencyToMegaPackage('monocdk', packageName, version, ['devDependencies']);
        await addDependencyToMegaPackage('decdk', packageName, version, ['dependencies']);
    }
}
/**
 * A few of our packages (e.g., decdk, aws-cdk-lib) require a dependency on every service package.
 * This automates adding the dependency (and peer dependency) to the package.json.
 */
async function addDependencyToMegaPackage(megaPackageName, packageName, version, dependencyTypes) {
    const packageJsonPath = path.join(__dirname, '..', '..', '..', megaPackageName, 'package.json');
    const packageJson = JSON.parse(await fs.readFile(packageJsonPath, 'utf8'));
    dependencyTypes.forEach(dependencyType => {
        const unorderedDeps = {
            ...packageJson[dependencyType],
            [packageName]: version,
        };
        packageJson[dependencyType] = {};
        Object.keys(unorderedDeps).sort().forEach(k => {
            packageJson[dependencyType][k] = unorderedDeps[k];
        });
    });
    await fs.writeFile(packageJsonPath, JSON.stringify(packageJson, null, 2) + '\n');
}
main().catch(e => {
    console.error(e);
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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