"use strict";
// ----------------------------------------------------
// CROSS REFERENCES
// ----------------------------------------------------
Object.defineProperty(exports, "__esModule", { value: true });
exports.referenceNestedStackValueInParent = exports.resolveReferences = void 0;
const cfn_element_1 = require("../cfn-element");
const cfn_output_1 = require("../cfn-output");
const cfn_parameter_1 = require("../cfn-parameter");
const names_1 = require("../names");
const stack_1 = require("../stack");
const token_1 = require("../token");
const cfn_reference_1 = require("./cfn-reference");
const resolve_1 = require("./resolve");
/**
 * This is called from the App level to resolve all references defined. Each
 * reference is resolved based on it's consumption context.
 */
function resolveReferences(scope) {
    const edges = findAllReferences(scope);
    for (const { source, value } of edges) {
        const consumer = stack_1.Stack.of(source);
        // resolve the value in the context of the consumer
        if (!value.hasValueForStack(consumer)) {
            const resolved = resolveValue(consumer, value);
            value.assignValueForStack(consumer, resolved);
        }
    }
}
exports.resolveReferences = resolveReferences;
/**
 * Resolves the value for `reference` in the context of `consumer`.
 */
function resolveValue(consumer, reference) {
    const producer = stack_1.Stack.of(reference.target);
    // produce and consumer stacks are the same, we can just return the value itself.
    if (producer === consumer) {
        return reference;
    }
    // unsupported: stacks from different apps
    if (producer.node.root !== consumer.node.root) {
        throw new Error('Cannot reference across apps. Consuming and producing stacks must be defined within the same CDK app.');
    }
    // unsupported: stacks are not in the same environment
    if (producer.environment !== consumer.environment) {
        throw new Error(`Stack "${consumer.node.path}" cannot consume a cross reference from stack "${producer.node.path}". ` +
            'Cross stack references are only supported for stacks deployed to the same environment or between nested stacks and their parent stack');
    }
    // ----------------------------------------------------------------------
    // consumer is nested in the producer (directly or indirectly)
    // ----------------------------------------------------------------------
    // if the consumer is nested within the producer (directly or indirectly),
    // wire through a CloudFormation parameter and then resolve the reference with
    // the parent stack as the consumer.
    if (consumer.nestedStackParent && isNested(consumer, producer)) {
        const parameterValue = resolveValue(consumer.nestedStackParent, reference);
        return createNestedStackParameter(consumer, reference, parameterValue);
    }
    // ----------------------------------------------------------------------
    // producer is a nested stack
    // ----------------------------------------------------------------------
    // if the producer is nested, always publish the value through a
    // cloudformation output and resolve recursively with the Fn::GetAtt
    // of the output in the parent stack.
    // one might ask, if the consumer is not a parent of the producer,
    // why not just use export/import? the reason is that we cannot
    // generate an "export name" from a nested stack because the export
    // name must contain the stack name to ensure uniqueness, and we
    // don't know the stack name of a nested stack before we deploy it.
    // therefore, we can only export from a top-level stack.
    if (producer.nested) {
        const outputValue = createNestedStackOutput(producer, reference);
        return resolveValue(consumer, outputValue);
    }
    // ----------------------------------------------------------------------
    // export/import
    // ----------------------------------------------------------------------
    // export the value through a cloudformation "export name" and use an
    // Fn::ImportValue in the consumption site.
    // add a dependency between the producer and the consumer. dependency logic
    // will take care of applying the dependency at the right level (e.g. the
    // top-level stacks).
    consumer.addDependency(producer, `${consumer.node.path} -> ${reference.target.node.path}.${reference.displayName}`);
    return createImportValue(reference);
}
/**
 * Finds all the CloudFormation references in a construct tree.
 */
function findAllReferences(root) {
    const result = new Array();
    for (const consumer of root.node.findAll()) {
        // include only CfnElements (i.e. resources)
        if (!cfn_element_1.CfnElement.isCfnElement(consumer)) {
            continue;
        }
        try {
            const tokens = resolve_1.findTokens(consumer, () => consumer._toCloudFormation());
            // iterate over all the tokens (e.g. intrinsic functions, lazies, etc) that
            // were found in the cloudformation representation of this resource.
            for (const token of tokens) {
                // include only CfnReferences (i.e. "Ref" and "Fn::GetAtt")
                if (!cfn_reference_1.CfnReference.isCfnReference(token)) {
                    continue;
                }
                result.push({
                    source: consumer,
                    value: token,
                });
            }
        }
        catch (e) {
            // Note: it might be that the properties of the CFN object aren't valid.
            // This will usually be preventatively caught in a construct's validate()
            // and turned into a nicely descriptive error, but we're running prepare()
            // before validate(). Swallow errors that occur because the CFN layer
            // doesn't validate completely.
            //
            // This does make the assumption that the error will not be rectified,
            // but the error will be thrown later on anyway. If the error doesn't
            // get thrown down the line, we may miss references.
            if (e.type === 'CfnSynthesisError') {
                continue;
            }
            throw e;
        }
    }
    return result;
}
// ------------------------------------------------------------------------------------------------
// export/import
// ------------------------------------------------------------------------------------------------
/**
 * Imports a value from another stack by creating an "Output" with an "ExportName"
 * and returning an "Fn::ImportValue" token.
 */
function createImportValue(reference) {
    const exportingStack = stack_1.Stack.of(reference.target);
    const importExpr = exportingStack.exportValue(reference);
    // I happen to know this returns a Fn.importValue() which implements Intrinsic.
    return token_1.Tokenization.reverseCompleteString(importExpr);
}
// ------------------------------------------------------------------------------------------------
// nested stacks
// ------------------------------------------------------------------------------------------------
/**
 * Adds a CloudFormation parameter to a nested stack and assigns it with the
 * value of the reference.
 */
function createNestedStackParameter(nested, reference, value) {
    // we call "this.resolve" to ensure that tokens do not creep in (for example, if the reference display name includes tokens)
    const paramId = nested.resolve(`reference-to-${names_1.Names.nodeUniqueId(reference.target.node)}.${reference.displayName}`);
    let param = nested.node.tryFindChild(paramId);
    if (!param) {
        param = new cfn_parameter_1.CfnParameter(nested, paramId, { type: 'String' });
        // Ugly little hack until we move NestedStack to this module.
        if (!('setParameter' in nested)) {
            throw new Error('assertion failed: nested stack should have a "setParameter" method');
        }
        nested.setParameter(param.logicalId, token_1.Token.asString(value));
    }
    return param.value;
}
/**
 * Adds a CloudFormation output to a nested stack and returns an "Fn::GetAtt"
 * intrinsic that can be used to reference this output in the parent stack.
 */
function createNestedStackOutput(producer, reference) {
    const outputId = `${names_1.Names.nodeUniqueId(reference.target.node)}${reference.displayName}`;
    let output = producer.node.tryFindChild(outputId);
    if (!output) {
        output = new cfn_output_1.CfnOutput(producer, outputId, { value: token_1.Token.asString(reference) });
    }
    if (!producer.nestedStackResource) {
        throw new Error('assertion failed');
    }
    return producer.nestedStackResource.getAtt(`Outputs.${output.logicalId}`);
}
/**
 * Translate a Reference into a nested stack into a value in the parent stack
 *
 * Will create Outputs along the chain of Nested Stacks, and return the final `{ Fn::GetAtt }`.
 */
function referenceNestedStackValueInParent(reference, targetStack) {
    let currentStack = stack_1.Stack.of(reference.target);
    if (currentStack !== targetStack && !isNested(currentStack, targetStack)) {
        throw new Error(`Referenced resource must be in stack '${targetStack.node.path}', got '${reference.target.node.path}'`);
    }
    while (currentStack !== targetStack) {
        reference = createNestedStackOutput(stack_1.Stack.of(reference.target), reference);
        currentStack = stack_1.Stack.of(reference.target);
    }
    return reference;
}
exports.referenceNestedStackValueInParent = referenceNestedStackValueInParent;
/**
 * @returns true if this stack is a direct or indirect parent of the nested
 * stack `nested`.
 *
 * If `child` is not a nested stack, always returns `false` because it can't
 * have a parent, dah.
 */
function isNested(nested, parent) {
    // if the parent is a direct parent
    if (nested.nestedStackParent === parent) {
        return true;
    }
    // we reached a top-level (non-nested) stack without finding the parent
    if (!nested.nestedStackParent) {
        return false;
    }
    // recurse with the child's direct parent
    return isNested(nested.nestedStackParent, parent);
}
//# sourceMappingURL=data:application/json;base64,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