import * as iam from 'aws-cdk-lib/aws-iam';
import { IResource, Resource } from 'aws-cdk-lib/core';
import { Construct } from 'constructs';
import { Policy, PolicyDefinitionProperty } from './policy';
export interface ISchema {
    cedarJson: string;
}
export interface IValidationSettings {
    mode: ValidationSettingsMode;
}
/**
 * Validation Settings mode, according to the Cloudformation PolicyStore resource
 */
export declare enum ValidationSettingsMode {
    OFF = "OFF",
    STRICT = "STRICT"
}
export interface IPolicyStore extends IResource {
    /**
     * ARN of the Policy Store.
     *
     * @attribute
     */
    readonly policyStoreArn: string;
    /**
     * ID of the Policy Store.
     *
     * @attribute
     */
    readonly policyStoreId: string;
    /**
     *
     * Adds an IAM policy statement associated with this policy store to an IAM
     * principal's policy.
     *
     * @param grantee The principal (no-op if undefined)
     * @param actions The set of actions to allow (i.e. "verifiedpermissions:IsAuthorized", "verifiedpermissions:ListPolicies", ...)
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Permits an IAM principal all read operations on the policy store: GetIdentitySource,
     * GetPolicy, GetPolicyStore, GetPolicyTemplate,
     * GetSchema, ListIdentitySources, ListPolicies, ListPolicyTemplates.
     *
     * @param grantee
     */
    grantRead(grantee: iam.IGrantable): iam.Grant;
    /**
     * Permits an IAM principal all write & read operations on the policy store: CreateIdentitySource,
     * CreatePolicy,CreatePolicyTemplate, DeleteIdentitySource, DeletePolicy,
     * DeletePolicyTemplate, PutSchema, UpdateIdentitySource, UpdatePolicy, UpdatePolicyTemplate.
     *
     * @param grantee
     */
    grantWrite(grantee: iam.IGrantable): iam.Grant;
    /**
     * Permits an IAM principal all auth operations on the policy store:
     * IsAuthorized, IsAuthorizedWithToken
     * @param grantee
     */
    grantAuth(grantee: iam.IGrantable): iam.Grant;
}
export interface PolicyStoreProps {
    /**
     * This attribute is not required from an API point of view.
     * It represents the schema (in Cedar) to be applied to the PolicyStore.
     *
     * @default - No schema.
     */
    readonly schema?: ISchema;
    /**
     * The policy store's validation settings.
     *
     * @default - If not provided, the Policy store will be created with ValidationSettingsMode = "OFF"
     */
    readonly validationSettings: IValidationSettings;
    /**
     * The policy store's description
     *
     * @default - No description.
     */
    readonly description?: string;
}
export interface AddPolicyOptions {
    /**
     * The id of the Policy.
     */
    readonly policyId: string;
    /**
     * The configuration of the Policy.
     */
    readonly policyConfiguration: PolicyDefinitionProperty;
}
export interface PolicyStoreAttributes {
    /**
     * The ARN of the Amazon Verified Permissions Policy Store.
     * One of this, or `policyStoreId`, is required.
     *
     * @default - no PolicyStore arn
     */
    readonly policyStoreArn?: string;
    /**
     * The id of the Amazon Verified Permissions PolicyStore.
     * One of this, or `policyStoreArn`, is required.
     *
     * @default - no PolicyStore id
     */
    readonly policyStoreId?: string;
}
declare abstract class PolicyStoreBase extends Resource implements IPolicyStore {
    abstract readonly policyStoreArn: string;
    abstract readonly policyStoreId: string;
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    grantAuth(grantee: iam.IGrantable): iam.Grant;
    grantRead(grantee: iam.IGrantable): iam.Grant;
    grantWrite(grantee: iam.IGrantable): iam.Grant;
}
export declare class PolicyStore extends PolicyStoreBase {
    /**
     * Create a PolicyStore construct that represents an external policy store via policy store id.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param policyStoreId The PolicyStore's id.
     */
    static fromPolicyStoreId(scope: Construct, id: string, policyStoreId: string): IPolicyStore;
    /**
     * Create a PolicyStore construct that represents an external PolicyStore via policy store arn.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param policyStoreArn The PolicyStore's ARN.
     */
    static fromPolicyStoreArn(scope: Construct, id: string, policyStoreArn: string): IPolicyStore;
    /**
     * Creates a PolicyStore construct that represents an external Policy Store.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `PolicyStoreAttributes` object.
     */
    static fromPolicyStoreAttributes(scope: Construct, id: string, attrs: PolicyStoreAttributes): IPolicyStore;
    private readonly policyStore;
    /**
     * ARN of the Policy Store.
     *
     * @attribute
     */
    readonly policyStoreArn: string;
    /**
     * ID of the Policy Store.
     *
     * @attribute
     */
    readonly policyStoreId: string;
    /**
     * Name of the Policy Store.
     */
    readonly policyStoreName: string;
    /**
     * Schema definition of the Policy Store.
     */
    readonly schema?: ISchema;
    /**
     * Validation Settings of the Policy Store.
     */
    readonly validationSettings: IValidationSettings;
    /**
     * Description of the Policy Store
     */
    readonly description?: string;
    constructor(scope: Construct, id: string, props?: PolicyStoreProps);
    /**
     * Add multiple policies to the policy store
     *
     * @param policyDefinitions An array of policy options for the policy stores policies.
     * @returns An array of created policy constructs.
     */
    addPolicies(policyDefinitions: AddPolicyOptions[]): Policy[];
    /**
     * Takes in an absolute path to a directory containing .cedar files and adds the contents of each
     * .cedar file as policies to this policy store. Parses the policies with cedar-wasm and, if the policy store has a schema,
     * performs semantic validation of the policies as well.
     * @param absolutePath a string representing an absolute path to the directory containing your policies
     * @returns An array of created Policy constructs.
     */
    addPoliciesFromPath(absolutePath: string): Policy[];
}
export {};
