"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HostPathVolumeType = exports.AzureDiskPersistentVolumeCachingMode = exports.AzureDiskPersistentVolumeKind = exports.EmptyDirMedium = exports.Volume = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const constructs_1 = require("constructs");
const k8s = require("./imports/k8s");
/**
 * Volume represents a named volume in a pod that may be accessed by any
 * container in the pod.
 *
 * Docker also has a concept of volumes, though it is somewhat looser and less
 * managed. In Docker, a volume is simply a directory on disk or in another
 * Container. Lifetimes are not managed and until very recently there were only
 * local-disk-backed volumes. Docker now provides volume drivers, but the
 * functionality is very limited for now (e.g. as of Docker 1.7 only one volume
 * driver is allowed per Container and there is no way to pass parameters to
 * volumes).
 *
 * A Kubernetes volume, on the other hand, has an explicit lifetime - the same
 * as the Pod that encloses it. Consequently, a volume outlives any Containers
 * that run within the Pod, and data is preserved across Container restarts. Of
 * course, when a Pod ceases to exist, the volume will cease to exist, too.
 * Perhaps more importantly than this, Kubernetes supports many types of
 * volumes, and a Pod can use any number of them simultaneously.
 *
 * At its core, a volume is just a directory, possibly with some data in it,
 * which is accessible to the Containers in a Pod. How that directory comes to
 * be, the medium that backs it, and the contents of it are determined by the
 * particular volume type used.
 *
 * To use a volume, a Pod specifies what volumes to provide for the Pod (the
 * .spec.volumes field) and where to mount those into Containers (the
 * .spec.containers[*].volumeMounts field).
 *
 * A process in a container sees a filesystem view composed from their Docker
 * image and volumes. The Docker image is at the root of the filesystem
 * hierarchy, and any volumes are mounted at the specified paths within the
 * image. Volumes can not mount onto other volumes
 */
class Volume extends constructs_1.Construct {
    constructor(scope, id, name, config) {
        super(scope, id);
        this.name = name;
        this.config = config;
    }
    /**
     * Mounts an Amazon Web Services (AWS) EBS volume into your pod.
     * Unlike emptyDir, which is erased when a pod is removed, the contents of an EBS volume are
     * persisted and the volume is unmounted. This means that an EBS volume can be pre-populated with data,
     * and that data can be shared between pods.
     *
     * There are some restrictions when using an awsElasticBlockStore volume:
     *
     * - the nodes on which pods are running must be AWS EC2 instances.
     * - those instances need to be in the same region and availability zone as the EBS volume.
     * - EBS only supports a single EC2 instance mounting a volume.
     */
    static fromAwsElasticBlockStore(scope, id, volumeId, options = {}) {
        return new Volume(scope, id, options.name ?? `ebs-${volumeId}`, {
            awsElasticBlockStore: {
                volumeId,
                fsType: options.fsType ?? 'ext4',
                partition: options.partition,
                readOnly: options.readOnly ?? false,
            },
        });
    }
    /**
     * Mounts a Microsoft Azure Data Disk into a pod.
     */
    static fromAzureDisk(scope, id, diskName, diskUri, options = {}) {
        return new Volume(scope, id, options.name ?? `azuredisk-${diskName}`, {
            azureDisk: {
                diskName,
                diskUri,
                cachingMode: options.cachingMode ?? AzureDiskPersistentVolumeCachingMode.NONE,
                fsType: options.fsType ?? 'ext4',
                kind: options.kind ?? AzureDiskPersistentVolumeKind.SHARED,
                readOnly: options.readOnly ?? false,
            },
        });
    }
    /**
     * Mounts a Google Compute Engine (GCE) persistent disk (PD) into your Pod.
     * Unlike emptyDir, which is erased when a pod is removed, the contents of a PD are
     * preserved and the volume is merely unmounted. This means that a PD can be pre-populated
     * with data, and that data can be shared between pods.
     *
     * There are some restrictions when using a gcePersistentDisk:
     *
     * - the nodes on which Pods are running must be GCE VMs
     * - those VMs need to be in the same GCE project and zone as the persistent disk
     */
    static fromGcePersistentDisk(scope, id, pdName, options = {}) {
        return new Volume(scope, id, options.name ?? `gcedisk-${pdName}`, {
            gcePersistentDisk: {
                pdName,
                fsType: options.fsType ?? 'ext4',
                partition: options.partition,
                readOnly: options.readOnly ?? false,
            },
        });
    }
    /**
     * Populate the volume from a ConfigMap.
     *
     * The configMap resource provides a way to inject configuration data into
     * Pods. The data stored in a ConfigMap object can be referenced in a volume
     * of type configMap and then consumed by containerized applications running
     * in a Pod.
     *
     * When referencing a configMap object, you can simply provide its name in the
     * volume to reference it. You can also customize the path to use for a
     * specific entry in the ConfigMap.
     *
     * @param configMap The config map to use to populate the volume.
     * @param options Options
     */
    static fromConfigMap(scope, id, configMap, options = {}) {
        return new Volume(scope, id, options.name ?? `configmap-${configMap.name}`, {
            configMap: {
                name: configMap.name,
                defaultMode: options.defaultMode,
                optional: options.optional,
                items: Volume.renderItems(options.items),
            },
        });
    }
    /**
     * An emptyDir volume is first created when a Pod is assigned to a Node, and
     * exists as long as that Pod is running on that node. As the name says, it is
     * initially empty. Containers in the Pod can all read and write the same
     * files in the emptyDir volume, though that volume can be mounted at the same
     * or different paths in each Container. When a Pod is removed from a node for
     * any reason, the data in the emptyDir is deleted forever.
     *
     * @see http://kubernetes.io/docs/user-guide/volumes#emptydir
     *
     * @param options - Additional options.
     */
    static fromEmptyDir(scope, id, name, options = {}) {
        return new Volume(scope, id, name, {
            emptyDir: {
                medium: options.medium,
                sizeLimit: options.sizeLimit
                    ? k8s.Quantity.fromString(`${options.sizeLimit.toMebibytes()}Mi`)
                    : undefined,
            },
        });
    }
    /**
     * Populate the volume from a Secret.
     *
     * A secret volume is used to pass sensitive information, such as passwords, to Pods.
     * You can store secrets in the Kubernetes API and mount them as files for use by pods
     * without coupling to Kubernetes directly.
     *
     * secret volumes are backed by tmpfs (a RAM-backed filesystem)
     * so they are never written to non-volatile storage.
     *
     * @see https://kubernetes.io/docs/concepts/storage/volumes/#secret
     *
     * @param secr The secret to use to populate the volume.
     * @param options Options
     */
    static fromSecret(scope, id, secr, options = {}) {
        return new Volume(scope, id, options.name ?? `secret-${secr.name}`, {
            secret: {
                secretName: secr.name,
                defaultMode: options.defaultMode,
                optional: options.optional,
                items: Volume.renderItems(options.items),
            },
        });
    }
    /**
     * Used to mount a PersistentVolume into a Pod.
     * PersistentVolumeClaims are a way for users to "claim" durable storage (such as a GCE PersistentDisk or an iSCSI volume)
     * without knowing the details of the particular cloud environment.
     *
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/
     */
    static fromPersistentVolumeClaim(scope, id, claim, options = {}) {
        return new Volume(scope, id, options.name ?? `pvc-${claim.name}`, {
            persistentVolumeClaim: {
                claimName: claim.name,
                readOnly: options.readOnly ?? false,
            },
        });
    }
    /**
     * Used to mount a file or directory from the host node's filesystem into a Pod.
     * This is not something that most Pods will need, but it offers a powerful
     * escape hatch for some applications.
     *
     * @see https://kubernetes.io/docs/concepts/storage/volumes/#hostpath
     */
    static fromHostPath(scope, id, name, options) {
        return new Volume(scope, id, name, {
            hostPath: {
                path: options.path,
                type: options.type ?? HostPathVolumeType.DEFAULT,
            },
        });
    }
    asVolume() {
        return this;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            name: this.name,
            ...this.config,
        };
    }
}
exports.Volume = Volume;
_a = JSII_RTTI_SYMBOL_1;
Volume[_a] = { fqn: "cdk8s-plus-23.Volume", version: "2.3.15" };
/**
  * @internal
 */
Volume.renderItems = (items) => {
    if (!items) {
        return undefined;
    }
    const result = new Array();
    for (const key of Object.keys(items).sort()) {
        result.push({
            key,
            path: items[key].path,
            mode: items[key].mode,
        });
    }
    return result;
};
/**
 * The medium on which to store the volume.
 */
var EmptyDirMedium;
(function (EmptyDirMedium) {
    /**
     * The default volume of the backing node.
     */
    EmptyDirMedium["DEFAULT"] = "";
    /**
     * Mount a tmpfs (RAM-backed filesystem) for you instead. While tmpfs is very
     * fast, be aware that unlike disks, tmpfs is cleared on node reboot and any
     * files you write will count against your Container's memory limit.
     */
    EmptyDirMedium["MEMORY"] = "Memory";
})(EmptyDirMedium = exports.EmptyDirMedium || (exports.EmptyDirMedium = {}));
/**
 * Azure Disk kinds.
 */
var AzureDiskPersistentVolumeKind;
(function (AzureDiskPersistentVolumeKind) {
    /**
     * Multiple blob disks per storage account.
     */
    AzureDiskPersistentVolumeKind["SHARED"] = "Shared";
    /**
     * Single blob disk per storage account.
     */
    AzureDiskPersistentVolumeKind["DEDICATED"] = "Dedicated";
    /**
     * Azure managed data disk.
     */
    AzureDiskPersistentVolumeKind["MANAGED"] = "Managed";
})(AzureDiskPersistentVolumeKind = exports.AzureDiskPersistentVolumeKind || (exports.AzureDiskPersistentVolumeKind = {}));
/**
 * Azure disk caching modes.
 */
var AzureDiskPersistentVolumeCachingMode;
(function (AzureDiskPersistentVolumeCachingMode) {
    /**
     * None.
     */
    AzureDiskPersistentVolumeCachingMode["NONE"] = "None";
    /**
     * ReadOnly.
     */
    AzureDiskPersistentVolumeCachingMode["READ_ONLY"] = "ReadOnly";
    /**
     * ReadWrite.
     */
    AzureDiskPersistentVolumeCachingMode["READ_WRITE"] = "ReadWrite";
})(AzureDiskPersistentVolumeCachingMode = exports.AzureDiskPersistentVolumeCachingMode || (exports.AzureDiskPersistentVolumeCachingMode = {}));
/**
 * Host path types.
 */
var HostPathVolumeType;
(function (HostPathVolumeType) {
    /**
     * Empty string (default) is for backward compatibility, which means that no
     * checks will be performed before mounting the hostPath volume.
     */
    HostPathVolumeType["DEFAULT"] = "";
    /**
     * If nothing exists at the given path, an empty directory will be created
     * there as needed with permission set to 0755, having the same group and
     * ownership with Kubelet.
     */
    HostPathVolumeType["DIRECTORY_OR_CREATE"] = "DirectoryOrCreate";
    /**
     * A directory must exist at the given path.
     */
    HostPathVolumeType["DIRECTORY"] = "Directory";
    /**
     * If nothing exists at the given path, an empty file will be created there
     * as needed with permission set to 0644, having the same group and ownership
     * with Kubelet.
     */
    HostPathVolumeType["FILE_OR_CREATE"] = "FileOrCreate";
    /**
     * A file must exist at the given path.
     */
    HostPathVolumeType["FILE"] = "File";
    /**
     * A UNIX socket must exist at the given path.
     */
    HostPathVolumeType["SOCKET"] = "Socket";
    /**
     * A character device must exist at the given path.
     */
    HostPathVolumeType["CHAR_DEVICE"] = "CharDevice";
    /**
     * A block device must exist at the given path.
     */
    HostPathVolumeType["BLOCK_DEVICE"] = "BlockDevice";
})(HostPathVolumeType = exports.HostPathVolumeType || (exports.HostPathVolumeType = {}));
//# sourceMappingURL=data:application/json;base64,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