"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
const construct_compat_1 = require("./construct-compat");
const lazy_1 = require("./lazy");
const token_1 = require("./token");
const CFN_ELEMENT_SYMBOL = Symbol.for('@aws-cdk/core.CfnElement');
/**
 * An element of a CloudFormation stack.
 */
class CfnElement extends construct_compat_1.Construct {
    /**
     * Creates an entity and binds it to a tree.
     * Note that the root of the tree must be a Stack object (not just any Root).
     *
     * @param scope The parent construct
     * @param props Construct properties
     */
    constructor(scope, id) {
        super(scope, id);
        Object.defineProperty(this, CFN_ELEMENT_SYMBOL, { value: true });
        this.stack = stack_1.Stack.of(this);
        this.logicalId = lazy_1.Lazy.stringValue({ produce: () => this.synthesizeLogicalId() }, {
            displayHint: `${notTooLong(this.node.path)}.LogicalID`
        });
        this.node.addMetadata(cxapi.LOGICAL_ID_METADATA_KEY, this.logicalId, this.constructor);
    }
    /**
     * Returns `true` if a construct is a stack element (i.e. part of the
     * synthesized cloudformation template).
     *
     * Uses duck-typing instead of `instanceof` to allow stack elements from different
     * versions of this library to be included in the same stack.
     *
     * @returns The construct as a stack element or undefined if it is not a stack element.
     */
    static isCfnElement(x) {
        return CFN_ELEMENT_SYMBOL in x;
    }
    /**
     * Overrides the auto-generated logical ID with a specific ID.
     * @param newLogicalId The new logical ID to use for this stack element.
     */
    overrideLogicalId(newLogicalId) {
        this._logicalIdOverride = newLogicalId;
    }
    /**
     * @returns the stack trace of the point where this Resource was created from, sourced
     *      from the +metadata+ entry typed +aws:cdk:logicalId+, and with the bottom-most
     *      node +internal+ entries filtered.
     */
    get creationStack() {
        const trace = this.node.metadata.find(md => md.type === cxapi.LOGICAL_ID_METADATA_KEY).trace;
        if (!trace) {
            return [];
        }
        return filterStackTrace(trace);
        function filterStackTrace(stack) {
            const result = Array.of(...stack);
            while (result.length > 0 && shouldFilter(result[result.length - 1])) {
                result.pop();
            }
            // It's weird if we filtered everything, so return the whole stack...
            return result.length === 0 ? stack : result;
        }
        function shouldFilter(str) {
            return str.match(/[^(]+\(internal\/.*/) !== null;
        }
    }
    /**
     * Called during synthesize to render the logical ID of this element. If
     * `overrideLogicalId` was it will be used, otherwise, we will allocate the
     * logical ID through the stack.
     */
    synthesizeLogicalId() {
        if (this._logicalIdOverride) {
            return this._logicalIdOverride;
        }
        else {
            return this.stack.getLogicalId(this);
        }
    }
}
exports.CfnElement = CfnElement;
/**
 * Base class for referenceable CloudFormation constructs which are not Resources
 *
 * These constructs are things like Conditions and Parameters, can be
 * referenced by taking the `.ref` attribute.
 *
 * Resource constructs do not inherit from CfnRefElement because they have their
 * own, more specific types returned from the .ref attribute. Also, some
 * resources aren't referenceable at all (such as BucketPolicies or GatewayAttachments).
 */
class CfnRefElement extends CfnElement {
    /**
     * Return a string that will be resolved to a CloudFormation `{ Ref }` for this element.
     *
     * If, by any chance, the intrinsic reference of a resource is not a string, you could
     * coerce it to an IResolvable through `Lazy.any({ produce: resource.ref })`.
     */
    get ref() {
        return token_1.Token.asString(cfn_reference_1.CfnReference.for(this, 'Ref'));
    }
}
exports.CfnRefElement = CfnRefElement;
function notTooLong(x) {
    if (x.length < 100) {
        return x;
    }
    return x.substr(0, 47) + '...' + x.substr(x.length - 47);
}
const cfn_reference_1 = require("./private/cfn-reference");
const stack_1 = require("./stack");
//# sourceMappingURL=data:application/json;base64,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