"use strict";
const fc = require("fast-check");
const _ = require("lodash");
const nodeunit = require("nodeunit");
const lib_1 = require("../lib");
const intrinsic_1 = require("../lib/private/intrinsic");
function asyncTest(cb) {
    return async (test) => {
        let error;
        try {
            await cb(test);
        }
        catch (e) {
            error = e;
        }
        finally {
            test.doesNotThrow(() => {
                if (error) {
                    throw error;
                }
            });
            test.done();
        }
    };
}
const nonEmptyString = fc.string(1, 16);
const tokenish = fc.array(nonEmptyString, 2, 2).map(arr => ({ [arr[0]]: arr[1] }));
const anyValue = fc.oneof(nonEmptyString, tokenish);
function stringListToken(o) {
    return lib_1.Token.asList(new intrinsic_1.Intrinsic(o));
}
function stringToken(o) {
    return lib_1.Token.asString(new intrinsic_1.Intrinsic(o));
}
module.exports = nodeunit.testCase({
    'eager resolution for non-tokens': {
        'Fn.select'(test) {
            test.deepEqual(lib_1.Fn.select(2, ['hello', 'you', 'dude']), 'dude');
            test.done();
        },
        'Fn.split'(test) {
            test.deepEqual(lib_1.Fn.split(':', 'hello:world:yeah'), ['hello', 'world', 'yeah']);
            test.done();
        }
    },
    'FnJoin': {
        'rejects empty list of arguments to join'(test) {
            test.throws(() => lib_1.Fn.join('.', []));
            test.done();
        },
        'collapse nested FnJoins even if they contain tokens'(test) {
            const stack = new lib_1.Stack();
            const obj = lib_1.Fn.join('', [
                'a',
                lib_1.Fn.join('', [lib_1.Fn.getAtt('a', 'bc').toString(), 'c']),
                'd'
            ]);
            test.deepEqual(stack.resolve(obj), { 'Fn::Join': ["",
                    [
                        "a",
                        { 'Fn::GetAtt': ['a', 'bc'] },
                        'cd',
                    ]
                ] });
            test.done();
        },
        'resolves to the value if only one value is joined': asyncTest(async () => {
            const stack = new lib_1.Stack();
            fc.assert(fc.property(fc.string(), anyValue, (delimiter, value) => _.isEqual(stack.resolve(lib_1.Fn.join(delimiter, [value])), value)), { verbose: true });
        }),
        'pre-concatenates string literals': asyncTest(async () => {
            const stack = new lib_1.Stack();
            fc.assert(fc.property(fc.string(), fc.array(nonEmptyString, 1, 15), (delimiter, values) => stack.resolve(lib_1.Fn.join(delimiter, values)) === values.join(delimiter)), { verbose: true });
        }),
        'pre-concatenates around tokens': asyncTest(async () => {
            const stack = new lib_1.Stack();
            fc.assert(fc.property(fc.string(), fc.array(nonEmptyString, 1, 3), tokenish, fc.array(nonEmptyString, 1, 3), (delimiter, prefix, obj, suffix) => _.isEqual(stack.resolve(lib_1.Fn.join(delimiter, [...prefix, stringToken(obj), ...suffix])), { 'Fn::Join': [delimiter, [prefix.join(delimiter), obj, suffix.join(delimiter)]] })), { verbose: true, seed: 1539874645005, path: "0:0:0:0:0:0:0:0:0" });
        }),
        'flattens joins nested under joins with same delimiter': asyncTest(async () => {
            const stack = new lib_1.Stack();
            fc.assert(fc.property(fc.string(), fc.array(anyValue), fc.array(anyValue, 1, 3), fc.array(anyValue), (delimiter, prefix, nested, suffix) => 
            // Gonna test
            _.isEqual(stack.resolve(lib_1.Fn.join(delimiter, [...prefix, lib_1.Fn.join(delimiter, nested), ...suffix])), stack.resolve(lib_1.Fn.join(delimiter, [...prefix, ...nested, ...suffix])))), { verbose: true });
        }),
        'does not flatten joins nested under joins with different delimiter': asyncTest(async () => {
            const stack = new lib_1.Stack();
            fc.assert(fc.property(fc.string(), fc.string(), fc.array(anyValue, 1, 3), fc.array(tokenish, 2, 3), fc.array(anyValue, 3), (delimiter1, delimiter2, prefix, nested, suffix) => {
                fc.pre(delimiter1 !== delimiter2);
                const join = lib_1.Fn.join(delimiter1, [...prefix, lib_1.Fn.join(delimiter2, stringListToken(nested)), ...suffix]);
                const resolved = stack.resolve(join);
                return resolved['Fn::Join'][1].find((e) => typeof e === 'object'
                    && ('Fn::Join' in e)
                    && e['Fn::Join'][0] === delimiter2) != null;
            }), { verbose: true });
        }),
        'Fn::EachMemberIn': asyncTest(async (test) => {
            const stack = new lib_1.Stack();
            const eachMemberIn = lib_1.Fn.conditionEachMemberIn(lib_1.Fn.valueOfAll('AWS::EC2::Subnet::Id', 'VpcId'), lib_1.Fn.refAll('AWS::EC2::VPC::Id'));
            test.deepEqual(stack.resolve(eachMemberIn), {
                'Fn::EachMemberIn': [
                    { 'Fn::ValueOfAll': ['AWS::EC2::Subnet::Id', 'VpcId'] },
                    { 'Fn::RefAll': 'AWS::EC2::VPC::Id' }
                ]
            });
        }),
        'cross-stack FnJoin elements are properly resolved': asyncTest(async (test) => {
            // GIVEN
            const app = new lib_1.App();
            const stack1 = new lib_1.Stack(app, 'Stack1');
            const stack2 = new lib_1.Stack(app, 'Stack2');
            // WHEN
            new lib_1.CfnOutput(stack2, 'Stack1Id', {
                value: lib_1.Fn.join(' = ', ['Stack1Id', stack1.stackId])
            });
            // THEN
            const template = app.synth().getStackByName('Stack2').template;
            test.deepEqual(template, {
                Outputs: {
                    Stack1Id: {
                        Value: {
                            'Fn::Join': [' = ', [
                                    'Stack1Id',
                                    { 'Fn::ImportValue': 'Stack1:ExportsOutputRefAWSStackIdB2DD5BAA' }
                                ]]
                        }
                    }
                }
            });
        }),
    },
    'Ref': {
        'returns a reference given a logical name'(test) {
            const stack = new lib_1.Stack();
            test.deepEqual(stack.resolve(lib_1.Fn.ref('hello')), {
                Ref: 'hello'
            });
            test.done();
        }
    },
    'nested Fn::Join with list token'(test) {
        const stack = new lib_1.Stack();
        const inner = lib_1.Fn.join(',', lib_1.Token.asList({ NotReallyList: true }));
        const outer = lib_1.Fn.join(',', [inner, 'Foo']);
        test.deepEqual(stack.resolve(outer), {
            'Fn::Join': [
                ',',
                [
                    { 'Fn::Join': [',', { NotReallyList: true }] },
                    'Foo'
                ]
            ]
        });
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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