"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Application = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const iam = require("aws-cdk-lib/aws-iam");
const aws_kinesisanalytics_1 = require("aws-cdk-lib/aws-kinesisanalytics");
const logs = require("aws-cdk-lib/aws-logs");
const core = require("aws-cdk-lib");
const environment_properties_1 = require("./private/environment-properties");
const flink_application_configuration_1 = require("./private/flink-application-configuration");
const validation_1 = require("./private/validation");
/**
 * Implements the functionality shared between CDK created and imported
 * IApplications.
 */
class ApplicationBase extends core.Resource {
    /** Implement the convenience {@link IApplication.addToPrincipalPolicy} method. */
    addToRolePolicy(policyStatement) {
        if (this.role) {
            this.role.addToPrincipalPolicy(policyStatement);
            return true;
        }
        return false;
    }
    /**
     * Return a CloudWatch metric associated with this Flink application.
     *
     * @param metricName The name of the metric
     * @param props Customization properties
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/KinesisAnalytics',
            metricName,
            dimensionsMap: { Application: this.applicationName },
            ...props,
        }).attachTo(this);
    }
    /**
     * The number of Kinesis Processing Units that are used to run your stream
     * processing application. The average number of KPUs used each hour
     * determines the billing for your application.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default average over 5 minutes
     */
    metricKpus(props) {
        return this.metric('KPUs', { statistic: 'Average', ...props });
    }
    /**
     * The time elapsed during an outage for failing/recovering jobs.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application
     *
     * @default average over 5 minutes
     */
    metricDowntime(props) {
        return this.metric('downtime', { statistic: 'Average', ...props });
    }
    /**
     * The time that the job has been running without interruption.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application
     *
     * @default average over 5 minutes
     */
    metricUptime(props) {
        return this.metric('uptime', { statistic: 'Average', ...props });
    }
    /**
     * The total number of times this job has fully restarted since it was
     * submitted. This metric does not measure fine-grained restarts.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default sum over 5 minutes
     */
    metricFullRestarts(props) {
        return this.metric('fullRestarts', { statistic: 'Sum', ...props });
    }
    /**
     * The number of times checkpointing has failed.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default sum over 5 minutes
     */
    metricNumberOfFailedCheckpoints(props) {
        return this.metric('numberOfFailedCheckpoints', { statistic: 'Sum', ...props });
    }
    /**
     * The time it took to complete the last checkpoint.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application
     *
     * @default maximum over 5 minutes
     */
    metricLastCheckpointDuration(props) {
        return this.metric('lastCheckpointDuration', { statistic: 'Maximum', ...props });
    }
    /**
     * The total size of the last checkpoint.
     *
     * Units: Bytes
     *
     * Reporting Level: Application
     *
     * @default maximum over 5 minutes
     */
    metricLastCheckpointSize(props) {
        return this.metric('lastCheckpointSize', { statistic: 'Maximum', ...props });
    }
    /**
     * The overall percentage of CPU utilization across task managers. For
     * example, if there are five task managers, Kinesis Data Analytics publishes
     * five samples of this metric per reporting interval.
     *
     * Units: Percentage
     *
     * Reporting Level: Application
     *
     * @default average over 5 minutes
     */
    metricCpuUtilization(props) {
        return this.metric('cpuUtilization', { statistic: 'Average', ...props });
    }
    /**
     * Overall heap memory utilization across task managers. For example, if there
     * are five task managers, Kinesis Data Analytics publishes five samples of
     * this metric per reporting interval.
     *
     * Units: Percentage
     *
     * Reporting Level: Application
     *
     * @default average over 5 minutes
     */
    metricHeapMemoryUtilization(props) {
        return this.metric('heapMemoryUtilization', { statistic: 'Average', ...props });
    }
    /**
     * The total time spent performing old garbage collection operations.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application
     *
     * @default sum over 5 minutes
     */
    metricOldGenerationGCTime(props) {
        return this.metric('oldGenerationGCTime', { statistic: 'Sum', ...props });
    }
    /**
     * The total number of old garbage collection operations that have occurred
     * across all task managers.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default sum over 5 minutes
     */
    metricOldGenerationGCCount(props) {
        return this.metric('oldGenerationGCCount', { statistic: 'Sum', ...props });
    }
    /**
     * The total number of live threads used by the application.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default average over 5 minutes
     */
    metricThreadsCount(props) {
        return this.metric('threadsCount', { statistic: 'Average', ...props });
    }
    /**
     * The total number of records this application, operator, or task has
     * received.
     *
     * Units: Count
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default average over 5 minutes
     */
    metricNumRecordsIn(props) {
        return this.metric('numRecordsIn', { statistic: 'Average', ...props });
    }
    /**
     * The total number of records this application, operator or task has received
     * per second.
     *
     * Units: Count/Second
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default average over 5 minutes
     */
    metricNumRecordsInPerSecond(props) {
        return this.metric('numRecordsInPerSecond', { statistic: 'Average', ...props });
    }
    /**
     * The total number of records this application, operator or task has emitted.
     *
     * Units: Count
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default average over 5 minutes
     */
    metricNumRecordsOut(props) {
        return this.metric('numRecordsOut', { statistic: 'Average', ...props });
    }
    /**
     * The total number of records this application, operator or task has emitted
     * per second.
     *
     * Units: Count/Second
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default average over 5 minutes
     */
    metricNumRecordsOutPerSecond(props) {
        return this.metric('numRecordsOutPerSecond', { statistic: 'Average', ...props });
    }
    /**
     * The number of records this operator or task has dropped due to arriving
     * late.
     *
     * Units: Count
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default sum over 5 minutes
     */
    metricNumLateRecordsDropped(props) {
        return this.metric('numLateRecordsDropped', { statistic: 'Sum', ...props });
    }
    /**
     * The last watermark this application/operator/task/thread has received.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default maximum over 5 minutes
     */
    metricCurrentInputWatermark(props) {
        return this.metric('currentInputWatermark', { statistic: 'Maximum', ...props });
    }
    /**
     * The last watermark this application/operator/task/thread has received.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default maximum over 5 minutes
     */
    metricCurrentOutputWatermark(props) {
        return this.metric('currentOutputWatermark', { statistic: 'Maximum', ...props });
    }
    /**
     * The amount of managed memory currently used.
     *
     * Units: Bytes
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default average over 5 minutes
     */
    metricManagedMemoryUsed(props) {
        return this.metric('managedMemoryUsed', { statistic: 'Average', ...props });
    }
    /**
     * The total amount of managed memory.
     *
     * Units: Bytes
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default average over 5 minutes
     */
    metricManagedMemoryTotal(props) {
        return this.metric('managedMemoryTotal', { statistic: 'Average', ...props });
    }
    /**
     * Derived from managedMemoryUsed/managedMemoryTotal.
     *
     * Units: Percentage
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default average over 5 minutes
     */
    metricManagedMemoryUtilization(props) {
        return this.metric('managedMemoryUtilization', { statistic: 'Average', ...props });
    }
    /**
     * The time (in milliseconds) this task or operator is idle (has no data to
     * process) per second. Idle time excludes back pressured time, so if the task
     * is back pressured it is not idle.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Operator, Task, Parallelism
     *
     * @default average over 5 minutes
     */
    metricIdleTimeMsPerSecond(props) {
        return this.metric('idleTimeMsPerSecond', { statistic: 'Average', ...props });
    }
    /**
     * The time (in milliseconds) this task or operator is back pressured per
     * second.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Operator, Task, Parallelism
     *
     * @default average over 5 minutes
     */
    metricBackPressuredTimeMsPerSecond(props) {
        return this.metric('backPressuredTimeMsPerSecond', { statistic: 'Average', ...props });
    }
    /**
     * The time (in milliseconds) this task or operator is busy (neither idle nor
     * back pressured) per second. Can be NaN, if the value could not be
     * calculated.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Operator, Task, Parallelism
     *
     * @default average over 5 minutes
     */
    metricBusyTimePerMsPerSecond(props) {
        return this.metric('busyTimePerMsPerSecond', { statistic: 'Average', ...props });
    }
}
/**
 * An imported Flink application.
 */
class Import extends ApplicationBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        // Imported applications have no associated role or grantPrincipal
        this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
        this.role = undefined;
        this.applicationArn = attrs.applicationArn;
        this.applicationName = attrs.applicationName;
    }
}
/**
 * The L2 construct for Flink Kinesis Data Applications.
 *
 * @resource AWS::KinesisAnalyticsV2::Application
 *
 */
class Application extends ApplicationBase {
    constructor(scope, id, props) {
        super(scope, id, { physicalName: props.applicationName });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_kinesisanalytics_flink_alpha_ApplicationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Application);
            }
            throw error;
        }
        validation_1.validateFlinkApplicationProps(props);
        this.role = props.role ?? new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('kinesisanalytics.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        // Permit metric publishing to CloudWatch
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['cloudwatch:PutMetricData'],
            resources: ['*'],
        }));
        const code = props.code.bind(this);
        code.bucket.grantRead(this);
        const resource = new aws_kinesisanalytics_1.CfnApplicationV2(this, 'Resource', {
            applicationName: props.applicationName,
            runtimeEnvironment: props.runtime.value,
            serviceExecutionRole: this.role.roleArn,
            applicationConfiguration: {
                ...code.applicationCodeConfigurationProperty,
                environmentProperties: environment_properties_1.environmentProperties(props.propertyGroups),
                flinkApplicationConfiguration: flink_application_configuration_1.flinkApplicationConfiguration({
                    checkpointingEnabled: props.checkpointingEnabled,
                    checkpointInterval: props.checkpointInterval,
                    minPauseBetweenCheckpoints: props.minPauseBetweenCheckpoints,
                    logLevel: props.logLevel,
                    metricsLevel: props.metricsLevel,
                    autoScalingEnabled: props.autoScalingEnabled,
                    parallelism: props.parallelism,
                    parallelismPerKpu: props.parallelismPerKpu,
                }),
                applicationSnapshotConfiguration: {
                    snapshotsEnabled: props.snapshotsEnabled ?? true,
                },
            },
        });
        resource.node.addDependency(this.role);
        const logGroup = props.logGroup ?? new logs.LogGroup(this, 'LogGroup');
        const logStream = new logs.LogStream(this, 'LogStream', { logGroup });
        /* Permit logging */
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:DescribeLogGroups'],
            resources: [
                core.Stack.of(this).formatArn({
                    service: 'logs',
                    resource: 'log-group',
                    arnFormat: core.ArnFormat.COLON_RESOURCE_NAME,
                    resourceName: '*',
                }),
            ],
        }));
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:DescribeLogStreams'],
            resources: [logGroup.logGroupArn],
        }));
        const logStreamArn = `arn:${core.Aws.PARTITION}:logs:${core.Aws.REGION}:${core.Aws.ACCOUNT_ID}:log-group:${logGroup.logGroupName}:log-stream:${logStream.logStreamName}`;
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:PutLogEvents'],
            resources: [logStreamArn],
        }));
        new aws_kinesisanalytics_1.CfnApplicationCloudWatchLoggingOptionV2(this, 'LoggingOption', {
            applicationName: resource.ref,
            cloudWatchLoggingOption: {
                logStreamArn,
            },
        });
        this.applicationName = this.getResourceNameAttribute(resource.ref);
        this.applicationArn = this.getResourceArnAttribute(core.Stack.of(this).formatArn(applicationArnComponents(resource.ref)), applicationArnComponents(this.physicalName));
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: core.RemovalPolicy.DESTROY,
        });
    }
    /**
     * Import an existing Flink application defined outside of CDK code by
     * applicationName.
     */
    static fromApplicationName(scope, id, applicationName) {
        const applicationArn = core.Stack.of(scope).formatArn(applicationArnComponents(applicationName));
        return new Import(scope, id, { applicationArn, applicationName });
    }
    /**
     * Import an existing application defined outside of CDK code by
     * applicationArn.
     */
    static fromApplicationArn(scope, id, applicationArn) {
        const applicationName = core.Stack.of(scope).splitArn(applicationArn, core.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        if (!applicationName) {
            throw new Error(`applicationArn for fromApplicationArn (${applicationArn}) must include resource name`);
        }
        return new Import(scope, id, { applicationArn, applicationName });
    }
}
exports.Application = Application;
_a = JSII_RTTI_SYMBOL_1;
Application[_a] = { fqn: "@aws-cdk/aws-kinesisanalytics-flink-alpha.Application", version: "2.58.0-alpha.0" };
function applicationArnComponents(resourceName) {
    return {
        service: 'kinesisanalytics',
        resource: 'application',
        resourceName,
    };
}
//# sourceMappingURL=data:application/json;base64,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