import numpy as np
from numpy.random import random
import itertools as itertools

from numpy.testing import (assert_equal, assert_almost_equal, assert_,
                           assert_array_almost_equal, assert_allclose,
                           assert_array_equal)
import pytest
from pytest import raises as assert_raises
import PSpincalc as sp

Q1 = np.array([-0.1677489, -0.7369231, -0.3682588, 0.5414703]);
Q2 = (-0.8735598,  0.1145235, -0.2093062, 0.4242270);
Q3 = [0.426681700, -0.20287610,  0.43515810, -0.76643420];
Q4 = np.array([[-0.1677489, -0.7369231, -0.3682588, 0.5414703],
[-0.8735598,  0.1145235, -0.2093062, 0.4242270],
[0.426681700, -0.20287610,  0.43515810, -0.76643420]]);

EV1 = np.array([[-0.1995301, -0.8765382, -0.4380279, 114.4324]]);
EV2 = np.array([-9.646669e-001, 1.264676e-001, -2.311356e-001, 1.297965e+002]);
EV3 = np.array([6.642793e-001, -3.158476e-001, 6.774757e-001, 2.800695e+002]);
EV4 = np.array([[-1.995301e-001, -8.765382e-001, -4.380280e-001, 1.144324e+002],
[-9.646669e-001, 1.264676e-001, -2.311356e-001, 1.297965e+002],
[6.642793e-001, -3.158476e-001, 6.774757e-001, 2.800695e+002]]);

DCM1 = np.array([[-0.3573404, -0.1515663, 0.9215940], [0.6460385, 0.6724915, 0.3610947], [-0.6744939, 0.7244189, -0.1423907]]);
DCM2 = np.array([[0.88615060, -0.3776729,  0.2685150],[-0.02249957, -0.6138316, -0.7891163],[0.46285090,  0.6932344, -0.5524447]]);
DCM3 = np.array([[0.5389574, -0.8401672,  0.06036564],[0.4939131, 0.2571603, -0.83061330],[0.6823304, 0.4774806, 0.55356800]]);
DCM4 = np.array([[[-0.3573404, -0.1515663, 0.9215940], [0.6460385, 0.6724915, 0.3610947], [-0.6744939, 0.7244189, -0.1423907]],
[[0.88615060, -0.3776729,  0.2685150],[-0.02249957, -0.6138316, -0.7891163],[0.46285090,  0.6932344, -0.5524447]],
[[0.5389574, -0.8401672,  0.06036564],[0.4939131, 0.2571603, -0.83061330],[0.6823304, 0.4774806, 0.55356800]]]);

EAxyx = [[0.3734309, 1.427920, 2.3205212],[-1.2428130, 0.985502, 0.9821003],[-1.4982479, 2.157439, 0.6105777]];
EAxyz = [[2.07603606, -0.7402790, -1.376712],[-0.02538478,  0.4812086, -0.897943], [0.74181498,  0.7509456, -2.429857]];
EAxzy = [[-2.9199163, -0.8101911, -1.3627437],[-0.5515727, -0.7659671,  0.6973821],[-1.8678238, -0.4977850,  2.2523838]];
EAyzx = [[1.4175036,  0.3694416,  2.3762543],[0.4524244, -0.9093689, -0.0366379],[3.0329736, -0.9802081,  2.0508342]];
EAyxz = [[-2.0579912,  0.70238301,  2.6488233], [0.4813408, -0.02250147, -0.9096943], [0.9022582,  0.51658433, -1.8710396]];
EAzxy = [[-2.3190385, -0.1521527, 1.9406908],[-0.8460724, -0.3872818, 0.2942186],[-2.0648275, -0.9975913, 0.1115396]];
EAzyx = [[1.1951869, 1.17216717, -2.7404413],[-0.9600165, 0.27185113, -0.4028824],[-2.1586537, 0.06040236, -1.0004280]];
EAxzx = [[-1.197365, 1.427920, -2.391868],[-2.813609, 0.985502,  2.552897],[-3.069044, 2.157439,  2.181374]];
EAyxy = [[1.777046, 2.3083664,  0.5096786],[2.069637, 0.9098912, -1.5993010],[2.624796, 1.8308788, -1.0343298]];
EAyzy = [[-2.935343, 2.3083664, -1.061118],[-2.642752, 0.9098912,  3.113088],[-2.087593, 1.8308788, -2.605126]];
EAzxz = [[-0.8069433, 1.9362151, -1.733798], [1.6193689, 0.4818253, -2.523541], [0.9442344, 1.0015975, -3.069866]];
EAzyz = [[-2.3777396, 1.9362151, -0.1630019], [0.0485726, 0.4818253, -0.9527442],[-0.6265619, 1.0015975, -1.4990700]];
EAall = np.vstack([EAxyx,EAxyz,EAxzy,EAyzx,EAyxz,EAzxy,EAzyx,EAxzx,EAyxy,EAyzy,EAzxz,EAzyz]);
EAvct = ["xyx","xyz","xzy","yzx","yxz","zxy","zyx","xzx","yxy","yzy","zxz","zyz"];

EAResult1 = np.array([[0.16774889, 0.73692311, 0.36825881, -0.5414703],
    [0.87355981, -0.1145235, 0.20930621, -0.42422699],
    [0.42668164, -0.20287607, 0.43515814, -0.76643426],
    [0.16774893, 0.73692312, 0.36825879, -0.54147029],
    [0.8735598, -0.1145235, 0.20930618, -0.42422702],
    [0.42668176, -0.20287604, 0.43515812, -0.76643421],
    [-0.16774891, -0.7369231, -0.36825879, 0.54147033],
    [0.87355981, -0.11452349, 0.20930623, -0.42422698],
    [0.42668168, -0.20287611, 0.43515813, -0.76643423],
    [0.1677489, 0.73692311, 0.36825881, -0.5414703],
    [0.87355981, -0.1145235, 0.2093062, -0.42422699],
    [0.42668169, -0.2028761, 0.43515811, -0.76643424],
    [-0.16774887, -0.73692309, -0.36825881, 0.54147033],
    [0.87355981, -0.11452349, 0.20930618, -0.424227],
    [0.42668174, -0.2028761, 0.4351581, -0.76643421],
    [0.16774891, 0.73692311, 0.36825882, -0.54147028],
    [0.87355981, -0.1145235, 0.20930619, -0.42422699],
    [0.42668175, -0.20287608, 0.43515806, -0.76643424],
    [-0.16774891, -0.73692311, -0.36825884, 0.54147028],
    [0.8735598, -0.11452351, 0.2093062, -0.424227],
    [0.42668173, -0.2028761, 0.4351581, -0.76643422],
    [-0.16774881, -0.73692313, -0.36825899, 0.54147018],
    [0.87355985, -0.11452319, 0.2093062, -0.424227],
    [0.42668166, -0.20287603, 0.43515823, -0.7664342],
    [0.16774889, 0.73692305, 0.36825882, -0.54147038],
    [0.87355983, -0.11452347, 0.20930611, -0.42422701],
    [0.42668171, -0.2028761, 0.43515812, -0.76643422],
    [-0.16774894, -0.73692298, -0.3682588, 0.54147047],
    [0.87355983, -0.11452346, 0.20930611, -0.42422701],
    [-0.42668169, 0.20287605, -0.43515815, 0.76643423],
    [0.16774896, 0.73692316, 0.36825871, -0.54147028],
    [0.8735597, -0.11452355, 0.20930618, -0.42422721],
    [0.42668184, -0.20287603, 0.43515817, -0.76643414],
    [0.16774891, 0.73692312, 0.3682588, -0.54147029],
    [0.87355981, -0.1145235, 0.20930621, -0.424227],
    [0.42668172, -0.20287611, 0.43515813, -0.76643421]])


EAResult2 = np.array([[[0.16774889, 0.73692311, 0.36825881, -0.5414703],
    [0.87355981, -0.1145235, 0.20930621, -0.42422699],
    [0.42668164, -0.20287607, 0.43515814, -0.76643426],],
    [[0.16774893, 0.73692312, 0.36825879, -0.54147029],
    [0.8735598, -0.1145235, 0.20930618, -0.42422702],
    [0.42668176, -0.20287604, 0.43515812, -0.76643421],],
    [[-0.16774891, -0.7369231, -0.36825879, 0.54147033],
    [0.87355981, -0.11452349, 0.20930623, -0.42422698],
    [0.42668168, -0.20287611, 0.43515813, -0.76643423],],
    [[0.1677489, 0.73692311, 0.36825881, -0.5414703],
    [0.87355981, -0.1145235, 0.2093062, -0.42422699],
    [0.42668169, -0.2028761, 0.43515811, -0.76643424],],
    [[-0.16774887, -0.73692309, -0.36825881, 0.54147033],
    [0.87355981, -0.11452349, 0.20930618, -0.424227],
    [0.42668174, -0.2028761, 0.4351581, -0.76643421],],
    [[0.16774891, 0.73692311, 0.36825882, -0.54147028],
    [0.87355981, -0.1145235, 0.20930619, -0.42422699],
    [0.42668175, -0.20287608, 0.43515806, -0.76643424],],
    [[-0.16774891, -0.73692311, -0.36825884, 0.54147028],
    [0.8735598, -0.11452351, 0.2093062, -0.424227],
    [0.42668173, -0.2028761, 0.4351581, -0.76643422],],
    [[-0.16774881, -0.73692313, -0.36825899, 0.54147018],
    [0.87355985, -0.11452319, 0.2093062, -0.424227],
    [0.42668166, -0.20287603, 0.43515823, -0.7664342],],
    [[0.16774889, 0.73692305, 0.36825882, -0.54147038],
    [0.87355983, -0.11452347, 0.20930611, -0.42422701],
    [0.42668171, -0.2028761, 0.43515812, -0.76643422],],
    [[-0.16774894, -0.73692298, -0.3682588, 0.54147047],
    [0.87355983, -0.11452346, 0.20930611, -0.42422701],
    [-0.42668169, 0.20287605, -0.43515815, 0.76643423],],
    [[0.16774896, 0.73692316, 0.36825871, -0.54147028],
    [0.8735597, -0.11452355, 0.20930618, -0.42422721],
    [0.42668184, -0.20287603, 0.43515817, -0.76643414],],
    [[0.16774891, 0.73692312, 0.3682588, -0.54147029],
    [0.87355981, -0.1145235, 0.20930621, -0.424227],
    [0.42668172, -0.20287611, 0.43515813, -0.76643421]]])

EAResult3 = np.array([[-2.97859077, 1.93621503, 2.37773962],
    [-2.18884844, 0.48182525, -0.04857256],
    [-1.64252264, 1.00159744, 0.62656191],
    [-1.76488062, -0.74027901, -2.07603602],
    [-2.24364965, 0.48120861, 0.02538476],
    [-0.71173553, 0.75094569, -0.74181497],
    [0.49276926, -0.70238302, -2.05799123],
    [-2.23189831, 0.02250145, 0.48134083],
    [-1.27055302, -0.51658437, 0.90225821],
    [-1.94069072, -0.15215268, -0.82255409],
    [-0.2942186, -0.38728179, -2.29552018],
    [-0.11153961, -0.99759145, -1.07676513],
    [-1.77884896, -0.81019108, -0.22167632],
    [2.4442106, -0.7659672, -2.59001993],
    [0.88920891, -0.49778507, -1.27376888],
    [-0.76533834, 0.36944152, -1.72408905],
    [3.10495479, -0.90936899, -2.68916825],
    [-1.0907585, -0.98020819, -0.1086191],
    [-2.74044132, -1.1721672, 1.94640586],
    [-0.40288236, -0.27185112, -2.18157612],
    [-1.00042803, -0.06040237, -0.98293893],
    [1.73379821, 1.93621503, -2.33464936],
    [2.52354055, 0.48182525, 1.52222377],
    [3.06986634, 1.00159744, 2.19735824],
    [1.06111775, 0.83322623, -2.93534318],
    [-3.11308804, 2.23170141, -2.64275184],
    [2.60512614, 1.31071384, -2.087593],
    [2.63191408, 0.83322623, 1.7770458],
    [-1.54229171, 2.23170141, 2.06963714],
    [-2.10726284, 1.31071384, 2.62479598],
    [-2.39186777, 1.71367266, 1.19736551],
    [2.5528967, 2.15609061, 2.81360939],
    [2.18137404, 0.98415383, 3.06904421],
    [2.32052121, 1.71367266, 2.76816184],
    [0.98210037, 2.15609061, -1.89877959],
    [0.61057772, 0.98415383, -1.64334477]])

EAResult4 = np.array([[-2.83416594, 1.30355819, 2.52216447],
    [3.1314547, 0.46779845, -1.0114546],
    [-2.36098822, 1.62827676, -0.09190377],
    [-0.90869174, -0.90319547, -1.13013935],
    [2.07021285, 0.24161615, 0.40458574],
    [1.79399875, 1.46244063, 2.13084225],
    [0.12131271, -0.59433956, -1.24641348],
    [2.1723151, 0.39217997, 0.26191495],
    [-2.3556936, 0.09175156, 1.62851982],
    [-1.29107298, -0.29617681, -0.53578399],
    [-0.46777768, 0.00457136, 2.12108923],
    [-1.65161185, -0.77896999, 2.99284956],
    [-1.11949233, -0.51012591, -0.34101766],
    [2.66636391, 1.02048633, 3.1328511],
    [1.59487098, 0.10565777, -2.35707122],
    [-0.59776651, 0.10043222, -1.17825355],
    [2.5104365, 0.86621248, -2.37159131],
    [3.0121339, -0.78169746, -1.60453168],
    [-0.83535709, -1.16625622, 0.25757576],
    [0.00512155, -0.46777238, 2.11877991],
    [-1.65238697, -0.78852586, -1.60476278],
    [1.87822304, 1.30355819, -2.19022451],
    [1.56065838, 0.46779845, 0.55934172],
    [2.35140076, 1.62827676, 1.47889256],
    [1.3936197, 0.60512785, -2.60284129],
    [-0.46497313, 2.12108279, 0.0053631],
    [-3.01226208, 2.35150037, -1.42179603],
    [2.96441603, 0.60512785, 2.10954769],
    [1.1058232, 2.12108279, -1.56543323],
    [-1.44146575, 2.35150037, -2.99259236],
    [-2.12707453, 1.18033899, 1.46215877],
    [0.27367292, 2.05443339, 0.53438559],
    [1.46510816, 1.59473668, 2.35277834],
    [2.58531445, 1.18033899, 3.03295509],
    [-1.29712341, 2.05443339, 2.10518192],
    [-0.10568816, 1.59473668, -2.35961065]])

def random_quaternion():
# based on: https://stackoverflow.com/questions/31600717/how-to-generate-a-random-quaternion-quickly
    while True:
        x = random.uniform(-1,1)
        y = random.uniform(-1,1)
        z = x**2 + y**2
        if (z <= 1):
            break
    while True:
        u = random.uniform(-1,1)
        v = random.uniform(-1,1)
        w = u**2 + v**2
        if (w <= 1):
            break
    s = np.sqrt((1-z) / w)
    return [x, y, s*u, s*v]

def test_Qnorm():
    r1 = sp.Qnorm(Q1)
    r4 = sp.Qnorm(Q4)
    e1 = np.array([1.])
    e4 = np.array([0.99999999, 0.99999999, 0.99999997])
    assert_array_almost_equal(r1, e1)
    assert_array_almost_equal(r4, e4)
    
def test_Qnormalize():
    r1 = sp.Qnormalize(Q1).reshape(4)
    r4 = sp.Qnormalize(Q4)
    e1 = np.array([-0.1677489, -0.73692311, -0.3682588, 0.54147031])
    e4 = np.array([[-0.1677489,  -0.73692311, -0.3682588,   0.54147031],
        [-0.8735598,   0.1145235,  -0.2093062,   0.424227  ],
        [ 0.42668171, -0.20287611,  0.43515811, -0.76643422]])
    assert_array_almost_equal(r1, e1)
    assert_array_almost_equal(r4, e4)

def test_Q2EV_EV2Q():
    r1 = sp.Q2EV(Q1).reshape(4)
    r2 = sp.Q2EV(Q2).reshape(4)
    r3 = sp.Q2EV(Q3).reshape(4)
    r4 = sp.Q2EV(Q4)
    e1 = np.array([0.19953008,0.87653823, 0.43802795, 4.28596264])
    e2 = np.array([ 0.96466692, -0.12646762,  0.2311356,   4.01780885])
    e3 = np.array([-0.66427924,  0.31584758, -0.67747572,  1.39505018])
    e4 = np.array([[ 0.19953008,  0.87653822,  0.43802795,  4.28596264],
        [ 0.96466692, -0.12646762,  0.2311356,   4.01780885],
        [-0.66427924,  0.31584758, -0.67747572,  1.39505018]])
    assert_array_almost_equal(r1, e1)
    assert_array_almost_equal(r2, e2)
    assert_array_almost_equal(r3, e3)
    assert_array_almost_equal(r4, e4)

def test_EV2Q():
    r1 = sp.EV2Q(EV1,1e-7).reshape(4)
    r2 = sp.EV2Q(EV2,1e-7).reshape(4)
    r3 = sp.EV2Q(EV3,1e-7).reshape(4)
    r4 = sp.EV2Q(EV4,1e-7)
    e1 = np.array([ 0.78535172, -0.12351907, -0.54262081, -0.27116109])
    e2 = np.array([-0.47556008, -0.84860131,  0.11125143, -0.20332611])
    e3 = np.array([-0.23175062,  0.64619445, -0.30724872,  0.65903158])
    e4 = np.array([[ 0.78535172, -0.12351907, -0.54262081, -0.27116115],
        [-0.47556008, -0.84860131,  0.11125143, -0.20332611],
        [-0.23175062,  0.64619445, -0.30724872,  0.65903158]])
    assert_array_almost_equal(r1, e1)
    assert_array_almost_equal(r2, e2)
    assert_array_almost_equal(r3, e3)
    assert_array_almost_equal(r4, e4)

def test_DCM2Q():
    r1 = sp.DCM2Q(DCM1).reshape(4)
    r2 = sp.DCM2Q(DCM2).reshape(4)
    r3 = sp.DCM2Q(DCM3).reshape(4)
    r4 = sp.DCM2Q(DCM4)
    e1 = np.array([-0.54147031,  0.16774892,  0.7369231,   0.36825878])
    e2 = np.array([-0.42422702,  0.8735598,  -0.11452349,  0.20930619])
    e3 = np.array([ 0.76643423, -0.42668172,  0.2028761,  -0.43515812])
    e4 = np.array([[-0.54147031,  0.16774892,  0.7369231,   0.36825878],
        [-0.42422702,  0.8735598,  -0.11452349,  0.20930619],
        [ 0.76643423, -0.42668172,  0.2028761,  -0.43515812]])
    assert_array_almost_equal(r1, e1)
    assert_array_almost_equal(r2, e2)
    assert_array_almost_equal(r3, e3)
    assert_array_almost_equal(r4, e4)

def test_Q2DCM():
    r1 = sp.Q2DCM(Q1)
    r2 = sp.Q2DCM(Q2)
    r3 = sp.Q2DCM(Q3)
    r4 = sp.Q2DCM(Q4)
    e1 = np.array([[ 0.14239072,  0.36109475, -0.92159398],
        [ 0.72441894, -0.67249152, -0.15156633],
        [-0.67449394, -0.6460385,  -0.35734043]])
    e2 = np.array([[ 0.55244473, -0.78911627, -0.26851505],
        [ 0.69323436,  0.61383164, -0.37767294],
        [ 0.46285089,  0.02249957,  0.88615056]])
    e3 = np.array([[-0.553568,   -0.8306133,  -0.06036564],
        [ 0.47748057, -0.25716027, -0.84016719],
        [ 0.6823304,  -0.49391309,  0.5389574 ]])
    e4 = np.array([[[ 0.14239072,  0.36109475, -0.92159398],
        [ 0.72441894, -0.67249152, -0.15156633],
        [-0.67449394, -0.6460385,  -0.35734043]],
        [[ 0.55244473, -0.78911627, -0.26851505],
        [ 0.69323436,  0.61383164, -0.37767294],
        [ 0.46285089,  0.02249957,  0.88615056]],
        [[-0.553568,   -0.8306133,  -0.06036564],
        [ 0.47748057, -0.25716027, -0.84016719],
        [ 0.6823304,  -0.49391309,  0.5389574 ]]])
    assert_array_almost_equal(r1, e1)
    assert_array_almost_equal(r2, e2)
    assert_array_almost_equal(r3, e3)
    assert_array_almost_equal(r4, e4)

def test_Q2EA_xyx():
    r1 = sp.Q2EA(Q1,'xyx').reshape(3)
    r2 = sp.Q2EA(Q2,'xyx').reshape(3)
    r3 = sp.Q2EA(Q3,'xyx').reshape(3)
    r4 = sp.Q2EA(Q4,'xyx')
    e1 = np.array([0.37343087, 1.42792,    2.3205212 ])
    e2 = np.array([-1.24281303,  0.98550202,  0.98210032])
    e3 = np.array([-1.4982479,   2.15743878,  0.61057771])
    e4 = np.array([[ 0.37343087,  1.42792,     2.3205212 ],
        [-1.24281303,  0.98550202,  0.98210032],
        [-1.4982479,   2.15743878,  0.61057771]])
    assert_array_almost_equal(r1, e1)
    assert_array_almost_equal(r2, e2)
    assert_array_almost_equal(r3, e3)
    assert_array_almost_equal(r4, e4)

def test_Q2EA_xyz():
    r1 = sp.Q2EA(Q1,'xyz').reshape(3)
    r2 = sp.Q2EA(Q2,'xyz').reshape(3)
    r3 = sp.Q2EA(Q3,'xyz').reshape(3)
    r4 = sp.Q2EA(Q4,'xyz')
    e1 = np.array([ 2.07603606, -0.74027903, -1.37671205])
    e2 = np.array([-0.02538478,  0.48120863, -0.89794296])
    e3 = np.array([ 0.74181498,  0.75094563, -2.42985714])
    e4 = np.array([[ 2.07603606, -0.74027901, -1.37671205],
        [-0.02538478,  0.48120863, -0.89794296],
        [ 0.74181498,  0.75094563, -2.42985714]])
    assert_array_almost_equal(r1, e1)
    assert_array_almost_equal(r2, e2)
    assert_array_almost_equal(r3, e3)
    assert_array_almost_equal(r4, e4)

def test_Q2EA_xzy():
    r1 = sp.Q2EA(Q1,'xzy').reshape(3)
    r2 = sp.Q2EA(Q2,'xzy').reshape(3)
    r3 = sp.Q2EA(Q3,'xzy').reshape(3)
    r4 = sp.Q2EA(Q4,'xzy')
    e1 = np.array([-2.91991628, -0.81019113, -1.36274366])
    e2 = np.array([-0.55157273, -0.76596714,  0.69738206])
    e3 = np.array([-1.86782379, -0.49778502,  2.25238376])
    e4 = np.array([[-2.91991628, -0.8101911,  -1.36274366],
        [-0.55157273, -0.76596714,  0.69738206],
        [-1.86782379, -0.49778502,  2.25238376]])
    assert_array_almost_equal(r1, e1)
    assert_array_almost_equal(r2, e2)
    assert_array_almost_equal(r3, e3)
    assert_array_almost_equal(r4, e4)

def test_Q2EA_yzx():
    r1 = sp.Q2EA(Q1,'yzx').reshape(3)
    r2 = sp.Q2EA(Q2,'yzx').reshape(3)
    r3 = sp.Q2EA(Q3,'yzx').reshape(3)
    r4 = sp.Q2EA(Q4,'yzx')
    e1 = np.array([1.41750361, 0.36944158, 2.37625431])
    e2 = np.array([ 0.45242441, -0.90936893, -0.0366379 ])
    e3 = np.array([ 3.03297357, -0.98020807,  2.05083416])
    e4 = np.array([[ 1.41750361,  0.36944157,  2.37625431],
        [ 0.45242441, -0.90936893, -0.0366379 ],
        [ 3.03297357, -0.98020807,  2.05083416]])
    assert_array_almost_equal(r1, e1)
    assert_array_almost_equal(r2, e2)
    assert_array_almost_equal(r3, e3)
    assert_array_almost_equal(r4, e4)

def test_Q2EA_yxz():
    r1 = sp.Q2EA(Q1,'yxz').reshape(3)
    r2 = sp.Q2EA(Q2,'yxz').reshape(3)
    r3 = sp.Q2EA(Q3,'yxz').reshape(3)
    r4 = sp.Q2EA(Q4,'yxz')
    e1 = np.array([-2.05799127,  0.70238303,  2.64882332])
    e2 = np.array([ 0.48134085, -0.02250147, -0.90969431])
    e3 = np.array([ 0.90225816,  0.51658433, -1.87103965])
    e4 = np.array([[-2.05799125,  0.70238301,  2.64882332],
        [ 0.48134085, -0.02250147, -0.90969431],
        [ 0.90225816,  0.51658433, -1.87103965]])
    assert_array_almost_equal(r1, e1)
    assert_array_almost_equal(r2, e2)
    assert_array_almost_equal(r3, e3)
    assert_array_almost_equal(r4, e4)

def test_Q2EA_zxy():
    r1 = sp.Q2EA(Q1,'zxy').reshape(3)
    r2 = sp.Q2EA(Q2,'zxy').reshape(3)
    r3 = sp.Q2EA(Q3,'zxy').reshape(3)
    r4 = sp.Q2EA(Q4,'zxy')
    e1 = np.array([-2.31903855, -0.15215272,  1.94069075])
    e2 = np.array([-0.84607242, -0.38728181,  0.29421862])
    e3 = np.array([-2.06482755, -0.99759134,  0.1115396 ])
    e4 = np.array([[-2.31903855, -0.15215272,  1.94069075],
        [-0.84607242, -0.38728181,  0.29421862],
        [-2.06482755, -0.99759134,  0.1115396 ]])
    assert_array_almost_equal(r1, e1)
    assert_array_almost_equal(r2, e2)
    assert_array_almost_equal(r3, e3)
    assert_array_almost_equal(r4, e4)

def test_Q2EA_zyx():
    r1 = sp.Q2EA(Q1,'zyx').reshape(3)
    r2 = sp.Q2EA(Q2,'zyx').reshape(3)
    r3 = sp.Q2EA(Q3,'zyx').reshape(3)
    r4 = sp.Q2EA(Q4,'zyx')
    e1 = np.array([ 1.19518686,  1.17216723, -2.74044126])
    e2 = np.array([-0.9600165,   0.27185113, -0.40288239])
    e3 = np.array([-2.15865372,  0.06040236, -1.00042803])
    e4 = np.array([[ 1.19518686,  1.17216717, -2.74044126],
        [-0.9600165,   0.27185113, -0.40288239],
        [-2.15865372,  0.06040236, -1.00042803]])
    assert_array_almost_equal(r1, e1)
    assert_array_almost_equal(r2, e2)
    assert_array_almost_equal(r3, e3)
    assert_array_almost_equal(r4, e4)

def test_Q2EA_xzx():
    r1 = sp.Q2EA(Q1,'xzx').reshape(3)
    r2 = sp.Q2EA(Q2,'xzx').reshape(3)
    r3 = sp.Q2EA(Q3,'xzx').reshape(3)
    r4 = sp.Q2EA(Q4,'xzx')
    e1 = np.array([-1.19736546,  1.42792,    -2.39186778])
    e2 = np.array([-2.81360936,  0.98550202,  2.55289665])
    e3 = np.array([-3.06904422,  2.15743878,  2.18137403])
    e4 = np.array([[-1.19736546,  1.42792,    -2.39186778],
        [-2.81360936,  0.98550202 , 2.55289665],
        [-3.06904422,  2.15743878,  2.18137403]])
    assert_array_almost_equal(r1, e1)
    assert_array_almost_equal(r2, e2)
    assert_array_almost_equal(r3, e3)
    assert_array_almost_equal(r4, e4)

def test_Q2EA_yxy():
    r1 = sp.Q2EA(Q1,'yxy').reshape(3)
    r2 = sp.Q2EA(Q2,'yxy').reshape(3)
    r3 = sp.Q2EA(Q3,'yxy').reshape(3)
    r4 = sp.Q2EA(Q4,'yxy')
    e1 = np.array([1.77704584, 2.30836643, 0.50967864])
    e2 = np.array([ 2.06963718,  0.90989122, -1.59930097])
    e3 = np.array([ 2.62479599,  1.8308788,  -1.03432982])
    e4 = np.array([[ 1.77704584,  2.30836641,  0.50967864],
 [ 2.06963718,  0.90989122, -1.59930097],
 [ 2.62479599,  1.8308788,  -1.03432982]])
    assert_array_almost_equal(r1, e1)
    assert_array_almost_equal(r2, e2)
    assert_array_almost_equal(r3, e3)
    assert_array_almost_equal(r4, e4)

def test_Q2EA_yzy():
    r1 = sp.Q2EA(Q1,'yzy').reshape(3)
    r2 = sp.Q2EA(Q2,'yzy').reshape(3)
    r3 = sp.Q2EA(Q3,'yzy').reshape(3)
    r4 = sp.Q2EA(Q4,'yzy')
    e1 = np.array([-2.93534314,  2.30836643, -1.06111769])
    e2 = np.array([-2.6427518,   0.90989122,  3.11308801])
    e3 = np.array([-2.08759299,  1.8308788,  -2.60512614])
    e4 = np.array([[-2.93534314,  2.30836641, -1.06111769],
        [-2.6427518,   0.90989122,  3.11308801],
        [-2.08759299 , 1.8308788,  -2.60512614]])
    assert_array_almost_equal(r1, e1)
    assert_array_almost_equal(r2, e2)
    assert_array_almost_equal(r3, e3)
    assert_array_almost_equal(r4, e4)

def test_Q2EA_zxz():
    r1 = sp.Q2EA(Q1,'zxz').reshape(3)
    r2 = sp.Q2EA(Q2,'zxz').reshape(3)
    r3 = sp.Q2EA(Q3,'zxz').reshape(3)
    r4 = sp.Q2EA(Q4,'zxz')
    e1 = np.array([-0.80694329,  1.93621508, -1.73379825])
    e2 = np.array([ 1.61936893,  0.48182531, -2.52354054])
    e3 = np.array([ 0.94423442,  1.00159749, -3.06986636])
    e4 = np.array([[-0.80694329,  1.93621507, -1.73379825],
        [ 1.61936893,  0.48182531, -2.52354054],
        [ 0.94423442,  1.00159749, -3.06986636]])
    assert_array_almost_equal(r1, e1)
    assert_array_almost_equal(r2, e2)
    assert_array_almost_equal(r3, e3)
    assert_array_almost_equal(r4, e4)

def test_Q2EA_zyz():
    r1 = sp.Q2EA(Q1,'zyz').reshape(3)
    r2 = sp.Q2EA(Q2,'zyz').reshape(3)
    r3 = sp.Q2EA(Q3,'zyz').reshape(3)
    r4 = sp.Q2EA(Q4,'zyz')
    e1 = np.array([-2.37773962,  1.93621508, -0.16300192])
    e2 = np.array([ 0.0485726,   0.48182531, -0.95274421])
    e3 = np.array([-0.62656191,  1.00159749, -1.49907003])
    e4 = np.array([[-2.37773962,  1.93621507, -0.16300192],
        [ 0.0485726,   0.48182531, -0.95274421],
        [-0.62656191,  1.00159749, -1.49907003]])
    assert_array_almost_equal(r1, e1)
    assert_array_almost_equal(r2, e2)
    assert_array_almost_equal(r3, e3)
    assert_array_almost_equal(r4, e4)

def test_EA2Q():
    n = 0
    for EAv in list(itertools.chain(*zip(EAvct,EAvct,EAvct))):
        r1 = sp.EA2Q(EAall[n,:],EAv).reshape(4)
        e1 = EAResult1[n,:]
        assert_array_almost_equal(r1, e1)
        n = n+1
    n = 0
    for EAv in EAvct:
        r1 = sp.EA2Q(EAall[range(n,1+(n+2)),:],EAv)
        e1 = EAResult2[n//3,:]
        assert_array_almost_equal(r1, e1)
        n = n+3

def test_DCM2EV():
    r1 = sp.DCM2EV(DCM1).reshape(4)
    r2 = sp.DCM2EV(DCM2).reshape(4)
    r3 = sp.DCM2EV(DCM3).reshape(4)
    r4 = sp.DCM2EV(DCM4)
    e1 = np.array([ 0.5823994,  -0.18042885, -0.79262623,  3.89586364])
    e2 = np.array([ 0.43383644, -0.89334731,  0.11711763,  3.56332342])
    e3 = np.array([-0.85125871,  0.47390437, -0.22532925,  2.24116489])
    e4 = np.array([[ 0.5823994,  -0.18042885, -0.79262623,  3.89586364],
        [ 0.43383644, -0.89334731,  0.11711763,  3.56332342],
        [-0.85125871,  0.47390437, -0.22532925,  2.24116489]])
    assert_array_almost_equal(r1, e1)
    assert_array_almost_equal(r2, e2)
    assert_array_almost_equal(r3, e3)
    assert_array_almost_equal(r4, e4)

def test_EV2DCM():
    r1 = sp.EV2DCM(EV1,1e-7)
    r2 = sp.EV2DCM(EV2,1e-7)
    r3 = sp.EV2DCM(EV3,1e-7)
    r4 = sp.EV2DCM(EV4,1e-7)
    e1 = np.array([[ 0.26406862, -0.29186563,  0.91928353],
        [ 0.55996171,  0.8224294,   0.10026347],
        [-0.78530927,  0.48828714,  0.38061138]])
    e2 = np.array([[ 0.89256322,  0.00457135,  0.45089911],
        [-0.3822038,  -0.52293145,  0.7618812 ],
        [ 0.23927215, -0.85236248, -0.46500219]])
    e3 = np.array([[-0.05744878, -0.70254678,  0.70931492],
        [-0.09162288, -0.70377975, -0.70448514],
        [ 0.99413525, -0.10546129, -0.02393807]])
    e4 = np.array([[[ 0.26406858, -0.29186572,  0.91928352],
        [ 0.55996179,  0.82242934,  0.10026353],
        [-0.78530922,  0.48828719,  0.3806114 ]],
        [[ 0.89256322,  0.00457135,  0.45089911],
        [-0.3822038,  -0.52293145,  0.7618812 ],
        [ 0.23927215, -0.85236248, -0.46500219]],
        [[-0.05744878, -0.70254678,  0.70931492],
        [-0.09162288, -0.70377975, -0.70448514],
        [ 0.99413525, -0.10546129, -0.02393807]]])
    assert_array_almost_equal(r1, e1)
    assert_array_almost_equal(r2, e2)
    assert_array_almost_equal(r3, e3)
    assert_array_almost_equal(r4, e4)

def test_DCM2EA():
    n = 0
    for EAv in EAvct:
        for DCMtmp in [DCM1, DCM2, DCM3]:
            r1 = sp.DCM2EA(DCMtmp,EAv).reshape(3)
            e1 = EAResult3[n,:]
            assert_array_almost_equal(r1, e1)
            n = n + 1
    n = 0
    for EAv in EAvct:
        r1 = sp.DCM2EA(DCM4,EAv)
        e1 = EAResult3[range(n,1+(n+2)),:]
        assert_array_almost_equal(r1, e1)
        n = n + 3
    r1 = sp.EA2DCM(EAxyx,'xyx',1e-7)
    e1 = np.array([[[0.14239072, 0.36109478, -0.92159397],
        [0.72441895, -0.67249151, -0.15156635],
        [-0.67449394, -0.64603849, -0.35734045]],
        [[0.55244474, -0.78911626, -0.26851507],
        [0.69323434, 0.61383166, -0.37767293],
        [0.4628509, 0.02249956, 0.88615056]],
        [[-0.55356815, -0.8306132, -0.06036563],
        [0.4774805, -0.25716035, -0.8401672],
        [0.68233032, -0.49391321, 0.53895739]]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2DCM(EAxyz,'xyz',1e-7)
    e1 = np.array([[[0.14239076, 0.36109472, -0.92159399],
        [0.72441896, -0.67249152, -0.15156627],
        [-0.67449392, -0.64603852, -0.35734044]],
        [[0.55244471, -0.7891163, -0.26851501],
        [0.69323439, 0.6138316, -0.37767293],
        [0.46285086, 0.02249957, 0.88615058]],
        [[-0.55356798, -0.83061331, -0.06036579],
        [0.47748068, -0.25716018, -0.84016715],
        [0.68233034, -0.49391313, 0.53895744]]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2DCM(EAxzy,'xzy',1e-7)
    e1 = np.array([[[0.1423907, 0.3610947, -0.921594],
        [0.72441892, -0.67249154, -0.15156632],
        [-0.67449397, -0.6460385, -0.35734037]],
        [[0.55244474, -0.78911624, -0.26851512],
        [0.69323432, 0.61383167, -0.37767294],
        [0.46285093, 0.02249953, 0.88615054]],
        [[-0.55356805, -0.83061327, -0.06036562],
        [0.47748052, -0.25716028, -0.84016721],
        [0.6823304, -0.49391313, 0.53895737]]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2DCM(EAyzx,'yzx',1e-7)
    e1 = np.array([[[0.14239073, 0.36109476, -0.92159397],
        [0.72441895, -0.67249151, -0.15156633],
        [-0.67449393, -0.6460385, -0.35734045]],
        [[0.55244476, -0.78911625, -0.26851505],
        [0.69323434, 0.61383167, -0.37767292],
        [0.46285088, 0.02249957, 0.88615057]],
        [[-0.55356805, -0.83061327, -0.06036562],
        [0.47748055, -0.25716031, -0.84016719],
        [0.68233037, -0.49391312, 0.53895741]]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2DCM(EAyxz,'yxz',1e-7)
    e1 = np.array(        [[[0.14239065, 0.36109477, -0.92159398],
        [0.72441892, -0.67249152, -0.15156641],
        [-0.67449398, -0.64603848, -0.35734039]],
        [[0.55244475, -0.78911627, -0.26851502],
        [0.69323437, 0.61383164, -0.3776729],
        [0.46285084, 0.02249957, 0.88615059]],
        [[-0.55356796, -0.83061333, -0.06036567],
        [0.47748061, -0.25716023, -0.84016717],
        [0.68233041, -0.49391306, 0.53895742]]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2DCM(EAzxy,'zxy',1e-7)
    e1 = np.array([[[0.14239075, 0.36109477, -0.92159397],
        [0.72441898, -0.67249149, -0.15156631],
        [-0.6744939, -0.64603852, -0.35734047]],
        [[0.55244476, -0.78911626, -0.26851503],
        [0.69323435, 0.61383165, -0.37767293],
        [0.46285087, 0.02249959, 0.88615057]],
        [[-0.55356796, -0.83061333, -0.06036565],
        [0.47748068, -0.25716029, -0.84016712],
        [0.68233036, -0.49391303, 0.53895751]]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2DCM(EAzyx,'zyx',1e-7)
    e1 = np.array([[[0.14239073, 0.3610948, -0.92159396],
        [0.72441899, -0.67249146, -0.15156634],
        [-0.67449388, -0.64603853, -0.35734048]],
        [[0.55244473, -0.78911628, -0.26851504],
        [0.69323435, 0.61383163, -0.37767295],
        [0.4628509, 0.02249958, 0.88615056]],
        [[-0.55356798, -0.83061331, -0.06036564],
        [0.4774806, -0.25716027, -0.84016717],
        [0.6823304, -0.49391307, 0.53895743]]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2DCM(EAxzx,'xzx',1e-7)
    e1 = np.array([[[0.14239072, 0.36109517, -0.92159382],
        [0.72441909, -0.6724913, -0.15156656],
        [-0.67449378, -0.64603848, -0.35734076]],
        [[0.55244474, -0.78911617, -0.26851533],
        [0.69323451, 0.61383179, -0.3776724],
        [0.46285064, 0.02249904, 0.88615071]],
        [[-0.55356815, -0.83061319, -0.06036581],
        [0.47748049, -0.25716015, -0.84016727],
        [0.68233033, -0.49391334, 0.53895726]]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2DCM(EAyxy,'yxy',1e-7)
    e1 = np.array([[[0.14239054, 0.36109473, -0.92159402],
        [0.72441894, -0.6724915, -0.15156645],
        [-0.67449399, -0.64603853, -0.35734029]],
        [[0.5524448, -0.78911627, -0.26851492],
        [0.69323442, 0.61383164, -0.37767281],
        [0.46285071, 0.0224996, 0.88615066]],
        [[-0.55356801, -0.8306133, -0.06036565],
        [0.47748056, -0.25716025, -0.8401672],
        [0.6823304, -0.49391311, 0.53895739]]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2DCM(EAyzy,'yzy',1e-7)
    e1 = np.array(        [[[0.14239037, 0.36109455, -0.92159411],
        [0.72441894, -0.6724915, -0.15156643],
        [-0.67449402, -0.64603863, -0.35734005]],
        [[0.5524448, -0.78911627, -0.26851492],
        [0.69323443, 0.61383164, -0.37767279],
        [0.4628507, 0.02249958, 0.88615066]],
        [[-0.55356809, -0.83061323, -0.06036573],
        [0.47748057, -0.25716025, -0.84016719],
        [0.68233032, -0.49391321, 0.53895739]]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2DCM(EAzxz,'zxz',1e-7)
    e1 = np.array(        [[[0.14239092, 0.36109459, -0.92159401],
        [0.7244189, -0.67249161, -0.1515661],
        [-0.67449394, -0.64603849, -0.35734045]],
        [[0.55244439, -0.78911657, -0.26851488],
        [0.69323462, 0.61383125, -0.37767307],
        [0.4628509, 0.02249956, 0.88615056]],
        [[-0.55356785, -0.83061338, -0.06036594],
        [0.47748073, -0.25715996, -0.84016719],
        [0.68233041, -0.49391312, 0.53895737]]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2DCM(EAzyz,'zyz',1e-7)
    e1 = np.array(        [[[0.14239075, 0.36109475, -0.92159398],
        [0.72441895, -0.67249151, -0.15156631],
        [-0.67449392, -0.6460385, -0.35734045]],
        [[0.55244474, -0.78911626, -0.26851506],
        [0.69323435, 0.61383165, -0.37767294],
        [0.4628509, 0.02249957, 0.88615056]],
        [[-0.55356798, -0.83061331, -0.06036566],
        [0.47748056, -0.25716021, -0.84016721],
        [0.68233042, -0.4939131, 0.53895737]]])
    assert_array_almost_equal(r1, e1)

def test_EA2EV():
    r1 = sp.EA2EV(EAxyx,'xyx',1e-7)
    e1 = np.array([[-0.19953006, -0.87653822, -0.43802796, 1.99722268],
        [-0.96466692, 0.12646762, -0.23113561, 2.26537648],
        [-0.66427919, 0.31584756, -0.67747582, 1.39505007]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2EV(EAxyz,'xyz',1e-7)
    e1 = np.array([[-0.1995301, -0.87653823, -0.43802793, 1.9972227],
        [-0.96466693, 0.12646762, -0.23113558, 2.26537641],
        [-0.66427931, 0.31584748, -0.67747573, 1.39505022]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2EV(EAxzy,'xzy',1e-7)
    e1 = np.array([[0.19953009, 0.87653823, 0.43802794, 4.28596269],
        [-0.96466692, 0.12646761, -0.23113563, 2.2653765],
        [-0.66427922, 0.31584759, -0.67747577, 1.39505017]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2EV(EAyzx,'yzx',1e-7)
    e1 = np.array([[-0.19953008, -0.87653822, -0.43802796, 1.99722269],
        [-0.96466693, 0.12646762, -0.23113559, 2.2653765],
        [-0.66427924, 0.31584758, -0.67747576, 1.39505013]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2EV(EAyxz,'yxz',1e-7)
    e1 = np.array([[0.19953005, 0.87653822, 0.43802797, 4.2859627],
        [-0.96466693, 0.12646761, -0.23113557, 2.26537646],
        [-0.66427929, 0.31584757, -0.67747571, 1.39505021]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2EV(EAzxy,'zxy',1e-7)
    e1 = np.array([[-0.19953009, -0.87653822, -0.43802796, 1.99722273],
        [-0.96466693, 0.12646762, -0.23113559, 2.26537648],
        [-0.66427933, 0.31584756, -0.67747567, 1.39505014]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2EV(EAzyx,'zyx',1e-7)
    e1 = np.array([[0.19953008, 0.87653821, 0.43802798, 4.28596257],
        [-0.96466692, 0.12646763, -0.2311356, 2.26537646],
        [-0.66427928, 0.31584758, -0.67747572, 1.39505018]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2EV(EAxzx,'xzx',1e-7)
    e1 = np.array([[0.19952995, 0.87653817, 0.43802813, 4.28596234],
        [-0.96466697, 0.12646728, -0.2311356, 2.26537647],
        [-0.66427915, 0.31584746, -0.6774759, 1.39505024]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2EV(EAyxy,'yxy',1e-7)
    e1 = np.array([[-0.19953007, -0.8765382, -0.438028, 1.9972225],
        [-0.96466695, 0.12646759, -0.2311355, 2.26537645],
        [-0.66427925, 0.31584758, -0.67747575, 1.39505019]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2EV(EAyzy,'yzy',1e-7)
    e1 = np.array([[0.19953015, 0.87653819, 0.438028, 4.28596302],
        [-0.96466695, 0.12646758, -0.2311355, 2.26537644],
        [-0.66427923, 0.3158475, -0.6774758, 1.39505015]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2EV(EAzxz,'zxz',1e-7)
    e1 = np.array([[-0.19953015, -0.87653827, -0.43802783, 1.99722274],
        [-0.96466692, 0.12646769, -0.2311356, 2.26537599],
        [-0.66427935, 0.31584742, -0.67747572, 1.39505044]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.EA2EV(EAzyz,'zyz',1e-7)
    e1 = np.array([[-0.19953009, -0.87653823, -0.43802795, 1.9972227],
        [-0.96466692, 0.12646762, -0.2311356, 2.26537647],
        [-0.66427925, 0.31584758, -0.67747575, 1.39505024]])
    assert_array_almost_equal(r1, e1)

def test_EV2EA():
    n = 0
    for EAv in EAvct:
        for EVtmp in [EV1, EV2, EV3]:
            r1 = sp.EV2EA(EVtmp,EAv).reshape(3)
            e1 = EAResult4[n]
            assert_array_almost_equal(r1, e1)
            n = n + 1
    n = 0
    for EAv in EAvct:
        r1 = sp.EV2EA(EV4,EAv)
        e1 = EAResult4[range(n,1+(n+2)),:]
        assert_array_almost_equal(r1, e1)
        n = n + 3
        
def test_Q2GL():
    r1 = sp.Q2GL(Q1)
    e1 = np.array([[-0.67249152, -0.15156633, -0.72441894, 0.],
        [0.6460385, 0.35734043, -0.67449394, 0.],
        [0.36109475, -0.92159398, -0.14239072, 0.],
        [0., 0., 0., 1.]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.Q2GL(Q2)
    e1 = np.array([[0.61383164, -0.37767294, -0.69323436, 0.],
        [-0.02249957, -0.88615056, 0.46285089, 0.],
        [-0.78911627, -0.26851505, -0.55244473, 0.],
        [0., 0., 0., 1.]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.Q2GL(Q3)
    e1 = np.array([[-0.25716027, -0.84016719, -0.47748057, 0.],
        [0.49391309, -0.5389574, 0.6823304, 0.],
        [-0.8306133, -0.06036564, 0.553568, 0.],
        [0., 0., 0., 1.]])
    assert_array_almost_equal(r1, e1)
    r1 = sp.Q2GL(Q4)
    e1 = np.array([[-0.67249152, -0.15156633, -0.72441894, 0.],
        [0.6460385, 0.35734043, -0.67449394, 0.],
        [0.36109475, -0.92159398, -0.14239072, 0.],
        [0., 0., 0., 1.],
        [0.61383164, -0.37767294, -0.69323436, 0.],
        [-0.02249957, -0.88615056, 0.46285089, 0.],
        [-0.78911627, -0.26851505, -0.55244473, 0.],
        [0., 0., 0., 1.],
        [-0.25716027, -0.84016719, -0.47748057, 0.],
        [0.49391309, -0.5389574, 0.6823304, 0.],
        [-0.8306133, -0.06036564, 0.553568, 0.],
        [0., 0., 0., 1.]])
    assert_array_almost_equal(r1, e1)

def test_EA2EA():
    r1 = sp.EA2EA(EAxyx,"xyx","xyz")
    assert_array_almost_equal(r1, EAxyz)
    r1 = sp.EA2EA(EAxyx,"xyx","xzy")
    assert_array_almost_equal(r1, EAxzy)
    r1 = sp.EA2EA(EAxyx,"xyx","yzx")
    assert_array_almost_equal(r1, EAyzx)
    r1 = sp.EA2EA(EAxyx,"xyx","yxz")
    assert_array_almost_equal(r1, EAyxz)
    r1 = sp.EA2EA(EAxyx,"xyx","zxy")
    assert_array_almost_equal(r1, EAzxy)
    r1 = sp.EA2EA(EAxyx,"xyx","zyx")
    assert_array_almost_equal(r1, EAzyx)
    r1 = sp.EA2EA(EAxyx,"xyx","xzx")
    assert_array_almost_equal(r1, EAxzx)
    r1 = sp.EA2EA(EAxyx,"xyx","yxy")
    assert_array_almost_equal(r1, EAyxy)
    r1 = sp.EA2EA(EAxyx,"xyx","yzy")
    assert_array_almost_equal(r1, EAyzy)
    r1 = sp.EA2EA(EAxyx,"xyx","zxz")
    assert_array_almost_equal(r1, EAzxz)
    r1 = sp.EA2EA(EAxyx,"xyx","zyz")
    assert_array_almost_equal(r1, EAzyz)
