from django.db import models

# Create your models here.
# pg 102 max of 5 models per app

from django.urls import reverse  # To generate URLS by reversing URL patterns
import uuid  # Required for unique book instances
from datetime import date
# Required to assign User as a borrower
from django.contrib.auth.models import User


class Genre(models.Model):
    # This model represents book genre.
    name = models.CharField(
        max_length=200,
        help_text="Enter a book genre (e.g. Science Fiction etc.)")

    def __str__(self):
        # Gets friendly name of object
        return self.name


class Language(models.Model):
    # Model that represents a language
    name = models.CharField(max_length=200,
                            help_text="Enter the book's language)")

    def __str__(self):
        return self.name


class Book(models.Model):
    # Model representing a book
    title = models.CharField(max_length=200)
    author = models.ForeignKey('Author', on_delete=models.SET_NULL, null=True)
    # Foreign Key used because book can only have one author,
    # but authors can have multiple books
    # Author as a string rather than object
    # because it hasn't been declared yet in file.
    summary = models.TextField(max_length=1000,
                               help_text="Enter a description of the book")
    isbn = models.CharField('ISBN', max_length=13,
                            help_text='13 Character <a href="isbn.org'
                                      '">ISBN number</a>')
    genre = models.ManyToManyField(Genre,
                                   help_text="Select a genre for this book")
    # ManyToManyField used because a genre can
    # contain many books and a Book can cover many genres.
    # Genre class has already been defined so we can specify the object above.
    language = models.ForeignKey('Language',
                                 on_delete=models.SET_NULL, null=True)

    def display_genre(self):
        """Creates a string for the Genre.
           This is required to display genre in Admin."""
        return ', '.join([genre.name for genre in self.genre.all()[:3]])

    display_genre.short_description = 'Genre'

    def get_absolute_url(self):
        """Returns the url to access a particular book instance."""
        return reverse('book-detail', args=[str(self.id)])

    def __str__(self):
        """String for representing the Model object."""
        return self.title


class BookInstance(models.Model):
    # Unlike book model, this one represents specific instance of a book
    id = models.UUIDField(primary_key=True, default=uuid.uuid4,
                          help_text="Unique ID for this book across library")
    book = models.ForeignKey('Book', on_delete=models.SET_NULL, null=True)
    imprint = models.CharField(max_length=200)
    due_back = models.DateField(null=True, blank=True)
    borrower = models.ForeignKey(User, on_delete=models.SET_NULL,
                                 null=True, blank=True)

    # this is built in function that turns the is_overdue
    # method into a getter for a read only attribute with
    # the same name, you can now use @is_overdue.setter
    @property
    def is_overdue(self):
        if self.due_back and date.today() > self.due_back:
            return True
        return False

    LOAN_STATUS = (
        ('d', 'Maintenance'),
        ('o', 'On loan'),
        ('a', 'Available'),
        ('r', 'Reserved'),
    )

    status = models.CharField(
        max_length=1,
        choices=LOAN_STATUS,
        blank=True,
        default='d',
        help_text='Book availability')

    class Meta:
        ordering = ['due_back']
        permissions = (("can_mark_returned", "Set book as returned"),)

    def __str__(self):
        """String for representing the Model object."""
        return '{0} ({1})'.format(self.id, self.book.title)


class Author(models.Model):
    # Model representing an author
    first_name = models.CharField(max_length=100)
    last_name = models.CharField(max_length=100)
    date_of_birth = models.DateField(null=True, blank=True)
    date_of_death = models.DateField('died', null=True, blank=True)

    class Meta:
        ordering = ['last_name', 'first_name']

    def get_absolute_url(self):
        """Returns the url to access a particular author instance."""
        return reverse('author-detail', args=[str(self.id)])

    def __str__(self):
        """String for representing the Model object."""
        return '{0}, {1}'.format(self.last_name, self.first_name)
