# !/usr/bin/env python
# -*- coding: utf-8 -*-

"""
@Time    : 2023-10-14 23:05:35
@Author  : Rey
@Contact : reyxbo@163.com
@Explain : Database build methods.
"""


from typing import Any, List, Tuple, Dict, Optional, Union, Literal, NoReturn, overload

from .rdatabase_engine import REngine
from ..rprint import rprint
from ..rsystem import get_first_notnull, rexc


class RBuild(object):
    """
    Rey's `database build` type.
    """


    def __init__(self, rengine: REngine) -> None:
        """
        Build `database build` instance.

        Parameters
        ----------
        rengine : REngine object.
        """

        # Set attribute.
        self.rengine = rengine


    def create_database(
        self,
        database: str,
        character: str = "utf8mb3",
        collate: str = "utf8_general_ci",
        execute: bool = True
    ) -> str:
        """
        Create `database`.

        Parameters
        ----------
        database : Database name.
        character : Character set.
        collate : Collate rule.
        execute : Whether directly execute.

        Returns
        -------
        Execute SQL.
        """

        # Generate.
        sql = f"CREATE DATABASE `{database}` CHARACTER SET {character} COLLATE {collate}"

        # Execute.
        if execute:
            self.rengine(sql)

        return sql


    def _get_field_sql(
        self,
        name: str,
        type_: str,
        constraint: str = "DEFAULT NULL",
        comment: Optional[str] = None,
        position: Optional[str] = None,
        old_name: Optional[str] = None
    ) -> str:
        """
        Get a field set SQL.

        Parameters
        ----------
        name : Field name.
        type_ : Field type.
        constraint : Field constraint.
        comment : Field comment.
        position : Field position.
        old_name : Field old name.

        Returns
        -------
        Field set SQL.
        """

        # Get parameter.

        ## Constraint.
        constraint = " " + constraint

        ## Comment.
        if comment is None:
            comment = ""
        else:
            comment = f" COMMENT '{comment}'"

        ## Position.
        if position is None:
            position = ""
        elif position == "first":
            position = " FIRST"
        else:
            position = f" AFTER `{position}`"

        ## Old name.
        if old_name is None:
            old_name = ""
        else:
            old_name = f"`{old_name}` "

        # Generate.
        sql = f"{old_name}`{name}` {type_}{constraint}{comment}{position}"

        return sql


    @overload
    def _get_index_sql(
        self,
        name: str,
        fields: Union[str, List[str]],
        type_: Literal["noraml", "unique", "fulltext", "spatial"] = "noraml",
        comment: Optional[str] = None
    ) -> str: ...

    @overload
    def _get_index_sql(
        self,
        name: str,
        fields: Union[str, List[str]],
        type_: str = "noraml",
        comment: Optional[str] = None
    ) -> NoReturn: ...

    def _get_index_sql(
        self,
        name: str,
        fields: Union[str, List[str]],
        type_: Literal["noraml", "unique", "fulltext", "spatial"] = "noraml",
        comment: Optional[str] = None
    ) -> str:
        """
        Get a index set SQL.

        Parameters
        ----------
        name : Index name.
        fields : Index fileds.
        type_ : Index type.
        comment : Index comment.

        Returns
        -------
        Index set SQL.
        """

        # Get parameter.
        if fields.__class__ == str:
            fields = [fields]
        if type_ == "noraml":
            type_ = "KEY"
            method = " USING BTREE"
        elif type_ == "unique":
            type_ = "UNIQUE KEY"
            method = " USING BTREE"
        elif type_ == "fulltext":
            type_ = "FULLTEXT KEY"
            method = ""
        elif type_ == "spatial":
            type_ = "SPATIAL KEY"
            method = ""
        else:
            rexc(ValueError, type_)
        if comment in (None, ""):
            comment = ""
        else:
            comment = f" COMMENT '{comment}'"

        # Generate.

        ## Fields.
        sql_fields = ", ".join(
            [
                f"`{field}`"
                for field in fields
            ]
        )

        ## Join.
        sql = f"{type_} `{name}` ({sql_fields}){method}{comment}"

        return sql


    def create_table(
        self,
        path: Union[str, Tuple[str, str]],
        fields: Union[Dict, List[Dict]],
        primary: Optional[Union[str, List[str]]] = None,
        indexes: Optional[Union[Dict, List[Dict]]] = None,
        engine: str = "InnoDB",
        increment: int = 1,
        charset: str = "utf8mb3",
        collate: str = "utf8_general_ci",
        execute: bool = True
    ) -> str:
        """
        Create `table`.

        Parameters
        ----------
        path : Table name, can contain database name, otherwise use `self.rengine.database`.
            - `str` : Automatic extract database name and table name.
            - `Tuple[str, str]` : Database name and table name.

        fields : Fields set table.
            - `Key 'name'` : Field name.
            - `Key 'type_'` : Field type.
            - `Key 'constraint'` : Field constraint.
                * `Empty or None` : Use 'DEFAULT NULL'.
                * `str` : Use this value.
            - `Key 'comment'` : Field comment.
                * `Empty or None` : Not comment.
                * `str` : Use this value.

        primary : Primary key fields.
            - `str` : One field.
            - `List[str]` : Multiple fileds.

        indexes : Index set table.
            - `Key 'name'` : Index name.
            - `Key 'fields'` : Index fields.
                * `str` : One field.
                * `List[str]` : Multiple fileds.
            - `Key 'type_'` : Index type.
                * `Literal['noraml']` : Noraml key.
                * `Literal['unique']` : Unique key.
                * `Literal['fulltext']` : Full text key.
                * `Literal['spatial']` : Spatial key.
            - `Key 'comment'` : Field comment.
                * `Empty or None` : Not comment.
                * `str` : Use this value.

        engine : Engine type.
        increment : Automatic Increment start value.
        charset : Charset type.
        collate : Collate type.
        execute : Whether directly execute.

        Returns
        -------
        Execute SQL.
        """

        # Handle parameter.
        if path.__class__ == str:
            database, table, _ = self.rengine.extract_path(path)
        else:
            database, table = path
        database = get_first_notnull(database, self.rengine.database, default="exception")
        if fields.__class__ == dict:
            fields = [fields]
        if primary.__class__ == str:
            primary = [primary]
        if primary in ([], [""]):
            primary = None
        if indexes.__class__ == dict:
            indexes = [indexes]

        # Generate.

        ## Fields.
        sql_fields = [
            self._get_field_sql(**field)
            for field in fields
        ]

        ## Primary.
        if primary is not None:
            keys = ", ".join(
                [
                    f"`{key}`"
                    for key in primary
                ]
            )
            sql_primary = f"PRIMARY KEY ({keys}) USING BTREE"
            sql_fields.append(sql_primary)

        ## Indexes.
        if indexes is not None:
            sql_indexes = [
                self._get_index_sql(**index)
                for index in indexes
            ]
            sql_fields.extend(sql_indexes)

        ## Join.
        sql_fields = ",\n    ".join(sql_fields)
        sql = (
            f"CREATE TABLE `{database}`.`{table}`(\n"
            f"    {sql_fields}\n"
            f") ENGINE={engine} AUTO_INCREMENT={increment} CHARSET={charset} COLLATE={collate}"
        )

        # Execute.
        if execute:
            self.rengine(sql)

        return sql


    def drop_database(
        self,
        database: str,
        execute: bool = True
    ) -> str:
        """
        Drop `database`.

        Parameters
        ----------
        database : Database name.
        execute : Whether directly execute.

        Returns
        -------
        Execute SQL.
        """

        # Generate.
        sql = f"DROP DATABASE `{database}`"

        # Execute.
        if execute:
            self.rengine(sql)

        return sql


    def drop_table(
        self,
        path: Union[str, Tuple[str, str]],
        execute: bool = True
    ) -> str:
        """
        Drop `table`.

        Parameters
        ----------
        path : Table name, can contain database name, otherwise use `self.rengine.database`.
            - `str` : Automatic extract database name and table name.
            - `Tuple[str, str]` : Database name and table name.

        execute : Whether directly execute.

        Returns
        -------
        Execute SQL.
        """

        # Handle parameter.
        if path.__class__ == str:
            database, table, _ = self.rengine.extract_path(path)
        else:
            database, table = path
        database = get_first_notnull(database, self.rengine.database, default="exception")

        # Generate.
        sql = f"DROP TABLE `{database}`.`{table}`"

        # Execute.
        if execute:
            self.rengine(sql)

        return sql


    def alter_database(
        self,
        database: str,
        character: Optional[str] = None,
        collate: Optional[str] = None,
        execute: bool = True
    ) -> str:
        """
        Alter `database`.

        Parameters
        ----------
        database : Database name.
        character : Character set.
            - `None` : Not alter.
            - `str` : Alter to this value.

        collate : Collate rule.
            - `None` : Not alter.
            - `str` : Alter to this value.

        execute : Whether directly execute.

        Returns
        -------
        Execute SQL.
        """

        # Generate.

        ## Character.
        if character is None:
            sql_character = ""
        else:
            sql_character = f" CHARACTER SET {character}"

        ## Collate.
        if collate is None:
            sql_collate = ""
        else:
            sql_collate = f" COLLATE {collate}"

        ## Join.
        sql = f"ALTER DATABASE `{database}`{sql_character}{sql_collate}"

        # Execute.
        if execute:
            self.rengine(sql)

        return sql


    def alter_table_add(
        self,
        path: Union[str, Tuple[str, str]],
        fields: Optional[Union[Dict, List[Dict]]] = None,
        primary: Optional[Union[str, List[str]]] = None,
        indexes: Optional[Union[Dict, List[Dict]]] = None,
        execute: bool = True
    ) -> str:
        """
        Alter `table` add `filed`.

        Parameters
        ----------
        path : Table name, can contain database name, otherwise use `self.rengine.database`.
            - `str` : Automatic extract database name and table name.
            - `Tuple[str, str]` : Database name and table name.

        fields : Fields set table.
            - `Key 'name'` : Field name.
            - `Key 'type_'` : Field type.
            - `Key 'constraint'` : Field constraint.
                * `Empty or None` : Use 'DEFAULT NULL'.
                * `str` : Use this value.
            - `Key 'comment'` : Field comment.
                * `Empty or None` : Not comment.
                * `str` : Use this value.
            - `Key 'position'` : Field position.
                * `None` : Last.
                * `Literal['first']` : First.
                * `str` : After this field.

        primary : Primary key fields.
            - `str` : One field.
            - `List[str]` : Multiple fileds.

        indexes : Index set table.
            - `Key 'name'` : Index name.
            - `Key 'fields'` : Index fields.
                * `str` : One field.
                * `List[str]` : Multiple fileds.
            - `Key 'type_'` : Index type.
                * `Literal['noraml']` : Noraml key.
                * `Literal['unique']` : Unique key.
                * `Literal['fulltext']` : Full text key.
                * `Literal['spatial']` : Spatial key.
            - `Key 'comment'` : Field comment.
                * `Empty or None` : Not comment.
                * `str` : Use this value.

        Returns
        -------
        Execute SQL.
        """

        # Handle parameter.
        if path.__class__ == str:
            database, table, _ = self.rengine.extract_path(path)
        else:
            database, table = path
        database = get_first_notnull(database, self.rengine.database, default="exception")
        if fields.__class__ == dict:
            fields = [fields]
        if primary.__class__ == str:
            primary = [primary]
        if primary in ([], [""]):
            primary = None
        if indexes.__class__ == dict:
            indexes = [indexes]

        # Generate.
        sql_content = []

        ## Fields.
        if fields is not None:
            sql_fields = [
                "COLUMN " + self._get_field_sql(**field)
                for field in fields
            ]
            sql_content.extend(sql_fields)

        ## Primary.
        if primary is not None:
            keys = ", ".join(
                [
                    f"`{key}`"
                    for key in primary
                ]
            )
            sql_primary = f"PRIMARY KEY ({keys}) USING BTREE"
            sql_content.append(sql_primary)

        ## Indexes.
        if indexes is not None:
            sql_indexes = [
                self._get_index_sql(**index)
                for index in indexes
            ]
            sql_content.extend(sql_indexes)

        ## Join.
        sql_content = ",\n    ADD ".join(sql_content)
        sql = (
            f"ALTER TABLE `{database}`.`{table}`\n"
            f"    ADD {sql_content}"
        )

        # Execute.
        if execute:
            self.rengine(sql)

        return sql


    def alter_table_drop(
        self,
        path: Union[str, Tuple[str, str]],
        fields: Optional[List[str]] = None,
        primary: bool = False,
        indexes: Optional[List[str]] = None,
        execute: bool = True
    ) -> str:
        """
        Alter `table` drop `field`.

        Parameters
        ----------
        path : Table name, can contain database name, otherwise use `self.rengine.database`.
            - `str` : Automatic extract database name and table name.
            - `Tuple[str, str]` : Database name and table name.

        fields : Delete fields name.
        primary : Whether delete primary key.
        indexes : Delete indexes name.
        execute : Whether directly execute.

        Returns
        -------
        Execute SQL.
        """

        # Handle parameter.
        if path.__class__ == str:
            database, table, _ = self.rengine.extract_path(path)
        else:
            database, table = path
        database = get_first_notnull(database, self.rengine.database, default="exception")

        # Generate.
        sql_content = []

        ## Fields.
        if fields is not None:
            sql_fields = [
                "COLUMN " + field
                for field in fields
            ]
            sql_content.extend(sql_fields)

        ## Primary.
        if primary:
            sql_primary = "PRIMARY KEY"
            sql_content.append(sql_primary)

        ## Indexes.
        if indexes is not None:
            sql_indexes = [
                "INDEX " + index
                for index in indexes
            ]
            sql_content.extend(sql_indexes)

        ## Join.
        sql_content = ",\n    DROP ".join(sql_content)
        sql = (
            f"ALTER TABLE `{database}`.`{table}`\n"
            f"    DROP {sql_content}"
        )

        # Execute.
        if execute:
            self.rengine(sql)

        return sql


    def alter_table_change(
        self,
        path: Union[str, Tuple[str, str]],
        fields: Optional[Union[Dict, List[Dict]]] = None,
        rename: Optional[str] = None,
        engine: Optional[str] = None,
        increment: Optional[int] = None,
        charset: Optional[str] = None,
        collate: Optional[str] = None,
        execute: bool = True
    ) -> str:
        """
        Alter `database`.

        Parameters
        ----------
        path : Table name, can contain database name, otherwise use `self.rengine.database`.
            - `str` : Automatic extract database name and table name.
            - `Tuple[str, str]` : Database name and table name.

        fields : Fields set table.
            - `Key 'name'` : Field name.
            - `Key 'type_'` : Field type.
            - `Key 'constraint'` : Field constraint.
                * `Empty or None` : Use 'DEFAULT NULL'.
                * `str` : Use this value.
            - `Key 'comment'` : Field comment.
                * `Empty or None` : Not comment.
                * `str` : Use this value.
            - `Key 'position'` : Field position.
                * `None` : Last.
                * `Literal['first']` : First.
                * `str` : After this field.
            - `Key 'old_name'` : Field old name.

        rename : Table new name.
        engine : Engine type.
        increment : Automatic Increment start value.
        charset : Charset type.
        collate : Collate type.
        execute : Whether directly execute.

        Returns
        -------
        Execute SQL.
        """

        # Handle parameter.
        if path.__class__ == str:
            database, table, _ = self.rengine.extract_path(path)
        else:
            database, table = path
        database = get_first_notnull(database, self.rengine.database, default="exception")
        if fields.__class__ == dict:
            fields = [fields]

        # Generate.
        sql_content = []

        ## Rename.
        if rename is not None:
            sql_rename = f"RENAME `{database}`.`{rename}`"
            sql_content.append(sql_rename)

        ## Fields.
        if fields is not None:
            sql_fields = [
                "CHANGE %s" % self._get_field_sql(**field)
                for field in fields
            ]
            sql_content.extend(sql_fields)

        ## Attribute.
        sql_attr = []

        ### Engine.
        if engine is not None:
            sql_engine = f"ENGINE={engine}"
            sql_attr.append(sql_engine)

        ### Increment.
        if increment is not None:
            sql_increment = f"AUTO_INCREMENT={increment}"
            sql_attr.append(sql_increment)

        ### Charset.
        if charset is not None:
            sql_charset = f"CHARSET={charset}"
            sql_attr.append(sql_charset)

        ### Collate.
        if collate is not None:
            sql_collate = f"COLLATE={collate}"
            sql_attr.append(sql_collate)

        if sql_attr != []:
            sql_attr = " ".join(sql_attr)
            sql_content.append(sql_attr)

        ## Join.
        sql_content = ",\n    ".join(sql_content)
        sql = (
            f"ALTER TABLE `{database}`.`{table}`\n"
            f"    {sql_content}"
        )

        # Execute.
        if execute:
            self.rengine(sql)

        return sql


    def truncate_table(
        self,
        path: Union[str, Tuple[str, str]],
        execute: bool = True
    ) -> str:
        """
        Truncate `table`.

        Parameters
        ----------
        path : Table name, can contain database name, otherwise use `self.rengine.database`.
            - `str` : Automatic extract database name and table name.
            - `Tuple[str, str]` : Database name and table name.

        execute : Whether directly execute.

        Returns
        -------
        Execute SQL.
        """

        # Handle parameter.
        if path.__class__ == str:
            database, table, _ = self.rengine.extract_path(path)
        else:
            database, table = path
        database = get_first_notnull(database, self.rengine.database, default="exception")

        # Generate.
        sql = f"TRUNCATE TABLE `{database}`.`{table}`"

        # Execute.
        if execute:
            self.rengine(sql)

        return sql


    def exist(
        self,
        path: Union[str, Tuple[str, Optional[str], Optional[str]]]
    ) -> bool:
        """
        Judge database or table or column exists.

        Parameters
        ----------
        path : Database name and table name and column name.
            - `str` : Automatic extract.
            - `Tuple[str, Optional[str], Optional[str]]` : Database name, table name and column name is optional.

        Returns
        -------
        Judge result.
        """

        # Handle parameter.
        if path.__class__ == str:
            database, table, column = self.rengine.extract_path(path, "database")
        else:
            database, table, column = path

        # Judge.
        if table is None:
            rinfo = self.rengine.info(database)
        elif column is None:
            rinfo = self.rengine.info(database)(table)
        else:
            rinfo = self.rengine.info(database)(table)(column)
        try:
            rinfo["*"]
        except AssertionError:
            judge = False
        else:
            judge = True

        return judge


    def input_confirm_build(
        self,
        sql: str
    ) -> None:
        """
        `Print` tip text, and `confirm` execute SQL. If reject, throw exception.

        Parameters
        ----------
        sql : SQL.
        """

        # Confirm.
        command = rprint.rinput(sql, "Do you want to execute SQL to build the database? Otherwise stop program. (y/n): ", frame="top", title="SQL")

        # Check.
        command = command.lower()
        if command == "n":
            raise AssertionError("program stop")