u"""
LICENSE:
Copyright 2015,2016 Hermann Krumrey

This file is part of toktokkie.

    toktokkie is a program that allows convenient managing of various
    local media collections, mostly focused on video.

    toktokkie is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    toktokkie is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with toktokkie.  If not, see <http://www.gnu.org/licenses/>.
LICENSE
"""

# imports
from __future__ import absolute_import
import os
from typing import List, Dict
from toktokkie.modules.utils.Renamer import Renamer
from toktokkie.modules.gui.framework import GlobalGuiFramework


class RenamerGui(GlobalGuiFramework.selected_grid_gui_framework):
    u"""
    GUI for the Renamer plugin that allows the user to select a directory to rename the contents of
    """

    button = None
    u"""
    A button that starts the renaming of the currently entered directory
    """

    entry = None
    u"""
    A Text Entry that contains the path to the directory to be renamed
    """

    browse = None
    u"""
    A button that lets the user browse for a directory using a directory chooser dialog
    """

    def __init__(self, parent):
        u"""
        Constructor of the RenamerGui class. It calls the selected gfworks framework's constructor and
        hides the parent window

        :param parent: the parent gui window
        :return: None
        """
        super(RenamerGui, self).__init__(u"Renamer", parent, True)

    def lay_out(self):
        u"""
        Sets up all interface elements of the GUI and positions them in a grid layout

        :return: None
        """
        # Generate the button
        self.button = self.generate_button(u"Start", self.start_rename)
        self.position_absolute(self.button, 4, 0, 1, 1)

        # Generate the browse button
        self.browse = self.generate_button(u"Browse", self.browse_directory)
        self.position_absolute(self.browse, 0, 0, 1, 1)

        # Generate the Text Entry
        self.entry = self.generate_text_entry(u"", self.start_rename)
        self.position_absolute(self.entry, 1, 0, 2, 1)

    def start_rename(self, widget):
        u"""
        Starts the renaming process.

        During this time, the user is asked to confirm his selection

        :param widget: the button that started this method
        :return: None
        """
        # DO this to avoid IDE warnings about unused variables
        if widget is None:
            return

        # Get the currently entered directory path
        abs_dir = self.get_string_from_text_entry(self.entry)

        # Check if the path is a valid directory
        if not os.path.isdir(abs_dir):
            self.show_message_dialog(u"Error, ", u"Not a valid directory")
            return

        renamer = Renamer(abs_dir)  # Create a new Renamer object
        confirmation = renamer.request_confirmation()  # Request the confirmation dictionary from the Renamer
        if self.confirmer(confirmation):  # Ask the user for confirmation, and only continue if the answer is positive
            renamer.confirm(confirmation)  # Confirm the confirmation with the Renamer
            renamer.start_rename()  # Start renaming

        self.show_message_dialog(u"Renaming Complete!", u"")

    def browse_directory(self, widget):
        u"""
        Shows a directory chooser dialog and sets the entry to the result of the browse

        :param widget: the button that called this method
        :return: None
        """
        if widget is not None:  # Suppress IDE warnings
            selected_directory = self.show_directory_chooser_dialog()  # Open directory chooser dialog
            if selected_directory:  # If a directory was selected set the text entry to that directory path
                self.set_text_entry_string(self.entry, selected_directory)

    def confirmer(self, confirmation):
        u"""
        Asks the user for confirmation before continuing the renaming process

        This is done by the user clicking the 'Yes' Button on every Yes/No Dialog shown,
        each representing one renaming operation to be commited

        :param confirmation: the confirmation dictionary
        :return: False if the user did not confirm the rename, True otherwise.
        """

        for element in confirmation:  # Iterate over every element in the dictionary
            # Generate the message String
            message = u"Rename\n"
            message += element[u"old"]  # Include the old name
            message += u"\nto\n"
            message += element[u"new"]  # Include the new name
            message += u"\n?"
            response = self.show_yes_no_dialog(u"Confirmation", message)  # Show a yes/no dialog
            if not response:
                return False  # as soon as the user disapproves one operation, halt the renaming process
        return True  # If that doesn't happen, return True
