"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const node_version_1 = require("./node-version");
// list of NPM scopes included in version reporting e.g. @aws-cdk and @aws-solutions-konstruk
const WHITELIST_SCOPES = ['@aws-cdk', '@aws-solutions-konstruk'];
/**
 * Returns a list of loaded modules and their versions.
 */
function collectRuntimeInformation() {
    const libraries = {};
    for (const fileName of Object.keys(require.cache)) {
        const pkg = findNpmPackage(fileName);
        if (pkg && !pkg.private) {
            libraries[pkg.name] = pkg.version;
        }
    }
    // include only libraries that are in the whitelistLibraries list
    for (const name of Object.keys(libraries)) {
        let foundMatch = false;
        for (const scope of WHITELIST_SCOPES) {
            if (name.startsWith(`${scope}/`)) {
                foundMatch = true;
            }
        }
        if (!foundMatch) {
            delete libraries[name];
        }
    }
    // add jsii runtime version
    libraries['jsii-runtime'] = getJsiiAgentVersion();
    return { libraries };
}
exports.collectRuntimeInformation = collectRuntimeInformation;
/**
 * Determines which NPM module a given loaded javascript file is from.
 *
 * The only infromation that is available locally is a list of Javascript files,
 * and every source file is associated with a search path to resolve the further
 * ``require`` calls made from there, which includes its own directory on disk,
 * and parent directories - for example:
 *
 * [ '...repo/packages/aws-cdk-resources/lib/cfn/node_modules',
 *   '...repo/packages/aws-cdk-resources/lib/node_modules',
 *   '...repo/packages/aws-cdk-resources/node_modules',
 *   '...repo/packages/node_modules',
 *   // etc...
 * ]
 *
 * We are looking for ``package.json`` that is anywhere in the tree, except it's
 * in the parent directory, not in the ``node_modules`` directory. For this
 * reason, we strip the ``/node_modules`` suffix off each path and use regular
 * module resolution to obtain a reference to ``package.json``.
 *
 * @param fileName a javascript file name.
 * @returns the NPM module infos (aka ``package.json`` contents), or
 *      ``undefined`` if the lookup was unsuccessful.
 */
function findNpmPackage(fileName) {
    const mod = require.cache[fileName];
    if (!mod.paths) {
        // sometimes this can be undefined. for example when querying for .json modules
        // inside a jest runtime environment.
        // see https://github.com/aws/aws-cdk/issues/7657
        // potentially we can remove this if it turns out to be a bug in how jest implemented the 'require' module.
        return undefined;
    }
    const paths = mod.paths.map(stripNodeModules);
    try {
        const packagePath = require.resolve(
        // Resolution behavior changed in node 12.0.0 - https://github.com/nodejs/node/issues/27583
        node_version_1.major >= 12 ? './package.json' : 'package.json', { paths });
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require(packagePath);
    }
    catch (e) {
        return undefined;
    }
    /**
     * @param s a path.
     * @returns ``s`` with any terminating ``/node_modules``
     *      (or ``\\node_modules``) stripped off.)
     */
    function stripNodeModules(s) {
        if (s.endsWith('/node_modules') || s.endsWith('\\node_modules')) {
            // /node_modules is 13 characters
            return s.substr(0, s.length - 13);
        }
        return s;
    }
}
function getJsiiAgentVersion() {
    let jsiiAgent = process.env.JSII_AGENT;
    // if JSII_AGENT is not specified, we will assume this is a node.js runtime
    // and plug in our node.js version
    if (!jsiiAgent) {
        jsiiAgent = `node.js/${process.version}`;
    }
    return jsiiAgent;
}
//# sourceMappingURL=data:application/json;base64,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