"use strict";
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
const app_1 = require("../lib/app");
function withApp(props, block) {
    const app = new app_1.App({
        runtimeInfo: false,
        stackTraces: false,
        ...props,
    });
    block(app);
    return app.synth();
}
function synth(context) {
    return withApp({ context }, app => {
        const stack1 = new lib_1.Stack(app, 'stack1', { env: { account: '12345', region: 'us-east-1' } });
        new lib_1.CfnResource(stack1, 's1c1', { type: 'DummyResource', properties: { Prop1: 'Prop1' } });
        const r2 = new lib_1.CfnResource(stack1, 's1c2', { type: 'DummyResource', properties: { Foo: 123 } });
        const stack2 = new lib_1.Stack(app, 'stack2');
        new lib_1.CfnResource(stack2, 's2c1', { type: 'DummyResource', properties: { Prog2: 'Prog2' } });
        const c1 = new MyConstruct(stack2, 's1c2');
        // add some metadata
        stack1.node.addMetadata('meta', 111);
        r2.node.addWarning('warning1');
        r2.node.addWarning('warning2');
        c1.node.addMetadata('meta', { key: 'value' });
        app.node.addMetadata('applevel', 123); // apps can also have metadata
    });
}
function synthStack(name, includeMetadata = false, context) {
    const response = synth(context);
    const stack = response.getStackByName(name);
    if (!includeMetadata) {
        delete stack.metadata;
    }
    return stack;
}
class MyConstruct extends lib_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        new lib_1.CfnResource(this, 'r1', { type: 'ResourceType1' });
        new lib_1.CfnResource(this, 'r2', { type: 'ResourceType2', properties: { FromContext: this.node.tryGetContext('ctx1') } });
    }
}
module.exports = {
    'synthesizes all stacks and returns synthesis result'(test) {
        const response = synth();
        delete response.dir;
        test.deepEqual(response.stacks.length, 2);
        const stack1 = response.stacks[0];
        test.deepEqual(stack1.stackName, 'stack1');
        test.deepEqual(stack1.id, 'stack1');
        test.deepEqual(stack1.environment.account, 12345);
        test.deepEqual(stack1.environment.region, 'us-east-1');
        test.deepEqual(stack1.environment.name, 'aws://12345/us-east-1');
        test.deepEqual(stack1.template, { Resources: { s1c1: { Type: 'DummyResource', Properties: { Prop1: 'Prop1' } },
                s1c2: { Type: 'DummyResource', Properties: { Foo: 123 } } } });
        test.deepEqual(stack1.manifest.metadata, {
            '/stack1': [{ type: 'meta', data: 111 }],
            '/stack1/s1c1': [{ type: 'aws:cdk:logicalId', data: 's1c1' }],
            '/stack1/s1c2': [{ type: 'aws:cdk:logicalId', data: 's1c2' },
                { type: 'aws:cdk:warning', data: 'warning1' },
                { type: 'aws:cdk:warning', data: 'warning2' }],
        });
        const stack2 = response.stacks[1];
        test.deepEqual(stack2.stackName, 'stack2');
        test.deepEqual(stack2.id, 'stack2');
        test.deepEqual(stack2.environment.name, 'aws://unknown-account/unknown-region');
        test.deepEqual(stack2.template, { Resources: { s2c1: { Type: 'DummyResource', Properties: { Prog2: 'Prog2' } },
                s1c2r1D1791C01: { Type: 'ResourceType1' },
                s1c2r25F685FFF: { Type: 'ResourceType2' } } });
        test.deepEqual(stack2.manifest.metadata, {
            '/stack2/s2c1': [{ type: 'aws:cdk:logicalId', data: 's2c1' }],
            '/stack2/s1c2': [{ type: 'meta', data: { key: 'value' } }],
            '/stack2/s1c2/r1': [{ type: 'aws:cdk:logicalId', data: 's1c2r1D1791C01' }],
            '/stack2/s1c2/r2': [{ type: 'aws:cdk:logicalId', data: 's1c2r25F685FFF' }],
        });
        test.done();
    },
    'context can be passed through CDK_CONTEXT'(test) {
        process.env[cxapi.CONTEXT_ENV] = JSON.stringify({
            key1: 'val1',
            key2: 'val2',
        });
        const prog = new app_1.App();
        test.deepEqual(prog.node.tryGetContext('key1'), 'val1');
        test.deepEqual(prog.node.tryGetContext('key2'), 'val2');
        test.done();
    },
    'context passed through CDK_CONTEXT has precedence'(test) {
        process.env[cxapi.CONTEXT_ENV] = JSON.stringify({
            key1: 'val1',
            key2: 'val2',
        });
        const prog = new app_1.App({
            context: {
                key1: 'val3',
                key2: 'val4',
            },
        });
        test.deepEqual(prog.node.tryGetContext('key1'), 'val1');
        test.deepEqual(prog.node.tryGetContext('key2'), 'val2');
        test.done();
    },
    'context from the command line can be used when creating the stack'(test) {
        const output = synthStack('stack2', false, { ctx1: 'HELLO' });
        test.deepEqual(output.template, {
            Resources: {
                s2c1: {
                    Type: 'DummyResource',
                    Properties: {
                        Prog2: 'Prog2',
                    },
                },
                s1c2r1D1791C01: {
                    Type: 'ResourceType1',
                },
                s1c2r25F685FFF: {
                    Type: 'ResourceType2',
                    Properties: {
                        FromContext: 'HELLO',
                    },
                },
            },
        });
        test.done();
    },
    'setContext(k,v) can be used to set context programmatically'(test) {
        const prog = new app_1.App({
            context: {
                foo: 'bar',
            },
        });
        test.deepEqual(prog.node.tryGetContext('foo'), 'bar');
        test.done();
    },
    'setContext(k,v) cannot be called after stacks have been added because stacks may use the context'(test) {
        const prog = new app_1.App();
        new lib_1.Stack(prog, 's1');
        test.throws(() => prog.node.setContext('foo', 'bar'));
        test.done();
    },
    'app.synth() performs validation first and if there are errors, it returns the errors'(test) {
        class Child extends lib_1.Construct {
            validate() {
                return [`Error from ${this.node.id}`];
            }
        }
        class Parent extends lib_1.Stack {
        }
        const app = new app_1.App();
        const parent = new Parent(app, 'Parent');
        new Child(parent, 'C1');
        new Child(parent, 'C2');
        test.throws(() => app.synth(), /Validation failed with the following errors/);
        test.done();
    },
    'app.synthesizeStack(stack) will return a list of missing contextual information'(test) {
        class MyStack extends lib_1.Stack {
            constructor(scope, id, props) {
                super(scope, id, props);
                this.reportMissingContext({
                    key: 'missing-context-key',
                    provider: cloud_assembly_schema_1.ContextProvider.AVAILABILITY_ZONE_PROVIDER,
                    props: {
                        account: '12345689012',
                        region: 'ab-north-1',
                    },
                });
                this.reportMissingContext({
                    key: 'missing-context-key-2',
                    provider: cloud_assembly_schema_1.ContextProvider.AVAILABILITY_ZONE_PROVIDER,
                    props: {
                        account: '12345689012',
                        region: 'ab-south-1',
                    },
                });
            }
        }
        const assembly = withApp({}, app => {
            new MyStack(app, 'MyStack');
        });
        test.deepEqual(assembly.manifest.missing, [
            {
                key: 'missing-context-key',
                provider: cloud_assembly_schema_1.ContextProvider.AVAILABILITY_ZONE_PROVIDER,
                props: {
                    account: '12345689012',
                    region: 'ab-north-1',
                },
            },
            {
                key: 'missing-context-key-2',
                provider: cloud_assembly_schema_1.ContextProvider.AVAILABILITY_ZONE_PROVIDER,
                props: {
                    account: '12345689012',
                    region: 'ab-south-1',
                },
            },
        ]);
        test.done();
    },
    'runtime library versions disabled'(test) {
        const context = {};
        context[cxapi.DISABLE_VERSION_REPORTING] = true;
        const assembly = withApp(context, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        test.deepEqual(assembly.runtime, { libraries: {} });
        test.done();
    },
    'runtime library versions'(test) {
        const response = withApp({ runtimeInfo: true }, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        const libs = (response.runtime && response.runtime.libraries) || {};
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const version = require('../package.json').version;
        test.deepEqual(libs['@aws-cdk/core'], version);
        test.deepEqual(libs['@aws-cdk/cx-api'], version);
        test.deepEqual(libs['jsii-runtime'], `node.js/${process.version}`);
        test.done();
    },
    'jsii-runtime version loaded from JSII_AGENT'(test) {
        process.env.JSII_AGENT = 'Java/1.2.3.4';
        const response = withApp({ runtimeInfo: true }, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        const libs = (response.runtime && response.runtime.libraries) || {};
        test.deepEqual(libs['jsii-runtime'], 'Java/1.2.3.4');
        delete process.env.JSII_AGENT;
        test.done();
    },
    'version reporting includes only @aws-cdk, aws-cdk and jsii libraries'(test) {
        const response = withApp({ runtimeInfo: true }, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        const libs = (response.runtime && response.runtime.libraries) || {};
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const version = require('../package.json').version;
        test.deepEqual(libs, {
            '@aws-cdk/core': version,
            '@aws-cdk/cx-api': version,
            '@aws-cdk/cloud-assembly-schema': version,
            'jsii-runtime': `node.js/${process.version}`,
        });
        test.done();
    },
    'deep stack is shown and synthesized properly'(test) {
        // WHEN
        const response = withApp({}, (app) => {
            const topStack = new lib_1.Stack(app, 'Stack');
            const topResource = new lib_1.CfnResource(topStack, 'Res', { type: 'CDK::TopStack::Resource' });
            const bottomStack = new lib_1.Stack(topResource, 'Stack');
            new lib_1.CfnResource(bottomStack, 'Res', { type: 'CDK::BottomStack::Resource' });
        });
        // THEN
        test.deepEqual(response.stacks.map(s => ({ name: s.stackName, template: s.template })), [
            {
                name: 'Stack',
                template: { Resources: { Res: { Type: 'CDK::TopStack::Resource' } } },
            },
            {
                name: 'StackResStack7E4AFA86',
                template: { Resources: { Res: { Type: 'CDK::BottomStack::Resource' } } },
            },
        ]);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5hcHAuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0LmFwcC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEsMEVBQWlFO0FBQ2pFLHlDQUF5QztBQUV6QyxnQ0FBbUU7QUFDbkUsb0NBQTJDO0FBRTNDLFNBQVMsT0FBTyxDQUFDLEtBQWUsRUFBRSxLQUF5QjtJQUN6RCxNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsQ0FBQztRQUNsQixXQUFXLEVBQUUsS0FBSztRQUNsQixXQUFXLEVBQUUsS0FBSztRQUNsQixHQUFHLEtBQUs7S0FDVCxDQUFDLENBQUM7SUFFSCxLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7SUFFWCxPQUFPLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztBQUNyQixDQUFDO0FBRUQsU0FBUyxLQUFLLENBQUMsT0FBZ0M7SUFDN0MsT0FBTyxPQUFPLENBQUMsRUFBRSxPQUFPLEVBQUUsRUFBRSxHQUFHLENBQUMsRUFBRTtRQUNoQyxNQUFNLE1BQU0sR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsUUFBUSxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQzVGLElBQUksaUJBQVcsQ0FBQyxNQUFNLEVBQUUsTUFBTSxFQUFFLEVBQUUsSUFBSSxFQUFFLGVBQWUsRUFBRSxVQUFVLEVBQUUsRUFBRSxLQUFLLEVBQUUsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQzNGLE1BQU0sRUFBRSxHQUFHLElBQUksaUJBQVcsQ0FBQyxNQUFNLEVBQUUsTUFBTSxFQUFFLEVBQUUsSUFBSSxFQUFFLGVBQWUsRUFBRSxVQUFVLEVBQUUsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWhHLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUN4QyxJQUFJLGlCQUFXLENBQUMsTUFBTSxFQUFFLE1BQU0sRUFBRSxFQUFFLElBQUksRUFBRSxlQUFlLEVBQUUsVUFBVSxFQUFFLEVBQUUsS0FBSyxFQUFFLE9BQU8sRUFBRSxFQUFFLENBQUMsQ0FBQztRQUMzRixNQUFNLEVBQUUsR0FBRyxJQUFJLFdBQVcsQ0FBQyxNQUFNLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFFM0Msb0JBQW9CO1FBQ3BCLE1BQU0sQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLE1BQU0sRUFBRSxHQUFHLENBQUMsQ0FBQztRQUNyQyxFQUFFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUMvQixFQUFFLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUMvQixFQUFFLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxNQUFNLEVBQUUsRUFBRSxHQUFHLEVBQUUsT0FBTyxFQUFFLENBQUMsQ0FBQztRQUM5QyxHQUFHLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxVQUFVLEVBQUUsR0FBRyxDQUFDLENBQUMsQ0FBQyw4QkFBOEI7SUFDdkUsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDO0FBRUQsU0FBUyxVQUFVLENBQUMsSUFBWSxFQUFFLGtCQUEyQixLQUFLLEVBQUUsT0FBYTtJQUMvRSxNQUFNLFFBQVEsR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDaEMsTUFBTSxLQUFLLEdBQUcsUUFBUSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUU1QyxJQUFJLENBQUMsZUFBZSxFQUFFO1FBQ3BCLE9BQVEsS0FBYSxDQUFDLFFBQVEsQ0FBQztLQUNoQztJQUVELE9BQU8sS0FBSyxDQUFDO0FBQ2YsQ0FBQztBQTBSRCxNQUFNLFdBQVksU0FBUSxlQUFTO0lBQ2pDLFlBQVksS0FBZ0IsRUFBRSxFQUFVO1FBQ3RDLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxpQkFBVyxDQUFDLElBQUksRUFBRSxJQUFJLEVBQUUsRUFBRSxJQUFJLEVBQUUsZUFBZSxFQUFFLENBQUMsQ0FBQztRQUN2RCxJQUFJLGlCQUFXLENBQUMsSUFBSSxFQUFFLElBQUksRUFBRSxFQUFFLElBQUksRUFBRSxlQUFlLEVBQUUsVUFBVSxFQUFFLEVBQUUsV0FBVyxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQ3ZILENBQUM7Q0FDRjtBQS9SRCxpQkFBUztJQUNQLHFEQUFxRCxDQUFDLElBQVU7UUFDOUQsTUFBTSxRQUFRLEdBQUcsS0FBSyxFQUFFLENBQUM7UUFDekIsT0FBUSxRQUFnQixDQUFDLEdBQUcsQ0FBQztRQUU3QixJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBRTFDLE1BQU0sTUFBTSxHQUFHLFFBQVEsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDbEMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsU0FBUyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQzNDLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLEVBQUUsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUNwQyxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxXQUFXLENBQUMsT0FBTyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ2xELElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxNQUFNLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDdkQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSx1QkFBdUIsQ0FBQyxDQUFDO1FBQ2pFLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLFFBQVEsRUFBRSxFQUFFLFNBQVMsRUFDekMsRUFBRSxJQUFJLEVBQUUsRUFBRSxJQUFJLEVBQUUsZUFBZSxFQUFFLFVBQVUsRUFBRSxFQUFFLEtBQUssRUFBRSxPQUFPLEVBQUUsRUFBRTtnQkFDL0QsSUFBSSxFQUFFLEVBQUUsSUFBSSxFQUFFLGVBQWUsRUFBRSxVQUFVLEVBQUUsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEVBQUUsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNuRSxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsUUFBUSxFQUFFO1lBQ3ZDLFNBQVMsRUFBRSxDQUFDLEVBQUUsSUFBSSxFQUFFLE1BQU0sRUFBRSxJQUFJLEVBQUUsR0FBRyxFQUFFLENBQUM7WUFDeEMsY0FBYyxFQUFFLENBQUMsRUFBRSxJQUFJLEVBQUUsbUJBQW1CLEVBQUUsSUFBSSxFQUFFLE1BQU0sRUFBRSxDQUFDO1lBQzdELGNBQWMsRUFDWixDQUFDLEVBQUUsSUFBSSxFQUFFLG1CQUFtQixFQUFFLElBQUksRUFBRSxNQUFNLEVBQUU7Z0JBQzFDLEVBQUUsSUFBSSxFQUFFLGlCQUFpQixFQUFFLElBQUksRUFBRSxVQUFVLEVBQUU7Z0JBQzdDLEVBQUUsSUFBSSxFQUFFLGlCQUFpQixFQUFFLElBQUksRUFBRSxVQUFVLEVBQUUsQ0FBQztTQUNuRCxDQUFDLENBQUM7UUFFSCxNQUFNLE1BQU0sR0FBRyxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ2xDLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUMzQyxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxFQUFFLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDcEMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSxzQ0FBc0MsQ0FBQyxDQUFDO1FBQ2hGLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLFFBQVEsRUFBRSxFQUFFLFNBQVMsRUFDekMsRUFBRSxJQUFJLEVBQUUsRUFBRSxJQUFJLEVBQUUsZUFBZSxFQUFFLFVBQVUsRUFBRSxFQUFFLEtBQUssRUFBRSxPQUFPLEVBQUUsRUFBRTtnQkFDL0QsY0FBYyxFQUFFLEVBQUUsSUFBSSxFQUFFLGVBQWUsRUFBRTtnQkFDekMsY0FBYyxFQUFFLEVBQUUsSUFBSSxFQUFFLGVBQWUsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ25ELElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxRQUFRLEVBQUU7WUFDdkMsY0FBYyxFQUFFLENBQUMsRUFBRSxJQUFJLEVBQUUsbUJBQW1CLEVBQUUsSUFBSSxFQUFFLE1BQU0sRUFBRSxDQUFDO1lBQzdELGNBQWMsRUFBRSxDQUFDLEVBQUUsSUFBSSxFQUFFLE1BQU0sRUFBRSxJQUFJLEVBQUUsRUFBRSxHQUFHLEVBQUUsT0FBTyxFQUFFLEVBQUUsQ0FBQztZQUMxRCxpQkFBaUIsRUFDZixDQUFDLEVBQUUsSUFBSSxFQUFFLG1CQUFtQixFQUFFLElBQUksRUFBRSxnQkFBZ0IsRUFBRSxDQUFDO1lBQ3pELGlCQUFpQixFQUNmLENBQUMsRUFBRSxJQUFJLEVBQUUsbUJBQW1CLEVBQUUsSUFBSSxFQUFFLGdCQUFnQixFQUFFLENBQUM7U0FDMUQsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDJDQUEyQyxDQUFDLElBQVU7UUFDcEQsT0FBTyxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUM5QyxJQUFJLEVBQUUsTUFBTTtZQUNaLElBQUksRUFBRSxNQUFNO1NBQ2IsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxJQUFJLEdBQUcsSUFBSSxTQUFHLEVBQUUsQ0FBQztRQUN2QixJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sQ0FBQyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3hELElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxDQUFDLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDeEQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELG1EQUFtRCxDQUFDLElBQVU7UUFDNUQsT0FBTyxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUM5QyxJQUFJLEVBQUUsTUFBTTtZQUNaLElBQUksRUFBRSxNQUFNO1NBQ2IsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxJQUFJLEdBQUcsSUFBSSxTQUFHLENBQUM7WUFDbkIsT0FBTyxFQUFFO2dCQUNQLElBQUksRUFBRSxNQUFNO2dCQUNaLElBQUksRUFBRSxNQUFNO2FBQ2I7U0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLE1BQU0sQ0FBQyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3hELElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsTUFBTSxDQUFDLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDeEQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELG1FQUFtRSxDQUFDLElBQVU7UUFDNUUsTUFBTSxNQUFNLEdBQUcsVUFBVSxDQUFDLFFBQVEsRUFBRSxLQUFLLEVBQUUsRUFBRSxJQUFJLEVBQUUsT0FBTyxFQUFFLENBQUMsQ0FBQztRQUU5RCxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxRQUFRLEVBQUU7WUFDOUIsU0FBUyxFQUFFO2dCQUNULElBQUksRUFBRTtvQkFDSixJQUFJLEVBQUUsZUFBZTtvQkFDckIsVUFBVSxFQUFFO3dCQUNWLEtBQUssRUFBRSxPQUFPO3FCQUNmO2lCQUNGO2dCQUNELGNBQWMsRUFBRTtvQkFDZCxJQUFJLEVBQUUsZUFBZTtpQkFDdEI7Z0JBQ0QsY0FBYyxFQUFFO29CQUNkLElBQUksRUFBRSxlQUFlO29CQUNyQixVQUFVLEVBQUU7d0JBQ1YsV0FBVyxFQUFFLE9BQU87cUJBQ3JCO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNkRBQTZELENBQUMsSUFBVTtRQUN0RSxNQUFNLElBQUksR0FBRyxJQUFJLFNBQUcsQ0FBQztZQUNuQixPQUFPLEVBQUU7Z0JBQ1AsR0FBRyxFQUFFLEtBQUs7YUFDWDtTQUNGLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDdEQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGtHQUFrRyxDQUFDLElBQVU7UUFDM0csTUFBTSxJQUFJLEdBQUcsSUFBSSxTQUFHLEVBQUUsQ0FBQztRQUN2QixJQUFJLFdBQUssQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDdEIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUMsQ0FBQztRQUN0RCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsc0ZBQXNGLENBQUMsSUFBVTtRQUUvRixNQUFNLEtBQU0sU0FBUSxlQUFTO1lBQ2pCLFFBQVE7Z0JBQ2hCLE9BQU8sQ0FBQyxjQUFjLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztZQUN4QyxDQUFDO1NBQ0Y7UUFFRCxNQUFNLE1BQU8sU0FBUSxXQUFLO1NBRXpCO1FBRUQsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLEVBQUUsQ0FBQztRQUV0QixNQUFNLE1BQU0sR0FBRyxJQUFJLE1BQU0sQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDekMsSUFBSSxLQUFLLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQ3hCLElBQUksS0FBSyxDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMsQ0FBQztRQUV4QixJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsRUFBRSw2Q0FBNkMsQ0FBQyxDQUFDO1FBRTlFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxpRkFBaUYsQ0FBQyxJQUFVO1FBQzFGLE1BQU0sT0FBUSxTQUFRLFdBQUs7WUFDekIsWUFBWSxLQUFVLEVBQUUsRUFBVSxFQUFFLEtBQWtCO2dCQUNwRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztnQkFFeEIsSUFBSSxDQUFDLG9CQUFvQixDQUFDO29CQUN4QixHQUFHLEVBQUUscUJBQXFCO29CQUMxQixRQUFRLEVBQUUsdUNBQWUsQ0FBQywwQkFBMEI7b0JBQ3BELEtBQUssRUFBRTt3QkFDTCxPQUFPLEVBQUUsYUFBYTt3QkFDdEIsTUFBTSxFQUFFLFlBQVk7cUJBQ3JCO2lCQUNGLENBQ0EsQ0FBQztnQkFFRixJQUFJLENBQUMsb0JBQW9CLENBQUM7b0JBQ3hCLEdBQUcsRUFBRSx1QkFBdUI7b0JBQzVCLFFBQVEsRUFBRSx1Q0FBZSxDQUFDLDBCQUEwQjtvQkFDcEQsS0FBSyxFQUFFO3dCQUNMLE9BQU8sRUFBRSxhQUFhO3dCQUN0QixNQUFNLEVBQUUsWUFBWTtxQkFDckI7aUJBQ0YsQ0FDQSxDQUFDO1lBQ0osQ0FBQztTQUNGO1FBRUQsTUFBTSxRQUFRLEdBQUcsT0FBTyxDQUFDLEVBQUUsRUFBRSxHQUFHLENBQUMsRUFBRTtZQUNqQyxJQUFJLE9BQU8sQ0FBQyxHQUFHLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDOUIsQ0FBQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsT0FBTyxFQUFFO1lBQ3hDO2dCQUNFLEdBQUcsRUFBRSxxQkFBcUI7Z0JBQzFCLFFBQVEsRUFBRSx1Q0FBZSxDQUFDLDBCQUEwQjtnQkFDcEQsS0FBSyxFQUFFO29CQUNMLE9BQU8sRUFBRSxhQUFhO29CQUN0QixNQUFNLEVBQUUsWUFBWTtpQkFDckI7YUFDRjtZQUNEO2dCQUNFLEdBQUcsRUFBRSx1QkFBdUI7Z0JBQzVCLFFBQVEsRUFBRSx1Q0FBZSxDQUFDLDBCQUEwQjtnQkFDcEQsS0FBSyxFQUFFO29CQUNMLE9BQU8sRUFBRSxhQUFhO29CQUN0QixNQUFNLEVBQUUsWUFBWTtpQkFDckI7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxtQ0FBbUMsQ0FBQyxJQUFVO1FBQzVDLE1BQU0sT0FBTyxHQUFRLEVBQUUsQ0FBQztRQUN4QixPQUFPLENBQUMsS0FBSyxDQUFDLHlCQUF5QixDQUFDLEdBQUcsSUFBSSxDQUFDO1FBRWhELE1BQU0sUUFBUSxHQUFHLE9BQU8sQ0FBQyxPQUFPLEVBQUUsR0FBRyxDQUFDLEVBQUU7WUFDdEMsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1lBQ3ZDLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFLEVBQUUsSUFBSSxFQUFFLGdCQUFnQixFQUFFLENBQUMsQ0FBQztRQUNuRSxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRSxFQUFFLFNBQVMsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ3BELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwwQkFBMEIsQ0FBQyxJQUFVO1FBQ25DLE1BQU0sUUFBUSxHQUFHLE9BQU8sQ0FBQyxFQUFFLFdBQVcsRUFBRSxJQUFJLEVBQUUsRUFBRSxHQUFHLENBQUMsRUFBRTtZQUNwRCxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLENBQUM7WUFDdkMsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUUsRUFBRSxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsQ0FBQyxDQUFDO1FBQ25FLENBQUMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxJQUFJLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxJQUFJLFFBQVEsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxDQUFDO1FBRXBFLGlFQUFpRTtRQUNqRSxNQUFNLE9BQU8sR0FBRyxPQUFPLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxPQUFPLENBQUM7UUFDbkQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDL0MsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUNqRCxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsRUFBRSxXQUFXLE9BQU8sQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO1FBQ25FLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw2Q0FBNkMsQ0FBQyxJQUFVO1FBQ3RELE9BQU8sQ0FBQyxHQUFHLENBQUMsVUFBVSxHQUFHLGNBQWMsQ0FBQztRQUV4QyxNQUFNLFFBQVEsR0FBRyxPQUFPLENBQUMsRUFBRSxXQUFXLEVBQUUsSUFBSSxFQUFFLEVBQUUsR0FBRyxDQUFDLEVBQUU7WUFDcEQsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1lBQ3ZDLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFLEVBQUUsSUFBSSxFQUFFLGdCQUFnQixFQUFFLENBQUMsQ0FBQztRQUNuRSxDQUFDLENBQUMsQ0FBQztRQUVILE1BQU0sSUFBSSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sSUFBSSxRQUFRLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNwRSxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsRUFBRSxjQUFjLENBQUMsQ0FBQztRQUVyRCxPQUFPLE9BQU8sQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDO1FBQzlCLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxzRUFBc0UsQ0FBQyxJQUFVO1FBQy9FLE1BQU0sUUFBUSxHQUFHLE9BQU8sQ0FBQyxFQUFFLFdBQVcsRUFBRSxJQUFJLEVBQUUsRUFBRSxHQUFHLENBQUMsRUFBRTtZQUNwRCxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLENBQUM7WUFDdkMsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUUsRUFBRSxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsQ0FBQyxDQUFDO1FBQ25FLENBQUMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxJQUFJLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxJQUFJLFFBQVEsQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxDQUFDO1FBRXBFLGlFQUFpRTtRQUNqRSxNQUFNLE9BQU8sR0FBRyxPQUFPLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxPQUFPLENBQUM7UUFDbkQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUU7WUFDbkIsZUFBZSxFQUFFLE9BQU87WUFDeEIsaUJBQWlCLEVBQUUsT0FBTztZQUMxQixnQ0FBZ0MsRUFBRSxPQUFPO1lBQ3pDLGNBQWMsRUFBRSxXQUFXLE9BQU8sQ0FBQyxPQUFPLEVBQUU7U0FDN0MsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDhDQUE4QyxDQUFDLElBQVU7UUFDekQsT0FBTztRQUNMLE1BQU0sUUFBUSxHQUFHLE9BQU8sQ0FBQyxFQUFFLEVBQUUsQ0FBQyxHQUFHLEVBQUUsRUFBRTtZQUNuQyxNQUFNLFFBQVEsR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsT0FBTyxDQUFDLENBQUM7WUFDekMsTUFBTSxXQUFXLEdBQUcsSUFBSSxpQkFBVyxDQUFDLFFBQVEsRUFBRSxLQUFLLEVBQUUsRUFBRSxJQUFJLEVBQUUseUJBQXlCLEVBQUUsQ0FBQyxDQUFDO1lBRTFGLE1BQU0sV0FBVyxHQUFHLElBQUksV0FBSyxDQUFDLFdBQVcsRUFBRSxPQUFPLENBQUMsQ0FBQztZQUNwRCxJQUFJLGlCQUFXLENBQUMsV0FBVyxFQUFFLEtBQUssRUFBRSxFQUFFLElBQUksRUFBRSw0QkFBNEIsRUFBRSxDQUFDLENBQUM7UUFDOUUsQ0FBQyxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLFNBQVMsRUFBRSxRQUFRLEVBQUUsQ0FBQyxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUMsRUFBRTtZQUN0RjtnQkFDRSxJQUFJLEVBQUUsT0FBTztnQkFDYixRQUFRLEVBQUUsRUFBRSxTQUFTLEVBQUUsRUFBRSxHQUFHLEVBQUUsRUFBRSxJQUFJLEVBQUUseUJBQXlCLEVBQUUsRUFBRSxFQUFFO2FBQ3RFO1lBQ0Q7Z0JBQ0UsSUFBSSxFQUFFLHVCQUF1QjtnQkFDN0IsUUFBUSxFQUFFLEVBQUUsU0FBUyxFQUFFLEVBQUUsR0FBRyxFQUFFLEVBQUUsSUFBSSxFQUFFLDRCQUE0QixFQUFFLEVBQUUsRUFBRTthQUN6RTtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7Q0FDRixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQ29udGV4dFByb3ZpZGVyIH0gZnJvbSAnQGF3cy1jZGsvY2xvdWQtYXNzZW1ibHktc2NoZW1hJztcbmltcG9ydCAqIGFzIGN4YXBpIGZyb20gJ0Bhd3MtY2RrL2N4LWFwaSc7XG5pbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0IHsgQ2ZuUmVzb3VyY2UsIENvbnN0cnVjdCwgU3RhY2ssIFN0YWNrUHJvcHMgfSBmcm9tICcuLi9saWInO1xuaW1wb3J0IHsgQXBwLCBBcHBQcm9wcyB9IGZyb20gJy4uL2xpYi9hcHAnO1xuXG5mdW5jdGlvbiB3aXRoQXBwKHByb3BzOiBBcHBQcm9wcywgYmxvY2s6IChhcHA6IEFwcCkgPT4gdm9pZCk6IGN4YXBpLkNsb3VkQXNzZW1ibHkge1xuICBjb25zdCBhcHAgPSBuZXcgQXBwKHtcbiAgICBydW50aW1lSW5mbzogZmFsc2UsXG4gICAgc3RhY2tUcmFjZXM6IGZhbHNlLFxuICAgIC4uLnByb3BzLFxuICB9KTtcblxuICBibG9jayhhcHApO1xuXG4gIHJldHVybiBhcHAuc3ludGgoKTtcbn1cblxuZnVuY3Rpb24gc3ludGgoY29udGV4dD86IHsgW2tleTogc3RyaW5nXTogYW55IH0pOiBjeGFwaS5DbG91ZEFzc2VtYmx5IHtcbiAgcmV0dXJuIHdpdGhBcHAoeyBjb250ZXh0IH0sIGFwcCA9PiB7XG4gICAgY29uc3Qgc3RhY2sxID0gbmV3IFN0YWNrKGFwcCwgJ3N0YWNrMScsIHsgZW52OiB7IGFjY291bnQ6ICcxMjM0NScsIHJlZ2lvbjogJ3VzLWVhc3QtMScgfSB9KTtcbiAgICBuZXcgQ2ZuUmVzb3VyY2Uoc3RhY2sxLCAnczFjMScsIHsgdHlwZTogJ0R1bW15UmVzb3VyY2UnLCBwcm9wZXJ0aWVzOiB7IFByb3AxOiAnUHJvcDEnIH0gfSk7XG4gICAgY29uc3QgcjIgPSBuZXcgQ2ZuUmVzb3VyY2Uoc3RhY2sxLCAnczFjMicsIHsgdHlwZTogJ0R1bW15UmVzb3VyY2UnLCBwcm9wZXJ0aWVzOiB7IEZvbzogMTIzIH0gfSk7XG5cbiAgICBjb25zdCBzdGFjazIgPSBuZXcgU3RhY2soYXBwLCAnc3RhY2syJyk7XG4gICAgbmV3IENmblJlc291cmNlKHN0YWNrMiwgJ3MyYzEnLCB7IHR5cGU6ICdEdW1teVJlc291cmNlJywgcHJvcGVydGllczogeyBQcm9nMjogJ1Byb2cyJyB9IH0pO1xuICAgIGNvbnN0IGMxID0gbmV3IE15Q29uc3RydWN0KHN0YWNrMiwgJ3MxYzInKTtcblxuICAgIC8vIGFkZCBzb21lIG1ldGFkYXRhXG4gICAgc3RhY2sxLm5vZGUuYWRkTWV0YWRhdGEoJ21ldGEnLCAxMTEpO1xuICAgIHIyLm5vZGUuYWRkV2FybmluZygnd2FybmluZzEnKTtcbiAgICByMi5ub2RlLmFkZFdhcm5pbmcoJ3dhcm5pbmcyJyk7XG4gICAgYzEubm9kZS5hZGRNZXRhZGF0YSgnbWV0YScsIHsga2V5OiAndmFsdWUnIH0pO1xuICAgIGFwcC5ub2RlLmFkZE1ldGFkYXRhKCdhcHBsZXZlbCcsIDEyMyk7IC8vIGFwcHMgY2FuIGFsc28gaGF2ZSBtZXRhZGF0YVxuICB9KTtcbn1cblxuZnVuY3Rpb24gc3ludGhTdGFjayhuYW1lOiBzdHJpbmcsIGluY2x1ZGVNZXRhZGF0YTogYm9vbGVhbiA9IGZhbHNlLCBjb250ZXh0PzogYW55KTogY3hhcGkuQ2xvdWRGb3JtYXRpb25TdGFja0FydGlmYWN0IHtcbiAgY29uc3QgcmVzcG9uc2UgPSBzeW50aChjb250ZXh0KTtcbiAgY29uc3Qgc3RhY2sgPSByZXNwb25zZS5nZXRTdGFja0J5TmFtZShuYW1lKTtcblxuICBpZiAoIWluY2x1ZGVNZXRhZGF0YSkge1xuICAgIGRlbGV0ZSAoc3RhY2sgYXMgYW55KS5tZXRhZGF0YTtcbiAgfVxuXG4gIHJldHVybiBzdGFjaztcbn1cblxuZXhwb3J0ID0ge1xuICAnc3ludGhlc2l6ZXMgYWxsIHN0YWNrcyBhbmQgcmV0dXJucyBzeW50aGVzaXMgcmVzdWx0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3QgcmVzcG9uc2UgPSBzeW50aCgpO1xuICAgIGRlbGV0ZSAocmVzcG9uc2UgYXMgYW55KS5kaXI7XG5cbiAgICB0ZXN0LmRlZXBFcXVhbChyZXNwb25zZS5zdGFja3MubGVuZ3RoLCAyKTtcblxuICAgIGNvbnN0IHN0YWNrMSA9IHJlc3BvbnNlLnN0YWNrc1swXTtcbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjazEuc3RhY2tOYW1lLCAnc3RhY2sxJyk7XG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sxLmlkLCAnc3RhY2sxJyk7XG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sxLmVudmlyb25tZW50LmFjY291bnQsIDEyMzQ1KTtcbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjazEuZW52aXJvbm1lbnQucmVnaW9uLCAndXMtZWFzdC0xJyk7XG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sxLmVudmlyb25tZW50Lm5hbWUsICdhd3M6Ly8xMjM0NS91cy1lYXN0LTEnKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjazEudGVtcGxhdGUsIHsgUmVzb3VyY2VzOlxuICAgICAgeyBzMWMxOiB7IFR5cGU6ICdEdW1teVJlc291cmNlJywgUHJvcGVydGllczogeyBQcm9wMTogJ1Byb3AxJyB9IH0sXG4gICAgICAgIHMxYzI6IHsgVHlwZTogJ0R1bW15UmVzb3VyY2UnLCBQcm9wZXJ0aWVzOiB7IEZvbzogMTIzIH0gfSB9IH0pO1xuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrMS5tYW5pZmVzdC5tZXRhZGF0YSwge1xuICAgICAgJy9zdGFjazEnOiBbeyB0eXBlOiAnbWV0YScsIGRhdGE6IDExMSB9XSxcbiAgICAgICcvc3RhY2sxL3MxYzEnOiBbeyB0eXBlOiAnYXdzOmNkazpsb2dpY2FsSWQnLCBkYXRhOiAnczFjMScgfV0sXG4gICAgICAnL3N0YWNrMS9zMWMyJzpcbiAgICAgICAgW3sgdHlwZTogJ2F3czpjZGs6bG9naWNhbElkJywgZGF0YTogJ3MxYzInIH0sXG4gICAgICAgICAgeyB0eXBlOiAnYXdzOmNkazp3YXJuaW5nJywgZGF0YTogJ3dhcm5pbmcxJyB9LFxuICAgICAgICAgIHsgdHlwZTogJ2F3czpjZGs6d2FybmluZycsIGRhdGE6ICd3YXJuaW5nMicgfV0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBzdGFjazIgPSByZXNwb25zZS5zdGFja3NbMV07XG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2syLnN0YWNrTmFtZSwgJ3N0YWNrMicpO1xuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrMi5pZCwgJ3N0YWNrMicpO1xuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrMi5lbnZpcm9ubWVudC5uYW1lLCAnYXdzOi8vdW5rbm93bi1hY2NvdW50L3Vua25vd24tcmVnaW9uJyk7XG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2syLnRlbXBsYXRlLCB7IFJlc291cmNlczpcbiAgICAgIHsgczJjMTogeyBUeXBlOiAnRHVtbXlSZXNvdXJjZScsIFByb3BlcnRpZXM6IHsgUHJvZzI6ICdQcm9nMicgfSB9LFxuICAgICAgICBzMWMycjFEMTc5MUMwMTogeyBUeXBlOiAnUmVzb3VyY2VUeXBlMScgfSxcbiAgICAgICAgczFjMnIyNUY2ODVGRkY6IHsgVHlwZTogJ1Jlc291cmNlVHlwZTInIH0gfSB9KTtcbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjazIubWFuaWZlc3QubWV0YWRhdGEsIHtcbiAgICAgICcvc3RhY2syL3MyYzEnOiBbeyB0eXBlOiAnYXdzOmNkazpsb2dpY2FsSWQnLCBkYXRhOiAnczJjMScgfV0sXG4gICAgICAnL3N0YWNrMi9zMWMyJzogW3sgdHlwZTogJ21ldGEnLCBkYXRhOiB7IGtleTogJ3ZhbHVlJyB9IH1dLFxuICAgICAgJy9zdGFjazIvczFjMi9yMSc6XG4gICAgICAgIFt7IHR5cGU6ICdhd3M6Y2RrOmxvZ2ljYWxJZCcsIGRhdGE6ICdzMWMycjFEMTc5MUMwMScgfV0sXG4gICAgICAnL3N0YWNrMi9zMWMyL3IyJzpcbiAgICAgICAgW3sgdHlwZTogJ2F3czpjZGs6bG9naWNhbElkJywgZGF0YTogJ3MxYzJyMjVGNjg1RkZGJyB9XSxcbiAgICB9KTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjb250ZXh0IGNhbiBiZSBwYXNzZWQgdGhyb3VnaCBDREtfQ09OVEVYVCcodGVzdDogVGVzdCkge1xuICAgIHByb2Nlc3MuZW52W2N4YXBpLkNPTlRFWFRfRU5WXSA9IEpTT04uc3RyaW5naWZ5KHtcbiAgICAgIGtleTE6ICd2YWwxJyxcbiAgICAgIGtleTI6ICd2YWwyJyxcbiAgICB9KTtcbiAgICBjb25zdCBwcm9nID0gbmV3IEFwcCgpO1xuICAgIHRlc3QuZGVlcEVxdWFsKHByb2cubm9kZS50cnlHZXRDb250ZXh0KCdrZXkxJyksICd2YWwxJyk7XG4gICAgdGVzdC5kZWVwRXF1YWwocHJvZy5ub2RlLnRyeUdldENvbnRleHQoJ2tleTInKSwgJ3ZhbDInKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnY29udGV4dCBwYXNzZWQgdGhyb3VnaCBDREtfQ09OVEVYVCBoYXMgcHJlY2VkZW5jZScodGVzdDogVGVzdCkge1xuICAgIHByb2Nlc3MuZW52W2N4YXBpLkNPTlRFWFRfRU5WXSA9IEpTT04uc3RyaW5naWZ5KHtcbiAgICAgIGtleTE6ICd2YWwxJyxcbiAgICAgIGtleTI6ICd2YWwyJyxcbiAgICB9KTtcbiAgICBjb25zdCBwcm9nID0gbmV3IEFwcCh7XG4gICAgICBjb250ZXh0OiB7XG4gICAgICAgIGtleTE6ICd2YWwzJyxcbiAgICAgICAga2V5MjogJ3ZhbDQnLFxuICAgICAgfSxcbiAgICB9KTtcbiAgICB0ZXN0LmRlZXBFcXVhbChwcm9nLm5vZGUudHJ5R2V0Q29udGV4dCgna2V5MScpLCAndmFsMScpO1xuICAgIHRlc3QuZGVlcEVxdWFsKHByb2cubm9kZS50cnlHZXRDb250ZXh0KCdrZXkyJyksICd2YWwyJyk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2NvbnRleHQgZnJvbSB0aGUgY29tbWFuZCBsaW5lIGNhbiBiZSB1c2VkIHdoZW4gY3JlYXRpbmcgdGhlIHN0YWNrJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgb3V0cHV0ID0gc3ludGhTdGFjaygnc3RhY2syJywgZmFsc2UsIHsgY3R4MTogJ0hFTExPJyB9KTtcblxuICAgIHRlc3QuZGVlcEVxdWFsKG91dHB1dC50ZW1wbGF0ZSwge1xuICAgICAgUmVzb3VyY2VzOiB7XG4gICAgICAgIHMyYzE6IHtcbiAgICAgICAgICBUeXBlOiAnRHVtbXlSZXNvdXJjZScsXG4gICAgICAgICAgUHJvcGVydGllczoge1xuICAgICAgICAgICAgUHJvZzI6ICdQcm9nMicsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAgczFjMnIxRDE3OTFDMDE6IHtcbiAgICAgICAgICBUeXBlOiAnUmVzb3VyY2VUeXBlMScsXG4gICAgICAgIH0sXG4gICAgICAgIHMxYzJyMjVGNjg1RkZGOiB7XG4gICAgICAgICAgVHlwZTogJ1Jlc291cmNlVHlwZTInLFxuICAgICAgICAgIFByb3BlcnRpZXM6IHtcbiAgICAgICAgICAgIEZyb21Db250ZXh0OiAnSEVMTE8nLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdzZXRDb250ZXh0KGssdikgY2FuIGJlIHVzZWQgdG8gc2V0IGNvbnRleHQgcHJvZ3JhbW1hdGljYWxseScodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHByb2cgPSBuZXcgQXBwKHtcbiAgICAgIGNvbnRleHQ6IHtcbiAgICAgICAgZm9vOiAnYmFyJyxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgdGVzdC5kZWVwRXF1YWwocHJvZy5ub2RlLnRyeUdldENvbnRleHQoJ2ZvbycpLCAnYmFyJyk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3NldENvbnRleHQoayx2KSBjYW5ub3QgYmUgY2FsbGVkIGFmdGVyIHN0YWNrcyBoYXZlIGJlZW4gYWRkZWQgYmVjYXVzZSBzdGFja3MgbWF5IHVzZSB0aGUgY29udGV4dCcodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHByb2cgPSBuZXcgQXBwKCk7XG4gICAgbmV3IFN0YWNrKHByb2csICdzMScpO1xuICAgIHRlc3QudGhyb3dzKCgpID0+IHByb2cubm9kZS5zZXRDb250ZXh0KCdmb28nLCAnYmFyJykpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdhcHAuc3ludGgoKSBwZXJmb3JtcyB2YWxpZGF0aW9uIGZpcnN0IGFuZCBpZiB0aGVyZSBhcmUgZXJyb3JzLCBpdCByZXR1cm5zIHRoZSBlcnJvcnMnKHRlc3Q6IFRlc3QpIHtcblxuICAgIGNsYXNzIENoaWxkIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgICAgIHByb3RlY3RlZCB2YWxpZGF0ZSgpIHtcbiAgICAgICAgcmV0dXJuIFtgRXJyb3IgZnJvbSAke3RoaXMubm9kZS5pZH1gXTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBjbGFzcyBQYXJlbnQgZXh0ZW5kcyBTdGFjayB7XG5cbiAgICB9XG5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG5cbiAgICBjb25zdCBwYXJlbnQgPSBuZXcgUGFyZW50KGFwcCwgJ1BhcmVudCcpO1xuICAgIG5ldyBDaGlsZChwYXJlbnQsICdDMScpO1xuICAgIG5ldyBDaGlsZChwYXJlbnQsICdDMicpO1xuXG4gICAgdGVzdC50aHJvd3MoKCkgPT4gYXBwLnN5bnRoKCksIC9WYWxpZGF0aW9uIGZhaWxlZCB3aXRoIHRoZSBmb2xsb3dpbmcgZXJyb3JzLyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnYXBwLnN5bnRoZXNpemVTdGFjayhzdGFjaykgd2lsbCByZXR1cm4gYSBsaXN0IG9mIG1pc3NpbmcgY29udGV4dHVhbCBpbmZvcm1hdGlvbicodGVzdDogVGVzdCkge1xuICAgIGNsYXNzIE15U3RhY2sgZXh0ZW5kcyBTdGFjayB7XG4gICAgICBjb25zdHJ1Y3RvcihzY29wZTogQXBwLCBpZDogc3RyaW5nLCBwcm9wcz86IFN0YWNrUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG5cbiAgICAgICAgdGhpcy5yZXBvcnRNaXNzaW5nQ29udGV4dCh7XG4gICAgICAgICAga2V5OiAnbWlzc2luZy1jb250ZXh0LWtleScsXG4gICAgICAgICAgcHJvdmlkZXI6IENvbnRleHRQcm92aWRlci5BVkFJTEFCSUxJVFlfWk9ORV9QUk9WSURFUixcbiAgICAgICAgICBwcm9wczoge1xuICAgICAgICAgICAgYWNjb3VudDogJzEyMzQ1Njg5MDEyJyxcbiAgICAgICAgICAgIHJlZ2lvbjogJ2FiLW5vcnRoLTEnLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgICk7XG5cbiAgICAgICAgdGhpcy5yZXBvcnRNaXNzaW5nQ29udGV4dCh7XG4gICAgICAgICAga2V5OiAnbWlzc2luZy1jb250ZXh0LWtleS0yJyxcbiAgICAgICAgICBwcm92aWRlcjogQ29udGV4dFByb3ZpZGVyLkFWQUlMQUJJTElUWV9aT05FX1BST1ZJREVSLFxuICAgICAgICAgIHByb3BzOiB7XG4gICAgICAgICAgICBhY2NvdW50OiAnMTIzNDU2ODkwMTInLFxuICAgICAgICAgICAgcmVnaW9uOiAnYWItc291dGgtMScsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgICAgKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBjb25zdCBhc3NlbWJseSA9IHdpdGhBcHAoe30sIGFwcCA9PiB7XG4gICAgICBuZXcgTXlTdGFjayhhcHAsICdNeVN0YWNrJyk7XG4gICAgfSk7XG5cbiAgICB0ZXN0LmRlZXBFcXVhbChhc3NlbWJseS5tYW5pZmVzdC5taXNzaW5nLCBbXG4gICAgICB7XG4gICAgICAgIGtleTogJ21pc3NpbmctY29udGV4dC1rZXknLFxuICAgICAgICBwcm92aWRlcjogQ29udGV4dFByb3ZpZGVyLkFWQUlMQUJJTElUWV9aT05FX1BST1ZJREVSLFxuICAgICAgICBwcm9wczoge1xuICAgICAgICAgIGFjY291bnQ6ICcxMjM0NTY4OTAxMicsXG4gICAgICAgICAgcmVnaW9uOiAnYWItbm9ydGgtMScsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgICAge1xuICAgICAgICBrZXk6ICdtaXNzaW5nLWNvbnRleHQta2V5LTInLFxuICAgICAgICBwcm92aWRlcjogQ29udGV4dFByb3ZpZGVyLkFWQUlMQUJJTElUWV9aT05FX1BST1ZJREVSLFxuICAgICAgICBwcm9wczoge1xuICAgICAgICAgIGFjY291bnQ6ICcxMjM0NTY4OTAxMicsXG4gICAgICAgICAgcmVnaW9uOiAnYWItc291dGgtMScsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIF0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3J1bnRpbWUgbGlicmFyeSB2ZXJzaW9ucyBkaXNhYmxlZCcodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IGNvbnRleHQ6IGFueSA9IHt9O1xuICAgIGNvbnRleHRbY3hhcGkuRElTQUJMRV9WRVJTSU9OX1JFUE9SVElOR10gPSB0cnVlO1xuXG4gICAgY29uc3QgYXNzZW1ibHkgPSB3aXRoQXBwKGNvbnRleHQsIGFwcCA9PiB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjayhhcHAsICdzdGFjazEnKTtcbiAgICAgIG5ldyBDZm5SZXNvdXJjZShzdGFjaywgJ015UmVzb3VyY2UnLCB7IHR5cGU6ICdSZXNvdXJjZTo6VHlwZScgfSk7XG4gICAgfSk7XG5cbiAgICB0ZXN0LmRlZXBFcXVhbChhc3NlbWJseS5ydW50aW1lLCB7IGxpYnJhcmllczoge30gfSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3J1bnRpbWUgbGlicmFyeSB2ZXJzaW9ucycodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHJlc3BvbnNlID0gd2l0aEFwcCh7IHJ1bnRpbWVJbmZvOiB0cnVlIH0sIGFwcCA9PiB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjayhhcHAsICdzdGFjazEnKTtcbiAgICAgIG5ldyBDZm5SZXNvdXJjZShzdGFjaywgJ015UmVzb3VyY2UnLCB7IHR5cGU6ICdSZXNvdXJjZTo6VHlwZScgfSk7XG4gICAgfSk7XG5cbiAgICBjb25zdCBsaWJzID0gKHJlc3BvbnNlLnJ1bnRpbWUgJiYgcmVzcG9uc2UucnVudGltZS5saWJyYXJpZXMpIHx8IHt9O1xuXG4gICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIEB0eXBlc2NyaXB0LWVzbGludC9uby1yZXF1aXJlLWltcG9ydHNcbiAgICBjb25zdCB2ZXJzaW9uID0gcmVxdWlyZSgnLi4vcGFja2FnZS5qc29uJykudmVyc2lvbjtcbiAgICB0ZXN0LmRlZXBFcXVhbChsaWJzWydAYXdzLWNkay9jb3JlJ10sIHZlcnNpb24pO1xuICAgIHRlc3QuZGVlcEVxdWFsKGxpYnNbJ0Bhd3MtY2RrL2N4LWFwaSddLCB2ZXJzaW9uKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChsaWJzWydqc2lpLXJ1bnRpbWUnXSwgYG5vZGUuanMvJHtwcm9jZXNzLnZlcnNpb259YCk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2pzaWktcnVudGltZSB2ZXJzaW9uIGxvYWRlZCBmcm9tIEpTSUlfQUdFTlQnKHRlc3Q6IFRlc3QpIHtcbiAgICBwcm9jZXNzLmVudi5KU0lJX0FHRU5UID0gJ0phdmEvMS4yLjMuNCc7XG5cbiAgICBjb25zdCByZXNwb25zZSA9IHdpdGhBcHAoeyBydW50aW1lSW5mbzogdHJ1ZSB9LCBhcHAgPT4ge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnc3RhY2sxJyk7XG4gICAgICBuZXcgQ2ZuUmVzb3VyY2Uoc3RhY2ssICdNeVJlc291cmNlJywgeyB0eXBlOiAnUmVzb3VyY2U6OlR5cGUnIH0pO1xuICAgIH0pO1xuXG4gICAgY29uc3QgbGlicyA9IChyZXNwb25zZS5ydW50aW1lICYmIHJlc3BvbnNlLnJ1bnRpbWUubGlicmFyaWVzKSB8fCB7fTtcbiAgICB0ZXN0LmRlZXBFcXVhbChsaWJzWydqc2lpLXJ1bnRpbWUnXSwgJ0phdmEvMS4yLjMuNCcpO1xuXG4gICAgZGVsZXRlIHByb2Nlc3MuZW52LkpTSUlfQUdFTlQ7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3ZlcnNpb24gcmVwb3J0aW5nIGluY2x1ZGVzIG9ubHkgQGF3cy1jZGssIGF3cy1jZGsgYW5kIGpzaWkgbGlicmFyaWVzJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3QgcmVzcG9uc2UgPSB3aXRoQXBwKHsgcnVudGltZUluZm86IHRydWUgfSwgYXBwID0+IHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ3N0YWNrMScpO1xuICAgICAgbmV3IENmblJlc291cmNlKHN0YWNrLCAnTXlSZXNvdXJjZScsIHsgdHlwZTogJ1Jlc291cmNlOjpUeXBlJyB9KTtcbiAgICB9KTtcblxuICAgIGNvbnN0IGxpYnMgPSAocmVzcG9uc2UucnVudGltZSAmJiByZXNwb25zZS5ydW50aW1lLmxpYnJhcmllcykgfHwge307XG5cbiAgICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgQHR5cGVzY3JpcHQtZXNsaW50L25vLXJlcXVpcmUtaW1wb3J0c1xuICAgIGNvbnN0IHZlcnNpb24gPSByZXF1aXJlKCcuLi9wYWNrYWdlLmpzb24nKS52ZXJzaW9uO1xuICAgIHRlc3QuZGVlcEVxdWFsKGxpYnMsIHtcbiAgICAgICdAYXdzLWNkay9jb3JlJzogdmVyc2lvbixcbiAgICAgICdAYXdzLWNkay9jeC1hcGknOiB2ZXJzaW9uLFxuICAgICAgJ0Bhd3MtY2RrL2Nsb3VkLWFzc2VtYmx5LXNjaGVtYSc6IHZlcnNpb24sXG4gICAgICAnanNpaS1ydW50aW1lJzogYG5vZGUuanMvJHtwcm9jZXNzLnZlcnNpb259YCxcbiAgICB9KTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdkZWVwIHN0YWNrIGlzIHNob3duIGFuZCBzeW50aGVzaXplZCBwcm9wZXJseScodGVzdDogVGVzdCkge1xuICAvLyBXSEVOXG4gICAgY29uc3QgcmVzcG9uc2UgPSB3aXRoQXBwKHt9LCAoYXBwKSA9PiB7XG4gICAgICBjb25zdCB0b3BTdGFjayA9IG5ldyBTdGFjayhhcHAsICdTdGFjaycpO1xuICAgICAgY29uc3QgdG9wUmVzb3VyY2UgPSBuZXcgQ2ZuUmVzb3VyY2UodG9wU3RhY2ssICdSZXMnLCB7IHR5cGU6ICdDREs6OlRvcFN0YWNrOjpSZXNvdXJjZScgfSk7XG5cbiAgICAgIGNvbnN0IGJvdHRvbVN0YWNrID0gbmV3IFN0YWNrKHRvcFJlc291cmNlLCAnU3RhY2snKTtcbiAgICAgIG5ldyBDZm5SZXNvdXJjZShib3R0b21TdGFjaywgJ1JlcycsIHsgdHlwZTogJ0NESzo6Qm90dG9tU3RhY2s6OlJlc291cmNlJyB9KTtcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LmRlZXBFcXVhbChyZXNwb25zZS5zdGFja3MubWFwKHMgPT4gKHsgbmFtZTogcy5zdGFja05hbWUsIHRlbXBsYXRlOiBzLnRlbXBsYXRlIH0pKSwgW1xuICAgICAge1xuICAgICAgICBuYW1lOiAnU3RhY2snLFxuICAgICAgICB0ZW1wbGF0ZTogeyBSZXNvdXJjZXM6IHsgUmVzOiB7IFR5cGU6ICdDREs6OlRvcFN0YWNrOjpSZXNvdXJjZScgfSB9IH0sXG4gICAgICB9LFxuICAgICAge1xuICAgICAgICBuYW1lOiAnU3RhY2tSZXNTdGFjazdFNEFGQTg2JyxcbiAgICAgICAgdGVtcGxhdGU6IHsgUmVzb3VyY2VzOiB7IFJlczogeyBUeXBlOiAnQ0RLOjpCb3R0b21TdGFjazo6UmVzb3VyY2UnIH0gfSB9LFxuICAgICAgfSxcbiAgICBdKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxufTtcblxuY2xhc3MgTXlDb25zdHJ1Y3QgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIG5ldyBDZm5SZXNvdXJjZSh0aGlzLCAncjEnLCB7IHR5cGU6ICdSZXNvdXJjZVR5cGUxJyB9KTtcbiAgICBuZXcgQ2ZuUmVzb3VyY2UodGhpcywgJ3IyJywgeyB0eXBlOiAnUmVzb3VyY2VUeXBlMicsIHByb3BlcnRpZXM6IHsgRnJvbUNvbnRleHQ6IHRoaXMubm9kZS50cnlHZXRDb250ZXh0KCdjdHgxJykgfSB9KTtcbiAgfVxufVxuIl19