# Copyright (c) 2025 Beijing Volcano Engine Technology Co., Ltd. and/or its affiliates.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import asyncio

from veadk.utils.logger import get_logger

logger = get_logger(__name__)


def patch_asyncio():
    """Patch asyncio to ignore 'Event loop is closed' error.

    After invoking MCPToolset, we met the `RuntimeError: Event loop is closed` error. Related issue see:
    - https://github.com/google/adk-python/issues/1429
    - https://github.com/google/adk-python/pull/1420
    """
    original_del = asyncio.base_subprocess.BaseSubprocessTransport.__del__

    def patched_del(self):
        try:
            original_del(self)
        except RuntimeError as e:
            if "Event loop is closed" not in str(e):
                raise

    asyncio.base_subprocess.BaseSubprocessTransport.__del__ = patched_del

    from anyio._backends._asyncio import CancelScope

    original_cancel_scope_exit = CancelScope.__exit__

    def patched_cancel_scope_exit(self, exc_type, exc_val, exc_tb):
        try:
            return original_cancel_scope_exit(self, exc_type, exc_val, exc_tb)
        except RuntimeError as e:
            if (
                "Attempted to exit cancel scope in a different task than it was entered in"
                in str(e)
            ):
                return False
            raise

    CancelScope.__exit__ = patched_cancel_scope_exit


# def enable_veadk_tracing(tracing_func):
#     """Enable Veadk tracing by add after_agent_callback to Google ADK Agent.

#     Args:
#         tracing_func: The tracing function to be called after the Runner.run_async is done.
#     """
#     import inspect
#     from functools import wraps

#     def async_generator_decorator(generator_func):
#         sig = inspect.signature(generator_func)

#         @wraps(generator_func)
#         async def wrapper(*args, **kwargs):
#             bound_args = sig.bind_partial(*args, **kwargs)
#             user_id = bound_args.arguments.get("user_id", None)
#             session_id = bound_args.arguments.get("session_id", None)
#             try:
#                 async for item in generator_func(*args, **kwargs):
#                     yield item
#             finally:
#                 tracing_func(user_id, session_id)

#         return wrapper

#     from google.adk.runners import Runner

#     Runner.run_async = async_generator_decorator(Runner.run_async)
