import type { IFileSystem } from "aws-cdk-lib/aws-efs";
import type { IBucket } from "aws-cdk-lib/aws-s3";
import type { IFirewallRules } from "./firewall";
import { IStarterAddOn, Starter } from "./starter";
/**
 * Properties for starter add-ons that add a single script.
 */
export interface SinglePriorityProps {
    /**
     * The priority for the script added by this add-on.
     *
     * @default - 10
     */
    readonly priority?: number;
}
/**
 * An add-on that configures an on-instance firewall.
 */
export declare class InstanceFirewallAddOn implements IStarterAddOn {
    private readonly rules;
    private readonly props?;
    /**
     * An add-on that configures an on-instance firewall.
     *
     * @param rules - The instance firewall rules
     * @param props - Optional configuration properties
     */
    constructor(rules: IFirewallRules, props?: SinglePriorityProps | undefined);
    configure(starter: Starter): void;
}
/**
 * An add-on that synchronizes files from S3 to directories on the instance.
 *
 * This add-on also grants read access to the bucket.
 */
export declare class BucketSyncAddOn implements IStarterAddOn {
    private readonly bucket;
    private readonly destinations;
    private readonly props?;
    /**
     * An add-on that synchronizes files from S3 to directories on the instance.
     *
     * This add-on also grants read access to the bucket.
     *
     * @param bucket - The S3 bucket from which files can be downloaded
     * @param destinations - An object where keys are S3 object key prefixes and values are filesystem directories
     * @param props - Optional configuration properties
     */
    constructor(bucket: IBucket, destinations: Record<string, string>, props?: SinglePriorityProps | undefined);
    configure(starter: Starter): void;
}
/**
 * Constructor properties for ElasticFileSystemAddOn.
 */
export interface ElasticFileSystemAddOnProps extends SinglePriorityProps {
    /**
     * The intended Linux username or ID of the owner of the mounted directory.
     *
     * @default - No chown command is executed
     */
    readonly chown?: string;
    /**
     * The intended Linux group name or ID of the group of the mounted directory.
     *
     * @default - No chrp command is executed
     */
    readonly chgrp?: string;
    /**
     * The intended file mode of the mounted directory.
     *
     * @default - No chmod command is executed
     */
    readonly chmod?: number;
}
/**
 * An add-on that configures a mount point for an EFS filesystem.
 *
 * This add-on will produce a startup script to:
 * - Create the mount directory
 * - Mount the NFS filesystem to the mount point
 * - Optionally change the mode or ownership of the mount point
 *
 * This visitor also configures the Security Groups on both ends.
 */
export declare class ElasticFileSystemAddOn implements IStarterAddOn {
    private readonly filesystem;
    private readonly destination;
    private readonly props?;
    /**
     * An add-on that configures a mount point for an EFS filesystem.
     *
     * This add-on will produce a startup script to:
     * - Create the mount directory
     * - Mount the NFS filesystem to the mount point
     * - Optionally change the mode or ownership of the mount point
     *
     * This visitor also configures the Security Groups on both ends.
     *
     * @param filesystem - The elastic filesystem to mount
     * @param destination - The directory to use as the mount point
     * @param props - Optional configuration properties
     */
    constructor(filesystem: IFileSystem, destination: string, props?: ElasticFileSystemAddOnProps | undefined);
    configure(starter: Starter): void;
}
