"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CidrContext = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const constructs_1 = require("constructs");
const assign_on_launch_1 = require("./assign-on-launch");
const util_1 = require("../util");
/**
 * Allocates IPv6 CIDRs and routes for subnets in a VPC.
 *
 * @see {@link https://github.com/aws/aws-cdk/issues/894}
 * @see {@link https://github.com/aws/aws-cdk/issues/5927}
 */
class CidrContext extends constructs_1.Construct {
    /**
     * Creates a new BetterVpc.
     *
     * @param scope - The construct scope.
     * @param id - The construct ID.
     * @param options - The constructor options.
     */
    constructor(scope, id, options) {
        super(scope, id);
        this.assignAddressOnLaunch = options.assignAddressOnLaunch !== false;
        const vpc = (0, util_1.assertType)(options.vpc, aws_ec2_1.Vpc);
        this.vpc = vpc;
        // Create the VPC CIDR block.
        const cidrBlock = new aws_ec2_1.CfnVPCCidrBlock(this, "Ipv6CidrBlock", {
            vpcId: vpc.vpcId,
            amazonProvidedIpv6CidrBlock: options.addressPool === undefined,
            ipv6Pool: options.addressPool,
            ipv6CidrBlock: options.cidrBlock,
        });
        // Create a placeholder for CloudFormation to provide a CIDR array.
        const cidrs = aws_cdk_lib_1.Fn.cidr(aws_cdk_lib_1.Fn.select(0, vpc.vpcIpv6CidrBlocks), this.validateCidrCount(vpc, options.cidrCount), "64");
        if (vpc.publicSubnets.length > 0) {
            this.assignPublicSubnetCidrs(vpc, cidrs, cidrBlock);
        }
        if (vpc.privateSubnets.length > 0) {
            this.assignPrivateSubnetCidrs(vpc, cidrs, cidrBlock);
        }
        if (vpc.isolatedSubnets.length > 0) {
            this.assignIsolatedSubnetCidrs(vpc, cidrs, cidrBlock);
        }
        if (this.assignAddressOnLaunch) {
            const vpcSubnets = {
                subnets: [
                    ...vpc.publicSubnets,
                    ...vpc.privateSubnets,
                    ...vpc.isolatedSubnets,
                ],
            };
            new assign_on_launch_1.AssignOnLaunch(this, "AssignOnLaunch", { vpc, vpcSubnets });
        }
    }
    /**
     * Figure out the minimun required CIDR subnets and the number desired.
     *
     * @param vpc - The VPC.
     * @param cidrCount - Optional. Divide the VPC CIDR into this many subsets.
     */
    validateCidrCount(vpc, cidrCount) {
        const leastCidrBlocksNeeded = vpc.publicSubnets.length +
            vpc.privateSubnets.length +
            vpc.isolatedSubnets.length;
        const cidrBlockCount = cidrCount ?? leastCidrBlocksNeeded;
        if (cidrBlockCount < leastCidrBlocksNeeded) {
            throw new Error(`You must create at least ${leastCidrBlocksNeeded} CIDR blocks in this VPC`);
        }
        return cidrBlockCount;
    }
    /**
     * Override the template; set the IPv6 CIDR for isolated subnets.
     *
     * @param vpc - The VPC of the subnets.
     * @param cidrs - The possible IPv6 CIDRs to assign.
     * @param cidrBlock - The CfnVPCCidrBlock the subnets depend on.
     */
    assignPublicSubnetCidrs(vpc, cidrs, cidrBlock) {
        const gateway = (0, util_1.findInstanceOf)(vpc.node.children, aws_ec2_1.CfnInternetGateway);
        for (const [index, subnet] of vpc.publicSubnets.entries()) {
            subnet.node.addDependency(cidrBlock);
            // Add a Route to the Internet gateway for Internet-bound IPv6 traffic.
            const concreteSubnet = (0, util_1.assertType)(subnet, aws_ec2_1.Subnet);
            concreteSubnet.addRoute("DefaultIpv6Route", {
                routerId: gateway.ref,
                routerType: aws_ec2_1.RouterType.GATEWAY,
                destinationIpv6CidrBlock: "::/0",
            });
            // Define the IPv6 CIDR block for this subnet.
            const cfnSubnet = (0, util_1.assertType)(subnet.node.defaultChild, aws_ec2_1.CfnSubnet);
            cfnSubnet.ipv6CidrBlock = aws_cdk_lib_1.Fn.select(index, cidrs);
        }
    }
    /**
     * Override the template; set the IPv6 CIDR for private subnets.
     *
     * @param vpc - The VPC of the subnets.
     * @param cidrs - The possible IPv6 CIDRs to assign.
     * @param cidrBlock - The CfnVPCCidrBlock the subnets depend on.
     */
    assignPrivateSubnetCidrs(vpc, cidrs, cidrBlock) {
        const gateway = new aws_ec2_1.CfnEgressOnlyInternetGateway(this, "EgressOnlyIgw", {
            vpcId: vpc.vpcId,
        });
        for (const [index, subnet] of vpc.privateSubnets.entries()) {
            subnet.node.addDependency(cidrBlock);
            // Add a Route to the Egress-only Internet Gateway.
            const concreteSubnet = (0, util_1.assertType)(subnet, aws_ec2_1.Subnet);
            concreteSubnet.addRoute("DefaultIpv6Route", {
                routerId: `${gateway.ref}`,
                routerType: aws_ec2_1.RouterType.EGRESS_ONLY_INTERNET_GATEWAY,
                destinationIpv6CidrBlock: "::/0",
            });
            // Define the IPv6 CIDR block for this subnet.
            const cfnSubnet = (0, util_1.assertType)(subnet.node.defaultChild, aws_ec2_1.CfnSubnet);
            cfnSubnet.ipv6CidrBlock = aws_cdk_lib_1.Fn.select(index + vpc.publicSubnets.length, cidrs);
        }
    }
    /**
     * Override the template; set the IPv6 CIDR for isolated subnets.
     *
     * @param vpc - The VPC of the subnets.
     * @param cidrs - The possible IPv6 CIDRs to assign.
     * @param cidrBlock - The CfnVPCCidrBlock the subnets depend on.
     */
    assignIsolatedSubnetCidrs(vpc, cidrs, cidrBlock) {
        for (const [index, subnet] of vpc.isolatedSubnets.entries()) {
            subnet.node.addDependency(cidrBlock);
            // Define the IPv6 CIDR block for this subnet.
            const cfnSubnet = (0, util_1.assertType)(subnet.node.defaultChild, aws_ec2_1.CfnSubnet);
            cfnSubnet.ipv6CidrBlock = aws_cdk_lib_1.Fn.select(index + vpc.publicSubnets.length + vpc.privateSubnets.length, cidrs);
        }
    }
}
exports.CidrContext = CidrContext;
_a = JSII_RTTI_SYMBOL_1;
CidrContext[_a] = { fqn: "shady-island.CidrContext", version: "0.1.46" };
//# sourceMappingURL=data:application/json;base64,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