"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FargateTask = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_ecs_1 = require("aws-cdk-lib/aws-ecs");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_stepfunctions_tasks_1 = require("aws-cdk-lib/aws-stepfunctions-tasks");
const constructs_1 = require("constructs");
/**
 * An ECS Fargate Task.
 *
 * If `vpcSubnets` is blank but `assignPublicIp` is set, the task will launch
 * in Public subnets, otherwise the first available one of Private, Isolated,
 * Public, in that order.
 */
class FargateTask extends constructs_1.Construct {
    /**
     * Creates a new FargateTask.
     */
    constructor(scope, id, props) {
        super(scope, id);
        let { securityGroups = [], assignPublicIp = false, vpcSubnets } = props;
        this.cluster = props.cluster;
        this.taskDefinition = props.taskDefinition;
        this.assignPublicIp = assignPublicIp;
        if (vpcSubnets === undefined) {
            vpcSubnets = assignPublicIp ? { subnetType: aws_ec2_1.SubnetType.PUBLIC } : {};
        }
        this.subnetSelection = vpcSubnets;
        const vpc = this.cluster.vpc;
        if (securityGroups.length === 0) {
            securityGroups = [new aws_ec2_1.SecurityGroup(this, "SecurityGroup", { vpc })];
        }
        this.connections = new aws_ec2_1.Connections({ securityGroups });
        this.awsVpcNetworkConfig = {
            assignPublicIp: assignPublicIp ? "ENABLED" : "DISABLED",
            subnets: this.cluster.vpc.selectSubnets(vpcSubnets).subnetIds,
            securityGroups: this.connections.securityGroups.map((sg) => sg.securityGroupId),
        };
    }
    grantRun(grantee) {
        const grant = this.taskDefinition.grantRun(grantee);
        const statement = grant.principalStatements.find((ps) => ps.actions.includes("ecs:RunTask"));
        if (statement) {
            statement.addCondition("ArnEquals", {
                "ecs:cluster": this.cluster.clusterArn,
            });
        }
        return grant;
    }
    createRuleTarget(props) {
        return new aws_events_targets_1.EcsTask({
            ...props,
            platformVersion: aws_ecs_1.FargatePlatformVersion.LATEST,
            cluster: this.cluster,
            taskDefinition: this.taskDefinition,
            assignPublicIp: this.assignPublicIp,
            subnetSelection: this.subnetSelection,
            securityGroups: this.connections.securityGroups,
        });
    }
    createStateMachineTask(id, props) {
        return new aws_stepfunctions_tasks_1.EcsRunTask(this, id, {
            ...props,
            launchTarget: new aws_stepfunctions_tasks_1.EcsFargateLaunchTarget({
                platformVersion: aws_ecs_1.FargatePlatformVersion.LATEST,
            }),
            cluster: this.cluster,
            taskDefinition: this.taskDefinition,
            assignPublicIp: this.assignPublicIp,
            subnets: this.subnetSelection,
            securityGroups: this.connections.securityGroups,
        });
    }
}
exports.FargateTask = FargateTask;
_a = JSII_RTTI_SYMBOL_1;
FargateTask[_a] = { fqn: "shady-island.FargateTask", version: "0.1.46" };
//# sourceMappingURL=data:application/json;base64,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