from freezegun import freeze_time
from .mock_service import BeginWorkerMock

APP_ID = 1
LICENSE_KEY = 10

@freeze_time("2021-05-16")
def test_learn_from_data():
    bw = BeginWorkerMock(APP_ID, LICENSE_KEY)
    bw.set_data({
        "user":{
            "111":{
                "dateofbirth":"09-12-1989",
                "userlocation":{
                    "latitude":36.8507689,
                    "longitude":-76.2858726
                },
                "numberfield": 10,
                "textfield": "Hello!"
            },
            "222": {
                "dateofbirth":"09-12-1990",
                "userlocation":{
                    "latitude":36.8507689,
                    "longitude":-76.2858726
                },
                "numberfield": 120,
                "textfield": "Hellooooooo!"
            }
        },
        "interactions":{
            "111": {
                "product":{
                    "10":[
                        "like",
                        "comment",
                        "like"
                    ],
                    "20":[
                        "dislike"
                    ],
                    "30":[
                        "comment"
                    ]

                }
            }
        },
        "product":{
            "10":{
                "description":"....the description...",
                "randomnumber": 100.0,
                "publisheddate": "09-12-1999"
            },
            "20":{
                "description":"hi!",
                "randomnumber": 100.0,
                "publisheddate": "09-12-1991"
            },
            "30":{
                "description":"hi!!",
                "randomnumber": 100.0,
                "publisheddate": "09-12-1991"
            }
        }
    })

    bw.learn_from_data()

    result = bw.get_embeddings()

    expected = {
        "user":{
            "111":{
                'embedding': [ 31.0, 4.0, 1.0, 6.0 ],
                'labels': []
            },
            "222": {
                'embedding': [ 30.0, 4.0, 8.0, 12.0 ],
                'labels': []
            }
        },
        "interactions":{
            "111": {
                "product": {
                    "10": {
                        "sent_bin": 2,
                        "sentiment": 5,
                        "label": "POSITIVE"
                    },
                    '20': {
                        "sent_bin": 1,
                        "sentiment": 1,
                        "label": "NEGATIVE"
                    },
                    '30': {
                        "sent_bin": 1,
                        "sentiment": 4,
                        "label": "NEUTRAL"
                    },
                }
            }
        },
        "product":{
            "10": {
                'embedding': [ 22.0, 7.0, 21.0 ],
                'labels': []
            },
            '20':{
                'embedding': [ 3.0, 7.0, 29.0 ],
                'labels': []
            },
            '30':{
                'embedding': [ 4.0, 7.0, 29.0 ],
                'labels': []
            }
        }
    }
    assert expected == result

@freeze_time("2021-05-16")
def test_learn_from_data_with_different_types_of_interaction():
    bw = BeginWorkerMock(APP_ID, LICENSE_KEY)
    bw.set_data({
        "user":{
            "111":{
                "dateofbirth":"09-12-1989",
                "userlocation":{
                    "latitude":36.8507689,
                    "longitude":-76.2858726
                },
                "numberfield": 10,
                "textfield": "Hellooooooo!"
            }
        },
        "interactions":{
            "111": {
                "product":{
                    "10":[
                        "like",
                        "comment",
                        "like"
                    ],
                    "20":[
                        "dislike"
                    ]
                },
                "user":{
                    "1234":[
                        "report"
                    ]
                }
            }
        },
        "product":{
            "10":{
                "description":"....the description...",
                "randomnumber": 100.0,
                "publisheddate": "09-12-1990"
            },
            "20":{
                "description":"hi!",
                "randomnumber": 50.0,
                "publisheddate": "09-12-1991"
            }
        }
    })

    bw.learn_from_data()

    result = bw.get_embeddings()

    expected = {
        "user":{
            "111":{
                'embedding': [ 31.0, 4.0, 1.0, 12.0 ],
                'labels': []
            }
        },
        "interactions":{
            "111": {
                "product":{
                    "10": {
                        "sent_bin": 2,
                        "sentiment": 5,
                        "label": "POSITIVE"
                    },
                    '20': {
                        "sent_bin": 1,
                        "sentiment": 1,
                        "label": "NEGATIVE"
                    },
                },
                "user": {
                    "1234": {
                        "sent_bin": 1,
                        "sentiment": 2,
                        "label": "NEGATIVE"
                    },
                }
            }
        },
        "product":{
            "10":{
                'embedding': [ 22.0, 7.0, 30.0 ],
                'labels': []
            },
            '20':{
                'embedding': [ 3.0, 4.0, 29.0 ],
                'labels': []
            }
        }
    }
    assert expected == result


def test_learn_from_data_remove_empty_objects():
    bw = BeginWorkerMock(APP_ID, LICENSE_KEY)
    bw.set_data({
        "user":{
            "111":{}
        },
        "interactions":{},
        "product":{
            "10":{
                "description":"....the description...",
                "randomnumber": 10.0,
                "publisheddate": "09-12-1991"
            },
            "20":{
                "description":"hi!",
                "randomnumber": 1,
                "publisheddate": "09-12-1993"
            }
        }
    })

    bw.learn_from_data()

    result = bw.get_embeddings()

    expected = {
        "user":{},
        "interactions":{},
        "product":{
            "10":{
                'embedding': [ 22.0, 1.0, 30.0 ],
                'labels': []
            },
            '20':{
                'embedding': [ 3.0, 1.0, 28.0 ],
                'labels': []
            }
        }
    }
    assert expected == result

@freeze_time("2021-05-16")
def test_learn_from_data_with_labels():
    bw = BeginWorkerMock(APP_ID, LICENSE_KEY)
    bw.set_data({
        "user":{
            "111":{
                "dateofbirth":"09-12-1989",
                "userlocation":{
                    "latitude":36.8507689,
                    "longitude":-76.2858726
                },
                "numberfield": 10,
                "textfield": "Hellooooooo!",
                "labels": ["fake"]
            }
        },
        "interactions":{},
        "product":{
            "10":{
                "description":"....the description...",
                "randomnumber": 10.0,
                "publisheddate": "09-12-1991",
                "labels": ["comedy"]
            },
            "20":{
                "description":"hi!",
                "randomnumber": 1,
                "publisheddate": "09-12-1993",
                "labels": ["mystery"]
            }
        }
    })

    bw.learn_from_data()

    result = bw.get_embeddings()

    expected = {
        "user":{
            "111": {
                "embedding": [ 31.0, 4.0, 1.0, 12.0 ],
                "labels": ["fake"]
            }
        },
        "interactions":{},
        "product":{
            "10":{
                "embedding": [ 22.0, 1.0, 29.0 ],
                "labels": ["comedy"]
            },
            '20':{
                "embedding": [ 3.0, 1.0, 27.0 ],
                "labels": ["mystery"]
            }
        }
    }
    assert expected == result