# -*- coding: utf-8 -*-
"""CONM_20250408_00.ipynb

Automatically generated by Colab.

Original file is located at
    https://colab.research.google.com/drive/1JSgmW_mCFwt8e7-ckPGHjKXv7NIfDJwS

# CondaManager
Conda 仮想環境におけるCUDA コンポーネントなどの管理を行うクラス

```
2025/03/19 0.1.0 BranchRouter 用ダミー生成
2025/03/19 0.1.1 デバッグ
2025/03/27 0.2.0 EnvironmentChecker 追加
2025/03/27 0.2.1 EnvironmentChecker 取付完了
2025/03/31 0.3.0 第一次改装開始。パラメーターでインストールする対象を選択できるようにすることが目標
2025/04/01 1.0.0 第一次改装終了。
2025/04/03 1.1.0 ブラッシュアップ。ToolKit 導入
2025/04/04 1.1.1 enva_path 追加に対応
2025/04/04 1.2.0 archive_enva, restore_enva 追加
2025/04/06 1.2.2 ブラッシュアップ
2025/04/08 1.3.2 Project Unicornに基づき、環境変数を導入
2025/04/08 1.3.7 EnvironmentChecker 取り外し
```

# CondaManager

## 1. condacolab 設定
- **概要:** Google Drive接続と condacolabのインストールを行う
- **使用例:** コード冒頭に設置。condacolabと周辺プログラムが起動する。
"""

# @title a. GDrive接続とcondacolab設定
if __name__ == '__main__':
    get_ipython().system( "pip install an_QuickViper2" )
    from an_quickviper2 import CondaInitializer

if __name__ == '__main__':
    condainitializer = CondaInitializer()

"""## 2. モジュール定義"""

# %%writefile /content/drive/MyDrive/code/CondaManager.py
# @title a. CondaManager 定義
import os
import sys
import json
import shutil
import subprocess
import urllib.parse
from importlib.metadata import distributions


from pathlib import Path
get_ipython().system( "pip install an_DebugHelper" )
from an_debughelper import DebugHelper
get_ipython().system( "pip install an_CudaUtility" )
from an_cudautility import CudaUtility
get_ipython().system( "pip install an_Toolkit" )
from an_toolkit import ToolKit
get_ipython().system( "pip install an_QuickViper2" )
from an_quickviper2 import QuickViper2
get_ipython().system( "pip install an_UniqueEnvVar" )
from an_uniqueenvvar import UniqueEnvVar
get_ipython().system( "pip install an_EasyVen" )
from an_easyven import EasyVen


class CondaManager:
    """Conda 仮想環境における CUDA コンポーネントの管理を行うクラス"""
    def __init__( self, venv_name = "kohya_env"  ):
        """
        CondaManager の初期化を行います。
        args:
            venv_name( str ): 対象とする仮想環境の名前
        """
        self.venv_name = venv_name

        self.debug = DebugHelper( instance_name = "CondaManager" )
        self.debug.enable_log_to_file_stdout()
        self.debug.enable_log_to_file_stderr()
        self.debug.enable_timestamp()

        if shutil.which("conda") is None:
            self.debug.log_step("Conda is not available. Ensure Conda is installed and activated.", success=False)
            raise Exception("Conda is not available. Ensure Conda is installed and activated.")

        self.toolkit = ToolKit()
        self.quickviper = QuickViper2( venv_name = self.venv_name )
        self.cudau = CudaUtility()
        self.even = EasyVen()
        [ self.bch_path,
          self.dst_path,
          self.cur_path,
          self.src_path,
          self.arc_path,
          self.arz_path,
          self.lcl_path,
          self.bin_path,
          self.lib_path,
          self.tmp_path,
          self.org_path,
          self.orz_path,
          self.enva_path,
          self.envb_path,
          self.uni_path ] = self.even.setup( venv_name = self.venv_name )

        # 圧縮仮想環境にインストールしたアプリが、Cacheを仮想環境上に作成するよう、環境変数を設定する。
        self.toolkit.executor( cmd = [ "bash", str( self.uni_path ), self.venv_name ], shell = False )
        self.make_environment_non_editable()

    def create_env( self, cuda_version = "CU118" ):
        """
        アプリ起動用仮想環境を作成します。
        args:
            cuda_version( str ) : 設定するCUDAのバージョン("CU118", "CU124"のみ対応)
        """
        self.debug.log_step( "create_env", success = None )
        self.cuda_version = cuda_version
        self.quickviper.unlock( archive = self.org_path, destination = self.lcl_path, force_unlock = True )

        self.install_cuda_toolkit()
        self.install_cuda_dependencies()
        self.install_pytorch()
        self.install_xformers()
        self.install_bitsandbytes()
        self.set_cuda_environment_vars()
        # self.verify_cuda_installation() 環境チェックをするプログラムをつくったが、時間がかかるのでパス。
        self.debug.log_step( f"仮想環境: { self.venv_name } がリセットされ、CUDAが { cuda_version } にセットされました。", success = True )

        self.toolkit.file_remover( self.arc_path )
        self.quickviper.pack( archive = self.arc_path, target = self.lcl_path, destination = self.lcl_path )

    def extract_env( self ):
        """
        エクストラクト
        アプリ起動用仮想環境ファイルを展開します。
        """
        self.debug.log_step( "exract エクストラクト")
        self.quickviper.unlock( archive = self.arc_path, destination = self.lcl_path, force_unlock = True )



    def archive_env( self, env_path: str ):
        """
        アーカイブ
        仮想環境を、引数に指定した圧縮仮想圧縮ファイルに圧縮します。
        args:
            env_path( str ): 出力する圧縮仮想環境のパス
        """
        self.debug.log_step( "arcive アーカイブ", success = None )
        self.toolkit.file_remover( env_path )
        self.quickviper.pack( archive = env_path, target = self.lcl_path, destination = self.lcl_path )

    def restore_env( self, env_path: str ):
        """
        レストア
        引数に指定した圧縮仮想環境ファイルを展開し、activateします。
        """
        self.debug.log_step( "restore レストア", success = None )
        self.toolkit.dir_remover( self.lcl_path )
        self.debug.log_step( f"{ env_path.name } を探しています。")
        self.debug.log_step( f"調査フォルダ :{ env_path }", success = None )
        if os.path.exists( env_path ):
            self.quickviper.unlock( archive = env_path, destination = self.lcl_path, force_unlock = True )
        else:
            self.debug.log_step( f"{ env_path.name } が見つかりませんでした。", success = False )
            self.debug.log_step( f"同梱の{ env_path.name } を", success = False )
            self.debug.log_step( f"{ env_path.parent } に配置してください。", success = False )
            raise "指定されたフォルダに.env ファイルがありません"



    def archive_enva( self ):
        """
        アーカイブ
        仮想環境をenva仮想圧縮ファイルに圧縮します。
        """
        self.debug.log_step( "arcive アーカイブ", success = None )
        self.toolkit.file_remover( self.enva_path )
        self.quickviper.pack( archive = self.enva_path, target = self.lcl_path, destination = self.lcl_path )

    def restore_enva( self ):
        """
        レストア
        enva圧縮仮想環境ファイルを展開し、activateします。
        """
        self.debug.log_step( "restore レストア", success = None )
        self.toolkit.dir_remover( self.lcl_path )
        self.debug.log_step( f"{ self.enva_path.name } を探しています。")
        self.debug.log_step( f"調査フォルダ :{ self.enva_path }", success = None )
        if os.path.exists( self.enva_path ):
            self.quickviper.unlock( archive = self.enva_path, destination = self.lcl_path, force_unlock = True )
        else:
            self.debug.log_step( f"{ self.enva_path.name } が見つかりませんでした。", success = False )
            self.debug.log_step( f"同梱の{ self.enva_path.name } を", success = False )
            self.debug.log_step( f"{ self.enva_path.parent } に配置してください。", success = False )
            raise "指定されたフォルダに.enva ファイルがありません"

    def unlock( self, venv_name = "kohya_env" ):
        """
        アンロック
        env圧縮仮想環境ファイルを展開し、activateします。
        args:
            venv_name( str ): 仮想環境の名前
        """
        self.debug.log_step( "unlock", success = None )
        self.venv_name = venv_name
        self.quickviper = QuickViper2( venv_name = self.venv_name )
        self.quickviper.unlock( force_unlock = True )

    def executor( self, cmdlist = [ "pip", "--version" ] ):
        """
        アプリを実行します。
        """
        self.quickviper.executor( target = self.lcl_path, cmdlist = cmdlist )

    def install_cuda_toolkit( self ):
        """
        cudatoolkitをインストールします。
        self.cuda_versionがセットされているのを暗黙の了解にしています。
        """
        self.debug.log_step( "install_cuda_toolkit", success= None )
        self.debug.log_step(f"Checking for CUDA Toolkit {self.cudau.version_number( self.cuda_version )} in the Conda environment...", success=None)
        cmd = f"mamba install -y cudatoolkit={self.cudau.version_number( self.cuda_version )} -c nvidia -c conda-forge"
        result = self.toolkit.executor_sync( cmd = cmd, shell = True )


    def install_cuda_dependencies( self ):
        """
        CUDA依存関係パッケージをインストールします。
        self.cuda_versionがセットされているのを暗黙の了解にしています。
        """
        self.debug.log_step("install_cuda_dependencies", success=None)
        self.debug.log_step("Checking for CUDA Dependencies in the Conda environment...", success=None)
        if self.cuda_version not in ("CU118", "CU124"):
            self.debug.log_step(f"サポートされていない cuda_version '{ self.cuda_version }' が指定されました。", success=False)
            raise ValueError("cuda_version は 'CU118' または 'CU124' のみ指定可能です。")
        if self.cuda_version == "CU118":
            commands = [
                "mamba install -y cudatoolkit=11.8 -c nvidia -c conda-forge",
                "mamba install -y cudnn=8.4.1.50 -c conda-forge"
            ]
        else:  # cuda_version == "CU124"
            commands = [
                "mamba install -y cudatoolkit=12.4 -c nvidia -c conda-forge",
                "mamba install -y cudnn=8.9 -c conda-forge"
            ]
        self.debug.log_step(f"CUDA バージョン {self.cuda_version} 用の依存パッケージをインストールします。", success=None)
        for cmd in commands:
            result = self.toolkit.executor_sync( cmd = cmd, shell = True )
        self.debug.log_step("CUDA 関連の依存パッケージのインストールが正常に完了しました。", success=True)

    def install_pytorch( self ):
        """PyTorch をインストールする。既にインストールされている場合はスキップする。"""
        self.debug.log_step("install_pytorch", success=None)
        self.debug.log_step("Checking for PyTorch in the Conda environment...", success=None)
        cmd = f"mamba install pytorch torchvision torchaudio -q -c pytorch -c nvidia"
        result = self.toolkit.executor_sync( cmd, shell = True )

    def install_xformers(self):
        """xformers パッケージを仮想環境にインストールする。"""
        self.debug.log_step("install_xformers", success=None)
        self.debug.log_step("xformers をインストールします...", success=None)
        python_path = self.lcl_path.joinpath( self.venv_name, self.bin_path, "python" )
        cmd = f"{python_path} -m pip install xformers"
        result = self.toolkit.executor_sync( cmd = cmd, shell = True )

    def install_bitsandbytes( self ):
        """ bitsandbytes を仮想環境にインストールする """
        self.debug.log_step("install_bitsandbytes", success=None)
        self.debug.log_step("Checking for Bitsandbytes in the Conda environment...", success=None)
        cuda_version_number = float( self.cudau.version_number( self.cuda_version) )
        if not (11.0 <= cuda_version_number <= 12.8):
            raise Exception("サポートされていない cuda_version が指定されました。")
        cmd = f"pip install bitsandbytes"
        result = self.toolkit.executor_sync( cmd = cmd, shell = True )

    def set_cuda_environment_vars(self):
        """CUDA_HOME 環境変数を設定し、LD_LIBRARY_PATH に CUDA ライブラリパスを追加する。"""
        self.debug.log_step("set_cuda_environment_vars", success=None)
        self.unv_cuda_home = UniqueEnvVar( var_name = "CUDA_HOME" )
        self.unv_ld_library = UniqueEnvVar( var_name = "LD_LIBRARY_PATH" )
        self.unv_cuda_home.add_value( "/usr/local/cuda" )
        self.debug.log_step(f"環境変数 CUDA_HOME を { self.unv_cuda_home.get_values() } に設定しました。", success = True )
        self.unv_ld_library.add_value( "/usr/local/cuda/lib" )
        self.debug.log_step(f"環境変数 LD_LIBRARY_PATH を {self.unv_ld_library.get_values()} に設定しました。", success = True)


    def make_environment_non_editable(self):
        """
        editableインストールされたパッケージをnon-editableに再インストールします。
        """
        self.debug.log_step("make_environment_non_editable 開始", success=None)

        for dist in distributions():
            direct_url_text = dist.read_text("direct_url.json")
            if not direct_url_text:
                continue
            info = json.loads(direct_url_text)
            if info.get("dir_info", {}).get("editable", False):
                package_name = dist.metadata["Name"]
                url = info.get("url", "")
                if url.startswith("file://"):
                    src_path = urllib.parse.unquote(url[len("file://"):])
                else:
                    src_path = url

                self.debug.log_step(f"{package_name} をnon-editableに再インストールします。", success=None)
                subprocess.run(["pip", "uninstall", "-y", package_name], check=True)
                subprocess.run(["pip", "install", self.src_path], check=True)

        self.debug.log_step("make_environment_non_editable 完了", success=True)




    # def verify_cuda_installation(self):
    #     """CUDA 関連パッケージのインストール状況と PyTorch による GPU 認識を検証する。"""
    #     # インストール済みの CUDA 関連パッケージを確認
    #     # packages = ["cudatoolkit", "cudnn", "cublas"]
    #     packages = ["cudatoolkit", "cudnn" ]
    #     all_installed = True
    #     for pkg in packages:
    #         result, stdout, stderr = self.debug.run_command(f"conda list {pkg}", stdout = True, stderr = True)
    #         output = stdout
    #         if pkg not in output:
    #             self.debug.log_step(f"{pkg} が環境にインストールされていません。", success = False)
    #             all_installed = False
    #         else:
    #             # パッケージ行（名前とバージョン）をログに記録
    #             for line in output.splitlines():
    #                 if line.strip().startswith(pkg):
    #                     self.debug.log_step(f"{pkg} インストール済み: {line.strip()}")
    #                     break
    #     if not all_installed:
    #         raise RuntimeError("一部のCUDA関連パッケージが正しくインストールされていません。")
    #     # PyTorch による GPU 認識を確認  'print(\"Hello World!\")'
    #     cmd = f"{os.path.join( self.envmanager.get_env_var( 'Ven_venv_folder' ), 'kohya_env', 'bin', 'python')} -c 'import torch; print(torch.cuda.is_available())'"
    #     self.debug.log_step( f"cmd : { cmd }", success = None )
    #     result, stdout, stderr = self.debug.run_command( command = cmd , stdout = True, stderr= True )
    #     self.debug.log_step( f"stdout: { stdout }", success = True )
    #     self.debug.log_step( f"stderr: { stderr }", success = False )
    #     torch_check = stdout.strip()
    #     self.debug.log_step(f"torch.cuda.is_available() の結果: {torch_check}")
    #     if result == 0:
    #         if torch_check != "True":
    #             self.debug.log_step("PyTorch が CUDA 対応GPUを認識できませんでした。", success = False )
    #             raise RuntimeError("PyTorch が GPU を認識できません。CUDA のインストールに問題がある可能性があります。")
    #         else:
    #             self.debug.log_step("CUDA 環境は正常です (PyTorch が GPU を認識しました)。", success = True )
    #     else:
    #         raise Exception( f"resultが0以外の値を返しています{ result }")

"""## 3. テスト"""

# @title a. CondaManager 実行
get_ipython().system( "pip install an_EasyVen" )
from an_easyven import EasyVen

class MainLoop:
    def __init__( self, venv_name ):
        self.venv_name = venv_name
        self.conda_manager = CondaManager( venv_name = self.venv_name )
        self.even = EasyVen()
        [ self.bch_path,
          self.dst_path,
          self.cur_path,
          self.src_path,
          self.arc_path,
          self.arz_path,
          self.lcl_path,
          self.bin_path,
          self.lib_path,
          self.tmp_path,
          self.org_path,
          self.orz_path,
          self.enva_path,
          self.envb_path,
          self.uni_path ] = self.even.setup( venv_name = self.venv_name )

    def execute( self ):
        # self.conda_manager.create_env( cuda_version = "CU118" )
        # self.conda_manager.extract_env()
        # self.conda_manager.executor()
        # self.conda_manager.archive_env( env_path = self.enva_path )
        pass

if __name__ == "__main__":
    main_loop = MainLoop( "kohya_env")
    main_loop.execute()

# if __name__ == "__main__":
#     conda_manager = CondaManager( venv_name = "kohya_env" )
#     # conda_manager.create_env( cuda_version = "CU118" )
#     # conda_manager.extract_env()
#     # conda_manager.executor()
#     conda_manager.archive_env( env_path = self.enva_path )
#     # conda_manager.restore_env( env_path = self.enva_path )