## Subresource Integrity

If you are loading Highlight.js via CDN you may wish to use [Subresource Integrity](https://developer.mozilla.org/en-US/docs/Web/Security/Subresource_Integrity) to guarantee that you are using a legimitate build of the library.

To do this you simply need to add the `integrity` attribute for each JavaScript file you download via CDN. These digests are used by the browser to confirm the files downloaded have not been modified.

```html
<script
  src="//cdnjs.cloudflare.com/ajax/libs/highlight.js/11.10.0/highlight.min.js"
  integrity="sha384-GdEWAbCjn+ghjX0gLx7/N1hyTVmPAjdC2OvoAA0RyNcAOhqwtT8qnbCxWle2+uJX"></script>
<!-- including any other grammars you might need to load -->
<script
  src="//cdnjs.cloudflare.com/ajax/libs/highlight.js/11.10.0/languages/go.min.js"
  integrity="sha384-Mtb4EH3R9NMDME1sPQALOYR8KGqwrXAtmc6XGxDd0XaXB23irPKsuET0JjZt5utI"></script>
```

The full list of digests for every file can be found below.

### Digests

```
sha384-GdEWAbCjn+ghjX0gLx7/N1hyTVmPAjdC2OvoAA0RyNcAOhqwtT8qnbCxWle2+uJX highlight.min.js
sha384-N3MQjGtCiJF/kKIC3PKqEqipvPYD+SrXFFLDJXJqoRiTAG+c7uIDvyC82jm4u4L5 highlight.js
sha384-elZp05EZ8AgjT52C+3CgO6PvSzU18iuXkxKUTMCy/gpbrZPv8R9heZdSBWH7+Lvx es/highlight.min.js
sha384-WDg469eZfwaynBgOmDGfBjrGPkiczHs3VRgxCzioJwIXh6dRffiRyj8mlTDzUOlL es/highlight.js
sha384-EIywSCcGq7pL1vkNuY4/seae64HgBbGjUGRDNnjJPhUVSDZzuSRWKAr2OQSJCLS+ es/core.min.js
sha384-/+PkBHw9Vx63rslCCTgDJQoxcdmNAB8QkkE+mataJ/gYHnroxXbbUqpZXI/i8CzK es/core.js
sha384-jSRPpx6rrJge30xNPZiRzm9KMbyWxprw/Are7Y34dZ90vGIUfntrU1T0W+a1RhBq languages/accesslog.min.js
sha384-yNCP0sdwolnj5W6b+rBC4O9HhTi89S+pMO6g3dhA28E4Nh8U2sODAuKsz6C0BL76 languages/1c.min.js
sha384-B/SpbqqZVZQIr1t1sf7pgdpGvHqad0b3akYTAoMPuibwOcgvip7stUnu1FDCOToc languages/angelscript.min.js
sha384-AUNDJl44vrpeLXyYqx/XsT+1pMNhWBAImuI7Vq7AaWEbiAlfsD+c2eeP5MevsDBK languages/abnf.min.js
sha384-C/sy2YHBBqSYuLpOIPwmowl8FAib+W+qL+kLPC16NYJzd2mPa2sKsVTUGHVGMNKl languages/ada.min.js
sha384-YSzii8vq7yQVui2TRE60pvJTcXrIfgfUQv5Q3NFkGZKEQoblb4GCykfouqDDPP+x languages/actionscript.min.js
sha384-iUQ9BcUA+p4w9sZAfo8E9iui0Ay0KwBSTIkTLZB8yc3+ExNWmxWMYMLxB1Q1NOZt languages/apache.min.js
sha384-/ZGnKBtFlWz3dap5TeG4EtewOydrfxdAP22FQI2zXLM9BtAVmutZf6VLqcnNKwoy languages/applescript.min.js
sha384-gGstQAp8sFZHpWlwzuHHRdaRSHf5zRmuKOOxna89kSS1igYH4b1uXlkPq5619I28 languages/arcade.min.js
sha384-Xg2bPVtjGJ1EYwh3oKsNkG6p/YNRk1ZGW9IRCbYfFccqH8e7Lm9JPcFZQnMOzNOr languages/armasm.min.js
sha384-oiof9guf8fns9264wmrURvzAJJ5mv1Ih1dah3NeHJ/IrzCN2d4cA9DObasCXwcA4 languages/autohotkey.min.js
sha384-UgvIYJpRQTiRlgz+ApmrQmnucu+VVWocV/co5t/el2inOm9cOQFcIyupaYsoviX9 languages/aspectj.min.js
sha384-q+4aQVHnY1FwKezXFV3xmaRkdKNu1ux80j8yzOSaAjn9EeziuDYQhRFPdqeNXsds languages/asciidoc.min.js
sha384-nHtU1E068tLBBDCv0xqeAbGJxWj+5WhJyMYtzOdNJI+k+a+2aqW85pZvqINeKhUl languages/avrasm.min.js
sha384-eFizm8+WBkIU2d3kalqH2vV0pDtyQ1xPXQyCROjwwsryMMpzIMQgw0CSz+6AUssT languages/awk.min.js
sha384-430zfNyjJ94+NW90/wejFObS2O5/ArJ3y1LxoAj8uIFCEjiFW+/lE04XJySkVyKj languages/autoit.min.js
sha384-I2QKqUbed9Y7Cn3Burg4468VsVX6chc0t/APBZlDAOWzvCA1JupJfLcU1fCzpgKI languages/axapta.min.js
sha384-83HvQQdGTWqnRLmgm19NjCb1+/awKJGytUX9sm3HJb2ddZ9Fs1Bst2bZogFjt9rr languages/bash.min.js
sha384-4PYLazOmBsX2CrXzJcLCBLwOPrvovuNEQptMmXmjugO0spaxi/+Qh4FQrCxZ9FvG languages/basic.min.js
sha384-1rEjTEiJfyW/zOc6+fBt0yfAomOXS+PVFhbrsEchIZn1Og+BC1hZd1ITiibC+mqp languages/bnf.min.js
sha384-D7Zn/BuvbS4hog8nu/2oCUV7tIcV71MqakBo/xlKsbI76kqn/SkcI9XRYNfnBU5v languages/brainfuck.min.js
sha384-z1DF5ZgOM0rgwam7tgSDoIrODSfOFl4KGa8qxiQTCDUR1JGxqBlQQyS32jAvA8nT languages/cal.min.js
sha384-jtwnwOYA+K4zYN55fA4z4U0PTK5oEp4RcLYaXkYRKO3UUzge1o21ArmvKmTRdh/d languages/c.min.js
sha384-u6XTarvRv5GvfCSwf1iwzegX6cqsIt+JhRL7+Aq1KnnBOhBPBCn+ysur3NhSl0/m languages/capnproto.min.js
sha384-E041Cil/vgxM9X9fEtcmpBtC9xKD65kaUVTyBinVmPhk/EU8FcKUaKzTaMeW+yrV languages/clean.min.js
sha384-k42POIunJdHLLeoPI9dqcGAz5bFhJwmxqAiGK+QSK4wXUfPvOBwgBekZQ95jcTcA languages/ceylon.min.js
sha384-xvji4qjk1yTch1tZsD6rflf99DB1cFqcQ6OeO1nwb/JXQ6/6y6pKzgumMTW7yrCQ languages/clojure-repl.min.js
sha384-Wzcbu1NwxLbZ7syk2AG7Uel/YM0wnO+k1hZgcKkqX+Rtbc8ezBkEZTVwsSu6T1M8 languages/clojure.min.js
sha384-NXUiKRE4iB8J7h3t1Z2aLjDRJslCWi4SWqJpec1z0Y927kNqWejg1uGaEh1P3GCK languages/cmake.min.js
sha384-LFxbMDVPRH4AeQG7k/EHmUwU0yT7Pl+q4JL7xqBnZXy/v8HY+RNAssnYJKzbaJ6t languages/coq.min.js
sha384-pMjYEVk6bW/Li5fVrTdpSGd5dGEGD/jrgHSU08VrcmsqyZB2B/8245L9/m9F4r0D languages/cos.min.js
sha384-pWWH+XECnKlN3vERExCAowNTyRmNartFm9vBbd9NCeMhtyAFqHE/EQLE1fpGlS3c languages/crmsh.min.js
sha384-/yf54L01PbO6NtVs1Pu9rgfNHbKXanLdNcGVuNa0m5+KiyH+1NpZRDK6idm5VoVl languages/cpp.min.js
sha384-GfsA+L6x4v9I1iiHCTfElgEMXs4tsXCw+gC+bC89n7nSXfUEfNemTt48EJD4jndY languages/csp.min.js
sha384-vsyuH3iggrwNYvo/3BmSS7DJq9olpDb+P2wlR7ziu2vDbXfyGGzE+V+3BYVUoJTo languages/crystal.min.js
sha384-6NsOlZtv7W2iSoDU+Yi+hENfl3MuiECvnl7emdRUvpIpDbLvoCjpAU1fjE6HxIQp languages/csharp.min.js
sha384-H5nvRzsubVRK0OxJ9XKWgoOkkYJTKWWthvtIgsZIWYkopc102uu8/yEaIr6Jv8lj languages/d.min.js
sha384-xV+yQq6yNCJpHU1bXc40ygKBQS6BF2u/5RgLa0A64Rn/ue1fW52gr1gF8rko0wkG languages/dart.min.js
sha384-WPXcFtkOLxX0vKdl+zKh+zWbdMF2GV5aWrNbGT4L2qBYLSUxq1KQDWYhJHh8qRrt languages/delphi.min.js
sha384-IZ99gU0P2i3O8itOlz4exVdl6lGFAgj7zq4hgyoe29bt1KyJykBSCxdH8ubn3DSk languages/diff.min.js
sha384-E8a05vTFeTlJrGsYy6uvHSN7YyYtYVDfuTB0NYKryrTqkKAh0Kljndtou/5x9KxE languages/django.min.js
sha384-jg4vR4ePpACdBVLAe+31BrI3MW4sfv1AS62HlXRXmQWk2q98yJqKR5VxHzuABw8X languages/dockerfile.min.js
sha384-KEzcnJSlyQjvNpDwNeh6LUBpvcvNSFDj5NMNxUiH/1XzSiutw7KXKxsqF+LJm6+P languages/dns.min.js
sha384-5NYiSl0YAfUusQHfYmh7TNKO2XTd4alWFiwdqemWAjv7qIAJt3Z6vm8J7XDj+8dY languages/dos.min.js
sha384-KmsHM96M8R4zfTswDLJs69vAc2OJUogOt8Cm4o8l40GllTIg9C9eZk5Nh0JlKvDI languages/dts.min.js
sha384-UOr3RVijENphF7cBcCuiwwXBTABQ1c2v6GGYxV/cBVDjPU5Ikmw4dIYJXc51cSsu languages/dsconfig.min.js
sha384-yF7iAW+5SyekbVk9JwdcD9FvBYGbwXM9+uNjWDiZwXY3K3sQIweb00ODpskdmvoG languages/dust.min.js
sha384-zrN3w76G13sFPJ4BPRDmYrPe1TW4P+1wpgAcXcj9J9VCpDkGj+H3HbK3xuQvnXfq languages/ebnf.min.js
sha384-1yAJ5qBg8K80+przprGQnQPGeFIb3tKmAVjHnSSEZiq4HKjh3cNy/nwzJ7l2nFK+ languages/elixir.min.js
sha384-/FY+XRWmaEh7r6TTcq3KEOxwB6sp4MEUUh0Ls2PNyfch1P2kpYlIm6HF9INVd7MS languages/erlang-repl.min.js
sha384-KGAFeqpaGSoOdl/gNDQIpZxHTyRKmSxbQrHZ6amSeeDxFyRUffqIt+10aCHUUXF3 languages/elm.min.js
sha384-gEX03lFlUpoA8rfkmZ2UaKi2820eK1l0gtckxIGjq6Czr2uCyyEfekMrSoFwej+C languages/erb.min.js
sha384-cE562Xij2Xk4SxySz0GKZX6CH4pFe4wSgZgEyKEzTYRhiORoEnlvzFeW/2ZINZ6z languages/excel.min.js
sha384-971pUduvIswF8i7V63S9oUoSCed6WGwmHM3Imgy/209KXdtphB1Jh/85+fJPI3TA languages/fix.min.js
sha384-0CP4DzMXyug19cwGTs3ONRRpUBBFTa4O5Hv9YjyJY10yvQOl5W+x5LOifval6ZBt languages/erlang.min.js
sha384-4DbMwCqRqcxBersl7bPX3KEVstVtQ9CgGKwlo79Iu61BRrE3O0uHhhoXhuDriGZn languages/fortran.min.js
sha384-Qz1f08u7z8GyotqYiyJEc0CSlFIt0sbnl9z0bRmcKh1K8Tj7EJXwJtBxpwdTvogd languages/flix.min.js
sha384-zwiFuwrESVX82ZurqEjUprf78YrmlWWXZTA7XQy4wCSw8liRMvTT+lzEeI0t7gBa languages/gams.min.js
sha384-ZXXsPS+ORWfAch+VsKTpwqC2ZAO1Ih6Uqk+5I8Si9yAO1UXLNuK+CgkbSGly2GQA languages/gauss.min.js
sha384-YLuaMuZSDa3eAMAZ6uLgoNUFGWVjLLQy8eJ4C9qhla3Yjx6hlGwiUXrs8jq8Lf1a languages/gcode.min.js
sha384-ppssxHN00pV6RryFPHORyMZIe5RpB6yrukW7w4JdFAIrCpJVJCVTC3pR9KEjDpAS languages/gherkin.min.js
sha384-ubPsF/0wBxL7zEYtpuIL27F0zYtzIZknk+3GdMEcQw1v0Td6xAJ6gDhxHy0UFhk+ languages/glsl.min.js
sha384-VaLJgrWWfCVtbAeOjBIOf/CbAIqj1IbdKtJhHh+gh+2H8F7XEhfK/a5dcg/lhxXH languages/golo.min.js
sha384-Mtb4EH3R9NMDME1sPQALOYR8KGqwrXAtmc6XGxDd0XaXB23irPKsuET0JjZt5utI languages/go.min.js
sha384-Zsb2SvyyyZPjojn07qmkDNfdiapkw/4t6kb2Abbh14QEJBHNsXBhkE9wWfbVPYOW languages/groovy.min.js
sha384-2vKrGN0+i0vPRkMvpEZkOgi35/Vyn5pGG9Oi+baBPODVLjCrGyhQpwDG+BBEEeA3 languages/graphql.min.js
sha384-Qh5sDx8XR/gARjQgTGqZrWsG9IWURWytCVZFqqUDtx4M16przYikOSpBofp/smEj languages/gradle.min.js
sha384-uQS8ZRxc1TFIVag12uukmzjpuxnV7ATxgtJ9AQ4V2bcFtNVQK0mpSh3oIjLcvipv languages/haml.min.js
sha384-1+L6SezuX8EOUBfYZyp1GMjjPHwIiJhMhRsLebXPt5ItYffwrwQII8OW4Zu/hndV languages/haskell.min.js
sha384-B7PNh5MCEV6Mz/dVdekDqHupFqAlPHt9qknPa+BHdOlUd95jeTW7QByrAalohboI languages/handlebars.min.js
sha384-BvW6WsJimyvFXi5xeOmolLeUQ+pMLER6sQzMGgq+6aOstIyN7go9WEdmE19Vbyry languages/haxe.min.js
sha384-FApAVPv3mT368h+qYeULjOQHEoMZXmF2+3ULgzAtrpjRJHxFjAn+Xq2S5TQXrZBd languages/gml.min.js
sha384-ZaQba+1+217wybfm4ZhQcYeyF3wdFyT+AeWx9tv3saciWITHU2rWUhYSYxunVb0r languages/hy.min.js
sha384-AdGtHB9BrPseyh43FlhtWRXQDcI2rRahPtIB3Dp2XBvhE+SB4kPfMyh2G8HSHXmy languages/hsp.min.js
sha384-ubRntct0s40ZDtDRLkxA3/xYX51o5yC2U8SKlky8dhIRsjSnvZiUKLhz0gNTewno languages/http.min.js
sha384-eQgUZRyLniTo0FTkATb327W+4GXuOZwxmBM8T9lQDg6JtwltIb6Kvv+eNe8RsCFy languages/inform7.min.js
sha384-nOpxTlWRgLR13aZWWqxVWHxB0hn+sCSwnTWCcC2TNJISGlmzyU9A/R1Rm/0/GUw9 languages/irpf90.min.js
sha384-0/1VV9gfjl+ZuUf+R7fvp6dQlJ5JVh+WzEqjzOwd+PCh8fa104Vm13MBaJjTz+cG languages/ini.min.js
sha384-lUROqArhKl6lBtLfj7c/6cvdo9MeijqmAlpclz5Qr0EvK8ttOBHJQ7CZPoINACyM languages/jboss-cli.min.js
sha384-Bb6DhE3tUpBROwypL78TbhRUs9QbCt2GxcxVSYglt2l3MefrYkm4CfwjfWhRfQaX languages/json.min.js
sha384-nAq7dvjNYEoyfKcI9TVNM4NmxAF68eGURBz/2PtasIfUkEnkb5POJ7vzb5feIlUI languages/isbl.min.js
sha384-sjuRI7LKtdlJaa/cLtQZMJIJKg2ZLleJSmPtcXZxe3FxEhBMktIkyiQTpkP7byoq languages/julia-repl.min.js
sha384-aXaL309hpQjGj0iiC05w9SImeyJiZ/9XKOnzrg9IiBEj5iaO2UTe3Or/KJHfOKFc languages/julia.min.js
sha384-sJB50VGTeIqTEHDgjZSgd1y1E7nJz0ZY8+7RXplspljiErNkS2/0c7/EY85jhIs4 languages/ldif.min.js
sha384-ZZRcOijvFcSYDPwKpgTEOGSm+Ww/Dmj2lwxelFyQsl4U7MHHlYDt5vEyCdo82FhG languages/lasso.min.js
sha384-4g0nL2h6+jF1QzsdR/uAFRuswDrXVI5SQaKq0Ig06T820naSMaIFWtWb9gs4/RB+ languages/leaf.min.js
sha384-YnQI+TwzZ1xRk2HJrBDWxf28lhGoOKeLoiMTznB2mX2JDclqH/iuPsQC86Qtwh6X languages/latex.min.js
sha384-Rt/SKBZQ3kgIiK9pZeM7wlw3VIC9GqKB5CWElZkRBdmqW8LgAQyzMIxNaTw0M2j3 languages/lisp.min.js
sha384-bVyarbf6VCg82623c+ljGhsOzwmgyE177jfM0o28keFDzQfsv9ytARIaag/I6100 languages/livecodeserver.min.js
sha384-2E2Olv5Np0C9fY2pt3A6kvYCcGDTyGzMc1HxkVA5d0gczQAz7YiPND12qr1xDfzG languages/llvm.min.js
sha384-uVH2rCvbWFMkwVPsVokIzvS1BHV1h/0W8K4Tkdo705C4Zsdz240UI5llM735JDmb languages/lsl.min.js
sha384-WES4Ky9nlehN2gBM6oY3/98hVtWF5PZnBO+7jK8ZVKFo+QR32zhfuRK+Mv9jQsOK languages/lua.min.js
sha384-pYbMiHWycMKEfJaSEsquFRDTjCY7QHvQN0FIfDK0lVMd9DPJuOA7Kq5wZGecvYwM languages/makefile.min.js
sha384-SEUclvWzbtiTGabHIlauQmWVPE/JkSVlJe5BKAnSErPApBGiH4/5zu8+yWtPCUqN languages/matlab.min.js
sha384-sFh+6oaRBb6kdaMLf8x7qeH7NTvm2u1Ta6PtI0S8hoA+bP7UtHCyKFzaI1JBXwhT languages/markdown.min.js
sha384-F9NP072dYWBOnlVFrv0gpeItWR7egIQiE8rXgvnYEUZLZJ+J8jmb7gcaEp+UaUk0 languages/maxima.min.js
sha384-5KB5j2LLN83tz772MpfqgmaHbgNJoGpqncjtBcrCE1c0l0AKBBRmHt3STu3jYxoF languages/mercury.min.js
sha384-dlO3LB/5yjSdavmxp9TaKZblcqj5zaTJRDKgojwiVYPxMdmAUtAzPQsTW/0haJZJ languages/mel.min.js
sha384-Hc5qAKAeMWj7W95V6BNPzRqhhA2leur3hyGlhd4ncVW5kO+LReHa1M+XfT9qrRC5 languages/mipsasm.min.js
sha384-RVJmhZMr2jBnaa18Qeu4zmda0XwGYYPv0C17h+6NlgD5LZMPKn4tJRQuVPxhOvFu languages/mizar.min.js
sha384-iq3FlJUEnFxl1esoj6R7yvUpe4xwB8SRMskdl49eoywRd5vxOTDuiVdb4L+X3LyM languages/mojolicious.min.js
sha384-Sc8ePJIQQwgAs0e2t3ISMAssMi6OXVyCfhyUsf+YDfFAtc++oz1hnwPydbojFvXI languages/monkey.min.js
sha384-nPXR87ewAxssB/YfzBwp1kfvUtkNP/qe/feDB0JUoBzlbkVbI14vOyEp/BCEUYm6 languages/moonscript.min.js
sha384-lkHkblFGkRtnpUfwjwgwfsqu/ugnms7NU6Ao+EhElIIPVTyAKANHf9HxMyxkqYHf languages/n1ql.min.js
sha384-exQrQUL2NBuy5l35FwAYS9jFajBnrRCmhIM9yrQgdWOoM4s6aIzqW9gkfn+LaC4f languages/nim.min.js
sha384-LuyPddXj3XboLoEpfIjtRhUGLKeTPi+/8SBIKEm+sYpG/IQexvq3zIAS86PRP9hM languages/nestedtext.min.js
sha384-TG8jUbt29ktiHxVaCkA6FLnJkL/PYG3zQwEYexdbr+Z6mMkFf+c0ONpHyuIY4vvG languages/nginx.min.js
sha384-TQ3GD74bUmAbxGCTp6y09Ujxio1hxtA79eu7/gYKwgFanLtczRqmlf81EJQmY8gH languages/nix.min.js
sha384-aZXuOYsqBQ2RFl2pfS/f5zLA9yUjgNr0/zeuus2h55Mp6eWqQapLIIJYDsG14A89 languages/node-repl.min.js
sha384-evxKJTmk5AChNAzYM6uh+DVNAm7QCNu2x/OynLEpbH+h/TVMellCIci2K+uvXWID languages/objectivec.min.js
sha384-Dhxsr9ocIQrnHlppkSen7Dge0kK5KC2XBDPG2PVCDVQdHcem7THu/X8ed9RYhZ0g languages/openscad.min.js
sha384-JVC+S3b8TEbM5xgapLjbhcnLc7YbakzETqPtT3xLLvnm6etYjVjsB0uDPMRkmagW languages/oxygene.min.js
sha384-aFCHrYMEyAT2T8i9tOfudmHsGqppJfwRKg8n8hbAMVPV9NfdCCtdw9DFBWd/xr8O languages/ocaml.min.js
sha384-SIQiLYyI1nDqsAsAbw/Jws1FZPmWmAyhR1eu3MCDZ3NahqRoelEK14C6q6j1Kign languages/parser3.min.js
sha384-HBc3JQgC+i/l43bOuIE9xtQz9ZFXZDEjPCyiFD7O5Wauvl79zHEQmV4uDStGEQLu languages/perl.min.js
sha384-YDzioK2VUfo7ZdArDjWEhtgR/zHsogx8SJ9xfToavGK4J4Jq16QZMwo5+mEtHgqL languages/pf.min.js
sha384-qgAIdznqUzBBAS4nOYdZKnhkSxER8jn7g10EW176MLksxvnQCBcXOdawfqoRb67X languages/php-template.min.js
sha384-Xd0AQIkWCEjBL8BNMtDMGVqFXkf445J6PNUJdXbh334ckjeSa90wtSUjXySuz+rt languages/php.min.js
sha384-hE+znpd5xggEBW6IccZoCI0mgFHAfLVuqT/7aW8RakaQ4UJnI058SfIX3lhdGxtE languages/plaintext.min.js
sha384-7llnZwKgcknuqjXcEeUP/boMrzjXxhhfH9afFHqRtqRpnfdGWZJzcAqfq9QHWx5E languages/pony.min.js
sha384-oJqzx2KYttoB62Br3yGkuDMB7q08+JIjBH1jiTmGfz94kIjSpP8WFYdneQESWp3z languages/pgsql.min.js
sha384-0u0NM3ve01ej9h9zRzZ/ztDGe1h07d6TStpNoJ4f/50I/vtoCsDHI2PfzDZSYz8q languages/powershell.min.js
sha384-VSUySmqeUUatYNFPFFZAHEmgpdfedIGFcZyQv/6OIsLc1vR1Wl1SimExn23za0FB languages/profile.min.js
sha384-a+nwjMBrHHj4K59nk1SmGnkQ19/zLAQrOQaVUYsiTcBr+HeKqUrjvw5Znx6HQeS7 languages/processing.min.js
sha384-G0I/5pg6Q0by4a7utLoEKB0+vqUs6FrSfTgIyjVkOCW8I1R5A7R1mDVPpgvHKd0r languages/prolog.min.js
sha384-sgNTtpv5Paw0Ux4jH4t4kp0frMSOTtws8wtoYOeystt+hVEKHDinMYYW/ZR5SKI9 languages/properties.min.js
sha384-S+PDVP8B6WN9X4KJd8TaIK8Sj6BTdd9QAqCRzIYSuwxAQHpcGbZjHNBVrC9qUfrh languages/protobuf.min.js
sha384-EXxXI12fO4TsAJfYNsrfXpOeaiw4OyAte9Qly3UTiBavCYLG4t3LEgjP1KzvWs7b languages/purebasic.min.js
sha384-WYz+BIjSZsSWkXPxYY/cN0c3yN3N98f1+JASXaEegpsqQqN0/OpVe9PbiyQ8IFA4 languages/python-repl.min.js
sha384-NzOBHXRL0Tvx3y08Z9UFLnfEortdDn+7h0HagnNff40zn4M3PrMQqY6tPYO+APjJ languages/puppet.min.js
sha384-YDj7s2Wf0QEwarV3OB8lvoNJJCH032vOLMDo2HDrYiEpQ+QmKN+e++x3hElX5S+w languages/python.min.js
sha384-mj9QNbYd7ibOAOnrCkY/1jRyke9TPExrkIU7ouV8xtHVmUrM1w/7OoGjiVin6YdB languages/qml.min.js
sha384-IdLfiBtws/HD6C/aQaDlw4/xILR8XJV16CBZtTPLs80HNyGsUpNwQmc2+g0aXPhg languages/reasonml.min.js
sha384-0wLjaNaUVh/kcNbtLmLZkz9NX8VtSkyPUrRqKAjtf2u056iIzXgTRn1fonvwhvJU languages/q.min.js
sha384-KR6QhCPclzoQUVJZiCDWkDytFrjI7H3z8gCN39ACvRPSTZ0enCcrDTxB33nJ+N8J languages/roboconf.min.js
sha384-Jc59QIC9wiM+tNA9lnx0Cb23+WrXa6PT2LgOAhVPwyx490IJRKdt9/NGT1ZyQc6y languages/rib.min.js
sha384-u+x0P8zfV33Hs+FIbrPIw38IUbqKyYA1yEdL911mrwtUJxaNJtMOdB2injOcZMmF languages/r.min.js
sha384-1lod0Mw40xIsxTmsdIbezbyXhu12CZLPOjFSHJrEiWgJteWmTeSQlluzoW1Sx/aw languages/routeros.min.js
sha384-LOFdRHZ5u+oZg5Wh8DpkiJQR/w8egcUhJJoo95XVQ846DEwuRxGUujx+tQKyqXme languages/ruby.min.js
sha384-VTNxHMz2AmpHxzSm8SvRI0As5+wID2j2XJBFtWTic2iEK8WbXgR1fymVQS9S2DvY languages/rust.min.js
sha384-9rzhEAd8EfJCPELcwmTS0OeWys1RBPz26Oghv+u7GT0oXa1PrDPxfsPU7SHQ3fKn languages/rsl.min.js
sha384-Jf3WKZhkMK2CEuMszIpp3i7w+QbmgOIzDd7ueAeHTH5GXOTZK38bHB9f1JZDETC7 languages/sas.min.js
sha384-Ydu8edRhwf0Q3mwbCRUPCkhYopMD4JzvyZL6YsOLcDFpQ+7WdKmlBk1jFbCfgfJ6 languages/ruleslanguage.min.js
sha384-0BOfMS8PT03WW7dkaI3LgBx21PwSLCxWucHG9qTsAXRKMLHcIMHuu9spz3606P3o languages/scheme.min.js
sha384-SA6LxB7n4Y1G9cv1TTU9P9QCpQXJaNURCzOyoGjV/D9gkhcHYSsYERw8IVffZ8uD languages/scala.min.js
sha384-7qPMH09XA8AJjij3fvK4DR/tZf1ajIf1EEmvQfpXkGKeC8tAqz+TqdavHtkxKHet languages/scilab.min.js
sha384-olAuUjfRvTi/iEH4RXRpaq/G1iJGizn7OefkyJLQYuqNhh1xAV5dnUrkH/LlPd9j languages/shell.min.js
sha384-9xPGl0IyZCFsMCTnDoxktyJm/1P4qtRk+WCscFY2JCkrXgrZv1TYV8tw/iLVgQeH languages/smalltalk.min.js
sha384-Lmk7Z08oieDOChjC++YInGivPR7XvfTJFXunqImOsFq71wdP7fqHZs4p2E68NW3z languages/smali.min.js
sha384-Ahqruee4TcqHPxJ0qWMd8AwCzxn32ztYC5BsgO5a+9StIlucQfHfbxf3shJ4knjC languages/sml.min.js
sha384-2uzCjI3OPwJce6i2hphsYs1qqTqRrDnfPXbfjZggPWy2/Lgl8gzV9Hyl0jhhoWy4 languages/sql.min.js
sha384-9ckCCmGBhiko6gWuy8HUDGhwBCAW7OuTPEPZcHh5qfpr78oz2zfuU5svTcxdiYvC languages/sqf.min.js
sha384-7k+vuglPXrNadgl5jySkKUUjOjk6NsVV9+M+JCRp3wP++kZjRK2hogRJrGBrsURm languages/stan.min.js
sha384-VOaOLJhjy62zrnq3N6nBKbU8yCsDVEZI+l6Ch11xdI8FuUHFqU2AT7FX3TDwKChO languages/step21.min.js
sha384-b+wzPiDP5nqHGvkTkMg9KIp7xjHmb+1LGAlpH21g+Oyksvq3NHknM39ZLdZ8RdVE languages/stata.min.js
sha384-OYfRTEinJxuatPmhe9R3rJ+lG6jNXUHvSsBpVDDqU+RWJZ6wjkD+QYuM7dBnUbG6 languages/subunit.min.js
sha384-39o8THRvVOl9DEzvSrCMsUCub41ibGqjj/O73e4Mz08I6fmgUSfHplFTJkBqxKz1 languages/tap.min.js
sha384-A6Duvm756b9xHe0gOvjVWitUxu8PLX7g461lNhVaPtpoSJ7GyUNdLCD+OBRj63Ue languages/taggerscript.min.js
sha384-mLeI0s1+uHK7HpID/RvvgQ1QGBKOce7RvWMnxBDy/boNuW0W9JcaiaT+0vOpdMy7 languages/thrift.min.js
sha384-r9OPcD/aFYhqKM2SiJ9rWMahBytgRLrbYvJ77m1AaO3ghVCWBSK7f1hWyc3HuWX1 languages/tcl.min.js
sha384-XlDRKuo3Ea9pIqT0CUrf4Vp4ZN/AItGl/aUiL/4AgHSC2N7xJI1commJxYZo0I1T languages/twig.min.js
sha384-qy3fDByfNQpIPnTx39nNL76o4ofcek+QnbxyLOWY1qgtRXq/TYz2vHv59LWC8i/e languages/tp.min.js
sha384-cHdAVgva6ADwmJQeO9JPevHoX/8tL0MEqz3vMAxu+Jy6X3m1TJ+U9qYeb3E1Rny0 languages/vbnet.min.js
sha384-qsiVnf7Cwh6dClTlwLWKcQ/3IY15+qMaVWIDEliKyO80Cpb7z/ie9etw4fo0ydyo languages/vala.min.js
sha384-q0sJ1YFiPW+i+XVme9b3M6adegU2pnXPpwhLvGukidQZIkV9QIzBFdSCkf4LjQAf languages/vbscript-html.min.js
sha384-dO0iYHNUbREvGXvDcxZsIcqf9DCTUASjhhyStXtJFhj9dipN3Ypc94KaHXIQQN24 languages/vbscript.min.js
sha384-i+ovBPuX6/6IYKo/hvzjiDeWmFZGjQ8GcN5P2JlCvsQQz7tmGhwFBZ7L8O5j6Uih languages/verilog.min.js
sha384-lRfKmfeanfEEsne7ndto0E7FFUaI9By4vEr1RZkq+pVhHu4dRKebAURBKne7Dzdm languages/vhdl.min.js
sha384-6R+wTNBHlaPfbkMRDtjTf5quccaGcE9dSnW1yvuCyvVQWhUxAnDoMfcW5iPuVAZv languages/wren.min.js
sha384-vfNnhhcZoO470LVTMwy7VO8IGy2qRDfw+NOnU7pdz/V8y2uNfjq5Ea7Gljj0dZEn languages/vim.min.js
sha384-cw6YAmvOjrczJAb9NkjJRzsXso0VEJrJsLZl8G49s9vVlIFVwG8cR6K8yvpDlGn5 languages/x86asm.min.js
sha384-TTxGby06lDIEB1Y6l2/Hd2VJbewWQSCESXUcDWC24gJN0IiptruI6DE1l5+w4Qzh languages/wasm.min.js
sha384-9kjy3NF6jS9dLGRFPKl8hlTtPEwFGUkscHqovyfsjFqblpaGB05HpBzFgHD8Fj8c languages/xl.min.js
sha384-CN3No+n1UZXCFYyl+ge5yAPGTNGuH23BdIsFJxntDmEYL94AmoZlNBHGSdjVSjKG languages/xml.min.js
sha384-L+i0yBdAaDJKM0LjWrOba9BR58bmNU4O1e7339zfF3ZW0CjnaTJsQ6AuK22TNIVc languages/xquery.min.js
sha384-bMkvdnz+wPu1ro0fqO3BaDWztc7RzSvw05MQFP6bhJKDcwpkrFYTfTFI9ndkP11l languages/yaml.min.js
sha384-7A2HawNmptSPS479EB34q/+e8WPft+1zFF1OETV5ZYY/JzzUkfkkqXxng6jV4wB8 languages/zephir.min.js
sha384-QqfeezSFk3jFhiaBPAGWdCdQGBdBckwrd9PXxjsb1OObrgikKVL+tcnJYLsSReEH es/languages/accesslog.min.js
sha384-gybzJRGZrWV/j50oGMw124LpN/Tc/C+33TDgAq/xuPRDKEgJufzztHa5DdmMks2u es/languages/angelscript.min.js
sha384-wKaGu1ROMZaxQzYoWdEpbgPUpNoyGYkIEIZdhHnNN5NI19e02sXHIifCPwQ/ecdj es/languages/abnf.min.js
sha384-8EECOILmBYlAkigZ+z/CNnC/mSVvoGBCdsfhUIp9Eh2nw228k0J27PUwfzzrXVwJ es/languages/ada.min.js
sha384-xVGCBpZS3OpdCcbBo14EIYO8M5OMnVTQF4yKcGp0grCGx6EQkkM7b+ufTu6OCDpf es/languages/actionscript.min.js
sha384-xuOg8cLlaKYVwu8KcJCX2nbs9mrVa7Ngz0c9NQiwViwl408M5TX3w1FC22RHhUuS es/languages/apache.min.js
sha384-2kZ9vXS4PfcoGlwJ5gNbv5QnUY15tcWNHxJePONFXmRi7pgWs7WAOaJlfA1JRFN6 es/languages/applescript.min.js
sha384-1xPG1kTsRkF3U6CRoWoLs4n4UOvAFC6opoueI1vWZ0GYANwTuOv0+um9ecHAG3qU es/languages/arcade.min.js
sha384-OSLa36uZ8MoDqW/oUAJd5PV1sHIr0nDREiPwSlmVEa82KJDAVc6zTpFt+/zSNJf0 es/languages/1c.min.js
sha384-ocl2WOnIR3HmBi1ZJ2kfjMxq7rNhHxV1t4GFEmTWFMZLbHHXgdhR6HfCneMNCkt2 es/languages/armasm.min.js
sha384-cxMaZnqujtZiWcazO+5AIOVVliqyyDYRy8Nl21t796HxgHx0zOQTID4v/by9acdM es/languages/autohotkey.min.js
sha384-IWEQTAPRfs5zaxhI0JynkX/u/7+0r4uZMoUGsHB0vr4RpLjuwW6qoHGq+28ZNvFS es/languages/aspectj.min.js
sha384-P5meR23mmKq5yLOMg2BF3KK/kka+PgB0U1YJ/Bn/IlzO6M1ZaRMeJ8Mq16Z8ujnB es/languages/avrasm.min.js
sha384-9XM0lQmJ4T2vT5fovmdMwihL4baMIxlzYPj6DUTgkGIQPNur/r5HVQmz7NufVMHp es/languages/asciidoc.min.js
sha384-ZtSXWAZLFp62aXWYyynhvbh2tt+/B8/dq0PoZd8lNMmhGj9kec6R9TAkI2SPFKAx es/languages/awk.min.js
sha384-HzACo/Fj/mHoBXnxhCwNJnNAXGEE+r+6eUg302MQcRdDw9mVJJJkuFyWnl7Pl45V es/languages/autoit.min.js
sha384-C8/VHu7qq1MrYknXkGgr1E97n47lEC3tu3i6pjOvWZgx1ObPWIYoO7MOvUdjvKQs es/languages/axapta.min.js
sha384-4KNGcjC61b5C3mM3IHO/Cd+a9B/V8erOfOCkhFGQGV0SleDcX+Fx5u50WJNy+8rp es/languages/basic.min.js
sha384-ZQaScy+UAXWHHTqErXqOiguzuFM9KQG3KY70cNyfXUtM+Q5OwX5SxrEiIrFuLYZj es/languages/bnf.min.js
sha384-u2nRnIxVxHkjnpxFScw/XgxNVuLz4dXXiT56xbp+Kk2b8AuStNgggHNpM9HO569A es/languages/bash.min.js
sha384-tO9yiZ3yAyGDiWGcmVH+On5wYksAKu2TooLr2eQpIFEzasuxCj+JgEwfXc4Rd5Kp es/languages/brainfuck.min.js
sha384-yTXZ55gT3vqZ1Utfj/D4XrgwicCH36zfHc4Q0H+0WiaSOOTONI5unPZOsvecZvJr es/languages/cal.min.js
sha384-5fESKgrRcGs7I/89bn7NKFcHyvIVcmQIG4JfCEAV5Rg5VVtskrmGkHVOIsD1642v es/languages/c.min.js
sha384-7NkWzcNLtsHJrgq5lYW2LsWVHUMP3BojV5mpmKOxagVDi9V8Y+OLDT19sZOw18Q7 es/languages/capnproto.min.js
sha384-1FpO7j5PcwDe8+fxjYb7fQJsfGEEDXe6Cf/BKX5BJB757pciR60wM+kwggBS/dNf es/languages/clean.min.js
sha384-xpVB1ReXdLGX2qOVBnOTgLfq0H+gUcc2dftOpl/RKha1RYSYyBYQ4FFKypy4oBul es/languages/ceylon.min.js
sha384-pVO3LX5PlcHHL0Xu0V9NXHtYEkwIbhY9VijcsVsRIzBCzYvbh5ynInvcldfmQbxd es/languages/clojure-repl.min.js
sha384-XOcAzhOmbwUPkV0n4c9nRSfK4Ro1fPKehnn/hOns5fvsieIWnx/U5Czz5ljWIPgR es/languages/clojure.min.js
sha384-JdSkDmOvEqN4fTPk/zl0QpN/wGh53J+xJRFtOMdEwhZ2iDgT5a/Dl18e/TChDqjK es/languages/cmake.min.js
sha384-YrTksxGF5q+81b7ZbllSjNa8awfGhHwVlnk7SALU+8jtDUAcEvNUDNA0KeUEyw5V es/languages/coq.min.js
sha384-jnfP4KFzlfLE+tWZkDc/927s+A6R6Ks7wM4JLP1x72zUBQBoJSE48o7B/6r1kb/a es/languages/cos.min.js
sha384-BmL28rW1ggTPsUjUVLEN3lP+Vd5xub2ZRM83QPZHcU4m8bj3vzWMP30wvDWIygNL es/languages/crmsh.min.js
sha384-+tDHTmLKfBxXgVksRhLEJM4z9PfcGQ2XsrZMDcdJ1SIlPZrtAR4+m4XUX+zJf5nf es/languages/cpp.min.js
sha384-h92gzO66Hk1++bH+fXGmNay2Yf9WBsijCsnwubxdSE86gVf6UMC3drQ5VDjZaoaK es/languages/csp.min.js
sha384-3zYTwQ2p/pFsHA7rfbbRX6gggHPVk25IfZxXHsh/zCNy1YfDOlxBKV6YdkTy01Dy es/languages/crystal.min.js
sha384-k4z6XdU7qI35NxUF8SGumv5kMerrVg/xoat0iSaWnu/dHKoNZKdxZN3gI2WYgMfe es/languages/csharp.min.js
sha384-guDR3VGIj675PAxqDTJxjM2Ew82al5msD9ySiJOzXfehlaoslILRHNUyrlxBSnZG es/languages/d.min.js
sha384-XQur/dNXGWVSWOuMk+TsDoXfe5GzflLu972jJ72dqRV4N65jOT+h4GGpXQDstHWC es/languages/dart.min.js
sha384-KYBMbsTREEtzM5JsppWsdLqQsXtLiJkMauViGW5/rWaPzT0wDsU7yRHcjbBKmOt2 es/languages/delphi.min.js
sha384-3J9ZKxCAysZ+DowS+TRZQFLDNVJwRq0pxq9t/JYsuFRmBSwgvJrbRDH4Av46yJft es/languages/diff.min.js
sha384-xmWxMjSXleWbbNXqoQdpI8OkRJnRmiuJ8cmkovyFJSZxyjycv8DA4XuENvWPDLDk es/languages/django.min.js
sha384-Dw9HMdbM7eZULiZ40cTZJ0NU88GUU5VQ22H+PVZ0IzxPGdnPPqKdsg4Uk3D2wbCB es/languages/dockerfile.min.js
sha384-FyQp/yTYnwHmm+XuKfVGWxFNrGlgQ5KeEfWuNLOkj9AVIMPZPAx74qjwIQz2V5rt es/languages/dns.min.js
sha384-XSgDKTvGzlj/ixthrrw81gg7YE2TFYtyoqKTrwrQ6Eb/VIsbP8Krclh2BmNnJLfr es/languages/dos.min.js
sha384-iGPPZyVXd7W9GgH6h9I03x+/J/rrwdsAHvSvvOkKy4RmMs146fYhCx9ywrKTBML+ es/languages/dts.min.js
sha384-Uc+MP2N7ERUL43nljo8V2Xtaw9xDiyS9EgFBrshaBEObPiqyY+IwIxzkkHsLM11a es/languages/dsconfig.min.js
sha384-7baMO61bDaNT3XOhMkeU9+88gLiAq763qqHp0olY9dgfDtIxisZo3LnoRrtLRsWT es/languages/dust.min.js
sha384-i+XFBJyknuWMcn7936ZYNZ+DDJ2mtBBAShqFgZkAVYz5ngJxMqatdUiD0gPr+4BP es/languages/ebnf.min.js
sha384-Uo+KLVze+/A9YCgMDJ4XtbATny5jt/ABQshKJVHut77YNUuS/zPVYAGKeLFdthjP es/languages/elixir.min.js
sha384-9HUVbJs7C6TSuS3lg1gYv9UONx7830CSmNM1C4hxswmRia0EkeP3w0VbufumsO+B es/languages/erlang-repl.min.js
sha384-hv9MdXKrd5QCWe9kkMdVR8bUwU6grFv3l9i5Qce+rRe56KnkTSGFKBeojjnMaOAG es/languages/elm.min.js
sha384-jKl1B10yci152fjeFFWOiIxAnyuqjj+GrabCB2d6Dq0jHIYs0kqsCF31KINUC9i5 es/languages/erb.min.js
sha384-uMMeKU6xBWD301AZB6o7aBYDWroG+K8OUYDrA+cXf286lrDcciS6wBjzUnXFp1fS es/languages/excel.min.js
sha384-g3uTC9nkSFSU4IGAEpkWH6/I3NJWikEbfrIpsA+SsOs9Qz3tYB7AoP97wTnd4ZDW es/languages/fix.min.js
sha384-ffehxKri+ZLFXqRZcDwKHI+OuCy3tWvJDjxmY7kTJNnF4EB//blTQga2gnOuh3UT es/languages/erlang.min.js
sha384-UkVn2T3qZDBm9TduPrIPBBINJQyzIVSJMGa2DRpLOc2zp6QZ3wA/6zUw190hnX9T es/languages/fortran.min.js
sha384-RrbUBKaPfrWutGiVHiEJ7EdCUNW1Ca39lVIM4XBA0ymj/DJ6CYlLcWqPiRZZio2h es/languages/flix.min.js
sha384-ZT7iPLg9oVzTWin6/2HhkKNkDNBHkwlxAAolZ4tmwLUD8Jb4EFhkyZIEnMlmLRDg es/languages/gams.min.js
sha384-/crZfccBqGs5cAN8T8hFz+SYpNxD4xepOX5baIlqV0G8zaV9SuzTonApAXeiKy0/ es/languages/gcode.min.js
sha384-7iky+DInqFxLz45s1mmnohe42XafDC49jX2Uy8qT39RfxdNcrgtHiYe9MirdFQp3 es/languages/gauss.min.js
sha384-aOobeFVQ4Sux+80R0437f74uXacvfbsC01OHMCwOf9YqqqXdCv3xmdltL65Ztp6M es/languages/gherkin.min.js
sha384-0QXwzWESgZAQ8866ALLJEDZ9l14HhJTWnECXBxJ4DHDmywVn0LRIMqz7JJQWG/AM es/languages/glsl.min.js
sha384-1KwZxIzx78RvcZJCildSMOxijbCro6qqaPG/Wj3ustmMwgTYIiid0nenpiXqDxRh es/languages/golo.min.js
sha384-5Mzx2XTmXU2XQ0AiQg/4HA9SbBDrPySZgpsOfSfflGdzC4bIpCjWSxIP62fOIFkO es/languages/go.min.js
sha384-187D/dkURBQpq1omwwr2ia/1/OxrA8ok8uGZN7I5rHV0muATfS04w+mWc/kYTC2w es/languages/groovy.min.js
sha384-y8dkUR8w2ApytztUtIg+27fJuiL5sH1oq9uRGEcpH9zcoYPU/ZFvroB77kwDE7d6 es/languages/graphql.min.js
sha384-D7ZaQQrEs5BBMC2isU1LZZm9ITf31H0h+uPhGRdse290qklkkvt+WXc4sFh3m70B es/languages/gradle.min.js
sha384-TaV6bIvb0PAdNigZuAtH9T5CUU7BCEJxOLBsVOeAb4O412gzt++0xyZtxbpQTwcw es/languages/haskell.min.js
sha384-W3AwKM0k7ufGY1rXi/NC/GCvcGw1BQfGeDa7qXqV2IQbLoxnIQ0JojJtNxl33ybu es/languages/haml.min.js
sha384-npJ2g3mwXMh9Py849SyQMEdEzTUaC5NsUuYXZ+C936ZpalKkv8V/XgbWzs1n2tzJ es/languages/handlebars.min.js
sha384-qU/sDJJjc964Hn5j3KPIyChNS1VdgAQEeHQ/cjvYF0C255fgR//m65y9kdU6WF+B es/languages/haxe.min.js
sha384-G7+SF7XAzzELSBGtt76Sc3Gi1ix+fhD9ChAAvcduFY56+C4XGNqnQsIo4PbAjXru es/languages/hy.min.js
sha384-X9TbaX7oKsQOaCnW/+tchCet/NzBVPbeOBFooLVGsQiqit+b66sUhevct82JLuSG es/languages/hsp.min.js
sha384-j+2AgmE+4SlZjmviwPvbGypcb9BQNHQj043l9Bb3F2fnlusdNxxA5/INHsOrSE6g es/languages/http.min.js
sha384-8b3VeDRERFE/95Quioly3m60mn7GJEY8SOw8a1HMaheNvs9NJgS073+c9OaVurML es/languages/inform7.min.js
sha384-nz75K0pqPS/d8Q+HyuddJgG2PLLdsjnE1IX03ZlQP1cWx3uGZhHbgBF84cUfcx63 es/languages/gml.min.js
sha384-Qk1583V3PAnmXJ00e8ufkLJOuIZIrqrg5sTGoghEOwikzdWrdpiJv8lQqrURXjBG es/languages/ini.min.js
sha384-Rj8CWMElH9F9ZzXkxTheLhrrvROGyKtHkmZqCRmANiYQHjMWDbKQ04PY6OQVOxC2 es/languages/irpf90.min.js
sha384-m8qbMX40lQiAOaUTw6V/2bp3qnXMR2vYuOBjTm1DHcBQY5GRKisUsdnuFUR7k6dc es/languages/jboss-cli.min.js
sha384-QFDPNpqtrgZCuAr70TZJFM4VCY+xNnyGKwJw2EsWIBJOVcWAns9PHcLzecyDVv+x es/languages/json.min.js
sha384-19YzhrU5CG+oV8eJJVC64Hir10cuNEEEvqj/Stura7ZqUqbgDsldpSfoWunbHnVO es/languages/julia-repl.min.js
sha384-IaB8ennafOrABIWzfmPoJLQ1Bnd6HSlO8sXjy4lXzu4rPXbGN9ul2o+a1omLUqKO es/languages/ldif.min.js
sha384-yvS3Aa9ptrkJ7+gPUPKPo7dC6E0MIoYJQamXheN75r4LEbhr2NCHR3QuBfwus+L0 es/languages/julia.min.js
sha384-vIQM0UP3EhcE9MH94YYzQE6uBcXu6nq/fKKiBnmUccpbatE/jkZ23Tc0K3NdiiAd es/languages/isbl.min.js
sha384-fvrCkD6kUH0Y5flDk5fM43/nR1vGkZ0HzIoNLyT3GQsRZFOy7ZWY7lwwjDS0TV26 es/languages/lasso.min.js
sha384-zSVAwJVXLuIs4ZlAQiIC+Cd4cEMWt6w02VRWDhzVKPHMdmzSHKb6xTRgQcuvXc6X es/languages/leaf.min.js
sha384-Nj46FmS2fPnUrHYhGDrVZjS+SqPPtCVnjiMaPB2WFMnYAWV8lW1T2EQ2gxq8gDs1 es/languages/latex.min.js
sha384-S8uA27fsmX6Fx6EhK08sAFvjkDhHZSMfrFbNgqDDsstYlqK8NVqKK5TqYldxjDny es/languages/lisp.min.js
sha384-qu4HCOB3caPjcm12Aq1NoxdXwJNKp/Doig00wy36re3AWCpnG1jl5H//vkqBJk2B es/languages/livecodeserver.min.js
sha384-gpuZNMDkXtGAGcK7Oc23L5g4M6m2q3cDgegXlnpTN7euamQLboWR9c5lPIrINbrq es/languages/llvm.min.js
sha384-UwHRKsg/zBFMdfj6qNa9tgf5dOEgf/9aylmmn10aghnjWhWKzxSqRLvPC4tL9jAT es/languages/lsl.min.js
sha384-cjlU8DPjZ3XY/dLzIx+CaoB2jzKXf43s2KSU2rZQGmxwR1d7k9p+SFt1IsNkFnnV es/languages/lua.min.js
sha384-Ev1PV0+HiSwEbi0IfJYmpMoxv3E0sWhAALg1frIiitM9zh2BVDe871H9Z/RGXqFM es/languages/makefile.min.js
sha384-wNVkpFZOn59qnU1wCeePf9PzQGAA1JeukodpxeRj5ldSPyLeLU9FPMTKFPcFFym4 es/languages/matlab.min.js
sha384-65/lNNIY+ayhHTtFznjnyZ5w2NDdZIpSEcqjss/p4ex5JamZ46FdYwDDf6IBLCmE es/languages/markdown.min.js
sha384-L2bh78DiQIIiXEEyhBU58rhag6CSLrSk9ezrNrTIdXlvzltGVPqN0SBPA8QVIchm es/languages/mercury.min.js
sha384-sLZolVMgRf+rWbsksKXmI8jcmLv0pLNgsJHWxfegD8vxxrtepmgxMZbHg424Xo4X es/languages/maxima.min.js
sha384-hzQcVB7Qny4YQPUuHQtZ+wHvcpHWNYimKmHONk5g77Fca1MwwkNixH7uPmFnZSIB es/languages/mipsasm.min.js
sha384-ktem8uFt3dpR7cjM5U2q/rcOhjrg9RTRnjRvEN2Y3X7F7CO2/NI/9fVNhUWqL51+ es/languages/mel.min.js
sha384-riyMTSZ7x2BtezT0NuSUcLWe5L4yYLXp2QWu4NWdo2kZOs8Y3iBTcd3M469B49Qh es/languages/mojolicious.min.js
sha384-GMt6gh6Sam0b4tO3URi3Aayc/mBtERcOztn8gNiizdKCmWKcZdBKDa7VOKczR2T0 es/languages/mizar.min.js
sha384-yNAYZJH77tgJ42VoPB0eZbDd3SscEe/QadDk4z0uLMvq2Y6DV3G++6XuP0I6bh9S es/languages/monkey.min.js
sha384-hXY2GFW+d9dPGMJ2+S4MTTRFSRE2QstWS1MWWDmjNDc0FOrcC5fNbr90mh8awwa6 es/languages/moonscript.min.js
sha384-qVnCpZxP0vME5fvdyeGfdfPu5NbR3mSIi7UOWNGtON/opL6OfVYYSdK1u2S6rB8P es/languages/n1ql.min.js
sha384-2xkAL6oEvjXQcavQSUBRR0o9ZmaPa7KtykaGxiSp7Sy42KpSCsG4cvIpIXk3pJHJ es/languages/nim.min.js
sha384-VY5MjPeVE5QrSnKCEuQlNC60blB6fFELWFgSoeS99bkXap5NQw/D2PsurTmoZwOd es/languages/nestedtext.min.js
sha384-XOua+gbAwDawIeMkI2pkXOZH9Xxl9/XLoGuPJD5Bs3WS4bMn207o20s+aQtpsqqE es/languages/nginx.min.js
sha384-ZDRzqYOSMHFfcbhNtV0wdJX9mMQ4GIp+km7Re8Dife/IK+k9BqYeoYWAS26Ld6IQ es/languages/nix.min.js
sha384-WUouP1qTnssBcr1Txhf991xYhbMsu29J0AeMm8Hk0Vx/jBKehRztpDrtLV5chOyH es/languages/node-repl.min.js
sha384-hFyITHOZcjIH+h44SyQkx6EmN8ato2cZ9DaY9N6C7jLs9dNEmZrLKcCKI50zvOSm es/languages/objectivec.min.js
sha384-y3Xa2254pmPcQWr6Wljhj2sgqJounTcPFIrMkKQrWlWp7zzJHO7EpPdU4YxOVll6 es/languages/oxygene.min.js
sha384-Nm4R0c/T+nnd8HUuYtwqTIhgYXMPvx2fzjIQB30fzczD4Uk86ljdbr7GfP7VgSOO es/languages/openscad.min.js
sha384-crG9/szkNiDEO5kEofdhAugXK4heD4+rVymcFWdMD2zju1jMnmwj0+zRbryhqS7P es/languages/ocaml.min.js
sha384-gXlSFjAQdfSvHNnvsQzb6gF3Tg2XcgxEoaoJNt3eQM/zYp7bbECyesWhIDCim3Ro es/languages/parser3.min.js
sha384-2CHs1KY5b+PYxl5fEs1H9IwHwE8eglxOgjVwD9dlhDWZfWKj/w3uHtDDH+1P8nYO es/languages/perl.min.js
sha384-WeABEWvvGME3DkpaC8x0980x/hDPRS0Wb2w4RkSlEpzNfBjK1pBJMsziSkv6+SzK es/languages/pf.min.js
sha384-xcrjtkX6DU8emOzX0JkgBdUnCB6E5abIF80h1JDwLKDJPzmIJSWyaUELrPg28SsM es/languages/php-template.min.js
sha384-C28kAnknmyKQUME+2sa0tyQkFMN1S/aUSB1gQ+PKUf0+ewgxNh8CwAmd+C0UUaZ7 es/languages/php.min.js
sha384-AkqanemYxn6S3BQnW2++1+xqywaq2bJfFlfiAkPNd7Yv5t9YsS8tFzVVopyOa747 es/languages/plaintext.min.js
sha384-SWQ8CV4qeox34/B5OTaJlcCHnCUyGjTzwB7ZB0Z+VmjtG/1/0DK/XXk80l8/duqr es/languages/pony.min.js
sha384-OH8IAQNE+/4/z6sdDTBio6BftdgqOsadsVYkSooVoUwdZYg95qly80XMjUigH5zk es/languages/pgsql.min.js
sha384-LA+GHsDHAleSQDZ+9MEiJmTnrJ392Nkux5b1RIiGmzAkRo+4dwKsjS0z/pzCXIFr es/languages/powershell.min.js
sha384-pIXuqck4LfmmmkzhWmHbUMb1AHwLVFuI523GlUYcuNZ2X+wVdGyB7x9erE0Jeerv es/languages/profile.min.js
sha384-8KwenjihmQdgtsLsST675iXMgQbH8MZ8M4yyaaUY34BK27FxLwO3t6Vf/XnrTuLN es/languages/processing.min.js
sha384-OzCC3CRxx3WbANdD1Kmda4no08jqEbOptoYfXKZGrDWQ+vwLK5eHbx0r3AeMQkVV es/languages/prolog.min.js
sha384-igG9bugojNt63V6EXJUpd8a75vlEeh+G9suHO7+lk9/sbdQ70Rn0KYsJNRcRxJDC es/languages/properties.min.js
sha384-S0SG5LPfLffrYWmoL26SJIw83wbqXQz7YzSwoYH1Fv9oYhT4tok23r4XwLjhwdot es/languages/protobuf.min.js
sha384-BWESDJxF3tlHE7zC3x1g+vFLrvXtfyWcmOnyJ5iuaVdJzysOe/C8Rl6fii9KXKYV es/languages/purebasic.min.js
sha384-RE0OsKNDPyfYMmPfLkTxJXJj6I0NAc689xyUOr3+EcFqTQSaBqP2TF2UPSY0Qpj4 es/languages/python-repl.min.js
sha384-/piqZSA5WyjAU/0Qsz4/SJqEJY1cc8p+caPB7I6Vt70YqB7wmFBtLZQIPKHsRTcO es/languages/puppet.min.js
sha384-CPHh+9FxkWF3OtMZdTj5oQN1Qti0p4/5XBABz/JdgssOKHmfAOFz6Gu4tsG6MQiH es/languages/python.min.js
sha384-kYUxqwRpm3RkSmBRGE5WNwR959eF2Gq1+oyEE5twaZ5PfZnLRwTUv/5DwIpEwItv es/languages/qml.min.js
sha384-brwFcYLUPw5tzs2aTT7QpBBDJr8KOM0d565m9nqmZVnkGOro7yZiY1LMVkdd+Cz1 es/languages/reasonml.min.js
sha384-ipG/kK3puSleu0UZUpK0ddy+AdAc838rDe1h1gTse2tLdUq3HSlK2V7PIzZmsqQW es/languages/q.min.js
sha384-U819GJWTJDz2catnyvYguPUtCVda4qeZQ3y44GTRlIZxbThKuG+X90YNPN3VA79q es/languages/roboconf.min.js
sha384-/iToU4LYE5K1+u9mjsp4ezWGcJP7+loe1xp2GQbhNepeNtU1a9y1bC3V/hjyP+IC es/languages/rib.min.js
sha384-+SoUnimzbx9Uf3b+zsNgb58sz3J3nRHE7ftjhuB8hvcb3K8D8G0Fe3Bu4lBrA3Fj es/languages/r.min.js
sha384-zYk2sUFnVptEfXlETVtcHjgCsqnwsPAmiFluI+w6buaS3NABQ3A1+wrfLbpCL9wF es/languages/routeros.min.js
sha384-INdPgGNAH51T3uWXoaYVa0ag70hxlbvjTlhyLicF3SuuG0BVuycs/GrFGi7gt/8a es/languages/ruby.min.js
sha384-/ktfWRgwL+kZAZeeXDl9mwkD/3atjwjkzLCCoSHtME7MzP87wMhUmNUZ83AoqYx2 es/languages/rust.min.js
sha384-TTaTGhHJtVCiuponl3c3W41yk7oceUGvbthSIf0YIAy1Ua89nyjf3/xROIRa0ZeO es/languages/rsl.min.js
sha384-VwBoSEidUQdvDdOHc+Rxj1gak8QvAOq3NMw8vBrGXqR8HVTshFM9DLOZ3a6Xki5h es/languages/sas.min.js
sha384-2cU0Ky0iglRofoEhsHFwvU28aU2vNqby7FLYyABx2Et4CSHg1SMqT59fBIOdhD1S es/languages/ruleslanguage.min.js
sha384-sODLwd9vR209AK944nKCuMOHKvQLBtDc+6pxw2Ofgc0aPWbglehFBTIYya3hqakl es/languages/scheme.min.js
sha384-Ww9na6pFMpl0LFINRDLDfPQf6xtS31SnqKhTud2UgsqMjEYdiU0UCoL6aNcUg0Gc es/languages/scala.min.js
sha384-V5+UMZKcI4a5pMwGEj/UhnlXoW42mVpPU/8bZCoGiCZS+96s88J1Vy7Ne6Dx97NE es/languages/scilab.min.js
sha384-gRgqaIgHrAR1qlo866rkV1AUahqopv0dnpFFLAC4X3jYTDagRCTxQBTMFGV1+x1U es/languages/shell.min.js
sha384-uDAwcPj8hOxgjPWTW9AQoNeZKeCtfo/e9dBHntXD3L3KpY8wRg3BgmO57iLd4ZyT es/languages/smalltalk.min.js
sha384-fPEykjaz8PnjzAA8A14l3GD0L1xBV5CIr6Zj2No+5QcdbKY9qkjAOzV3lTHzyUbL es/languages/smali.min.js
sha384-tYQNE11b91fF0Yxb1oIax5gYF2FHibKKcMNwvFoEmgewqH6YWeAiLHe7VJiAb+1q es/languages/sml.min.js
sha384-DloKeCkj/pTPHeqWu3keGoEPpZJGm44yQjSmSfpWasykAMUobM0hcYFFPsg1PE6K es/languages/sql.min.js
sha384-6HVe/DzulN4+JYhskNbIQjIrL54HujHvB8kkYeroH8llfD703+UkYbSRcoV/Q5Gl es/languages/stan.min.js
sha384-cTYdWLIEJyEvihgV6KuwAmH44Rvisp+2xGep35NuqtSukZC9AtiV/zjownVDD/a/ es/languages/sqf.min.js
sha384-LK7AZp04o1ZCkbd4H0RCrNpbgt2Sqtoiyy5XFtrxi1Kl1qsemyf8dJWSP5Eg1vpp es/languages/step21.min.js
sha384-S47i75uP8zrBJ1LgUUP540uu2oQOVjSqqxk9xUhiLSGTsaCOzNdzNTsTBM0KrtMv es/languages/stata.min.js
sha384-ThEjd0cqeUoMo1okPph0zKqAkp2PTn7lSRyVO+Raqps94ti82wYRn2F96Fh/VJC2 es/languages/tap.min.js
sha384-YPX2/Q71sJT2APyCCwlDiAoevUjzpv9PQQtucLfqN65dDVt1qJOai1nW75AsJcld es/languages/taggerscript.min.js
sha384-eljTpkJZPyqNx/Ey5VJAmULjWQXdW4nf3Hw796BQza7d2zZjPeGhkAVAweLgTwSG es/languages/subunit.min.js
sha384-ywhHmG9ovsaI1Tafu3/GwgITztNxKBDgXWUX6K3tZx6OdHiyG9dt63QWtjpNIYXY es/languages/thrift.min.js
sha384-3jRAK2MRHgHVTGdcVv6Ldg8/S3ObC0JTg6o0hPnMsE1gRHtUeHTC9UEyLajfWRh5 es/languages/tcl.min.js
sha384-PEMVkXQ9Mi3tt5Jjr17Xob3ABkIXSzTpFeN4Nl4VXwAWVM9vVFIjBbDzci0pZxFu es/languages/twig.min.js
sha384-1aXkr1dOVq5cVBBRu/0PkrcKm0E4T3AGa9M+Z7FZ8xeaeEeyuhZLwm4q2KHBM+1g es/languages/tp.min.js
sha384-V5qIvBTXBgJ/JFrBZbiDxU5hJxnRsbsU1iLPvYES8a21SdylhibdxjursRkrWFU/ es/languages/vbnet.min.js
sha384-nqUhAVmjedbsaoQqA4WSUO/h0PXZ3jktQTj6QCi8D/13poAy2IEhdlydTvJgw+vW es/languages/vala.min.js
sha384-hX31lddqojtc+QTe5Wx5sLiCvFSEVdP4vg2FEqwbM206W2c3qfhqz+rjC19nxA3k es/languages/vbscript-html.min.js
sha384-C6xz8NJdWRpf3kD8yy3Cr+y8Wo4bcc9KvUK2gOcMDxwR/AlcBvi2mECx7wjStQoK es/languages/vbscript.min.js
sha384-3cj0wcUVuyPZD7/NmQZVe+R/5hKUXDXlyokh6aPqJBzdN65LnXrrDuR8JJyNrKvN es/languages/verilog.min.js
sha384-mdQMmzz/pL52qaQqnXG63Jewnl+lDaXRUeF60ffBbU45RuinqP9EDGlxzyALbe+/ es/languages/vhdl.min.js
sha384-lPuskV4gmQ7AGr7uwBvKGWgqsw8XALGRb4p75G1Bsy/543YwsCNLejOVJvGQSTci es/languages/wren.min.js
sha384-iGCHxJTxvjzc4eyRff/+05+KaernFwSKiHceFSeDycK+vr6zh9ba2fMppsaMscuO es/languages/vim.min.js
sha384-tu9x6stUNSb4pXeVRKavZtTp8ha5aQOjCp464ykdPSuI1BMHmN/07U/iqeaoKLgE es/languages/wasm.min.js
sha384-vHiTR3Tc9YfZRBaKmg0YH47W9B+ElJ6EH5jEEaV+HSoSPOtWe5TdrIHaHqf6uyfo es/languages/xl.min.js
sha384-y1FgDrVkuSox/kv/Oib26ZqzunABWyUkirzc4i4sthq8Z7c/ReWp2XA9DCh7MHQh es/languages/x86asm.min.js
sha384-n9ZezaAVj8pK1BIFZQxmC1BM9yGuBNRgvsOxHMHPCXzqYd1gSYIu9KjgGEm8K57w es/languages/xml.min.js
sha384-4v/r6DeVAX3yqdbicGVQeihQqLgJfIGnc39OTy42Up397bntu9taUUidWN338v10 es/languages/xquery.min.js
sha384-aWVzxOWmD+p/ZLEDee7zrLBVPmcuxVaxvXqyi25ezJS887Gk3cWf4Z+PUpU1UUsH es/languages/zephir.min.js
sha384-vXmhozexi2dHQBoniIEbWI5ZqDxyVfUs96BUGpqjWL1aberSp9pyxbvK8WCNASGB es/languages/yaml.min.js
sha384-1fjvRLYZVspF/TsOQHIoipbUkoNfSWsc5/FG6aYc1rnvRhLFDZmjuEJRiRZQBgzb languages/arduino.min.js
sha384-nYNTSA4nfVQ5esuMpVP111sGLunWJMnBXsYufX4ExWifybw5u48EAUxr+SsgIowR languages/fsharp.min.js
sha384-4a8Q6mmzSuhT+l+DrhqYu4KlkhOyLJiF16gkWqVqjXgPSaK1sO1W5YV20xOoNg43 languages/nsis.min.js
sha384-+MO3D3y/aZzZq7QMAAA5KiuAcqBZivJHFmVUXfwdBoLxEXeGTeQGsNMll4fpnegg languages/css.min.js
sha384-NZS9QHMmwN86GbNpIvGs4HOB2EzfqPjhIHqZviwZIw1HLumSMRnl4X/OIgpyP0+K languages/less.min.js
sha384-7CdPzO1jMDolQK+LCRlGiW1Oka6fSiQdIiqfTvHeBk2Hm2iYNoqMn60aJ9HdKkjb languages/scss.min.js
sha384-L/XmDiyusXomLRGcRmcBpPlboRFjpQNV747OJvg+sEOpgGYvUsNwcC4JLNQ2dI6O languages/javascript.min.js
sha384-j8a5RCWRIwrFdX0knM7cz9bqtHX4osYmTT2CBpXf/rlQfiSKq6cRbCsxLcJWPoxe languages/coffeescript.min.js
sha384-MoaYmkURrT/giCrgWXgBEOOXP+nEFtMKjRY2XIzWehdGT2GN73w+dTvyP+jZhEZL languages/stylus.min.js
sha384-8mc5ynnm3AlnXn8P3ccSqVAaZIDoijPM08/Hp4DABy6GMy7EHCQFwiIUoGAaGJiO languages/java.min.js
sha384-s0inyAR7LE5SVvn9VCZrQaiUxkDi+RsQdSKzFh2CjWf+LFd01DAjt9wtxGhT+4qT languages/kotlin.min.js
sha384-8yK16EmWIAHNcs1eB22u5N46+eXXbOw5Ca13l7ExJmgEw13ecvKn4qQ3Vri/neZR languages/livescript.min.js
sha384-w9lwucuIeYK8hzWrD0H1CZhvQLisv4eUmJc7pL7oMNkpBVpuuSDAwWJB7XW0ys+z es/languages/arduino.min.js
sha384-RwyqgvGLMtAbdxb0JwJS46Hs0xvJDJ7ZhVOBb7vaGQOxJVPL9LYfT7FE10EgHGus es/languages/fsharp.min.js
sha384-OKDl+BTp91z+xOwY30l5qXV194b5FceaKoTbqgMAEg9Sy0mhWN90wTJeuJ7IvFBp es/languages/nsis.min.js
sha384-1WyRNxnxiRASoImr/x5teZlt2xlwOotnfkJAj33imuGcVgndwsuCeyHAIke2RKmN languages/mathematica.min.js
sha384-G44u1/pUATC8754FIKYqkCxCl9AQYnspnFxzuR3RB1YVnTvqOEofqvZNQMUWcY/1 es/languages/css.min.js
sha384-vmEaRQjBedDEQk0PmHXdDeC6kbUsP+AhxtGkI2jQ/VdSRZ9ndnDOBUuxp+cZ7pQV es/languages/less.min.js
sha384-mlx1v98SAsuX2QnqD9+dsXOfilvT7QQjfyOD3HTrvA4TXsxV4yjW/+TqhUADUuOH es/languages/scss.min.js
sha384-3T8DJ91yCa1//uY9h8Bo4QLrgEtbz4ILN2x9kSM4QWX9/1kKu6UXC3RAbVQV85UQ es/languages/javascript.min.js
sha384-fZRN66gLZ6Pkn58UJ93oApMbUf8HwDyjx+eP4bLuepyEegCg1DKbKOHHmHEZvHgA es/languages/coffeescript.min.js
sha384-p/nPinGn5chrt4cTKyQIBe6+Mq3I9wWfqtxG1zDeE1MFRpZgR5u7OMwne2uePVM2 es/languages/stylus.min.js
sha384-716dHwZ7zbZcnEwTe7DLxlm5tH3Iyl8vSW5a2kYPgusEdp5k3A3jeZt0Ck+CjYE0 es/languages/java.min.js
sha384-yM8aIXHTDq77S7Ar7r/O8Ix2yH07cPC1z48Qh6HBzcG2+plE8YMQn9goknWvawaH es/languages/kotlin.min.js
sha384-P0AzyZPotdKwowy/WdkLtC0E5/ezqsShC3N2Q9q4f6TIB6Hn3vxIOCyXAKKyJyLK es/languages/livescript.min.js
sha384-NrQRnRP+ItgqOqxjc5lml+y913cYS5pG+mKH2P0J342fuZ21EcABNNJf547J+Cg1 es/languages/mathematica.min.js
sha384-TfALNLT6HJzZieazgsVvFM0DzFWQsgl0d7mdwPLyg1yg7XE4QwLY4jqmJRNnI1S4 languages/swift.min.js
sha384-rfwxAwuWzb2XdSU7HN3IhrSyCq96Nj4p1ZYPCNAGbqtnPsaWl8d5eSypxPbW6alT languages/typescript.min.js
sha384-a+qBlyBQMw6U8FrlXL2OodB7uR5S1diebujSfAM76WdYSl3ydnlxf8MCpGaiGgWG es/languages/swift.min.js
sha384-Mfjt0R07QBpLXhGWeCetLw7s2mTQSoXmcyYnfsSNq4V4YG3FwayBjxod9MxjSB1x es/languages/typescript.min.js
```
