"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn_resource_1 = require("./cfn-resource");
const removal_policy_1 = require("./removal-policy");
const resource_1 = require("./resource");
const token_1 = require("./token");
/**
 * Custom resource that is implemented using a Lambda
 *
 * As a custom resource author, you should be publishing a subclass of this class
 * that hides the choice of provider, and accepts a strongly-typed properties
 * object with the properties your provider accepts.
 *
 * @resource AWS::CloudFormation::CustomResource
 */
class CustomResource extends resource_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const type = renderResourceType(props.resourceType);
        const pascalCaseProperties = (_a = props.pascalCaseProperties) !== null && _a !== void 0 ? _a : false;
        const properties = pascalCaseProperties ? uppercaseProperties(props.properties || {}) : (props.properties || {});
        this.resource = new cfn_resource_1.CfnResource(this, 'Default', {
            type,
            properties: {
                ServiceToken: props.serviceToken,
                ...properties,
            },
        });
        this.resource.applyRemovalPolicy(props.removalPolicy, {
            default: removal_policy_1.RemovalPolicy.DESTROY,
        });
    }
    /**
     * The physical name of this custom resource.
     */
    get ref() {
        return this.resource.ref;
    }
    /**
     * Returns the value of an attribute of the custom resource of an arbitrary
     * type. Attributes are returned from the custom resource provider through the
     * `Data` map where the key is the attribute name.
     *
     * @param attributeName the name of the attribute
     * @returns a token for `Fn::GetAtt`. Use `Token.asXxx` to encode the returned `Reference` as a specific type or
     * use the convenience `getAttString` for string attributes.
     */
    getAtt(attributeName) {
        return this.resource.getAtt(attributeName);
    }
    /**
     * Returns the value of an attribute of the custom resource of type string.
     * Attributes are returned from the custom resource provider through the
     * `Data` map where the key is the attribute name.
     *
     * @param attributeName the name of the attribute
     * @returns a token for `Fn::GetAtt` encoded as a string.
     */
    getAttString(attributeName) {
        return token_1.Token.asString(this.getAtt(attributeName));
    }
}
exports.CustomResource = CustomResource;
/**
 * Uppercase the first letter of every property name
 *
 * It's customary for CloudFormation properties to start with capitals, and our
 * properties to start with lowercase, so this function translates from one
 * to the other
 */
function uppercaseProperties(props) {
    const ret = {};
    Object.keys(props).forEach(key => {
        const upper = key.substr(0, 1).toUpperCase() + key.substr(1);
        ret[upper] = props[key];
    });
    return ret;
}
function renderResourceType(resourceType) {
    if (!resourceType) {
        return 'AWS::CloudFormation::CustomResource';
    }
    if (!resourceType.startsWith('Custom::')) {
        throw new Error(`Custom resource type must begin with "Custom::" (${resourceType})`);
    }
    const typeName = resourceType.substr(resourceType.indexOf('::') + 2);
    if (typeName.length > 60) {
        throw new Error(`Custom resource type length > 60 (${resourceType})`);
    }
    if (!/^[a-z0-9_@-]+$/i.test(typeName)) {
        throw new Error(`Custom resource type name can only include alphanumeric characters and _@- (${typeName})`);
    }
    return resourceType;
}
//# sourceMappingURL=data:application/json;base64,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