from __future__ import annotations

import copy
from abc import ABC
from functools import partial

import optuna
import pandas as pd
from optuna.samplers import TPESampler
from optuna.trial import TrialState

from dataclr._evaluate import train_eval
from dataclr._typing import DataSplits
from dataclr.methods.method import Method
from dataclr.metrics import Metric, is_maximizing_metric, is_regression
from dataclr.metrics.metrics import CLASSIFICATION_METRICS, REGRESSION_METRICS
from dataclr.results import Result, ResultPerformance

optuna.logging.set_verbosity(optuna.logging.WARNING)


class FilterMethod(Method, ABC):
    """
    A base class for filter feature selection methods.

    Filter methods evaluate the relevance of features using statistical tests,
    correlations, or other criteria independent of a machine learning model.
    This class serves as the foundation for implementing specific filter-based
    feature selection algorithms.

    Inherits from:
        :class:`Method`: The base class that provides the structure for feature
                         selection methods.
    """

    def __init__(
        self, model, metric: Metric, n_results: int = 3, seed: int = 42
    ) -> None:
        super().__init__(model, metric, n_results, seed)

        self.ranked_features_: pd.Series[float] = pd.Series(dtype=float)

    def transform(
        self,
        X_train: pd.DataFrame,
        X_test: pd.DataFrame,
        y_train: pd.Series,
        y_test: pd.Series,
    ) -> list[Result]:
        """
        Returns results based on the ranked features determined during fitting.

        This method applies the feature selection process to the provided datasets
        using the rankings obtained from the `fit` method. It raises an error if
        `fit` has not been called beforehand.

        Args:
            X_train (pd.DataFrame): The training features.
            X_test (pd.DataFrame): The test features.
            y_train (pd.Series): The training target variable.
            y_test (pd.Series): The test target variable.

        Returns:
            list[Result]: A list of results generated by optimizing the feature subset
                          based on the ranked features.

        Raises:
            ValueError: If `fit` has not been called before `transform`.

        Notes:
            - The method relies on the `ranked_features_` attribute, which must be
              populated during the `fit` process.
            - Internally, it calls the `_optimize` method to perform the feature
              selection.

        See Also:
            - `fit`: Method to determine the ranked features.
            - `_optimize`: Internal method for feature subset optimization.
        """
        if self.ranked_features_.empty:
            raise ValueError("You need to call fit() before transform()!")

        data_splits: DataSplits = {
            "X_train": X_train,
            "y_train": y_train,
            "X_test": X_test,
            "y_test": y_test,
        }

        return self._optimize(
            data_splits=data_splits,
            sorted_list=self.ranked_features_,
            cached_performance={},
        )

    def _optimize(
        self,
        data_splits: DataSplits,
        sorted_list: pd.Series,
        cached_performance: dict[int, ResultPerformance],
        keep_features: list[str] = [],
    ) -> list[Result]:
        study = optuna.create_study(
            directions=[
                "maximize" if is_maximizing_metric(self.metric) else "minimize"
            ],
            sampler=TPESampler(seed=self.seed, n_startup_trials=5),
        )

        objective_with_params = partial(
            self._objective,
            sorted_list=sorted_list,
            data_splits=data_splits,
            cached_performance=cached_performance,
            keep_features=keep_features,
        )
        study.optimize(
            objective_with_params,
            n_trials=max(5, len(data_splits["X_test"].columns) // 4),
        )

        base_result = train_eval(self.model, self.metric, data_splits)
        base_sorted_features = sorted_list.index

        best_params = self._get_n_best_params(
            study=study,
            base_result=base_result,
            base_features=base_sorted_features,
            keep_features=keep_features,
        )

        return best_params

    def _get_results(
        self,
        data_splits: DataSplits,
        cached_performance: dict[int, ResultPerformance],
        keep_features: list[str] = [],
    ) -> list[Result]:
        try:
            self.fit(data_splits["X_train"], data_splits["y_train"])
        except ValueError:
            self.ranked_features_ = pd.Series(dtype=float)

        return self._optimize(
            data_splits=data_splits,
            sorted_list=self.ranked_features_,
            cached_performance=cached_performance,
            keep_features=keep_features,
        )

    def _objective(
        self,
        trial: optuna.Trial,
        sorted_list: pd.Series,
        data_splits: DataSplits,
        cached_performance: dict[int, ResultPerformance],
        keep_features: list[str] = [],
    ) -> float:
        filtered_list = sorted_list[~sorted_list.index.isin(keep_features)]

        k = trial.suggest_int("k", 1, max(len(filtered_list) - 1, 1))
        for previous_trial in trial.study.trials:
            if (
                previous_trial.state == TrialState.COMPLETE
                and trial.params == previous_trial.params
            ):
                if previous_trial.value is None:
                    return float("inf")
                columns = list(filtered_list[k:].index)
                columns.extend(keep_features)
                features_key = hash(tuple(columns))
                self._set_trial_attributes(trial, cached_performance[features_key])

                return previous_trial.value

        columns = list(filtered_list[k:].index)
        columns.extend(keep_features)

        new_data_splits = copy.deepcopy(data_splits)

        new_data_splits["X_train"] = data_splits["X_train"][columns]
        new_data_splits["X_test"] = data_splits["X_test"][columns]

        X_train = new_data_splits["X_train"]

        features_key = hash(tuple(X_train.columns))

        if features_key in cached_performance:
            self._set_trial_attributes(trial, cached_performance[features_key])

            value = cached_performance[features_key][self.metric]
            if value is None:
                raise ValueError("Could not retrieve criterion")

            return value

        self.total_combinations += 1

        performance_result = train_eval(self.model, self.metric, new_data_splits)

        cached_performance[features_key] = performance_result
        self._set_trial_attributes(trial, performance_result)

        value = performance_result[self.metric]
        if value is None:
            raise ValueError("Could not retrieve criterion")

        return value

    def _set_trial_attributes(
        self, trial: optuna.Trial, performance_result: ResultPerformance
    ) -> None:
        for attr in (
            REGRESSION_METRICS if is_regression(self.metric) else CLASSIFICATION_METRICS
        ):
            trial.set_user_attr(attr, getattr(performance_result, attr))

    def _get_n_best_params(
        self,
        study: optuna.Study,
        base_result: ResultPerformance,
        base_features: pd.Index,
        keep_features: list[str] = [],
    ) -> list[Result]:
        df: pd.DataFrame = study.trials_dataframe()

        tolerance = 1e-9
        result = base_result[self.metric]
        if result is None:
            raise ValueError("Base result is None!")

        if is_maximizing_metric(self.metric):
            better_values = df["value"] > result - tolerance
        else:
            better_values = df["value"] < result + tolerance

        selected_df = df[better_values].drop_duplicates(subset=["params_k"])
        sorted_df = selected_df.sort_values(
            "value", ascending=(not is_maximizing_metric(self.metric))
        )

        best_params = []
        base_features = list(base_features)

        if base_features:
            for feature in keep_features:
                base_features.remove(feature)
        for i in range(min(self.n_results, len(sorted_df))):
            best_trial = study.trials[sorted_df.index[i]]

            params = best_trial.params

            performance_metrics = {
                key: best_trial.user_attrs.get(key)
                for key in ["r2", "rmse", "accuracy", "precision", "recall", "f1"]
            }
            feature_list = base_features[params["k"] :]
            feature_list.extend(keep_features)
            best_params.append(
                Result(
                    params=params,
                    performance=ResultPerformance(**performance_metrics),
                    feature_list=feature_list,
                )
            )

        return best_params
