# Model: molecular_descriptors
#
# Description: The molecular_descriptors model uses RDKit and Mordred to compute a wide array of
#     molecular descriptors from SMILES strings. These descriptors quantify various chemical properties
#     and structural features, providing critical insights for cheminformatics applications such as
#     QSAR modeling and drug discovery.
#
import argparse
import os
import joblib
from io import StringIO
import pandas as pd

# Local imports
from chem_utils import compute_molecular_descriptors


# TRAINING SECTION
#
# This section (__main__) is where SageMaker will execute the training job
# and save the model artifacts to the model directory.
#
if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("--output-data-dir", type=str, default=os.environ["SM_OUTPUT_DATA_DIR"])
    parser.add_argument("--model-dir", type=str, default=os.environ["SM_MODEL_DIR"])
    parser.add_argument("--train", type=str, default=os.environ["SM_CHANNEL_TRAIN"])
    args = parser.parse_args()

    # This model doesn't get trained, it just a feature creation 'model'

    # Sagemaker seems to get upset if we don't save a model, so we'll create a placeholder model
    placeholder_model = {}
    joblib.dump(placeholder_model, os.path.join(args.model_dir, "model.joblib"))


# Model loading and prediction functions
def model_fn(model_dir):
    return joblib.load(os.path.join(model_dir, "model.joblib"))


# Input and output functions (basically convert CSV to DataFrame)
def input_fn(input_data, content_type):
    if content_type == "text/csv":
        return pd.read_csv(StringIO(input_data))
    raise ValueError(f"{content_type} not supported!")


def output_fn(output_df, accept_type):
    if accept_type == "text/csv":
        return output_df.to_csv(index=False), "text/csv"
    raise RuntimeError(f"{accept_type} not supported!")


# Prediction function
def predict_fn(df, model):

    # Compute the Molecular Descriptors
    df = compute_molecular_descriptors(df)
    return df
