"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const core_1 = require("@aws-solutions-constructs/core");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
const lambda = require("aws-cdk-lib/aws-lambda");
const defaults = require("@aws-solutions-constructs/core");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
// Note: All integration tests for alb are for HTTP APIs, as certificates require
// validation through DNS and email. This validation is impossible during our integration
// tests and the stack will fail to launch with an unvalidated certificate.
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, core_1.generateIntegStackName(__filename), {
    env: { account: aws_cdk_lib_1.Aws.ACCOUNT_ID, region: "us-east-1" },
});
stack.templateOptions.description =
    "Integration Test for public HTTP API with a existing function and ALB";
const myVpc = defaults.getTestVpc(stack);
const testSg = new aws_ec2_1.SecurityGroup(stack, "lambda-sg", {
    vpc: myVpc,
    allowAllOutbound: false,
});
const lambdaFunction = defaults.buildLambdaFunction(stack, {
    lambdaFunctionProps: {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_16_X,
        handler: "index.handler",
        vpc: myVpc,
        securityGroups: [testSg],
    },
});
const loadBalancer = defaults.ObtainAlb(stack, "existing-alb", {
    vpc: myVpc,
    publicApi: false,
    loadBalancerProps: {
        internetFacing: false,
        vpc: myVpc,
    },
    logAccessLogs: true,
    loggingBucketProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        autoDeleteObjects: true,
    }
});
const props = {
    existingLambdaObj: lambdaFunction,
    existingLoadBalancerObj: loadBalancer,
    existingVpc: myVpc,
    listenerProps: {
        protocol: "HTTP",
    },
    publicApi: true,
};
const albToLambda = new lib_1.AlbToLambda(stack, "test-one", props);
defaults.addCfnSuppressRules(albToLambda.listener, [
    {
        id: "W56",
        reason: "All integration tests must be HTTP because of certificate limitations.",
    },
]);
const newSecurityGroup = albToLambda.loadBalancer.connections.securityGroups[0]
    .node.defaultChild;
defaults.addCfnSuppressRules(newSecurityGroup, [
    { id: "W29", reason: "CDK created rule that blocks all traffic." },
    { id: "W2", reason: "Rule does not apply for ELB." },
    { id: "W9", reason: "Rule does not apply for ELB." },
]);
defaults.addCfnSuppressRules(testSg, [
    { id: "W29", reason: "CDK created rule that blocks all traffic." },
]);
defaults.suppressCustomHandlerCfnNagWarnings(stack, 'Custom::S3AutoDeleteObjectsCustomResourceProvider');
defaults.suppressCustomHandlerCfnNagWarnings(stack, 'Custom::VpcRestrictDefaultSGCustomResourceProvider');
// Synth
new integ_tests_alpha_1.IntegTest(stack, 'Integ', { testCases: [
        stack
    ] });
//# sourceMappingURL=data:application/json;base64,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