from functools import wraps
import inspect
import typing

import pydantic
from openai import OpenAI

# This is a global variable that will store the OpenAI client. This
# enables any user to set the client  under their own terms (key, project...)
# via `gpt_function.SETTINGS["openai_client"] = OpenAI()`.
# If the user doesn't set up the client, it gets set up automatically with
# OpenAI() the first time a client is needed
SETTINGS = {"openai_client": None}

ADDITIONAL_DOCS = """

Function auto-generated by @gpt_function.
- The execution happens on a chatbot, and may require an API key.
- Answers can be changing, their quality and validity are not guaranteed.
- The following auto-added parameters can control the GPT execution:

Parameters
-----------

gpt_model: str
    The OpenAI model, current choices are gpt-4o-mini, gpt-4o-2024-08-06, and later.
    `gpt-4o-mini` is faster, and 10 times less expensive, but the answers can be
    less good.

gpt_system_prompt: Optional[str]
    Additional prompt to be added before the user's prompt. This can be used
"""


def gpt_function(func):
    """Decorator that runs a function on a GPT model."""

    requested_format = typing.get_type_hints(func).get("return", str)
    if not isinstance(requested_format, pydantic.BaseModel):

        class __Response(pydantic.BaseModel):
            response: requested_format

        requested_format = __Response

    @add_kwargs(gpt_model="gpt-4o-mini", gpt_system_prompt=None)
    @wraps(func)  # This preserves the docstring and other attributes
    def wrapper(*args, **kwargs):
        # Get and remove parameters used by the wrapper only
        gpt_model = kwargs.pop("gpt_model")
        gpt_system_prompt = kwargs.pop("gpt_system_prompt")

        # Build the prompt: format the docstring with the function's arguments
        # Step 1: Name the args that weren't named:
        args_names = [
            name
            for name, param in inspect.signature(func).parameters.items()
            if param.default == inspect.Parameter.empty
        ]
        named_args = dict(zip(args_names, args))

        all_named_args = {**named_args, **kwargs}

        # Step 2: Add the values un-specified kwargs from func
        unspecified_kwargs = {
            name: param.default
            for name, param in inspect.signature(func).parameters.items()
            if param.default != inspect.Parameter.empty and name not in all_named_args
        }
        prompt = func.__doc__.format(**all_named_args, **unspecified_kwargs)

        system_prompt = "Answer using the provided output schema."
        if gpt_system_prompt:
            system_prompt = gpt_system_prompt + "\n" + system_prompt
        gpt_messages = [
            {"role": "system", "content": system_prompt},
            {"role": "user", "content": prompt},
        ]

        # Initiate the global GPT client if it's not already set
        client = SETTINGS["openai_client"]
        if client is None:
            client = SETTINGS["openai_client"] = OpenAI()

        # Query the GPT model, extract the response
        response = client.beta.chat.completions.parse(
            messages=gpt_messages, model=gpt_model, response_format=requested_format
        )
        formatted_response = response.choices[0].message.parsed
        if formatted_response.__class__.__name__ == "__Response":
            formatted_response = formatted_response.response

        return formatted_response

    # Add a text to the docstring so it will be clear to users that the
    # function is actually running on a chatbot.
    wrapper.__doc__ += ADDITIONAL_DOCS
    return wrapper


def add_kwargs(**new_kwargs):
    """Decorator to add keyword arguments to a function's signature."""

    def decorator(func):
        # Get the original function's signature
        sig = inspect.signature(func)
        params = list(sig.parameters.values())

        # Add new kwargs to the parameters
        parameter_type = inspect.Parameter.POSITIONAL_OR_KEYWORD
        keyword_only = inspect.Parameter.KEYWORD_ONLY
        if any([param.kind == keyword_only for param in params]):
            parameter_type = keyword_only

        new_params = [
            inspect.Parameter(name, parameter_type, default=value)
            for name, value in new_kwargs.items()
        ]

        # Update the function's signature
        params.extend(new_params)
        new_sig = sig.replace(parameters=params)

        @wraps(func)
        def wrapper(*args, **kwargs):
            # The new kwargs are already added to the signature, so they will
            # be available in kwargs when the function is called.
            kwargs = {**new_kwargs, **kwargs}
            return func(*args, **kwargs)

        # Update the signature of the wrapper function
        wrapper.__signature__ = new_sig

        return wrapper

    return decorator


def ReasonedAnswer(T) -> pydantic.BaseModel:
    class _ReasonedAnswer(pydantic.BaseModel):
        reasoning: str
        result: T

        def __str__(self):
            return f"{self.result}\n\nGPT reasoning: {self.reasoning}"

    return _ReasonedAnswer
