"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const lazy_1 = require("../lazy");
const resolvable_1 = require("../resolvable");
const token_1 = require("../token");
const intrinsic_1 = require("./intrinsic");
const resolve_1 = require("./resolve");
/**
 * Routines that know how to do operations at the CloudFormation document language level
 */
class CloudFormationLang {
    /**
     * Turn an arbitrary structure potentially containing Tokens into a JSON string.
     *
     * Returns a Token which will evaluate to CloudFormation expression that
     * will be evaluated by CloudFormation to the JSON representation of the
     * input structure.
     *
     * All Tokens substituted in this way must return strings, or the evaluation
     * in CloudFormation will fail.
     *
     * @param obj The object to stringify
     * @param space Indentation to use (default: no pretty-printing)
     */
    static toJSON(obj, space) {
        // This works in two stages:
        //
        // First, resolve everything. This gets rid of the lazy evaluations, evaluation
        // to the real types of things (for example, would a function return a string, an
        // intrinsic, or a number? We have to resolve to know).
        //
        // We then to through the returned result, identify things that evaluated to
        // CloudFormation intrinsics, and re-wrap those in Tokens that have a
        // toJSON() method returning their string representation. If we then call
        // JSON.stringify() on that result, that gives us essentially the same
        // string that we started with, except with the non-token characters quoted.
        //
        //    {"field": "${TOKEN}"} --> {\"field\": \"${TOKEN}\"}
        //
        // A final resolve() on that string (done by the framework) will yield the string
        // we're after.
        //
        // Resolving and wrapping are done in go using the resolver framework.
        class IntrinsincWrapper extends resolvable_1.DefaultTokenResolver {
            constructor() {
                super(CLOUDFORMATION_CONCAT);
            }
            resolveToken(t, context, postProcess) {
                return wrap(super.resolveToken(t, context, postProcess));
            }
            resolveString(fragments, context) {
                return wrap(super.resolveString(fragments, context));
            }
            resolveList(l, context) {
                return wrap(super.resolveList(l, context));
            }
        }
        // We need a ResolveContext to get started so return a Token
        return lazy_1.Lazy.stringValue({ produce: (ctx) => JSON.stringify(resolve_1.resolve(obj, {
                scope: ctx.scope,
                resolver: new IntrinsincWrapper()
            }), undefined, space)
        });
        function wrap(value) {
            return isIntrinsic(value) ? new JsonToken(deepQuoteStringsForJSON(value)) : value;
        }
    }
    /**
     * Produce a CloudFormation expression to concat two arbitrary expressions when resolving
     */
    static concat(left, right) {
        if (left === undefined && right === undefined) {
            return '';
        }
        const parts = new Array();
        if (left !== undefined) {
            parts.push(left);
        }
        if (right !== undefined) {
            parts.push(right);
        }
        // Some case analysis to produce minimal expressions
        if (parts.length === 1) {
            return parts[0];
        }
        if (parts.length === 2 && typeof parts[0] === 'string' && typeof parts[1] === 'string') {
            return parts[0] + parts[1];
        }
        // Otherwise return a Join intrinsic (already in the target document language to avoid taking
        // circular dependencies on FnJoin & friends)
        return { 'Fn::Join': ['', minimalCloudFormationJoin('', parts)] };
    }
}
exports.CloudFormationLang = CloudFormationLang;
/**
 * Token that also stringifies in the toJSON() operation.
 */
class JsonToken extends intrinsic_1.Intrinsic {
    /**
     * Special handler that gets called when JSON.stringify() is used.
     */
    toJSON() {
        return this.toString();
    }
}
/**
 * Deep escape strings for use in a JSON context
 */
function deepQuoteStringsForJSON(x) {
    if (typeof x === 'string') {
        // Whenever we escape a string we strip off the outermost quotes
        // since we're already in a quoted context.
        const stringified = JSON.stringify(x);
        return stringified.substring(1, stringified.length - 1);
    }
    if (Array.isArray(x)) {
        return x.map(deepQuoteStringsForJSON);
    }
    if (typeof x === 'object') {
        for (const key of Object.keys(x)) {
            x[key] = deepQuoteStringsForJSON(x[key]);
        }
    }
    return x;
}
const CLOUDFORMATION_CONCAT = {
    join(left, right) {
        return CloudFormationLang.concat(left, right);
    }
};
/**
 * Default Token resolver for CloudFormation templates
 */
exports.CLOUDFORMATION_TOKEN_RESOLVER = new resolvable_1.DefaultTokenResolver(CLOUDFORMATION_CONCAT);
/**
 * Do an intelligent CloudFormation join on the given values, producing a minimal expression
 */
function minimalCloudFormationJoin(delimiter, values) {
    let i = 0;
    while (i < values.length) {
        const el = values[i];
        if (isSplicableFnJoinIntrinsic(el)) {
            values.splice(i, 1, ...el['Fn::Join'][1]);
        }
        else if (i > 0 && isPlainString(values[i - 1]) && isPlainString(values[i])) {
            values[i - 1] += delimiter + values[i];
            values.splice(i, 1);
        }
        else {
            i += 1;
        }
    }
    return values;
    function isPlainString(obj) {
        return typeof obj === 'string' && !token_1.Token.isUnresolved(obj);
    }
    function isSplicableFnJoinIntrinsic(obj) {
        return isIntrinsic(obj)
            && Object.keys(obj)[0] === 'Fn::Join'
            && obj['Fn::Join'][0] === delimiter;
    }
}
exports.minimalCloudFormationJoin = minimalCloudFormationJoin;
/**
 * Return whether the given value represents a CloudFormation intrinsic
 */
function isIntrinsic(x) {
    if (Array.isArray(x) || x === null || typeof x !== 'object') {
        return false;
    }
    const keys = Object.keys(x);
    if (keys.length !== 1) {
        return false;
    }
    return keys[0] === 'Ref' || isNameOfCloudFormationIntrinsic(keys[0]);
}
function isNameOfCloudFormationIntrinsic(name) {
    if (!name.startsWith('Fn::')) {
        return false;
    }
    // these are 'fake' intrinsics, only usable inside the parameter overrides of a CFN CodePipeline Action
    return name !== 'Fn::GetArtifactAtt' && name !== 'Fn::GetParam';
}
exports.isNameOfCloudFormationIntrinsic = isNameOfCloudFormationIntrinsic;
//# sourceMappingURL=data:application/json;base64,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