"use strict";
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
const app_1 = require("../lib/app");
function withApp(props, block) {
    const app = new app_1.App({
        runtimeInfo: false,
        stackTraces: false,
        ...props,
    });
    block(app);
    return app.synth();
}
function synth(context) {
    return withApp({ context }, app => {
        const stack1 = new lib_1.Stack(app, 'stack1', { env: { account: '12345', region: 'us-east-1' } });
        new lib_1.CfnResource(stack1, 's1c1', { type: 'DummyResource', properties: { Prop1: 'Prop1' } });
        const r2 = new lib_1.CfnResource(stack1, 's1c2', { type: 'DummyResource', properties: { Foo: 123 } });
        const stack2 = new lib_1.Stack(app, 'stack2');
        new lib_1.CfnResource(stack2, 's2c1', { type: 'DummyResource', properties: { Prog2: 'Prog2' } });
        const c1 = new MyConstruct(stack2, 's1c2');
        // add some metadata
        stack1.node.addMetadata('meta', 111);
        r2.node.addWarning('warning1');
        r2.node.addWarning('warning2');
        c1.node.addMetadata('meta', { key: 'value' });
        app.node.addMetadata('applevel', 123); // apps can also have metadata
    });
}
function synthStack(name, includeMetadata = false, context) {
    const response = synth(context);
    const stack = response.getStack(name);
    if (!includeMetadata) {
        delete stack.metadata;
    }
    return stack;
}
class MyConstruct extends lib_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        new lib_1.CfnResource(this, 'r1', { type: 'ResourceType1' });
        new lib_1.CfnResource(this, 'r2', { type: 'ResourceType2', properties: { FromContext: this.node.tryGetContext('ctx1') } });
    }
}
module.exports = {
    'synthesizes all stacks and returns synthesis result'(test) {
        const response = synth();
        delete response.dir;
        test.deepEqual(response.stacks.length, 2);
        const stack1 = response.stacks[0];
        test.deepEqual(stack1.name, 'stack1');
        test.deepEqual(stack1.environment.account, 12345);
        test.deepEqual(stack1.environment.region, 'us-east-1');
        test.deepEqual(stack1.environment.name, 'aws://12345/us-east-1');
        test.deepEqual(stack1.template, { Resources: { s1c1: { Type: 'DummyResource', Properties: { Prop1: 'Prop1' } },
                s1c2: { Type: 'DummyResource', Properties: { Foo: 123 } } } });
        test.deepEqual(stack1.manifest.metadata, {
            '/stack1': [{ type: 'meta', data: 111 }],
            '/stack1/s1c1': [{ type: 'aws:cdk:logicalId', data: 's1c1' }],
            '/stack1/s1c2': [{ type: 'aws:cdk:logicalId', data: 's1c2' },
                { type: 'aws:cdk:warning', data: 'warning1' },
                { type: 'aws:cdk:warning', data: 'warning2' }],
        });
        const stack2 = response.stacks[1];
        test.deepEqual(stack2.name, 'stack2');
        test.deepEqual(stack2.environment.name, 'aws://unknown-account/unknown-region');
        test.deepEqual(stack2.template, { Resources: { s2c1: { Type: 'DummyResource', Properties: { Prog2: 'Prog2' } },
                s1c2r1D1791C01: { Type: 'ResourceType1' },
                s1c2r25F685FFF: { Type: 'ResourceType2' } } });
        test.deepEqual(stack2.manifest.metadata, {
            '/stack2/s2c1': [{ type: 'aws:cdk:logicalId', data: 's2c1' }],
            '/stack2/s1c2': [{ type: 'meta', data: { key: 'value' } }],
            '/stack2/s1c2/r1': [{ type: 'aws:cdk:logicalId', data: 's1c2r1D1791C01' }],
            '/stack2/s1c2/r2': [{ type: 'aws:cdk:logicalId', data: 's1c2r25F685FFF' }],
        });
        test.done();
    },
    'context can be passed through CDK_CONTEXT'(test) {
        process.env[cxapi.CONTEXT_ENV] = JSON.stringify({
            key1: 'val1',
            key2: 'val2'
        });
        const prog = new app_1.App();
        test.deepEqual(prog.node.tryGetContext('key1'), 'val1');
        test.deepEqual(prog.node.tryGetContext('key2'), 'val2');
        test.done();
    },
    'context from the command line can be used when creating the stack'(test) {
        const output = synthStack('stack2', false, { ctx1: 'HELLO' });
        test.deepEqual(output.template, {
            Resources: {
                s2c1: {
                    Type: "DummyResource",
                    Properties: {
                        Prog2: "Prog2"
                    }
                },
                s1c2r1D1791C01: {
                    Type: "ResourceType1"
                },
                s1c2r25F685FFF: {
                    Type: "ResourceType2",
                    Properties: {
                        FromContext: "HELLO"
                    }
                }
            }
        });
        test.done();
    },
    'setContext(k,v) can be used to set context programmatically'(test) {
        const prog = new app_1.App();
        prog.node.setContext('foo', 'bar');
        test.deepEqual(prog.node.tryGetContext('foo'), 'bar');
        test.done();
    },
    'setContext(k,v) cannot be called after stacks have been added because stacks may use the context'(test) {
        const prog = new app_1.App();
        new lib_1.Stack(prog, 's1');
        test.throws(() => prog.node.setContext('foo', 'bar'));
        test.done();
    },
    'app.synth() performs validation first and if there are errors, it returns the errors'(test) {
        class Child extends lib_1.Construct {
            validate() {
                return [`Error from ${this.node.id}`];
            }
        }
        class Parent extends lib_1.Stack {
        }
        const app = new app_1.App();
        const parent = new Parent(app, 'Parent');
        new Child(parent, 'C1');
        new Child(parent, 'C2');
        test.throws(() => app.synth(), /Validation failed with the following errors/);
        test.done();
    },
    'app.synthesizeStack(stack) will return a list of missing contextual information'(test) {
        class MyStack extends lib_1.Stack {
            constructor(scope, id, props) {
                super(scope, id, props);
                this.reportMissingContext({
                    key: 'missing-context-key',
                    provider: 'fake',
                    props: {
                        account: '12345689012',
                        region: 'ab-north-1',
                    },
                });
                this.reportMissingContext({
                    key: 'missing-context-key-2',
                    provider: 'fake2',
                    props: {
                        foo: 'bar',
                        account: '12345689012',
                        region: 'ab-south-1',
                    },
                });
            }
        }
        const assembly = withApp({}, app => {
            new MyStack(app, 'MyStack');
        });
        test.deepEqual(assembly.manifest.missing, [
            {
                key: "missing-context-key",
                provider: 'fake',
                props: {
                    account: '12345689012',
                    region: 'ab-north-1',
                },
            },
            {
                key: "missing-context-key-2",
                provider: 'fake2',
                props: {
                    account: '12345689012',
                    region: 'ab-south-1',
                    foo: 'bar',
                },
            }
        ]);
        test.done();
    },
    'runtime library versions disabled'(test) {
        const context = {};
        context[cxapi.DISABLE_VERSION_REPORTING] = true;
        const assembly = withApp(context, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        test.deepEqual(assembly.runtime, { libraries: {} });
        test.done();
    },
    'runtime library versions'(test) {
        const response = withApp({ runtimeInfo: true }, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        const libs = (response.runtime && response.runtime.libraries) || {};
        const version = require('../package.json').version;
        test.deepEqual(libs['@aws-cdk/core'], version);
        test.deepEqual(libs['@aws-cdk/cx-api'], version);
        test.deepEqual(libs['jsii-runtime'], `node.js/${process.version}`);
        test.done();
    },
    'jsii-runtime version loaded from JSII_AGENT'(test) {
        process.env.JSII_AGENT = 'Java/1.2.3.4';
        const response = withApp({ runtimeInfo: true }, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        const libs = (response.runtime && response.runtime.libraries) || {};
        test.deepEqual(libs['jsii-runtime'], `Java/1.2.3.4`);
        delete process.env.JSII_AGENT;
        test.done();
    },
    'version reporting includes only @aws-cdk, aws-cdk and jsii libraries'(test) {
        const response = withApp({ runtimeInfo: true }, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        const libs = (response.runtime && response.runtime.libraries) || {};
        const version = require('../package.json').version;
        test.deepEqual(libs, {
            '@aws-cdk/core': version,
            '@aws-cdk/cx-api': version,
            'jsii-runtime': `node.js/${process.version}`
        });
        test.done();
    },
    'deep stack is shown and synthesized properly'(test) {
        // WHEN
        const response = withApp({}, (app) => {
            const topStack = new lib_1.Stack(app, 'Stack');
            const topResource = new lib_1.CfnResource(topStack, 'Res', { type: 'CDK::TopStack::Resource' });
            const bottomStack = new lib_1.Stack(topResource, 'Stack');
            new lib_1.CfnResource(bottomStack, 'Res', { type: 'CDK::BottomStack::Resource' });
        });
        // THEN
        test.deepEqual(response.stacks.map(s => ({ name: s.name, template: s.template })), [
            {
                name: 'Stack',
                template: { Resources: { Res: { Type: 'CDK::TopStack::Resource' } } },
            },
            {
                name: 'StackResStack7E4AFA86',
                template: { Resources: { Res: { Type: 'CDK::BottomStack::Resource' } } },
            },
        ]);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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