"use strict";
const cdk = require("@aws-cdk/core");
const util_1 = require("../lib/util");
module.exports = {
    parseBucketArn: {
        'explicit arn'(test) {
            const stack = new cdk.Stack();
            const bucketArn = 'my:bucket:arn';
            test.deepEqual(util_1.parseBucketArn(stack, { bucketArn }), bucketArn);
            test.done();
        },
        'produce arn from bucket name'(test) {
            const stack = new cdk.Stack();
            const bucketName = 'hello';
            test.deepEqual(stack.resolve(util_1.parseBucketArn(stack, { bucketName })), {
                'Fn::Join': ['',
                    ['arn:',
                        { Ref: 'AWS::Partition' },
                        ':s3:::hello']]
            });
            test.done();
        },
        'fails if neither arn nor name are provided'(test) {
            const stack = new cdk.Stack();
            test.throws(() => util_1.parseBucketArn(stack, {}), /Cannot determine bucket ARN. At least `bucketArn` or `bucketName` is needed/);
            test.done();
        }
    },
    parseBucketName: {
        'explicit name'(test) {
            const stack = new cdk.Stack();
            const bucketName = 'foo';
            test.deepEqual(stack.resolve(util_1.parseBucketName(stack, { bucketName })), 'foo');
            test.done();
        },
        'extract bucket name from string arn'(test) {
            const stack = new cdk.Stack();
            const bucketArn = 'arn:aws:s3:::my-bucket';
            test.deepEqual(stack.resolve(util_1.parseBucketName(stack, { bucketArn })), 'my-bucket');
            test.done();
        },
        'can parse bucket name even if it contains a token'(test) {
            const stack = new cdk.Stack();
            const bucketArn = `arn:aws:s3:::${cdk.Token.asString({ Ref: 'my-bucket' })}`;
            test.deepEqual(stack.resolve(util_1.parseBucketName(stack, { bucketArn })), {
                "Fn::Select": [
                    0,
                    {
                        "Fn::Split": [
                            "/",
                            {
                                "Fn::Select": [
                                    5,
                                    {
                                        "Fn::Split": [
                                            ":",
                                            {
                                                "Fn::Join": [
                                                    "",
                                                    [
                                                        "arn:aws:s3:::",
                                                        {
                                                            Ref: "my-bucket"
                                                        }
                                                    ]
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    }
                ]
            });
            test.done();
        },
        'fails if ARN has invalid format'(test) {
            const stack = new cdk.Stack();
            const bucketArn = 'invalid-arn';
            test.throws(() => util_1.parseBucketName(stack, { bucketArn }), /ARNs must have at least 6 components/);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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