"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnParser = exports.CfnParsingContext = exports.FromCloudFormation = void 0;
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cfn_resource_policy_1 = require("./cfn-resource-policy");
const lazy_1 = require("./lazy");
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
/**
 * This class contains static methods called when going from
 * translated values received from {@link CfnParser.parseValue}
 * to the actual L1 properties -
 * things like changing IResolvable to the appropriate type
 * (string, string array, or number), etc.
 *
 * While this file not exported from the module
 * (to not make it part of the public API),
 * it is directly referenced in the generated L1 code.
 *
 * @experimental
 */
class FromCloudFormation {
    // nothing to for any but return it
    static getAny(value) { return value; }
    static getBoolean(value) {
        if (typeof value === 'string') {
            // CloudFormation allows passing strings as boolean
            switch (value) {
                case 'true': return true;
                case 'false': return false;
                default: throw new Error(`Expected 'true' or 'false' for boolean value, got: '${value}'`);
            }
        }
        // in all other cases, just return the value,
        // and let a validator handle if it's not a boolean
        return value;
    }
    static getDate(value) {
        // if the date is a deploy-time value, just return it
        if (token_1.isResolvableObject(value)) {
            return value;
        }
        // if the date has been given as a string, convert it
        if (typeof value === 'string') {
            return new Date(value);
        }
        // all other cases - just return the value,
        // if it's not a Date, a validator should catch it
        return value;
    }
    // won't always return a string; if the input can't be resolved to a string,
    // the input will be returned.
    static getString(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return value.toString();
        }
        // CloudFormation treats numbers and strings interchangeably;
        // so, if we get a number here, convert it to a string
        if (typeof value === 'number') {
            return value.toString();
        }
        // in all other cases, just return the input,
        // and let a validator handle it if it's not a string
        return value;
    }
    // won't always return a number; if the input can't be parsed to a number,
    // the input will be returned.
    static getNumber(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asNumber(value);
        }
        // return a number, if the input can be parsed as one
        if (typeof value === 'string') {
            const parsedValue = parseFloat(value);
            if (!isNaN(parsedValue)) {
                return parsedValue;
            }
        }
        // otherwise return the input,
        // and let a validator handle it if it's not a number
        return value;
    }
    static getStringArray(value) {
        // if the array is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asList(value);
        }
        // in all other cases, delegate to the standard mapping logic
        return this.getArray(this.getString)(value);
    }
    static getArray(mapper) {
        return (value) => {
            if (!Array.isArray(value)) {
                // break the type system, and just return the given value,
                // which hopefully will be reported as invalid by the validator
                // of the property we're transforming
                // (unless it's a deploy-time value,
                // which we can't map over at build time anyway)
                return value;
            }
            return value.map(mapper);
        };
    }
    static getMap(mapper) {
        return (value) => {
            if (typeof value !== 'object') {
                // if the input is not a map (= object in JS land),
                // just return it, and let the validator of this property handle it
                // (unless it's a deploy-time value,
                // which we can't map over at build time anyway)
                return value;
            }
            const ret = {};
            for (const [key, val] of Object.entries(value)) {
                ret[key] = mapper(val);
            }
            return ret;
        };
    }
    static getCfnTag(tag) {
        return tag == null
            ? {} // break the type system - this should be detected at runtime by a tag validator
            : {
                key: tag.Key,
                value: tag.Value,
            };
    }
    /**
     * Return a function that, when applied to a value, will return the first validly deserialized one
     */
    static getTypeUnion(validators, mappers) {
        return (value) => {
            for (let i = 0; i < validators.length; i++) {
                const candidate = mappers[i](value);
                if (validators[i](candidate).isSuccess) {
                    return candidate;
                }
            }
            // if nothing matches, just return the input unchanged, and let validators catch it
            return value;
        };
    }
}
exports.FromCloudFormation = FromCloudFormation;
/**
 * The context in which the parsing is taking place.
 *
 * Some fragments of CloudFormation templates behave differently than others
 * (for example, the 'Conditions' sections treats { "Condition": "NameOfCond" }
 * differently than the 'Resources' section).
 * This enum can be used to change the created {@link CfnParser} behavior,
 * based on the template context.
 */
var CfnParsingContext;
(function (CfnParsingContext) {
    /** We're currently parsing the 'Conditions' section. */
    CfnParsingContext[CfnParsingContext["CONDITIONS"] = 0] = "CONDITIONS";
    /** We're currently parsing the 'Rules' section. */
    CfnParsingContext[CfnParsingContext["RULES"] = 1] = "RULES";
})(CfnParsingContext = exports.CfnParsingContext || (exports.CfnParsingContext = {}));
/**
 * This class contains methods for translating from a pure CFN value
 * (like a JS object { "Ref": "Bucket" })
 * to a form CDK understands
 * (like Fn.ref('Bucket')).
 *
 * While this file not exported from the module
 * (to not make it part of the public API),
 * it is directly referenced in the generated L1 code,
 * so any renames of it need to be reflected in cfn2ts/codegen.ts as well.
 *
 * @experimental
 */
class CfnParser {
    constructor(options) {
        this.options = options;
    }
    handleAttributes(resource, resourceAttributes, logicalId) {
        var _a;
        const cfnOptions = resource.cfnOptions;
        cfnOptions.creationPolicy = this.parseCreationPolicy(resourceAttributes.CreationPolicy);
        cfnOptions.updatePolicy = this.parseUpdatePolicy(resourceAttributes.UpdatePolicy);
        cfnOptions.deletionPolicy = this.parseDeletionPolicy(resourceAttributes.DeletionPolicy);
        cfnOptions.updateReplacePolicy = this.parseDeletionPolicy(resourceAttributes.UpdateReplacePolicy);
        cfnOptions.version = this.parseValue(resourceAttributes.Version);
        cfnOptions.metadata = this.parseValue(resourceAttributes.Metadata);
        // handle Condition
        if (resourceAttributes.Condition) {
            const condition = this.finder.findCondition(resourceAttributes.Condition);
            if (!condition) {
                throw new Error(`Resource '${logicalId}' uses Condition '${resourceAttributes.Condition}' that doesn't exist`);
            }
            cfnOptions.condition = condition;
        }
        // handle DependsOn
        resourceAttributes.DependsOn = (_a = resourceAttributes.DependsOn) !== null && _a !== void 0 ? _a : [];
        const dependencies = Array.isArray(resourceAttributes.DependsOn) ?
            resourceAttributes.DependsOn : [resourceAttributes.DependsOn];
        for (const dep of dependencies) {
            const depResource = this.finder.findResource(dep);
            if (!depResource) {
                throw new Error(`Resource '${logicalId}' depends on '${dep}' that doesn't exist`);
            }
            resource.node.addDependency(depResource);
        }
    }
    parseCreationPolicy(policy) {
        if (typeof policy !== 'object') {
            return undefined;
        }
        // change simple JS values to their CDK equivalents
        policy = this.parseValue(policy);
        return undefinedIfAllValuesAreEmpty({
            autoScalingCreationPolicy: parseAutoScalingCreationPolicy(policy.AutoScalingCreationPolicy),
            resourceSignal: parseResourceSignal(policy.ResourceSignal),
        });
        function parseAutoScalingCreationPolicy(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                minSuccessfulInstancesPercent: FromCloudFormation.getNumber(p.MinSuccessfulInstancesPercent),
            });
        }
        function parseResourceSignal(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                count: FromCloudFormation.getNumber(p.Count),
                timeout: FromCloudFormation.getString(p.Timeout),
            });
        }
    }
    parseUpdatePolicy(policy) {
        if (typeof policy !== 'object') {
            return undefined;
        }
        // change simple JS values to their CDK equivalents
        policy = this.parseValue(policy);
        return undefinedIfAllValuesAreEmpty({
            autoScalingReplacingUpdate: parseAutoScalingReplacingUpdate(policy.AutoScalingReplacingUpdate),
            autoScalingRollingUpdate: parseAutoScalingRollingUpdate(policy.AutoScalingRollingUpdate),
            autoScalingScheduledAction: parseAutoScalingScheduledAction(policy.AutoScalingScheduledAction),
            codeDeployLambdaAliasUpdate: parseCodeDeployLambdaAliasUpdate(policy.CodeDeployLambdaAliasUpdate),
            enableVersionUpgrade: FromCloudFormation.getBoolean(policy.EnableVersionUpgrade),
            useOnlineResharding: FromCloudFormation.getBoolean(policy.UseOnlineResharding),
        });
        function parseAutoScalingReplacingUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                willReplace: p.WillReplace,
            });
        }
        function parseAutoScalingRollingUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                maxBatchSize: FromCloudFormation.getNumber(p.MaxBatchSize),
                minInstancesInService: FromCloudFormation.getNumber(p.MinInstancesInService),
                minSuccessfulInstancesPercent: FromCloudFormation.getNumber(p.MinSuccessfulInstancesPercent),
                pauseTime: FromCloudFormation.getString(p.PauseTime),
                suspendProcesses: FromCloudFormation.getStringArray(p.SuspendProcesses),
                waitOnResourceSignals: FromCloudFormation.getBoolean(p.WaitOnResourceSignals),
            });
        }
        function parseCodeDeployLambdaAliasUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return {
                beforeAllowTrafficHook: FromCloudFormation.getString(p.BeforeAllowTrafficHook),
                afterAllowTrafficHook: FromCloudFormation.getString(p.AfterAllowTrafficHook),
                applicationName: FromCloudFormation.getString(p.ApplicationName),
                deploymentGroupName: FromCloudFormation.getString(p.DeploymentGroupName),
            };
        }
        function parseAutoScalingScheduledAction(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                ignoreUnmodifiedGroupSizeProperties: p.IgnoreUnmodifiedGroupSizeProperties,
            });
        }
    }
    parseDeletionPolicy(policy) {
        switch (policy) {
            case null: return undefined;
            case undefined: return undefined;
            case 'Delete': return cfn_resource_policy_1.CfnDeletionPolicy.DELETE;
            case 'Retain': return cfn_resource_policy_1.CfnDeletionPolicy.RETAIN;
            case 'Snapshot': return cfn_resource_policy_1.CfnDeletionPolicy.SNAPSHOT;
            default: throw new Error(`Unrecognized DeletionPolicy '${policy}'`);
        }
    }
    parseValue(cfnValue) {
        // == null captures undefined as well
        if (cfnValue == null) {
            return undefined;
        }
        // if we have any late-bound values,
        // just return them
        if (token_1.isResolvableObject(cfnValue)) {
            return cfnValue;
        }
        if (Array.isArray(cfnValue)) {
            return cfnValue.map(el => this.parseValue(el));
        }
        if (typeof cfnValue === 'object') {
            // an object can be either a CFN intrinsic, or an actual object
            const cfnIntrinsic = this.parseIfCfnIntrinsic(cfnValue);
            if (cfnIntrinsic !== undefined) {
                return cfnIntrinsic;
            }
            const ret = {};
            for (const [key, val] of Object.entries(cfnValue)) {
                ret[key] = this.parseValue(val);
            }
            return ret;
        }
        // in all other cases, just return the input
        return cfnValue;
    }
    get finder() {
        return this.options.finder;
    }
    parseIfCfnIntrinsic(object) {
        const key = this.looksLikeCfnIntrinsic(object);
        switch (key) {
            case undefined:
                return undefined;
            case 'Ref': {
                const refTarget = object[key];
                const specialRef = this.specialCaseRefs(refTarget);
                if (specialRef !== undefined) {
                    return specialRef;
                }
                else {
                    const refElement = this.finder.findRefTarget(refTarget);
                    if (!refElement) {
                        throw new Error(`Element used in Ref expression with logical ID: '${refTarget}' not found`);
                    }
                    return cfn_reference_1.CfnReference.for(refElement, 'Ref');
                }
            }
            case 'Fn::GetAtt': {
                // Fn::GetAtt takes a 2-element list as its argument
                const value = object[key];
                const target = this.finder.findResource(value[0]);
                if (!target) {
                    throw new Error(`Resource used in GetAtt expression with logical ID: '${value[0]}' not found`);
                }
                return target.getAtt(value[1]);
            }
            case 'Fn::Join': {
                // Fn::Join takes a 2-element list as its argument,
                // where the first element is the delimiter,
                // and the second is the list of elements to join
                const value = this.parseValue(object[key]);
                // wrap the array as a Token,
                // as otherwise Fn.join() will try to concatenate
                // the non-token parts,
                // causing a diff with the original template
                return cfn_fn_1.Fn.join(value[0], lazy_1.Lazy.listValue({ produce: () => value[1] }));
            }
            case 'Fn::Cidr': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.cidr(value[0], value[1], value[2]);
            }
            case 'Fn::FindInMap': {
                const value = this.parseValue(object[key]);
                // the first argument to FindInMap is the mapping name
                const mapping = this.finder.findMapping(value[0]);
                if (!mapping) {
                    throw new Error(`Mapping used in FindInMap expression with name '${value[0]}' was not found in the template`);
                }
                return cfn_fn_1.Fn.findInMap(mapping.logicalId, value[1], value[2]);
            }
            case 'Fn::Select': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.select(value[0], value[1]);
            }
            case 'Fn::GetAZs': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.getAzs(value);
            }
            case 'Fn::ImportValue': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.importValue(value);
            }
            case 'Fn::Split': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.split(value[0], value[1]);
            }
            case 'Fn::Transform': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.transform(value.Name, value.Parameters);
            }
            case 'Fn::Base64': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.base64(value);
            }
            case 'Fn::If': {
                // Fn::If takes a 3-element list as its argument,
                // where the first element is the name of a Condition
                const value = this.parseValue(object[key]);
                const condition = this.finder.findCondition(value[0]);
                if (!condition) {
                    throw new Error(`Condition '${value[0]}' used in an Fn::If expression does not exist in the template`);
                }
                return cfn_fn_1.Fn.conditionIf(condition.logicalId, value[1], value[2]);
            }
            case 'Fn::Equals': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionEquals(value[0], value[1]);
            }
            case 'Fn::And': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionAnd(...value);
            }
            case 'Fn::Not': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionNot(value[0]);
            }
            case 'Fn::Or': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionOr(...value);
            }
            case 'Fn::Sub': {
                const value = this.parseValue(object[key]);
                let fnSubString;
                let map;
                if (typeof value === 'string') {
                    fnSubString = value;
                    map = undefined;
                }
                else {
                    fnSubString = value[0];
                    map = value[1];
                }
                return cfn_fn_1.Fn.sub(this.parseFnSubString(fnSubString, map), map);
            }
            case 'Condition': {
                // a reference to a Condition from another Condition
                const condition = this.finder.findCondition(object[key]);
                if (!condition) {
                    throw new Error(`Referenced Condition with name '${object[key]}' was not found in the template`);
                }
                return { Condition: condition.logicalId };
            }
            default:
                if (this.options.context === CfnParsingContext.RULES) {
                    return this.handleRulesIntrinsic(key, object);
                }
                else {
                    throw new Error(`Unsupported CloudFormation function '${key}'`);
                }
        }
    }
    looksLikeCfnIntrinsic(object) {
        const objectKeys = Object.keys(object);
        // a CFN intrinsic is always an object with a single key
        if (objectKeys.length !== 1) {
            return undefined;
        }
        const key = objectKeys[0];
        return key === 'Ref' || key.startsWith('Fn::') ||
            // special intrinsic only available in the 'Conditions' section
            (this.options.context === CfnParsingContext.CONDITIONS && key === 'Condition')
            ? key
            : undefined;
    }
    parseFnSubString(value, map = {}) {
        const leftBrace = value.indexOf('${');
        const rightBrace = value.indexOf('}') + 1;
        // don't include left and right braces when searching for the target of the reference
        if (leftBrace === -1 || leftBrace >= rightBrace) {
            return value;
        }
        const leftHalf = value.substring(0, leftBrace);
        const rightHalf = value.substring(rightBrace);
        const refTarget = value.substring(leftBrace + 2, rightBrace - 1).trim();
        if (refTarget[0] === '!') {
            return value.substring(0, rightBrace) + this.parseFnSubString(rightHalf, map);
        }
        // lookup in map
        if (refTarget in map) {
            return leftHalf + '${' + refTarget + '}' + this.parseFnSubString(rightHalf, map);
        }
        // since it's not in the map, check if it's a pseudo parameter
        const specialRef = this.specialCaseSubRefs(refTarget);
        if (specialRef !== undefined) {
            return leftHalf + specialRef + this.parseFnSubString(rightHalf, map);
        }
        const dotIndex = refTarget.indexOf('.');
        const isRef = dotIndex === -1;
        if (isRef) {
            const refElement = this.finder.findRefTarget(refTarget);
            if (!refElement) {
                throw new Error(`Element referenced in Fn::Sub expression with logical ID: '${refTarget}' was not found in the template`);
            }
            return leftHalf + cfn_reference_1.CfnReference.for(refElement, 'Ref', true).toString() + this.parseFnSubString(rightHalf, map);
        }
        else {
            const targetId = refTarget.substring(0, dotIndex);
            const refResource = this.finder.findResource(targetId);
            if (!refResource) {
                throw new Error(`Resource referenced in Fn::Sub expression with logical ID: '${targetId}' was not found in the template`);
            }
            const attribute = refTarget.substring(dotIndex + 1);
            return leftHalf + cfn_reference_1.CfnReference.for(refResource, attribute, true).toString() + this.parseFnSubString(rightHalf, map);
        }
    }
    handleRulesIntrinsic(key, object) {
        // Rules have their own set of intrinsics:
        // https://docs.aws.amazon.com/servicecatalog/latest/adminguide/intrinsic-function-reference-rules.html
        switch (key) {
            case 'Fn::ValueOf': {
                // ValueOf is special,
                // as it takes the name of a Parameter as its first argument
                const value = this.parseValue(object[key]);
                const parameterName = value[0];
                if (parameterName in this.parameters) {
                    // since ValueOf returns the value of a specific attribute,
                    // fail here - this substitution is not allowed
                    throw new Error(`Cannot substitute parameter '${parameterName}' used in Fn::ValueOf expression with attribute '${value[1]}'`);
                }
                const param = this.finder.findRefTarget(parameterName);
                if (!param) {
                    throw new Error(`Rule references parameter '${parameterName}' which was not found in the template`);
                }
                // create an explicit IResolvable,
                // as Fn.valueOf() returns a string,
                // which is incorrect
                // (Fn::ValueOf can also return an array)
                return lazy_1.Lazy.anyValue({ produce: () => ({ 'Fn::ValueOf': [param.logicalId, value[1]] }) });
            }
            default:
                // I don't want to hard-code the list of supported Rules-specific intrinsics in this function;
                // so, just return undefined here,
                // and they will be treated as a regular JSON object
                return undefined;
        }
    }
    specialCaseRefs(value) {
        if (value in this.parameters) {
            return this.parameters[value];
        }
        switch (value) {
            case 'AWS::AccountId': return cfn_pseudo_1.Aws.ACCOUNT_ID;
            case 'AWS::Region': return cfn_pseudo_1.Aws.REGION;
            case 'AWS::Partition': return cfn_pseudo_1.Aws.PARTITION;
            case 'AWS::URLSuffix': return cfn_pseudo_1.Aws.URL_SUFFIX;
            case 'AWS::NotificationARNs': return cfn_pseudo_1.Aws.NOTIFICATION_ARNS;
            case 'AWS::StackId': return cfn_pseudo_1.Aws.STACK_ID;
            case 'AWS::StackName': return cfn_pseudo_1.Aws.STACK_NAME;
            case 'AWS::NoValue': return cfn_pseudo_1.Aws.NO_VALUE;
            default: return undefined;
        }
    }
    specialCaseSubRefs(value) {
        if (value in this.parameters) {
            return this.parameters[value];
        }
        return value.indexOf('::') === -1 ? undefined : '${' + value + '}';
    }
    get parameters() {
        return this.options.parameters || {};
    }
}
exports.CfnParser = CfnParser;
function undefinedIfAllValuesAreEmpty(object) {
    return Object.values(object).some(v => v !== undefined) ? object : undefined;
}
//# sourceMappingURL=data:application/json;base64,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