"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomResourceProvider = exports.CustomResourceProviderRuntime = void 0;
const fs = require("fs");
const path = require("path");
const asset_staging_1 = require("../asset-staging");
const assets_1 = require("../assets");
const cfn_resource_1 = require("../cfn-resource");
const construct_compat_1 = require("../construct-compat");
const duration_1 = require("../duration");
const size_1 = require("../size");
const stack_1 = require("../stack");
const token_1 = require("../token");
const ENTRYPOINT_FILENAME = '__entrypoint__';
const ENTRYPOINT_NODEJS_SOURCE = path.join(__dirname, 'nodejs-entrypoint.js');
/**
 * The lambda runtime to use for the resource provider. This also indicates
 * which language is used for the handler.
 * @experimental
 */
var CustomResourceProviderRuntime;
(function (CustomResourceProviderRuntime) {
    /**
     * Node.js 12.x
     */
    CustomResourceProviderRuntime["NODEJS_12"] = "nodejs12";
})(CustomResourceProviderRuntime = exports.CustomResourceProviderRuntime || (exports.CustomResourceProviderRuntime = {}));
/**
 * An AWS-Lambda backed custom resource provider.
 *
 * @experimental
 */
class CustomResourceProvider extends construct_compat_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const stack = stack_1.Stack.of(scope);
        // copy the entry point to the code directory
        fs.copyFileSync(ENTRYPOINT_NODEJS_SOURCE, path.join(props.codeDirectory, `${ENTRYPOINT_FILENAME}.js`));
        // verify we have an index file there
        if (!fs.existsSync(path.join(props.codeDirectory, 'index.js'))) {
            throw new Error(`cannot find ${props.codeDirectory}/index.js`);
        }
        const staging = new asset_staging_1.AssetStaging(this, 'Staging', {
            sourcePath: props.codeDirectory,
        });
        const asset = stack.addFileAsset({
            fileName: staging.stagedPath,
            sourceHash: staging.sourceHash,
            packaging: assets_1.FileAssetPackaging.ZIP_DIRECTORY,
        });
        const policies = !props.policyStatements ? undefined : [
            {
                PolicyName: 'Inline',
                PolicyDocument: {
                    Version: '2012-10-17',
                    Statement: props.policyStatements,
                },
            },
        ];
        const role = new cfn_resource_1.CfnResource(this, 'Role', {
            type: 'AWS::IAM::Role',
            properties: {
                AssumeRolePolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [{ Action: 'sts:AssumeRole', Effect: 'Allow', Principal: { Service: 'lambda.amazonaws.com' } }],
                },
                ManagedPolicyArns: [
                    { 'Fn::Sub': 'arn:${AWS::Partition}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole' },
                ],
                Policies: policies,
            },
        });
        const timeout = (_a = props.timeout) !== null && _a !== void 0 ? _a : duration_1.Duration.minutes(15);
        const memory = (_b = props.memorySize) !== null && _b !== void 0 ? _b : size_1.Size.mebibytes(128);
        const handler = new cfn_resource_1.CfnResource(this, 'Handler', {
            type: 'AWS::Lambda::Function',
            properties: {
                Code: {
                    S3Bucket: asset.bucketName,
                    S3Key: asset.objectKey,
                },
                Timeout: timeout.toSeconds(),
                MemorySize: memory.toMebibytes(),
                Handler: `${ENTRYPOINT_FILENAME}.handler`,
                Role: role.getAtt('Arn'),
                Runtime: 'nodejs12.x',
            },
        });
        handler.addDependsOn(role);
        this.serviceToken = token_1.Token.asString(handler.getAtt('Arn'));
    }
    /**
     * Returns a stack-level singleton ARN (service token) for the custom resource
     * provider.
     *
     * @param scope Construct scope
     * @param uniqueid A globally unique id that will be used for the stack-level
     * construct.
     * @param props Provider properties which will only be applied when the
     * provider is first created.
     * @returns the service token of the custom resource provider, which should be
     * used when defining a `CustomResource`.
     */
    static getOrCreate(scope, uniqueid, props) {
        var _a;
        const id = `${uniqueid}CustomResourceProvider`;
        const stack = stack_1.Stack.of(scope);
        const provider = (_a = stack.node.tryFindChild(id)) !== null && _a !== void 0 ? _a : new CustomResourceProvider(stack, id, props);
        return provider.serviceToken;
    }
}
exports.CustomResourceProvider = CustomResourceProvider;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY3VzdG9tLXJlc291cmNlLXByb3ZpZGVyLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY3VzdG9tLXJlc291cmNlLXByb3ZpZGVyLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHlCQUF5QjtBQUN6Qiw2QkFBNkI7QUFDN0Isb0RBQWdEO0FBQ2hELHNDQUErQztBQUMvQyxrREFBOEM7QUFDOUMsMERBQWdEO0FBQ2hELDBDQUF1QztBQUN2QyxrQ0FBK0I7QUFDL0Isb0NBQWlDO0FBQ2pDLG9DQUFpQztBQUVqQyxNQUFNLG1CQUFtQixHQUFHLGdCQUFnQixDQUFDO0FBQzdDLE1BQU0sd0JBQXdCLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsc0JBQXNCLENBQUMsQ0FBQztBQWdEOUU7Ozs7R0FJRztBQUNILElBQVksNkJBS1g7QUFMRCxXQUFZLDZCQUE2QjtJQUN2Qzs7T0FFRztJQUNILHVEQUFzQixDQUFBO0FBQ3hCLENBQUMsRUFMVyw2QkFBNkIsR0FBN0IscUNBQTZCLEtBQTdCLHFDQUE2QixRQUt4QztBQUVEOzs7O0dBSUc7QUFDSCxNQUFhLHNCQUF1QixTQUFRLDRCQUFTO0lBb0NuRCxZQUFzQixLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFrQzs7UUFDcEYsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixNQUFNLEtBQUssR0FBRyxhQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRTlCLDZDQUE2QztRQUM3QyxFQUFFLENBQUMsWUFBWSxDQUFDLHdCQUF3QixFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWEsRUFBRSxHQUFHLG1CQUFtQixLQUFLLENBQUMsQ0FBQyxDQUFDO1FBRXZHLHFDQUFxQztRQUNyQyxJQUFJLENBQUMsRUFBRSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxhQUFhLEVBQUUsVUFBVSxDQUFDLENBQUMsRUFBRTtZQUM5RCxNQUFNLElBQUksS0FBSyxDQUFDLGVBQWUsS0FBSyxDQUFDLGFBQWEsV0FBVyxDQUFDLENBQUM7U0FDaEU7UUFFRCxNQUFNLE9BQU8sR0FBRyxJQUFJLDRCQUFZLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUNoRCxVQUFVLEVBQUUsS0FBSyxDQUFDLGFBQWE7U0FDaEMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxLQUFLLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztZQUMvQixRQUFRLEVBQUUsT0FBTyxDQUFDLFVBQVU7WUFDNUIsVUFBVSxFQUFFLE9BQU8sQ0FBQyxVQUFVO1lBQzlCLFNBQVMsRUFBRSwyQkFBa0IsQ0FBQyxhQUFhO1NBQzVDLENBQUMsQ0FBQztRQUVILE1BQU0sUUFBUSxHQUFHLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO1lBQ3JEO2dCQUNFLFVBQVUsRUFBRSxRQUFRO2dCQUNwQixjQUFjLEVBQUU7b0JBQ2QsT0FBTyxFQUFFLFlBQVk7b0JBQ3JCLFNBQVMsRUFBRSxLQUFLLENBQUMsZ0JBQWdCO2lCQUNsQzthQUNGO1NBQ0YsQ0FBQztRQUVGLE1BQU0sSUFBSSxHQUFHLElBQUksMEJBQVcsQ0FBQyxJQUFJLEVBQUUsTUFBTSxFQUFFO1lBQ3pDLElBQUksRUFBRSxnQkFBZ0I7WUFDdEIsVUFBVSxFQUFFO2dCQUNWLHdCQUF3QixFQUFFO29CQUN4QixPQUFPLEVBQUUsWUFBWTtvQkFDckIsU0FBUyxFQUFFLENBQUMsRUFBRSxNQUFNLEVBQUUsZ0JBQWdCLEVBQUUsTUFBTSxFQUFFLE9BQU8sRUFBRSxTQUFTLEVBQUUsRUFBRSxPQUFPLEVBQUUsc0JBQXNCLEVBQUUsRUFBRSxDQUFDO2lCQUMzRztnQkFDRCxpQkFBaUIsRUFBRTtvQkFDakIsRUFBRSxTQUFTLEVBQUUsZ0ZBQWdGLEVBQUU7aUJBQ2hHO2dCQUNELFFBQVEsRUFBRSxRQUFRO2FBQ25CO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxPQUFPLFNBQUcsS0FBSyxDQUFDLE9BQU8sbUNBQUksbUJBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDdEQsTUFBTSxNQUFNLFNBQUcsS0FBSyxDQUFDLFVBQVUsbUNBQUksV0FBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUV2RCxNQUFNLE9BQU8sR0FBRyxJQUFJLDBCQUFXLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUMvQyxJQUFJLEVBQUUsdUJBQXVCO1lBQzdCLFVBQVUsRUFBRTtnQkFDVixJQUFJLEVBQUU7b0JBQ0osUUFBUSxFQUFFLEtBQUssQ0FBQyxVQUFVO29CQUMxQixLQUFLLEVBQUUsS0FBSyxDQUFDLFNBQVM7aUJBQ3ZCO2dCQUNELE9BQU8sRUFBRSxPQUFPLENBQUMsU0FBUyxFQUFFO2dCQUM1QixVQUFVLEVBQUUsTUFBTSxDQUFDLFdBQVcsRUFBRTtnQkFDaEMsT0FBTyxFQUFFLEdBQUcsbUJBQW1CLFVBQVU7Z0JBQ3pDLElBQUksRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQztnQkFDeEIsT0FBTyxFQUFFLFlBQVk7YUFDdEI7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRTNCLElBQUksQ0FBQyxZQUFZLEdBQUcsYUFBSyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7SUFDNUQsQ0FBQztJQXZHRDs7Ozs7Ozs7Ozs7T0FXRztJQUNJLE1BQU0sQ0FBQyxXQUFXLENBQUMsS0FBZ0IsRUFBRSxRQUFnQixFQUFFLEtBQWtDOztRQUM5RixNQUFNLEVBQUUsR0FBRyxHQUFHLFFBQVEsd0JBQXdCLENBQUM7UUFDL0MsTUFBTSxLQUFLLEdBQUcsYUFBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUM5QixNQUFNLFFBQVEsU0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQTJCLG1DQUNqRSxJQUFJLHNCQUFzQixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFbEQsT0FBTyxRQUFRLENBQUMsWUFBWSxDQUFDO0lBQy9CLENBQUM7Q0FxRkY7QUF6R0Qsd0RBeUdDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgZnMgZnJvbSAnZnMnO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCB7IEFzc2V0U3RhZ2luZyB9IGZyb20gJy4uL2Fzc2V0LXN0YWdpbmcnO1xuaW1wb3J0IHsgRmlsZUFzc2V0UGFja2FnaW5nIH0gZnJvbSAnLi4vYXNzZXRzJztcbmltcG9ydCB7IENmblJlc291cmNlIH0gZnJvbSAnLi4vY2ZuLXJlc291cmNlJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJy4uL2NvbnN0cnVjdC1jb21wYXQnO1xuaW1wb3J0IHsgRHVyYXRpb24gfSBmcm9tICcuLi9kdXJhdGlvbic7XG5pbXBvcnQgeyBTaXplIH0gZnJvbSAnLi4vc2l6ZSc7XG5pbXBvcnQgeyBTdGFjayB9IGZyb20gJy4uL3N0YWNrJztcbmltcG9ydCB7IFRva2VuIH0gZnJvbSAnLi4vdG9rZW4nO1xuXG5jb25zdCBFTlRSWVBPSU5UX0ZJTEVOQU1FID0gJ19fZW50cnlwb2ludF9fJztcbmNvbnN0IEVOVFJZUE9JTlRfTk9ERUpTX1NPVVJDRSA9IHBhdGguam9pbihfX2Rpcm5hbWUsICdub2RlanMtZW50cnlwb2ludC5qcycpO1xuXG4vKipcbiAqIEluaXRpYWxpemF0aW9uIHByb3BlcnRpZXMgZm9yIGBDdXN0b21SZXNvdXJjZVByb3ZpZGVyYC5cbiAqXG4gKiBAZXhwZXJpbWVudGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ3VzdG9tUmVzb3VyY2VQcm92aWRlclByb3BzIHtcbiAgLyoqXG4gICAqIEEgbG9jYWwgZmlsZSBzeXN0ZW0gZGlyZWN0b3J5IHdpdGggdGhlIHByb3ZpZGVyJ3MgY29kZS4gVGhlIGNvZGUgd2lsbCBiZVxuICAgKiBidW5kbGVkIGludG8gYSB6aXAgYXNzZXQgYW5kIHdpcmVkIHRvIHRoZSBwcm92aWRlcidzIEFXUyBMYW1iZGEgZnVuY3Rpb24uXG4gICAqL1xuICByZWFkb25seSBjb2RlRGlyZWN0b3J5OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBBV1MgTGFtYmRhIHJ1bnRpbWUgYW5kIHZlcnNpb24gdG8gdXNlIGZvciB0aGUgcHJvdmlkZXIuXG4gICAqL1xuICByZWFkb25seSBydW50aW1lOiBDdXN0b21SZXNvdXJjZVByb3ZpZGVyUnVudGltZTtcblxuICAvKipcbiAgICogQSBzZXQgb2YgSUFNIHBvbGljeSBzdGF0ZW1lbnRzIHRvIGluY2x1ZGUgaW4gdGhlIGlubGluZSBwb2xpY3kgb2YgdGhlXG4gICAqIHByb3ZpZGVyJ3MgbGFtYmRhIGZ1bmN0aW9uLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGFkZGl0aW9uYWwgaW5saW5lIHBvbGljeVxuICAgKlxuICAgKiBAZXhhbXBsZVxuICAgKlxuICAgKiAgIHBvbGljeVN0YXRlbWVudHM6IFsgeyBFZmZlY3Q6ICdBbGxvdycsIEFjdGlvbjogJ3MzOlB1dE9iamVjdConLCBSZXNvdXJjZTogJyonIH0gXVxuICAgKlxuICAgKi9cbiAgcmVhZG9ubHkgcG9saWN5U3RhdGVtZW50cz86IGFueVtdO1xuXG4gIC8qKlxuICAgKiBBV1MgTGFtYmRhIHRpbWVvdXQgZm9yIHRoZSBwcm92aWRlci5cbiAgICpcbiAgICogQGRlZmF1bHQgRHVyYXRpb24ubWludXRlcygxNSlcbiAgICovXG4gIHJlYWRvbmx5IHRpbWVvdXQ/OiBEdXJhdGlvbjtcblxuICAvKipcbiAgICogVGhlIGFtb3VudCBvZiBtZW1vcnkgdGhhdCB5b3VyIGZ1bmN0aW9uIGhhcyBhY2Nlc3MgdG8uIEluY3JlYXNpbmcgdGhlXG4gICAqIGZ1bmN0aW9uJ3MgbWVtb3J5IGFsc28gaW5jcmVhc2VzIGl0cyBDUFUgYWxsb2NhdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgU2l6ZS5tZWJpYnl0ZXMoMTI4KVxuICAgKi9cbiAgcmVhZG9ubHkgbWVtb3J5U2l6ZT86IFNpemU7XG59XG5cbi8qKlxuICogVGhlIGxhbWJkYSBydW50aW1lIHRvIHVzZSBmb3IgdGhlIHJlc291cmNlIHByb3ZpZGVyLiBUaGlzIGFsc28gaW5kaWNhdGVzXG4gKiB3aGljaCBsYW5ndWFnZSBpcyB1c2VkIGZvciB0aGUgaGFuZGxlci5cbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGVudW0gQ3VzdG9tUmVzb3VyY2VQcm92aWRlclJ1bnRpbWUge1xuICAvKipcbiAgICogTm9kZS5qcyAxMi54XG4gICAqL1xuICBOT0RFSlNfMTIgPSAnbm9kZWpzMTInXG59XG5cbi8qKlxuICogQW4gQVdTLUxhbWJkYSBiYWNrZWQgY3VzdG9tIHJlc291cmNlIHByb3ZpZGVyLlxuICpcbiAqIEBleHBlcmltZW50YWxcbiAqL1xuZXhwb3J0IGNsYXNzIEN1c3RvbVJlc291cmNlUHJvdmlkZXIgZXh0ZW5kcyBDb25zdHJ1Y3Qge1xuICAvKipcbiAgICogUmV0dXJucyBhIHN0YWNrLWxldmVsIHNpbmdsZXRvbiBBUk4gKHNlcnZpY2UgdG9rZW4pIGZvciB0aGUgY3VzdG9tIHJlc291cmNlXG4gICAqIHByb3ZpZGVyLlxuICAgKlxuICAgKiBAcGFyYW0gc2NvcGUgQ29uc3RydWN0IHNjb3BlXG4gICAqIEBwYXJhbSB1bmlxdWVpZCBBIGdsb2JhbGx5IHVuaXF1ZSBpZCB0aGF0IHdpbGwgYmUgdXNlZCBmb3IgdGhlIHN0YWNrLWxldmVsXG4gICAqIGNvbnN0cnVjdC5cbiAgICogQHBhcmFtIHByb3BzIFByb3ZpZGVyIHByb3BlcnRpZXMgd2hpY2ggd2lsbCBvbmx5IGJlIGFwcGxpZWQgd2hlbiB0aGVcbiAgICogcHJvdmlkZXIgaXMgZmlyc3QgY3JlYXRlZC5cbiAgICogQHJldHVybnMgdGhlIHNlcnZpY2UgdG9rZW4gb2YgdGhlIGN1c3RvbSByZXNvdXJjZSBwcm92aWRlciwgd2hpY2ggc2hvdWxkIGJlXG4gICAqIHVzZWQgd2hlbiBkZWZpbmluZyBhIGBDdXN0b21SZXNvdXJjZWAuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGdldE9yQ3JlYXRlKHNjb3BlOiBDb25zdHJ1Y3QsIHVuaXF1ZWlkOiBzdHJpbmcsIHByb3BzOiBDdXN0b21SZXNvdXJjZVByb3ZpZGVyUHJvcHMpIHtcbiAgICBjb25zdCBpZCA9IGAke3VuaXF1ZWlkfUN1c3RvbVJlc291cmNlUHJvdmlkZXJgO1xuICAgIGNvbnN0IHN0YWNrID0gU3RhY2sub2Yoc2NvcGUpO1xuICAgIGNvbnN0IHByb3ZpZGVyID0gc3RhY2subm9kZS50cnlGaW5kQ2hpbGQoaWQpIGFzIEN1c3RvbVJlc291cmNlUHJvdmlkZXJcbiAgICAgID8/IG5ldyBDdXN0b21SZXNvdXJjZVByb3ZpZGVyKHN0YWNrLCBpZCwgcHJvcHMpO1xuXG4gICAgcmV0dXJuIHByb3ZpZGVyLnNlcnZpY2VUb2tlbjtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgQVJOIG9mIHRoZSBwcm92aWRlcidzIEFXUyBMYW1iZGEgZnVuY3Rpb24gd2hpY2ggc2hvdWxkIGJlIHVzZWQgYXMgdGhlXG4gICAqIGBzZXJ2aWNlVG9rZW5gIHdoZW4gZGVmaW5pbmcgYSBjdXN0b20gcmVzb3VyY2UuXG4gICAqXG4gICAqIEBleGFtcGxlXG4gICAqXG4gICAqICAgbmV3IEN1c3RvbVJlc291cmNlKHRoaXMsICdNeUN1c3RvbVJlc291cmNlJywge1xuICAgKiAgICAgLy8gLi4uXG4gICAqICAgICBzZXJ2aWNlVG9rZW46IHByb3ZpZGVyLnNlcnZpY2VUb2tlbiAvLyA8LS0tIGhlcmVcbiAgICogICB9KVxuICAgKlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHNlcnZpY2VUb2tlbjogc3RyaW5nO1xuXG4gIHByb3RlY3RlZCBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQ3VzdG9tUmVzb3VyY2VQcm92aWRlclByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGNvbnN0IHN0YWNrID0gU3RhY2sub2Yoc2NvcGUpO1xuXG4gICAgLy8gY29weSB0aGUgZW50cnkgcG9pbnQgdG8gdGhlIGNvZGUgZGlyZWN0b3J5XG4gICAgZnMuY29weUZpbGVTeW5jKEVOVFJZUE9JTlRfTk9ERUpTX1NPVVJDRSwgcGF0aC5qb2luKHByb3BzLmNvZGVEaXJlY3RvcnksIGAke0VOVFJZUE9JTlRfRklMRU5BTUV9LmpzYCkpO1xuXG4gICAgLy8gdmVyaWZ5IHdlIGhhdmUgYW4gaW5kZXggZmlsZSB0aGVyZVxuICAgIGlmICghZnMuZXhpc3RzU3luYyhwYXRoLmpvaW4ocHJvcHMuY29kZURpcmVjdG9yeSwgJ2luZGV4LmpzJykpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYGNhbm5vdCBmaW5kICR7cHJvcHMuY29kZURpcmVjdG9yeX0vaW5kZXguanNgKTtcbiAgICB9XG5cbiAgICBjb25zdCBzdGFnaW5nID0gbmV3IEFzc2V0U3RhZ2luZyh0aGlzLCAnU3RhZ2luZycsIHtcbiAgICAgIHNvdXJjZVBhdGg6IHByb3BzLmNvZGVEaXJlY3RvcnksXG4gICAgfSk7XG5cbiAgICBjb25zdCBhc3NldCA9IHN0YWNrLmFkZEZpbGVBc3NldCh7XG4gICAgICBmaWxlTmFtZTogc3RhZ2luZy5zdGFnZWRQYXRoLFxuICAgICAgc291cmNlSGFzaDogc3RhZ2luZy5zb3VyY2VIYXNoLFxuICAgICAgcGFja2FnaW5nOiBGaWxlQXNzZXRQYWNrYWdpbmcuWklQX0RJUkVDVE9SWSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHBvbGljaWVzID0gIXByb3BzLnBvbGljeVN0YXRlbWVudHMgPyB1bmRlZmluZWQgOiBbXG4gICAgICB7XG4gICAgICAgIFBvbGljeU5hbWU6ICdJbmxpbmUnLFxuICAgICAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICAgIFZlcnNpb246ICcyMDEyLTEwLTE3JyxcbiAgICAgICAgICBTdGF0ZW1lbnQ6IHByb3BzLnBvbGljeVN0YXRlbWVudHMsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIF07XG5cbiAgICBjb25zdCByb2xlID0gbmV3IENmblJlc291cmNlKHRoaXMsICdSb2xlJywge1xuICAgICAgdHlwZTogJ0FXUzo6SUFNOjpSb2xlJyxcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgQXNzdW1lUm9sZVBvbGljeURvY3VtZW50OiB7XG4gICAgICAgICAgVmVyc2lvbjogJzIwMTItMTAtMTcnLFxuICAgICAgICAgIFN0YXRlbWVudDogW3sgQWN0aW9uOiAnc3RzOkFzc3VtZVJvbGUnLCBFZmZlY3Q6ICdBbGxvdycsIFByaW5jaXBhbDogeyBTZXJ2aWNlOiAnbGFtYmRhLmFtYXpvbmF3cy5jb20nIH0gfV0sXG4gICAgICAgIH0sXG4gICAgICAgIE1hbmFnZWRQb2xpY3lBcm5zOiBbXG4gICAgICAgICAgeyAnRm46OlN1Yic6ICdhcm46JHtBV1M6OlBhcnRpdGlvbn06aWFtOjphd3M6cG9saWN5L3NlcnZpY2Utcm9sZS9BV1NMYW1iZGFCYXNpY0V4ZWN1dGlvblJvbGUnIH0sXG4gICAgICAgIF0sXG4gICAgICAgIFBvbGljaWVzOiBwb2xpY2llcyxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBjb25zdCB0aW1lb3V0ID0gcHJvcHMudGltZW91dCA/PyBEdXJhdGlvbi5taW51dGVzKDE1KTtcbiAgICBjb25zdCBtZW1vcnkgPSBwcm9wcy5tZW1vcnlTaXplID8/IFNpemUubWViaWJ5dGVzKDEyOCk7XG5cbiAgICBjb25zdCBoYW5kbGVyID0gbmV3IENmblJlc291cmNlKHRoaXMsICdIYW5kbGVyJywge1xuICAgICAgdHlwZTogJ0FXUzo6TGFtYmRhOjpGdW5jdGlvbicsXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIENvZGU6IHtcbiAgICAgICAgICBTM0J1Y2tldDogYXNzZXQuYnVja2V0TmFtZSxcbiAgICAgICAgICBTM0tleTogYXNzZXQub2JqZWN0S2V5LFxuICAgICAgICB9LFxuICAgICAgICBUaW1lb3V0OiB0aW1lb3V0LnRvU2Vjb25kcygpLFxuICAgICAgICBNZW1vcnlTaXplOiBtZW1vcnkudG9NZWJpYnl0ZXMoKSxcbiAgICAgICAgSGFuZGxlcjogYCR7RU5UUllQT0lOVF9GSUxFTkFNRX0uaGFuZGxlcmAsXG4gICAgICAgIFJvbGU6IHJvbGUuZ2V0QXR0KCdBcm4nKSxcbiAgICAgICAgUnVudGltZTogJ25vZGVqczEyLngnLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGhhbmRsZXIuYWRkRGVwZW5kc09uKHJvbGUpO1xuXG4gICAgdGhpcy5zZXJ2aWNlVG9rZW4gPSBUb2tlbi5hc1N0cmluZyhoYW5kbGVyLmdldEF0dCgnQXJuJykpO1xuICB9XG59XG4iXX0=