"use strict";
const nodeunit = require("nodeunit");
const core = require("../lib");
module.exports = nodeunit.testCase({
    '._toCloudFormation': {
        'does not call renderProperties with an undefined value'(test) {
            const app = new core.App();
            const stack = new core.Stack(app, 'TestStack');
            const resource = new core.CfnResource(stack, 'DefaultResource', { type: 'Test::Resource::Fake' });
            let called = false;
            resource.renderProperties = (val) => {
                called = true;
                test.notEqual(val, null);
            };
            test.deepEqual(app.synth().getStackByName(stack.stackName).template, {
                Resources: {
                    DefaultResource: {
                        Type: 'Test::Resource::Fake',
                    },
                },
            });
            test.ok(called, 'renderProperties must be called called');
            test.done();
        },
    },
    'applyRemovalPolicy default includes Update policy'(test) {
        // GIVEN
        const app = new core.App();
        const stack = new core.Stack(app, 'TestStack');
        const resource = new core.CfnResource(stack, 'DefaultResource', { type: 'Test::Resource::Fake' });
        // WHEN
        resource.applyRemovalPolicy(core.RemovalPolicy.RETAIN);
        // THEN
        test.deepEqual(app.synth().getStackByName(stack.stackName).template, {
            Resources: {
                DefaultResource: {
                    Type: 'Test::Resource::Fake',
                    DeletionPolicy: 'Retain',
                    UpdateReplacePolicy: 'Retain',
                },
            },
        });
        test.done();
    },
    'can switch off updating Update policy'(test) {
        // GIVEN
        const app = new core.App();
        const stack = new core.Stack(app, 'TestStack');
        const resource = new core.CfnResource(stack, 'DefaultResource', { type: 'Test::Resource::Fake' });
        // WHEN
        resource.applyRemovalPolicy(core.RemovalPolicy.RETAIN, {
            applyToUpdateReplacePolicy: false,
        });
        // THEN
        test.deepEqual(app.synth().getStackByName(stack.stackName).template, {
            Resources: {
                DefaultResource: {
                    Type: 'Test::Resource::Fake',
                    DeletionPolicy: 'Retain',
                },
            },
        });
        test.done();
    },
    'can add metadata'(test) {
        // GIVEN
        const app = new core.App();
        const stack = new core.Stack(app, 'TestStack');
        const resource = new core.CfnResource(stack, 'DefaultResource', { type: 'Test::Resource::Fake' });
        // WHEN
        resource.addMetadata('Beep', 'Boop');
        // THEN
        test.deepEqual(app.synth().getStackByName(stack.stackName).template, {
            Resources: {
                DefaultResource: {
                    Type: 'Test::Resource::Fake',
                    Metadata: {
                        Beep: 'Boop',
                    },
                },
            },
        });
        test.done();
    },
    'subclasses can override "shouldSynthesize" to lazy-determine if the resource should be included'(test) {
        // GIVEN
        class HiddenCfnResource extends core.CfnResource {
            shouldSynthesize() {
                return false;
            }
        }
        const app = new core.App();
        const stack = new core.Stack(app, 'TestStack');
        const subtree = new core.Construct(stack, 'subtree');
        // WHEN
        new HiddenCfnResource(subtree, 'R1', { type: 'Foo::R1' });
        const r2 = new core.CfnResource(stack, 'R2', { type: 'Foo::R2' });
        // also try to take a dependency on the parent of `r1` and expect the dependency not to materialize
        r2.node.addDependency(subtree);
        // THEN - only R2 is synthesized
        test.deepEqual(app.synth().getStackByName(stack.stackName).template, {
            Resources: { R2: { Type: 'Foo::R2' } },
        });
        test.done();
    },
    'CfnResource cannot be created outside Stack'(test) {
        const app = new core.App();
        test.throws(() => {
            new core.CfnResource(app, 'Resource', {
                type: 'Some::Resource',
            });
        }, /should be created in the scope of a Stack, but no Stack found/);
        test.done();
    },
    /**
     * Stages start a new scope, which does not count as a Stack anymore
     */
    'CfnResource cannot be in Stage in Stack'(test) {
        const app = new core.App();
        const stack = new core.Stack(app, 'Stack');
        const stage = new core.Stage(stack, 'Stage');
        test.throws(() => {
            new core.CfnResource(stage, 'Resource', {
                type: 'Some::Resource',
            });
        }, /should be created in the scope of a Stack, but no Stack found/);
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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