"use strict";
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
const intrinsic_1 = require("../lib/private/intrinsic");
const refs_1 = require("../lib/private/refs");
const util_1 = require("../lib/util");
const util_2 = require("./util");
class StackWithPostProcessor extends lib_1.Stack {
    // ...
    _toCloudFormation() {
        const template = super._toCloudFormation();
        // manipulate template (e.g. rename "Key" to "key")
        template.Resources.myResource.Properties.Environment.key =
            template.Resources.myResource.Properties.Environment.Key;
        delete template.Resources.myResource.Properties.Environment.Key;
        return template;
    }
}
module.exports = {
    'a stack can be serialized into a CloudFormation template, initially it\'s empty'(test) {
        const stack = new lib_1.Stack();
        test.deepEqual(util_2.toCloudFormation(stack), {});
        test.done();
    },
    'stack objects have some template-level propeties, such as Description, Version, Transform'(test) {
        const stack = new lib_1.Stack();
        stack.templateOptions.templateFormatVersion = 'MyTemplateVersion';
        stack.templateOptions.description = 'This is my description';
        stack.templateOptions.transforms = ['SAMy'];
        test.deepEqual(util_2.toCloudFormation(stack), {
            Description: 'This is my description',
            AWSTemplateFormatVersion: 'MyTemplateVersion',
            Transform: 'SAMy',
        });
        test.done();
    },
    'Stack.isStack indicates that a construct is a stack'(test) {
        const stack = new lib_1.Stack();
        const c = new lib_1.Construct(stack, 'Construct');
        test.ok(lib_1.Stack.isStack(stack));
        test.ok(!lib_1.Stack.isStack(c));
        test.done();
    },
    'stack.id is not included in the logical identities of resources within it'(test) {
        const stack = new lib_1.Stack(undefined, 'MyStack');
        new lib_1.CfnResource(stack, 'MyResource', { type: 'MyResourceType' });
        test.deepEqual(util_2.toCloudFormation(stack), { Resources: { MyResource: { Type: 'MyResourceType' } } });
        test.done();
    },
    'stack.templateOptions can be used to set template-level options'(test) {
        const stack = new lib_1.Stack();
        stack.templateOptions.description = 'StackDescription';
        stack.templateOptions.templateFormatVersion = 'TemplateVersion';
        stack.templateOptions.transform = 'DeprecatedField';
        stack.templateOptions.transforms = ['Transform'];
        stack.templateOptions.metadata = {
            MetadataKey: 'MetadataValue',
        };
        test.deepEqual(util_2.toCloudFormation(stack), {
            Description: 'StackDescription',
            Transform: ['Transform', 'DeprecatedField'],
            AWSTemplateFormatVersion: 'TemplateVersion',
            Metadata: { MetadataKey: 'MetadataValue' },
        });
        test.done();
    },
    'stack.templateOptions.transforms removes duplicate values'(test) {
        const stack = new lib_1.Stack();
        stack.templateOptions.transforms = ['A', 'B', 'C', 'A'];
        test.deepEqual(util_2.toCloudFormation(stack), {
            Transform: ['A', 'B', 'C'],
        });
        test.done();
    },
    'stack.addTransform() adds a transform'(test) {
        const stack = new lib_1.Stack();
        stack.addTransform('A');
        stack.addTransform('B');
        stack.addTransform('C');
        test.deepEqual(util_2.toCloudFormation(stack), {
            Transform: ['A', 'B', 'C'],
        });
        test.done();
    },
    // This approach will only apply to TypeScript code, but at least it's a temporary
    // workaround for people running into issues caused by SDK-3003.
    // We should come up with a proper solution that involved jsii callbacks (when they exist)
    // so this can be implemented by jsii languages as well.
    'Overriding `Stack._toCloudFormation` allows arbitrary post-processing of the generated template during synthesis'(test) {
        const stack = new StackWithPostProcessor();
        new lib_1.CfnResource(stack, 'myResource', {
            type: 'AWS::MyResource',
            properties: {
                MyProp1: 'hello',
                MyProp2: 'howdy',
                Environment: {
                    Key: 'value',
                },
            },
        });
        test.deepEqual(stack._toCloudFormation(), {
            Resources: {
                myResource: {
                    Type: 'AWS::MyResource',
                    Properties: {
                        MyProp1: 'hello',
                        MyProp2: 'howdy',
                        Environment: { key: 'value' },
                    },
                },
            },
        });
        test.done();
    },
    'Stack.getByPath can be used to find any CloudFormation element (Parameter, Output, etc)'(test) {
        const stack = new lib_1.Stack();
        const p = new lib_1.CfnParameter(stack, 'MyParam', { type: 'String' });
        const o = new lib_1.CfnOutput(stack, 'MyOutput', { value: 'boom' });
        const c = new lib_1.CfnCondition(stack, 'MyCondition');
        test.equal(stack.node.findChild(p.node.id), p);
        test.equal(stack.node.findChild(o.node.id), o);
        test.equal(stack.node.findChild(c.node.id), c);
        test.done();
    },
    'Stack names can have hyphens in them'(test) {
        const root = new lib_1.App();
        new lib_1.Stack(root, 'Hello-World');
        // Did not throw
        test.done();
    },
    'Stacks can have a description given to them'(test) {
        const stack = new lib_1.Stack(new lib_1.App(), 'MyStack', { description: 'My stack, hands off!' });
        const output = util_2.toCloudFormation(stack);
        test.equal(output.Description, 'My stack, hands off!');
        test.done();
    },
    'Stack descriptions have a limited length'(test) {
        const desc = `Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmod tempor
     incididunt ut labore et dolore magna aliqua. Consequat interdum varius sit amet mattis vulputate
     enim nulla aliquet. At imperdiet dui accumsan sit amet nulla facilisi morbi. Eget lorem dolor sed
     viverra ipsum. Diam volutpat commodo sed egestas egestas. Sit amet porttitor eget dolor morbi non.
     Lorem dolor sed viverra ipsum. Id porta nibh venenatis cras sed felis. Augue interdum velit euismod
     in pellentesque. Suscipit adipiscing bibendum est ultricies integer quis. Condimentum id venenatis a
     condimentum vitae sapien pellentesque habitant morbi. Congue mauris rhoncus aenean vel elit scelerisque
     mauris pellentesque pulvinar.
     Faucibus purus in massa tempor nec. Risus viverra adipiscing at in. Integer feugiat scelerisque varius
     morbi. Malesuada nunc vel risus commodo viverra maecenas accumsan lacus. Vulputate sapien nec sagittis
     aliquam malesuada bibendum arcu vitae. Augue neque gravida in fermentum et sollicitudin ac orci phasellus.
     Ultrices tincidunt arcu non sodales neque sodales.`;
        test.throws(() => new lib_1.Stack(new lib_1.App(), 'MyStack', { description: desc }));
        test.done();
    },
    'Include should support non-hash top-level template elements like "Description"'(test) {
        const stack = new lib_1.Stack();
        const template = {
            Description: 'hello, world',
        };
        new lib_1.CfnInclude(stack, 'Include', { template });
        const output = util_2.toCloudFormation(stack);
        test.equal(typeof output.Description, 'string');
        test.done();
    },
    'Pseudo values attached to one stack can be referenced in another stack'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another stack
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: account1 });
        // THEN
        const assembly = app.synth();
        const template1 = assembly.getStackByName(stack1.stackName).template;
        const template2 = assembly.getStackByName(stack2.stackName).template;
        test.deepEqual(template1, {
            Outputs: {
                ExportsOutputRefAWSAccountIdAD568057: {
                    Value: { Ref: 'AWS::AccountId' },
                    Export: { Name: 'Stack1:ExportsOutputRefAWSAccountIdAD568057' },
                },
            },
        });
        test.deepEqual(template2, {
            Parameters: {
                SomeParameter: {
                    Type: 'String',
                    Default: { 'Fn::ImportValue': 'Stack1:ExportsOutputRefAWSAccountIdAD568057' },
                },
            },
        });
        test.done();
    },
    'Cross-stack references are detected in resource properties'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const resource1 = new lib_1.CfnResource(stack1, 'Resource', { type: 'BLA' });
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another resource
        new lib_1.CfnResource(stack2, 'SomeResource', {
            type: 'AWS::Some::Resource',
            properties: {
                someProperty: new intrinsic_1.Intrinsic(resource1.ref),
            },
        });
        // THEN
        const assembly = app.synth();
        const template2 = assembly.getStackByName(stack2.stackName).template;
        test.deepEqual(template2, {
            Resources: {
                SomeResource: {
                    Type: 'AWS::Some::Resource',
                    Properties: {
                        someProperty: { 'Fn::ImportValue': 'Stack1:ExportsOutputRefResource1D5D905A' },
                    },
                },
            },
        });
        test.done();
    },
    'Cross-stack reference export names are relative to the stack (when the flag is set)'(test) {
        // GIVEN
        const app = new lib_1.App({
            context: {
                '@aws-cdk/core:stackRelativeExports': 'true',
            },
        });
        const indifferentScope = new lib_1.Construct(app, 'ExtraScope');
        const stack1 = new lib_1.Stack(indifferentScope, 'Stack1', {
            stackName: 'Stack1',
        });
        const resource1 = new lib_1.CfnResource(stack1, 'Resource', { type: 'BLA' });
        const stack2 = new lib_1.Stack(indifferentScope, 'Stack2');
        // WHEN - used in another resource
        new lib_1.CfnResource(stack2, 'SomeResource', {
            type: 'AWS::Some::Resource',
            properties: {
                someProperty: new intrinsic_1.Intrinsic(resource1.ref),
            },
        });
        // THEN
        const assembly = app.synth();
        const template2 = assembly.getStackByName(stack2.stackName).template;
        test.deepEqual(template2, {
            Resources: {
                SomeResource: {
                    Type: 'AWS::Some::Resource',
                    Properties: {
                        someProperty: { 'Fn::ImportValue': 'Stack1:ExportsOutputRefResource1D5D905A' },
                    },
                },
            },
        });
        test.done();
    },
    'cross-stack references in lazy tokens work'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another stack
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: lib_1.Lazy.stringValue({ produce: () => account1 }) });
        const assembly = app.synth();
        const template1 = assembly.getStackByName(stack1.stackName).template;
        const template2 = assembly.getStackByName(stack2.stackName).template;
        // THEN
        test.deepEqual(template1, {
            Outputs: {
                ExportsOutputRefAWSAccountIdAD568057: {
                    Value: { Ref: 'AWS::AccountId' },
                    Export: { Name: 'Stack1:ExportsOutputRefAWSAccountIdAD568057' },
                },
            },
        });
        test.deepEqual(template2, {
            Parameters: {
                SomeParameter: {
                    Type: 'String',
                    Default: { 'Fn::ImportValue': 'Stack1:ExportsOutputRefAWSAccountIdAD568057' },
                },
            },
        });
        test.done();
    },
    'Cross-stack use of Region and account returns nonscoped intrinsic because the two stacks must be in the same region anyway'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another stack
        new lib_1.CfnOutput(stack2, 'DemOutput', { value: stack1.region });
        new lib_1.CfnOutput(stack2, 'DemAccount', { value: stack1.account });
        // THEN
        const assembly = app.synth();
        const template2 = assembly.getStackByName(stack2.stackName).template;
        test.deepEqual(template2, {
            Outputs: {
                DemOutput: {
                    Value: { Ref: 'AWS::Region' },
                },
                DemAccount: {
                    Value: { Ref: 'AWS::AccountId' },
                },
            },
        });
        test.done();
    },
    'cross-stack references in strings work'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN - used in another stack
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: `TheAccountIs${account1}` });
        const assembly = app.synth();
        const template2 = assembly.getStackByName(stack2.stackName).template;
        // THEN
        test.deepEqual(template2, {
            Parameters: {
                SomeParameter: {
                    Type: 'String',
                    Default: { 'Fn::Join': ['', ['TheAccountIs', { 'Fn::ImportValue': 'Stack1:ExportsOutputRefAWSAccountIdAD568057' }]] },
                },
            },
        });
        test.done();
    },
    'cross stack references and dependencies work within child stacks (non-nested)'(test) {
        // GIVEN
        const app = new lib_1.App();
        const parent = new lib_1.Stack(app, 'Parent');
        const child1 = new lib_1.Stack(parent, 'Child1');
        const child2 = new lib_1.Stack(parent, 'Child2');
        const resourceA = new lib_1.CfnResource(child1, 'ResourceA', { type: 'RA' });
        const resourceB = new lib_1.CfnResource(child1, 'ResourceB', { type: 'RB' });
        // WHEN
        const resource2 = new lib_1.CfnResource(child2, 'Resource1', {
            type: 'R2',
            properties: {
                RefToResource1: resourceA.ref,
            },
        });
        resource2.addDependsOn(resourceB);
        // THEN
        const assembly = app.synth();
        const parentTemplate = assembly.getStackArtifact(parent.artifactId).template;
        const child1Template = assembly.getStackArtifact(child1.artifactId).template;
        const child2Template = assembly.getStackArtifact(child2.artifactId).template;
        test.deepEqual(parentTemplate, {});
        test.deepEqual(child1Template, {
            Resources: {
                ResourceA: { Type: 'RA' },
                ResourceB: { Type: 'RB' },
            },
            Outputs: {
                ExportsOutputRefResourceA461B4EF9: {
                    Value: { Ref: 'ResourceA' },
                    Export: { Name: 'ParentChild18FAEF419:Child1ExportsOutputRefResourceA7BF20B37' },
                },
            },
        });
        test.deepEqual(child2Template, {
            Resources: {
                Resource1: {
                    Type: 'R2',
                    Properties: {
                        RefToResource1: { 'Fn::ImportValue': 'ParentChild18FAEF419:Child1ExportsOutputRefResourceA7BF20B37' },
                    },
                },
            },
        });
        test.deepEqual(assembly.getStackArtifact(child1.artifactId).dependencies.map(x => x.id), []);
        test.deepEqual(assembly.getStackArtifact(child2.artifactId).dependencies.map(x => x.id), ['ParentChild18FAEF419']);
        test.done();
    },
    'CfnSynthesisError is ignored when preparing cross references'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'my-stack');
        // WHEN
        class CfnTest extends lib_1.CfnResource {
            _toCloudFormation() {
                return new util_1.PostResolveToken({
                    xoo: 1234,
                }, props => {
                    lib_1.validateString(props).assertSuccess();
                });
            }
        }
        new CfnTest(stack, 'MyThing', { type: 'AWS::Type' });
        // THEN
        refs_1.resolveReferences(app);
        test.done();
    },
    'Stacks can be children of other stacks (substack) and they will be synthesized separately'(test) {
        // GIVEN
        const app = new lib_1.App();
        // WHEN
        const parentStack = new lib_1.Stack(app, 'parent');
        const childStack = new lib_1.Stack(parentStack, 'child');
        new lib_1.CfnResource(parentStack, 'MyParentResource', { type: 'Resource::Parent' });
        new lib_1.CfnResource(childStack, 'MyChildResource', { type: 'Resource::Child' });
        // THEN
        const assembly = app.synth();
        test.deepEqual(assembly.getStackByName(parentStack.stackName).template, { Resources: { MyParentResource: { Type: 'Resource::Parent' } } });
        test.deepEqual(assembly.getStackByName(childStack.stackName).template, { Resources: { MyChildResource: { Type: 'Resource::Child' } } });
        test.done();
    },
    'cross-stack reference (substack references parent stack)'(test) {
        // GIVEN
        const app = new lib_1.App();
        const parentStack = new lib_1.Stack(app, 'parent');
        const childStack = new lib_1.Stack(parentStack, 'child');
        // WHEN (a resource from the child stack references a resource from the parent stack)
        const parentResource = new lib_1.CfnResource(parentStack, 'MyParentResource', { type: 'Resource::Parent' });
        new lib_1.CfnResource(childStack, 'MyChildResource', {
            type: 'Resource::Child',
            properties: {
                ChildProp: parentResource.getAtt('AttOfParentResource'),
            },
        });
        // THEN
        const assembly = app.synth();
        test.deepEqual(assembly.getStackByName(parentStack.stackName).template, {
            Resources: { MyParentResource: { Type: 'Resource::Parent' } },
            Outputs: {
                ExportsOutputFnGetAttMyParentResourceAttOfParentResourceC2D0BB9E: {
                    Value: { 'Fn::GetAtt': ['MyParentResource', 'AttOfParentResource'] },
                    Export: { Name: 'parent:ExportsOutputFnGetAttMyParentResourceAttOfParentResourceC2D0BB9E' },
                },
            },
        });
        test.deepEqual(assembly.getStackByName(childStack.stackName).template, {
            Resources: {
                MyChildResource: {
                    Type: 'Resource::Child',
                    Properties: {
                        ChildProp: {
                            'Fn::ImportValue': 'parent:ExportsOutputFnGetAttMyParentResourceAttOfParentResourceC2D0BB9E',
                        },
                    },
                },
            },
        });
        test.done();
    },
    'cross-stack reference (parent stack references substack)'(test) {
        // GIVEN
        const app = new lib_1.App();
        const parentStack = new lib_1.Stack(app, 'parent');
        const childStack = new lib_1.Stack(parentStack, 'child');
        // WHEN (a resource from the child stack references a resource from the parent stack)
        const childResource = new lib_1.CfnResource(childStack, 'MyChildResource', { type: 'Resource::Child' });
        new lib_1.CfnResource(parentStack, 'MyParentResource', {
            type: 'Resource::Parent',
            properties: {
                ParentProp: childResource.getAtt('AttributeOfChildResource'),
            },
        });
        // THEN
        const assembly = app.synth();
        test.deepEqual(assembly.getStackByName(parentStack.stackName).template, {
            Resources: {
                MyParentResource: {
                    Type: 'Resource::Parent',
                    Properties: {
                        ParentProp: { 'Fn::ImportValue': 'parentchild13F9359B:childExportsOutputFnGetAttMyChildResourceAttributeOfChildResource420052FC' },
                    },
                },
            },
        });
        test.deepEqual(assembly.getStackByName(childStack.stackName).template, {
            Resources: { MyChildResource: { Type: 'Resource::Child' } },
            Outputs: {
                ExportsOutputFnGetAttMyChildResourceAttributeOfChildResource52813264: {
                    Value: { 'Fn::GetAtt': ['MyChildResource', 'AttributeOfChildResource'] },
                    Export: { Name: 'parentchild13F9359B:childExportsOutputFnGetAttMyChildResourceAttributeOfChildResource420052FC' },
                },
            },
        });
        test.done();
    },
    'cannot create cyclic reference between stacks'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        const account2 = new lib_1.ScopedAws(stack2).accountId;
        // WHEN
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: account1 });
        new lib_1.CfnParameter(stack1, 'SomeParameter', { type: 'String', default: account2 });
        test.throws(() => {
            app.synth();
            // eslint-disable-next-line max-len
        }, "'Stack2' depends on 'Stack1' (Stack2/SomeParameter -> Stack1.AWS::AccountId). Adding this dependency (Stack1/SomeParameter -> Stack2.AWS::AccountId) would create a cyclic reference.");
        test.done();
    },
    'stacks know about their dependencies'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1');
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2');
        // WHEN
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: account1 });
        app.synth();
        // THEN
        test.deepEqual(stack2.dependencies.map(s => s.node.id), ['Stack1']);
        test.done();
    },
    'cannot create references to stacks in other regions/accounts'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack1 = new lib_1.Stack(app, 'Stack1', { env: { account: '123456789012', region: 'es-norst-1' } });
        const account1 = new lib_1.ScopedAws(stack1).accountId;
        const stack2 = new lib_1.Stack(app, 'Stack2', { env: { account: '123456789012', region: 'es-norst-2' } });
        // WHEN
        new lib_1.CfnParameter(stack2, 'SomeParameter', { type: 'String', default: account1 });
        test.throws(() => {
            app.synth();
        }, /Stack "Stack2" cannot consume a cross reference from stack "Stack1"/);
        test.done();
    },
    'urlSuffix does not imply a stack dependency'(test) {
        // GIVEN
        const app = new lib_1.App();
        const first = new lib_1.Stack(app, 'First');
        const second = new lib_1.Stack(app, 'Second');
        // WHEN
        new lib_1.CfnOutput(second, 'Output', {
            value: first.urlSuffix,
        });
        // THEN
        app.synth();
        test.equal(second.dependencies.length, 0);
        test.done();
    },
    'stack with region supplied via props returns literal value'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'Stack1', { env: { account: '123456789012', region: 'es-norst-1' } });
        // THEN
        test.equal(stack.resolve(stack.region), 'es-norst-1');
        test.done();
    },
    'overrideLogicalId(id) can be used to override the logical ID of a resource'(test) {
        // GIVEN
        const stack = new lib_1.Stack();
        const bonjour = new lib_1.CfnResource(stack, 'BonjourResource', { type: 'Resource::Type' });
        // { Ref } and { GetAtt }
        new lib_1.CfnResource(stack, 'RefToBonjour', {
            type: 'Other::Resource',
            properties: {
                RefToBonjour: bonjour.ref,
                GetAttBonjour: bonjour.getAtt('TheAtt').toString(),
            },
        });
        bonjour.overrideLogicalId('BOOM');
        // THEN
        test.deepEqual(util_2.toCloudFormation(stack), {
            Resources: {
                BOOM: { Type: 'Resource::Type' },
                RefToBonjour: {
                    Type: 'Other::Resource',
                    Properties: {
                        RefToBonjour: { Ref: 'BOOM' },
                        GetAttBonjour: { 'Fn::GetAtt': ['BOOM', 'TheAtt'] },
                    },
                },
            },
        });
        test.done();
    },
    'Stack name can be overridden via properties'(test) {
        // WHEN
        const stack = new lib_1.Stack(undefined, 'Stack', { stackName: 'otherName' });
        // THEN
        test.deepEqual(stack.stackName, 'otherName');
        test.done();
    },
    'Stack name is inherited from App name if available'(test) {
        // WHEN
        const root = new lib_1.App();
        const app = new lib_1.Construct(root, 'Prod');
        const stack = new lib_1.Stack(app, 'Stack');
        // THEN
        test.deepEqual(stack.stackName, 'ProdStackD5279B22');
        test.done();
    },
    'stack construct id does not go through stack name validation if there is an explicit stack name'(test) {
        // GIVEN
        const app = new lib_1.App();
        // WHEN
        const stack = new lib_1.Stack(app, 'invalid as : stack name, but thats fine', {
            stackName: 'valid-stack-name',
        });
        // THEN
        const session = app.synth();
        test.deepEqual(stack.stackName, 'valid-stack-name');
        test.ok(session.tryGetArtifact(stack.artifactId));
        test.done();
    },
    'stack validation is performed on explicit stack name'(test) {
        // GIVEN
        const app = new lib_1.App();
        // THEN
        test.throws(() => new lib_1.Stack(app, 'boom', { stackName: 'invalid:stack:name' }), /Stack name must match the regular expression/);
        test.done();
    },
    'Stack.of(stack) returns the correct stack'(test) {
        const stack = new lib_1.Stack();
        test.same(lib_1.Stack.of(stack), stack);
        const parent = new lib_1.Construct(stack, 'Parent');
        const construct = new lib_1.Construct(parent, 'Construct');
        test.same(lib_1.Stack.of(construct), stack);
        test.done();
    },
    'Stack.of() throws when there is no parent Stack'(test) {
        const root = new lib_1.Construct(undefined, 'Root');
        const construct = new lib_1.Construct(root, 'Construct');
        test.throws(() => lib_1.Stack.of(construct), /should be created in the scope of a Stack, but no Stack found/);
        test.done();
    },
    'Stack.of() works for substacks'(test) {
        // GIVEN
        const app = new lib_1.App();
        // WHEN
        const parentStack = new lib_1.Stack(app, 'ParentStack');
        const parentResource = new lib_1.CfnResource(parentStack, 'ParentResource', { type: 'parent::resource' });
        // we will define a substack under the /resource/... just for giggles.
        const childStack = new lib_1.Stack(parentResource, 'ChildStack');
        const childResource = new lib_1.CfnResource(childStack, 'ChildResource', { type: 'child::resource' });
        // THEN
        test.same(lib_1.Stack.of(parentStack), parentStack);
        test.same(lib_1.Stack.of(parentResource), parentStack);
        test.same(lib_1.Stack.of(childStack), childStack);
        test.same(lib_1.Stack.of(childResource), childStack);
        test.done();
    },
    'stack.availabilityZones falls back to Fn::GetAZ[0],[2] if region is not specified'(test) {
        // GIVEN
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'MyStack');
        // WHEN
        const azs = stack.availabilityZones;
        // THEN
        test.deepEqual(stack.resolve(azs), [
            { 'Fn::Select': [0, { 'Fn::GetAZs': '' }] },
            { 'Fn::Select': [1, { 'Fn::GetAZs': '' }] },
        ]);
        test.done();
    },
    'stack.templateFile is the name of the template file emitted to the cloud assembly (default is to use the stack name)'(test) {
        // GIVEN
        const app = new lib_1.App();
        // WHEN
        const stack1 = new lib_1.Stack(app, 'MyStack1');
        const stack2 = new lib_1.Stack(app, 'MyStack2', { stackName: 'MyRealStack2' });
        // THEN
        test.deepEqual(stack1.templateFile, 'MyStack1.template.json');
        test.deepEqual(stack2.templateFile, 'MyRealStack2.template.json');
        test.done();
    },
    'when feature flag is enabled we will use the artifact id as the template name'(test) {
        // GIVEN
        const app = new lib_1.App({
            context: {
                [cxapi.ENABLE_STACK_NAME_DUPLICATES_CONTEXT]: 'true',
            },
        });
        // WHEN
        const stack1 = new lib_1.Stack(app, 'MyStack1');
        const stack2 = new lib_1.Stack(app, 'MyStack2', { stackName: 'MyRealStack2' });
        // THEN
        test.deepEqual(stack1.templateFile, 'MyStack1.template.json');
        test.deepEqual(stack2.templateFile, 'MyStack2.template.json');
        test.done();
    },
    '@aws-cdk/core:enableStackNameDuplicates': {
        'disabled (default)': {
            'artifactId and templateFile use the stack name'(test) {
                // GIVEN
                const app = new lib_1.App();
                // WHEN
                const stack1 = new lib_1.Stack(app, 'MyStack1', { stackName: 'thestack' });
                const assembly = app.synth();
                // THEN
                test.deepEqual(stack1.artifactId, 'thestack');
                test.deepEqual(stack1.templateFile, 'thestack.template.json');
                test.deepEqual(assembly.getStackArtifact(stack1.artifactId).templateFile, 'thestack.template.json');
                test.done();
            },
        },
        'enabled': {
            'allows using the same stack name for two stacks (i.e. in different regions)'(test) {
                // GIVEN
                const app = new lib_1.App({ context: { [cxapi.ENABLE_STACK_NAME_DUPLICATES_CONTEXT]: 'true' } });
                // WHEN
                const stack1 = new lib_1.Stack(app, 'MyStack1', { stackName: 'thestack' });
                const stack2 = new lib_1.Stack(app, 'MyStack2', { stackName: 'thestack' });
                const assembly = app.synth();
                // THEN
                test.deepEqual(assembly.getStackArtifact(stack1.artifactId).templateFile, 'MyStack1.template.json');
                test.deepEqual(assembly.getStackArtifact(stack2.artifactId).templateFile, 'MyStack2.template.json');
                test.deepEqual(stack1.templateFile, 'MyStack1.template.json');
                test.deepEqual(stack2.templateFile, 'MyStack2.template.json');
                test.done();
            },
            'artifactId and templateFile use the unique id and not the stack name'(test) {
                // GIVEN
                const app = new lib_1.App({ context: { [cxapi.ENABLE_STACK_NAME_DUPLICATES_CONTEXT]: 'true' } });
                // WHEN
                const stack1 = new lib_1.Stack(app, 'MyStack1', { stackName: 'thestack' });
                const assembly = app.synth();
                // THEN
                test.deepEqual(stack1.artifactId, 'MyStack1');
                test.deepEqual(stack1.templateFile, 'MyStack1.template.json');
                test.deepEqual(assembly.getStackArtifact(stack1.artifactId).templateFile, 'MyStack1.template.json');
                test.done();
            },
        },
    },
    'metadata is collected at the stack boundary'(test) {
        // GIVEN
        const app = new lib_1.App({
            context: {
                [cxapi.DISABLE_METADATA_STACK_TRACE]: 'true',
            },
        });
        const parent = new lib_1.Stack(app, 'parent');
        const child = new lib_1.Stack(parent, 'child');
        // WHEN
        child.node.addMetadata('foo', 'bar');
        // THEN
        const asm = app.synth();
        test.deepEqual(asm.getStackByName(parent.stackName).findMetadataByType('foo'), []);
        test.deepEqual(asm.getStackByName(child.stackName).findMetadataByType('foo'), [
            { path: '/parent/child', type: 'foo', data: 'bar' },
        ]);
        test.done();
    },
    'stack tags are reflected in the stack cloud assembly artifact'(test) {
        // GIVEN
        const app = new lib_1.App({ stackTraces: false });
        const stack1 = new lib_1.Stack(app, 'stack1');
        const stack2 = new lib_1.Stack(stack1, 'stack2');
        // WHEN
        lib_1.Tags.of(app).add('foo', 'bar');
        // THEN
        const asm = app.synth();
        const expected = [
            {
                type: 'aws:cdk:stack-tags',
                data: [{ key: 'foo', value: 'bar' }],
            },
        ];
        test.deepEqual(asm.getStackArtifact(stack1.artifactId).manifest.metadata, { '/stack1': expected });
        test.deepEqual(asm.getStackArtifact(stack2.artifactId).manifest.metadata, { '/stack1/stack2': expected });
        test.done();
    },
    'Termination Protection is reflected in Cloud Assembly artifact'(test) {
        // if the root is an app, invoke "synth" to avoid double synthesis
        const app = new lib_1.App();
        const stack = new lib_1.Stack(app, 'Stack', { terminationProtection: true });
        const assembly = app.synth();
        const artifact = assembly.getStackArtifact(stack.artifactId);
        test.equals(artifact.terminationProtection, true);
        test.done();
    },
    'users can (still) override "synthesize()" in stack'(test) {
        let called = false;
        class MyStack extends lib_1.Stack {
            synthesize(session) {
                called = true;
                test.ok(session.outdir);
                test.equal(session.assembly.outdir, session.outdir);
            }
        }
        const app = new lib_1.App();
        new MyStack(app, 'my-stack');
        app.synth();
        test.ok(called, 'synthesize() not called for Stack');
        test.done();
    },
    'context can be set on a stack using a LegacySynthesizer'(test) {
        // WHEN
        const stack = new lib_1.Stack(undefined, undefined, {
            synthesizer: new lib_1.LegacyStackSynthesizer(),
        });
        stack.node.setContext('something', 'value');
        // THEN: no exception
        test.done();
    },
    'context can be set on a stack using a DefaultSynthesizer'(test) {
        // WHEN
        const stack = new lib_1.Stack(undefined, undefined, {
            synthesizer: new lib_1.DefaultStackSynthesizer(),
        });
        stack.node.setContext('something', 'value');
        // THEN: no exception
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5zdGFjay5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRlc3Quc3RhY2sudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLHlDQUF5QztBQUV6QyxnQ0FHZ0I7QUFDaEIsd0RBQXFEO0FBQ3JELDhDQUF3RDtBQUN4RCxzQ0FBK0M7QUFDL0MsaUNBQTBDO0FBMDhCMUMsTUFBTSxzQkFBdUIsU0FBUSxXQUFLO0lBRXhDLE1BQU07SUFFQyxpQkFBaUI7UUFDdEIsTUFBTSxRQUFRLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixFQUFFLENBQUM7UUFFM0MsbURBQW1EO1FBQ25ELFFBQVEsQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxXQUFXLENBQUMsR0FBRztZQUN0RCxRQUFRLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxVQUFVLENBQUMsV0FBVyxDQUFDLEdBQUcsQ0FBQztRQUMzRCxPQUFPLFFBQVEsQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFDLFVBQVUsQ0FBQyxXQUFXLENBQUMsR0FBRyxDQUFDO1FBRWhFLE9BQU8sUUFBUSxDQUFDO0lBQ2xCLENBQUM7Q0FDRjtBQXQ5QkQsaUJBQVM7SUFDUCxpRkFBaUYsQ0FBQyxJQUFVO1FBQzFGLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7UUFDMUIsSUFBSSxDQUFDLFNBQVMsQ0FBQyx1QkFBZ0IsQ0FBQyxLQUFLLENBQUMsRUFBRSxFQUFHLENBQUMsQ0FBQztRQUM3QyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsMkZBQTJGLENBQUMsSUFBVTtRQUNwRyxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1FBQzFCLEtBQUssQ0FBQyxlQUFlLENBQUMscUJBQXFCLEdBQUcsbUJBQW1CLENBQUM7UUFDbEUsS0FBSyxDQUFDLGVBQWUsQ0FBQyxXQUFXLEdBQUcsd0JBQXdCLENBQUM7UUFDN0QsS0FBSyxDQUFDLGVBQWUsQ0FBQyxVQUFVLEdBQUcsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUM1QyxJQUFJLENBQUMsU0FBUyxDQUFDLHVCQUFnQixDQUFDLEtBQUssQ0FBQyxFQUFFO1lBQ3RDLFdBQVcsRUFBRSx3QkFBd0I7WUFDckMsd0JBQXdCLEVBQUUsbUJBQW1CO1lBQzdDLFNBQVMsRUFBRSxNQUFNO1NBQ2xCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxxREFBcUQsQ0FBQyxJQUFVO1FBQzlELE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxDQUFDLEdBQUcsSUFBSSxlQUFTLENBQUMsS0FBSyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQzVDLElBQUksQ0FBQyxFQUFFLENBQUMsV0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBQzlCLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxXQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDM0IsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDJFQUEyRSxDQUFDLElBQVU7UUFDcEYsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQzlDLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsWUFBWSxFQUFFLEVBQUUsSUFBSSxFQUFFLGdCQUFnQixFQUFFLENBQUMsQ0FBQztRQUVqRSxJQUFJLENBQUMsU0FBUyxDQUFDLHVCQUFnQixDQUFDLEtBQUssQ0FBQyxFQUFFLEVBQUUsU0FBUyxFQUFFLEVBQUUsVUFBVSxFQUFFLEVBQUUsSUFBSSxFQUFFLGdCQUFnQixFQUFFLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDbkcsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGlFQUFpRSxDQUFDLElBQVU7UUFDMUUsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztRQUUxQixLQUFLLENBQUMsZUFBZSxDQUFDLFdBQVcsR0FBRyxrQkFBa0IsQ0FBQztRQUN2RCxLQUFLLENBQUMsZUFBZSxDQUFDLHFCQUFxQixHQUFHLGlCQUFpQixDQUFDO1FBQ2hFLEtBQUssQ0FBQyxlQUFlLENBQUMsU0FBUyxHQUFHLGlCQUFpQixDQUFDO1FBQ3BELEtBQUssQ0FBQyxlQUFlLENBQUMsVUFBVSxHQUFHLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDakQsS0FBSyxDQUFDLGVBQWUsQ0FBQyxRQUFRLEdBQUc7WUFDL0IsV0FBVyxFQUFFLGVBQWU7U0FDN0IsQ0FBQztRQUVGLElBQUksQ0FBQyxTQUFTLENBQUMsdUJBQWdCLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDdEMsV0FBVyxFQUFFLGtCQUFrQjtZQUMvQixTQUFTLEVBQUUsQ0FBQyxXQUFXLEVBQUUsaUJBQWlCLENBQUM7WUFDM0Msd0JBQXdCLEVBQUUsaUJBQWlCO1lBQzNDLFFBQVEsRUFBRSxFQUFFLFdBQVcsRUFBRSxlQUFlLEVBQUU7U0FDM0MsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDJEQUEyRCxDQUFDLElBQVU7UUFDcEUsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztRQUUxQixLQUFLLENBQUMsZUFBZSxDQUFDLFVBQVUsR0FBRyxDQUFDLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxFQUFFLEdBQUcsQ0FBQyxDQUFDO1FBRXhELElBQUksQ0FBQyxTQUFTLENBQUMsdUJBQWdCLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDdEMsU0FBUyxFQUFFLENBQUMsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLENBQUM7U0FDM0IsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHVDQUF1QyxDQUFDLElBQVU7UUFDaEQsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztRQUUxQixLQUFLLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3hCLEtBQUssQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDeEIsS0FBSyxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUV4QixJQUFJLENBQUMsU0FBUyxDQUFDLHVCQUFnQixDQUFDLEtBQUssQ0FBQyxFQUFFO1lBQ3RDLFNBQVMsRUFBRSxDQUFDLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxDQUFDO1NBQzNCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxrRkFBa0Y7SUFDbEYsZ0VBQWdFO0lBQ2hFLDBGQUEwRjtJQUMxRix3REFBd0Q7SUFDeEQsa0hBQWtILENBQUMsSUFBVTtRQUUzSCxNQUFNLEtBQUssR0FBRyxJQUFJLHNCQUFzQixFQUFFLENBQUM7UUFFM0MsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7WUFDbkMsSUFBSSxFQUFFLGlCQUFpQjtZQUN2QixVQUFVLEVBQUU7Z0JBQ1YsT0FBTyxFQUFFLE9BQU87Z0JBQ2hCLE9BQU8sRUFBRSxPQUFPO2dCQUNoQixXQUFXLEVBQUU7b0JBQ1gsR0FBRyxFQUFFLE9BQU87aUJBQ2I7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLGlCQUFpQixFQUFFLEVBQUU7WUFDeEMsU0FBUyxFQUNUO2dCQUNFLFVBQVUsRUFDVDtvQkFDRSxJQUFJLEVBQUUsaUJBQWlCO29CQUN2QixVQUFVLEVBQ1g7d0JBQ0UsT0FBTyxFQUFFLE9BQU87d0JBQ2hCLE9BQU8sRUFBRSxPQUFPO3dCQUNoQixXQUFXLEVBQUUsRUFBRSxHQUFHLEVBQUUsT0FBTyxFQUFFO3FCQUM5QjtpQkFDRDthQUNIO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHlGQUF5RixDQUFDLElBQVU7UUFFbEcsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztRQUUxQixNQUFNLENBQUMsR0FBRyxJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBQ2pFLE1BQU0sQ0FBQyxHQUFHLElBQUksZUFBUyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFFLENBQUMsQ0FBQztRQUM5RCxNQUFNLENBQUMsR0FBRyxJQUFJLGtCQUFZLENBQUMsS0FBSyxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBRWpELElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUMvQyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDL0MsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBRS9DLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxzQ0FBc0MsQ0FBQyxJQUFVO1FBQy9DLE1BQU0sSUFBSSxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFFdkIsSUFBSSxXQUFLLENBQUMsSUFBSSxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBQy9CLGdCQUFnQjtRQUVoQixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNkNBQTZDLENBQUMsSUFBVTtRQUN0RCxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxJQUFJLFNBQUcsRUFBRSxFQUFFLFNBQVMsRUFBRSxFQUFFLFdBQVcsRUFBRSxzQkFBc0IsRUFBRSxDQUFDLENBQUM7UUFDdkYsTUFBTSxNQUFNLEdBQUcsdUJBQWdCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDdkMsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsV0FBVyxFQUFFLHNCQUFzQixDQUFDLENBQUM7UUFDdkQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDBDQUEwQyxDQUFDLElBQVU7UUFDbkQsTUFBTSxJQUFJLEdBQUc7Ozs7Ozs7Ozs7O3dEQVd1QyxDQUFDO1FBQ3JELElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxXQUFLLENBQUMsSUFBSSxTQUFHLEVBQUUsRUFBRSxTQUFTLEVBQUUsRUFBRSxXQUFXLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQzFFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxnRkFBZ0YsQ0FBQyxJQUFVO1FBQ3pGLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7UUFFMUIsTUFBTSxRQUFRLEdBQUc7WUFDZixXQUFXLEVBQUUsY0FBYztTQUM1QixDQUFDO1FBRUYsSUFBSSxnQkFBVSxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUUsRUFBRSxRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBRS9DLE1BQU0sTUFBTSxHQUFHLHVCQUFnQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBRXZDLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxNQUFNLENBQUMsV0FBVyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ2hELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx3RUFBd0UsQ0FBQyxJQUFVO1FBQ2pGLFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUN4QyxNQUFNLFFBQVEsR0FBRyxJQUFJLGVBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFDakQsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBRXhDLCtCQUErQjtRQUMvQixJQUFJLGtCQUFZLENBQUMsTUFBTSxFQUFFLGVBQWUsRUFBRSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsT0FBTyxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUM7UUFFakYsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM3QixNQUFNLFNBQVMsR0FBRyxRQUFRLENBQUMsY0FBYyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxRQUFRLENBQUM7UUFDckUsTUFBTSxTQUFTLEdBQUcsUUFBUSxDQUFDLGNBQWMsQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUMsUUFBUSxDQUFDO1FBRXJFLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxFQUFFO1lBQ3hCLE9BQU8sRUFBRTtnQkFDUCxvQ0FBb0MsRUFBRTtvQkFDcEMsS0FBSyxFQUFFLEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFO29CQUNoQyxNQUFNLEVBQUUsRUFBRSxJQUFJLEVBQUUsNkNBQTZDLEVBQUU7aUJBQ2hFO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsRUFBRTtZQUN4QixVQUFVLEVBQUU7Z0JBQ1YsYUFBYSxFQUFFO29CQUNiLElBQUksRUFBRSxRQUFRO29CQUNkLE9BQU8sRUFBRSxFQUFFLGlCQUFpQixFQUFFLDZDQUE2QyxFQUFFO2lCQUM5RTthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDREQUE0RCxDQUFDLElBQVU7UUFDckUsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ3hDLE1BQU0sU0FBUyxHQUFHLElBQUksaUJBQVcsQ0FBQyxNQUFNLEVBQUUsVUFBVSxFQUFFLEVBQUUsSUFBSSxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7UUFDdkUsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBRXhDLGtDQUFrQztRQUNsQyxJQUFJLGlCQUFXLENBQUMsTUFBTSxFQUFFLGNBQWMsRUFBRTtZQUN0QyxJQUFJLEVBQUUscUJBQXFCO1lBQzNCLFVBQVUsRUFBRTtnQkFDVixZQUFZLEVBQUUsSUFBSSxxQkFBUyxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUM7YUFDM0M7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzdCLE1BQU0sU0FBUyxHQUFHLFFBQVEsQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLFFBQVEsQ0FBQztRQUVyRSxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsRUFBRTtZQUN4QixTQUFTLEVBQUU7Z0JBQ1QsWUFBWSxFQUFFO29CQUNaLElBQUksRUFBRSxxQkFBcUI7b0JBQzNCLFVBQVUsRUFBRTt3QkFDVixZQUFZLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSx5Q0FBeUMsRUFBRTtxQkFDL0U7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxxRkFBcUYsQ0FBQyxJQUFVO1FBQzlGLFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsQ0FBQztZQUNsQixPQUFPLEVBQUU7Z0JBQ1Asb0NBQW9DLEVBQUUsTUFBTTthQUM3QztTQUNGLENBQUMsQ0FBQztRQUNILE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSxlQUFTLENBQUMsR0FBRyxFQUFFLFlBQVksQ0FBQyxDQUFDO1FBRTFELE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLGdCQUFnQixFQUFFLFFBQVEsRUFBRTtZQUNuRCxTQUFTLEVBQUUsUUFBUTtTQUNwQixDQUFDLENBQUM7UUFDSCxNQUFNLFNBQVMsR0FBRyxJQUFJLGlCQUFXLENBQUMsTUFBTSxFQUFFLFVBQVUsRUFBRSxFQUFFLElBQUksRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDO1FBQ3ZFLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLGdCQUFnQixFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBRXJELGtDQUFrQztRQUNsQyxJQUFJLGlCQUFXLENBQUMsTUFBTSxFQUFFLGNBQWMsRUFBRTtZQUN0QyxJQUFJLEVBQUUscUJBQXFCO1lBQzNCLFVBQVUsRUFBRTtnQkFDVixZQUFZLEVBQUUsSUFBSSxxQkFBUyxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUM7YUFDM0M7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzdCLE1BQU0sU0FBUyxHQUFHLFFBQVEsQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLFFBQVEsQ0FBQztRQUVyRSxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsRUFBRTtZQUN4QixTQUFTLEVBQUU7Z0JBQ1QsWUFBWSxFQUFFO29CQUNaLElBQUksRUFBRSxxQkFBcUI7b0JBQzNCLFVBQVUsRUFBRTt3QkFDVixZQUFZLEVBQUUsRUFBRSxpQkFBaUIsRUFBRSx5Q0FBeUMsRUFBRTtxQkFDL0U7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw0Q0FBNEMsQ0FBQyxJQUFVO1FBQ3JELFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUN4QyxNQUFNLFFBQVEsR0FBRyxJQUFJLGVBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFDakQsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBRXhDLCtCQUErQjtRQUMvQixJQUFJLGtCQUFZLENBQUMsTUFBTSxFQUFFLGVBQWUsRUFBRSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsT0FBTyxFQUFFLFVBQUksQ0FBQyxXQUFXLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsUUFBUSxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUM7UUFFdEgsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzdCLE1BQU0sU0FBUyxHQUFHLFFBQVEsQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLFFBQVEsQ0FBQztRQUNyRSxNQUFNLFNBQVMsR0FBRyxRQUFRLENBQUMsY0FBYyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxRQUFRLENBQUM7UUFFckUsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxFQUFFO1lBQ3hCLE9BQU8sRUFBRTtnQkFDUCxvQ0FBb0MsRUFBRTtvQkFDcEMsS0FBSyxFQUFFLEVBQUUsR0FBRyxFQUFFLGdCQUFnQixFQUFFO29CQUNoQyxNQUFNLEVBQUUsRUFBRSxJQUFJLEVBQUUsNkNBQTZDLEVBQUU7aUJBQ2hFO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsRUFBRTtZQUN4QixVQUFVLEVBQUU7Z0JBQ1YsYUFBYSxFQUFFO29CQUNiLElBQUksRUFBRSxRQUFRO29CQUNkLE9BQU8sRUFBRSxFQUFFLGlCQUFpQixFQUFFLDZDQUE2QyxFQUFFO2lCQUM5RTthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDRIQUE0SCxDQUFDLElBQVU7UUFDckksUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ3hDLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUV4QywrQkFBK0I7UUFDL0IsSUFBSSxlQUFTLENBQUMsTUFBTSxFQUFFLFdBQVcsRUFBRSxFQUFFLEtBQUssRUFBRSxNQUFNLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQztRQUM3RCxJQUFJLGVBQVMsQ0FBQyxNQUFNLEVBQUUsWUFBWSxFQUFFLEVBQUUsS0FBSyxFQUFFLE1BQU0sQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO1FBRS9ELE9BQU87UUFDUCxNQUFNLFFBQVEsR0FBRyxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDN0IsTUFBTSxTQUFTLEdBQUcsUUFBUSxDQUFDLGNBQWMsQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLENBQUMsUUFBUSxDQUFDO1FBRXJFLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxFQUFFO1lBQ3hCLE9BQU8sRUFBRTtnQkFDUCxTQUFTLEVBQUU7b0JBQ1QsS0FBSyxFQUFFLEVBQUUsR0FBRyxFQUFFLGFBQWEsRUFBRTtpQkFDOUI7Z0JBQ0QsVUFBVSxFQUFFO29CQUNWLEtBQUssRUFBRSxFQUFFLEdBQUcsRUFBRSxnQkFBZ0IsRUFBRTtpQkFDakM7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx3Q0FBd0MsQ0FBQyxJQUFVO1FBQ2pELFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUN4QyxNQUFNLFFBQVEsR0FBRyxJQUFJLGVBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFDakQsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBRXhDLCtCQUErQjtRQUMvQixJQUFJLGtCQUFZLENBQUMsTUFBTSxFQUFFLGVBQWUsRUFBRSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsT0FBTyxFQUFFLGVBQWUsUUFBUSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWxHLE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM3QixNQUFNLFNBQVMsR0FBRyxRQUFRLENBQUMsY0FBYyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxRQUFRLENBQUM7UUFFckUsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxFQUFFO1lBQ3hCLFVBQVUsRUFBRTtnQkFDVixhQUFhLEVBQUU7b0JBQ2IsSUFBSSxFQUFFLFFBQVE7b0JBQ2QsT0FBTyxFQUFFLEVBQUUsVUFBVSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsY0FBYyxFQUFFLEVBQUUsaUJBQWlCLEVBQUUsNkNBQTZDLEVBQUUsQ0FBQyxDQUFDLEVBQUU7aUJBQ3RIO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsK0VBQStFLENBQUMsSUFBVTtRQUN4RixRQUFRO1FBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLEVBQUUsQ0FBQztRQUN0QixNQUFNLE1BQU0sR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDeEMsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsTUFBTSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQzNDLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLE1BQU0sRUFBRSxRQUFRLENBQUMsQ0FBQztRQUMzQyxNQUFNLFNBQVMsR0FBRyxJQUFJLGlCQUFXLENBQUMsTUFBTSxFQUFFLFdBQVcsRUFBRSxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQ3ZFLE1BQU0sU0FBUyxHQUFHLElBQUksaUJBQVcsQ0FBQyxNQUFNLEVBQUUsV0FBVyxFQUFFLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7UUFFdkUsT0FBTztRQUNQLE1BQU0sU0FBUyxHQUFHLElBQUksaUJBQVcsQ0FBQyxNQUFNLEVBQUUsV0FBVyxFQUFFO1lBQ3JELElBQUksRUFBRSxJQUFJO1lBQ1YsVUFBVSxFQUFFO2dCQUNWLGNBQWMsRUFBRSxTQUFTLENBQUMsR0FBRzthQUM5QjtTQUNGLENBQUMsQ0FBQztRQUNILFNBQVMsQ0FBQyxZQUFZLENBQUMsU0FBUyxDQUFDLENBQUM7UUFFbEMsT0FBTztRQUNQLE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM3QixNQUFNLGNBQWMsR0FBRyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxDQUFDLFFBQVEsQ0FBQztRQUM3RSxNQUFNLGNBQWMsR0FBRyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxDQUFDLFFBQVEsQ0FBQztRQUM3RSxNQUFNLGNBQWMsR0FBRyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxDQUFDLFFBQVEsQ0FBQztRQUU3RSxJQUFJLENBQUMsU0FBUyxDQUFDLGNBQWMsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNuQyxJQUFJLENBQUMsU0FBUyxDQUFDLGNBQWMsRUFBRTtZQUM3QixTQUFTLEVBQUU7Z0JBQ1QsU0FBUyxFQUFFLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRTtnQkFDekIsU0FBUyxFQUFFLEVBQUUsSUFBSSxFQUFFLElBQUksRUFBRTthQUMxQjtZQUNELE9BQU8sRUFBRTtnQkFDUCxpQ0FBaUMsRUFBRTtvQkFDakMsS0FBSyxFQUFFLEVBQUUsR0FBRyxFQUFFLFdBQVcsRUFBRTtvQkFDM0IsTUFBTSxFQUFFLEVBQUUsSUFBSSxFQUFFLDhEQUE4RCxFQUFFO2lCQUNqRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQyxjQUFjLEVBQUU7WUFDN0IsU0FBUyxFQUFFO2dCQUNULFNBQVMsRUFBRTtvQkFDVCxJQUFJLEVBQUUsSUFBSTtvQkFDVixVQUFVLEVBQUU7d0JBQ1YsY0FBYyxFQUFFLEVBQUUsaUJBQWlCLEVBQUUsOERBQThELEVBQUU7cUJBQ3RHO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUM3RixJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUMsQ0FBQztRQUNuSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsOERBQThELENBQUMsSUFBVTtRQUN2RSxRQUFRO1FBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLEVBQUUsQ0FBQztRQUN0QixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFFekMsT0FBTztRQUNQLE1BQU0sT0FBUSxTQUFRLGlCQUFXO1lBQ3hCLGlCQUFpQjtnQkFDdEIsT0FBTyxJQUFJLHVCQUFnQixDQUFDO29CQUMxQixHQUFHLEVBQUUsSUFBSTtpQkFDVixFQUFFLEtBQUssQ0FBQyxFQUFFO29CQUNULG9CQUFjLENBQUMsS0FBSyxDQUFDLENBQUMsYUFBYSxFQUFFLENBQUM7Z0JBQ3hDLENBQUMsQ0FBQyxDQUFDO1lBQ0wsQ0FBQztTQUNGO1FBRUQsSUFBSSxPQUFPLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRSxFQUFFLElBQUksRUFBRSxXQUFXLEVBQUUsQ0FBQyxDQUFDO1FBRXJELE9BQU87UUFDUCx3QkFBaUIsQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUV2QixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsMkZBQTJGLENBQUMsSUFBVTtRQUNwRyxRQUFRO1FBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLEVBQUUsQ0FBQztRQUV0QixPQUFPO1FBQ1AsTUFBTSxXQUFXLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQzdDLE1BQU0sVUFBVSxHQUFHLElBQUksV0FBSyxDQUFDLFdBQVcsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUNuRCxJQUFJLGlCQUFXLENBQUMsV0FBVyxFQUFFLGtCQUFrQixFQUFFLEVBQUUsSUFBSSxFQUFFLGtCQUFrQixFQUFFLENBQUMsQ0FBQztRQUMvRSxJQUFJLGlCQUFXLENBQUMsVUFBVSxFQUFFLGlCQUFpQixFQUFFLEVBQUUsSUFBSSxFQUFFLGlCQUFpQixFQUFFLENBQUMsQ0FBQztRQUU1RSxPQUFPO1FBQ1AsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzdCLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLENBQUMsUUFBUSxFQUFFLEVBQUUsU0FBUyxFQUFFLEVBQUUsZ0JBQWdCLEVBQUUsRUFBRSxJQUFJLEVBQUUsa0JBQWtCLEVBQUUsRUFBRSxFQUFFLENBQUMsQ0FBQztRQUMzSSxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxjQUFjLENBQUMsVUFBVSxDQUFDLFNBQVMsQ0FBQyxDQUFDLFFBQVEsRUFBRSxFQUFFLFNBQVMsRUFBRSxFQUFFLGVBQWUsRUFBRSxFQUFFLElBQUksRUFBRSxpQkFBaUIsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ3hJLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwwREFBMEQsQ0FBQyxJQUFVO1FBQ25FLFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sV0FBVyxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUM3QyxNQUFNLFVBQVUsR0FBRyxJQUFJLFdBQUssQ0FBQyxXQUFXLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFFbkQscUZBQXFGO1FBQ3JGLE1BQU0sY0FBYyxHQUFHLElBQUksaUJBQVcsQ0FBQyxXQUFXLEVBQUUsa0JBQWtCLEVBQUUsRUFBRSxJQUFJLEVBQUUsa0JBQWtCLEVBQUUsQ0FBQyxDQUFDO1FBQ3RHLElBQUksaUJBQVcsQ0FBQyxVQUFVLEVBQUUsaUJBQWlCLEVBQUU7WUFDN0MsSUFBSSxFQUFFLGlCQUFpQjtZQUN2QixVQUFVLEVBQUU7Z0JBQ1YsU0FBUyxFQUFFLGNBQWMsQ0FBQyxNQUFNLENBQUMscUJBQXFCLENBQUM7YUFDeEQ7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzdCLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLGNBQWMsQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLENBQUMsUUFBUSxFQUFFO1lBQ3RFLFNBQVMsRUFBRSxFQUFFLGdCQUFnQixFQUFFLEVBQUUsSUFBSSxFQUFFLGtCQUFrQixFQUFFLEVBQUU7WUFDN0QsT0FBTyxFQUFFO2dCQUNQLGdFQUFnRSxFQUFFO29CQUNoRSxLQUFLLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBQyxrQkFBa0IsRUFBRSxxQkFBcUIsQ0FBQyxFQUFFO29CQUNwRSxNQUFNLEVBQUUsRUFBRSxJQUFJLEVBQUUseUVBQXlFLEVBQUU7aUJBQzVGO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxjQUFjLENBQUMsVUFBVSxDQUFDLFNBQVMsQ0FBQyxDQUFDLFFBQVEsRUFBRTtZQUNyRSxTQUFTLEVBQUU7Z0JBQ1QsZUFBZSxFQUFFO29CQUNmLElBQUksRUFBRSxpQkFBaUI7b0JBQ3ZCLFVBQVUsRUFBRTt3QkFDVixTQUFTLEVBQUU7NEJBQ1QsaUJBQWlCLEVBQUUseUVBQXlFO3lCQUM3RjtxQkFDRjtpQkFDRjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDBEQUEwRCxDQUFDLElBQVU7UUFDbkUsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxXQUFXLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQzdDLE1BQU0sVUFBVSxHQUFHLElBQUksV0FBSyxDQUFDLFdBQVcsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUVuRCxxRkFBcUY7UUFDckYsTUFBTSxhQUFhLEdBQUcsSUFBSSxpQkFBVyxDQUFDLFVBQVUsRUFBRSxpQkFBaUIsRUFBRSxFQUFFLElBQUksRUFBRSxpQkFBaUIsRUFBRSxDQUFDLENBQUM7UUFDbEcsSUFBSSxpQkFBVyxDQUFDLFdBQVcsRUFBRSxrQkFBa0IsRUFBRTtZQUMvQyxJQUFJLEVBQUUsa0JBQWtCO1lBQ3hCLFVBQVUsRUFBRTtnQkFDVixVQUFVLEVBQUUsYUFBYSxDQUFDLE1BQU0sQ0FBQywwQkFBMEIsQ0FBQzthQUM3RDtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLFFBQVEsR0FBRyxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDN0IsSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsY0FBYyxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsQ0FBQyxRQUFRLEVBQUU7WUFDdEUsU0FBUyxFQUFFO2dCQUNULGdCQUFnQixFQUFFO29CQUNoQixJQUFJLEVBQUUsa0JBQWtCO29CQUN4QixVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFLEVBQUUsaUJBQWlCLEVBQUUsK0ZBQStGLEVBQUU7cUJBQ25JO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxjQUFjLENBQUMsVUFBVSxDQUFDLFNBQVMsQ0FBQyxDQUFDLFFBQVEsRUFBRTtZQUNyRSxTQUFTLEVBQUUsRUFBRSxlQUFlLEVBQUUsRUFBRSxJQUFJLEVBQUUsaUJBQWlCLEVBQUUsRUFBRTtZQUMzRCxPQUFPLEVBQUU7Z0JBQ1Asb0VBQW9FLEVBQUU7b0JBQ3BFLEtBQUssRUFBRSxFQUFFLFlBQVksRUFBRSxDQUFDLGlCQUFpQixFQUFFLDBCQUEwQixDQUFDLEVBQUU7b0JBQ3hFLE1BQU0sRUFBRSxFQUFFLElBQUksRUFBRSwrRkFBK0YsRUFBRTtpQkFDbEg7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwrQ0FBK0MsQ0FBQyxJQUFVO1FBQ3hELFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUN4QyxNQUFNLFFBQVEsR0FBRyxJQUFJLGVBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFDakQsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ3hDLE1BQU0sUUFBUSxHQUFHLElBQUksZUFBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUVqRCxPQUFPO1FBQ1AsSUFBSSxrQkFBWSxDQUFDLE1BQU0sRUFBRSxlQUFlLEVBQUUsRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFLE9BQU8sRUFBRSxRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBQ2pGLElBQUksa0JBQVksQ0FBQyxNQUFNLEVBQUUsZUFBZSxFQUFFLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxPQUFPLEVBQUUsUUFBUSxFQUFFLENBQUMsQ0FBQztRQUVqRixJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNmLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUNaLG1DQUFtQztRQUNyQyxDQUFDLEVBQUUsdUxBQXVMLENBQUMsQ0FBQztRQUU1TCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsc0NBQXNDLENBQUMsSUFBVTtRQUMvQyxRQUFRO1FBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLEVBQUUsQ0FBQztRQUN0QixNQUFNLE1BQU0sR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDeEMsTUFBTSxRQUFRLEdBQUcsSUFBSSxlQUFTLENBQUMsTUFBTSxDQUFDLENBQUMsU0FBUyxDQUFDO1FBQ2pELE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUV4QyxPQUFPO1FBQ1AsSUFBSSxrQkFBWSxDQUFDLE1BQU0sRUFBRSxlQUFlLEVBQUUsRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFLE9BQU8sRUFBRSxRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBRWpGLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUVaLE9BQU87UUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7UUFFcEUsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDhEQUE4RCxDQUFDLElBQVU7UUFDdkUsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsRUFBRSxFQUFFLEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUUsTUFBTSxFQUFFLFlBQVksRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNwRyxNQUFNLFFBQVEsR0FBRyxJQUFJLGVBQVMsQ0FBQyxNQUFNLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFDakQsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsRUFBRSxFQUFFLEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUUsTUFBTSxFQUFFLFlBQVksRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVwRyxPQUFPO1FBQ1AsSUFBSSxrQkFBWSxDQUFDLE1BQU0sRUFBRSxlQUFlLEVBQUUsRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFLE9BQU8sRUFBRSxRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBRWpGLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO1lBQ2YsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQ2QsQ0FBQyxFQUFFLHFFQUFxRSxDQUFDLENBQUM7UUFFMUUsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDZDQUE2QyxDQUFDLElBQVU7UUFDdEQsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ3RDLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUV4QyxPQUFPO1FBQ1AsSUFBSSxlQUFTLENBQUMsTUFBTSxFQUFFLFFBQVEsRUFBRTtZQUM5QixLQUFLLEVBQUUsS0FBSyxDQUFDLFNBQVM7U0FDdkIsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUVaLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLFlBQVksQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFFMUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDREQUE0RCxDQUFDLElBQVU7UUFDckUsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsRUFBRSxFQUFFLEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUUsTUFBTSxFQUFFLFlBQVksRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVuRyxPQUFPO1FBQ1AsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsRUFBRSxZQUFZLENBQUMsQ0FBQztRQUV0RCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNEVBQTRFLENBQUMsSUFBVTtRQUNyRixRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLE9BQU8sR0FBRyxJQUFJLGlCQUFXLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFLEVBQUUsSUFBSSxFQUFFLGdCQUFnQixFQUFFLENBQUMsQ0FBQztRQUV0Rix5QkFBeUI7UUFDekIsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7WUFDckMsSUFBSSxFQUFFLGlCQUFpQjtZQUN2QixVQUFVLEVBQUU7Z0JBQ1YsWUFBWSxFQUFFLE9BQU8sQ0FBQyxHQUFHO2dCQUN6QixhQUFhLEVBQUUsT0FBTyxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxRQUFRLEVBQUU7YUFDbkQ7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPLENBQUMsaUJBQWlCLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFbEMsT0FBTztRQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsdUJBQWdCLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDdEMsU0FBUyxFQUNUO2dCQUNFLElBQUksRUFBRSxFQUFFLElBQUksRUFBRSxnQkFBZ0IsRUFBRTtnQkFDaEMsWUFBWSxFQUNYO29CQUNFLElBQUksRUFBRSxpQkFBaUI7b0JBQ3ZCLFVBQVUsRUFDVDt3QkFDRSxZQUFZLEVBQUUsRUFBRSxHQUFHLEVBQUUsTUFBTSxFQUFFO3dCQUM3QixhQUFhLEVBQUUsRUFBRSxZQUFZLEVBQUUsQ0FBQyxNQUFNLEVBQUUsUUFBUSxDQUFDLEVBQUU7cUJBQ3BEO2lCQUNIO2FBQ0g7U0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNkNBQTZDLENBQUMsSUFBVTtRQUN0RCxPQUFPO1FBQ1AsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsU0FBUyxFQUFFLE9BQU8sRUFBRSxFQUFFLFNBQVMsRUFBRSxXQUFXLEVBQUUsQ0FBQyxDQUFDO1FBRXhFLE9BQU87UUFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFFN0MsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELG9EQUFvRCxDQUFDLElBQVU7UUFDN0QsT0FBTztRQUNQLE1BQU0sSUFBSSxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFDdkIsTUFBTSxHQUFHLEdBQUcsSUFBSSxlQUFTLENBQUMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3hDLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUV0QyxPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLG1CQUFtQixDQUFDLENBQUM7UUFFckQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGlHQUFpRyxDQUFDLElBQVU7UUFDMUcsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFFdEIsT0FBTztRQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSx5Q0FBeUMsRUFBRTtZQUN0RSxTQUFTLEVBQUUsa0JBQWtCO1NBQzlCLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLE9BQU8sR0FBRyxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDNUIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLGtCQUFrQixDQUFDLENBQUM7UUFDcEQsSUFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO1FBQ2xELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxzREFBc0QsQ0FBQyxJQUFVO1FBQy9ELFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBRXRCLE9BQU87UUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxNQUFNLEVBQUUsRUFBRSxTQUFTLEVBQUUsb0JBQW9CLEVBQUUsQ0FBQyxFQUMzRSw4Q0FBOEMsQ0FBQyxDQUFDO1FBRWxELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCwyQ0FBMkMsQ0FBQyxJQUFVO1FBQ3BELE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7UUFDMUIsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ2xDLE1BQU0sTUFBTSxHQUFHLElBQUksZUFBUyxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQztRQUM5QyxNQUFNLFNBQVMsR0FBRyxJQUFJLGVBQVMsQ0FBQyxNQUFNLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDckQsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFLLENBQUMsRUFBRSxDQUFDLFNBQVMsQ0FBQyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ3RDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxpREFBaUQsQ0FBQyxJQUFVO1FBQzFELE1BQU0sSUFBSSxHQUFHLElBQUksZUFBUyxDQUFDLFNBQWdCLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDckQsTUFBTSxTQUFTLEdBQUcsSUFBSSxlQUFTLENBQUMsSUFBSSxFQUFFLFdBQVcsQ0FBQyxDQUFDO1FBQ25ELElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsV0FBSyxDQUFDLEVBQUUsQ0FBQyxTQUFTLENBQUMsRUFBRSwrREFBK0QsQ0FBQyxDQUFDO1FBQ3hHLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCxnQ0FBZ0MsQ0FBQyxJQUFVO1FBQ3pDLFFBQVE7UUFDUixNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBRXRCLE9BQU87UUFDUCxNQUFNLFdBQVcsR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFDbEQsTUFBTSxjQUFjLEdBQUcsSUFBSSxpQkFBVyxDQUFDLFdBQVcsRUFBRSxnQkFBZ0IsRUFBRSxFQUFFLElBQUksRUFBRSxrQkFBa0IsRUFBRSxDQUFDLENBQUM7UUFFcEcsc0VBQXNFO1FBQ3RFLE1BQU0sVUFBVSxHQUFHLElBQUksV0FBSyxDQUFDLGNBQWMsRUFBRSxZQUFZLENBQUMsQ0FBQztRQUMzRCxNQUFNLGFBQWEsR0FBRyxJQUFJLGlCQUFXLENBQUMsVUFBVSxFQUFFLGVBQWUsRUFBRSxFQUFFLElBQUksRUFBRSxpQkFBaUIsRUFBRSxDQUFDLENBQUM7UUFFaEcsT0FBTztRQUNQLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBSyxDQUFDLEVBQUUsQ0FBQyxXQUFXLENBQUMsRUFBRSxXQUFXLENBQUMsQ0FBQztRQUM5QyxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQUssQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDakQsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFLLENBQUMsRUFBRSxDQUFDLFVBQVUsQ0FBQyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQzVDLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBSyxDQUFDLEVBQUUsQ0FBQyxhQUFhLENBQUMsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUMvQyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsbUZBQW1GLENBQUMsSUFBVTtRQUM1RixRQUFRO1FBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLEVBQUUsQ0FBQztRQUN0QixNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFFeEMsT0FBTztRQUNQLE1BQU0sR0FBRyxHQUFHLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQztRQUVwQyxPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxFQUFFO1lBQ2pDLEVBQUUsWUFBWSxFQUFFLENBQUMsQ0FBQyxFQUFFLEVBQUUsWUFBWSxFQUFFLEVBQUUsRUFBRSxDQUFDLEVBQUU7WUFDM0MsRUFBRSxZQUFZLEVBQUUsQ0FBQyxDQUFDLEVBQUUsRUFBRSxZQUFZLEVBQUUsRUFBRSxFQUFFLENBQUMsRUFBRTtTQUM1QyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsc0hBQXNILENBQUMsSUFBVTtRQUMvSCxRQUFRO1FBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLEVBQUUsQ0FBQztRQUV0QixPQUFPO1FBQ1AsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQzFDLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxVQUFVLEVBQUUsRUFBRSxTQUFTLEVBQUUsY0FBYyxFQUFFLENBQUMsQ0FBQztRQUV6RSxPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsWUFBWSxFQUFFLHdCQUF3QixDQUFDLENBQUM7UUFDOUQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsWUFBWSxFQUFFLDRCQUE0QixDQUFDLENBQUM7UUFDbEUsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELCtFQUErRSxDQUFDLElBQVU7UUFDeEYsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxDQUFDO1lBQ2xCLE9BQU8sRUFBRTtnQkFDUCxDQUFDLEtBQUssQ0FBQyxvQ0FBb0MsQ0FBQyxFQUFFLE1BQU07YUFDckQ7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQzFDLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxVQUFVLEVBQUUsRUFBRSxTQUFTLEVBQUUsY0FBYyxFQUFFLENBQUMsQ0FBQztRQUV6RSxPQUFPO1FBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsWUFBWSxFQUFFLHdCQUF3QixDQUFDLENBQUM7UUFDOUQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsWUFBWSxFQUFFLHdCQUF3QixDQUFDLENBQUM7UUFDOUQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHlDQUF5QyxFQUFFO1FBRXpDLG9CQUFvQixFQUFFO1lBRXBCLGdEQUFnRCxDQUFDLElBQVU7Z0JBQ3pELFFBQVE7Z0JBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLEVBQUUsQ0FBQztnQkFFdEIsT0FBTztnQkFDUCxNQUFNLE1BQU0sR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsVUFBVSxFQUFFLEVBQUUsU0FBUyxFQUFFLFVBQVUsRUFBRSxDQUFDLENBQUM7Z0JBQ3JFLE1BQU0sUUFBUSxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztnQkFFN0IsT0FBTztnQkFDUCxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxVQUFVLEVBQUUsVUFBVSxDQUFDLENBQUM7Z0JBQzlDLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLFlBQVksRUFBRSx3QkFBd0IsQ0FBQyxDQUFDO2dCQUM5RCxJQUFJLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLENBQUMsWUFBWSxFQUFFLHdCQUF3QixDQUFDLENBQUM7Z0JBQ3BHLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUNkLENBQUM7U0FDRjtRQUVELFNBQVMsRUFBRTtZQUNULDZFQUE2RSxDQUFDLElBQVU7Z0JBQ3RGLFFBQVE7Z0JBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLENBQUMsRUFBRSxPQUFPLEVBQUUsRUFBRSxDQUFDLEtBQUssQ0FBQyxvQ0FBb0MsQ0FBQyxFQUFFLE1BQU0sRUFBRSxFQUFFLENBQUMsQ0FBQztnQkFFM0YsT0FBTztnQkFDUCxNQUFNLE1BQU0sR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsVUFBVSxFQUFFLEVBQUUsU0FBUyxFQUFFLFVBQVUsRUFBRSxDQUFDLENBQUM7Z0JBQ3JFLE1BQU0sTUFBTSxHQUFHLElBQUksV0FBSyxDQUFDLEdBQUcsRUFBRSxVQUFVLEVBQUUsRUFBRSxTQUFTLEVBQUUsVUFBVSxFQUFFLENBQUMsQ0FBQztnQkFDckUsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO2dCQUU3QixPQUFPO2dCQUNQLElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUMsQ0FBQyxZQUFZLEVBQUUsd0JBQXdCLENBQUMsQ0FBQztnQkFDcEcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxDQUFDLFlBQVksRUFBRSx3QkFBd0IsQ0FBQyxDQUFDO2dCQUNwRyxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxZQUFZLEVBQUUsd0JBQXdCLENBQUMsQ0FBQztnQkFDOUQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsWUFBWSxFQUFFLHdCQUF3QixDQUFDLENBQUM7Z0JBQzlELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUNkLENBQUM7WUFFRCxzRUFBc0UsQ0FBQyxJQUFVO2dCQUMvRSxRQUFRO2dCQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxDQUFDLEVBQUUsT0FBTyxFQUFFLEVBQUUsQ0FBQyxLQUFLLENBQUMsb0NBQW9DLENBQUMsRUFBRSxNQUFNLEVBQUUsRUFBRSxDQUFDLENBQUM7Z0JBRTNGLE9BQU87Z0JBQ1AsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLFVBQVUsRUFBRSxFQUFFLFNBQVMsRUFBRSxVQUFVLEVBQUUsQ0FBQyxDQUFDO2dCQUNyRSxNQUFNLFFBQVEsR0FBRyxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7Z0JBRTdCLE9BQU87Z0JBQ1AsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLENBQUMsVUFBVSxFQUFFLFVBQVUsQ0FBQyxDQUFDO2dCQUM5QyxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxZQUFZLEVBQUUsd0JBQXdCLENBQUMsQ0FBQztnQkFDOUQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxDQUFDLFlBQVksRUFBRSx3QkFBd0IsQ0FBQyxDQUFDO2dCQUNwRyxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7WUFDZCxDQUFDO1NBQ0Y7S0FFRjtJQUVELDZDQUE2QyxDQUFDLElBQVU7UUFDdEQsUUFBUTtRQUNSLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxDQUFDO1lBQ2xCLE9BQU8sRUFBRTtnQkFDUCxDQUFDLEtBQUssQ0FBQyw0QkFBNEIsQ0FBQyxFQUFFLE1BQU07YUFDN0M7U0FDRixDQUFDLENBQUM7UUFDSCxNQUFNLE1BQU0sR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDeEMsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsTUFBTSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBRXpDLE9BQU87UUFDUCxLQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFckMsT0FBTztRQUNQLE1BQU0sR0FBRyxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUN4QixJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLGtCQUFrQixDQUFDLEtBQUssQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ25GLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLGNBQWMsQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDNUUsRUFBRSxJQUFJLEVBQUUsZUFBZSxFQUFFLElBQUksRUFBRSxLQUFLLEVBQUUsSUFBSSxFQUFFLEtBQUssRUFBRTtTQUNwRCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsK0RBQStELENBQUMsSUFBVTtRQUN4RSxRQUFRO1FBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxTQUFHLENBQUMsRUFBRSxXQUFXLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQztRQUM1QyxNQUFNLE1BQU0sR0FBRyxJQUFJLFdBQUssQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDeEMsTUFBTSxNQUFNLEdBQUcsSUFBSSxXQUFLLENBQUMsTUFBTSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBRTNDLE9BQU87UUFDUCxVQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFL0IsT0FBTztRQUNQLE1BQU0sR0FBRyxHQUFHLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUN4QixNQUFNLFFBQVEsR0FBRztZQUNmO2dCQUNFLElBQUksRUFBRSxvQkFBb0I7Z0JBQzFCLElBQUksRUFBRSxDQUFDLEVBQUUsR0FBRyxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLENBQUM7YUFDckM7U0FDRixDQUFDO1FBRUYsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxRQUFRLEVBQUUsRUFBRSxTQUFTLEVBQUUsUUFBUSxFQUFFLENBQUMsQ0FBQztRQUNuRyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLENBQUMsUUFBUSxDQUFDLFFBQVEsRUFBRSxFQUFFLGdCQUFnQixFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUM7UUFDMUcsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGdFQUFnRSxDQUFDLElBQVU7UUFDekUsa0VBQWtFO1FBQ2xFLE1BQU0sR0FBRyxHQUFHLElBQUksU0FBRyxFQUFFLENBQUM7UUFDdEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxXQUFLLENBQUMsR0FBRyxFQUFFLE9BQU8sRUFBRSxFQUFFLHFCQUFxQixFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7UUFFdkUsTUFBTSxRQUFRLEdBQUcsR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzdCLE1BQU0sUUFBUSxHQUFHLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUM7UUFFN0QsSUFBSSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMscUJBQXFCLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFFbEQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELG9EQUFvRCxDQUFDLElBQVU7UUFDN0QsSUFBSSxNQUFNLEdBQUcsS0FBSyxDQUFDO1FBRW5CLE1BQU0sT0FBUSxTQUFRLFdBQUs7WUFDekIsVUFBVSxDQUFDLE9BQTBCO2dCQUNuQyxNQUFNLEdBQUcsSUFBSSxDQUFDO2dCQUNkLElBQUksQ0FBQyxFQUFFLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxDQUFDO2dCQUN4QixJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFLE9BQU8sQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUN0RCxDQUFDO1NBQ0Y7UUFFRCxNQUFNLEdBQUcsR0FBRyxJQUFJLFNBQUcsRUFBRSxDQUFDO1FBQ3RCLElBQUksT0FBTyxDQUFDLEdBQUcsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUU3QixHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDWixJQUFJLENBQUMsRUFBRSxDQUFDLE1BQU0sRUFBRSxtQ0FBbUMsQ0FBQyxDQUFDO1FBQ3JELElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx5REFBeUQsQ0FBQyxJQUFVO1FBQ2xFLE9BQU87UUFDUCxNQUFNLEtBQUssR0FBRyxJQUFJLFdBQUssQ0FBQyxTQUFTLEVBQUUsU0FBUyxFQUFFO1lBQzVDLFdBQVcsRUFBRSxJQUFJLDRCQUFzQixFQUFFO1NBQzFDLENBQUMsQ0FBQztRQUNILEtBQUssQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLFdBQVcsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUU1QyxxQkFBcUI7UUFFckIsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELDBEQUEwRCxDQUFDLElBQVU7UUFDbkUsT0FBTztRQUNQLE1BQU0sS0FBSyxHQUFHLElBQUksV0FBSyxDQUFDLFNBQVMsRUFBRSxTQUFTLEVBQUU7WUFDNUMsV0FBVyxFQUFFLElBQUksNkJBQXVCLEVBQUU7U0FDM0MsQ0FBQyxDQUFDO1FBQ0gsS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsV0FBVyxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBRTVDLHFCQUFxQjtRQUVyQixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0NBQ0YsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGN4YXBpIGZyb20gJ0Bhd3MtY2RrL2N4LWFwaSc7XG5pbXBvcnQgeyBUZXN0IH0gZnJvbSAnbm9kZXVuaXQnO1xuaW1wb3J0IHtcbiAgQXBwLCBDZm5Db25kaXRpb24sIENmbkluY2x1ZGUsIENmbk91dHB1dCwgQ2ZuUGFyYW1ldGVyLFxuICBDZm5SZXNvdXJjZSwgQ29uc3RydWN0LCBMYXp5LCBTY29wZWRBd3MsIFN0YWNrLCB2YWxpZGF0ZVN0cmluZywgSVN5bnRoZXNpc1Nlc3Npb24sIFRhZ3MsIExlZ2FjeVN0YWNrU3ludGhlc2l6ZXIsIERlZmF1bHRTdGFja1N5bnRoZXNpemVyLFxufSBmcm9tICcuLi9saWInO1xuaW1wb3J0IHsgSW50cmluc2ljIH0gZnJvbSAnLi4vbGliL3ByaXZhdGUvaW50cmluc2ljJztcbmltcG9ydCB7IHJlc29sdmVSZWZlcmVuY2VzIH0gZnJvbSAnLi4vbGliL3ByaXZhdGUvcmVmcyc7XG5pbXBvcnQgeyBQb3N0UmVzb2x2ZVRva2VuIH0gZnJvbSAnLi4vbGliL3V0aWwnO1xuaW1wb3J0IHsgdG9DbG91ZEZvcm1hdGlvbiB9IGZyb20gJy4vdXRpbCc7XG5cbmV4cG9ydCA9IHtcbiAgJ2Egc3RhY2sgY2FuIGJlIHNlcmlhbGl6ZWQgaW50byBhIENsb3VkRm9ybWF0aW9uIHRlbXBsYXRlLCBpbml0aWFsbHkgaXRcXCdzIGVtcHR5Jyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICB0ZXN0LmRlZXBFcXVhbCh0b0Nsb3VkRm9ybWF0aW9uKHN0YWNrKSwgeyB9KTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnc3RhY2sgb2JqZWN0cyBoYXZlIHNvbWUgdGVtcGxhdGUtbGV2ZWwgcHJvcGV0aWVzLCBzdWNoIGFzIERlc2NyaXB0aW9uLCBWZXJzaW9uLCBUcmFuc2Zvcm0nKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIHN0YWNrLnRlbXBsYXRlT3B0aW9ucy50ZW1wbGF0ZUZvcm1hdFZlcnNpb24gPSAnTXlUZW1wbGF0ZVZlcnNpb24nO1xuICAgIHN0YWNrLnRlbXBsYXRlT3B0aW9ucy5kZXNjcmlwdGlvbiA9ICdUaGlzIGlzIG15IGRlc2NyaXB0aW9uJztcbiAgICBzdGFjay50ZW1wbGF0ZU9wdGlvbnMudHJhbnNmb3JtcyA9IFsnU0FNeSddO1xuICAgIHRlc3QuZGVlcEVxdWFsKHRvQ2xvdWRGb3JtYXRpb24oc3RhY2spLCB7XG4gICAgICBEZXNjcmlwdGlvbjogJ1RoaXMgaXMgbXkgZGVzY3JpcHRpb24nLFxuICAgICAgQVdTVGVtcGxhdGVGb3JtYXRWZXJzaW9uOiAnTXlUZW1wbGF0ZVZlcnNpb24nLFxuICAgICAgVHJhbnNmb3JtOiAnU0FNeScsXG4gICAgfSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ1N0YWNrLmlzU3RhY2sgaW5kaWNhdGVzIHRoYXQgYSBjb25zdHJ1Y3QgaXMgYSBzdGFjaycodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgYyA9IG5ldyBDb25zdHJ1Y3Qoc3RhY2ssICdDb25zdHJ1Y3QnKTtcbiAgICB0ZXN0Lm9rKFN0YWNrLmlzU3RhY2soc3RhY2spKTtcbiAgICB0ZXN0Lm9rKCFTdGFjay5pc1N0YWNrKGMpKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnc3RhY2suaWQgaXMgbm90IGluY2x1ZGVkIGluIHRoZSBsb2dpY2FsIGlkZW50aXRpZXMgb2YgcmVzb3VyY2VzIHdpdGhpbiBpdCcodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKHVuZGVmaW5lZCwgJ015U3RhY2snKTtcbiAgICBuZXcgQ2ZuUmVzb3VyY2Uoc3RhY2ssICdNeVJlc291cmNlJywgeyB0eXBlOiAnTXlSZXNvdXJjZVR5cGUnIH0pO1xuXG4gICAgdGVzdC5kZWVwRXF1YWwodG9DbG91ZEZvcm1hdGlvbihzdGFjayksIHsgUmVzb3VyY2VzOiB7IE15UmVzb3VyY2U6IHsgVHlwZTogJ015UmVzb3VyY2VUeXBlJyB9IH0gfSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3N0YWNrLnRlbXBsYXRlT3B0aW9ucyBjYW4gYmUgdXNlZCB0byBzZXQgdGVtcGxhdGUtbGV2ZWwgb3B0aW9ucycodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICBzdGFjay50ZW1wbGF0ZU9wdGlvbnMuZGVzY3JpcHRpb24gPSAnU3RhY2tEZXNjcmlwdGlvbic7XG4gICAgc3RhY2sudGVtcGxhdGVPcHRpb25zLnRlbXBsYXRlRm9ybWF0VmVyc2lvbiA9ICdUZW1wbGF0ZVZlcnNpb24nO1xuICAgIHN0YWNrLnRlbXBsYXRlT3B0aW9ucy50cmFuc2Zvcm0gPSAnRGVwcmVjYXRlZEZpZWxkJztcbiAgICBzdGFjay50ZW1wbGF0ZU9wdGlvbnMudHJhbnNmb3JtcyA9IFsnVHJhbnNmb3JtJ107XG4gICAgc3RhY2sudGVtcGxhdGVPcHRpb25zLm1ldGFkYXRhID0ge1xuICAgICAgTWV0YWRhdGFLZXk6ICdNZXRhZGF0YVZhbHVlJyxcbiAgICB9O1xuXG4gICAgdGVzdC5kZWVwRXF1YWwodG9DbG91ZEZvcm1hdGlvbihzdGFjayksIHtcbiAgICAgIERlc2NyaXB0aW9uOiAnU3RhY2tEZXNjcmlwdGlvbicsXG4gICAgICBUcmFuc2Zvcm06IFsnVHJhbnNmb3JtJywgJ0RlcHJlY2F0ZWRGaWVsZCddLFxuICAgICAgQVdTVGVtcGxhdGVGb3JtYXRWZXJzaW9uOiAnVGVtcGxhdGVWZXJzaW9uJyxcbiAgICAgIE1ldGFkYXRhOiB7IE1ldGFkYXRhS2V5OiAnTWV0YWRhdGFWYWx1ZScgfSxcbiAgICB9KTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdzdGFjay50ZW1wbGF0ZU9wdGlvbnMudHJhbnNmb3JtcyByZW1vdmVzIGR1cGxpY2F0ZSB2YWx1ZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgc3RhY2sudGVtcGxhdGVPcHRpb25zLnRyYW5zZm9ybXMgPSBbJ0EnLCAnQicsICdDJywgJ0EnXTtcblxuICAgIHRlc3QuZGVlcEVxdWFsKHRvQ2xvdWRGb3JtYXRpb24oc3RhY2spLCB7XG4gICAgICBUcmFuc2Zvcm06IFsnQScsICdCJywgJ0MnXSxcbiAgICB9KTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdzdGFjay5hZGRUcmFuc2Zvcm0oKSBhZGRzIGEgdHJhbnNmb3JtJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgIHN0YWNrLmFkZFRyYW5zZm9ybSgnQScpO1xuICAgIHN0YWNrLmFkZFRyYW5zZm9ybSgnQicpO1xuICAgIHN0YWNrLmFkZFRyYW5zZm9ybSgnQycpO1xuXG4gICAgdGVzdC5kZWVwRXF1YWwodG9DbG91ZEZvcm1hdGlvbihzdGFjayksIHtcbiAgICAgIFRyYW5zZm9ybTogWydBJywgJ0InLCAnQyddLFxuICAgIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgLy8gVGhpcyBhcHByb2FjaCB3aWxsIG9ubHkgYXBwbHkgdG8gVHlwZVNjcmlwdCBjb2RlLCBidXQgYXQgbGVhc3QgaXQncyBhIHRlbXBvcmFyeVxuICAvLyB3b3JrYXJvdW5kIGZvciBwZW9wbGUgcnVubmluZyBpbnRvIGlzc3VlcyBjYXVzZWQgYnkgU0RLLTMwMDMuXG4gIC8vIFdlIHNob3VsZCBjb21lIHVwIHdpdGggYSBwcm9wZXIgc29sdXRpb24gdGhhdCBpbnZvbHZlZCBqc2lpIGNhbGxiYWNrcyAod2hlbiB0aGV5IGV4aXN0KVxuICAvLyBzbyB0aGlzIGNhbiBiZSBpbXBsZW1lbnRlZCBieSBqc2lpIGxhbmd1YWdlcyBhcyB3ZWxsLlxuICAnT3ZlcnJpZGluZyBgU3RhY2suX3RvQ2xvdWRGb3JtYXRpb25gIGFsbG93cyBhcmJpdHJhcnkgcG9zdC1wcm9jZXNzaW5nIG9mIHRoZSBnZW5lcmF0ZWQgdGVtcGxhdGUgZHVyaW5nIHN5bnRoZXNpcycodGVzdDogVGVzdCkge1xuXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2tXaXRoUG9zdFByb2Nlc3NvcigpO1xuXG4gICAgbmV3IENmblJlc291cmNlKHN0YWNrLCAnbXlSZXNvdXJjZScsIHtcbiAgICAgIHR5cGU6ICdBV1M6Ok15UmVzb3VyY2UnLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICBNeVByb3AxOiAnaGVsbG8nLFxuICAgICAgICBNeVByb3AyOiAnaG93ZHknLFxuICAgICAgICBFbnZpcm9ubWVudDoge1xuICAgICAgICAgIEtleTogJ3ZhbHVlJyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjay5fdG9DbG91ZEZvcm1hdGlvbigpLCB7XG4gICAgICBSZXNvdXJjZXM6XG4gICAgICB7XG4gICAgICAgIG15UmVzb3VyY2U6XG4gICAgICAgICB7XG4gICAgICAgICAgIFR5cGU6ICdBV1M6Ok15UmVzb3VyY2UnLFxuICAgICAgICAgICBQcm9wZXJ0aWVzOlxuICAgICAgICAgIHtcbiAgICAgICAgICAgIE15UHJvcDE6ICdoZWxsbycsXG4gICAgICAgICAgICBNeVByb3AyOiAnaG93ZHknLFxuICAgICAgICAgICAgRW52aXJvbm1lbnQ6IHsga2V5OiAndmFsdWUnIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ1N0YWNrLmdldEJ5UGF0aCBjYW4gYmUgdXNlZCB0byBmaW5kIGFueSBDbG91ZEZvcm1hdGlvbiBlbGVtZW50IChQYXJhbWV0ZXIsIE91dHB1dCwgZXRjKScodGVzdDogVGVzdCkge1xuXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgIGNvbnN0IHAgPSBuZXcgQ2ZuUGFyYW1ldGVyKHN0YWNrLCAnTXlQYXJhbScsIHsgdHlwZTogJ1N0cmluZycgfSk7XG4gICAgY29uc3QgbyA9IG5ldyBDZm5PdXRwdXQoc3RhY2ssICdNeU91dHB1dCcsIHsgdmFsdWU6ICdib29tJyB9KTtcbiAgICBjb25zdCBjID0gbmV3IENmbkNvbmRpdGlvbihzdGFjaywgJ015Q29uZGl0aW9uJyk7XG5cbiAgICB0ZXN0LmVxdWFsKHN0YWNrLm5vZGUuZmluZENoaWxkKHAubm9kZS5pZCksIHApO1xuICAgIHRlc3QuZXF1YWwoc3RhY2subm9kZS5maW5kQ2hpbGQoby5ub2RlLmlkKSwgbyk7XG4gICAgdGVzdC5lcXVhbChzdGFjay5ub2RlLmZpbmRDaGlsZChjLm5vZGUuaWQpLCBjKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdTdGFjayBuYW1lcyBjYW4gaGF2ZSBoeXBoZW5zIGluIHRoZW0nKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCByb290ID0gbmV3IEFwcCgpO1xuXG4gICAgbmV3IFN0YWNrKHJvb3QsICdIZWxsby1Xb3JsZCcpO1xuICAgIC8vIERpZCBub3QgdGhyb3dcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdTdGFja3MgY2FuIGhhdmUgYSBkZXNjcmlwdGlvbiBnaXZlbiB0byB0aGVtJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2sobmV3IEFwcCgpLCAnTXlTdGFjaycsIHsgZGVzY3JpcHRpb246ICdNeSBzdGFjaywgaGFuZHMgb2ZmIScgfSk7XG4gICAgY29uc3Qgb3V0cHV0ID0gdG9DbG91ZEZvcm1hdGlvbihzdGFjayk7XG4gICAgdGVzdC5lcXVhbChvdXRwdXQuRGVzY3JpcHRpb24sICdNeSBzdGFjaywgaGFuZHMgb2ZmIScpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdTdGFjayBkZXNjcmlwdGlvbnMgaGF2ZSBhIGxpbWl0ZWQgbGVuZ3RoJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3QgZGVzYyA9IGBMb3JlbSBpcHN1bSBkb2xvciBzaXQgYW1ldCwgY29uc2VjdGV0dXIgYWRpcGlzY2luZyBlbGl0LCBzZWQgZG8gZWl1c21vZCB0ZW1wb3JcbiAgICAgaW5jaWRpZHVudCB1dCBsYWJvcmUgZXQgZG9sb3JlIG1hZ25hIGFsaXF1YS4gQ29uc2VxdWF0IGludGVyZHVtIHZhcml1cyBzaXQgYW1ldCBtYXR0aXMgdnVscHV0YXRlXG4gICAgIGVuaW0gbnVsbGEgYWxpcXVldC4gQXQgaW1wZXJkaWV0IGR1aSBhY2N1bXNhbiBzaXQgYW1ldCBudWxsYSBmYWNpbGlzaSBtb3JiaS4gRWdldCBsb3JlbSBkb2xvciBzZWRcbiAgICAgdml2ZXJyYSBpcHN1bS4gRGlhbSB2b2x1dHBhdCBjb21tb2RvIHNlZCBlZ2VzdGFzIGVnZXN0YXMuIFNpdCBhbWV0IHBvcnR0aXRvciBlZ2V0IGRvbG9yIG1vcmJpIG5vbi5cbiAgICAgTG9yZW0gZG9sb3Igc2VkIHZpdmVycmEgaXBzdW0uIElkIHBvcnRhIG5pYmggdmVuZW5hdGlzIGNyYXMgc2VkIGZlbGlzLiBBdWd1ZSBpbnRlcmR1bSB2ZWxpdCBldWlzbW9kXG4gICAgIGluIHBlbGxlbnRlc3F1ZS4gU3VzY2lwaXQgYWRpcGlzY2luZyBiaWJlbmR1bSBlc3QgdWx0cmljaWVzIGludGVnZXIgcXVpcy4gQ29uZGltZW50dW0gaWQgdmVuZW5hdGlzIGFcbiAgICAgY29uZGltZW50dW0gdml0YWUgc2FwaWVuIHBlbGxlbnRlc3F1ZSBoYWJpdGFudCBtb3JiaS4gQ29uZ3VlIG1hdXJpcyByaG9uY3VzIGFlbmVhbiB2ZWwgZWxpdCBzY2VsZXJpc3F1ZVxuICAgICBtYXVyaXMgcGVsbGVudGVzcXVlIHB1bHZpbmFyLlxuICAgICBGYXVjaWJ1cyBwdXJ1cyBpbiBtYXNzYSB0ZW1wb3IgbmVjLiBSaXN1cyB2aXZlcnJhIGFkaXBpc2NpbmcgYXQgaW4uIEludGVnZXIgZmV1Z2lhdCBzY2VsZXJpc3F1ZSB2YXJpdXNcbiAgICAgbW9yYmkuIE1hbGVzdWFkYSBudW5jIHZlbCByaXN1cyBjb21tb2RvIHZpdmVycmEgbWFlY2VuYXMgYWNjdW1zYW4gbGFjdXMuIFZ1bHB1dGF0ZSBzYXBpZW4gbmVjIHNhZ2l0dGlzXG4gICAgIGFsaXF1YW0gbWFsZXN1YWRhIGJpYmVuZHVtIGFyY3Ugdml0YWUuIEF1Z3VlIG5lcXVlIGdyYXZpZGEgaW4gZmVybWVudHVtIGV0IHNvbGxpY2l0dWRpbiBhYyBvcmNpIHBoYXNlbGx1cy5cbiAgICAgVWx0cmljZXMgdGluY2lkdW50IGFyY3Ugbm9uIHNvZGFsZXMgbmVxdWUgc29kYWxlcy5gO1xuICAgIHRlc3QudGhyb3dzKCgpID0+IG5ldyBTdGFjayhuZXcgQXBwKCksICdNeVN0YWNrJywgeyBkZXNjcmlwdGlvbjogZGVzYyB9KSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ0luY2x1ZGUgc2hvdWxkIHN1cHBvcnQgbm9uLWhhc2ggdG9wLWxldmVsIHRlbXBsYXRlIGVsZW1lbnRzIGxpa2UgXCJEZXNjcmlwdGlvblwiJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgIGNvbnN0IHRlbXBsYXRlID0ge1xuICAgICAgRGVzY3JpcHRpb246ICdoZWxsbywgd29ybGQnLFxuICAgIH07XG5cbiAgICBuZXcgQ2ZuSW5jbHVkZShzdGFjaywgJ0luY2x1ZGUnLCB7IHRlbXBsYXRlIH0pO1xuXG4gICAgY29uc3Qgb3V0cHV0ID0gdG9DbG91ZEZvcm1hdGlvbihzdGFjayk7XG5cbiAgICB0ZXN0LmVxdWFsKHR5cGVvZiBvdXRwdXQuRGVzY3JpcHRpb24sICdzdHJpbmcnKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnUHNldWRvIHZhbHVlcyBhdHRhY2hlZCB0byBvbmUgc3RhY2sgY2FuIGJlIHJlZmVyZW5jZWQgaW4gYW5vdGhlciBzdGFjaycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGNvbnN0IHN0YWNrMSA9IG5ldyBTdGFjayhhcHAsICdTdGFjazEnKTtcbiAgICBjb25zdCBhY2NvdW50MSA9IG5ldyBTY29wZWRBd3Moc3RhY2sxKS5hY2NvdW50SWQ7XG4gICAgY29uc3Qgc3RhY2syID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrMicpO1xuXG4gICAgLy8gV0hFTiAtIHVzZWQgaW4gYW5vdGhlciBzdGFja1xuICAgIG5ldyBDZm5QYXJhbWV0ZXIoc3RhY2syLCAnU29tZVBhcmFtZXRlcicsIHsgdHlwZTogJ1N0cmluZycsIGRlZmF1bHQ6IGFjY291bnQxIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IGFzc2VtYmx5ID0gYXBwLnN5bnRoKCk7XG4gICAgY29uc3QgdGVtcGxhdGUxID0gYXNzZW1ibHkuZ2V0U3RhY2tCeU5hbWUoc3RhY2sxLnN0YWNrTmFtZSkudGVtcGxhdGU7XG4gICAgY29uc3QgdGVtcGxhdGUyID0gYXNzZW1ibHkuZ2V0U3RhY2tCeU5hbWUoc3RhY2syLnN0YWNrTmFtZSkudGVtcGxhdGU7XG5cbiAgICB0ZXN0LmRlZXBFcXVhbCh0ZW1wbGF0ZTEsIHtcbiAgICAgIE91dHB1dHM6IHtcbiAgICAgICAgRXhwb3J0c091dHB1dFJlZkFXU0FjY291bnRJZEFENTY4MDU3OiB7XG4gICAgICAgICAgVmFsdWU6IHsgUmVmOiAnQVdTOjpBY2NvdW50SWQnIH0sXG4gICAgICAgICAgRXhwb3J0OiB7IE5hbWU6ICdTdGFjazE6RXhwb3J0c091dHB1dFJlZkFXU0FjY291bnRJZEFENTY4MDU3JyB9LFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIHRlc3QuZGVlcEVxdWFsKHRlbXBsYXRlMiwge1xuICAgICAgUGFyYW1ldGVyczoge1xuICAgICAgICBTb21lUGFyYW1ldGVyOiB7XG4gICAgICAgICAgVHlwZTogJ1N0cmluZycsXG4gICAgICAgICAgRGVmYXVsdDogeyAnRm46OkltcG9ydFZhbHVlJzogJ1N0YWNrMTpFeHBvcnRzT3V0cHV0UmVmQVdTQWNjb3VudElkQUQ1NjgwNTcnIH0sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ0Nyb3NzLXN0YWNrIHJlZmVyZW5jZXMgYXJlIGRldGVjdGVkIGluIHJlc291cmNlIHByb3BlcnRpZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICBjb25zdCBzdGFjazEgPSBuZXcgU3RhY2soYXBwLCAnU3RhY2sxJyk7XG4gICAgY29uc3QgcmVzb3VyY2UxID0gbmV3IENmblJlc291cmNlKHN0YWNrMSwgJ1Jlc291cmNlJywgeyB0eXBlOiAnQkxBJyB9KTtcbiAgICBjb25zdCBzdGFjazIgPSBuZXcgU3RhY2soYXBwLCAnU3RhY2syJyk7XG5cbiAgICAvLyBXSEVOIC0gdXNlZCBpbiBhbm90aGVyIHJlc291cmNlXG4gICAgbmV3IENmblJlc291cmNlKHN0YWNrMiwgJ1NvbWVSZXNvdXJjZScsIHtcbiAgICAgIHR5cGU6ICdBV1M6OlNvbWU6OlJlc291cmNlJyxcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgc29tZVByb3BlcnR5OiBuZXcgSW50cmluc2ljKHJlc291cmNlMS5yZWYpLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBjb25zdCBhc3NlbWJseSA9IGFwcC5zeW50aCgpO1xuICAgIGNvbnN0IHRlbXBsYXRlMiA9IGFzc2VtYmx5LmdldFN0YWNrQnlOYW1lKHN0YWNrMi5zdGFja05hbWUpLnRlbXBsYXRlO1xuXG4gICAgdGVzdC5kZWVwRXF1YWwodGVtcGxhdGUyLCB7XG4gICAgICBSZXNvdXJjZXM6IHtcbiAgICAgICAgU29tZVJlc291cmNlOiB7XG4gICAgICAgICAgVHlwZTogJ0FXUzo6U29tZTo6UmVzb3VyY2UnLFxuICAgICAgICAgIFByb3BlcnRpZXM6IHtcbiAgICAgICAgICAgIHNvbWVQcm9wZXJ0eTogeyAnRm46OkltcG9ydFZhbHVlJzogJ1N0YWNrMTpFeHBvcnRzT3V0cHV0UmVmUmVzb3VyY2UxRDVEOTA1QScgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnQ3Jvc3Mtc3RhY2sgcmVmZXJlbmNlIGV4cG9ydCBuYW1lcyBhcmUgcmVsYXRpdmUgdG8gdGhlIHN0YWNrICh3aGVuIHRoZSBmbGFnIGlzIHNldCknKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoe1xuICAgICAgY29udGV4dDoge1xuICAgICAgICAnQGF3cy1jZGsvY29yZTpzdGFja1JlbGF0aXZlRXhwb3J0cyc6ICd0cnVlJyxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgY29uc3QgaW5kaWZmZXJlbnRTY29wZSA9IG5ldyBDb25zdHJ1Y3QoYXBwLCAnRXh0cmFTY29wZScpO1xuXG4gICAgY29uc3Qgc3RhY2sxID0gbmV3IFN0YWNrKGluZGlmZmVyZW50U2NvcGUsICdTdGFjazEnLCB7XG4gICAgICBzdGFja05hbWU6ICdTdGFjazEnLFxuICAgIH0pO1xuICAgIGNvbnN0IHJlc291cmNlMSA9IG5ldyBDZm5SZXNvdXJjZShzdGFjazEsICdSZXNvdXJjZScsIHsgdHlwZTogJ0JMQScgfSk7XG4gICAgY29uc3Qgc3RhY2syID0gbmV3IFN0YWNrKGluZGlmZmVyZW50U2NvcGUsICdTdGFjazInKTtcblxuICAgIC8vIFdIRU4gLSB1c2VkIGluIGFub3RoZXIgcmVzb3VyY2VcbiAgICBuZXcgQ2ZuUmVzb3VyY2Uoc3RhY2syLCAnU29tZVJlc291cmNlJywge1xuICAgICAgdHlwZTogJ0FXUzo6U29tZTo6UmVzb3VyY2UnLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICBzb21lUHJvcGVydHk6IG5ldyBJbnRyaW5zaWMocmVzb3VyY2UxLnJlZiksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IGFzc2VtYmx5ID0gYXBwLnN5bnRoKCk7XG4gICAgY29uc3QgdGVtcGxhdGUyID0gYXNzZW1ibHkuZ2V0U3RhY2tCeU5hbWUoc3RhY2syLnN0YWNrTmFtZSkudGVtcGxhdGU7XG5cbiAgICB0ZXN0LmRlZXBFcXVhbCh0ZW1wbGF0ZTIsIHtcbiAgICAgIFJlc291cmNlczoge1xuICAgICAgICBTb21lUmVzb3VyY2U6IHtcbiAgICAgICAgICBUeXBlOiAnQVdTOjpTb21lOjpSZXNvdXJjZScsXG4gICAgICAgICAgUHJvcGVydGllczoge1xuICAgICAgICAgICAgc29tZVByb3BlcnR5OiB7ICdGbjo6SW1wb3J0VmFsdWUnOiAnU3RhY2sxOkV4cG9ydHNPdXRwdXRSZWZSZXNvdXJjZTFENUQ5MDVBJyB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjcm9zcy1zdGFjayByZWZlcmVuY2VzIGluIGxhenkgdG9rZW5zIHdvcmsnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICBjb25zdCBzdGFjazEgPSBuZXcgU3RhY2soYXBwLCAnU3RhY2sxJyk7XG4gICAgY29uc3QgYWNjb3VudDEgPSBuZXcgU2NvcGVkQXdzKHN0YWNrMSkuYWNjb3VudElkO1xuICAgIGNvbnN0IHN0YWNrMiA9IG5ldyBTdGFjayhhcHAsICdTdGFjazInKTtcblxuICAgIC8vIFdIRU4gLSB1c2VkIGluIGFub3RoZXIgc3RhY2tcbiAgICBuZXcgQ2ZuUGFyYW1ldGVyKHN0YWNrMiwgJ1NvbWVQYXJhbWV0ZXInLCB7IHR5cGU6ICdTdHJpbmcnLCBkZWZhdWx0OiBMYXp5LnN0cmluZ1ZhbHVlKHsgcHJvZHVjZTogKCkgPT4gYWNjb3VudDEgfSkgfSk7XG5cbiAgICBjb25zdCBhc3NlbWJseSA9IGFwcC5zeW50aCgpO1xuICAgIGNvbnN0IHRlbXBsYXRlMSA9IGFzc2VtYmx5LmdldFN0YWNrQnlOYW1lKHN0YWNrMS5zdGFja05hbWUpLnRlbXBsYXRlO1xuICAgIGNvbnN0IHRlbXBsYXRlMiA9IGFzc2VtYmx5LmdldFN0YWNrQnlOYW1lKHN0YWNrMi5zdGFja05hbWUpLnRlbXBsYXRlO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKHRlbXBsYXRlMSwge1xuICAgICAgT3V0cHV0czoge1xuICAgICAgICBFeHBvcnRzT3V0cHV0UmVmQVdTQWNjb3VudElkQUQ1NjgwNTc6IHtcbiAgICAgICAgICBWYWx1ZTogeyBSZWY6ICdBV1M6OkFjY291bnRJZCcgfSxcbiAgICAgICAgICBFeHBvcnQ6IHsgTmFtZTogJ1N0YWNrMTpFeHBvcnRzT3V0cHV0UmVmQVdTQWNjb3VudElkQUQ1NjgwNTcnIH0sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGVzdC5kZWVwRXF1YWwodGVtcGxhdGUyLCB7XG4gICAgICBQYXJhbWV0ZXJzOiB7XG4gICAgICAgIFNvbWVQYXJhbWV0ZXI6IHtcbiAgICAgICAgICBUeXBlOiAnU3RyaW5nJyxcbiAgICAgICAgICBEZWZhdWx0OiB7ICdGbjo6SW1wb3J0VmFsdWUnOiAnU3RhY2sxOkV4cG9ydHNPdXRwdXRSZWZBV1NBY2NvdW50SWRBRDU2ODA1NycgfSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnQ3Jvc3Mtc3RhY2sgdXNlIG9mIFJlZ2lvbiBhbmQgYWNjb3VudCByZXR1cm5zIG5vbnNjb3BlZCBpbnRyaW5zaWMgYmVjYXVzZSB0aGUgdHdvIHN0YWNrcyBtdXN0IGJlIGluIHRoZSBzYW1lIHJlZ2lvbiBhbnl3YXknKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICBjb25zdCBzdGFjazEgPSBuZXcgU3RhY2soYXBwLCAnU3RhY2sxJyk7XG4gICAgY29uc3Qgc3RhY2syID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrMicpO1xuXG4gICAgLy8gV0hFTiAtIHVzZWQgaW4gYW5vdGhlciBzdGFja1xuICAgIG5ldyBDZm5PdXRwdXQoc3RhY2syLCAnRGVtT3V0cHV0JywgeyB2YWx1ZTogc3RhY2sxLnJlZ2lvbiB9KTtcbiAgICBuZXcgQ2ZuT3V0cHV0KHN0YWNrMiwgJ0RlbUFjY291bnQnLCB7IHZhbHVlOiBzdGFjazEuYWNjb3VudCB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBjb25zdCBhc3NlbWJseSA9IGFwcC5zeW50aCgpO1xuICAgIGNvbnN0IHRlbXBsYXRlMiA9IGFzc2VtYmx5LmdldFN0YWNrQnlOYW1lKHN0YWNrMi5zdGFja05hbWUpLnRlbXBsYXRlO1xuXG4gICAgdGVzdC5kZWVwRXF1YWwodGVtcGxhdGUyLCB7XG4gICAgICBPdXRwdXRzOiB7XG4gICAgICAgIERlbU91dHB1dDoge1xuICAgICAgICAgIFZhbHVlOiB7IFJlZjogJ0FXUzo6UmVnaW9uJyB9LFxuICAgICAgICB9LFxuICAgICAgICBEZW1BY2NvdW50OiB7XG4gICAgICAgICAgVmFsdWU6IHsgUmVmOiAnQVdTOjpBY2NvdW50SWQnIH0sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2Nyb3NzLXN0YWNrIHJlZmVyZW5jZXMgaW4gc3RyaW5ncyB3b3JrJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3Qgc3RhY2sxID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrMScpO1xuICAgIGNvbnN0IGFjY291bnQxID0gbmV3IFNjb3BlZEF3cyhzdGFjazEpLmFjY291bnRJZDtcbiAgICBjb25zdCBzdGFjazIgPSBuZXcgU3RhY2soYXBwLCAnU3RhY2syJyk7XG5cbiAgICAvLyBXSEVOIC0gdXNlZCBpbiBhbm90aGVyIHN0YWNrXG4gICAgbmV3IENmblBhcmFtZXRlcihzdGFjazIsICdTb21lUGFyYW1ldGVyJywgeyB0eXBlOiAnU3RyaW5nJywgZGVmYXVsdDogYFRoZUFjY291bnRJcyR7YWNjb3VudDF9YCB9KTtcblxuICAgIGNvbnN0IGFzc2VtYmx5ID0gYXBwLnN5bnRoKCk7XG4gICAgY29uc3QgdGVtcGxhdGUyID0gYXNzZW1ibHkuZ2V0U3RhY2tCeU5hbWUoc3RhY2syLnN0YWNrTmFtZSkudGVtcGxhdGU7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwodGVtcGxhdGUyLCB7XG4gICAgICBQYXJhbWV0ZXJzOiB7XG4gICAgICAgIFNvbWVQYXJhbWV0ZXI6IHtcbiAgICAgICAgICBUeXBlOiAnU3RyaW5nJyxcbiAgICAgICAgICBEZWZhdWx0OiB7ICdGbjo6Sm9pbic6IFsnJywgWydUaGVBY2NvdW50SXMnLCB7ICdGbjo6SW1wb3J0VmFsdWUnOiAnU3RhY2sxOkV4cG9ydHNPdXRwdXRSZWZBV1NBY2NvdW50SWRBRDU2ODA1NycgfV1dIH0sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2Nyb3NzIHN0YWNrIHJlZmVyZW5jZXMgYW5kIGRlcGVuZGVuY2llcyB3b3JrIHdpdGhpbiBjaGlsZCBzdGFja3MgKG5vbi1uZXN0ZWQpJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3QgcGFyZW50ID0gbmV3IFN0YWNrKGFwcCwgJ1BhcmVudCcpO1xuICAgIGNvbnN0IGNoaWxkMSA9IG5ldyBTdGFjayhwYXJlbnQsICdDaGlsZDEnKTtcbiAgICBjb25zdCBjaGlsZDIgPSBuZXcgU3RhY2socGFyZW50LCAnQ2hpbGQyJyk7XG4gICAgY29uc3QgcmVzb3VyY2VBID0gbmV3IENmblJlc291cmNlKGNoaWxkMSwgJ1Jlc291cmNlQScsIHsgdHlwZTogJ1JBJyB9KTtcbiAgICBjb25zdCByZXNvdXJjZUIgPSBuZXcgQ2ZuUmVzb3VyY2UoY2hpbGQxLCAnUmVzb3VyY2VCJywgeyB0eXBlOiAnUkInIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHJlc291cmNlMiA9IG5ldyBDZm5SZXNvdXJjZShjaGlsZDIsICdSZXNvdXJjZTEnLCB7XG4gICAgICB0eXBlOiAnUjInLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICBSZWZUb1Jlc291cmNlMTogcmVzb3VyY2VBLnJlZixcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgcmVzb3VyY2UyLmFkZERlcGVuZHNPbihyZXNvdXJjZUIpO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IGFzc2VtYmx5ID0gYXBwLnN5bnRoKCk7XG4gICAgY29uc3QgcGFyZW50VGVtcGxhdGUgPSBhc3NlbWJseS5nZXRTdGFja0FydGlmYWN0KHBhcmVudC5hcnRpZmFjdElkKS50ZW1wbGF0ZTtcbiAgICBjb25zdCBjaGlsZDFUZW1wbGF0ZSA9IGFzc2VtYmx5LmdldFN0YWNrQXJ0aWZhY3QoY2hpbGQxLmFydGlmYWN0SWQpLnRlbXBsYXRlO1xuICAgIGNvbnN0IGNoaWxkMlRlbXBsYXRlID0gYXNzZW1ibHkuZ2V0U3RhY2tBcnRpZmFjdChjaGlsZDIuYXJ0aWZhY3RJZCkudGVtcGxhdGU7XG5cbiAgICB0ZXN0LmRlZXBFcXVhbChwYXJlbnRUZW1wbGF0ZSwge30pO1xuICAgIHRlc3QuZGVlcEVxdWFsKGNoaWxkMVRlbXBsYXRlLCB7XG4gICAgICBSZXNvdXJjZXM6IHtcbiAgICAgICAgUmVzb3VyY2VBOiB7IFR5cGU6ICdSQScgfSxcbiAgICAgICAgUmVzb3VyY2VCOiB7IFR5cGU6ICdSQicgfSxcbiAgICAgIH0sXG4gICAgICBPdXRwdXRzOiB7XG4gICAgICAgIEV4cG9ydHNPdXRwdXRSZWZSZXNvdXJjZUE0NjFCNEVGOToge1xuICAgICAgICAgIFZhbHVlOiB7IFJlZjogJ1Jlc291cmNlQScgfSxcbiAgICAgICAgICBFeHBvcnQ6IHsgTmFtZTogJ1BhcmVudENoaWxkMThGQUVGNDE5OkNoaWxkMUV4cG9ydHNPdXRwdXRSZWZSZXNvdXJjZUE3QkYyMEIzNycgfSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgdGVzdC5kZWVwRXF1YWwoY2hpbGQyVGVtcGxhdGUsIHtcbiAgICAgIFJlc291cmNlczoge1xuICAgICAgICBSZXNvdXJjZTE6IHtcbiAgICAgICAgICBUeXBlOiAnUjInLFxuICAgICAgICAgIFByb3BlcnRpZXM6IHtcbiAgICAgICAgICAgIFJlZlRvUmVzb3VyY2UxOiB7ICdGbjo6SW1wb3J0VmFsdWUnOiAnUGFyZW50Q2hpbGQxOEZBRUY0MTk6Q2hpbGQxRXhwb3J0c091dHB1dFJlZlJlc291cmNlQTdCRjIwQjM3JyB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGVzdC5kZWVwRXF1YWwoYXNzZW1ibHkuZ2V0U3RhY2tBcnRpZmFjdChjaGlsZDEuYXJ0aWZhY3RJZCkuZGVwZW5kZW5jaWVzLm1hcCh4ID0+IHguaWQpLCBbXSk7XG4gICAgdGVzdC5kZWVwRXF1YWwoYXNzZW1ibHkuZ2V0U3RhY2tBcnRpZmFjdChjaGlsZDIuYXJ0aWZhY3RJZCkuZGVwZW5kZW5jaWVzLm1hcCh4ID0+IHguaWQpLCBbJ1BhcmVudENoaWxkMThGQUVGNDE5J10pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdDZm5TeW50aGVzaXNFcnJvciBpcyBpZ25vcmVkIHdoZW4gcHJlcGFyaW5nIGNyb3NzIHJlZmVyZW5jZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjayhhcHAsICdteS1zdGFjaycpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNsYXNzIENmblRlc3QgZXh0ZW5kcyBDZm5SZXNvdXJjZSB7XG4gICAgICBwdWJsaWMgX3RvQ2xvdWRGb3JtYXRpb24oKSB7XG4gICAgICAgIHJldHVybiBuZXcgUG9zdFJlc29sdmVUb2tlbih7XG4gICAgICAgICAgeG9vOiAxMjM0LFxuICAgICAgICB9LCBwcm9wcyA9PiB7XG4gICAgICAgICAgdmFsaWRhdGVTdHJpbmcocHJvcHMpLmFzc2VydFN1Y2Nlc3MoKTtcbiAgICAgICAgfSk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgbmV3IENmblRlc3Qoc3RhY2ssICdNeVRoaW5nJywgeyB0eXBlOiAnQVdTOjpUeXBlJyB9KTtcblxuICAgIC8vIFRIRU5cbiAgICByZXNvbHZlUmVmZXJlbmNlcyhhcHApO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ1N0YWNrcyBjYW4gYmUgY2hpbGRyZW4gb2Ygb3RoZXIgc3RhY2tzIChzdWJzdGFjaykgYW5kIHRoZXkgd2lsbCBiZSBzeW50aGVzaXplZCBzZXBhcmF0ZWx5Jyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgcGFyZW50U3RhY2sgPSBuZXcgU3RhY2soYXBwLCAncGFyZW50Jyk7XG4gICAgY29uc3QgY2hpbGRTdGFjayA9IG5ldyBTdGFjayhwYXJlbnRTdGFjaywgJ2NoaWxkJyk7XG4gICAgbmV3IENmblJlc291cmNlKHBhcmVudFN0YWNrLCAnTXlQYXJlbnRSZXNvdXJjZScsIHsgdHlwZTogJ1Jlc291cmNlOjpQYXJlbnQnIH0pO1xuICAgIG5ldyBDZm5SZXNvdXJjZShjaGlsZFN0YWNrLCAnTXlDaGlsZFJlc291cmNlJywgeyB0eXBlOiAnUmVzb3VyY2U6OkNoaWxkJyB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBjb25zdCBhc3NlbWJseSA9IGFwcC5zeW50aCgpO1xuICAgIHRlc3QuZGVlcEVxdWFsKGFzc2VtYmx5LmdldFN0YWNrQnlOYW1lKHBhcmVudFN0YWNrLnN0YWNrTmFtZSkudGVtcGxhdGUsIHsgUmVzb3VyY2VzOiB7IE15UGFyZW50UmVzb3VyY2U6IHsgVHlwZTogJ1Jlc291cmNlOjpQYXJlbnQnIH0gfSB9KTtcbiAgICB0ZXN0LmRlZXBFcXVhbChhc3NlbWJseS5nZXRTdGFja0J5TmFtZShjaGlsZFN0YWNrLnN0YWNrTmFtZSkudGVtcGxhdGUsIHsgUmVzb3VyY2VzOiB7IE15Q2hpbGRSZXNvdXJjZTogeyBUeXBlOiAnUmVzb3VyY2U6OkNoaWxkJyB9IH0gfSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2Nyb3NzLXN0YWNrIHJlZmVyZW5jZSAoc3Vic3RhY2sgcmVmZXJlbmNlcyBwYXJlbnQgc3RhY2spJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3QgcGFyZW50U3RhY2sgPSBuZXcgU3RhY2soYXBwLCAncGFyZW50Jyk7XG4gICAgY29uc3QgY2hpbGRTdGFjayA9IG5ldyBTdGFjayhwYXJlbnRTdGFjaywgJ2NoaWxkJyk7XG5cbiAgICAvLyBXSEVOIChhIHJlc291cmNlIGZyb20gdGhlIGNoaWxkIHN0YWNrIHJlZmVyZW5jZXMgYSByZXNvdXJjZSBmcm9tIHRoZSBwYXJlbnQgc3RhY2spXG4gICAgY29uc3QgcGFyZW50UmVzb3VyY2UgPSBuZXcgQ2ZuUmVzb3VyY2UocGFyZW50U3RhY2ssICdNeVBhcmVudFJlc291cmNlJywgeyB0eXBlOiAnUmVzb3VyY2U6OlBhcmVudCcgfSk7XG4gICAgbmV3IENmblJlc291cmNlKGNoaWxkU3RhY2ssICdNeUNoaWxkUmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnUmVzb3VyY2U6OkNoaWxkJyxcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgQ2hpbGRQcm9wOiBwYXJlbnRSZXNvdXJjZS5nZXRBdHQoJ0F0dE9mUGFyZW50UmVzb3VyY2UnKSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgY29uc3QgYXNzZW1ibHkgPSBhcHAuc3ludGgoKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChhc3NlbWJseS5nZXRTdGFja0J5TmFtZShwYXJlbnRTdGFjay5zdGFja05hbWUpLnRlbXBsYXRlLCB7XG4gICAgICBSZXNvdXJjZXM6IHsgTXlQYXJlbnRSZXNvdXJjZTogeyBUeXBlOiAnUmVzb3VyY2U6OlBhcmVudCcgfSB9LFxuICAgICAgT3V0cHV0czoge1xuICAgICAgICBFeHBvcnRzT3V0cHV0Rm5HZXRBdHRNeVBhcmVudFJlc291cmNlQXR0T2ZQYXJlbnRSZXNvdXJjZUMyRDBCQjlFOiB7XG4gICAgICAgICAgVmFsdWU6IHsgJ0ZuOjpHZXRBdHQnOiBbJ015UGFyZW50UmVzb3VyY2UnLCAnQXR0T2ZQYXJlbnRSZXNvdXJjZSddIH0sXG4gICAgICAgICAgRXhwb3J0OiB7IE5hbWU6ICdwYXJlbnQ6RXhwb3J0c091dHB1dEZuR2V0QXR0TXlQYXJlbnRSZXNvdXJjZUF0dE9mUGFyZW50UmVzb3VyY2VDMkQwQkI5RScgfSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgdGVzdC5kZWVwRXF1YWwoYXNzZW1ibHkuZ2V0U3RhY2tCeU5hbWUoY2hpbGRTdGFjay5zdGFja05hbWUpLnRlbXBsYXRlLCB7XG4gICAgICBSZXNvdXJjZXM6IHtcbiAgICAgICAgTXlDaGlsZFJlc291cmNlOiB7XG4gICAgICAgICAgVHlwZTogJ1Jlc291cmNlOjpDaGlsZCcsXG4gICAgICAgICAgUHJvcGVydGllczoge1xuICAgICAgICAgICAgQ2hpbGRQcm9wOiB7XG4gICAgICAgICAgICAgICdGbjo6SW1wb3J0VmFsdWUnOiAncGFyZW50OkV4cG9ydHNPdXRwdXRGbkdldEF0dE15UGFyZW50UmVzb3VyY2VBdHRPZlBhcmVudFJlc291cmNlQzJEMEJCOUUnLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICB9KTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnY3Jvc3Mtc3RhY2sgcmVmZXJlbmNlIChwYXJlbnQgc3RhY2sgcmVmZXJlbmNlcyBzdWJzdGFjayknKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICBjb25zdCBwYXJlbnRTdGFjayA9IG5ldyBTdGFjayhhcHAsICdwYXJlbnQnKTtcbiAgICBjb25zdCBjaGlsZFN0YWNrID0gbmV3IFN0YWNrKHBhcmVudFN0YWNrLCAnY2hpbGQnKTtcblxuICAgIC8vIFdIRU4gKGEgcmVzb3VyY2UgZnJvbSB0aGUgY2hpbGQgc3RhY2sgcmVmZXJlbmNlcyBhIHJlc291cmNlIGZyb20gdGhlIHBhcmVudCBzdGFjaylcbiAgICBjb25zdCBjaGlsZFJlc291cmNlID0gbmV3IENmblJlc291cmNlKGNoaWxkU3RhY2ssICdNeUNoaWxkUmVzb3VyY2UnLCB7IHR5cGU6ICdSZXNvdXJjZTo6Q2hpbGQnIH0pO1xuICAgIG5ldyBDZm5SZXNvdXJjZShwYXJlbnRTdGFjaywgJ015UGFyZW50UmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnUmVzb3VyY2U6OlBhcmVudCcsXG4gICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgIFBhcmVudFByb3A6IGNoaWxkUmVzb3VyY2UuZ2V0QXR0KCdBdHRyaWJ1dGVPZkNoaWxkUmVzb3VyY2UnKSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgY29uc3QgYXNzZW1ibHkgPSBhcHAuc3ludGgoKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChhc3NlbWJseS5nZXRTdGFja0J5TmFtZShwYXJlbnRTdGFjay5zdGFja05hbWUpLnRlbXBsYXRlLCB7XG4gICAgICBSZXNvdXJjZXM6IHtcbiAgICAgICAgTXlQYXJlbnRSZXNvdXJjZToge1xuICAgICAgICAgIFR5cGU6ICdSZXNvdXJjZTo6UGFyZW50JyxcbiAgICAgICAgICBQcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgICBQYXJlbnRQcm9wOiB7ICdGbjo6SW1wb3J0VmFsdWUnOiAncGFyZW50Y2hpbGQxM0Y5MzU5QjpjaGlsZEV4cG9ydHNPdXRwdXRGbkdldEF0dE15Q2hpbGRSZXNvdXJjZUF0dHJpYnV0ZU9mQ2hpbGRSZXNvdXJjZTQyMDA1MkZDJyB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGVzdC5kZWVwRXF1YWwoYXNzZW1ibHkuZ2V0U3RhY2tCeU5hbWUoY2hpbGRTdGFjay5zdGFja05hbWUpLnRlbXBsYXRlLCB7XG4gICAgICBSZXNvdXJjZXM6IHsgTXlDaGlsZFJlc291cmNlOiB7IFR5cGU6ICdSZXNvdXJjZTo6Q2hpbGQnIH0gfSxcbiAgICAgIE91dHB1dHM6IHtcbiAgICAgICAgRXhwb3J0c091dHB1dEZuR2V0QXR0TXlDaGlsZFJlc291cmNlQXR0cmlidXRlT2ZDaGlsZFJlc291cmNlNTI4MTMyNjQ6IHtcbiAgICAgICAgICBWYWx1ZTogeyAnRm46OkdldEF0dCc6IFsnTXlDaGlsZFJlc291cmNlJywgJ0F0dHJpYnV0ZU9mQ2hpbGRSZXNvdXJjZSddIH0sXG4gICAgICAgICAgRXhwb3J0OiB7IE5hbWU6ICdwYXJlbnRjaGlsZDEzRjkzNTlCOmNoaWxkRXhwb3J0c091dHB1dEZuR2V0QXR0TXlDaGlsZFJlc291cmNlQXR0cmlidXRlT2ZDaGlsZFJlc291cmNlNDIwMDUyRkMnIH0sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjYW5ub3QgY3JlYXRlIGN5Y2xpYyByZWZlcmVuY2UgYmV0d2VlbiBzdGFja3MnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICBjb25zdCBzdGFjazEgPSBuZXcgU3RhY2soYXBwLCAnU3RhY2sxJyk7XG4gICAgY29uc3QgYWNjb3VudDEgPSBuZXcgU2NvcGVkQXdzKHN0YWNrMSkuYWNjb3VudElkO1xuICAgIGNvbnN0IHN0YWNrMiA9IG5ldyBTdGFjayhhcHAsICdTdGFjazInKTtcbiAgICBjb25zdCBhY2NvdW50MiA9IG5ldyBTY29wZWRBd3Moc3RhY2syKS5hY2NvdW50SWQ7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IENmblBhcmFtZXRlcihzdGFjazIsICdTb21lUGFyYW1ldGVyJywgeyB0eXBlOiAnU3RyaW5nJywgZGVmYXVsdDogYWNjb3VudDEgfSk7XG4gICAgbmV3IENmblBhcmFtZXRlcihzdGFjazEsICdTb21lUGFyYW1ldGVyJywgeyB0eXBlOiAnU3RyaW5nJywgZGVmYXVsdDogYWNjb3VudDIgfSk7XG5cbiAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICBhcHAuc3ludGgoKTtcbiAgICAgIC8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBtYXgtbGVuXG4gICAgfSwgXCInU3RhY2syJyBkZXBlbmRzIG9uICdTdGFjazEnIChTdGFjazIvU29tZVBhcmFtZXRlciAtPiBTdGFjazEuQVdTOjpBY2NvdW50SWQpLiBBZGRpbmcgdGhpcyBkZXBlbmRlbmN5IChTdGFjazEvU29tZVBhcmFtZXRlciAtPiBTdGFjazIuQVdTOjpBY2NvdW50SWQpIHdvdWxkIGNyZWF0ZSBhIGN5Y2xpYyByZWZlcmVuY2UuXCIpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3N0YWNrcyBrbm93IGFib3V0IHRoZWlyIGRlcGVuZGVuY2llcycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGNvbnN0IHN0YWNrMSA9IG5ldyBTdGFjayhhcHAsICdTdGFjazEnKTtcbiAgICBjb25zdCBhY2NvdW50MSA9IG5ldyBTY29wZWRBd3Moc3RhY2sxKS5hY2NvdW50SWQ7XG4gICAgY29uc3Qgc3RhY2syID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrMicpO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBDZm5QYXJhbWV0ZXIoc3RhY2syLCAnU29tZVBhcmFtZXRlcicsIHsgdHlwZTogJ1N0cmluZycsIGRlZmF1bHQ6IGFjY291bnQxIH0pO1xuXG4gICAgYXBwLnN5bnRoKCk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2syLmRlcGVuZGVuY2llcy5tYXAocyA9PiBzLm5vZGUuaWQpLCBbJ1N0YWNrMSddKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjYW5ub3QgY3JlYXRlIHJlZmVyZW5jZXMgdG8gc3RhY2tzIGluIG90aGVyIHJlZ2lvbnMvYWNjb3VudHMnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICBjb25zdCBzdGFjazEgPSBuZXcgU3RhY2soYXBwLCAnU3RhY2sxJywgeyBlbnY6IHsgYWNjb3VudDogJzEyMzQ1Njc4OTAxMicsIHJlZ2lvbjogJ2VzLW5vcnN0LTEnIH0gfSk7XG4gICAgY29uc3QgYWNjb3VudDEgPSBuZXcgU2NvcGVkQXdzKHN0YWNrMSkuYWNjb3VudElkO1xuICAgIGNvbnN0IHN0YWNrMiA9IG5ldyBTdGFjayhhcHAsICdTdGFjazInLCB7IGVudjogeyBhY2NvdW50OiAnMTIzNDU2Nzg5MDEyJywgcmVnaW9uOiAnZXMtbm9yc3QtMicgfSB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgQ2ZuUGFyYW1ldGVyKHN0YWNrMiwgJ1NvbWVQYXJhbWV0ZXInLCB7IHR5cGU6ICdTdHJpbmcnLCBkZWZhdWx0OiBhY2NvdW50MSB9KTtcblxuICAgIHRlc3QudGhyb3dzKCgpID0+IHtcbiAgICAgIGFwcC5zeW50aCgpO1xuICAgIH0sIC9TdGFjayBcIlN0YWNrMlwiIGNhbm5vdCBjb25zdW1lIGEgY3Jvc3MgcmVmZXJlbmNlIGZyb20gc3RhY2sgXCJTdGFjazFcIi8pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3VybFN1ZmZpeCBkb2VzIG5vdCBpbXBseSBhIHN0YWNrIGRlcGVuZGVuY3knKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgICBjb25zdCBmaXJzdCA9IG5ldyBTdGFjayhhcHAsICdGaXJzdCcpO1xuICAgIGNvbnN0IHNlY29uZCA9IG5ldyBTdGFjayhhcHAsICdTZWNvbmQnKTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgQ2ZuT3V0cHV0KHNlY29uZCwgJ091dHB1dCcsIHtcbiAgICAgIHZhbHVlOiBmaXJzdC51cmxTdWZmaXgsXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgYXBwLnN5bnRoKCk7XG5cbiAgICB0ZXN0LmVxdWFsKHNlY29uZC5kZXBlbmRlbmNpZXMubGVuZ3RoLCAwKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdzdGFjayB3aXRoIHJlZ2lvbiBzdXBwbGllZCB2aWEgcHJvcHMgcmV0dXJucyBsaXRlcmFsIHZhbHVlJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnU3RhY2sxJywgeyBlbnY6IHsgYWNjb3VudDogJzEyMzQ1Njc4OTAxMicsIHJlZ2lvbjogJ2VzLW5vcnN0LTEnIH0gfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5lcXVhbChzdGFjay5yZXNvbHZlKHN0YWNrLnJlZ2lvbiksICdlcy1ub3JzdC0xJyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnb3ZlcnJpZGVMb2dpY2FsSWQoaWQpIGNhbiBiZSB1c2VkIHRvIG92ZXJyaWRlIHRoZSBsb2dpY2FsIElEIG9mIGEgcmVzb3VyY2UnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgYm9uam91ciA9IG5ldyBDZm5SZXNvdXJjZShzdGFjaywgJ0JvbmpvdXJSZXNvdXJjZScsIHsgdHlwZTogJ1Jlc291cmNlOjpUeXBlJyB9KTtcblxuICAgIC8vIHsgUmVmIH0gYW5kIHsgR2V0QXR0IH1cbiAgICBuZXcgQ2ZuUmVzb3VyY2Uoc3RhY2ssICdSZWZUb0JvbmpvdXInLCB7XG4gICAgICB0eXBlOiAnT3RoZXI6OlJlc291cmNlJyxcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgUmVmVG9Cb25qb3VyOiBib25qb3VyLnJlZixcbiAgICAgICAgR2V0QXR0Qm9uam91cjogYm9uam91ci5nZXRBdHQoJ1RoZUF0dCcpLnRvU3RyaW5nKCksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgYm9uam91ci5vdmVycmlkZUxvZ2ljYWxJZCgnQk9PTScpO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKHRvQ2xvdWRGb3JtYXRpb24oc3RhY2spLCB7XG4gICAgICBSZXNvdXJjZXM6XG4gICAgICB7XG4gICAgICAgIEJPT006IHsgVHlwZTogJ1Jlc291cmNlOjpUeXBlJyB9LFxuICAgICAgICBSZWZUb0JvbmpvdXI6XG4gICAgICAgICB7XG4gICAgICAgICAgIFR5cGU6ICdPdGhlcjo6UmVzb3VyY2UnLFxuICAgICAgICAgICBQcm9wZXJ0aWVzOlxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBSZWZUb0JvbmpvdXI6IHsgUmVmOiAnQk9PTScgfSxcbiAgICAgICAgICAgICAgR2V0QXR0Qm9uam91cjogeyAnRm46OkdldEF0dCc6IFsnQk9PTScsICdUaGVBdHQnXSB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdTdGFjayBuYW1lIGNhbiBiZSBvdmVycmlkZGVuIHZpYSBwcm9wZXJ0aWVzJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKHVuZGVmaW5lZCwgJ1N0YWNrJywgeyBzdGFja05hbWU6ICdvdGhlck5hbWUnIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnN0YWNrTmFtZSwgJ290aGVyTmFtZScpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ1N0YWNrIG5hbWUgaXMgaW5oZXJpdGVkIGZyb20gQXBwIG5hbWUgaWYgYXZhaWxhYmxlJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHJvb3QgPSBuZXcgQXBwKCk7XG4gICAgY29uc3QgYXBwID0gbmV3IENvbnN0cnVjdChyb290LCAnUHJvZCcpO1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrJyk7XG5cbiAgICAvLyBUSEVOXG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2suc3RhY2tOYW1lLCAnUHJvZFN0YWNrRDUyNzlCMjInKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdzdGFjayBjb25zdHJ1Y3QgaWQgZG9lcyBub3QgZ28gdGhyb3VnaCBzdGFjayBuYW1lIHZhbGlkYXRpb24gaWYgdGhlcmUgaXMgYW4gZXhwbGljaXQgc3RhY2sgbmFtZScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ2ludmFsaWQgYXMgOiBzdGFjayBuYW1lLCBidXQgdGhhdHMgZmluZScsIHtcbiAgICAgIHN0YWNrTmFtZTogJ3ZhbGlkLXN0YWNrLW5hbWUnLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IHNlc3Npb24gPSBhcHAuc3ludGgoKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChzdGFjay5zdGFja05hbWUsICd2YWxpZC1zdGFjay1uYW1lJyk7XG4gICAgdGVzdC5vayhzZXNzaW9uLnRyeUdldEFydGlmYWN0KHN0YWNrLmFydGlmYWN0SWQpKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnc3RhY2sgdmFsaWRhdGlvbiBpcyBwZXJmb3JtZWQgb24gZXhwbGljaXQgc3RhY2sgbmFtZScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QudGhyb3dzKCgpID0+IG5ldyBTdGFjayhhcHAsICdib29tJywgeyBzdGFja05hbWU6ICdpbnZhbGlkOnN0YWNrOm5hbWUnIH0pLFxuICAgICAgL1N0YWNrIG5hbWUgbXVzdCBtYXRjaCB0aGUgcmVndWxhciBleHByZXNzaW9uLyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnU3RhY2sub2Yoc3RhY2spIHJldHVybnMgdGhlIGNvcnJlY3Qgc3RhY2snKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIHRlc3Quc2FtZShTdGFjay5vZihzdGFjayksIHN0YWNrKTtcbiAgICBjb25zdCBwYXJlbnQgPSBuZXcgQ29uc3RydWN0KHN0YWNrLCAnUGFyZW50Jyk7XG4gICAgY29uc3QgY29uc3RydWN0ID0gbmV3IENvbnN0cnVjdChwYXJlbnQsICdDb25zdHJ1Y3QnKTtcbiAgICB0ZXN0LnNhbWUoU3RhY2sub2YoY29uc3RydWN0KSwgc3RhY2spO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdTdGFjay5vZigpIHRocm93cyB3aGVuIHRoZXJlIGlzIG5vIHBhcmVudCBTdGFjaycodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHJvb3QgPSBuZXcgQ29uc3RydWN0KHVuZGVmaW5lZCBhcyBhbnksICdSb290Jyk7XG4gICAgY29uc3QgY29uc3RydWN0ID0gbmV3IENvbnN0cnVjdChyb290LCAnQ29uc3RydWN0Jyk7XG4gICAgdGVzdC50aHJvd3MoKCkgPT4gU3RhY2sub2YoY29uc3RydWN0KSwgL3Nob3VsZCBiZSBjcmVhdGVkIGluIHRoZSBzY29wZSBvZiBhIFN0YWNrLCBidXQgbm8gU3RhY2sgZm91bmQvKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnU3RhY2sub2YoKSB3b3JrcyBmb3Igc3Vic3RhY2tzJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgcGFyZW50U3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnUGFyZW50U3RhY2snKTtcbiAgICBjb25zdCBwYXJlbnRSZXNvdXJjZSA9IG5ldyBDZm5SZXNvdXJjZShwYXJlbnRTdGFjaywgJ1BhcmVudFJlc291cmNlJywgeyB0eXBlOiAncGFyZW50OjpyZXNvdXJjZScgfSk7XG5cbiAgICAvLyB3ZSB3aWxsIGRlZmluZSBhIHN1YnN0YWNrIHVuZGVyIHRoZSAvcmVzb3VyY2UvLi4uIGp1c3QgZm9yIGdpZ2dsZXMuXG4gICAgY29uc3QgY2hpbGRTdGFjayA9IG5ldyBTdGFjayhwYXJlbnRSZXNvdXJjZSwgJ0NoaWxkU3RhY2snKTtcbiAgICBjb25zdCBjaGlsZFJlc291cmNlID0gbmV3IENmblJlc291cmNlKGNoaWxkU3RhY2ssICdDaGlsZFJlc291cmNlJywgeyB0eXBlOiAnY2hpbGQ6OnJlc291cmNlJyB9KTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LnNhbWUoU3RhY2sub2YocGFyZW50U3RhY2spLCBwYXJlbnRTdGFjayk7XG4gICAgdGVzdC5zYW1lKFN0YWNrLm9mKHBhcmVudFJlc291cmNlKSwgcGFyZW50U3RhY2spO1xuICAgIHRlc3Quc2FtZShTdGFjay5vZihjaGlsZFN0YWNrKSwgY2hpbGRTdGFjayk7XG4gICAgdGVzdC5zYW1lKFN0YWNrLm9mKGNoaWxkUmVzb3VyY2UpLCBjaGlsZFN0YWNrKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnc3RhY2suYXZhaWxhYmlsaXR5Wm9uZXMgZmFsbHMgYmFjayB0byBGbjo6R2V0QVpbMF0sWzJdIGlmIHJlZ2lvbiBpcyBub3Qgc3BlY2lmaWVkJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnTXlTdGFjaycpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IGF6cyA9IHN0YWNrLmF2YWlsYWJpbGl0eVpvbmVzO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrLnJlc29sdmUoYXpzKSwgW1xuICAgICAgeyAnRm46OlNlbGVjdCc6IFswLCB7ICdGbjo6R2V0QVpzJzogJycgfV0gfSxcbiAgICAgIHsgJ0ZuOjpTZWxlY3QnOiBbMSwgeyAnRm46OkdldEFacyc6ICcnIH1dIH0sXG4gICAgXSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3N0YWNrLnRlbXBsYXRlRmlsZSBpcyB0aGUgbmFtZSBvZiB0aGUgdGVtcGxhdGUgZmlsZSBlbWl0dGVkIHRvIHRoZSBjbG91ZCBhc3NlbWJseSAoZGVmYXVsdCBpcyB0byB1c2UgdGhlIHN0YWNrIG5hbWUpJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3Qgc3RhY2sxID0gbmV3IFN0YWNrKGFwcCwgJ015U3RhY2sxJyk7XG4gICAgY29uc3Qgc3RhY2syID0gbmV3IFN0YWNrKGFwcCwgJ015U3RhY2syJywgeyBzdGFja05hbWU6ICdNeVJlYWxTdGFjazInIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrMS50ZW1wbGF0ZUZpbGUsICdNeVN0YWNrMS50ZW1wbGF0ZS5qc29uJyk7XG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2syLnRlbXBsYXRlRmlsZSwgJ015UmVhbFN0YWNrMi50ZW1wbGF0ZS5qc29uJyk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ3doZW4gZmVhdHVyZSBmbGFnIGlzIGVuYWJsZWQgd2Ugd2lsbCB1c2UgdGhlIGFydGlmYWN0IGlkIGFzIHRoZSB0ZW1wbGF0ZSBuYW1lJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKHtcbiAgICAgIGNvbnRleHQ6IHtcbiAgICAgICAgW2N4YXBpLkVOQUJMRV9TVEFDS19OQU1FX0RVUExJQ0FURVNfQ09OVEVYVF06ICd0cnVlJyxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3Qgc3RhY2sxID0gbmV3IFN0YWNrKGFwcCwgJ015U3RhY2sxJyk7XG4gICAgY29uc3Qgc3RhY2syID0gbmV3IFN0YWNrKGFwcCwgJ015U3RhY2syJywgeyBzdGFja05hbWU6ICdNeVJlYWxTdGFjazInIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrMS50ZW1wbGF0ZUZpbGUsICdNeVN0YWNrMS50ZW1wbGF0ZS5qc29uJyk7XG4gICAgdGVzdC5kZWVwRXF1YWwoc3RhY2syLnRlbXBsYXRlRmlsZSwgJ015U3RhY2syLnRlbXBsYXRlLmpzb24nKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnQGF3cy1jZGsvY29yZTplbmFibGVTdGFja05hbWVEdXBsaWNhdGVzJzoge1xuXG4gICAgJ2Rpc2FibGVkIChkZWZhdWx0KSc6IHtcblxuICAgICAgJ2FydGlmYWN0SWQgYW5kIHRlbXBsYXRlRmlsZSB1c2UgdGhlIHN0YWNrIG5hbWUnKHRlc3Q6IFRlc3QpIHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuXG4gICAgICAgIC8vIFdIRU5cbiAgICAgICAgY29uc3Qgc3RhY2sxID0gbmV3IFN0YWNrKGFwcCwgJ015U3RhY2sxJywgeyBzdGFja05hbWU6ICd0aGVzdGFjaycgfSk7XG4gICAgICAgIGNvbnN0IGFzc2VtYmx5ID0gYXBwLnN5bnRoKCk7XG5cbiAgICAgICAgLy8gVEhFTlxuICAgICAgICB0ZXN0LmRlZXBFcXVhbChzdGFjazEuYXJ0aWZhY3RJZCwgJ3RoZXN0YWNrJyk7XG4gICAgICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrMS50ZW1wbGF0ZUZpbGUsICd0aGVzdGFjay50ZW1wbGF0ZS5qc29uJyk7XG4gICAgICAgIHRlc3QuZGVlcEVxdWFsKGFzc2VtYmx5LmdldFN0YWNrQXJ0aWZhY3Qoc3RhY2sxLmFydGlmYWN0SWQpLnRlbXBsYXRlRmlsZSwgJ3RoZXN0YWNrLnRlbXBsYXRlLmpzb24nKTtcbiAgICAgICAgdGVzdC5kb25lKCk7XG4gICAgICB9LFxuICAgIH0sXG5cbiAgICAnZW5hYmxlZCc6IHtcbiAgICAgICdhbGxvd3MgdXNpbmcgdGhlIHNhbWUgc3RhY2sgbmFtZSBmb3IgdHdvIHN0YWNrcyAoaS5lLiBpbiBkaWZmZXJlbnQgcmVnaW9ucyknKHRlc3Q6IFRlc3QpIHtcbiAgICAgICAgLy8gR0lWRU5cbiAgICAgICAgY29uc3QgYXBwID0gbmV3IEFwcCh7IGNvbnRleHQ6IHsgW2N4YXBpLkVOQUJMRV9TVEFDS19OQU1FX0RVUExJQ0FURVNfQ09OVEVYVF06ICd0cnVlJyB9IH0pO1xuXG4gICAgICAgIC8vIFdIRU5cbiAgICAgICAgY29uc3Qgc3RhY2sxID0gbmV3IFN0YWNrKGFwcCwgJ015U3RhY2sxJywgeyBzdGFja05hbWU6ICd0aGVzdGFjaycgfSk7XG4gICAgICAgIGNvbnN0IHN0YWNrMiA9IG5ldyBTdGFjayhhcHAsICdNeVN0YWNrMicsIHsgc3RhY2tOYW1lOiAndGhlc3RhY2snIH0pO1xuICAgICAgICBjb25zdCBhc3NlbWJseSA9IGFwcC5zeW50aCgpO1xuXG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgdGVzdC5kZWVwRXF1YWwoYXNzZW1ibHkuZ2V0U3RhY2tBcnRpZmFjdChzdGFjazEuYXJ0aWZhY3RJZCkudGVtcGxhdGVGaWxlLCAnTXlTdGFjazEudGVtcGxhdGUuanNvbicpO1xuICAgICAgICB0ZXN0LmRlZXBFcXVhbChhc3NlbWJseS5nZXRTdGFja0FydGlmYWN0KHN0YWNrMi5hcnRpZmFjdElkKS50ZW1wbGF0ZUZpbGUsICdNeVN0YWNrMi50ZW1wbGF0ZS5qc29uJyk7XG4gICAgICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrMS50ZW1wbGF0ZUZpbGUsICdNeVN0YWNrMS50ZW1wbGF0ZS5qc29uJyk7XG4gICAgICAgIHRlc3QuZGVlcEVxdWFsKHN0YWNrMi50ZW1wbGF0ZUZpbGUsICdNeVN0YWNrMi50ZW1wbGF0ZS5qc29uJyk7XG4gICAgICAgIHRlc3QuZG9uZSgpO1xuICAgICAgfSxcblxuICAgICAgJ2FydGlmYWN0SWQgYW5kIHRlbXBsYXRlRmlsZSB1c2UgdGhlIHVuaXF1ZSBpZCBhbmQgbm90IHRoZSBzdGFjayBuYW1lJyh0ZXN0OiBUZXN0KSB7XG4gICAgICAgIC8vIEdJVkVOXG4gICAgICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoeyBjb250ZXh0OiB7IFtjeGFwaS5FTkFCTEVfU1RBQ0tfTkFNRV9EVVBMSUNBVEVTX0NPTlRFWFRdOiAndHJ1ZScgfSB9KTtcblxuICAgICAgICAvLyBXSEVOXG4gICAgICAgIGNvbnN0IHN0YWNrMSA9IG5ldyBTdGFjayhhcHAsICdNeVN0YWNrMScsIHsgc3RhY2tOYW1lOiAndGhlc3RhY2snIH0pO1xuICAgICAgICBjb25zdCBhc3NlbWJseSA9IGFwcC5zeW50aCgpO1xuXG4gICAgICAgIC8vIFRIRU5cbiAgICAgICAgdGVzdC5kZWVwRXF1YWwoc3RhY2sxLmFydGlmYWN0SWQsICdNeVN0YWNrMScpO1xuICAgICAgICB0ZXN0LmRlZXBFcXVhbChzdGFjazEudGVtcGxhdGVGaWxlLCAnTXlTdGFjazEudGVtcGxhdGUuanNvbicpO1xuICAgICAgICB0ZXN0LmRlZXBFcXVhbChhc3NlbWJseS5nZXRTdGFja0FydGlmYWN0KHN0YWNrMS5hcnRpZmFjdElkKS50ZW1wbGF0ZUZpbGUsICdNeVN0YWNrMS50ZW1wbGF0ZS5qc29uJyk7XG4gICAgICAgIHRlc3QuZG9uZSgpO1xuICAgICAgfSxcbiAgICB9LFxuXG4gIH0sXG5cbiAgJ21ldGFkYXRhIGlzIGNvbGxlY3RlZCBhdCB0aGUgc3RhY2sgYm91bmRhcnknKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGFwcCA9IG5ldyBBcHAoe1xuICAgICAgY29udGV4dDoge1xuICAgICAgICBbY3hhcGkuRElTQUJMRV9NRVRBREFUQV9TVEFDS19UUkFDRV06ICd0cnVlJyxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgY29uc3QgcGFyZW50ID0gbmV3IFN0YWNrKGFwcCwgJ3BhcmVudCcpO1xuICAgIGNvbnN0IGNoaWxkID0gbmV3IFN0YWNrKHBhcmVudCwgJ2NoaWxkJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgY2hpbGQubm9kZS5hZGRNZXRhZGF0YSgnZm9vJywgJ2JhcicpO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IGFzbSA9IGFwcC5zeW50aCgpO1xuICAgIHRlc3QuZGVlcEVxdWFsKGFzbS5nZXRTdGFja0J5TmFtZShwYXJlbnQuc3RhY2tOYW1lKS5maW5kTWV0YWRhdGFCeVR5cGUoJ2ZvbycpLCBbXSk7XG4gICAgdGVzdC5kZWVwRXF1YWwoYXNtLmdldFN0YWNrQnlOYW1lKGNoaWxkLnN0YWNrTmFtZSkuZmluZE1ldGFkYXRhQnlUeXBlKCdmb28nKSwgW1xuICAgICAgeyBwYXRoOiAnL3BhcmVudC9jaGlsZCcsIHR5cGU6ICdmb28nLCBkYXRhOiAnYmFyJyB9LFxuICAgIF0pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdzdGFjayB0YWdzIGFyZSByZWZsZWN0ZWQgaW4gdGhlIHN0YWNrIGNsb3VkIGFzc2VtYmx5IGFydGlmYWN0Jyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBhcHAgPSBuZXcgQXBwKHsgc3RhY2tUcmFjZXM6IGZhbHNlIH0pO1xuICAgIGNvbnN0IHN0YWNrMSA9IG5ldyBTdGFjayhhcHAsICdzdGFjazEnKTtcbiAgICBjb25zdCBzdGFjazIgPSBuZXcgU3RhY2soc3RhY2sxLCAnc3RhY2syJyk7XG5cbiAgICAvLyBXSEVOXG4gICAgVGFncy5vZihhcHApLmFkZCgnZm9vJywgJ2JhcicpO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IGFzbSA9IGFwcC5zeW50aCgpO1xuICAgIGNvbnN0IGV4cGVjdGVkID0gW1xuICAgICAge1xuICAgICAgICB0eXBlOiAnYXdzOmNkazpzdGFjay10YWdzJyxcbiAgICAgICAgZGF0YTogW3sga2V5OiAnZm9vJywgdmFsdWU6ICdiYXInIH1dLFxuICAgICAgfSxcbiAgICBdO1xuXG4gICAgdGVzdC5kZWVwRXF1YWwoYXNtLmdldFN0YWNrQXJ0aWZhY3Qoc3RhY2sxLmFydGlmYWN0SWQpLm1hbmlmZXN0Lm1ldGFkYXRhLCB7ICcvc3RhY2sxJzogZXhwZWN0ZWQgfSk7XG4gICAgdGVzdC5kZWVwRXF1YWwoYXNtLmdldFN0YWNrQXJ0aWZhY3Qoc3RhY2syLmFydGlmYWN0SWQpLm1hbmlmZXN0Lm1ldGFkYXRhLCB7ICcvc3RhY2sxL3N0YWNrMic6IGV4cGVjdGVkIH0pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdUZXJtaW5hdGlvbiBQcm90ZWN0aW9uIGlzIHJlZmxlY3RlZCBpbiBDbG91ZCBBc3NlbWJseSBhcnRpZmFjdCcodGVzdDogVGVzdCkge1xuICAgIC8vIGlmIHRoZSByb290IGlzIGFuIGFwcCwgaW52b2tlIFwic3ludGhcIiB0byBhdm9pZCBkb3VibGUgc3ludGhlc2lzXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKGFwcCwgJ1N0YWNrJywgeyB0ZXJtaW5hdGlvblByb3RlY3Rpb246IHRydWUgfSk7XG5cbiAgICBjb25zdCBhc3NlbWJseSA9IGFwcC5zeW50aCgpO1xuICAgIGNvbnN0IGFydGlmYWN0ID0gYXNzZW1ibHkuZ2V0U3RhY2tBcnRpZmFjdChzdGFjay5hcnRpZmFjdElkKTtcblxuICAgIHRlc3QuZXF1YWxzKGFydGlmYWN0LnRlcm1pbmF0aW9uUHJvdGVjdGlvbiwgdHJ1ZSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAndXNlcnMgY2FuIChzdGlsbCkgb3ZlcnJpZGUgXCJzeW50aGVzaXplKClcIiBpbiBzdGFjaycodGVzdDogVGVzdCkge1xuICAgIGxldCBjYWxsZWQgPSBmYWxzZTtcblxuICAgIGNsYXNzIE15U3RhY2sgZXh0ZW5kcyBTdGFjayB7XG4gICAgICBzeW50aGVzaXplKHNlc3Npb246IElTeW50aGVzaXNTZXNzaW9uKSB7XG4gICAgICAgIGNhbGxlZCA9IHRydWU7XG4gICAgICAgIHRlc3Qub2soc2Vzc2lvbi5vdXRkaXIpO1xuICAgICAgICB0ZXN0LmVxdWFsKHNlc3Npb24uYXNzZW1ibHkub3V0ZGlyLCBzZXNzaW9uLm91dGRpcik7XG4gICAgICB9XG4gICAgfVxuXG4gICAgY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICAgIG5ldyBNeVN0YWNrKGFwcCwgJ215LXN0YWNrJyk7XG5cbiAgICBhcHAuc3ludGgoKTtcbiAgICB0ZXN0Lm9rKGNhbGxlZCwgJ3N5bnRoZXNpemUoKSBub3QgY2FsbGVkIGZvciBTdGFjaycpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdjb250ZXh0IGNhbiBiZSBzZXQgb24gYSBzdGFjayB1c2luZyBhIExlZ2FjeVN5bnRoZXNpemVyJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKHVuZGVmaW5lZCwgdW5kZWZpbmVkLCB7XG4gICAgICBzeW50aGVzaXplcjogbmV3IExlZ2FjeVN0YWNrU3ludGhlc2l6ZXIoKSxcbiAgICB9KTtcbiAgICBzdGFjay5ub2RlLnNldENvbnRleHQoJ3NvbWV0aGluZycsICd2YWx1ZScpO1xuXG4gICAgLy8gVEhFTjogbm8gZXhjZXB0aW9uXG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnY29udGV4dCBjYW4gYmUgc2V0IG9uIGEgc3RhY2sgdXNpbmcgYSBEZWZhdWx0U3ludGhlc2l6ZXInKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBXSEVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2sodW5kZWZpbmVkLCB1bmRlZmluZWQsIHtcbiAgICAgIHN5bnRoZXNpemVyOiBuZXcgRGVmYXVsdFN0YWNrU3ludGhlc2l6ZXIoKSxcbiAgICB9KTtcbiAgICBzdGFjay5ub2RlLnNldENvbnRleHQoJ3NvbWV0aGluZycsICd2YWx1ZScpO1xuXG4gICAgLy8gVEhFTjogbm8gZXhjZXB0aW9uXG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbn07XG5cbmNsYXNzIFN0YWNrV2l0aFBvc3RQcm9jZXNzb3IgZXh0ZW5kcyBTdGFjayB7XG5cbiAgLy8gLi4uXG5cbiAgcHVibGljIF90b0Nsb3VkRm9ybWF0aW9uKCkge1xuICAgIGNvbnN0IHRlbXBsYXRlID0gc3VwZXIuX3RvQ2xvdWRGb3JtYXRpb24oKTtcblxuICAgIC8vIG1hbmlwdWxhdGUgdGVtcGxhdGUgKGUuZy4gcmVuYW1lIFwiS2V5XCIgdG8gXCJrZXlcIilcbiAgICB0ZW1wbGF0ZS5SZXNvdXJjZXMubXlSZXNvdXJjZS5Qcm9wZXJ0aWVzLkVudmlyb25tZW50LmtleSA9XG4gICAgICB0ZW1wbGF0ZS5SZXNvdXJjZXMubXlSZXNvdXJjZS5Qcm9wZXJ0aWVzLkVudmlyb25tZW50LktleTtcbiAgICBkZWxldGUgdGVtcGxhdGUuUmVzb3VyY2VzLm15UmVzb3VyY2UuUHJvcGVydGllcy5FbnZpcm9ubWVudC5LZXk7XG5cbiAgICByZXR1cm4gdGVtcGxhdGU7XG4gIH1cbn1cbiJdfQ==