# -*- coding: utf-8 -*-
# schema package
"""
============
sfftk.schema
============

This module defines an usable adapter API to the one generated by
`generateDS <http://www.davekuhlman.org/generateDS.html>`_ API. It is designed to fulfill several requirements:

-   allow multiple versions of generateDS APIs to exist for the same adapter;

-   implement more convenient ways of working with the classes of the API;

-   easily integrate new EMDB-SFF entities into the API;

-   decouple the file format from the data model: most classes in this adapter implement methods to handle conversion
    to non-XML formats (XML is the only format that `generateDS <http://www.davekuhlman.org/generateDS.html>`_ writes
    to)

This adapter API therefore specifies two main base classes from which all other classes are built upon:

The :py:class:`SFFType` Base Class
==================================

This class contains all the magic to convert a subclass definition into a user-level class. View the
API documentation of this class for details on its attributes.

The :py:class:`SFFAttribute` Descriptor Class
================================================

Attributes of subclasses of :py:class:`SFFType` are instances of this descriptor class.

In addition to the two above classes there is a :py:class:`SFFTypeError` class which is raised whenever a type error
occurs which provides details on the required type.

Here is an example of how the :py:class:`SFFSegmentation` class is defined using these classes:

.. code:: python

    class SFFSegmentation(SFFType):
        \"""Adapter class to make using the output of ``generateDS`` easier to use\"""
        gds_type = sff.segmentation
        ref = "Segmentation"
        repr_string = ""

        # attributes
        name = SFFAttribute('name')
        version = SFFAttribute('version')
        software = SFFAttribute('software', sff_type=SFFSoftware)
        filePath = SFFAttribute('filePath')
        primaryDescriptor = SFFAttribute('primaryDescriptor')
        transforms = SFFAttribute('transformList', sff_type=SFFTransformList)
        boundingBox = SFFAttribute('boundingBox', sff_type=SFFBoundingBox)
        globalExternalReferences = SFFAttribute('globalExternalReferences', sff_type=SFFGlobalExternalReferences)
        segments = SFFAttribute('segmentList', sff_type=SFFSegmentList)
        details = SFFAttribute('details')

In the above example, the :py:class:`SFFSegmentation` class inherits from :py:class:`SFFType` which uses the first
three attributes (``gds_type``, ``ref`` and ``repr_string``) to configure the user-level :py:class:`SFFSegmentation`
class. A user can now use :py:class:`SFFSegmentation` normally:

.. code:: python

    from sfftk.schema import SFFSegmentation

    # to create a new empty EMDB-SFF segmentation object
    seg = SFFSegmentation()

    # to open an XML-, HDF5- or JSON-formatted file, respectively
    seg = SFFSegmentation('file.sff')
    seg = SFFSegmentation('file.hff')
    seg = SFFSegmentation('file.json')

The attributes (``name``, ``version``, etc.) are all instances of :py:class:`SFFAttribute`, which takes the name (first
positional argument) and an optional keyword argument ``sff_type`` denoting the class of the attribute. In the above
example, the ``software`` attribute will be of class :py:class:`SFFSoftware`.

"""
from __future__ import division, print_function

import base64
import os
import re
import struct
import sys
import zlib
from warnings import warn

import h5py
import numpy
import numpy as np

import emdb_sff as sff
from ..core.print_tools import print_date

__author__ = "Paul K. Korir, PhD"
__email__ = "pkorir@ebi.ac.uk, paul.korir@gmail.com"
__date__ = "2016-09-14"

# ensure that we can read/write encoded data
sff.ExternalEncoding = "utf-8"

# unused = 0
# containers = [
#     sff.contourListType, sff.meshListType,
#     sff.shapePrimitiveListType, sff.segmentListType,
#     sff.vertexListType, sff.polygonListType,
#     sff.transformListType
#     ]

FORMAT_CHARS = {
    'int8': 'b',
    'uint8': 'B',
    'int16': 'h',
    'uint16': 'H',
    'int32': 'i',
    'uint32': 'I',
    'int64': 'q',
    'uint64': 'Q',
    'float32': 'f',
    'float64': 'd',
}

ENDIANNESS = {
    'little': '<',
    'big': '>',
}


class SFFTypeError(Exception):
    """SFF type error"""

    def __init__(self, value, message=None):
        self.value = value
        self.message = message

    def __str__(self):
        if self.message is None:
            return repr("not object of {}".format(self.value))
        else:
            return repr("no object of {}; {}".format(self.value, self.message))


class SFFType(object):
    """Base class for all EMDB-SFF adapter classes
     
    This class reads and configures any subclass using the subclass' definition.

    Subclasses of :py:class:`SFFType` should define class attributes from the following list depending on whether
    the subclass defines a container or not. Containers typically have *List* in the class name signifying that they
    are a *list* of objects of some type. For example, the :py:class:`SFFSegmentList` objects are a *list of*
    :py:class:`SFFSegment` objects.

    :param var: argument value determines what the subclass does

    The :py:class:`type` of ``var`` determines what the subclass does.

    -   If the subclass is :py:class:`SFFSegmentation` then if ``var`` is:

        -   :py:class:`None` it will produce an empty :py:class:`SFFSegmentation` object

        -   an ``emdb_sff.segmentation`` object it will produce a populated :py:class:`SFFSegmentation` object;

        -   :py:class:`str` it will try to treat ``var`` as a filename containing data to populate an
            :py:class:`SFFSegmentation` object; the extension of the filename determines the format to be read:

            -   ``*.sff`` will read as XML

            -   ``*.hff`` will read as HDF5

            -   ``*.json`` will read as JSON

    -   If the subclass is any other class then if ``var`` is:

        -   :py:class:`None` it will produce an empty object of that class e.g. ``SFFSoftware()`` will produce an empty
            object for defining software data

        -   :py:class:`None` (or excluded) but keyword arguments to the generated generateDS class are specified it will
            produce a populated object of the corresponding adapter class e.g.
            ``SFFSoftware(name="name", version="version", processingDetails="details")`` will be an object loaded with
            the keyword data;

        -   any ``emdb_sff`` object it will produce a populated object of the corresponding adapter class;




    .. py:attribute:: gds_type

        The class generated by generateDS (*generate ds type*) that this class adapts

    .. py: attribute:: ref

        The *reference* name for objects of the subclass

    .. py:attribute:: repr_str

        A *string representation* for objects of the subclass that can be formatted. This string will be produced
        for calls to the :py:func:`print` function. Formatting is accomplished by included empty format delimiters
        (``{}``) which will be populated with values from the :py:attr:`repr_args` specified below otherwise
        literal braces will be displayed.

    .. py:attribute:: repr_args

        The *arguments of the string representation*, if any. This is a tuple containing any combination of either:

        -   the names of attributes of the subclass to populate the the :py:attr:`repr_string` attribute;

        -   a literal text string;

        -   the string ``len()`` to include the length of the subclass

    .. py:attribute:: iter_attr

        The *iterable attribute* of the class. It refers to an attribute of the class pointed to by
        :py:attr:`gds_type` which is iterable.

        It should be a two-tuple having the name of the generateDS class and ``SFF*`` class to adapt it to.

        Only one attribute per class can be specified.

        For example, consider the following class definition for some fictional ``SFFTest`` class that adapts a class
        called ``test`` that was generated by generateDS.

        .. code:: python

            import emdb_sff

            class SFFTest(SFFType):
                gds_type = emdb_sff.test
                repr_str = "SFFTest object with {} SFFItem objects inside"
                repr_args = ("len()",)
                iter_attr = ('i', SFFItem)

                # attributes
                items = SFFAttribute('items', sff_type=SFFItem)

        In this sample:

        -   the generateDS API is located in the module called ``emdb_sff``;

        -   ``test`` is a class in the ``emdb_sff`` module;

        -   ``i`` is an iterable attribute of ``test``;

        -   ``i`` is adapted by the ``SFFItem`` class (which should have its :py:attr:`gds_type` set to the class for
            ``i``);

        Now consider this class in action:

        .. code:: python

            from sfftk.schema import SFFTest

            T = SFFTest()
            # ...
            # populate T with some valid data
            # ...
            # now we can iterate over the data
            for i in T: # because it has an iter_attr specified
                # i is an SFFItem object
                print(i)
                # prints 'SFFTest object with 37 SFFItem objects inside'

        In addition to being iterable, the iterable attribute enable the following operations on objects of the
        subclass:

        -   length i.e. len(obj);

            .. code:: python

                len(T)
                # 37

        -   indexing e.g. ``obj[<int>]``;

            .. code:: python

                T[15]
    """
    gds_type = None
    ref = ""
    repr_string = ""
    repr_args = ()
    iter_attr = ()
    iter_dict = dict()

    def __init__(self, var=None, *args, **kwargs):
        """Base initialiser
        
        The top-level class has three forms
        #1 - SFFSegmentation() # empty segmentation object
        #2 - SFFSegmentation(emdb_sff.segmentation, *args, **kwargs) # build a segmentation object from an emdb_sff.segmentation object
        
        All other classes have two forms (e.g. SFFSoftware):
        #1 - SFFSoftware() # empty software object
        #2 - SFFSoftware(emdb_sff.softwareType) # build a software object from an emdb_sff.softwareType object
        #3 - SFFSoftware(name="name", version="version", processingDetails="details") # use gds_type kwargs
        
        """
        if self.gds_type:
            if var:
                if isinstance(var, self.gds_type):  # 2 - gds_type to SFFType
                    self._local = var
                else:
                    raise ValueError('{} is not of type {}'.format(var, self.gds_type))
            else:
                # restructure kwargs of type SFF* to their gds_type equivalents
                _kwargs = dict()
                for k in kwargs:
                    if isinstance(kwargs[k], SFFType):
                        _kwargs[k] = kwargs[k]._local
                    else:
                        _kwargs[k] = kwargs[k]
                self._local = self.gds_type(*args, **_kwargs)  # 1 and #3 - SFFType from (*a, **kw)
                # ensure that the version is copied without requiring user intervention
                if isinstance(self._local, sff.segmentation):
                    self.version = self._local.schemaVersion
        else:
            raise TypeError("attribute 'gds_type' cannot be 'None'")
        # load dict
        self._load_dict()

    def __repr__(self):
        return self.ref

    def __str__(self):
        if self.repr_string:
            if self.repr_args:
                assert isinstance(self.repr_args, tuple)
                if len(self.repr_args) == self.repr_string.count('{}'):
                    repr_args = list()
                    for arg in self.repr_args:
                        if arg == 'len()':
                            repr_args.append(len(self))
                        else:
                            repr_args.append(getattr(self, arg, None))
                    return self.repr_string.format(*repr_args)
                else:
                    raise ValueError("Unmatched number of '{}' and args in repr_args")
            else:
                return self.repr_string
        else:
            return str(type(self))

    def __iter__(self):
        if self.iter_attr:
            iter_name, iter_type = self.iter_attr
            if iter_name and iter_type:
                return iter(map(iter_type, getattr(self._local, iter_name)))
            elif iter_name:
                return iter(getattr(self._local, iter_name))
            elif iter_type:
                return iter(map(iter_type, self._local))
        else:
            raise TypeError("{} object is not iterable".format(self.__class__))

    def __len__(self):
        if self.iter_attr:
            iter_name, _ = self.iter_attr
            return len(getattr(self._local, iter_name))
        else:
            raise TypeError("object of type {} has no len()".format(self.__class__))

    def __getitem__(self, index):
        if self.iter_attr:
            iter_name, iter_type = self.iter_attr
            return iter_type(getattr(self._local, iter_name)[index])

    def __delitem__(self, index):
        if self.iter_attr:
            iter_name, _ = self.iter_attr
            # get the name of the iterable in _local (a list) then delete index pos from it
            del getattr(self._local, iter_name)[index]

    def _load_dict(self):
        self.iter_dict = dict()  #  initialise
        if self.iter_attr:
            for item in self:
                if isinstance(item, SFFType):
                    # if isinstance(item, SFFContourPoint):
                    #     pass  # contours points do not have ids (no reason why they can't though)
                    # else:
                    self.iter_dict[item.id] = item
                elif isinstance(item, int):
                    self.iter_dict[item] = item
                elif isinstance(item, str):
                    self.iter_dict[item] = item
                else:
                    raise ValueError("Unknown class {}".format(type(item)))

    def get_ids(self):
        """If the subclass is a container class, returns a list of IDs for items contained

        :return: a list of IDs
        :rtype: int
        """
        if self.iter_attr:
            return self.iter_dict.keys()

    def get_by_id(self, item_id):
        """If the subclass is a container class, returns the object of the specified ID

        :param int item_id: the ID whose item is required
        :return: a contained object
        :rtype: depends on the container class
        :raises ValueError: if the item is not found
        """
        if self.iter_attr:
            if item_id in self.iter_dict:
                return self.iter_dict[item_id]
            else:
                raise ValueError("ID {} not found".format(item_id))

    @classmethod
    def reset_id(cls):
        """Reset the ID for a subclass"""
        if issubclass(cls, SFFTransformationMatrix):
            cls.transform_id = -1
        elif issubclass(cls, SFFMesh):
            cls.mesh_id = -1
        elif issubclass(cls, SFFPolygon):
            cls.polygon_id = -1
        elif issubclass(cls, SFFSegment):
            cls.segment_id = 0
        elif issubclass(cls, SFFShape):
            cls.shape_id = -1
        elif issubclass(cls, SFFVertex):
            cls.vertex_id = -1
        elif issubclass(cls, SFFLattice):
            cls.lattice_id = -1

    def export(self, fn, *_args, **_kwargs):
        """Export to a file on disc

        :param str fn: filename to export to; the output format is determined by the extension:

        - ``.sff`` - XML
        - ``.hff`` - HDF5
        - ``.json`` - JSON
        """
        if isinstance(fn, str):  # fn is a file name
            fn_ext = fn.split('.')[-1]
            valid_extensions = ['sff', 'hff', 'json']
            try:
                assert fn_ext in valid_extensions
            except AssertionError:
                print_date("Invalid filename: extension should be one of {}: {}".format(
                    ", ".join(valid_extensions),
                    fn,
                ))
                sys.exit(1)
            if fn_ext == 'sff':
                with open(fn, 'w') as f:
                    # write version and encoding
                    version = _kwargs.get('version') if 'version' in _kwargs else "1.0"
                    encoding = _kwargs.get('encoding') if 'encoding' in _kwargs else "UTF-8"
                    f.write('<?xml version="{}" encoding="{}"?>\n'.format(version, encoding))
                    # always export from the root
                    self._local.export(f, 0, *_args, **_kwargs)
            elif fn_ext == 'hff':
                with h5py.File(fn, 'w') as f:
                    self.as_hff(f, *_args, **_kwargs)
            elif fn_ext == 'json':
                with open(fn, 'w') as f:
                    self.as_json(f, *_args, **_kwargs)
        elif isinstance(fn, file):
            self._local.export(fn, 0, *_args, **_kwargs)


class SFFAttribute(object):
    """Descriptor for SFFType subclass attributes"""

    def __init__(self, name, sff_type=None, get_from=None, set_to=None, del_from=None, help=""):
        """Initialiser for an attribute
        
        This class acts as an intermediary between ``SFFType`` and ``emdb_sff`` objects. Each ``SFFType``
        defines a ``_local`` attribute (defined from the ``gds_type`` class attribute, which points to 
        the ``emdb_sff`` object.
        
        Occassionally, the name of the ``emdb_sff`` attribute is different from the ``SFFType`` attribute.
        In this cases, a ``get_from`` argument controls where in the ``emdb_sff`` object the data should
        be obtained from and the ``set_to`` argument controls which attribute in ``emdb_sff`` it should 
        be set to. If both arguments are ``None`` (default) then get from the argument referred to by
        ``name``.
        
        :param str name: the name the attribute is referred to on the containing object
        :param sff_type: class of attribute (default: None - standard Python types like int, str, float)
        :type sff_type: ``SFFType``
        :param str get_from: which ``emdb_sff`` attribute to get the data from
        :param str set_to: which ``emdb_sff`` attribute to set the data to
        """
        self._name = name
        self.__doc__ = help
        self._sff_type = sff_type
        self._get_from = get_from
        self._set_to = set_to
        self._del_from = del_from

    def __get__(self, obj, _):  # replaced objtype with _
        if self._sff_type:
            if self._get_from:
                return self._sff_type(getattr(obj._local, self._get_from, None))
            else:
                return self._sff_type(getattr(obj._local, self._name, None))
        else:
            if self._get_from:
                return getattr(obj._local, self._get_from, None)
            else:
                return getattr(obj._local, self._name, None)

    def __set__(self, obj, value):
        if self._sff_type:
            if isinstance(value, self._sff_type):
                if self._set_to:
                    setattr(obj._local, self._set_to, value._local)
                else:
                    setattr(obj._local, self._name, value._local)
            else:
                raise SFFTypeError(self._sff_type)
        else:
            if self._set_to:
                setattr(obj._local, self._set_to, value)
            else:
                setattr(obj._local, self._name, value)

    def __delete__(self, obj):
        if self._del_from:
            delattr(obj._local, self._del_from)
        else:
            delattr(obj._local, self._name)


class SFFRGBA(SFFType):
    """RGBA colour"""
    gds_type = sff.rgbaType
    ref = "RGBA colour"
    repr_string = "({}, {}, {}, {})"
    repr_args = ('red', 'green', 'blue', 'alpha')

    # attributes
    red = SFFAttribute('red', help="red channel")
    green = SFFAttribute('green', help="green channel")
    blue = SFFAttribute('blue', help="blue channel")
    alpha = SFFAttribute('alpha', help="alpha (opacity) channel")

    @property
    def value(self):
        return self.red, self.green, self.blue, self.alpha

    @value.setter
    def value(self, c):
        if len(c) == 3:
            self.red, self.green, self.blue = c
        elif len(c) == 4:
            self.red, self.green, self.blue, self.alpha = c

    def __repr__(self):
        return str(self.value)

    def __nonzero__(self):
        if self.red is None or self.green is None or self.blue is None or self.alpha is None:
            return False
        else:
            return True

    def as_hff(self, parent_group, name="colour"):
        """Return the data of this object as an HDF5 group in the given parent group"""
        assert isinstance(parent_group, h5py.Group)
        parent_group[name] = self.value
        # group = parent_group.create_group(name)
        # group['rgba'] = self.value
        return parent_group

    @classmethod
    def from_hff(cls, hff_data):
        """Return an SFFType object given an HDF5 object"""
        assert isinstance(hff_data, h5py.Group)
        obj = cls()
        # r = SFFRGBA()
        obj.value = hff_data['colour'].value
        # obj.rgba = r
        return obj


class SFFComplexes(SFFType):
    """Class that encapsulates complex"""
    gds_type = sff.complexType
    ref = "Complexes"
    repr_string = "Complex list of length {}"
    repr_args = ('len()',)
    """
    :TODO: buggy; refers to emdb_sff attribute instead of SFFType attribute (which is inconsistent)
    """
    iter_attr = ('id', str)

    def set_complexes(self, cs):
        """Set value to the iterable of complexes

        :param list cs: an iterable of complex accessions
        """
        if isinstance(cs, list):
            self._local.set_id(cs)
        else:
            raise SFFTypeError(list)

    def add_complex(self, c):
        """Add a complex accession to the available complexes

        :param str c: a complex accession
        """
        if isinstance(c, str):
            self._local.add_id(c)
        else:
            raise SFFTypeError(str)

    def insert_complex_at(self, index, c):
        """Insert a complex accession at the given index

        :param int index: the index in the iterable of complexes at which to add the new complex accession
        :param str c: a complex accession
        """
        # todo: handle IndexError
        if isinstance(c, str):
            self._local.insert_id_at(index, c)
        else:
            raise SFFTypeError(str)

    def replace_complex_at(self, index, c):
        """Replace a complex accession at the given index

        :param int index: the index in the iterable of complexes at which to replace the complex accession
        :param str c: a complex accession
        """
        # todo: handle IndexError
        if isinstance(c, str):
            self._local.replace_id_at(index, c)
        else:
            raise SFFTypeError(str)

    def delete_at(self, index):
        """Delete the complex accession at the given index

        :param int index: index from which to delete the complex accession
        """
        del self._local.id[index]

    @classmethod
    def from_hff(cls, hff_data):
        """Return an SFFType object given an HDF5 object"""

        assert isinstance(hff_data, h5py.Dataset)

        obj = cls()
        [obj.add_complex(_) for _ in hff_data]

        return obj


class SFFMacromolecules(SFFType):
    """Class that encapsulates macromolecule"""
    gds_type = sff.macromoleculeType
    ref = "Macromolecules"
    repr_string = "Macromolecule list of length {}"
    repr_args = ("len()",)
    iter_attr = ('id', str)
    iter_dict = dict()

    def set_macromolecules(self, ms):
        """Set the value of macromoleclues to the provided list of macromolecule accessions

        :param list ms: a list of macromolecule accessions
        """
        if isinstance(ms, list):
            self._local.set_id(ms)
        else:
            raise SFFTypeError(list)

    def add_macromolecule(self, m):
        """Add the given macromolecule accession to this container

        :param str m: a macromolecule accession
        """
        if isinstance(m, str):
            self._local.add_id(m)
        else:
            raise SFFTypeError(str)

    def insert_macromolecule_at(self, index, m):
        """Insert the given macromolecule accession at the specified index bumping all others down the list

        :param int index: the index to insert at
        :param str m: a macromolecule accession
        """
        if isinstance(m, str):
            self._local.insert_id_at(index, m)
        else:
            raise SFFTypeError(str)

    def replace_macromolecule_at(self, index, m):
        """Replace the macromolecule accession at the specified index with the one specified

        :param int index: the index to insert at
        :param str m: a macromolecule accession
        """
        if isinstance(m, str):
            self._local.replace_id_at(index, m)
        else:
            raise SFFTypeError(str)

    def delete_at(self, index):
        """Delete the macromolecule accession at the give index

        :param int index: index of macromolecule accession to delete
        """
        del self._local.id[index]

    @classmethod
    def from_hff(cls, hff_data):
        """Return an SFFType object given an HDF5 object"""

        assert isinstance(hff_data, h5py.Dataset)

        obj = cls()
        [obj.add_macromolecule(_) for _ in hff_data]

        return obj


class SFFComplexesAndMacromolecules(SFFType):
    """Complexes and macromolecules"""
    gds_type = sff.macromoleculesAndComplexesType
    ref = "Complexes and macromolecules"
    repr_string = "Complexes: {}; Macromolecules: {}"
    repr_args = ('numComplexes', 'numMacromolecules')

    # attributes
    complexes = SFFAttribute('complex', sff_type=SFFComplexes, help="a list of complex accessions")
    macromolecules = SFFAttribute('macromolecule', sff_type=SFFMacromolecules,
                                  help="a list of macromolecule accessions")

    @property
    def numComplexes(self):
        return len(self.complexes)

    @property
    def numMacromolecules(self):
        return len(self.macromolecules)

    def __nonzero__(self):
        if self.complexes or self.macromolecules:
            return True
        else:
            return False

    def as_hff(self, parent_group, name="complexesAndMacromolecules"):
        """Return the data of this object as an HDF5 group in the given parent group"""
        assert isinstance(parent_group, h5py.Group)
        group = parent_group.create_group(name)
        if self.complexes:
            group['complexes'] = self.complexes
        if self.macromolecules:
            group['macromolecules'] = self.macromolecules
        return parent_group

    @classmethod
    def from_hff(cls, hff_data):
        """Return an SFFType object given an HDF5 object"""
        assert isinstance(hff_data, h5py.Group)
        obj = cls()
        obj.complexes = SFFComplexes.from_hff(hff_data['complexes'])
        obj.macromolecules = SFFMacromolecules.from_hff(hff_data['macromolecules'])

        return obj


class SFFExternalReference(SFFType):
    """Class that encapsulates an external reference"""
    gds_type = sff.externalReferenceType
    ref = "externalReference"
    repr_string = "Reference: {}; {}; {}"
    repr_args = ('type', 'otherType', 'value')

    # attributes
    id = SFFAttribute('id', help="this external reference's ID")
    type = SFFAttribute('type_', help="the ontology/archive name")
    otherType = SFFAttribute('otherType', help="a URL/IRI where data for this external reference may be obtained")
    value = SFFAttribute('value', help="the accession for this external reference")
    label = SFFAttribute('label', help="a short description of this external reference")
    description = SFFAttribute('description', help="a long description of this external reference")

    # methods
    def __init__(self, var=None, *args, **kwargs):
        # remap kwargs
        if 'type' in kwargs:
            kwargs['type_'] = kwargs['type']
            del kwargs['type']
        super(SFFExternalReference, self).__init__(var=var, *args, **kwargs)


class SFFExternalReferences(SFFType):
    """Container for external references"""
    gds_type = sff.externalReferencesType
    ref = "externalReferences"
    repr_string = "External references list with {} reference(s)"
    repr_args = ('len()',)
    iter_attr = ('ref', SFFExternalReference)
    iter_dict = dict()

    # methods
    def add_externalReference(self, eR):
        """Add the specified external reference object to this container

        :param eR: an external reference object
        :type eR: :py:class:`SFFExternalReference`
        """
        if isinstance(eR, SFFExternalReference):
            self._local.add_ref(eR._local)
        else:
            raise SFFTypeError(SFFExternalReference)

    def insert_externalReference(self, eR, index):
        """Insert the specified external reference object at the specified index

        :param eR: an external reference object
        :type eR: :py:class:`SFFExternalReference`
        :param int index: the index to insert to; bumps all other external references down the list
        """
        # todo: catch IndexError
        if isinstance(eR, SFFExternalReference) and isinstance(index, int):
            self._local.insert_ref_at(index, eR._local)
        else:
            if not isinstance(eR, SFFExternalReference):
                raise SFFTypeError(SFFExternalReference)
            elif not isinstance(index, int):
                raise SFFTypeError(int)

    def replace_externalReference(self, eR, index):
        """Replace the external reference at ``index`` with the specified external reference

        :param eR: an external reference object
        :type eR: :py:class:`SFFExternalReference`
        :param int index: the index to replace at
        """
        if isinstance(eR, SFFExternalReference) and isinstance(index, int):
            self._local.replace_ref_at(index, eR._local)
        else:
            if not isinstance(eR, SFFExternalReference):
                raise SFFTypeError(SFFExternalReference)
            elif not isinstance(index, int):
                raise SFFTypeError(int)


class SFFBiologicalAnnotation(SFFType):
    """Biological annotation"""
    gds_type = sff.biologicalAnnotationType
    ref = "biologicalAnnotation"
    repr_string = "Container for biological annotation with {} external references"
    repr_args = ('numExternalReferences',)

    # attributes
    name = SFFAttribute('name', help="the name of this segment")
    description = SFFAttribute('description', help="a brief description for this segment")
    externalReferences = SFFAttribute('externalReferences', SFFExternalReferences,
                                      help="the set of external references")
    numberOfInstances = SFFAttribute('numberOfInstances', help="the number of instances of this segment")

    # methods
    def __nonzero__(self):
        if not self.description and not self.externalReferences and not self.numberOfInstances:
            return False
        else:
            return True

    @property
    def numExternalReferences(self):
        return len(self.externalReferences)

    def as_hff(self, parent_group, name="biologicalAnnotation"):
        """Return the data of this object as an HDF5 group in the given parent group"""
        assert isinstance(parent_group, h5py.Group)
        group = parent_group.create_group(name)
        if self:
            vl_str = h5py.special_dtype(vlen=str)
            h_ext = group.create_dataset(
                "externalReferences",
                (self.numExternalReferences,),
                dtype=[
                    ('type', vl_str),
                    ('otherType', vl_str),
                    ('value', vl_str),
                    ('label', vl_str),
                    ('description', vl_str),
                ]
            )
            # description and nubmerOfInstances as attributes
            group['name'] = self.name if self.name else ''
            group['description'] = self.description if self.description else ''
            group['numberOfInstances'] = self.numberOfInstances if self.numberOfInstances > 0 else 0
            i = 0
            for extref in self.externalReferences:
                h_ext[i] = (extref.type, extref.otherType, extref.value, extref.label, extref.description)
                i += 1
        return parent_group

    @classmethod
    def from_hff(cls, hff_data):
        """Return an SFFType object given an HDF5 object"""
        assert isinstance(hff_data, h5py.Group)
        obj = cls()
        if 'name' in hff_data:
            obj.name = hff_data['name'].value
        if 'description' in hff_data:
            obj.description = hff_data['description'].value
        obj.numberOfInstances = int(hff_data['numberOfInstances'].value)
        if "externalReferences" in hff_data:
            obj.externalReferences = SFFExternalReferences()
            for ref in hff_data['externalReferences']:
                e = SFFExternalReference()
                e.type, e.otherType, e.value, e.label, e.description = ref
                obj.externalReferences.add_externalReference(e)
        return obj


class SFFThreeDVolume(SFFType):
    """Class representing segments described using a 3D volume"""
    gds_type = sff.threeDVolumeType
    ref = 'threeDVolume'
    repr_string = "3D formatted segment"

    # attributes
    latticeId = SFFAttribute('latticeId', help="the ID of the lattice that has the data for this 3D volume")
    value = SFFAttribute('value', help="the voxel values associated with this 3D volume")
    transformId = SFFAttribute('transformId', help="a transform applied to this 3D volume [optional]")

    def __nonzero__(self):
        if self.value is None:
            return False
        else:
            return True

    def as_hff(self, parent_group, name="volume"):
        """Return the data of this object as an HDF5 group in the given parent group"""
        assert isinstance(parent_group, h5py.Group)
        group = parent_group.create_group(name)
        group['latticeId'] = self.latticeId
        group['value'] = self.value
        if self.transformId is not None:
            group['transformId'] = self.transformId
        return parent_group

    @classmethod
    def from_hff(cls, hff_data):
        """Return an SFFType object given an HDF5 object"""
        assert isinstance(hff_data, h5py.Group)
        obj = cls()
        obj.latticeId = hff_data['latticeId'].value
        obj.value = hff_data['value'].value
        if 'transformId' in hff_data:
            obj.transformId = hff_data['transformId'].value
        return obj


class SFFVolume(SFFType):
    """Class for represention 3-space dimension"""
    # attributes
    cols = SFFAttribute('cols', help="number of columns")
    rows = SFFAttribute('rows', help="number of rows")
    sections = SFFAttribute('sections', help="number of sections (sets of congruent row-column collections)")

    @property
    def value(self):
        return self.cols, self.rows, self.sections

    @value.setter
    def value(self, value):
        if len(value) == 3:
            self.cols, self.rows, self.sections = value
        else:
            raise SFFTypeError("Iterable", "should be of length 3")

    @classmethod
    def from_hff(cls, hff_data):
        """Return an SFFType object given an HDF5 object"""
        assert isinstance(hff_data, h5py.Dataset)
        obj = cls()
        obj.cols = hff_data[0]
        obj.rows = hff_data[1]
        obj.sections = hff_data[2]
        return obj


class SFFVolumeStructure(SFFVolume):
    gds_type = sff.volumeStructureType
    ref = "3D volume structure: cols, rows, sections"
    repr_string = "3D volume structure: ({}, {}, {})"
    repr_args = ('cols', 'rows', 'sections')

    @property
    def voxelCount(self):
        """The number of voxels in this volume"""
        return self.cols * self.rows * self.sections


class SFFVolumeIndex(SFFVolume):
    gds_type = sff.volumeIndexType
    ref = "3D volume start index: cols, rows, sections"
    repr_string = "3D volume start index: [{}, {}, {}]"
    repr_args = ('cols', 'rows', 'sections')


class SFFLattice(SFFType):
    """Class representing 3D """
    gds_type = sff.latticeType
    ref = "3D lattice"
    repr_string = "Encoded 3D lattice with {}"
    repr_args = ('size',)
    lattice_id = -1

    # attributes
    id = SFFAttribute('id', help="the ID for this lattice (referenced by 3D volumes)")
    mode = SFFAttribute('mode',
                        help="type of data for each voxel; valid values are: int8, uint8, int16, uint16, int32, "
                             "uint32, int64, uint64, float32, float64")
    endianness = SFFAttribute('endianness', help="endianness; either 'little' (default) or 'big'")
    # todo: redundant to have size and data when size should be inferred from data
    size = SFFAttribute('size', sff_type=SFFVolumeStructure, help="size of the lattice described using a "
                                                                  ":py:class:`sfftk.schema.SFFVolumeStructure` object")
    start = SFFAttribute('start', sff_type=SFFVolumeIndex, help="starting index of the lattices described using a"
                                                                ":py:class:`sfftk.schema.SFFVolumeIndex` object")
    data = SFFAttribute('data', help="data provided by a numpy array; the dimensions should correspond with those "
                                     "specified in the 'size' attribute")

    def __new__(cls, *args, **kwargs):
        cls.lattice_id += 1
        return super(SFFLattice, cls).__new__(cls, *args, **kwargs)

    def __init__(self, var=None, *args, **kwargs):
        super(SFFLattice, self).__init__(var=var, *args, **kwargs)
        if 'id' in kwargs:
            self._local.id = kwargs['id']
        elif not var:
            self._local.id = self.lattice_id
        if not self.is_encoded:
            # encode on create
            self.encode()

    @property
    def is_encoded(self):
        """Tells whether the data in the lattice is encoded or not"""
        if isinstance(self.data, str):
            return True
        elif isinstance(self.data, numpy.ndarray):
            return False

    def encode(self):
        """Encode the numpy array provided in the initialiser

        Flatten -> Pack -> Zip -> Base64 encode
        """
        try:
            assert isinstance(self.data, numpy.ndarray)
        except AssertionError as a:
            print_date("Cannot encode data of type {}".format(type(self.data)))
            raise a
            sys.exit(os.EX_DATAERR)
        format_string = "{}{}{}".format(ENDIANNESS[self.endianness], self.size.voxelCount, FORMAT_CHARS[self.mode])
        binpack = struct.pack(format_string, *self.data.flat)
        # del binlist
        self.data = None
        binzip = zlib.compress(binpack)
        del binpack
        bin64 = base64.b64encode(binzip)
        del binzip
        self.data = bin64

    def decode(self):
        """Decode the data for processing

        Base64 decode -> Unzip -> Unpack -> Reshape
        """
        binzip = base64.b64decode(self.data)
        self.data = None
        binpack = zlib.decompress(binzip)
        del binzip
        _count = self.size.voxelCount
        bindata = struct.unpack("{}{}{}".format(ENDIANNESS[self.endianness], _count, FORMAT_CHARS[self.mode]), binpack)
        del binpack
        self.data = numpy.array(bindata).reshape(*self.size.value[::-1])
        del bindata
        # self.data = numpy.frombuffer(zlib.decompress(binzip)).reshape(*self.size.value)
        # del binzip

    @property
    def is_binary(self):
        """Quick check to see whether the 3D volume is binary or not"""
        voxel_values = set(self.data.flatten().tolist()).difference(set([0]))
        if len(voxel_values) == 1:
            return True
        else:
            return False

    def as_hff(self, parent_group, name="{}"):
        """Return the data of this object as an HDF5 group in the given parent group"""
        assert isinstance(parent_group, h5py.Group)
        group = parent_group.create_group(name.format(self.id))
        group['mode'] = self.mode
        group['endianness'] = self.endianness
        group['size'] = self.size.value
        group['start'] = self.start.value
        group['data'] = self.data
        return parent_group

    @classmethod
    def from_hff(cls, hff_data):
        """Return an SFFType object given an HDF5 object"""
        assert isinstance(hff_data, h5py.Group)
        obj = cls(
            mode=hff_data['mode'].value,
            endianness=hff_data['endianness'].value,
            size=SFFVolumeStructure.from_hff(hff_data['size']),
            start=SFFVolumeIndex.from_hff(hff_data['start']),
            data=hff_data['data'].value,
        )
        return obj


class SFFLatticeList(SFFType):
    """A container for lattice objects"""
    gds_type = sff.latticeListType
    ref = "Container for 3D lattices"
    repr_string = "Container with {} 3D lattices"
    repr_args = ("len()",)
    iter_attr = ('lattice', SFFLattice)
    iter_dict = dict()

    def __init__(self, var=None, *args, **kwargs):
        # reset id
        SFFLattice.reset_id()
        super(SFFLatticeList, self).__init__(var=var, *args, **kwargs)

    def add_lattice(self, l):
        """Add a lattice to the list of lattices

        :param l: a lattice object
        :type l: :py:class:`SFFLattice`
        """
        if isinstance(l, SFFLattice):
            self._local.add_lattice(l._local)
        else:
            raise SFFTypeError(SFFLattice)

    def as_hff(self, parent_group, name='lattices'):
        """Return the data of this object as an HDF5 group in the given parent group"""
        assert isinstance(parent_group, h5py.Group)
        group = parent_group.create_group(name)
        for lattice in self:
            group = lattice.as_hff(group)
        return parent_group

    @classmethod
    def from_hff(cls, hff_data):
        """Return an SFFType object given an HDF5 object"""
        assert isinstance(hff_data, h5py.Group)
        obj = cls()
        for lattice_id in hff_data:
            L = SFFLattice.from_hff(hff_data[lattice_id])
            L.id = int(lattice_id)
            obj.add_lattice(L)
        return obj


class SFFShape(SFFType):
    """Base shape class"""
    repr_string = "{} {}"
    repr_args = ('ref', 'id')
    shape_id = -1

    # attributes
    id = SFFAttribute('id', help="the ID of this shape")
    transformId = SFFAttribute('transformId', help="the transform applied to this shape to position it in the space")
    attribute = SFFAttribute('attribute', help="extra attribute information e.g. 'FOM'")


class SFFCone(SFFShape):
    """Cone shape class"""
    gds_type = sff.cone
    ref = "cone"

    # attributes
    height = SFFAttribute('height', help="cone height")
    bottomRadius = SFFAttribute('bottomRadius', help="cone bottom radius")

    def __new__(cls, *args, **kwargs):
        cls.shape_id = super(SFFCone, cls).shape_id + 1
        return super(SFFCone, cls).__new__(cls, *args, **kwargs)

    def __init__(self, var=None, *args, **kwargs):
        super(SFFCone, self).__init__(var=var, *args, **kwargs)
        if 'id' in kwargs:
            self._local.id = kwargs['id']
            SFFShape.shape_id = self.shape_id
        elif not var:
            self._local.id = self.shape_id
            SFFShape.shape_id = self.shape_id
        self._local.original_tagname_ = self.ref


class SFFCuboid(SFFShape):
    """Cuboid shape class"""
    gds_type = sff.cuboid
    ref = "cuboid"

    # attributes
    x = SFFAttribute('x', help="length in x")
    y = SFFAttribute('y', help="length in y")
    z = SFFAttribute('z', help="length in z")

    def __new__(cls, *args, **kwargs):
        cls.shape_id = super(SFFCuboid, cls).shape_id + 1
        return super(SFFCuboid, cls).__new__(cls, *args, **kwargs)

    def __init__(self, var=None, *args, **kwargs):
        super(SFFCuboid, self).__init__(var=var, *args, **kwargs)
        if 'id' in kwargs:
            self._local.id = kwargs['id']
            SFFShape.shape_id = self.shape_id
        elif not var:
            self._local.id = self.shape_id
            SFFShape.shape_id = self.shape_id
        self._local.original_tagname_ = self.ref


class SFFCylinder(SFFShape):
    """Cylinder shape class"""
    gds_type = sff.cylinder
    ref = "cylinder"

    # attributes
    height = SFFAttribute('height', help="cylinder height")
    diameter = SFFAttribute('diameter', help="cylinder diameter")

    def __new__(cls, *args, **kwargs):
        cls.shape_id = super(SFFCylinder, cls).shape_id + 1
        return super(SFFCylinder, cls).__new__(cls, *args, **kwargs)

    def __init__(self, var=None, *args, **kwargs):
        super(SFFCylinder, self).__init__(var=var, *args, **kwargs)
        if 'id' in kwargs:
            self._local.id = kwargs['id']
            SFFShape.shape_id = self.shape_id
        elif not var:
            self._local.id = self.shape_id
            SFFShape.shape_id = self.shape_id
        self._local.original_tagname_ = self.ref


class SFFEllipsoid(SFFShape):
    """Ellipsoid shape class"""
    gds_type = sff.ellipsoid
    ref = "ellipsoid"

    # attributes
    x = SFFAttribute('x', help="length in x")
    y = SFFAttribute('y', help="length in y")
    z = SFFAttribute('z', help="length in z")

    def __new__(cls, *args, **kwargs):
        cls.shape_id = super(SFFEllipsoid, cls).shape_id + 1
        return super(SFFEllipsoid, cls).__new__(cls, *args, **kwargs)

    def __init__(self, var=None, *args, **kwargs):
        super(SFFEllipsoid, self).__init__(var=var, *args, **kwargs)
        if 'id' in kwargs:
            self._local.id = kwargs['id']
            SFFShape.shape_id = self.shape_id
        elif not var:
            self._local.id = self.shape_id
            SFFShape.shape_id = self.shape_id
        self._local.original_tagname_ = self.ref


class SFFShapePrimitiveList(SFFType):
    """Container for shapes"""
    gds_type = sff.shapePrimitiveListType
    ref = 'shapePrimitiveList'
    repr_string = "Shape primitive list with some shapes"

    def __init__(self, var=None, *args, **kwargs):
        # reset id
        SFFShape.reset_id()
        super(SFFShapePrimitiveList, self).__init__(var=var, *args, **kwargs)

    def add_shape(self, s):
        """Add the provide shape into this shape container

        :param s: a shape object
        :type s: :py:class:`SFFShape`
        :raises SFFTypeError: if ``s`` is of the wrong type
        """
        if isinstance(s, SFFShape):
            self._local.shapePrimitive.append(s._local)
        else:
            raise SFFTypeError(SFFShape)

    def __len__(self):
        return len(self._local.shapePrimitive)

    def __getitem__(self, index):
        return self._shape_cast(self._local.shapePrimitive[index])

    @staticmethod
    def _shape_cast(shape):
        if isinstance(shape, sff.ellipsoid):
            return SFFEllipsoid(shape)
        elif isinstance(shape, sff.cuboid):
            return SFFCuboid(shape)
        elif isinstance(shape, sff.cylinder):
            return SFFCylinder(shape)
        elif isinstance(shape, sff.cone):
            return SFFCone(shape)
        else:
            raise TypeError("unknown shape type '{}'".format(type(shape)))

    def __iter__(self):
        return iter(map(self._shape_cast, self._local.shapePrimitive))

    def _shape_count(self, shape_type):
        return len(filter(lambda s: isinstance(s, shape_type), self._local.shapePrimitive))

    @property
    def numEllipsoids(self):
        """The number of ellipsoids in this container"""
        return self._shape_count(sff.ellipsoid)

    @property
    def numCuboids(self):
        """The number of cuboids in this container"""
        return self._shape_count(sff.cuboid)

    @property
    def numCylinders(self):
        """The number of cylinders in this container"""
        return self._shape_count(sff.cylinder)

    @property
    def numCones(self):
        """The number of cones in this container"""
        return self._shape_count(sff.cone)

    #     @property
    #     def numSubtomogramAverages(self):
    #         return self._shape_count(sff.subtomogramAverage)

    @classmethod
    def from_hff(cls, hff_data):
        """Return an SFFType object given an HDF5 object"""

        assert isinstance(hff_data, h5py.Group)

        obj = cls()
        if "ellipsoids" in hff_data:
            for ellipsoid in hff_data["ellipsoids"]:
                e = SFFEllipsoid()
                e.id = int(ellipsoid['id'])
                e.x = float(ellipsoid['x'])
                e.y = float(ellipsoid['y'])
                e.z = float(ellipsoid['z'])
                e.transformId = int(ellipsoid['transformId'])
                if not np.isnan(ellipsoid['attribute']):
                    e.attribute = float(ellipsoid['attribute'])
                obj.add_shape(e)
        if "cones" in hff_data:
            for cone in hff_data["cones"]:
                c = SFFCone()
                c.id = int(cone['id'])
                c.bottomRadius = float(cone['bottomRadius'])
                c.height = float(cone['height'])
                c.transformId = int(cone['transformId'])
                if not np.isnan(cone['attribute']):
                    c.attribute = float(cone['attribute'])
                obj.add_shape(c)
        if "cuboids" in hff_data:
            for cuboid in hff_data["cuboids"]:
                c = SFFCuboid()
                c.id = int(cuboid['id'])
                c.x = float(cuboid['x'])
                c.y = float(cuboid['y'])
                c.z = float(cuboid['z'])
                c.transformId = int(cuboid['transformId'])
                if not np.isnan(cuboid['attribute']):
                    c.attribute = float(cuboid['attribute'])
                obj.add_shape(c)
        if "cylinders" in hff_data:
            for cylinder in hff_data["cylinders"]:
                c = SFFCylinder()
                c.id = int(cylinder['id'])
                c.height = float(cylinder['height'])
                c.diameter = float(cylinder['diameter'])
                c.transformId = int(cylinder['transformId'])
                if not np.isnan(cylinder['attribute']):
                    c.attribute = float(cylinder['attribute'])
                obj.add_shape(c)

        return obj


class SFFVertex(SFFType):
    """Single vertex"""
    gds_type = sff.vertexType
    ref = "Vertex"
    repr_string = "{} vertex {}: ({}, {}, {})"
    repr_args = ('designation', 'vID', 'x', 'y', 'z')
    vertex_id = -1

    # attributes
    vID = SFFAttribute('vID', help="vertex ID; referenced by polygons")
    designation = SFFAttribute('designation', help="type of vertex ('surface' (default) or 'normal')")
    x = SFFAttribute('x', help="x co-ordinate")
    y = SFFAttribute('y', help="y co-ordinate")
    z = SFFAttribute('z', help="z co-ordinate")

    def __new__(cls, *args, **kwargs):
        cls.vertex_id += 1
        return super(SFFVertex, cls).__new__(cls, *args, **kwargs)

    def __init__(self, var=None, *args, **kwargs):
        super(SFFVertex, self).__init__(var=var, *args, **kwargs)
        """
        :TODO: vID fails to take effect; fails with IMODSegmentation
        """
        if 'vID' in kwargs:
            self._local.vID = kwargs['vID']
        elif not var:
            self._local.vID = self.vertex_id

    @property
    def point(self):
        """The co-ordinates for this vertex"""
        return self.x, self.y, self.z

    @point.setter
    def point(self, p):
        if isinstance(p, tuple):
            if len(p) == 3:
                self.x, self.y, self.z = p
            else:
                raise TypeError("point does not have three values")
        else:
            raise SFFTypeError(tuple)


class SFFPolygon(SFFType):
    """Single polygon"""
    gds_type = sff.polygonType
    ref = "Polygon"
    repr_string = "Polygon {}"
    repr_args = ('PID',)
    iter_attr = ('v', int)
    polygon_id = -1
    iter_dict = dict()

    # attributes
    PID = SFFAttribute('PID', help="the ID for this polygon")

    def __new__(cls, *args, **kwargs):
        cls.polygon_id += 1
        return super(SFFPolygon, cls).__new__(cls, *args, **kwargs)

    def __init__(self, var=None, *args, **kwargs):
        super(SFFPolygon, self).__init__(var=var, *args, **kwargs)
        if 'PID' in kwargs:
            self._local.PID = kwargs['PID']
        elif not var:
            self._local.PID = self.polygon_id

    @property
    def vertex_ids(self):
        """An iterable of vertex IDs of this polygon"""
        return [v for v in self]

    def add_vertex(self, v):
        """Add the vertex ID to this polygon

        :param int v: a vertex ID
        """
        if isinstance(v, int):
            self._local.add_v(v)
        else:
            raise SFFTypeError(int)


class SFFVertexList(SFFType):
    """List of vertices"""
    gds_type = sff.vertexListType
    ref = "List of vertices"

    def __init__(self, var=None, *args, **kwargs):
        # reset id
        SFFVertex.reset_id()
        super(SFFVertexList, self).__init__(var=var, *args, **kwargs)
        self._vertex_dict = {v.vID: v for v in map(SFFVertex, self._local.v)}

    @property
    def numVertices(self):
        """The number of vertices in this vertex container"""
        return len(self)

    def __str__(self):
        return "Vertex dict with {} vertices".format(len(self))

    def __len__(self):
        return len(self._local.v)

    def __iter__(self):
        return iter(self._vertex_dict.values())

    @property
    def vertex_ids(self):
        """Iterable of vertex IDs contained in this vertex container"""
        return iter(self._vertex_dict.keys())

    def __getitem__(self, vertex_id):
        return self._vertex_dict[vertex_id]

    def add_vertex(self, v):
        """Add the provided vertex to this vertex container

        :param v: a vertex object
        :type v: :py:class:`SFFVertex`
        :raises SFFTypeError: if ``v`` is of the wrong type
        """
        if isinstance(v, SFFVertex):
            self._local.add_v(v._local)
            self._local.numVertices = self.numVertices
        else:
            raise SFFTypeError(SFFVertex)

    @classmethod
    def from_hff(cls, hff_data):
        """Return an SFFType object given an HDF5 object"""
        assert isinstance(hff_data, h5py.Dataset)
        obj = cls()
        for vertex in hff_data:
            obj.add_vertex(
                SFFVertex(
                    vID=vertex['vID'],
                    designation=vertex['designation'],
                    x=float(vertex['x']),
                    y=float(vertex['y']),
                    z=float(vertex['z'])
                )
            )
        return obj


class SFFPolygonList(SFFType):
    """List of polygons"""
    gds_type = sff.polygonListType
    ref = "List of polygons"
    repr_string = "Polygon list with {} polygons"
    repr_args = ('len()',)

    def __init__(self, var=None, *args, **kwargs):
        # reset id
        SFFPolygon.reset_id()
        super(SFFPolygonList, self).__init__(var=var, *args, **kwargs)
        self._polygon_dict = {P.PID: P for P in map(SFFPolygon, self._local.P)}

    @property
    def numPolygons(self):
        """The number of polygons in this list of polygons"""
        return len(self)

    def __len__(self):
        return len(self._local.P)

    def __iter__(self):
        return iter(self._polygon_dict.values())

    @property
    def polygon_ids(self):
        """An iterable over the polygon IDs of the contained polygons"""
        return self.__iter__()

    def __getitem__(self, polygon_id):
        return self._polygon_dict[polygon_id]

    def __str__(self):
        return "Polygon list with {} polygons".format(len(self))

    def add_polygon(self, P):
        """Add a polygon to this polygon container

        :param p: a polygon object
        :type p: :py:class:`SFFPolygon`
        """
        if isinstance(P, SFFPolygon):
            self._local.add_P(P._local)
            self._local.numPolygons = self.numPolygons
        else:
            raise SFFTypeError(SFFPolygon)

    @classmethod
    def from_hff(cls, hff_data):
        """Return an SFFType object given an HDF5 object"""
        assert isinstance(hff_data, h5py.Dataset)
        obj = cls()
        for polygon in hff_data:
            P = SFFPolygon()
            P.PID = int(polygon['PID'])
            [P.add_vertex(int(_)) for _ in polygon['v']]
            obj.add_polygon(P)
        return obj


class SFFMesh(SFFType):
    """Single mesh"""
    gds_type = sff.meshType
    ref = "Mesh"
    repr_string = "Mesh {} with {} and {}"
    repr_args = ('id', 'vertices', 'polygons')
    mesh_id = -1

    # attributes
    id = SFFAttribute('id')
    vertices = SFFAttribute('vertexList', sff_type=SFFVertexList, help="a list of vertices (object of class :py:class:`sfftk.schema.SFFVertexList`)")
    polygons = SFFAttribute('polygonList', sff_type=SFFPolygonList, help="a list of derived polygons (object of class :py:class:`sfftk.schema.SFFPolygonList`)")
    transformId = SFFAttribute('transformId', help="a transform applied to the mesh")

    def __new__(cls, *args, **kwargs):
        cls.mesh_id += 1
        return super(SFFMesh, cls).__new__(cls, *args, **kwargs)

    def __init__(self, var=None, *args, **kwargs):
        super(SFFMesh, self).__init__(var=var, *args, **kwargs)
        if 'id' in kwargs:
            self._local.id = kwargs['id']
        elif not var:
            self._local.id = self.mesh_id

    @property
    def numVertices(self):
        """The number of vertices in this mesh"""
        return len(self.vertices)

    @property
    def numPolygons(self):
        """The number of polygons in this mesh"""
        return len(self.polygons)

    @classmethod
    def from_hff(cls, hff_data):
        """Return an SFFType object given an HDF5 object"""
        assert isinstance(hff_data, h5py.Group)
        obj = cls()
        obj.vertices = SFFVertexList.from_hff(hff_data['vertices'])
        obj.polygons = SFFPolygonList.from_hff(hff_data['polygons'])
        return obj


class SFFMeshList(SFFType):
    """Mesh list representation"""
    gds_type = sff.meshListType
    ref = "meshList"
    repr_string = "Mesh list with {} meshe(s)"
    repr_args = ('len()',)
    iter_attr = ('mesh', SFFMesh)
    iter_dict = dict()

    def __init__(self, var=None, *args, **kwargs):
        # reset id
        SFFMesh.reset_id()
        super(SFFMeshList, self).__init__(var=var, *args, **kwargs)

    def add_mesh(self, m):
        """Add a mesh into the list of meshes

        :param m: a mesh object
        :type m: :py:class:`SFFMesh`
        """
        if isinstance(m, SFFMesh):
            self._local.add_mesh(m._local)
        else:
            raise SFFType(SFFMesh)

    def as_hff(self, parent_group, name="meshes"):
        """Return the data of this object as an HDF5 group in the given parent group"""
        assert isinstance(parent_group, h5py.Group)
        group = parent_group.create_group(name)
        # structures
        vlen_str = h5py.special_dtype(vlen=str)
        vertex_array = h5py.special_dtype(vlen=np.dtype('u4'))  # create a variable-length for vertices
        for mesh in self:
            # /sff/segments/1/meshes/0 - mesh 0
            h_mesh = group.create_group("{}".format(mesh.id))
            # /sff/segments/1/meshes/0/vertices
            h_v = h_mesh.create_dataset(
                "vertices",
                (mesh.numVertices,),
                dtype=[
                    ('vID', 'u4'),
                    ('designation', vlen_str),
                    ('x', 'f4'),
                    ('y', 'f4'),
                    ('z', 'f4'),
                ],
                #                 compression="gzip",
            )
            # load vertex data
            i = 0
            for vertex in mesh.vertices:
                """
                :FIXME: recurrent bug
                RuntimeError: Unable to register datatype id (Can't insert duplicate key)
                """
                h_v[i] = (vertex.vID, vertex.designation, vertex.x, vertex.y, vertex.z)
                i += 1
            #         # attempt to avoid RuntimeError
            #         for mesh in self:
            # /sff/segments/1/meshes/0/polygons
            h_P = h_mesh.create_dataset(
                "polygons",
                (mesh.numPolygons,),
                dtype=[
                    ('PID', 'u4'),
                    ('v', vertex_array),
                ],
                #                 compression="gzip",
            )
            #  load polygon data
            j = 0
            for polygon in mesh.polygons:
                h_P[j] = (polygon.PID, np.array(polygon.vertex_ids))
                j += 1
            if mesh.transformId:
                h_mesh["transformId"] = mesh.transformId
        return parent_group

    @classmethod
    def from_hff(cls, hff_data):
        """Return an SFFType object given an HDF5 object"""
        assert isinstance(hff_data, h5py.Group)
        obj = cls()
        for mesh_id in hff_data:
            M = SFFMesh.from_hff(hff_data["{}".format(mesh_id)])
            M.id = int(mesh_id)
            obj.add_mesh(M)
        return obj


class SFFSegment(SFFType):
    """Class that encapsulates segment data"""
    gds_type = sff.segmentType
    ref = "Segment"
    repr_string = "Segment {}"
    repr_args = ('id',)
    segment_id = 0
    segment_parentID = 0

    # attributes
    id = SFFAttribute('id', help="the ID for this segment; segment IDs begin at 1 with the value of 0 implying the segmentation i.e. all segments are children of the root segment (the segmentation)")
    parentID = SFFAttribute('parentID', help="the ID for the segment that contains this segment; defaults to 0 (the whole segmentation)")
    biologicalAnnotation = SFFAttribute('biologicalAnnotation', sff_type=SFFBiologicalAnnotation, help="the biological annotation for this segment; described using a :py:class:`sfftk.schema.SFFBiologicalAnnotation` object")
    complexesAndMacromolecules = SFFAttribute('complexesAndMacromolecules', sff_type=SFFComplexesAndMacromolecules, help="the complexes and macromolecules associated with this segment; described using a :py:class:`sfftk.schema.SFFComplexesAndMacromolecules` object")
    colour = SFFAttribute('colour', sff_type=SFFRGBA, help="this segments colour; described using a :py:class:`sfftk.schema.SFFRGBA` object")
    meshes = SFFAttribute('meshList', sff_type=SFFMeshList, help="the list of meshes (if any) that describe this segment; a :py:class:`sfftk.schema.SFFMeshList` object")
    # contours = SFFAttribute('contourList', sff_type=SFFContourList)
    volume = SFFAttribute('threeDVolume', sff_type=SFFThreeDVolume, help="the 3D volume (if any) that describes this segment; a :py:class:`sfftk.schema.SFFThreeDVolume` object ")
    shapes = SFFAttribute('shapePrimitiveList', sff_type=SFFShapePrimitiveList, help="the list of shape primitives that describe this segment; a :py:class:`sfftk.schema.SFFShapePrimitiveList` object")

    def __new__(cls, *args, **kwargs):
        cls.segment_id += 1
        return super(SFFType, cls).__new__(cls, *args, **kwargs)

    def __init__(self, var=None, *args, **kwargs):
        super(SFFSegment, self).__init__(var=var, *args, **kwargs)
        """
        :TODO: if I want to add a new segment to a set of available segments does the id begin at the right value?
        """
        # id
        if 'id' in kwargs:
            self._local.id = kwargs['id']
        elif not var:
            self._local.id = self.segment_id
        # parentID
        if 'parentID' in kwargs:
            self._local.parentID = kwargs['parentID']
        elif not var:
            self._local.parentID = self.segment_parentID

    def as_hff(self, parent_group, name="{}"):
        """Return the data of this object as an HDF5 group in the given parent group"""
        assert isinstance(parent_group, h5py.Group)
        group = parent_group.create_group(name.format(self.id))
        group['parentID'] = self.parentID
        # add annotation data
        if self.biologicalAnnotation:
            group = self.biologicalAnnotation.as_hff(group)
        if self.complexesAndMacromolecules:
            group = self.complexesAndMacromolecules.as_hff(group)
        if self.colour:
            group = self.colour.as_hff(group)
        # add segmentation data
        if self.meshes:
            group = self.meshes.as_hff(group)
        if self.shapes:
            # /sff/segments/1/shapes
            h_shapes = group.create_group("shapes")
            # /sff/segments/1/shapes/ellipsoids
            h_ell = h_shapes.create_dataset(
                "ellipsoids",
                (self.shapes.numEllipsoids,),
                dtype=[
                    ('id', 'u4'),
                    ('x', 'f4'),
                    ('y', 'f4'),
                    ('z', 'f4'),
                    ('transformId', 'u4'),
                    ('attribute', 'f4'),
                ]
            )
            h_cub = h_shapes.create_dataset(
                "cuboids",
                (self.shapes.numCuboids,),
                dtype=[
                    ('id', 'u4'),
                    ('x', 'f4'),
                    ('y', 'f4'),
                    ('z', 'f4'),
                    ('transformId', 'u4'),
                    ('attribute', 'f4'),
                ]
            )

            h_cyl = h_shapes.create_dataset(
                "cylinders",
                (self.shapes.numCylinders,),
                dtype=[
                    ('id', 'u4'),
                    ('height', 'f4'),
                    ('diameter', 'f4'),
                    ('transformId', 'u4'),
                    ('attribute', 'f4'),
                ]
            )

            h_con = h_shapes.create_dataset(
                "cones",
                (self.shapes.numCones,),
                dtype=[
                    ('id', 'u4'),
                    ('height', 'f4'),
                    ('bottomRadius', 'f4'),
                    ('transformId', 'u4'),
                    ('attribute', 'f4'),
                ]
            )
            i = 0  # ellipsoid
            j = 0  # cuboid
            k = 0  # cylinder
            m = 0  # cone
            # n = 0 # subtomogram average
            for shape in self.shapes:
                if shape.ref == "Ellipsoid":
                    h_ell[i] = (shape.id, shape.x, shape.y, shape.z, shape.transformId,
                                shape.attribute if hasattr(shape, 'attribute') else None)
                    i += 1
                elif shape.ref == "Cuboid":
                    h_cub[j] = (shape.id, shape.x, shape.y, shape.z, shape.transformId,
                                shape.attribute if hasattr(shape, 'attribute') else None)
                    j += 1
                elif shape.ref == "Cylinder":
                    h_cyl[k] = (shape.id, shape.height, shape.diameter, shape.transformId,
                                shape.attribute if hasattr(shape, 'attribute') else None)
                    k += 1
                elif shape.ref == "Cone":
                    h_con[m] = (shape.id, shape.height, shape.bottomRadius, shape.transformId,
                                shape.attribute if hasattr(shape, 'attribute') else None)
                    m += 1
                elif shape.ref == "Subtomogram average":
                    warn("Unimplemented portion")
        if self.volume:
            # /sff/segments/1/volume
            group = self.volume.as_hff(group)
        return parent_group

    @classmethod
    def from_hff(cls, hff_data):
        """Return an SFFType object given an HDF5 object"""
        assert isinstance(hff_data, h5py.Group)
        obj = cls()
        obj.parentID = int(hff_data['parentID'].value)
        if "biologicalAnnotation" in hff_data:
            obj.biologicalAnnotation = SFFBiologicalAnnotation.from_hff(hff_data["biologicalAnnotation"])
        if "complexesAndMacromolecules" in hff_data:
            obj.complexesAndMacromolecules = SFFComplexesAndMacromolecules.from_hff(
                hff_data["complexesAndMacromolecules"])
        if "colour" in hff_data:
            obj.colour = SFFRGBA.from_hff(hff_data)
        if "meshes" in hff_data:
            obj.meshes = SFFMeshList.from_hff(hff_data["meshes"])
        if "shapes" in hff_data:
            obj.shapes = SFFShapePrimitiveList.from_hff(hff_data["shapes"])
        if "volume" in hff_data:
            obj.volume = SFFThreeDVolume.from_hff(hff_data["volume"])
        return obj


class SFFSegmentList(SFFType):
    """Container for segments"""
    gds_type = sff.segmentListType
    ref = "segmentList"
    repr_string = "Segment container"
    iter_attr = ('segment', SFFSegment)
    iter_dict = dict()

    def __init__(self, var=None, *args, **kwargs):
        # reset id
        SFFSegment.reset_id()
        super(SFFSegmentList, self).__init__(var=var, *args, **kwargs)

    def add_segment(self, s):
        """Add a segment to this segment container

        :param s: a segment object
        :type s: :py:class:`SFFSegment`
        :raises SFFTypeError: if ``s`` is of the wrong type
        """
        if isinstance(s, SFFSegment):
            self._local.add_segment(s._local)
        else:
            raise SFFTypeError(SFFSegment)

    def as_hff(self, parent_group, name="segments"):
        """Return the data of this object as an HDF5 group in the given parent group"""
        assert isinstance(parent_group, h5py.Group)
        group = parent_group.create_group(name)
        for segment in self:
            group = segment.as_hff(group)
        return parent_group

    @classmethod
    def from_hff(cls, hff_data):
        """Return an SFFType object given an HDF5 object"""
        assert isinstance(hff_data, h5py.Group)
        obj = cls()
        for segment_id in hff_data:
            S = SFFSegment.from_hff(hff_data[segment_id])
            S.id = int(segment_id)
            obj.add_segment(S)
        return obj


# class SFFTransform(SFFType):
#     """Transform"""
#     ref = "transform"
#     transform_id = -1
#
#     # attributes
#     id = SFFAttribute('id')


class SFFTransformationMatrix(SFFType):
    """Transformation matrix transform"""
    gds_type = sff.transformationMatrixType
    ref = "transformationMatrix"
    transform_id = -1

    # attributes
    id = SFFAttribute('id', help="an ID for this transform")
    rows = SFFAttribute('rows', help="the number of rows in this matrix")
    cols = SFFAttribute('cols', help="the number of columns in this matrix")
    data = SFFAttribute('data', help="the data in this matrix")
    # todo: work with numpy arrays transparently

    def __new__(cls, *args, **kwargs):
        cls.transform_id += 1
        return super(SFFTransformationMatrix, cls).__new__(cls, *args, **kwargs)

    def __init__(self, var=None, *args, **kwargs):
        super(SFFTransformationMatrix, self).__init__(var=var, *args, **kwargs)
        # override id if it is included
        if 'id' in kwargs:
            self._local.id = kwargs['id']
        elif not var:
            self._local.id = self.transform_id

        self._local.original_tagname_ = self.ref

    @property
    def data_array(self):
        """The data in this matrix as an array"""
        data_list = self.data.split(' ')
        data_array = np.array(data_list).reshape(self.rows, self.cols)
        return data_array
    #TODO: a setter for the above attribute

    def __str__(self):
        return (("[" + "{:.4f} " * self.cols + "]\n") * self.rows).format(*map(float, self.data.split(' ')))



class SFFTransformList(SFFType):
    """Container for transforms"""
    gds_type = sff.transformListType
    ref = "Transform list"
    repr_string = "List of transforms"
    iter_attr = ('transform', SFFTransformationMatrix)
    iter_dict = dict()

    def __init__(self, var=None, *args, **kwargs):
        # a new container of transforms needs the transform ID reset
        SFFTransformationMatrix.reset_id()
        super(SFFTransformList, self).__init__(var=var, *args, **kwargs)

    @property
    def transformationMatrixCount(self):
        """The number of :py:class:`SFFTransformationMatrix` objects in this transform container"""
        return len(self._local.transform)

    def add_transform(self, T):
        """Add the specified transform to this transform container"""
        if isinstance(T, SFFTransformationMatrix):
            self._local.add_transform(T._local)
        else:
            raise SFFTypeError(SFFTransformationMatrix)

    def check_transformation_matrix_homogeneity(self):
        """Helper method to check transformation matrix homogeneity
        
        If the transformation matrices are not homogeneous then we cannot use
        structured arrays in numpy :'(        
        """
        transformationMatricesSimilar = True  # assume they are all similar
        first = True
        rows = None
        cols = None
        for transform in self:
            if transform.ref == "transformationMatrix":
                if first:
                    rows = transform.rows
                    cols = transform.cols
                    first = False
                    continue
                else:
                    if transform.rows != rows or transform.cols != cols:
                        transformationMatricesSimilar = False
                        break
        return transformationMatricesSimilar, rows, cols

    def as_hff(self, parent_group, name="transforms"):
        """Return the data of this object as an HDF5 group in the given parent group"""
        assert isinstance(parent_group, h5py.Group)
        group = parent_group.create_group(name)
        # we need to check whether all transformationMatrices are of the same dimension
        # what we need to know:
        # - rows
        #  - cols
        # if they are then we just use rows and cols
        # else we should
        transformationMatricesSimilar, rows, cols = self.check_transformation_matrix_homogeneity()
        if self.transformationMatrixCount:
            if transformationMatricesSimilar:
                h_tM = group.create_dataset(
                    "transformationMatrix",
                    (self.transformationMatrixCount,),
                    dtype=[
                        ('id', 'u4'),
                        ('rows', 'u1'),
                        ('cols', 'u1'),
                        ('data', 'f4', (rows, cols)),
                    ]
                )
            else:
                h_tM = group.create_group("transformationMatrix")
        i = 0  # h_tM index
        j = 0  # h_cEA index
        k = 0  # h_vVR index
        for transform in self:
            if transform.ref == "transformationMatrix":
                if transformationMatricesSimilar:
                    h_tM[i] = (transform.id, transform.rows, transform.cols, transform.data_array)
                    i += 1
                else:
                    tM = h_tM.create_dataset(
                        "{}".format(transform.id),
                        (1,),
                        dtype=[
                            ('id', 'u4'),
                            ('rows', 'u1'),
                            ('cols', 'u1'),
                            ('data', 'f4', (rows, cols)),
                        ]
                    )
                    tM[0] = (transform.id, transform.rows, transform.cols, transform.data_array)
                    i += 1
        return parent_group

    @classmethod
    def from_hff(cls, hff_data):
        """Return an SFFType object given an HDF5 object"""
        assert isinstance(hff_data, h5py.Group)
        obj = cls()
        if "transformationMatrix" in hff_data:
            for _transform in hff_data['transformationMatrix']:
                if isinstance(hff_data['transformationMatrix'], h5py.Group):
                    transform = hff_data['transformationMatrix'][_transform][0]
                else:
                    transform = _transform
                T = SFFTransformationMatrix()
                T.id = transform['id']
                T.rows = transform['rows']
                T.cols = transform['cols']
                T.data = " ".join(map(str, transform['data'].flatten()))
                obj.add_transform(T)
        return obj


class SFFSoftware(SFFType):
    """Class definition for specifying software used to create this segmentation

    .. py:attribute:: name

        The name of the software used

    .. py:attribute:: version

        The version string

    .. py:attribute:: processingDetails

        Details of how the segmentation was produced
    """
    gds_type = sff.softwareType
    ref = "Software"
    repr_string = "Software object"

    # attributes
    name = SFFAttribute('name', help="the software/programme's name")
    version = SFFAttribute('version', help="the version used")
    processingDetails = SFFAttribute('processingDetails', help="a description of how the data was processed to produce the segmentation")

    def as_hff(self, parent_group, name="software"):
        """Return the data of this object as an HDF5 group in the given parent group"""
        assert isinstance(parent_group, h5py.Group)
        group = parent_group.create_group(name)
        group['name'] = self.name if self.name else ''
        group['version'] = self.version if self.version else ''
        if self.processingDetails:
            group['processingDetails'] = self.processingDetails
        return parent_group

    @classmethod
    def from_hff(cls, hff_data):
        """Return an SFFType object given an HDF5 object"""
        assert isinstance(hff_data, h5py.Group)
        obj = cls()
        obj.name = hff_data['name'].value
        obj.version = str(hff_data['version'].value)
        if 'processingDetails' in hff_data:
            obj.processingDetails = hff_data['processingDetails'].value
        return obj


class SFFBoundingBox(SFFType):
    """Dimensions of bounding box"""
    #  config
    gds_type = sff.boundingBoxType
    ref = "Bounding box"
    repr_string = "Bounding box: ({}, {}, {}, {}, {}, {})"
    repr_args = ('xmin', 'xmax', 'ymin', 'ymax', 'zmin', 'zmax')

    # attributes
    xmin = SFFAttribute('xmin', help="minimum x co-ordinate value")
    xmax = SFFAttribute('xmax', help="maximum x co-ordinate value")
    ymin = SFFAttribute('ymin', help="minimum y co-ordinate value")
    ymax = SFFAttribute('ymax', help="maximum y co-ordinate value")
    zmin = SFFAttribute('zmin', help="minimum z co-ordinate value")
    zmax = SFFAttribute('zmax', help="maximum z co-ordinate value")

    # methods
    def as_hff(self, parent_group, name="boundingBox"):
        """Bounding box as HDF5 group"""
        assert isinstance(parent_group, h5py.Group)
        group = parent_group.create_group(name)
        group['xmin'] = self.xmin
        group['xmax'] = self.xmax
        group['ymin'] = self.ymin
        group['ymax'] = self.ymax
        group['zmin'] = self.zmin
        group['zmax'] = self.zmax
        return parent_group

    @classmethod
    def from_hff(cls, hff_data):
        """Bounding box from HDF5 group"""
        assert isinstance(hff_data, h5py.Group)
        obj = cls()
        obj.xmin = hff_data['xmin'].value
        obj.xmax = hff_data['xmax'].value
        obj.ymin = hff_data['ymin'].value
        obj.ymax = hff_data['ymax'].value
        obj.zmin = hff_data['zmin'].value
        obj.zmax = hff_data['zmax'].value
        return obj


class SFFGlobalExternalReferences(SFFType):
    """Container for global external references"""
    gds_type = sff.globalExternalReferencesType
    ref = "globalExternalReference"
    repr_string = "Global external reference list with {} reference(s)"
    repr_args = ('len()',)
    iter_attr = ('ref', SFFExternalReference)
    iter_dict = dict()

    # methods
    def add_externalReference(self, eR):
        """Add the specified external reference object to this container

        :param eR: an external reference object
        :type eR: :py:class:`SFFExternalReference`
        """
        if isinstance(eR, SFFExternalReference):
            self._local.add_ref(eR._local)
        else:
            raise SFFTypeError(SFFExternalReference)

    def insert_externalReference(self, eR, index):
        """Insert the specified external reference object at the specified index

        :param eR: an external reference object
        :type eR: :py:class:`SFFExternalReference`
        :param int index: the index to insert to; bumps all other external references down the list
        """
        if isinstance(eR, SFFExternalReference) and isinstance(index, int):
            self._local.insert_ref_at(index, eR._local)
        else:
            if not isinstance(eR, SFFExternalReference):
                raise SFFTypeError(SFFExternalReference)
            elif not isinstance(index, int):
                raise SFFTypeError(int)

    def replace_externalReference(self, eR, index):
        """Replace the external reference at ``index`` with the specified external reference

        :param eR: an external reference object
        :type eR: :py:class:`SFFExternalReference`
        :param int index: the index to replace at
        """
        if isinstance(eR, SFFExternalReference) and isinstance(index, int):
            self._local.replace_ref_at(index, eR._local)
        else:
            if not isinstance(eR, SFFExternalReference):
                raise SFFTypeError(SFFExternalReference)
            elif not isinstance(index, int):
                raise SFFTypeError(int)


class SFFSegmentation(SFFType):
    """Adapter class to make using the output of ``generateDS`` easier to use"""
    gds_type = sff.segmentation
    ref = "Segmentation"
    repr_string = ""

    # attributes
    name = SFFAttribute('name', help="the name of this segmentation")
    version = SFFAttribute('version', help="EMDB-SFF version")
    software = SFFAttribute('software', sff_type=SFFSoftware, help="the software details used to generate this segmentationa :py:class:`sfftk.schema.SFFSoftware` object")
    primaryDescriptor = SFFAttribute('primaryDescriptor', help="the main type of representation used for this segmentation; can be one of 'meshList', 'shapePrimitiveList' or 'threeDVolume'")
    transforms = SFFAttribute('transformList', sff_type=SFFTransformList, help="a list of transforms; a :py:class:`sfftk.schema.SFFTransformList` object")
    boundingBox = SFFAttribute('boundingBox', sff_type=SFFBoundingBox, help="the bounding box in which the segmentation sits; a :py:class:`sfftk.schema.SFFBoundingBox` object")
    globalExternalReferences = SFFAttribute('globalExternalReferences', sff_type=SFFGlobalExternalReferences, help="a list of external references that apply to the whole segmentation (global); a :py:class:`sfftk.schema.SFFGlobalExternalReferences` object")
    segments = SFFAttribute('segmentList', sff_type=SFFSegmentList, help="the list of annotated segments; a :py:class:`sfftk.schema.SFFSegmentList` object")
    lattices = SFFAttribute('latticeList', sff_type=SFFLatticeList, help="the list of lattices (if any) containing 3D volumes referred to; a :py:class:`sfftk.schema.SFFLatticeList` object")
    details = SFFAttribute('details', help="any other details about this segmentation (free text)")

    # properties, methods
    def __init__(self, var=None, *args, **kwargs):
        """Initialiser to handle opening from EMDB-SFF files (XML, HDF5, JSON)"""
        if isinstance(var, str) or isinstance(var, unicode):
            # Experimental
            if re.match(r'.*\.sff$', var, re.IGNORECASE):
                try:
                    self._local = sff.parse(var, silence=True, *args, **kwargs)
                except IOError:
                    print_date("File {} not found".format(var))
                    sys.exit(os.EX_IOERR)
            elif re.match(r'.*\.hff$', var, re.IGNORECASE):
                with h5py.File(var) as h:
                    self._local = self.__class__.from_hff(h, *args, **kwargs)._local
            elif re.match(r'.*\.json$', var, re.IGNORECASE):
                self._local = self.__class__.from_json(var, *args, **kwargs)._local
            else:
                print_date("Invalid EMDB-SFF file name: {}".format(var))
                sys.exit(os.EX_USAGE)
        else:
            super(SFFSegmentation, self).__init__(var, *args, **kwargs)

    @property
    def numGlobalExternalReferences(self):
        """The number of global external references"""
        return len(self.globalExternalReferences)

    def as_hff(self, parent_group, name=None):
        """Return the data of this object as an HDF5 group in the given parent group"""
        assert isinstance(parent_group, h5py.File)
        if name:
            group = parent_group.create_group(name)
        else:
            group = parent_group
        group['name'] = self.name if self.name else ''
        group['version'] = self.version
        group['primaryDescriptor'] = self.primaryDescriptor
        # if we are adding another group then don't set dict style; just return the populated group
        group = self.software.as_hff(group)
        group = self.transforms.as_hff(group)
        if self.boundingBox.xmax:
            group = self.boundingBox.as_hff(group)
        if self.globalExternalReferences:
            vl_str = h5py.special_dtype(vlen=str)
            h_gext = group.create_dataset(
                "globalExternalReferences",
                (self.numGlobalExternalReferences,),
                dtype=[
                    ('type', vl_str),
                    ('otherType', vl_str),
                    ('value', vl_str),
                    ('label', vl_str),
                    ('description', vl_str),
                ]
            )
            i = 0
            for gExtRef in self.globalExternalReferences:
                h_gext[i] = (gExtRef.type, gExtRef.otherType, gExtRef.value, gExtRef.label, gExtRef.description)
                i += 1
        group = self.segments.as_hff(group)
        group = self.lattices.as_hff(group)
        group['details'] = self.details if self.details else ''
        return parent_group

    @classmethod
    def from_hff(cls, hff_data):
        """Create an :py:class:`sfftk.schema.SFFSegmentation` object from HDF5 formatted data
        
        :param hff_data: an HDF5 File object
        :type hff_data: ``h5py.File``
        :return sff_seg: an EMDB-SFF segmentation
        :rtype sff_seg: :py:class:`sfftk.schema.SFFSegmentation`
        """
        assert isinstance(hff_data, h5py.File)
        obj = cls()
        try:
            obj.name = hff_data['name'].value
        except KeyError:
            print_date('Segmentation name not found. Please check that {} is a valid EMDB-SFF file'.format(
                hff_data.filename
            ))
            sys.exit(1)
        obj.version = str(hff_data['version'].value)
        obj.software = SFFSoftware.from_hff(hff_data['software'])
        obj.transforms = SFFTransformList.from_hff(hff_data['transforms'])
        # obj.filePath = hff_data['filePath'].value
        obj.primaryDescriptor = hff_data['primaryDescriptor'].value
        if 'boundingBox' in hff_data:
            obj.boundingBox = SFFBoundingBox.from_hff(hff_data['boundingBox'])
        if "globalExternalReferences" in hff_data:
            obj.globalExternalReferences = SFFGlobalExternalReferences()
            for gref in hff_data['globalExternalReferences']:
                g = SFFExternalReference()
                g.type, g.otherType, g.value, g.label, g.description = gref
                obj.globalExternalReferences.add_externalReference(g)
        obj.segments = SFFSegmentList.from_hff(hff_data['segments'])
        obj.lattices = SFFLatticeList.from_hff(hff_data['lattices'])
        obj.details = hff_data['details'].value
        return obj

    def as_json(self, f, sort_keys=True, indent_width=2):
        """Render an EMDB-SFF to JSON
        
        :param file f: open file handle
        :param bool annotation_only: only extract annotation information and do not render geometric data
        :param bool sort_keys: whether (default) or not to sort keys in the dictionaries
        :param int indent_width: indent width (default: 2)
        """
        """
        :TODO: also extract geometrical data
        """
        sff_data = dict()
        # can be simplified
        sff_data['name'] = self.name
        sff_data['version'] = self.version
        sff_data['software'] = {
            'name': self.software.name,
            'version': self.software.version,
            'processingDetails': self.software.processingDetails if self.software.processingDetails is not None else None,
        }
        sff_data['primaryDescriptor'] = self.primaryDescriptor
        # sff_data['filePath'] = self.filePath
        sff_data['details'] = self.details
        sff_data['transforms'] = list()
        boundingBox = {
            'xmin': self.boundingBox.xmin,
            'xmax': self.boundingBox.xmax,
            'ymin': self.boundingBox.ymin,
            'ymax': self.boundingBox.ymax,
            'zmin': self.boundingBox.zmin,
            'zmax': self.boundingBox.zmax,
        }
        sff_data['boundingBox'] = boundingBox
        globalExternalReferences = list()
        for gextref in self.globalExternalReferences:
            globalExternalReferences.append({
                'type': gextref.type,
                'otherType': gextref.otherType,
                'value': gextref.value,
                'label': gextref.label,
                'description': gextref.description
            })
        sff_data['globalExternalReferences'] = globalExternalReferences
        sff_data['segments'] = list()
        for segment in self.segments:
            seg_data = dict()
            seg_data['id'] = int(segment.id)
            seg_data['parentID'] = int(segment.parentID)
            bioAnn = dict()
            bioAnn[
                'name'] = segment.biologicalAnnotation.name if segment.biologicalAnnotation.name is not None else None
            bioAnn['description'] = str(
                segment.biologicalAnnotation.description) if segment.biologicalAnnotation.description is not None else None
            bioAnn[
                'numberOfInstances'] = segment.biologicalAnnotation.numberOfInstances if segment.biologicalAnnotation.numberOfInstances is not None else None

            bioAnn['externalReferences'] = list()
            if segment.biologicalAnnotation.externalReferences:
                for extref in segment.biologicalAnnotation.externalReferences:
                    bioAnn['externalReferences'].append(
                        {
                            'type': extref.type,
                            'otherType': extref.otherType,
                            'value': extref.value,
                            'label': extref.label,
                            'description': extref.description,
                        }
                    )
            seg_data['biologicalAnnotation'] = bioAnn
            if segment.complexesAndMacromolecules:
                complexes = list()
                for _complex in segment.complexesAndMacromolecules.complexes:
                    complexes.append(_complex)
                macromolecules = list()
                for macromolecule in segment.complexesAndMacromolecules.macromolecules:
                    macromolecules.append(macromolecule)
                seg_data['complexesAndMacromolecules'] = {
                    'complexes': complexes,
                    'macromolecules': macromolecules,
                }
            seg_data['colour'] = map(float, segment.colour.value)
            if segment.meshes:
                seg_data['meshList'] = len(segment.meshes)
            # exclude geometric data in JSON
            # if segment.volume is not None:
            #     seg_data['threeDVolume'] = {
            #         'latticeId': segment.volume.latticeId,
            #         'value': int(segment.volume.value),
            #     }
            #     if segment.volume.transformId is not None:
            #         seg_data['threeDVolume']['transformId'] = segment.volume.transformId,
            if segment.shapes:
                seg_data['shapePrimitiveList'] = len(segment.shapes)
            sff_data['segments'].append(seg_data)
        sff_data['details'] = self.details
        # write to f
        with f:
            import json
            json.dump(sff_data, f, sort_keys=sort_keys, indent=indent_width)

    @classmethod
    def from_json(cls, json_file):
        """Create an :py:class:`sfftk.schema.SFFSegmentation` object from JSON formatted data
        
        :param str json_file: name of a JSON-formatted file
        :return sff_seg: an EMDB-SFF segmentation
        :rtype sff_seg: :py:class:`sfftk.schema.SFFSegmentation`
        """
        with open(json_file) as j:
            import json
            J = json.load(j, 'utf-8')
        sff_seg = cls()
        # header
        sff_seg.name = J['name']
        sff_seg.version = J['version']
        sff_seg.software = SFFSoftware(
            name=J['software']['name'],
            version=J['software']['version'],
            processingDetails=J['software']['processingDetails'],
        )
        sff_seg.primaryDescriptor = J['primaryDescriptor']
        if 'boundingBox' in J:
            sff_seg.boundingBox = SFFBoundingBox(
                xmin=J['boundingBox']['xmin'],
                xmax=J['boundingBox']['xmax'],
                ymin=J['boundingBox']['ymin'],
                ymax=J['boundingBox']['ymax'],
                zmin=J['boundingBox']['zmin'],
                zmax=J['boundingBox']['zmax'],
            )
        if 'globalExternalReferences' in J:
            sff_seg.globalExternalReferences = SFFGlobalExternalReferences()
            for gextref in J['globalExternalReferences']:
                try:
                    label = gextref['label']
                except KeyError:
                    label = None
                try:
                    description = gextref['description']
                except KeyError:
                    description = None
                sff_seg.globalExternalReferences.add_externalReference(
                    SFFExternalReference(
                        type=gextref['type'],
                        otherType=gextref['otherType'],
                        value=gextref['value'],
                        label=label,
                        description=description,
                    )
                )
        # segments
        segments = SFFSegmentList()
        for s in J['segments']:
            r, g, b, a = s['colour']
            segment = SFFSegment()
            segment.id = s['id']
            segment.parentID = s['parentID']
            if 'biologicalAnnotation' in s:
                biologicalAnnotation = SFFBiologicalAnnotation()
                biologicalAnnotation.name = s['biologicalAnnotation']['name']
                biologicalAnnotation.description = s['biologicalAnnotation']['description']
                biologicalAnnotation.numberOfInstances = s['biologicalAnnotation']['numberOfInstances']
                if 'externalReferences' in s['biologicalAnnotation']:
                    biologicalAnnotation.externalReferences = SFFExternalReferences()
                    for extRef in s['biologicalAnnotation']['externalReferences']:
                        try:
                            label = extRef['label']
                        except KeyError:
                            label = None
                        try:
                            description = extRef['description']
                        except KeyError:
                            description = None
                        externalReference = SFFExternalReference(
                            type=extRef['type'],
                            otherType=extRef['otherType'],
                            value=extRef['value'],
                            label=label,
                            description=description,
                        )
                        biologicalAnnotation.externalReferences.add_externalReference(externalReference)
                segment.biologicalAnnotation = biologicalAnnotation
            if 'complexesAndMacromolecules' in s:
                complexesAndMacromolecules = SFFComplexesAndMacromolecules()
                if 'complexes' in s['complexesAndMacromolecules']:
                    complexes = SFFComplexes()
                    complexes.set_complexes(s['complexesAndMacromolecules']['complexes'])
                    complexesAndMacromolecules.complexes = complexes
                if 'macromolecules' in s['complexesAndMacromolecules']:
                    macromolecules = SFFMacromolecules()
                    macromolecules.set_macromolecules(s['complexesAndMacromolecules']['macromolecules'])
                    complexesAndMacromolecules.macromolecules = macromolecules
                segment.complexesAndMacromolecules = complexesAndMacromolecules
            segment.colour = SFFRGBA(
                red=r,
                green=g,
                blue=b,
                alpha=a,
            )
            # in order for sff notes to work with JSON there should be an empty geom
            if 'meshList' in s:
                segment.meshes = SFFMeshList()
                for _ in xrange(s['meshList']):
                    segment.meshes.add_mesh(SFFMesh())
            if 'threeDVolume' in s:
                # fixme: invalid model
                segment.volume = SFFThreeDVolume()
                # tDV = s['threeDVolume']
                # segment.volume.latticeId = tDV['latticeId']
                # segment.volume.value = tDV['value']
                # # segment.volume.file = tDV['file']
                # # segment.volume.format = tDV['format']
                # # segment.volume.objectPath = tDV['objectPath'] if 'objectPath' in tDV else None
                # # segment.volume.contourLevel = tDV['contourLevel'] if 'contourLevel' in tDV else None
                # segment.volume.transformId = tDV['transformId'] if 'transformId' in tDV else None
            if 'shapePrimitiveList' in s:
                segment.shapes = SFFShapePrimitiveList()
                for _ in xrange(s['shapePrimitiveList']):
                    segment.shapes.add_shape(SFFEllipsoid())
            segments.add_segment(segment)
        sff_seg.segments = segments
        # details
        sff_seg.details = J['details']
        return sff_seg

    def merge_annotation(self, other_seg):
        """Merge the annotation from another sff_seg to this one
        
        :param other_seg: segmentation to get annotations from
        :type other_seg: :py:class:`sfftk.schema.SFFSegmentation`
        """
        try:
            assert isinstance(other_seg, SFFSegmentation)
        except AssertionError:
            print_date("Invalid type for other_seg: {}".format(type(other_seg)))
            sys.exit(1)
        # global data
        self.name = other_seg.name
        # self.filePath = other_seg.filePath
        self.software = other_seg.software
        self.globalExternalReferences = other_seg.globalExternalReferences
        self.details = other_seg.details
        # loop through segments
        for segment in self.segments:
            other_segment = other_seg.segments.get_by_id(segment.id)
            segment.biologicalAnnotation = other_segment.biologicalAnnotation
            segment.complexesAndMacromolecules = other_segment.complexesAndMacromolecules

    def copy_annotation(self, from_id, to_id):
        """Copy annotation across segments

        :param int/list from_id: segment ID to get notes from; use -1 for for global notes
        :param int/list to_id: segment ID to copy notes to; use -1 for global notes
        """
        if from_id == -1:
            _from = self.globalExternalReferences
        else:
            _from = self.segments.get_by_id(from_id).biologicalAnnotation.externalReferences
        if to_id == -1:
            to = self.globalExternalReferences
        else:
            to = self.segments.get_by_id(to_id).biologicalAnnotation.externalReferences
        # the id for global notes
        for extref in _from:
            to.add_externalReference(extref)

    def clear_annotation(self, from_id):
        """Clear all annotations from the segment with ID specified

        :param from_id: segment ID
        :return:
        """
        if from_id == -1:
            self.globalExternalReferences = SFFGlobalExternalReferences()
        else:
            segment = self.segments.get_by_id(from_id)
            segment.biologicalAnnotation.externalReferences = SFFExternalReferences()
