import underworld as uw
import underworld.function as fn
from scaling import nonDimensionalize as nd
from scaling import UnitRegistry as u
from scaling import Dimensionalize
import json
from copy import copy


class _Polynom(object):

    def __init__(self, A1, A2=0., A3=0., A4=0.):

        self.A1 = A1
        self.A2 = A2
        self.A3 = A3
        self.A4 = A4

    def temperature(self, pressure, units=None):
        return nd(self.A1) + nd(self.A2) * pressure + nd(self.A3) * pressure**2 + nd(self.A4) * pressure**3

    def plot(self, pressure):
        import pylab as plt
        temperature = Dimensionalize(self.temperature(pressure), u.kelvin)
        pressure    = Dimensionalize(pressure, u.pascal)
        plt.plot(temperature, pressure)
        plt.gca().invert_yaxis()
        plt.show()

class Solidus(_Polynom):
    """ This class defines a solidus using the
    form suggested by Hirshmann, 2000"""

    pass

class Liquidus(_Polynom):
    """ This class defines a liquidus using the
    form suggested by Hirshmann, 2000"""

    pass

class SolidusRegistry(object):
    def __init__(self, filename=None):

        if not filename:
            import pkg_resources
            filename = pkg_resources.resource_filename(__name__, "ressources/Solidus.json")

        with open(filename, "r") as infile:
            _solidii = json.load(infile)

        for key in _solidii.keys():
            coefficients = _solidii[key]["coefficients"]
            for key2 in coefficients.keys():
                value = coefficients[key2]["value"]
                units = coefficients[key2]["units"]
                if units != "None":
                    coefficients[key2] = u.Quantity(value, units)
                else:
                    coefficients[key2] = value

        self._dir = {}
        for key in _solidii.keys():
            name = key.replace(" ","_").replace(",","").replace(".","")
            name = name.replace(")","").replace("(","")
            self._dir[name] = Solidus(**_solidii[key]["coefficients"])

    def __dir__(self):
        # Make all the rheology available through autocompletion
        return list(self._dir.keys())

    def __getattr__(self, item):
        # Make sure to return a new instance of ViscousCreep
        return copy(self._dir[item])

class LiquidusRegistry(object):
    def __init__(self, filename=None):

        if not filename:
            import pkg_resources
            filename = pkg_resources.resource_filename(__name__, "ressources/Liquidus.json")

        with open(filename, "r") as infile:
            _liquidii = json.load(infile)

        for key in _liquidii.keys():
            coefficients = _liquidii[key]["coefficients"]
            for key2 in coefficients.keys():
                value = coefficients[key2]["value"]
                units = coefficients[key2]["units"]
                if units != "None":
                    coefficients[key2] = u.Quantity(value, units)
                else:
                    coefficients[key2] = value

        self._dir = {}
        for key in _liquidii.keys():
            name = key.replace(" ","_").replace(",","").replace(".","")
            name = name.replace(")","").replace("(","")
            self._dir[name] = Liquidus(**_liquidii[key]["coefficients"])

    def __dir__(self):
        # Make all the rheology available through autocompletion
        return list(self._dir.keys())

    def __getattr__(self, item):
        # Make sure to return a new instance of ViscousCreep
        return copy(self._dir[item])
