"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const actions_secret_1 = require("@cdktf/provider-github/lib/actions-secret");
const constructs_1 = require("constructs");
const yaml_file_1 = require("../file/yaml-file");
const snakecase_1 = require("../private/snakecase");
class Workflow extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.jobs = [];
        this.node.addValidation({
            validate: () => {
                if (this.jobs.length === 0) {
                    return [
                        'No jobs were added to the workflow. At least one job is required.',
                    ];
                }
                this.buildWorkflow();
                return [];
            },
        });
        this.repoName = props.repoName;
        this.workflowPath = props.workflowPath ?? '.github/workflows/deploy.yml';
        if (!this.workflowPath.endsWith('.yml') && !this.workflowPath.endsWith('.yaml')) {
            throw new Error('workflow file is expected to be a yaml file');
        }
        if (!this.workflowPath.includes('.github/workflows/')) {
            throw new Error('workflow files must be stored in the \'.github/workflows\' directory of your repository');
        }
        this.workflowFile = new yaml_file_1.YamlFile(this.workflowPath);
        this.workflowName = props.workflowName ?? 'deploy';
        this.workflowTriggers = props.workflowTriggers ?? {
            push: { branches: ['main'] },
            workflowDispatch: {},
        };
        if (props.jobs) {
            for (const job of props.jobs) {
                this.addJob(job);
            }
        }
    }
    addJob(job) {
        this.jobs.push(job);
        if (job.secrets.length != 0) {
            for (const secret of job.secrets) {
                this.createSecret(secret.secretName, secret.secretValue);
            }
        }
    }
    createSecret(name, value) {
        new actions_secret_1.ActionsSecret(this, `${name}-secret`, {
            repository: this.repoName,
            secretName: name,
            plaintextValue: value,
        });
    }
    validateJobs() {
        const jobIds = this.jobs.map((job) => job.id);
        const duplicateJobIds = jobIds.filter((id, index) => jobIds.indexOf(id) !== index);
        if (duplicateJobIds.length > 0) {
            throw new Error(`Duplicate job ids found: ${duplicateJobIds.join(', ')}`);
        }
        const jobNames = this.jobs
            .map((job) => job.name)
            .filter((name) => name !== undefined);
        const duplicateJobNames = jobNames.filter((name, index) => jobNames.indexOf(name) !== index);
        if (duplicateJobNames.length > 0) {
            throw new Error(`Duplicate job names found: ${duplicateJobNames.join(', ')}`);
        }
        for (const job of this.jobs) {
            if (job.dependsOn.length > 0) {
                for (const need of job.dependsOn) {
                    if (!jobIds.includes(need)) {
                        throw new Error(`Job '${job.id}' needs job '${need}', but '${need}' is not defined`);
                    }
                }
            }
        }
    }
    jobsToRecords() {
        const j = {};
        this.jobs.forEach((job) => {
            j[job.id] = job.toObject();
        });
        return j;
    }
    buildWorkflow() {
        var dirname = path.dirname(this.workflowPath);
        if (!fs.existsSync(this.workflowPath)) {
            fs.mkdirSync(dirname, { recursive: true });
        }
        this.validateJobs();
        const file = new yaml_file_1.YamlFile(this.workflowPath);
        file.update({
            name: this.workflowName,
            on: snakecase_1.SnakeCaseKeys(this.workflowTriggers, '_'),
            jobs: this.jobsToRecords(),
        });
        file.writeFile();
    }
}
exports.Workflow = Workflow;
_a = JSII_RTTI_SYMBOL_1;
Workflow[_a] = { fqn: "@awlsring/cdktf-github-actions.Workflow", version: "0.0.17" };
//# sourceMappingURL=data:application/json;base64,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