"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaToKendra = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const defaults = require("@aws-solutions-constructs/core");
const iam = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
/**
 * @summary The LambdaToKendra class.
 */
class LambdaToKendra extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the LambdaToKendra class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {LambdaToKendraProps} props - user provided props for the construct.
     * @since 1.120.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckVpcProps(props);
        defaults.CheckLambdaProps(props);
        if (props.kendraIndexProps && props.existingKendraIndexObj) {
            throw new Error('You may not provide both kendraIndexProps and existingKendraIndexObj');
        }
        if (props.deployVpc || props.existingVpc) {
            this.vpc = defaults.buildVpc(scope, {
                defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
                existingVpc: props.existingVpc,
                userVpcProps: props.vpcProps,
                constructVpcProps: {
                    enableDnsHostnames: true,
                    enableDnsSupport: true,
                },
            });
            defaults.AddAwsServiceEndpoint(scope, this.vpc, defaults.ServiceEndpointTypes.KENDRA);
        }
        // Setup the Lambda function
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps,
            vpc: this.vpc,
        });
        this.kendraIndex = defaults.buildKendraIndex(this, id, {
            kendraIndexProps: props.kendraIndexProps,
            existingIndexObj: props.existingKendraIndexObj
        });
        this.kendraDataSources = defaults.AddMultipleKendraDataSources(this, id, this.kendraIndex, props.kendraDataSourcesProps);
        // Update Lambda function IAM policy with correct privileges to Kendra index
        const normalizedPermissions = props.indexPermissions ? defaults.normalizeKendraPermissions(props.indexPermissions) : undefined;
        if (!normalizedPermissions || normalizedPermissions.includes("READ")) {
            // Add policy with query permissions
            this.lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    "kendra:Query",
                    "kendra:Describe*",
                    "kendra:Get*",
                    "kendra:BatchGet*",
                    "kendra:List*",
                    "kendra:Retrieve"
                ],
                resources: [this.kendraIndex.attrArn]
            }));
        }
        if (!normalizedPermissions || normalizedPermissions.includes("SUBMITFEEDBACK")) {
            // Add policy with query permissions
            this.lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    "kendra:SubmitFeedback"
                ],
                resources: [this.kendraIndex.attrArn]
            }));
        }
        if (normalizedPermissions?.includes("WRITE")) {
            // Add policy with query permissions
            this.lambdaFunction.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    "kendra:Associate*",
                    "kendra:BatchPut*",
                    "kendra:Clear",
                    "kendra:Create*",
                    "kendra:Delete*",
                    "kendra:Disassociate*",
                    "kendra:Put*",
                    "kendra:Update*",
                    "kendra:Start*",
                    "kendra:Submit*",
                    "kendra:Stop*",
                    "kendra:TagResource",
                    "kendra:UntagResource"
                ],
                resources: [this.kendraIndex.attrArn]
            }));
        }
        // Configure environment variables
        const indexIdEnvironmentVariableName = props.indexIdEnvironmentVariableName || 'KENDRA_INDEX_ID';
        this.lambdaFunction.addEnvironment(indexIdEnvironmentVariableName, this.kendraIndex.attrId);
    }
}
exports.LambdaToKendra = LambdaToKendra;
_a = JSII_RTTI_SYMBOL_1;
LambdaToKendra[_a] = { fqn: "@aws-solutions-constructs/aws-lambda-kendra.LambdaToKendra", version: "2.92.0" };
//# sourceMappingURL=data:application/json;base64,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