"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomResourceProvider = exports.CustomResourceProviderRuntime = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const cxapi = require("@aws-cdk/cx-api");
const asset_staging_1 = require("../asset-staging");
const assets_1 = require("../assets");
const cfn_resource_1 = require("../cfn-resource");
const duration_1 = require("../duration");
const size_1 = require("../size");
const stack_1 = require("../stack");
const token_1 = require("../token");
const ENTRYPOINT_FILENAME = '__entrypoint__';
const ENTRYPOINT_NODEJS_SOURCE = path.join(__dirname, 'nodejs-entrypoint.js');
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const construct_compat_1 = require("../construct-compat");
/**
 * The lambda runtime to use for the resource provider. This also indicates
 * which language is used for the handler.
 */
var CustomResourceProviderRuntime;
(function (CustomResourceProviderRuntime) {
    /**
     * Node.js 12.x
     *
     * @deprecated Use {@link NODEJS_14_X}
     */
    CustomResourceProviderRuntime["NODEJS_12"] = "deprecated_nodejs12.x";
    /**
     * Node.js 12.x
     */
    CustomResourceProviderRuntime["NODEJS_12_X"] = "nodejs12.x";
    /**
     * Node.js 14.x
     */
    CustomResourceProviderRuntime["NODEJS_14_X"] = "nodejs14.x";
})(CustomResourceProviderRuntime = exports.CustomResourceProviderRuntime || (exports.CustomResourceProviderRuntime = {}));
/**
 * An AWS-Lambda backed custom resource provider, for CDK Construct Library constructs
 *
 * This is a provider for `CustomResource` constructs, backed by an AWS Lambda
 * Function. It only supports NodeJS runtimes.
 *
 * **This is not a generic custom resource provider class**. It is specifically
 * intended to be used only by constructs in the AWS CDK Construct Library, and
 * only exists here because of reverse dependency issues (for example, it cannot
 * use `iam.PolicyStatement` objects, since the `iam` library already depends on
 * the CDK `core` library and we cannot have cyclic dependencies).
 *
 * If you are not writing constructs for the AWS Construct Library, you should
 * use the `Provider` class in the `custom-resources` module instead, which has
 * a better API and supports all Lambda runtimes, not just Node.
 *
 * N.B.: When you are writing Custom Resource Providers, there are a number of
 * lifecycle events you have to pay attention to. These are documented in the
 * README of the `custom-resources` module. Be sure to give the documentation
 * in that module a read, regardless of whether you end up using the Provider
 * class in there or this one.
 */
class CustomResourceProvider extends construct_compat_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_core_CustomResourceProviderProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CustomResourceProvider);
            }
            throw error;
        }
        const stack = stack_1.Stack.of(scope);
        // copy the entry point to the code directory
        fs.copyFileSync(ENTRYPOINT_NODEJS_SOURCE, path.join(props.codeDirectory, `${ENTRYPOINT_FILENAME}.js`));
        // verify we have an index file there
        if (!fs.existsSync(path.join(props.codeDirectory, 'index.js'))) {
            throw new Error(`cannot find ${props.codeDirectory}/index.js`);
        }
        const staging = new asset_staging_1.AssetStaging(this, 'Staging', {
            sourcePath: props.codeDirectory,
        });
        const assetFileName = staging.relativeStagedPath(stack);
        const asset = stack.synthesizer.addFileAsset({
            fileName: assetFileName,
            sourceHash: staging.assetHash,
            packaging: assets_1.FileAssetPackaging.ZIP_DIRECTORY,
        });
        const policies = !props.policyStatements ? undefined : [
            {
                PolicyName: 'Inline',
                PolicyDocument: {
                    Version: '2012-10-17',
                    Statement: props.policyStatements,
                },
            },
        ];
        const role = new cfn_resource_1.CfnResource(this, 'Role', {
            type: 'AWS::IAM::Role',
            properties: {
                AssumeRolePolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [{ Action: 'sts:AssumeRole', Effect: 'Allow', Principal: { Service: 'lambda.amazonaws.com' } }],
                },
                ManagedPolicyArns: [
                    { 'Fn::Sub': 'arn:${AWS::Partition}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole' },
                ],
                Policies: policies,
            },
        });
        this.roleArn = token_1.Token.asString(role.getAtt('Arn'));
        const timeout = props.timeout ?? duration_1.Duration.minutes(15);
        const memory = props.memorySize ?? size_1.Size.mebibytes(128);
        const handler = new cfn_resource_1.CfnResource(this, 'Handler', {
            type: 'AWS::Lambda::Function',
            properties: {
                Code: {
                    S3Bucket: asset.bucketName,
                    S3Key: asset.objectKey,
                },
                Timeout: timeout.toSeconds(),
                MemorySize: memory.toMebibytes(),
                Handler: `${ENTRYPOINT_FILENAME}.handler`,
                Role: role.getAtt('Arn'),
                Runtime: customResourceProviderRuntimeToString(props.runtime),
                Environment: this.renderEnvironmentVariables(props.environment),
                Description: props.description ?? undefined,
            },
        });
        handler.addDependsOn(role);
        if (this.node.tryGetContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT)) {
            handler.addMetadata(cxapi.ASSET_RESOURCE_METADATA_PATH_KEY, assetFileName);
            handler.addMetadata(cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY, 'Code');
        }
        this.serviceToken = token_1.Token.asString(handler.getAtt('Arn'));
    }
    /**
     * Returns a stack-level singleton ARN (service token) for the custom resource
     * provider.
     *
     * @param scope Construct scope
     * @param uniqueid A globally unique id that will be used for the stack-level
     * construct.
     * @param props Provider properties which will only be applied when the
     * provider is first created.
     * @returns the service token of the custom resource provider, which should be
     * used when defining a `CustomResource`.
     */
    static getOrCreate(scope, uniqueid, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_core_CustomResourceProviderProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.getOrCreate);
            }
            throw error;
        }
        return this.getOrCreateProvider(scope, uniqueid, props).serviceToken;
    }
    /**
     * Returns a stack-level singleton for the custom resource provider.
     *
     * @param scope Construct scope
     * @param uniqueid A globally unique id that will be used for the stack-level
     * construct.
     * @param props Provider properties which will only be applied when the
     * provider is first created.
     * @returns the service token of the custom resource provider, which should be
     * used when defining a `CustomResource`.
     */
    static getOrCreateProvider(scope, uniqueid, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_core_CustomResourceProviderProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.getOrCreateProvider);
            }
            throw error;
        }
        const id = `${uniqueid}CustomResourceProvider`;
        const stack = stack_1.Stack.of(scope);
        const provider = stack.node.tryFindChild(id)
            ?? new CustomResourceProvider(stack, id, props);
        return provider;
    }
    renderEnvironmentVariables(env) {
        if (!env || Object.keys(env).length === 0) {
            return undefined;
        }
        // Sort environment so the hash of the function used to create
        // `currentVersion` is not affected by key order (this is how lambda does
        // it)
        const variables = {};
        const keys = Object.keys(env).sort();
        for (const key of keys) {
            variables[key] = env[key];
        }
        return { Variables: variables };
    }
}
exports.CustomResourceProvider = CustomResourceProvider;
_a = JSII_RTTI_SYMBOL_1;
CustomResourceProvider[_a] = { fqn: "@aws-cdk/core.CustomResourceProvider", version: "1.163.2" };
function customResourceProviderRuntimeToString(x) {
    switch (x) {
        case CustomResourceProviderRuntime.NODEJS_12:
        case CustomResourceProviderRuntime.NODEJS_12_X:
            return 'nodejs12.x';
        case CustomResourceProviderRuntime.NODEJS_14_X:
            return 'nodejs14.x';
    }
}
//# sourceMappingURL=data:application/json;base64,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