"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3ToLambda = void 0;
const core_1 = require("@aws-cdk/core");
const defaults = require("@aws-solutions-constructs/core");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
class S3ToLambda extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the S3ToLambda class.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        let bucket;
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        if (!props.existingBucketObj) {
            [this.s3Bucket, this.s3LoggingBucket] = defaults.buildS3Bucket(this, {
                bucketProps: props.bucketProps
            });
            bucket = this.s3Bucket;
        }
        else {
            bucket = props.existingBucketObj;
        }
        // Create S3 trigger to invoke lambda function
        this.lambdaFunction.addEventSource(new aws_lambda_event_sources_1.S3EventSource(bucket, defaults.S3EventSourceProps(props.s3EventSourceProps)));
        this.addCfnNagSuppress();
    }
    addCfnNagSuppress() {
        const root = core_1.Stack.of(this);
        const logicalId = 'BucketNotificationsHandler050a0587b7544547bf325f094a3db834';
        const notificationsResourceHandler = root.node.tryFindChild(logicalId);
        const notificationsResourceHandlerRoleRole = notificationsResourceHandler.node.findChild('Role');
        const notificationsResourceHandlerRolePolicy = notificationsResourceHandlerRoleRole.node.findChild('DefaultPolicy');
        // Extract the CfnFunction from the Function
        const fnResource = notificationsResourceHandler.node.findChild('Resource');
        fnResource.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W58',
                        reason: `Lambda function has permission to write CloudWatch Logs via AWSLambdaBasicExecutionRole policy attached to the lambda role`
                    }]
            }
        };
        // Extract the CfnPolicy from the iam.Policy
        const policyResource = notificationsResourceHandlerRolePolicy.node.findChild('Resource');
        policyResource.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W12',
                        reason: `Bucket resource is '*' due to circular dependency with bucket and role creation at the same time`
                    }]
            }
        };
    }
}
exports.S3ToLambda = S3ToLambda;
//# sourceMappingURL=data:application/json;base64,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