"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayToLambda = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const iam = require("aws-cdk-lib/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const resources = require("@aws-solutions-constructs/resources");
class OpenApiGatewayToLambda extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        CheckOpenapiProps(props);
        const apiDefinitionBucket = props.apiDefinitionBucket ?? props.apiDefinitionAsset?.bucket;
        const apiDefinitionKey = props.apiDefinitionKey ?? props.apiDefinitionAsset?.s3ObjectKey;
        // store a counter to be able to uniquely name lambda functions to avoid naming collisions
        let lambdaCounter = 0;
        this.apiLambdaFunctions = props.apiIntegrations.map(apiIntegration => {
            if (apiIntegration.existingLambdaObj && apiIntegration.lambdaFunctionProps) {
                throw new Error(`Error - Cannot provide both lambdaFunctionProps and existingLambdaObj in an ApiIntegrationfor the api integration with id: ${apiIntegration.id}`);
            }
            if (apiIntegration.existingLambdaObj || apiIntegration.lambdaFunctionProps) {
                return {
                    id: apiIntegration.id,
                    lambdaFunction: defaults.buildLambdaFunction(this, {
                        existingLambdaObj: apiIntegration.existingLambdaObj,
                        lambdaFunctionProps: apiIntegration.lambdaFunctionProps
                    }, `${apiIntegration.id}ApiFunction${lambdaCounter++}`)
                };
            }
            else {
                throw new Error(`One of existingLambdaObj or lambdaFunctionProps must be specified for the api integration with id: ${apiIntegration.id}`);
            }
        });
        // Map each id and lambda function pair to the required format for the template writer custom resource
        const apiIntegrationUris = this.apiLambdaFunctions.map(apiLambdaFunction => {
            // the placeholder string that will be replaced in the OpenAPI Definition
            const uriPlaceholderString = apiLambdaFunction.id;
            // the endpoint URI of the backing lambda function, as defined in the API Gateway extensions for OpenAPI here:
            // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-integration.html
            const uriResolvedValue = `arn:${aws_cdk_lib_1.Aws.PARTITION}:apigateway:${aws_cdk_lib_1.Aws.REGION}:lambda:path/2015-03-31/functions/${apiLambdaFunction.lambdaFunction.functionArn}/invocations`;
            return {
                id: uriPlaceholderString,
                value: uriResolvedValue
            };
        });
        // This custom resource will overwrite the string placeholders in the openapi definition with the resolved values of the lambda URIs
        const apiDefinitionWriter = resources.createTemplateWriterCustomResource(this, 'Api', {
            // CheckAlbProps() has confirmed the existence of these values
            templateBucket: apiDefinitionBucket,
            templateKey: apiDefinitionKey,
            templateValues: apiIntegrationUris,
            timeout: props.internalTransformTimeout ?? cdk.Duration.minutes(1),
            memorySize: props.internalTransformMemorySize ?? 1024
        });
        const specRestApiResponse = defaults.CreateSpecRestApi(this, {
            ...props.apiGatewayProps,
            apiDefinition: apigateway.ApiDefinition.fromBucket(apiDefinitionWriter.s3Bucket, apiDefinitionWriter.s3Key)
        }, props.logGroupProps);
        this.apiGateway = specRestApiResponse.api;
        this.apiGatewayCloudWatchRole = specRestApiResponse.role;
        this.apiGatewayLogGroup = specRestApiResponse.logGroup;
        // Redeploy the API any time the incoming API definition changes (from asset or s3 object)
        this.apiGateway.latestDeployment?.addToLogicalId(apiDefinitionKey);
        this.apiLambdaFunctions.forEach(apiLambdaFunction => {
            // Redeploy the API any time one of the lambda functions changes
            this.apiGateway.latestDeployment?.addToLogicalId(apiLambdaFunction.lambdaFunction.functionArn);
            // Grant APIGW invocation rights for each lambda function
            apiLambdaFunction.lambdaFunction.addPermission('PermitAPIGInvocation', {
                principal: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                sourceArn: this.apiGateway.arnForExecuteApi('*')
            });
        });
    }
}
exports.OpenApiGatewayToLambda = OpenApiGatewayToLambda;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayToLambda[_a] = { fqn: "@aws-solutions-constructs/aws-openapigateway-lambda.OpenApiGatewayToLambda", version: "2.54.1" };
function CheckOpenapiProps(props) {
    let errorMessages = '';
    let errorFound = false;
    if (props.apiDefinitionAsset && (props.apiDefinitionBucket || props.apiDefinitionKey)) {
        errorMessages += 'Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified, but not both\n';
        errorFound = true;
    }
    const apiDefinitionBucket = props.apiDefinitionBucket ?? props.apiDefinitionAsset?.bucket;
    const apiDefinitionKey = props.apiDefinitionKey ?? props.apiDefinitionAsset?.s3ObjectKey;
    if (apiDefinitionBucket === undefined || apiDefinitionKey === undefined) {
        errorMessages += 'Either apiDefinitionBucket/apiDefinitionKey or apiDefinitionAsset must be specified\n';
        errorFound = true;
    }
    if (props.apiIntegrations === undefined || props.apiIntegrations.length < 1) {
        errorMessages += 'At least one ApiIntegration must be specified in the apiIntegrations property\n';
        errorFound = true;
    }
    if (errorFound) {
        throw new Error(errorMessages);
    }
}
//# sourceMappingURL=data:application/json;base64,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