"""
Analytical solutions for fractional calculus problems.

This module provides known analytical solutions for various functions
under fractional differentiation, which can be used to validate
numerical methods.
"""

import numpy as np
from scipy.special import gamma
from typing import Callable, Dict, List, Optional, Tuple, Union
import warnings


class AnalyticalSolutions:
    """Base class for analytical solutions to fractional calculus problems."""

    def __init__(self):
        """Initialize the analytical solutions."""
        pass

    def power_function_derivative(
        self, x: np.ndarray, alpha: float, order: float
    ) -> np.ndarray:
        """
        Analytical fractional derivative of x^alpha.

        For Caputo derivative: D^order(x^alpha) = gamma(alpha + 1) / gamma(alpha - order + 1) * x^(alpha - order)
        For Riemann-Liouville: D^order(x^alpha) = gamma(alpha + 1) / gamma(alpha - order + 1) * x^(alpha - order)

        Args:
            x: Input array
            alpha: Power of the function
            order: Order of fractional derivative

        Returns:
            Analytical fractional derivative
        """
        if alpha <= -1:
            raise ValueError("Power function derivative requires alpha > -1")

        if order < 0:
            raise ValueError("Order must be non-negative")

        # Handle special cases
        if order == 0:
            return x**alpha

        if alpha == 0:
            if order == 1:
                return np.zeros_like(x)
            else:
                return np.zeros_like(x)

        # General formula
        coeff = gamma(alpha + 1) / gamma(alpha - order + 1)
        return coeff * x ** (alpha - order)

    def exponential_derivative(
        self, x: np.ndarray, a: float, order: float
    ) -> np.ndarray:
        """
        Analytical fractional derivative of exp(ax).

        For Caputo derivative: D^order(exp(ax)) = a^order * exp(ax) * E_{1,1-order}(ax^order)
        For small orders, we can use approximation: D^order(exp(ax)) ≈ a^order * exp(ax)

        Args:
            x: Input array
            a: Coefficient in exp(ax)
            order: Order of fractional derivative

        Returns:
            Analytical fractional derivative
        """
        if order < 0:
            raise ValueError("Order must be non-negative")

        if order == 0:
            return np.exp(a * x)

        if order == 1:
            return a * np.exp(a * x)

        # For fractional orders, use approximation
        # This is an approximation that works well for small orders
        return (a**order) * np.exp(a * x)

    def trigonometric_derivative(
        self, x: np.ndarray, func_type: str, omega: float, order: float
    ) -> np.ndarray:
        """
        Analytical fractional derivative of trigonometric functions.

        Args:
            x: Input array
            func_type: 'sin' or 'cos'
            omega: Frequency
            order: Order of fractional derivative

        Returns:
            Analytical fractional derivative
        """
        if order < 0:
            raise ValueError("Order must be non-negative")

        if order == 0:
            if func_type == "sin":
                return np.sin(omega * x)
            elif func_type == "cos":
                return np.cos(omega * x)
            else:
                raise ValueError("func_type must be 'sin' or 'cos'")

        if order == 1:
            if func_type == "sin":
                return omega * np.cos(omega * x)
            elif func_type == "cos":
                return -omega * np.sin(omega * x)

        # For fractional orders, use approximation
        # This is based on the fractional derivative of sin/cos
        phase_shift = order * np.pi / 2

        if func_type == "sin":
            return (omega**order) * np.sin(omega * x + phase_shift)
        elif func_type == "cos":
            return (omega**order) * np.cos(omega * x + phase_shift)
        else:
            raise ValueError("func_type must be 'sin' or 'cos'")

    def constant_function_derivative(
        self, x: np.ndarray, c: float, order: float
    ) -> np.ndarray:
        """
        Analytical fractional derivative of a constant function.

        For Caputo derivative: D^order(c) = c * x^(-order) / gamma(1 - order)
        For Riemann-Liouville: D^order(c) = c * x^(-order) / gamma(1 - order)

        Args:
            x: Input array
            c: Constant value
            order: Order of fractional derivative

        Returns:
            Analytical fractional derivative
        """
        if order < 0:
            raise ValueError("Order must be non-negative")

        if order == 0:
            return c * np.ones_like(x)

        if order == 1:
            return np.zeros_like(x)

        # For fractional orders
        coeff = c / gamma(1 - order)
        return coeff * (x ** (-order))


class PowerFunctionSolutions:
    """Specialized class for power function analytical solutions."""

    def __init__(self):
        """Initialize power function solutions."""
        self.base_solutions = AnalyticalSolutions()

    def get_solution(self, x: np.ndarray, alpha: float, order: float) -> np.ndarray:
        """Get analytical solution for x^alpha."""
        return self.base_solutions.power_function_derivative(x, alpha, order)

    def get_test_cases(self) -> List[Dict]:
        """Get standard test cases for power functions."""
        return [
            {"alpha": 1.0, "order": 0.5, "name": "Linear function, order 0.5"},
            {"alpha": 2.0, "order": 0.5, "name": "Quadratic function, order 0.5"},
            {"alpha": 1.0, "order": 1.0, "name": "Linear function, order 1.0"},
            {"alpha": 2.0, "order": 1.0, "name": "Quadratic function, order 1.0"},
            {"alpha": 0.5, "order": 0.25, "name": "Square root, order 0.25"},
        ]


class ExponentialSolutions:
    """Specialized class for exponential function analytical solutions."""

    def __init__(self):
        """Initialize exponential solutions."""
        self.base_solutions = AnalyticalSolutions()

    def get_solution(self, x: np.ndarray, a: float, order: float) -> np.ndarray:
        """Get analytical solution for exp(ax)."""
        return self.base_solutions.exponential_derivative(x, a, order)

    def get_test_cases(self) -> List[Dict]:
        """Get standard test cases for exponential functions."""
        return [
            {"a": 1.0, "order": 0.5, "name": "exp(x), order 0.5"},
            {"a": -1.0, "order": 0.5, "name": "exp(-x), order 0.5"},
            {"a": 1.0, "order": 1.0, "name": "exp(x), order 1.0"},
            {"a": 2.0, "order": 0.25, "name": "exp(2x), order 0.25"},
        ]


class TrigonometricSolutions:
    """Specialized class for trigonometric function analytical solutions."""

    def __init__(self):
        """Initialize trigonometric solutions."""
        self.base_solutions = AnalyticalSolutions()

    def get_solution(
        self, x: np.ndarray, func_type: str, omega: float, order: float
    ) -> np.ndarray:
        """Get analytical solution for trigonometric functions."""
        return self.base_solutions.trigonometric_derivative(x, func_type, omega, order)

    def get_test_cases(self) -> List[Dict]:
        """Get standard test cases for trigonometric functions."""
        return [
            {
                "func_type": "sin",
                "omega": 1.0,
                "order": 0.5,
                "name": "sin(x), order 0.5",
            },
            {
                "func_type": "cos",
                "omega": 1.0,
                "order": 0.5,
                "name": "cos(x), order 0.5",
            },
            {
                "func_type": "sin",
                "omega": 1.0,
                "order": 1.0,
                "name": "sin(x), order 1.0",
            },
            {
                "func_type": "cos",
                "omega": 1.0,
                "order": 1.0,
                "name": "cos(x), order 1.0",
            },
            {
                "func_type": "sin",
                "omega": 2.0,
                "order": 0.25,
                "name": "sin(2x), order 0.25",
            },
        ]


def get_analytical_solution(func_type: str, x: np.ndarray, **params) -> np.ndarray:
    """
    Get analytical solution for a given function type.

    Args:
        func_type: Type of function ('power', 'exponential', 'trigonometric', 'constant')
        x: Input array
        **params: Function parameters

    Returns:
        Analytical solution
    """
    solutions = AnalyticalSolutions()

    if func_type == "power":
        alpha = params.get("alpha", 1.0)
        order = params.get("order", 0.5)
        return solutions.power_function_derivative(x, alpha, order)

    elif func_type == "exponential":
        a = params.get("a", 1.0)
        order = params.get("order", 0.5)
        return solutions.exponential_derivative(x, a, order)

    elif func_type == "trigonometric":
        trig_type = params.get("func_type", "sin")
        omega = params.get("omega", 1.0)
        order = params.get("order", 0.5)
        return solutions.trigonometric_derivative(x, trig_type, omega, order)

    elif func_type == "constant":
        c = params.get("c", 1.0)
        order = params.get("order", 0.5)
        return solutions.constant_function_derivative(x, c, order)

    else:
        raise ValueError(f"Unknown function type: {func_type}")


def validate_against_analytical(
    numerical_func: Callable,
    analytical_func: Callable,
    test_params: List[Dict],
    x_range: Tuple[float, float] = (0, 1),
    n_points: int = 100,
) -> Dict:
    """
    Validate numerical method against analytical solution.

    Args:
        numerical_func: Function that computes numerical solution
        analytical_func: Function that computes analytical solution
        test_params: List of parameter dictionaries for test cases
        x_range: Range of x values to test
        n_points: Number of points in x array

    Returns:
        Validation results
    """
    from ..utils.error_analysis import ErrorAnalyzer

    x = np.linspace(x_range[0], x_range[1], n_points)
    error_analyzer = ErrorAnalyzer()
    results = []

    for i, params in enumerate(test_params):
        try:
            # Compute numerical solution
            numerical = numerical_func(x, **params)

            # Compute analytical solution
            analytical = analytical_func(x, **params)

            # Compute errors
            errors = error_analyzer.compute_all_errors(numerical, analytical)

            results.append(
                {
                    "test_case": i,
                    "params": params,
                    "success": True,
                    "errors": errors,
                    "max_error": errors["linf"],
                    "mean_error": errors["l2"],
                }
            )

        except Exception as e:
            results.append(
                {
                    "test_case": i,
                    "params": params,
                    "success": False,
                    "error": str(e),
                    "errors": None,
                }
            )

    # Compute summary statistics
    successful_tests = [r for r in results if r["success"]]

    if successful_tests:
        max_errors = [r["max_error"] for r in successful_tests]
        mean_errors = [r["mean_error"] for r in successful_tests]

        summary = {
            "total_tests": len(test_params),
            "successful_tests": len(successful_tests),
            "success_rate": len(successful_tests) / len(test_params),
            "max_error_overall": max(max_errors),
            "mean_error_overall": np.mean(mean_errors),
            "min_error_overall": min(max_errors),
        }
    else:
        summary = {
            "total_tests": len(test_params),
            "successful_tests": 0,
            "success_rate": 0.0,
            "max_error_overall": np.inf,
            "mean_error_overall": np.inf,
            "min_error_overall": np.inf,
        }

    return {
        "results": results,
        "summary": summary,
        "x_range": x_range,
        "n_points": n_points,
    }
