import pandas as pd
import numpy as np

"""
raw에서 받아와서 가공한 데이터를 가지고 있는, 
ppd의 dh.df를 ppd를 사용하는 다른 클래스를 위해서, 
칼럼들을 정리하는 모듈
"""
def convert_as_core_ppd_df(
    original_df: pd.DataFrame
    ) -> pd.DataFrame:
    """
    Returns:
    DataFrame:
        index: ['일자', '종목코드']
        columns: [
        '일자', '종목코드', '종목명', '마켓구분', 
        '변동률', '변동률_장후', '시가총액', '상장주식수', 
        '시가', '고가', '저가', '종가', '기준가', '전일대비', 
        '거래량', '거래대금',
        '거래량양수', '연속거래일수', 
        '종가상한가여부', '종가하한가여부', '시가상한가여부', '시가하한가여부', '종가상하한가', '시가상하한가' 
        ]
    """
    columns = [
        '일자', '종목코드', '종목명', '마켓구분', 
        '변동률', '변동률_장후', '시가총액', '상장주식수', 
        '시가', '고가', '저가', '종가', '기준가', '전일대비', 
        '거래량', '거래대금',
        '거래량양수', '연속거래일수', 
        '종가상한가여부', '종가하한가여부', '시가상한가여부', '시가하한가여부', '종가상하한가', '시가상하한가' 
        ]
    return convert_as_ppd_df(original_df)[columns]

def convert_as_ppd_df(
    original_df: pd.DataFrame,
) -> pd.DataFrame:
    """
    Params:
    original_df: pd.DataFrame - 원본 DataFrame
        index: ['일자', '종목코드']
        columns: [
        '일자', '종목코드', '표준코드', '종목명', '마켓구분', '관리구분', '종가', '변동코드', '전일대비',
        '변동률', '시가', '고가', '저가', '거래량', '거래대금', '시가총액', '상장주식수', '시장ID', '기준가',
        '종가_nxt', '시가_nxt', '고가_nxt', '저가_nxt', '거래량_nxt', '거래대금_nxt',
        '거래량_krx', '거래대금_krx', '전일종가', '익일기준가', '금일가격조정멀티플', '누적가격조정멀티플',
        '수정시가', '수정고가', '수정저가', '수정종가', '수정기준가', '수정전일대비', '수정거래량', '거래량양수',
        '연속거래그룹', '연속거래일수', '상한가', '하한가', '종가상한가여부', '종가하한가여부', '시가상한가여부',
        '시가하한가여부', '종가상하한가', '시가상하한가'
        ]

    Returns:
        pd.DataFrame:
        index: ['일자', '종목코드']
        columns: [
        '일자', '종목코드', '종목명', '마켓구분', # 그대로 유지
        '변동률', '시가총액', '상장주식수', # 그대로 유지
        '거래량', # 거래량_krx, 거래량_nxt를 합산한 이전 값은 버리고 수정거래량으로 대치
        '거래대금', # 합산값으로 그대로 유지
        '시가', '고가', '저가', '종가', '기준가', '전일대비', # 각각 '수정'xx 에서 '수정'을 제거
        '시가_krx', '고가_krx', '저가_krx', '종가_krx', '기준가_krx',# 원래 시가, 고가, 저가, 종가
        '거래량_krx', '거래대금_krx', # 그대로 유지
        '시가_nxt', '고가_nxt', '저가_nxt', '종가_nxt', '거래량_nxt', '거래대금_nxt',
        '변동률_krx', '변동률_nxt', '변동률_장후',# 추가
        '전일종가_krx', '익일기준가_krx', # '전일종가', '익일기준가' 사용
        '금일가격조정멀티플', '누적가격조정멀티플', # 그대로 유지
        '거래량양수', '연속거래일수', # 그대로 유지
        '상한가_krx', '하한가_krx', # '상한가', '하한가' 사용
        '종가상한가여부', '종가하한가여부', '시가상한가여부', '시가하한가여부', '종가상하한가', '시가상하한가' # 그대로 유지
        ]
    """
    df = original_df.copy()
    
    # 0. 최종칼럼 리스트 정리
    columns = [
        '일자', '종목코드', '종목명', '마켓구분', # 그대로 유지
        '변동률', '시가총액', '상장주식수', # 그대로 유지
        '거래량', # 거래량_krx, 거래량_nxt를 합산한 이전 값은 버리고 수정거래량으로 대치
        '거래대금', # 합산값으로 그대로 유지
        '시가', '고가', '저가', '종가', '기준가', '전일대비', # 각각 '수정'xx 에서 '수정'을 제거
        '시가_krx', '고가_krx', '저가_krx', '종가_krx', '기준가_krx',# 원래 시가, 고가, 저가, 종가
        '거래량_krx', '거래대금_krx', # 그대로 유지
        '변동률_krx', '변동률_nxt', '변동률_장후',# 추가
        '시가_nxt', '고가_nxt', '저가_nxt', '종가_nxt', '거래량_nxt', '거래대금_nxt',
        '전일종가_krx', '익일기준가_krx', # '전일종가', '익일기준가' 사용
        '금일가격조정멀티플', '누적가격조정멀티플', # 그대로 유지
        '거래량양수', '연속거래일수', # 그대로 유지
        '상한가_krx', '하한가_krx', # '상한가', '하한가' 사용
        '종가상한가여부', '종가하한가여부', '시가상한가여부', '시가하한가여부', '종가상하한가', '시가상하한가', # 그대로 유지
    ]

    # 1. 기존 시가, 고가, 저가, 종가, 기준가, 전일대비, 전일종가, 익일기준가, 상한가, 하한가 → *_krx로 먼저 변경
    df = df.rename(columns={
        '시가': '시가_krx',
        '고가': '고가_krx',
        '저가': '저가_krx',
        '종가': '종가_krx',
        '기준가': '기준가_krx',
        '전일대비': '전일대비_krx',
        '전일종가': '전일종가_krx',
        '익일기준가': '익일기준가_krx',
        '상한가': '상한가_krx',
        '하한가': '하한가_krx',
    })
    
    # 2. 수정시가 등에서 수정 생략
    df = df.rename(columns={
        '수정시가': '시가',
        '수정고가': '고가',
        '수정저가': '저가',
        '수정종가': '종가',
        '수정기준가': '기준가',
        '수정전일대비': '전일대비',
    })
    df['거래량'] = df['수정거래량']
    
    # 3. 변동률 추가
    df['변동률_krx'] = df['변동률']
    df['변동률_nxt'] = np.where(df['종가_nxt'] == 0, 0, df['종가_nxt'] / df['기준가_krx'] - 1)
    df['변동률_장후'] = np.where(
        df['종가_nxt'] == 0, 0, df['종가_nxt'] / df['종가_krx'] - 1
    )
    
    return df[columns].sort_index()