"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const kms_generated_1 = require("./kms.generated");
const REQUIRED_ALIAS_PREFIX = 'alias/';
const DISALLOWED_PREFIX = REQUIRED_ALIAS_PREFIX + 'aws/';
class AliasBase extends core_1.Resource {
}
/**
 * Defines a display name for a customer master key (CMK) in AWS Key Management
 * Service (AWS KMS). Using an alias to refer to a key can help you simplify key
 * management. For example, when rotating keys, you can just update the alias
 * mapping instead of tracking and changing key IDs. For more information, see
 * Working with Aliases in the AWS Key Management Service Developer Guide.
 *
 * You can also add an alias for a key by calling `key.addAlias(alias)`.
 *
 * @resource AWS::KMS::Alias
 */
class Alias extends AliasBase {
    static fromAliasAttributes(scope, id, attrs) {
        // tslint:disable-next-line: class-name
        class _Alias extends AliasBase {
            get aliasName() { return attrs.aliasName; }
            get aliasTargetKey() { return attrs.aliasTargetKey; }
        }
        return new _Alias(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id);
        if (!core_1.Token.isUnresolved(props.aliasName)) {
            if (!props.aliasName.startsWith(REQUIRED_ALIAS_PREFIX)) {
                throw new Error(`Alias must start with the prefix "${REQUIRED_ALIAS_PREFIX}": ${props.aliasName}`);
            }
            if (props.aliasName === REQUIRED_ALIAS_PREFIX) {
                throw new Error(`Alias must include a value after "${REQUIRED_ALIAS_PREFIX}": ${props.aliasName}`);
            }
            if (props.aliasName.startsWith(DISALLOWED_PREFIX)) {
                throw new Error(`Alias cannot start with ${DISALLOWED_PREFIX}: ${props.aliasName}`);
            }
            if (!props.aliasName.match(/^[a-zA-Z0-9:/_-]{1,256}$/)) {
                throw new Error(`Alias name must be between 1 and 256 characters in a-zA-Z0-9:/_-`);
            }
        }
        const resource = new kms_generated_1.CfnAlias(this, 'Resource', {
            aliasName: props.aliasName,
            targetKeyId: props.targetKey.keyArn
        });
        this.aliasName = resource.aliasName;
    }
}
exports.Alias = Alias;
//# sourceMappingURL=data:application/json;base64,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