/* $Id: mpi.c 117 2007-08-01 00:05:53Z dalcinl $ */

/*------------------------------------------------------------------*/

/* inline attribute */
#ifndef MPI4PYINLINE
# if defined(__cplusplus) || (defined(__GNUC__) && !defined(__STRICT_ANSI__))
#   define MPI4PYINLINE inline
# else
#   define MPI4PYINLINE
# endif
#endif

/* attribute recognised by some compilers to avoid 'unused' warnings */
#ifndef MPI4PYUNUSED
# if defined(__GNUC__)
#   if !defined(__cplusplus) || (__GNUC__ > 3 || (__GNUC__ == 3 && __GNUC_MINOR__ >= 4))
#     define MPI4PYUNUSED __attribute__ ((__unused__)) 
#   else
#     define MPI4PYUNUSED
#   endif
# elif defined(__INTEL_COMPILER) || defined(__ICC)
#   define MPI4PYUNUSED __attribute__ ((__unused__)) 
# else
#   define MPI4PYUNUSED 
# endif
#endif

/*------------------------------------------------------------------*/

#include <Python.h>

#if PY_VERSION_HEX < 0x02050000
typedef int Py_ssize_t;
#define PY_SSIZE_T_MAX INT_MAX
#define PY_SSIZE_T_MIN INT_MIN
#endif

#if !defined(Py_RETURN_NONE)
#define Py_RETURN_NONE return Py_INCREF(Py_None), Py_None
#endif

#if !defined(Py_BOOLOBJECT_H) && !defined(PyBool_FromLong)
#define PyBool_FromLong(b) PyInt_FromLong(((long)(b))?1:0)
#endif

#if !defined(PySequence_Fast_ITEMS)
#define PySequence_Fast_ITEMS(sf) \
	(PyList_Check(sf) ? ((PyListObject *)(sf))->ob_item \
			  : ((PyTupleObject *)(sf))->ob_item)
#endif

#if PY_VERSION_HEX < 0x02050000
#define PyErr_WarnEx(category, msg, stack_level) \
        PyErr_Warn(category, msg)
#endif

/*------------------------------------------------------------------*/

/* Below, some stuff from the top of MPICH2 mpicxx.h ...

   There is a name conflict between stdio.h and the MPI C++ binding 
   with respect to the names SEEK_SET, SEEK_CUR, and SEEK_END.  MPI
   wants these in the MPI namespace, but stdio.h will #define these
   to integer values.  #undef'ing these can cause obscure problems
   with other include files (such as iostream), so we instead use
   #error to indicate a fatal error.  Users can either #undef 
   the names before including mpi.h or include mpi.h *before* stdio.h
   or iostream.

   The MPI STANDARD HAS TO BE CHANGED to prevent this nonsense. 
*/

#define MPICH_IGNORE_CXX_SEEK
#define OMPI_IGNORE_CXX_SEEK
#include "libmpi.h"

/*------------------------------------------------------------------*/

#include "config.h"
#include "compat.h"
#include "macros.h"

/*------------------------------------------------------------------*/


/*------------------------------------------------------------------*/
/* Module Exception                                                 */
/*------------------------------------------------------------------*/

static PyObject *PyExc_MPIError = NULL;
static PyObject *PyExc_MPIWarning = NULL;

/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
/* Module Communicators                                             */
/*------------------------------------------------------------------*/

static MPI_Comm PyMPI_COMM_SELF;
static MPI_Comm PyMPI_COMM_WORLD;

/*------------------------------------------------------------------*/


/*------------------------------------------------------------------*/
/* Aint                                                             */
/*------------------------------------------------------------------*/

#define PyMPIAint_Check(o) \
        (PyInt_Check(o) || PyLong_Check(o))
#define PyMPIAint_CheckExact(o) \
        (PyInt_CheckExact(o) || PyLong_CheckExact(o))
#define PyMPIAint_AsAint(o) \
        ((MPI_Aint) PyLong_AsVoidPtr(o))
#define PyMPIAint_AS_AINT(o) \
        ((MPI_Aint) PyLong_AsVoidPtr(o))
#define PyMPIAint_FromAint(a) \
        (PyLong_FromVoidPtr((void *)(a)))

/*------------------------------------------------------------------*/


/*------------------------------------------------------------------*/
/* Offset                                                           */
/*------------------------------------------------------------------*/

#if !HAVE_MPI_DISPLACEMENT_CURRENT
#if !defined (MPI_Offset)
#if HAVE_LONG_LONG
#define MPI_Offset long long
#else
#define MPI_Offset long
#endif /* HAVE_LONG_LONG */
#endif /* !defined (MPI_Offset) */
#endif /* !HAVE_MPI_DISPLACEMENT_CURRENT */


#define PyMPIOffset_Check(o) \
        (PyInt_Check(o) || PyLong_Check(o))
#define PyMPIOffset_CheckExact(o) \
        (PyInt_CheckExact(o) || PyLong_CheckExact(o))
#if HAVE_LONG_LONG
#define PyMPIOffset_AsOffset(o) \
        ((MPI_Offset)PyLong_AsLongLong(o))
#define PyMPIOffset_AS_OFFSET(o) \
        ((MPI_Offset)PyLong_AsLongLong(o))
#define PyMPIOffset_FromOffset(a) \
        (PyLong_FromLongLong((PY_LONG_LONG)(a)))
#else
#define PyMPIOffset_AsOffset(o) \
        ((MPI_Offset)PyLong_AsLong(o))
#define PyMPIOffset_AS_OFFSET(o) \
        ((MPI_Offset)PyLong_AsLong(o))
#define PyMPIOffset_FromOffset(a) \
        (PyLong_FromLong((long)(a)))
#endif /* HAVE_LONG_LONG */

/*------------------------------------------------------------------*/


/*------------------------------------------------------------------*/
/* Python Object -> MPI_Type Converters                             */
/*------------------------------------------------------------------*/

#define MPI4PY_AS_VAL(name, Type)                \
static MPI4PYUNUSED int                          \
PyO2##name(PyObject *obj, MPI_##Type *val)       \
{                                                \
  *val =  PyMPI##Type##_As##Type(obj);           \
  return PyErr_Occurred()?0:1;                   \
}                                                 

#define MPI4PY_AS_PTR(name, Type)                \
static MPI4PYUNUSED int                          \
PyO2##name##P(PyObject *obj, MPI_##Type **ptr)   \
{                                                \
  *ptr =  PyMPI##Type##_As##Type##Ptr(obj);      \
  return ((*ptr == NULL) &&                      \
          PyErr_Occurred()) ? 0 : 1;             \
}                                                 

#define MPI4PY_AS_ALL(name,Type) \
        MPI4PY_AS_VAL(name,Type) \
        MPI4PY_AS_PTR(name,Type)              

/* MPI-1 */

MPI4PY_AS_VAL( Ad , Aint       )

MPI4PY_AS_ALL( Op , Op         )
MPI4PY_AS_ALL( Cm , Comm       )
MPI4PY_AS_ALL( Gr , Group      )
MPI4PY_AS_PTR( St , Status     )
MPI4PY_AS_ALL( Rq , Request    )
MPI4PY_AS_ALL( Dt , Datatype   )
MPI4PY_AS_ALL( Eh , Errhandler )

/* MPI-2 */

MPI4PY_AS_VAL( Of , Offset     )

MPI4PY_AS_ALL( In , Info       )
MPI4PY_AS_ALL( Wn , Win        )
MPI4PY_AS_ALL( Fl , File       )


/*------------------------------------------------------------------*/
/* Python Sequence - > C Array                                      */
/*------------------------------------------------------------------*/

#define MPI4PY_SEQ_AS_ARRAY(item_name, item_t,                       \
                            PyItem_Check, PyItem_AsItem)             \
static MPI4PYUNUSED int                                              \
seq_as_array_##item_name(PyObject *input, int req_size,              \
                         item_t **array, int *count)                 \
{                                                                    \
  PyObject *sequence = NULL;                                         \
  PyObject **items = NULL;                                           \
  item_t *ptr = NULL;                                                \
  Py_ssize_t len = 0;                                                \
  Py_ssize_t i;                                                      \
  /* get a sequence */                                               \
  sequence = PySequence_Fast(input, "sequence expected");            \
  if (sequence == NULL) goto fail;                                   \
  items = PySequence_Fast_ITEMS(sequence);                           \
  /* get sequence length */                                          \
  len = PySequence_Fast_GET_SIZE(sequence);                          \
  if (req_size >= 0 && ((Py_ssize_t)req_size) != len) {              \
    PyErr_Format(PyExc_ValueError,                                   \
                 "sequence length: expected %d, got %d",             \
                 req_size, (int)len); goto fail;                     \
  }                                                                  \
  /* allocate memory */                                              \
  ptr = PyMem_New(item_t, len+1);                                    \
  if (ptr == NULL) {                                                 \
    PyErr_Format(PyExc_MemoryError,                                  \
		 "calling PyMem_New(%s, %d+1)", #item_t, (int)len);  \
    goto fail;                                                       \
  }                                                                  \
  /* fill allocated memory */                                        \
  for (i = 0; i < len; ++i) {                                        \
    PyObject *item = items[i];                                       \
    if (!PyItem_Check(item)) {                                       \
      PyErr_Format(PyExc_TypeError,                                  \
                   "sequence item %d: %s expected, %.200s found",    \
                   (int)i, #item_name, item->ob_type->tp_name);      \
      goto fail;                                                     \
    }                                                                \
    ptr[i] = PyItem_AsItem(item);                                    \
  }                                                                  \
  /* return result */                                                \
  Py_DECREF(sequence);                                               \
  *array = ptr;                                                      \
  *count = Py_SAFE_DOWNCAST(len, Py_ssize_t, int);                   \
  return 1;                                                          \
 fail:                                                               \
  Py_XDECREF(sequence);                                              \
  PyMem_Del(ptr);                                                    \
  return 0;                                                          \
}

MPI4PY_SEQ_AS_ARRAY(integer, int,
		    PyInt_Check, PyInt_AS_LONG)

MPI4PY_SEQ_AS_ARRAY(address, MPI_Aint,
		    PyMPIAint_Check, PyMPIAint_AS_AINT)
     
MPI4PY_SEQ_AS_ARRAY(datatype, MPI_Datatype,
		    PyMPIDatatype_Check, PyMPIDatatype_AS_DATATYPE)
     
MPI4PY_SEQ_AS_ARRAY(status, MPI_Status,
		    PyMPIStatus_Check, PyMPIStatus_AS_STATUS)

MPI4PY_SEQ_AS_ARRAY(request, MPI_Request,
		    PyMPIRequest_Check, PyMPIRequest_AS_REQUEST)

#if !defined(PyAnyString_CheckExact)
#define PyAnyString_CheckExact(s) \
                    (PyString_CheckExact(s) || PyUnicode_CheckExact(s))
#endif

MPI4PY_SEQ_AS_ARRAY(string, char*,
		    PyAnyString_CheckExact, PyString_AsString)


/*------------------------------------------------------------------*/
/* C Array -> Python Sequence                                       */
/*------------------------------------------------------------------*/

#define MPI4PY_SEQ_FROM_ARRAY(sname, PySeq_New, PySeq_SET_ITEM,  \
                              iname, item_t, PyItem_FromItem)    \
static MPI4PYUNUSED PyObject *                                   \
sname##_from_array_##iname(item_t *array, int count)             \
{                                                                \
  int i;                                                         \
  PyObject *sequence = PySeq_New((Py_ssize_t) count);            \
  if (sequence == NULL) goto fail;                               \
  for (i = 0; i < count; i++) {                                  \
    PyObject *item = PyItem_FromItem(array[i]);                  \
    if (item == NULL) goto fail;                                 \
    PySeq_SET_ITEM(sequence, i, item);                           \
  }                                                              \
  return sequence;                                               \
 fail:                                                           \
  Py_XDECREF(sequence);                                          \
  return NULL;                                                   \
}

#define MPI4PY_TUPLE_FROM_ARRAY(iname, item_t, PyItem_FromItem)  \
  MPI4PY_SEQ_FROM_ARRAY(tuple, PyTuple_New, PyTuple_SET_ITEM,    \
                        iname, item_t, PyItem_FromItem)

#define MPI4PY_LIST_FROM_ARRAY(iname, item_t, PyItem_FromItem)   \
  MPI4PY_SEQ_FROM_ARRAY(list, PyList_New, PyList_SET_ITEM,       \
                        iname, item_t, PyItem_FromItem)

MPI4PY_LIST_FROM_ARRAY(int, int, PyInt_FromLong)
MPI4PY_TUPLE_FROM_ARRAY(int, int, PyInt_FromLong)



/*------------------------------------------------------------------*/
/* Init handles                                                     */
/*------------------------------------------------------------------*/
#define MPI4PY_HANDLE_INIT(name, Name, NAME)                  \
static PyObject *                                             \
name##_init(PyObject *self, PyObject *args)                   \
{                                                             \
  MPI_##Name *handle = NULL;                                  \
  PyObject *obj = NULL;                                       \
  if (!PyArg_ParseTuple(args, "O|O", &self, &obj))            \
    return NULL;                                              \
  handle = PyMPI##Name##_As##Name##Ptr(self);                 \
  if (handle == NULL && PyErr_Occurred())                     \
    goto fail;                                                \
  if (obj == NULL || obj == Py_None)                          \
    goto exit;                                                \
  else if (PyMPI##Name##_Check(obj)) {                        \
    *handle = PyMPI##Name##_AS_##NAME(obj);                   \
    goto exit;                                                \
  }                                                           \
  PyErr_Format(PyExc_TypeError,                               \
               "expecting a '%s' object, got %.200s",         \
               #Name, obj->ob_type->tp_name);  goto fail;     \
 exit:                                                        \
  Py_RETURN_NONE;                                             \
 fail:                                                        \
  return NULL;                                                \
}
/* MPI-1 */
MPI4PY_HANDLE_INIT(op         , Op         , OP        )
MPI4PY_HANDLE_INIT(comm       , Comm       , COMM      )
MPI4PY_HANDLE_INIT(group      , Group      , GROUP     )
MPI4PY_HANDLE_INIT(status     , Status     , STATUS    )
MPI4PY_HANDLE_INIT(request    , Request    , REQUEST   )
MPI4PY_HANDLE_INIT(datatype   , Datatype   , DATATYPE  )
MPI4PY_HANDLE_INIT(errhandler , Errhandler , ERRHANDLER)
/* MPI-2 */
MPI4PY_HANDLE_INIT(win        , Win        , WIN       )
MPI4PY_HANDLE_INIT(info       , Info       , INFO      )
MPI4PY_HANDLE_INIT(file       , File       , FILE      )
/*------------------------------------------------------------------*/


/*------------------------------------------------------------------*/
/* Clear handles (forcibly set them to MPI_XXX_NULL)                */
/*------------------------------------------------------------------*/
#define MPI4PY_HANDLE_CLEAR(name, Name, NAME)           \
static PyObject *                                       \
name##_clear(PyObject *self, PyObject *args)            \
{                                                       \
  MPI_##Name *handle = NULL;                            \
  if (!PyArg_ParseTuple(args, "O", &self)) return NULL; \
  handle = PyMPI##Name##_As##Name##Ptr(self);           \
  if (handle == NULL) return NULL;                      \
  *handle = MPI_##NAME##_NULL;                          \
  Py_RETURN_NONE;                                       \
}
/* MPI-1 */
MPI4PY_HANDLE_CLEAR(op         , Op         , OP        )
MPI4PY_HANDLE_CLEAR(comm       , Comm       , COMM      )
MPI4PY_HANDLE_CLEAR(group      , Group      , GROUP     )
/*MPI4PY_HANDLE_CLEAR(status     , Status     , STATUS    )*/
MPI4PY_HANDLE_CLEAR(request    , Request    , REQUEST   )
MPI4PY_HANDLE_CLEAR(datatype   , Datatype   , DATATYPE  )
MPI4PY_HANDLE_CLEAR(errhandler , Errhandler , ERRHANDLER)
/* MPI-2 */
MPI4PY_HANDLE_CLEAR(win        , Win        , WIN       )
MPI4PY_HANDLE_CLEAR(info       , Info       , INFO      )
MPI4PY_HANDLE_CLEAR(file       , File       , FILE      )
/*------------------------------------------------------------------*/



/*------------------------------------------------------------------*/
/* Info                                                             */
/*------------------------------------------------------------------*/

static MPI4PYUNUSED int
PyO2IN(PyObject *obj, MPI_Info *info)
{
  if (obj == Py_None) {
    *info = MPI_INFO_NULL;
    return 1;
  } else {
    *info =  PyMPIInfo_AsInfo(obj);
    return ((*info) == MPI_INFO_NULL && 
	    PyErr_Occurred()) ? 0 : 1;
  }
}

/*------------------------------------------------------------------*/
static PyObject *
info_create(PyObject *self, PyObject *args)
{
  MPI_Info info;

  TRY {
    /* create info */
#if HAVE_MPI_INFO_CREATE
    MPI4PY_CALL( MPI_Info_create(&info) );
#else
    info = MPI_INFO_NULL;
    MPI4PY_RAISE_MPINOIMPL("MPI_INFO");
#endif
    /* return info */
    return PyMPIInfo_FromInfo(info);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
info_free(PyObject *self, PyObject *args) 
{
  MPI_Info *info;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2InP, &info) ) return NULL;

  MPI4PY_CHECK_NULL(*info, MPI_INFO_NULL, MPI_ERR_INFO);

  TRY {
    /* free info */
#if HAVE_MPI_INFO_FREE
    MPI4PY_CALL_FREE( MPI_Info_free(info) );
#else
    *info = MPI_INFO_NULL;
    MPI4PY_RAISE_MPINOIMPL("MPI_INFO");
#endif
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/


#if HAVE_MPI_INFO_CREATE

/*------------------------------------------------------------------*/
static PyObject *
info_dup(PyObject *self, PyObject *args)
{
  MPI_Info info;
  MPI_Info newinfo;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2In, &info) ) return NULL;

  MPI4PY_CHECK_NULL(info, MPI_INFO_NULL, MPI_ERR_INFO);

  TRY {
    /* duplicate info  */
    MPI4PY_CALL( MPI_Info_dup(info, &newinfo) );
    /* return new info */
    return PyMPIInfo_FromInfo(newinfo);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
info_get(PyObject *self, PyObject *args)
{
  MPI_Info info;
  char *key;
  int valuelen = -1;
  char *value = NULL;
  int flag;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&s|i",
			 PyO2In, &info,
			 &key,
			 &valuelen) ) return NULL;

  MPI4PY_CHECK_NULL(info, MPI_INFO_NULL, MPI_ERR_INFO);

  TRY {
    PyObject *ovalue;
    if(valuelen < 0 || valuelen > MPI_MAX_INFO_VAL)
      valuelen = MPI_MAX_INFO_VAL;
    value = PyMem_New(char, valuelen+1);
    if (value == NULL) PY_RAISE(PyExc_MemoryError, "");
    /* get value */
    MPI4PY_CALL( MPI_Info_get(info, key, valuelen, value, &flag) );
    ovalue = flag ? Py_BuildValue("s", value)
                  : (Py_INCREF(Py_None), Py_None);
    PyMem_Del(value);
    /* return */
    return Py_BuildValue("NN", ovalue, PyBool_FromLong((long)flag));
  }
  EXCEPT {
    PyMem_Del(value);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
info_set(PyObject *self, PyObject *args)
{
  MPI_Info info;
  char *key;
  char *value;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&ss",
			 PyO2In, &info,
			 &key,
			 &value) ) return NULL;

  MPI4PY_CHECK_NULL(info, MPI_INFO_NULL, MPI_ERR_INFO);

  TRY {
    /* set value */
    MPI4PY_CALL( MPI_Info_set(info, key, value) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
info_delete(PyObject *self, PyObject *args)
{
  MPI_Info info;
  char *key;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&s",
			 PyO2In, &info,
			 &key) ) return NULL;

  MPI4PY_CHECK_NULL(info, MPI_INFO_NULL, MPI_ERR_INFO);

  TRY {
    /* set value */
    MPI4PY_CALL( MPI_Info_delete(info, key) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
info_get_nkeys(PyObject *self, PyObject *args)
{
  MPI_Info info;
  int nkeys;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2In, &info) ) return NULL;

  MPI4PY_CHECK_NULL(info, MPI_INFO_NULL, MPI_ERR_INFO);

  TRY {
    /* get number of keys */
    MPI4PY_CALL( MPI_Info_get_nkeys(info, &nkeys) );
    /* return number of keys*/
    return PyInt_FromLong((long) nkeys);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
info_get_nthkey(PyObject *self, PyObject *args)
{
  MPI_Info info;
  int n;
  char *key;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&i",
			 PyO2In, &info,
			 &n) ) return NULL;

  MPI4PY_CHECK_NULL(info, MPI_INFO_NULL, MPI_ERR_INFO);

  TRY {
    char buf[MPI_MAX_INFO_KEY] = { '\0' };
    key = buf;
    /* get nth key */
    MPI4PY_CALL( MPI_Info_get_nthkey(info, n, key) );
    /* return key */
    return Py_BuildValue("s", key);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
info_get_vlen(PyObject *self, PyObject *args)
{
  MPI_Info info;
  char *key;
  int valuelen;
  int flag;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&s",
			 PyO2In, &info,
			 &key) ) return NULL;

  MPI4PY_CHECK_NULL(info, MPI_INFO_NULL, MPI_ERR_INFO);

  TRY {
    valuelen = 0;
    /* get value length */
    MPI4PY_CALL( MPI_Info_get_valuelen(info, key, &valuelen, &flag) );
    /* return value length */
    return Py_BuildValue("iN", valuelen, PyBool_FromLong((long)flag));
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/

#else /* !HAVE_MPI_INFO_CREATE */

static PyObject *
info_mpinoimpl(PyObject *self, PyObject *args)
{
  TRY {
    MPI4PY_RAISE_MPINOIMPL("MPI_INFO");
  }
  EXCEPT {
    return NULL;
  }
}

#define info_dup        info_mpinoimpl
#define info_get        info_mpinoimpl
#define info_set	info_mpinoimpl
#define info_delete     info_mpinoimpl
#define info_get_nkeys  info_mpinoimpl
#define info_get_nthkey	info_mpinoimpl
#define info_get_vlen   info_mpinoimpl

#endif /* HAVE_MPI_INFO_CREATE */




/*------------------------------------------------------------------*/
/* Memory                                                           */
/*------------------------------------------------------------------*/

/* XXX for safety, we need a new Python type here */

static PyObject *
PyMPIMemory_FromMemory(void *base, MPI_Aint size) {
  void *ptr = base;
  Py_ssize_t len = Py_SAFE_DOWNCAST(size, MPI_Aint, Py_ssize_t);
  return PyBuffer_FromReadWriteMemory(ptr, len);
}

static int
PyMPIMemory_AsMemory(PyObject *obj, void **base, MPI_Aint *size) {
  void *ptr;
  Py_ssize_t len;
  int res;
  res = PyObject_AsWriteBuffer(obj, &ptr, &len);
  if (res >= 0) {
    if (base) *base = ptr;
    if (size) *size = Py_SAFE_DOWNCAST(len, Py_ssize_t, MPI_Aint);
  }
  return res;
}

/*------------------------------------------------------------------*/
static PyObject *
get_address(PyObject *self, PyObject *args)
{
  PyObject *ob_buf;

  void *location;
  MPI_Aint address;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O",
			 &ob_buf) ) return NULL;

  TRY {
    /* get input and output buffer */
    void *bufptr = NULL;
    Py_ssize_t buflen = 0;
    if (PyObject_AsReadBuffer(ob_buf,
                              (const void **)&bufptr,
                              &buflen) < 0) RAISE;
    location = bufptr;
    /* get address */
#if HAVE_MPI_GET_ADDRESS
    MPI4PY_CALL( MPI_Get_address(location, &address) );
#else
    MPI4PY_CALL( MPI_Address(location, &address) );
#endif
    /* return address */
    return PyMPIAint_FromAint(address);
  }
  EXCEPT {
    return NULL;
  }
  
}
/*------------------------------------------------------------------*/
static PyObject *
alloc_mem(PyObject *self, PyObject *args)
{
  MPI_Aint size;
  MPI_Info info;
  void *baseptr;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O&",
			 PyO2Ad, &size,
			 PyO2IN, &info) ) return NULL;

  TRY {
    /* allocate memory */
#if HAVE_MPI_ALLOC_MEM
    MPI4PY_CALL( MPI_Alloc_mem(size, info, &baseptr) );
#else
    baseptr = NULL; size = 0;
    MPI4PY_RAISE_MPINOIMPL("MPI_ALLOC_MEM");
#endif
    /* return memory */
    return PyMPIMemory_FromMemory(baseptr, size);

  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
free_mem(PyObject *self, PyObject *args)
{
  PyObject *ob_base;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O",
			 &ob_base) ) return NULL;

  TRY {
    void *base;
    if (PyMPIMemory_AsMemory(ob_base, &base, NULL) < 0) RAISE;
    /* free memory */
#if HAVE_MPI_FREE_MEM
    MPI4PY_CALL( MPI_Free_mem(base) );
#else
    MPI4PY_RAISE_MPINOIMPL("MPI_FREE_MEM");
#endif
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
buffer_attach(PyObject *self, PyObject *args)
{  
  PyObject *ob_buffer;
  
  void *buffer = NULL;
  int size = 0;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O",
			 &ob_buffer) ) return NULL;
  TRY {
    void *ptr = NULL;
    MPI_Aint len = 0;
    if (ob_buffer == Py_None) { Py_RETURN_NONE; }
    if (PyMPIMemory_AsMemory(ob_buffer, &ptr, &len) < 0) RAISE;
    buffer = ptr; size = Py_SAFE_DOWNCAST(len, MPI_Aint, int);
    MPI4PY_CALL( MPI_Buffer_attach(buffer, size) );
    Py_RETURN_NONE;
  }  
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
buffer_detach(PyObject *self, PyObject *args)
{
  void *buffer = NULL;
  int size = 0;

  TRY {
    MPI4PY_CALL( MPI_Buffer_detach(&buffer, &size) );
    if (buffer == NULL || size == 0)
      Py_RETURN_NONE;
    else
      return PyMPIMemory_FromMemory(buffer, (MPI_Aint)size);
  }  
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/





/*------------------------------------------------------------------*/

typedef struct status_array_t {
  PyObject *pyseq;
  PyObject **items;
  MPI_Status *array;
  int count;
} status_array_t;

static int
status_array_init(status_array_t *statuses)
{
  statuses->pyseq = NULL;
  statuses->items = NULL;
  statuses->array = NULL;
  statuses->count = 0;
  return 0;
}

static int
status_array_fill(status_array_t *statuses, PyObject *obj)
{
  statuses->pyseq = PySequence_Fast(obj, "sequence expected");
  if (statuses->pyseq == NULL) goto fail;
  statuses->items = PySequence_Fast_ITEMS(statuses->pyseq);
  if(!seq_as_array_status(statuses->pyseq, -1,
                          &statuses->array,
                          &statuses->count)) goto fail;
  return 1;
 fail:
  Py_XDECREF(statuses->pyseq);
  statuses->pyseq = NULL;
  statuses->items = NULL;
  return 0;
}

static int
status_array_sync(status_array_t *statuses)
{
  register int n = statuses->count;
  register PyObject **items = statuses->items;
  register MPI_Status *array = statuses->array;
  while(n > 0) {
    --n; PyMPIStatus_AS_STATUS(items[n]) = array[n];
  }
  return 0;
}

static int
status_array_free(status_array_t *statuses)
{
  Py_XDECREF(statuses->pyseq);
  PyMem_Del(statuses->array);
  return 0;
}

/*------------------------------------------------------------------*/

typedef struct request_array_t {
  PyObject *pyseq;
  PyObject **items;
  MPI_Request *array;
  int count;
} request_array_t;

static int
request_array_init(request_array_t *requests)
{
  requests->pyseq = NULL;
  requests->items = NULL;
  requests->array = NULL;
  requests->count = 0;
  return 0;
}

static int
request_array_fill(request_array_t *requests, PyObject *obj)
{
  requests->pyseq = PySequence_Fast(obj, "sequence expected");
  if (requests->pyseq == NULL) goto fail;
  requests->items = PySequence_Fast_ITEMS(requests->pyseq);
  if(!seq_as_array_request(requests->pyseq, -1,
			   &requests->array,
			   &requests->count)) goto fail;
  return 1;
 fail:
  Py_XDECREF(requests->pyseq);
  requests->pyseq = NULL;
  requests->items = NULL;
  return 0;
}

static int
request_array_sync(request_array_t *requests)
{
  register int n = requests->count;
  register PyObject **items = requests->items;
  register MPI_Request *array = requests->array;
  while(n > 0) {
    --n; PyMPIRequest_AS_REQUEST(items[n]) = array[n];
  }
  return 0;
}

static int
request_array_free(request_array_t *requests)
{
  Py_XDECREF(requests->pyseq);
  PyMem_Del(requests->array);
  return 0;
}

/*------------------------------------------------------------------*/




/*------------------------------------------------------------------*/
/* Status                                                           */
/*------------------------------------------------------------------*/

#if HAVE_MPI_STATUS_IGNORE
#define PyMPI_STATUS_IGNORE MPI_STATUS_IGNORE;
#else
static MPI_Status _PyMPI_STATUS_DUMMY;
#define PyMPI_STATUS_IGNORE (&_PyMPI_STATUS_DUMMY);
#endif

static MPI4PYUNUSED int
PyO2SN(PyObject *obj, MPI_Status **status)
{
  if (obj == Py_None) {
    *status = PyMPI_STATUS_IGNORE;
    return 1;
  } else {
    *status =  PyMPIStatus_AsStatusPtr(obj);
    return ((*status) == NULL && 
	    PyErr_Occurred()) ? 0 : 1;
  }
}

/*------------------------------------------------------------------*/
static PyObject *
status_get_count(PyObject *self, PyObject *args)
{  
  MPI_Status *status;
  MPI_Datatype datatype;
  int count;

  if ( !PyArg_ParseTuple(args, "O&O&",
			 PyO2StP, &status,
			 PyO2Dt, &datatype) ) return NULL;

  MPI4PY_CHECK_NULL(datatype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);
  
  TRY {
    /* get count */
    MPI4PY_CALL( MPI_Get_count(status, datatype, &count) );
    /* return count */
    return PyInt_FromLong((long)count);
  }
  EXCEPT {
    return NULL;
  }
  
}
/*------------------------------------------------------------------*/
static PyObject *
status_get_elems(PyObject *self, PyObject *args)
{
  MPI_Status *status;
  MPI_Datatype datatype;
  int count;

  if ( !PyArg_ParseTuple(args, "O&O&",
			 PyO2StP, &status,
			 PyO2Dt, &datatype) ) return NULL;
  
  MPI4PY_CHECK_NULL(datatype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);

  TRY {
    /* get count */
    MPI4PY_CALL( MPI_Get_elements(status, datatype, &count) );
    /* return count */
    return PyInt_FromLong((long)count);
  }
  EXCEPT {
    return NULL;
  }
  
}
/*------------------------------------------------------------------*/
static PyObject *
test_cancelled(PyObject *self, PyObject *args)
{
  MPI_Status *status;
  int flag;
 
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2StP, &status) ) return NULL;
  
  TRY {
    /* get flag */
    MPI4PY_CALL( MPI_Test_cancelled(status, &flag) );
    /* return flag */
    return PyBool_FromLong((long)flag);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/



/*------------------------------------------------------------------*/
/* Request                                                          */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static PyObject *
request_wait(PyObject *self, PyObject *args)
{
  MPI_Request *request;
  MPI_Status *status;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O&",
			 PyO2RqP, &request,
			 PyO2SN,  &status) ) return NULL;
  
  TRY {
    /* wait */
    MPI4PY_CALL( MPI_Wait(request, status) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
request_test(PyObject *self, PyObject *args)
{
  MPI_Request *request;
  int flag;
  MPI_Status *status;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O&",
			 PyO2RqP, &request,
			 PyO2SN,  &status) ) return NULL;

  TRY {
    /* test */
    MPI4PY_CALL( MPI_Test(request, &flag, status) );
    /* return flag */
    return PyBool_FromLong((long)flag);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
request_free(PyObject *self, PyObject *args)
{
  MPI_Request *request;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2RqP, &request) ) return NULL;
  
  MPI4PY_CHECK_NULL(*request, MPI_REQUEST_NULL, MPI_ERR_REQUEST);

  TRY {
    /* free request */
    MPI4PY_CALL_FREE( MPI_Request_free(request) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
request_get_status(PyObject *self, PyObject *args)
{
  MPI_Request request;
  int flag;
  MPI_Status *status;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O&",
			 PyO2Rq, &request,
			 PyO2SN, &status) ) return NULL;
  
  MPI4PY_CHECK_NULL(request, MPI_REQUEST_NULL, MPI_ERR_REQUEST);

  TRY {
    /* get flag and status */
#if HAVE_MPI_REQUEST_GET_STATUS
    MPI4PY_CALL( MPI_Request_get_status(request, &flag, status) );
    /* return flag */
#else
    flag = 0;
    MPI4PY_RAISE_MPINOIMPL("MPI_REQUEST_GET_STATUS");
#endif
    return PyBool_FromLong((long)flag);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
request_cancel(PyObject *self, PyObject *args)
{
  MPI_Request *request;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&", 
			 PyO2RqP, &request) ) return NULL;

  MPI4PY_CHECK_NULL(*request, MPI_REQUEST_NULL, MPI_ERR_REQUEST);

  TRY {
    /* cancel */
    MPI4PY_CALL( MPI_Cancel(request) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
request_waitany(PyObject *self, PyObject *args)
{
  PyObject *orequests;
  request_array_t requests;

  int count = 0;
  MPI_Request *array_of_requests = NULL;
  int index;
  MPI_Status *status;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OO&",
			 &orequests,
			 PyO2SN, &status) ) return NULL;

  request_array_init(&requests);
  if(!request_array_fill(&requests, orequests)) return NULL;

  count = requests.count;
  array_of_requests = requests.array;

  TRY {
    /* waitany */
    MPI4PY_CALL( MPI_Waitany(count, array_of_requests,
			     &index, status) );
    /* update the completed request */
    request_array_sync(&requests);
    /* deallocate array of requests */
    request_array_free(&requests);
    /* return index */
    return PyInt_FromLong((long)index);
  }
  EXCEPT {
    request_array_free(&requests);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
request_testany(PyObject *self, PyObject *args)
{
  PyObject *orequests;
  request_array_t requests;
  
  int count = 0;
  MPI_Request *array_of_requests = NULL;
  int index; 
  int flag;
  MPI_Status *status;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OO&",
			 &orequests,
			 PyO2SN, &status) ) return NULL;

  request_array_init(&requests);
  if(!request_array_fill(&requests, orequests)) return NULL;

  count = requests.count;
  array_of_requests = requests.array;

  TRY {
    /* waitany */
    MPI4PY_CALL( MPI_Testany(count, array_of_requests,
			     &index, &flag, status) );
    /* update the completed request */
    request_array_sync(&requests);
    /* deallocate array of requests */
    request_array_free(&requests);
    /* return index and flag */
    return Py_BuildValue("NN", PyInt_FromLong((long)index),
			       PyBool_FromLong((long)flag));
  }
  EXCEPT {
    request_array_free(&requests);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
request_waitall(PyObject *self, PyObject *args)
{
  PyObject *orequests;
  PyObject *ostatuses;
  request_array_t requests;
  status_array_t  statuses;
  
  int count = 0;
  MPI_Request *array_of_requests = NULL;
  MPI_Status *array_of_statuses = NULL;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OO",
			 &orequests,
			 &ostatuses) ) return NULL;

  request_array_init(&requests);
  if(!request_array_fill(&requests, orequests)) return NULL;
  
  status_array_init(&statuses);
  if(ostatuses != Py_None && 
     !status_array_fill(&statuses, ostatuses)) {
    request_array_free(&requests); return NULL;
  }

  count = requests.count;
  array_of_requests = requests.array;
  array_of_statuses = statuses.array;

  TRY {
    if ((statuses.count != 0) && (statuses.count != requests.count))
      PY_RAISE(PyExc_ValueError, "incompatible sequence lengths");
    if (requests.count > 0 && statuses.count == 0) {
#if HAVE_MPI_STATUSES_IGNORE
      array_of_statuses = MPI_STATUSES_IGNORE;
#else
      array_of_statuses = PyMem_New(MPI_Status, requests.count);
      if (array_of_statuses == NULL) PY_RAISE(PyExc_MemoryError, "");
#endif
    }
    /* waitall */
    MPI4PY_CALL( MPI_Waitall(count, array_of_requests,
			     array_of_statuses) );
    /* update the completed requests */
    request_array_sync(&requests);
    /* update the returned statuses */
    status_array_sync(&statuses);
    /* deallocate array of requests */
    request_array_free(&requests);
    /* deallocate array of statuses */
#if !HAVE_MPI_STATUSES_IGNORE
    if (array_of_statuses != statuses.array) PyMem_Del(array_of_statuses);
#endif
    status_array_free(&statuses);
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    request_array_free(&requests);
#if !HAVE_MPI_STATUSES_IGNORE
    if (array_of_statuses != statuses.array) PyMem_Del(array_of_statuses);
#endif
    status_array_free(&statuses);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
request_testall(PyObject *self, PyObject *args)
{
  PyObject *orequests;
  PyObject *ostatuses;
  request_array_t requests;
  status_array_t  statuses;
  
  int count = 0;
  MPI_Request *array_of_requests = NULL;
  int flag;
  MPI_Status *array_of_statuses = NULL;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OO",
			 &orequests,
			 &ostatuses) ) return NULL;

  request_array_init(&requests);
  if(!request_array_fill(&requests, orequests)) return NULL;
  
  status_array_init(&statuses);
  if(ostatuses != Py_None && 
     !status_array_fill(&statuses, ostatuses)) {
    request_array_free(&requests); return NULL;
  }

  count = requests.count;
  array_of_requests = requests.array;
  array_of_statuses = statuses.array;
  
  TRY {
    if ((statuses.count != 0) && (statuses.count != requests.count))
      PY_RAISE(PyExc_ValueError, "incompatible sequence lengths");
    if (requests.count > 0 && statuses.count == 0) {
#if HAVE_MPI_STATUSES_IGNORE
      array_of_statuses = MPI_STATUSES_IGNORE;
#else
      array_of_statuses = PyMem_New(MPI_Status, requests.count);
      if (array_of_statuses == NULL) PY_RAISE(PyExc_MemoryError, "");
#endif
    }
    /* testall */
    MPI4PY_CALL( MPI_Testall(count, array_of_requests,
			     &flag, array_of_statuses) );
    /* update the completed requests */
    request_array_sync(&requests);
    /* update the returned statuses */
    status_array_sync(&statuses);
    /* deallocate array of requests */
    request_array_free(&requests);
    /* deallocate array of statuses */
#if !HAVE_MPI_STATUSES_IGNORE
    if (array_of_statuses != statuses.array) PyMem_Del(array_of_statuses);
#endif
    status_array_free(&statuses);
    /* return flag */
    return PyBool_FromLong((long)flag);
  }
  EXCEPT {
    request_array_free(&requests);
#if !HAVE_MPI_STATUSES_IGNORE
    if (array_of_statuses != statuses.array) PyMem_Del(array_of_statuses);
#endif
    status_array_free(&statuses);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
request_waitsome(PyObject *self, PyObject *args)
{
  PyObject *orequests;
  PyObject *ostatuses;
  request_array_t requests;
  status_array_t  statuses;
  
  int incount = 0;
  MPI_Request *array_of_requests = NULL;
  int outcount = 0;
  int *array_of_indices = NULL;
  MPI_Status *array_of_statuses = NULL;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OO",
			 &orequests,
			 &ostatuses) ) return NULL;

  request_array_init(&requests);
  if(!request_array_fill(&requests, orequests)) return NULL;
  
  status_array_init(&statuses);
  if(ostatuses != Py_None && 
     !status_array_fill(&statuses, ostatuses)) {
    request_array_free(&requests); return NULL;
  }

  incount = requests.count;
  array_of_requests = requests.array;
  array_of_indices = PyMem_New(int, incount);
  array_of_statuses = statuses.array;

  TRY {
    int i;
    if ((statuses.count != 0) && (statuses.count != requests.count))
      PY_RAISE(PyExc_ValueError, "incompatible sequence lengths");
    if (requests.count > 0 && statuses.count == 0) {
#if HAVE_MPI_STATUSES_IGNORE
      array_of_statuses = MPI_STATUSES_IGNORE;
#else
      array_of_statuses = PyMem_New(MPI_Status, requests.count);
      if (array_of_statuses == NULL) PY_RAISE(PyExc_MemoryError, "");
#endif
    }
    if (array_of_indices == NULL) PY_RAISE(PyExc_MemoryError, "");
    
    /* call waitsome */
    MPI4PY_CALL( MPI_Waitsome(incount, array_of_requests,
			      &outcount, array_of_indices,
			      array_of_statuses) );
    /* update the completed requests */
    request_array_sync(&requests);
    /* update the returned statuses */
    status_array_sync(&statuses);
    /* deallocate array of requests */
    request_array_free(&requests);
    /* deallocate array of statuses */
#if !HAVE_MPI_STATUSES_IGNORE
    if (array_of_statuses != statuses.array) PyMem_Del(array_of_statuses);
#endif
    status_array_free(&statuses);
    /* build and return result */
    if (outcount == MPI_UNDEFINED) {
      /* deallocate array of indices */
      PyMem_Del(array_of_indices);
      /* return outcount and empty list */
      return Py_BuildValue("i[]", MPI_UNDEFINED);
    }
    else {
      /* fill list of indices */
      PyObject *indices = PyList_New((Py_ssize_t)outcount);
      if (indices == NULL) RAISE;
      for (i = 0; i < outcount; i++) {
	long index = (long) array_of_indices[i];
	PyObject *item = PyInt_FromLong(index);
	if (item == NULL) { Py_DECREF(indices); RAISE; }
	PyList_SET_ITEM(indices, i, item);
      }
      /* deallocate array of indices */
      PyMem_Del(array_of_indices);
      /* return outcount and index list */
      return Py_BuildValue("iN", outcount, indices);
    }
  }
  EXCEPT {
    request_array_free(&requests);
#if !HAVE_MPI_STATUSES_IGNORE
    if (array_of_statuses != statuses.array) PyMem_Del(array_of_statuses);
#endif
    status_array_free(&statuses);
    PyMem_Del(array_of_indices);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
request_testsome(PyObject *self, PyObject *args)
{
  PyObject *orequests;
  PyObject *ostatuses;
  request_array_t requests;
  status_array_t  statuses;
  
  int incount = 0;
  MPI_Request *array_of_requests = NULL;
  int outcount = 0;
  int *array_of_indices = NULL;
  MPI_Status *array_of_statuses = NULL;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OO",
			 &orequests,
			 &ostatuses) ) return NULL;

  request_array_init(&requests);
  if(!request_array_fill(&requests, orequests)) return NULL;
  
  status_array_init(&statuses);
  if(ostatuses != Py_None && 
     !status_array_fill(&statuses, ostatuses)) {
    request_array_free(&requests); return NULL;
  }

  incount = requests.count;
  array_of_requests = requests.array;
  array_of_indices = PyMem_New(int, incount);
  array_of_statuses = statuses.array;

  TRY {
    int i;
    if ((statuses.count != 0) && (statuses.count != requests.count))
      PY_RAISE(PyExc_ValueError, "incompatible sequence lengths");
    if (requests.count > 0 && statuses.count == 0) {
#if HAVE_MPI_STATUSES_IGNORE
      array_of_statuses = MPI_STATUSES_IGNORE;
#else
      array_of_statuses = PyMem_New(MPI_Status, requests.count);
      if (array_of_statuses == NULL) PY_RAISE(PyExc_MemoryError, "");
#endif
    }
    if (array_of_indices == NULL) PY_RAISE(PyExc_MemoryError, "");
    /* call testsome */
    MPI4PY_CALL( MPI_Testsome(incount, array_of_requests,
			      &outcount, array_of_indices,
			      array_of_statuses) );
    /* update the completed requests */
    request_array_sync(&requests);
    /* update the returned statuses */
    status_array_sync(&statuses);
    /* deallocate array of requests */
    request_array_free(&requests);
    /* deallocate array of statuses */
#if !HAVE_MPI_STATUSES_IGNORE
    if (array_of_statuses != statuses.array) PyMem_Del(array_of_statuses);
#endif
    status_array_free(&statuses);
    /* build and return result */
    if (outcount == MPI_UNDEFINED) {
      /* deallocate array of indices */
      PyMem_Del(array_of_indices);
      /* return outcount and empty list */
      return Py_BuildValue("i[]", MPI_UNDEFINED);
    }
    else {
      /* fill list of indices */
      PyObject *indices = PyList_New((Py_ssize_t)outcount);
      if (indices == NULL) RAISE;
      for (i = 0; i < outcount; i++) {
	long index = (long) array_of_indices[i];
	PyObject *item = PyInt_FromLong(index);
	if (item == NULL) { Py_DECREF(indices); RAISE; }
	PyList_SET_ITEM(indices, i, item);
      }
      /* deallocate array of indices */
      PyMem_Del(array_of_indices);
      /* return outcount and index list */
      return Py_BuildValue("iN", outcount, indices);
    }
  }
  EXCEPT {
    request_array_free(&requests);
#if !HAVE_MPI_STATUSES_IGNORE
    if (array_of_statuses != statuses.array) PyMem_Del(array_of_statuses);
#endif
    status_array_free(&statuses);
    PyMem_Del(array_of_indices);
    return NULL;
  }

}
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static PyObject *
request_start(PyObject *self, PyObject *args)
{
  MPI_Request *request;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2RqP, &request) ) return NULL;

  MPI4PY_CHECK_NULL(*request, MPI_REQUEST_NULL, MPI_ERR_REQUEST);

  TRY {
    /* start request */
    MPI4PY_CALL( MPI_Start(request) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
request_startall(PyObject *self, PyObject *args)
{
  PyObject *orequests; 
  request_array_t requests;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O",
			 &orequests) ) return NULL;
  
  request_array_init(&requests);
  if(!request_array_fill(&requests, orequests)) return NULL;

  { int i; for (i = 0; i < requests.count; i++) {
    MPI_Request req = requests.array[i];
    MPI4PY_CHECK_NULL(req, MPI_REQUEST_NULL, MPI_ERR_REQUEST);
  }}

  TRY {
    /* startall */
    MPI4PY_CALL( MPI_Startall(requests.count, requests.array) );
    /* update the started requests */
    request_array_sync(&requests);
    /* deallocate array of requests */
    request_array_free(&requests);
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    request_array_free(&requests);
    return NULL;
  }

}
/*------------------------------------------------------------------*/






/*------------------------------------------------------------------*/
/* Datatype                                                         */
/*------------------------------------------------------------------*/

static MPI4PYUNUSED int
PyO2DN(PyObject *obj, MPI_Datatype *datatype)
{
  if (obj == Py_None) {
    *datatype = MPI_DATATYPE_NULL;
    return 1;
  } else {
    *datatype =  PyMPIDatatype_AsDatatype(obj);
    return ((*datatype) == MPI_DATATYPE_NULL &&
	    PyErr_Occurred()) ? 0 : 1;
  }
}

/*------------------------------------------------------------------*/
static PyObject *
type_get_extent(PyObject *self, PyObject *args)
{  
  MPI_Datatype datatype;
  MPI_Aint lb;
  MPI_Aint extent;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Dt, &datatype) ) return NULL;

  MPI4PY_CHECK_NULL(datatype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);
 
  TRY {
    /* get lb and extent */
#if HAVE_MPI_TYPE_GET_EXTENT
    MPI4PY_CALL( MPI_Type_get_extent(datatype, &lb, &extent) );
#else
    MPI4PY_CALL( MPI_Type_lb(datatype, &lb) );
    MPI4PY_CALL( MPI_Type_extent(datatype, &extent) );
#endif
    /* return lb and extent */
    return Py_BuildValue("NN", PyMPIAint_FromAint(lb),
                               PyMPIAint_FromAint(extent));
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
type_size(PyObject *self, PyObject *args)
{  
  MPI_Datatype datatype;
  int size;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Dt, &datatype) ) return NULL;

  MPI4PY_CHECK_NULL(datatype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);
 
  TRY {
    /* get size */
    MPI4PY_CALL( MPI_Type_size(datatype, &size) );
    /* return size */
    return PyInt_FromLong((long)size);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
type_ex(PyObject *self, PyObject *args)
{  
  MPI_Datatype datatype;
  MPI_Aint extent;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Dt, &datatype) ) return NULL;

  MPI4PY_CHECK_NULL(datatype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);
 
  TRY {
    /* get extent */
#if HAVE_MPI_TYPE_GET_EXTENT
    MPI_Aint lb;
    MPI4PY_CALL( MPI_Type_get_extent(datatype, &lb, &extent) );
#else
    MPI4PY_CALL( MPI_Type_extent(datatype, &extent) );
#endif
    /* return extent */
    return PyMPIAint_FromAint(extent);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
type_lb(PyObject *self, PyObject *args)
{  
  MPI_Datatype datatype;
  MPI_Aint lb;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Dt, &datatype) ) return NULL;

  MPI4PY_CHECK_NULL(datatype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);
 
  TRY {
    /* get lower bound */
#if HAVE_MPI_TYPE_GET_EXTENT
    MPI_Aint extent;
    MPI4PY_CALL( MPI_Type_get_extent(datatype, &lb, &extent) );
#else
    MPI4PY_CALL( MPI_Type_lb(datatype, &lb) );
#endif
    /* return lower bound */
    return PyMPIAint_FromAint(lb);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
type_ub(PyObject *self, PyObject *args)
{  
  MPI_Datatype datatype;
  MPI_Aint ub;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Dt, &datatype) ) return NULL;

  MPI4PY_CHECK_NULL(datatype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);
 
  TRY {
    /* get upper bound */
#if HAVE_MPI_TYPE_GET_EXTENT
    MPI_Aint lb, extent;
    MPI4PY_CALL( MPI_Type_get_extent(datatype, &lb, &extent) );
    ub = lb + extent;
#else
    MPI4PY_CALL( MPI_Type_ub(datatype, &ub) );
#endif
    /* return upper bound */
    return PyMPIAint_FromAint(ub);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
type_dup(PyObject *self, PyObject *args)
{  
  MPI_Datatype oldtype;
  MPI_Datatype newtype;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Dt, &oldtype) ) return NULL;

  MPI4PY_CHECK_NULL(oldtype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);
 
  TRY {
    /* get new datatype */
#if HAVE_MPI_TYPE_DUP
    MPI4PY_CALL( MPI_Type_dup(oldtype, &newtype) );
#else
    MPI4PY_WARN_MPINOIMPL("MPI_TYPE_DUP",
			  "using MPI_TYPE_CONTIGUOUS(1, oldtype, newtype) "
			  "and MPI_TYPE_COMMIT(newtype)");
    MPI4PY_CALL( MPI_Type_contiguous(1, oldtype, &newtype) ||
		 MPI_Type_commit(&newtype) );
#endif
    /* return new datatype */
    return PyMPIDatatype_FromDatatype(newtype);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
type_contiguous(PyObject *self, PyObject *args)
{  
  int count;
  MPI_Datatype oldtype;
  MPI_Datatype newtype;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "iO&",
			 &count,
			 PyO2Dt, &oldtype) ) return NULL;

  MPI4PY_CHECK_NULL(oldtype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);
 
  TRY {
    /* get new dattype */
    MPI4PY_CALL( MPI_Type_contiguous(count, oldtype, &newtype) );
    /* return new datatype */
    return PyMPIDatatype_FromDatatype(newtype);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
type_vector(PyObject *self, PyObject *args)
{  
  int count;
  int blocklength;
  int stride;
  MPI_Datatype oldtype;
  MPI_Datatype newtype;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "iiiO&",
			 &count, 
			 &blocklength, 
			 &stride,
			 PyO2Dt, &oldtype) ) return NULL;

  MPI4PY_CHECK_NULL(oldtype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);
 
  TRY {
    /* get new dattype */
    MPI4PY_CALL( MPI_Type_vector(count, blocklength, stride,
				 oldtype, &newtype) );
    /* return new datatype */
    return PyMPIDatatype_FromDatatype(newtype);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
type_hvector(PyObject *self, PyObject *args)
{  
  int count; 
  int blocklength;
  MPI_Aint stride;
  MPI_Datatype oldtype;
  MPI_Datatype newtype;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "iiO&O&",
			 &count, 
			 &blocklength,
			 PyO2Ad, &stride,
			 PyO2Dt, &oldtype) ) return NULL;

  MPI4PY_CHECK_NULL(oldtype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);
 
  TRY {
    /* get new dattype */
#if HAVE_MPI_TYPE_CREATE_HVECTOR
    MPI4PY_CALL( MPI_Type_create_hvector(count, blocklength, stride,
					 oldtype, &newtype) );
#else
    MPI4PY_CALL( MPI_Type_hvector(count, blocklength, stride,
				  oldtype, &newtype) );
#endif
    /* return new datatype */
    return PyMPIDatatype_FromDatatype(newtype);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
type_indexed(PyObject *self, PyObject *args)
{
  PyObject *oblocks;
  PyObject *odisps;

  int count = 0;
  int *blocklengths = NULL;
  int *displacements = NULL;
  MPI_Datatype oldtype;
  MPI_Datatype newtype;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OOO&",
			 &oblocks, 
			 &odisps,
			 PyO2Dt, &oldtype) ) return NULL;

  MPI4PY_CHECK_NULL(oldtype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);

  TRY {
    /* get arrays */
    if (!PyInt_Check(oblocks)) {
      if (!seq_as_array_integer(oblocks,  -1,  &blocklengths,  &count)) RAISE;
      if (!seq_as_array_integer(odisps, count, &displacements, &count)) RAISE;
    } else {
      int i = 0, blocklen = (int) PyInt_AS_LONG(oblocks);
      if (!seq_as_array_integer(odisps, -1, &displacements, &count)) RAISE;
      blocklengths = PyMem_New(int, count);
      if (blocklengths == NULL) PY_RAISE(PyExc_MemoryError, "");
      for (; i < count; blocklengths[i++] = blocklen);
    }
    /* get new datatype */
    MPI4PY_CALL( MPI_Type_indexed(count, blocklengths, displacements,
				  oldtype, &newtype) );
    PyMem_Del(blocklengths);
    PyMem_Del(displacements);
    /* return new datatype */
    return PyMPIDatatype_FromDatatype(newtype);
  }
  EXCEPT {
    PyMem_Del(blocklengths);
    PyMem_Del(displacements);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
type_indexed_block(PyObject *self, PyObject *args)
{
  PyObject *odisps;

  int count=0;
  int blocklength=0;
#if !HAVE_MPI_TYPE_CREATE_INDEXED_BLOCK
  int *blocklengths = NULL;
#endif
  int *displacements = NULL;
  MPI_Datatype oldtype;
  MPI_Datatype newtype;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "iOO&",
			 &blocklength, 
			 &odisps,
			 PyO2Dt, &oldtype) ) return NULL;

  MPI4PY_CHECK_NULL(oldtype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);

  TRY {
    /* get array of displacements */
    if (!seq_as_array_integer(odisps, -1, &displacements, &count)) RAISE;
    /* get new datatype */
#if HAVE_MPI_TYPE_CREATE_INDEXED_BLOCK
    MPI4PY_CALL( MPI_Type_create_indexed_block(count, blocklength,
					       displacements,
					       oldtype, &newtype) );
#else
    MPI4PY_WARN_MPINOIMPL("MPI_TYPE_CREATE_INDEXED_BLOCK",
			  "using MPI_TYPE_INDEXED(...)");
    blocklengths = PyMem_New(int, count);
    if (blocklengths == NULL) PY_RAISE(PyExc_MemoryError, "");
    { int i = 0; for (; i < count; blocklengths[i++] = blocklength); }
    MPI4PY_CALL( MPI_Type_indexed(count, blocklengths, displacements,
				  oldtype, &newtype) );
    PyMem_Del(blocklengths);
#endif
    PyMem_Del(displacements);
    /* return new datatype */
    return PyMPIDatatype_FromDatatype(newtype);
  }
  EXCEPT {
#if !HAVE_MPI_TYPE_CREATE_INDEXED_BLOCK
    PyMem_Del(blocklengths);
#endif
    PyMem_Del(displacements);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
type_hindexed(PyObject *self, PyObject *args)
{
  PyObject *oblocks;
  PyObject *odisps;

  int count = 0;
  int *blocklengths = NULL;
  MPI_Aint *displacements = NULL;
  MPI_Datatype oldtype;
  MPI_Datatype newtype;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OOO&",
			 &oblocks,
			 &odisps,
			 PyO2Dt, &oldtype) ) return NULL;

  MPI4PY_CHECK_NULL(oldtype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);

  TRY {
    /* get arrays */
    if (!seq_as_array_address(odisps, -1, &displacements, &count)) RAISE;
    if (PyInt_Check(oblocks)) {
      int i, block;
      blocklengths = PyMem_New(int, count);
      if (blocklengths == NULL) PY_RAISE(PyExc_MemoryError, "");
      block = (int) PyInt_AS_LONG(oblocks);
      for (i = 0; i < count; i++) blocklengths[i] = block;
    } else{
      if (!seq_as_array_integer(oblocks, count, &blocklengths,  &count)) RAISE;
    }
    /* get new datatype */
#if HAVE_MPI_TYPE_CREATE_HINDEXED
    MPI4PY_CALL( MPI_Type_create_hindexed(count, blocklengths, displacements,
					  oldtype, &newtype) );
#else
    MPI4PY_CALL( MPI_Type_hindexed(count, blocklengths, displacements,
				   oldtype, &newtype) );
#endif
    PyMem_Del(blocklengths);
    PyMem_Del(displacements);
    /* return new datatype */
    return PyMPIDatatype_FromDatatype(newtype);
  }
  EXCEPT {
    PyMem_Del(blocklengths);
    PyMem_Del(displacements);
    return NULL;
  }

}
/*------------------------------------------------------------------*/

static MPI4PYUNUSED int
PyO2Or(PyObject *obj, int *order)
{
  if (obj == Py_None) {
    *order = MPI_ORDER_C;
  } 
  else if (PyInt_CheckExact(obj)) {
    *order = (int) PyInt_AS_LONG(obj);
  }
  else if (PyString_Check(obj) && PyString_GET_SIZE(obj) >= 1) {
    char *str = PyString_AS_STRING(obj);
    if (str[0] == 'C' || str[0] == 'c') {
      *order = MPI_ORDER_C;
    }
    else if (str[0] == 'F' || str[0] == 'f') {
      *order = MPI_ORDER_FORTRAN;
    }
    else if (str[0] == 'A' || str[0] == 'a') {
      *order = MPI_ORDER_C;
    }
    else {
      *order = MPI_ORDER_C;
      PyErr_SetString(PyExc_ValueError,
		      "order not understood");
      return 0;
    }
  } else {
    if (PyObject_IsTrue(obj))
      *order = MPI_ORDER_FORTRAN;
    else
      *order = MPI_ORDER_C;
    if (PyErr_Occurred()) return 0;
  }
  return 1;
}

static PyObject *
type_subarray(PyObject *self, PyObject *args)
{
  PyObject *osizes;
  PyObject *osubsizes;
  PyObject *ostarts;

  int ndim = 0;
  int *sizes = NULL;
  int *subsizes = NULL;
  int *starts = NULL;
  int order = MPI_ORDER_C;
  MPI_Datatype oldtype;
  MPI_Datatype newtype;


  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OOOO&O&",
			 &osizes, 
			 &osubsizes, 
			 &ostarts,
			 PyO2Or, &order,
			 PyO2Dt, &oldtype) ) return NULL;

  TRY {
    /* get arrays */
    if (!seq_as_array_integer(osizes,      -1, &sizes,    &ndim)) RAISE;
    if (!seq_as_array_integer(osubsizes, ndim, &subsizes, &ndim)) RAISE;
    if (!seq_as_array_integer(ostarts,   ndim, &starts,   &ndim)) RAISE;
    /* get new datatype */
#if HAVE_MPI_TYPE_CREATE_SUBARRAY
    MPI4PY_CALL( MPI_Type_create_subarray(ndim, sizes, subsizes, starts,
					  order, oldtype, &newtype) );
#else
    newtype = MPI_DATATYPE_NULL;
    MPI4PY_RAISE_MPINOIMPL("MPI_TYPE_CREATE_SUBARRAY");
#endif
    PyMem_Del(sizes);
    PyMem_Del(subsizes);
    PyMem_Del(starts);
    /* return new datatype */
    return PyMPIDatatype_FromDatatype(newtype);
  }
  EXCEPT {
    PyMem_Del(sizes);
    PyMem_Del(subsizes);
    PyMem_Del(starts);
    return NULL;
  }

}

static PyObject *
type_darray(PyObject *self, PyObject *args)
{
  PyObject *ogsizes;
  PyObject *odistribs;
  PyObject *odargs;
  PyObject *opsizes;
  
  int size;
  int rank;
  int ndims = 0;
  int *gsizes = NULL;
  int *distribs = NULL;
  int *dargs = NULL;
  int *psizes = NULL;
  int order = MPI_ORDER_C;
  MPI_Datatype oldtype;
  MPI_Datatype newtype;


  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "iiOOOO&O&",
			 &size,
			 &rank,
			 &ogsizes, 
			 &odistribs, 
			 &odargs,
			 &opsizes,
			 PyO2Or, &order,
			 PyO2Dt, &oldtype) ) return NULL;
  
  TRY {
    /* get arrays */
    if (!seq_as_array_integer(ogsizes,      -1, &gsizes,   &ndims)) RAISE;
    if (!seq_as_array_integer(odistribs, ndims, &distribs, &ndims)) RAISE;
    if (!seq_as_array_integer(odargs,    ndims, &dargs,    &ndims)) RAISE;
    if (!seq_as_array_integer(opsizes,   ndims, &psizes,   &ndims)) RAISE;
    /* get new datatype */
#if HAVE_MPI_TYPE_CREATE_DARRAY
    MPI4PY_CALL( MPI_Type_create_darray(size, rank, ndims, 
					gsizes, distribs, dargs, psizes,
					order, oldtype, &newtype) );
#else
    newtype = MPI_DATATYPE_NULL;
    MPI4PY_RAISE_MPINOIMPL("MPI_TYPE_CREATE_DARRAY");
#endif
    PyMem_Del(gsizes);
    PyMem_Del(distribs);
    PyMem_Del(dargs);
    PyMem_Del(psizes);
    /* return new datatype */
    return PyMPIDatatype_FromDatatype(newtype);
  }
  EXCEPT {
    PyMem_Del(gsizes);
    PyMem_Del(distribs);
    PyMem_Del(dargs);
    PyMem_Del(psizes);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
type_struct(PyObject *self, PyObject *args)
{
  PyObject *oblocks;
  PyObject *odisps;
  PyObject *otypes;

  int count = 0;
  int *blocklengths = NULL;
  MPI_Aint *displacements = NULL;
  MPI_Datatype *datatypes = NULL;
  MPI_Datatype newtype;


  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OOO",
			 &oblocks, &odisps, &otypes) ) return NULL;

  TRY {
    /* get arrays */
    if (!seq_as_array_datatype(otypes,     -1, &datatypes,     &count)) RAISE;
    if (!seq_as_array_integer (oblocks, count, &blocklengths,  &count)) RAISE;
    if (!seq_as_array_address (odisps,  count, &displacements, &count)) RAISE;
    /* get new datatype */
#if HAVE_MPI_TYPE_CREATE_STRUCT
    MPI4PY_CALL( MPI_Type_create_struct(count, blocklengths, displacements,
					datatypes, &newtype) );
#else
    MPI4PY_CALL( MPI_Type_struct(count, blocklengths, displacements,
				 datatypes, &newtype) );
#endif
    PyMem_Del(blocklengths);
    PyMem_Del(displacements);
    PyMem_Del(datatypes);
    /* return new datatype */
    return PyMPIDatatype_FromDatatype(newtype);
  }
  EXCEPT {
    PyMem_Del(blocklengths);
    PyMem_Del(displacements);
    PyMem_Del(datatypes);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
type_resized(PyObject *self, PyObject *args)
{  
  MPI_Datatype oldtype;
  MPI_Aint lb;
  MPI_Aint extent;
  MPI_Datatype newtype;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O&O&",
			 PyO2Dt, &oldtype,
			 PyO2Ad, &lb,
			 PyO2Ad, &extent) ) return NULL;

  MPI4PY_CHECK_NULL(oldtype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);

  TRY {
    /* get new datatype */
#if HAVE_MPI_TYPE_CREATE_RESIZED
    MPI4PY_CALL( MPI_Type_create_resized(oldtype, 
					 lb, extent,
					 &newtype) );
#else
    newtype = MPI_DATATYPE_NULL;
    MPI4PY_RAISE_MPINOIMPL("MPI_TYPE_CREATE_RESIZED");
#endif
    /* return new datatype */
    return PyMPIDatatype_FromDatatype(newtype);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
type_true_extent(PyObject *self, PyObject *args)
{  
  MPI_Datatype datatype;
  MPI_Aint lb;
  MPI_Aint extent;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Dt, &datatype) ) return NULL;

  MPI4PY_CHECK_NULL(datatype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);
 
  TRY {
    /* get lower bound and extent */
#if HAVE_MPI_TYPE_GET_TRUE_EXTENT
    MPI4PY_CALL( MPI_Type_get_true_extent(datatype, &lb, &extent) );
#else
    lb = extent = (MPI_Aint) 0;
    MPI4PY_RAISE_MPINOIMPL("MPI_TYPE_GET_TRUE_EXTENT");
#endif
    /* return lower bound and extent */
    return Py_BuildValue("NN", PyMPIAint_FromAint(lb),
			       PyMPIAint_FromAint(extent));
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
type_commit(PyObject *self, PyObject *args)
{  
  MPI_Datatype *datatype;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&", 
			 PyO2DtP, &datatype) ) return NULL;

  MPI4PY_CHECK_NULL(*datatype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);

  TRY {
    /* commit datatype */
    MPI4PY_CALL( MPI_Type_commit(datatype) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
type_free(PyObject *self, PyObject *args)
{  
  MPI_Datatype *datatype;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&", 
			 PyO2DtP, &datatype) ) return NULL;

  MPI4PY_CHECK_NULL(*datatype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);

  TRY {
    /* free datatype */
    MPI4PY_CALL_FREE( MPI_Type_free(datatype) );
    /* return */
    Py_RETURN_NONE;
  }

  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static PyObject *
type_pack(PyObject *self, PyObject *args)
{
  PyObject *ob_ibuf;
  PyObject *ob_obuf;

  void *inbuf=NULL;
  int incount=0;
  MPI_Datatype datatype;
  void *outbuf=NULL;
  int outsize=0;
  int position;
  MPI_Comm comm;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OO&OiO&",
			 &ob_ibuf,
			 PyO2Dt, &datatype,
			 &ob_obuf, 
			 &position,
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(datatype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);
  MPI4PY_CHECK_NULL(comm,     MPI_COMM_NULL,     MPI_ERR_COMM);

  TRY {
    const void *ibufptr;
    Py_ssize_t ibuflen;
    void *obufptr;
    Py_ssize_t obuflen;
    MPI_Aint extent;
    /* get input and output buffer */
    if (PyObject_AsReadBuffer (ob_ibuf, &ibufptr, &ibuflen) < 0) RAISE;
    if (PyObject_AsWriteBuffer(ob_obuf, &obufptr, &obuflen) < 0) RAISE;
    inbuf = (void *) ibufptr;
    incount = Py_SAFE_DOWNCAST(ibuflen, Py_ssize_t, int);
    outbuf = (void *) obufptr;
    outsize = Py_SAFE_DOWNCAST(obuflen, Py_ssize_t, int);
    /* get datatype extent */
    MPI4PY_CALL( MPI_Type_extent(datatype, &extent) );
    if (extent == (MPI_Aint)0)
      PY_RAISE(PyExc_ValueError, "datatype with zero extent");
    incount /= Py_SAFE_DOWNCAST(extent, MPI_Aint, int);
    /* pack */
    MPI4PY_CALL_COMM(comm,
		     MPI_Pack(inbuf, incount, datatype,
			      outbuf, outsize, &position,
			      comm) );
    /* return position */
    return PyInt_FromLong((long)position);
  }
  EXCEPT {
    return NULL;
  }
  
}
/*------------------------------------------------------------------*/
static PyObject *
type_unpack(PyObject *self, PyObject *args)
{
  PyObject *ob_ibuf;
  PyObject *ob_obuf;

  void *inbuf=NULL;
  int insize=0;
  int position;
  void *outbuf=NULL;
  int outcount=0;
  MPI_Datatype datatype;
  MPI_Comm comm;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OiOO&O&",
			 &ob_ibuf, 
			 &position,
			 &ob_obuf,
			 PyO2Dt, &datatype,
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(datatype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);
  MPI4PY_CHECK_NULL(comm,     MPI_COMM_NULL,     MPI_ERR_COMM);

  TRY {
    const void *ibufptr;
    Py_ssize_t ibuflen;
    void *obufptr;
    Py_ssize_t obuflen;
    MPI_Aint extent;
    /* get input and output buffer */
    if (PyObject_AsReadBuffer (ob_ibuf, &ibufptr, &ibuflen) < 0) RAISE;
    if (PyObject_AsWriteBuffer(ob_obuf, &obufptr, &obuflen) < 0) RAISE;
    inbuf = (void *) ibufptr;
    insize = Py_SAFE_DOWNCAST(ibuflen, Py_ssize_t, int);
    outbuf = (void *) obufptr;
    outcount = Py_SAFE_DOWNCAST(obuflen, Py_ssize_t, int);
    /* get datatype extent */
    MPI4PY_CALL( MPI_Type_extent(datatype, &extent) );
    if (extent == (MPI_Aint)0) 
      PY_RAISE(PyExc_ValueError, "datatype with zero extent");
    outcount /= Py_SAFE_DOWNCAST(extent, MPI_Aint, int);
    /* unpack */
    MPI4PY_CALL_COMM(comm,
		     MPI_Unpack(inbuf, insize, &position,
				outbuf, outcount, datatype,
				comm) );
    /* return position */
    return PyInt_FromLong((long)position);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
type_pack_size(PyObject *self, PyObject *args)
{  
  int incount;
  MPI_Datatype datatype;
  MPI_Comm comm;
  int size;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "iO&O&",
			 &incount,
			 PyO2Dt, &datatype,
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(datatype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);
  MPI4PY_CHECK_NULL(comm,     MPI_COMM_NULL,     MPI_ERR_COMM);

  TRY {
    /* get pack size */
    MPI4PY_CALL_COMM(comm,
		     MPI_Pack_size(incount, datatype, comm, &size) );
    /* return pack size */
    return PyInt_FromLong((long)size);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static PyObject *
type_pack_external(PyObject *self, PyObject *args)
{
  PyObject *ob_ibuf;
  PyObject *ob_obuf;

  char *datarep;
  void *inbuf=NULL;
  int incount=0;
  MPI_Datatype datatype;
  void *outbuf=NULL;
  MPI_Aint outsize=0;
  MPI_Aint position;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "sOO&OO&",
			 &datarep,
			 &ob_ibuf,
			 PyO2Dt, &datatype,
			 &ob_obuf, 
			 PyO2Ad, &position) ) return NULL;

  MPI4PY_CHECK_NULL(datatype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);

  TRY {
    const void *ibufptr;
    Py_ssize_t ibuflen;
    void *obufptr;
    Py_ssize_t obuflen;
    MPI_Aint extent;
    /* get input and output buffer */
    if (PyObject_AsReadBuffer (ob_ibuf, &ibufptr, &ibuflen) < 0) RAISE;
    if (PyObject_AsWriteBuffer(ob_obuf, &obufptr, &obuflen) < 0) RAISE;
    inbuf = (void *) ibufptr;
    incount = Py_SAFE_DOWNCAST(ibuflen, Py_ssize_t, int);
    outbuf = (void *) obufptr;
    outsize = Py_SAFE_DOWNCAST(obuflen, Py_ssize_t, MPI_Aint);
    /* get datatype extent */
    MPI4PY_CALL( MPI_Type_extent(datatype, &extent) );
    if (extent == (MPI_Aint)0)
      PY_RAISE(PyExc_ValueError, "datatype with zero extent");
    incount /= Py_SAFE_DOWNCAST(extent, MPI_Aint, int);
    /* pack */
#if HAVE_MPI_PACK_EXTERNAL
    MPI4PY_CALL( MPI_Pack_external(datarep,
				   inbuf, incount, datatype,
				   outbuf, outsize, &position) );
#else
    MPI4PY_RAISE_MPINOIMPL("MPI_PACK_EXTERNAL");
#endif
    /* return position */
    return PyMPIAint_FromAint(position);
  }
  EXCEPT {
    return NULL;
  }
  
}
/*------------------------------------------------------------------*/
static PyObject *
type_unpack_external(PyObject *self, PyObject *args)
{
  PyObject *ob_ibuf;
  PyObject *ob_obuf;

  char *datarep;
  void *inbuf=NULL;
  MPI_Aint insize=0;
  MPI_Aint position;
  void *outbuf=NULL;
  int outcount=0;
  MPI_Datatype datatype;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "sOO&OO&",
			 &datarep,
			 &ob_ibuf, 
			 PyO2Ad, &position,
			 &ob_obuf,
			 PyO2Dt, &datatype) ) return NULL;

  MPI4PY_CHECK_NULL(datatype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);

  TRY {
    const void *ibufptr;
    Py_ssize_t ibuflen;
    void *obufptr;
    Py_ssize_t obuflen;
    MPI_Aint extent;
    /* get input and output buffer */
    if (PyObject_AsReadBuffer (ob_ibuf, &ibufptr, &ibuflen) < 0) RAISE;
    if (PyObject_AsWriteBuffer(ob_obuf, &obufptr, &obuflen) < 0) RAISE;
    inbuf = (void *) ibufptr;
    insize = Py_SAFE_DOWNCAST(ibuflen, Py_ssize_t, int);
    outbuf = (void *) obufptr;
    outcount = Py_SAFE_DOWNCAST(obuflen, Py_ssize_t, int);
    /* get datatype extent */
    MPI4PY_CALL( MPI_Type_extent(datatype, &extent) );
    if (extent == (MPI_Aint)0) 
      PY_RAISE(PyExc_ValueError, "datatype with zero extent");
    outcount /= Py_SAFE_DOWNCAST(extent, MPI_Aint, int);
    /* unpack */
#if HAVE_MPI_UNPACK_EXTERNAL
    MPI4PY_CALL( MPI_Unpack_external(datarep,
				     inbuf, insize, &position,
				     outbuf, outcount, datatype) );
#else
    MPI4PY_RAISE_MPINOIMPL("MPI_UNPACK_EXTERNAL");
#endif
    /* return position */
    return PyMPIAint_FromAint(position);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
type_pack_external_size(PyObject *self, PyObject *args)
{  
  char *datarep;
  int incount;
  MPI_Datatype datatype;
  MPI_Aint size;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "siO&",
			 &datarep,
			 &incount,
			 PyO2Dt, &datatype) ) return NULL;

  MPI4PY_CHECK_NULL(datatype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);

  TRY {
    /* get pack size */
#if HAVE_MPI_PACK_EXTERNAL_SIZE
    MPI4PY_CALL( MPI_Pack_external_size(datarep,
					incount, datatype,
					&size) );
#else
    size = 0;
    MPI4PY_RAISE_MPINOIMPL("MPI_PACK_EXTERNAL_SIZE");
#endif
    /* return pack size */
    return PyMPIAint_FromAint(size);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/







/*------------------------------------------------------------------*/
/* Group                                                            */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static PyObject *
group_size(PyObject *self, PyObject *args)
{  
  MPI_Group group;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Gr, &group) ) return NULL;

  MPI4PY_CHECK_NULL(group, MPI_GROUP_NULL, MPI_ERR_GROUP);
 
  TRY {
    /* get size */
    int size;
    MPI4PY_CALL( MPI_Group_size(group, &size) );
    /* return size */
    return PyInt_FromLong((long)size);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
group_rank(PyObject *self, PyObject *args)
{
  MPI_Group group;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&", 
			 PyO2Gr, &group) ) return NULL;
  
  MPI4PY_CHECK_NULL(group, MPI_GROUP_NULL, MPI_ERR_GROUP);

  TRY {
    /* get rank */
    int rank;
    MPI4PY_CALL( MPI_Group_rank(group,&rank) );
    /* return rank */
    return PyInt_FromLong((long)rank);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
group_transl_rank(PyObject *self, PyObject *args)
{  
  PyObject *input  = NULL;
  PyObject *result = NULL;

  PyObject *s_ranks1 = NULL;
  PyObject *s_ranks2 = NULL;
  
  MPI_Group group1;
  int  n = 0;
  int *ranks1 = NULL;
  MPI_Group group2;
  int *ranks2 = NULL;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&OO&",
			 PyO2Gr, &group1, &input,
			 PyO2Gr, &group2) ) return NULL;

  MPI4PY_CHECK_NULL(group1, MPI_GROUP_NULL, MPI_ERR_GROUP);
  MPI4PY_CHECK_NULL(group2, MPI_GROUP_NULL, MPI_ERR_GROUP);
    
  TRY {

    /* handle integer input */
    if ( PyInt_Check(input) ) {
      int rank1 = (int) PyInt_AS_LONG(input);
      int rank2;
      /* translate ranks */
      MPI4PY_CALL( MPI_Group_translate_ranks(group1, 1, &rank1,
					     group2,    &rank2) );
      return PyInt_FromLong((long)rank2);
    }

    /* handle sequence input */
    s_ranks1 = PySequence_Fast(input,"integer or sequence expected");
    if (s_ranks1 == NULL) return NULL;
    n = PySequence_Fast_GET_SIZE(s_ranks1);

    /* allocate ranks* arrays */
    ranks1 = PyMem_New(int, n);
    ranks2 = PyMem_New(int, n);
    if ( !ranks1 || !ranks2 ) PY_RAISE(PyExc_MemoryError, "");

    /* fill ranks1 array */
    { 
      int i; PyObject **items = PySequence_Fast_ITEMS(s_ranks1);
      for (i = 0; i < n; i++) {
	if (!PyInt_Check(items[i]))
	  PY_RAISE(PyExc_TypeError, 
		   "sequence item: integer expected");
	ranks1[i] = (int) PyInt_AS_LONG(items[i]);
    }}
    
    /*translate ranks */
    MPI4PY_CALL( MPI_Group_translate_ranks(group1, n, ranks1,
					   group2,    ranks2) );
    /* allocate and fill result */
    /* output type tries to match input type */
    if (PyTuple_Check(s_ranks1)) 
      s_ranks2 = PyTuple_New((Py_ssize_t)n);
    else       
      s_ranks2 = PyList_New((Py_ssize_t)n);
    if (s_ranks2 == NULL) RAISE;
    { 
      int i; PyObject **items = PySequence_Fast_ITEMS(s_ranks2);
      for (i = 0; i < n; i++) {
	items[i] = PyInt_FromLong((long)ranks2[i]);
	if (items[i] == NULL) RAISE;
      }
    }
    result = s_ranks2; Py_INCREF(s_ranks2);
    
    /* deallocate objects */
    Py_DECREF(s_ranks1);
    Py_DECREF(s_ranks2);
    PyMem_Del(ranks1);
    PyMem_Del(ranks2);

    /* return result */
    return result;

  }

  EXCEPT {
    Py_XDECREF(s_ranks1);
    Py_XDECREF(s_ranks2);
    PyMem_Del(ranks1);
    PyMem_Del(ranks2);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
group_compare(PyObject *self, PyObject *args)
{
  MPI_Group group1;  
  MPI_Group group2;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O&", 
			 PyO2Gr, &group1,
			 PyO2Gr, &group2) ) return NULL;

  MPI4PY_CHECK_NULL(group1, MPI_GROUP_NULL, MPI_ERR_GROUP);
  MPI4PY_CHECK_NULL(group2, MPI_GROUP_NULL, MPI_ERR_GROUP);

  TRY {

    /* compare groups */
    int result;
    MPI4PY_CALL( MPI_Group_compare(group1,group2,&result) );
    
    /* return result */
    return PyInt_FromLong((long)result);
    
  }
  
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_group(PyObject *self, PyObject *args)
{
  MPI_Comm comm;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&", 
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);

  TRY {

    MPI_Group newgroup;

    /* get group */
    MPI4PY_CALL_COMM(comm,
		     MPI_Comm_group(comm, &newgroup) );
    
    /* return group  */
    return PyMPIGroup_FromGroup(newgroup);
    
  }

  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_remote_group(PyObject *self, PyObject *args)
{
  MPI_Comm comm;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&", 
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);

  TRY {    

    MPI_Group newgroup;

    /* get group */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Comm_remote_group(comm, &newgroup) );
    
    /* return group */
    return PyMPIGroup_FromGroup(newgroup);

  }

  EXCEPT { 
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
PyMPI_GROUP(MPI_Group group)
{
  PyObject *ob = PyMPIGroup_FromGroup(group);
  if (ob != NULL && group == MPI_GROUP_EMPTY)
    PyMPIGroup_Inner(ob)->isref = 1;
  return ob;
}
/*------------------------------------------------------------------*/
static PyObject *
group_union(PyObject *self, PyObject *args)
{
  MPI_Group group1;  
  MPI_Group group2;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O&",
			 PyO2Gr, &group1,
			 PyO2Gr, &group2) ) return NULL;

  MPI4PY_CHECK_NULL(group1, MPI_GROUP_NULL, MPI_ERR_GROUP);
  MPI4PY_CHECK_NULL(group2, MPI_GROUP_NULL, MPI_ERR_GROUP);

  TRY {
    
    MPI_Group newgroup;

    /* get group */
    MPI4PY_CALL( MPI_Group_union(group1, group2, &newgroup) );
    
    /* return group */
    return PyMPI_GROUP(newgroup);

  }

  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
group_intersection(PyObject *self, PyObject *args)
{
  MPI_Group group1;  
  MPI_Group group2;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O&",
			 PyO2Gr, &group1,
			 PyO2Gr, &group2) ) return NULL;


  MPI4PY_CHECK_NULL(group1, MPI_GROUP_NULL, MPI_ERR_GROUP);
  MPI4PY_CHECK_NULL(group2, MPI_GROUP_NULL, MPI_ERR_GROUP);

  TRY {

    MPI_Group newgroup;

    /* get group */
    MPI4PY_CALL( MPI_Group_intersection(group1, group2, &newgroup) );
    
    /* return group */
    return PyMPI_GROUP(newgroup);

  }

  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
group_difference(PyObject *self, PyObject *args)
{
  MPI_Group group1;
  MPI_Group group2;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O&",
			 PyO2Gr, &group1,
			 PyO2Gr, &group2) ) return NULL;

  MPI4PY_CHECK_NULL(group1, MPI_GROUP_NULL, MPI_ERR_GROUP);
  MPI4PY_CHECK_NULL(group2, MPI_GROUP_NULL, MPI_ERR_GROUP);

  TRY {

    MPI_Group newgroup;
  
    /* get group */
    MPI4PY_CALL( MPI_Group_difference(group1, group2, &newgroup) );
    
    /* return group */
    return PyMPI_GROUP(newgroup);

  }

  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
group_incl(PyObject *self, PyObject *args)
{
  PyObject *input  = NULL;
  MPI_Group group;

  PyObject *s_ranks = NULL;
  int       n       = 0;
  int*      ranks   = NULL;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O",
			 PyO2Gr, &group, &input) ) return NULL;
    
  MPI4PY_CHECK_NULL(group, MPI_GROUP_NULL, MPI_ERR_GROUP);

  TRY {
    
    MPI_Group newgroup;
  
    /* check and handle input objects */
    if ( PyInt_Check(input) ) {
      s_ranks = Py_BuildValue("(i)", (int)PyInt_AS_LONG(input));
      if (s_ranks==NULL) return NULL;
      n = 1;
    }
    else {
      s_ranks = PySequence_Fast(input, "integer or sequence expected");
      if (s_ranks==NULL) return NULL;
      n = PySequence_Fast_GET_SIZE(s_ranks);
    }

    /* allocate ranks array */
    ranks = PyMem_New(int, n);
    if (ranks==NULL) PY_RAISE(PyExc_MemoryError, "");

    /* fill ranks array */
    { int i; for (i = 0; i < n; i++) {
      PyObject *item = PySequence_Fast_GET_ITEM(s_ranks,i);
      if ( !PyInt_Check(item) )
	PY_RAISE(PyExc_TypeError, "sequence item: integer expected");
      ranks[i] = (int)PyInt_AS_LONG(item);
    }}

    /* obtain group */
    MPI4PY_CALL( MPI_Group_incl(group, n, ranks, &newgroup) );
    
    /* deallocate objects */
    Py_DECREF(s_ranks);
    PyMem_Del(ranks);

    /* return group */
    return PyMPI_GROUP(newgroup);

  }

  EXCEPT {
    Py_XDECREF(s_ranks);
    PyMem_Del(ranks);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
group_excl(PyObject *self, PyObject *args)
{
  PyObject *input  = NULL;
  MPI_Group group;

  PyObject *s_ranks = NULL;
  int       n       = 0;
  int*      ranks   = NULL;

  MPI_Group newgroup;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O",
			 PyO2Gr, &group, &input) ) return NULL;

  MPI4PY_CHECK_NULL(group, MPI_GROUP_NULL, MPI_ERR_GROUP);
  
  TRY {
    
    /* check and handle input objects */
    if ( PyInt_Check(input) ) {
      s_ranks = Py_BuildValue("(i)", (int)PyInt_AS_LONG(input));
      if (s_ranks==NULL) RAISE;
      n = 1;
    }
    else {
      s_ranks = PySequence_Fast(input,"integer or sequence expected");
      if (s_ranks==NULL) RAISE;
      n = PySequence_Fast_GET_SIZE(s_ranks);
    }

    /* allocate ranks array */
    ranks = PyMem_New(int, n);
    if (ranks==NULL) PY_RAISE(PyExc_MemoryError, "");

    /* fill ranks array */
    { int i; for (i = 0; i < n; i++) {
      PyObject *item = PySequence_Fast_GET_ITEM(s_ranks,i);
      if ( !PyInt_Check(item) )
	PY_RAISE(PyExc_TypeError, 
		 "sequence item: integer expected");
      ranks[i] = (int) PyInt_AS_LONG(item);
    }}

    /* obtain group */
    MPI4PY_CALL( MPI_Group_excl(group, n, ranks, &newgroup) );
    
    /* deallocate objects */
    Py_DECREF(s_ranks);
    PyMem_Del(ranks);
    
    /* return group */
    return PyMPI_GROUP(newgroup);

  }

  EXCEPT {
    Py_XDECREF(s_ranks);
    PyMem_Del(ranks);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
group_range_incl(PyObject *self, PyObject *args)
{
  PyObject *input = NULL;
  MPI_Group group;

  PyObject *s_ranges = NULL;
  PyObject *rseq     = NULL;

  typedef  int range_t[3];
  
  int      n      = 0;
  range_t* ranges = NULL;

  MPI_Group newgroup;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O",
			 PyO2Gr, &group, &input) ) return NULL;
  
  MPI4PY_CHECK_NULL(group, MPI_GROUP_NULL, MPI_ERR_GROUP);
    
  TRY {

    s_ranges = PySequence_Fast(input,"sequence expected");
    if (s_ranges==NULL) RAISE;
    n = PySequence_Fast_GET_SIZE(s_ranges);

    /* allocate ranges array */
    ranges = PyMem_New(range_t, n);
    if (ranges==NULL) PY_RAISE(PyExc_MemoryError, "");
    
    /* fill ranges array */
    { int i; for (i = 0; i < n; i++) {
      PyObject *item = PySequence_Fast_GET_ITEM(s_ranges,i);
      PyObject *rseq = PySequence_Fast(item,
				       "sequence item: sequence expected");
      if (rseq==NULL) RAISE;
      if (PySequence_Fast_GET_SIZE(rseq) != 3)
	PY_RAISE(PyExc_ValueError, 
		 "sequence item: invalid sequence length");
      { int j; for (j = 0; j < 3; j++) {
	PyObject *r = PySequence_Fast_GET_ITEM(rseq,j);
	if ( !PyInt_Check(r) )
	  PY_RAISE(PyExc_TypeError, 
		   "sequence item: sequence item: integer expected");
	ranges[i][j] = (int) PyInt_AS_LONG(r);
      }}
      Py_DECREF(rseq); rseq = NULL;
    }}
    
    /* obtain group */
    MPI4PY_CALL( MPI_Group_range_incl(group,
				      n, ranges,
				      &newgroup) );
    
    /* deallocate objects */
    PyMem_Del(ranges);
    Py_DECREF(s_ranges);

    /* return group */
    return PyMPI_GROUP(newgroup);

  }

  EXCEPT {
    Py_XDECREF(rseq);
    PyMem_Del(ranges);
    Py_XDECREF(s_ranges);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
group_range_excl(PyObject *self, PyObject *args)
{
  PyObject *input = NULL;
  MPI_Group group;

  PyObject *s_ranges = NULL;
  PyObject *rseq     = NULL;

  typedef  int range_t[3];
  
  int      n      = 0;
  range_t* ranges = NULL;

  MPI_Group newgroup;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O",
			 PyO2Gr, &group, &input) ) return NULL;
  
  MPI4PY_CHECK_NULL(group, MPI_GROUP_NULL, MPI_ERR_GROUP);
    
  TRY {

    s_ranges = PySequence_Fast(input,"sequence expected");
    if ( !s_ranges ) RAISE;
    n = PySequence_Fast_GET_SIZE(s_ranges);

    /* allocate ranges array */
    ranges = PyMem_New(range_t, n);
    if ( !ranges ) PY_RAISE(PyExc_MemoryError, "");
    
    /* fill ranges array */
    { int i; for (i = 0; i < n; i++) {
      PyObject *item = PySequence_Fast_GET_ITEM(s_ranges,i);
      PyObject *rseq = PySequence_Fast(item,
				       "sequence item: sequence expected");
      if (rseq==NULL) RAISE;
      if (PySequence_Fast_GET_SIZE(rseq) != 3) 
	PY_RAISE(PyExc_ValueError, 
		 "sequence item: invalid sequence length");
      { int j; for (j = 0; j < 3; j++) {
	PyObject *r = PySequence_Fast_GET_ITEM(rseq,j);
	if ( !PyInt_Check(r) )
	  PY_RAISE(PyExc_TypeError, 
		   "sequence item: sequence item: integer expected");
	ranges[i][j] = (int) PyInt_AS_LONG(r);
      }}
      Py_DECREF(rseq); rseq = NULL;
    }}
    
    /* obtain group */
    MPI4PY_CALL( MPI_Group_range_excl(group,
				      n, ranges,
				      &newgroup) );
    
    /* deallocate objects */
    PyMem_Del(ranges);
    Py_DECREF(s_ranges);

    /* return group */
    return PyMPI_GROUP(newgroup);

  }

  EXCEPT {
    Py_XDECREF(rseq);
    PyMem_Del(ranges);
    Py_XDECREF(s_ranges);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
group_free(PyObject *self, PyObject *args)
{  
  MPI_Group* group;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&", 
			 PyO2GrP, &group) ) return NULL;

  MPI4PY_CHECK_NULL(*group, MPI_GROUP_NULL, MPI_ERR_GROUP);

  TRY {
    /* free group */
    MPI4PY_CALL_FREE( MPI_Group_free(group) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/









/*------------------------------------------------------------------*/
/* Comm - Intracomm                                                 */
/*------------------------------------------------------------------*/

typedef enum {
  COMM_INTRA,
  COMM_INTER,
  COMM_CART,
  COMM_GRAPH,
  COMM_ANY
} COMM_KIND;

static PyObject *
comm_check_kind(COMM_KIND kind, PyObject *self, PyObject *args)
{
  MPI_Comm comm;
  if ( !PyArg_ParseTuple(args, "O&", PyO2Cm, &comm) )
    return NULL;
  if (comm == MPI_COMM_NULL)
    Py_RETURN_NONE;
  TRY {
    int flag;
    switch (kind) {
    case COMM_INTRA:
      MPI4PY_CALL( MPI_Comm_test_inter(comm, &flag) );
      if (flag) 
        PY_RAISE(PyExc_TypeError,
                 "expecting an intracommunicator");
      break;
    case COMM_INTER:
      MPI4PY_CALL( MPI_Comm_test_inter(comm, &flag) );
      if (!flag) 
        PY_RAISE(PyExc_TypeError,
                 "expecting an intercommunicator");
      break;
    case COMM_CART:
      MPI4PY_CALL( MPI_Topo_test(comm, &flag) );
      if (flag != MPI_CART)
        PY_RAISE(PyExc_TypeError,
                 "expecting a Cartesian communicator");
      break;
    case COMM_GRAPH:
      MPI4PY_CALL( MPI_Topo_test(comm, &flag) );
      if (flag != MPI_GRAPH)
        PY_RAISE(PyExc_TypeError,
                 "expecting a graph communicator");
      break;
    case COMM_ANY:
    default: 
      {
        int size=-1, rank=-1;
        MPI4PY_CALL( MPI_Comm_size(comm, &size) );
        MPI4PY_CALL( MPI_Comm_rank(comm, &rank) );
        if (size <= 0)
          PY_RAISE(PyExc_ValueError,
                   "communicator returned a nonpositive size");
        if (rank < 0 || rank >= size)
          PY_RAISE(PyExc_ValueError,
                   "communicator returned an out of range rank");
      }
    }
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }
}

#define MPI4PY_COMM_CHECK_KIND(kind, KIND)         \
static PyObject *                                  \
comm_check_##kind(PyObject *self, PyObject *args)  \
{                                                  \
  return comm_check_kind(COMM_##KIND, self, args); \
}

MPI4PY_COMM_CHECK_KIND(intra, INTRA)
MPI4PY_COMM_CHECK_KIND(inter, INTER)
MPI4PY_COMM_CHECK_KIND(cart,  CART)
MPI4PY_COMM_CHECK_KIND(graph, GRAPH)
MPI4PY_COMM_CHECK_KIND(any,   ANY)

/*------------------------------------------------------------------*/
static PyObject *
comm_world(PyObject *self, PyObject *args)
{
  return PyMPIComm_FromComm(MPI_COMM_WORLD);
}
static PyObject *
comm_self(PyObject *self, PyObject *args)
{
  return PyMPIComm_FromComm(MPI_COMM_SELF);
}
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static PyObject *
comm_size(PyObject *self, PyObject *args)
{  
  MPI_Comm comm;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);

  TRY {
    /* get size */
    int size;
    MPI4PY_CALL_COMM( comm, MPI_Comm_size(comm, &size) );
    /* return size */
    return PyInt_FromLong((long)size);
  }
  
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_rank(PyObject *self, PyObject *args)
{  
  MPI_Comm comm;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&", 
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);

  TRY {  
    /* get rank */
    int rank;
    MPI4PY_CALL_COMM( comm, MPI_Comm_rank(comm, &rank) );
    /* return rank */
    return PyInt_FromLong((long)rank);
  }

  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_compare(PyObject *self, PyObject *args)
{  
  MPI_Comm comm1; 
  MPI_Comm comm2;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O&", 
			 PyO2Cm, &comm1,
			 PyO2Cm, &comm2) ) return NULL;

  MPI4PY_CHECK_NULL(comm1, MPI_COMM_NULL, MPI_ERR_COMM);
  MPI4PY_CHECK_NULL(comm2, MPI_COMM_NULL, MPI_ERR_COMM);

  TRY {
    /* compare communicators */
    int result;
    MPI4PY_CALL( MPI_Comm_compare(comm1, comm2, &result) );
    /* return result */
    return PyInt_FromLong((long)result);
  }

  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_dup(PyObject *self, PyObject *args)
{
  MPI_Comm comm;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&", 
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    /* duplicate communicator */
    MPI_Comm newcomm;
    MPI4PY_CALL_COMM_NEW(comm, newcomm,
			 MPI_Comm_dup(comm, &newcomm) );
    /* return result */
    return PyMPIComm_FromComm(newcomm);
  }
  
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_create(PyObject *self, PyObject *args)
{  
  MPI_Comm  comm;
  MPI_Group group;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O&", 
			 PyO2Cm, &comm,
			 PyO2Gr, &group) ) return NULL;
  
  MPI4PY_CHECK_NULL(comm,  MPI_COMM_NULL,  MPI_ERR_COMM);
  MPI4PY_CHECK_NULL(group, MPI_GROUP_NULL, MPI_ERR_GROUP);

  TRY {    
    /* create communicator */
    MPI_Comm newcomm;
    MPI4PY_CALL_COMM_NEW(comm, newcomm,
			 MPI_Comm_create(comm, group, &newcomm) );
    /* return result */
    return PyMPIComm_FromComm(newcomm);
  }

  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_split(PyObject *self, PyObject *args)
{
  MPI_Comm comm;
  int color, key;
  MPI_Comm newcomm;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&ii", 
			 PyO2Cm, &comm,
			 &color, &key) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);

  TRY {
    /* split communicator */
    MPI4PY_CALL_COMM_NEW(comm, newcomm,
			 MPI_Comm_split(comm, color, key, &newcomm) );
    /* return new communicator */
    return PyMPIComm_FromComm(newcomm);
  }

  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_free(PyObject *self, PyObject *args)
{  
  MPI_Comm *comm;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&", 
			 PyO2CmP, &comm) ) return NULL;
  
  MPI4PY_CHECK_NULL(*comm, MPI_COMM_NULL, MPI_ERR_COMM);

  MPI4PY_CHECK_NOT_EQ(*comm, PyMPI_COMM_SELF,  MPI_ERR_COMM);
  MPI4PY_CHECK_NOT_EQ(*comm, PyMPI_COMM_WORLD, MPI_ERR_COMM);

  TRY {
    /* free communicator */
    MPI4PY_CALL_COMM_FREE(*comm, MPI_Comm_free(comm) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_test_intra(PyObject *self, PyObject *args)
{  
  MPI_Comm comm;
  int flag;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&", 
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);

  TRY {
    /* test inter */
    MPI4PY_CALL_COMM(comm,
		     MPI_Comm_test_inter(comm, &flag) );
    flag = flag ? 0 : 1;
    /* return result */
    return PyBool_FromLong((long)flag);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/




/*------------------------------------------------------------------*/
/* Intercomm                                                        */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static PyObject *
comm_test_inter(PyObject *self, PyObject *args)
{  
  MPI_Comm comm;
  int flag;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&", 
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);

  TRY {
    /* test */
    MPI4PY_CALL_COMM(comm,
		     MPI_Comm_test_inter(comm, &flag) );
    /* return result */
    return PyBool_FromLong((long)flag);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_remote_size(PyObject *self, PyObject *args)
{  
  MPI_Comm comm;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&", 
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    
    /* get size */
    int size;
    MPI4PY_CALL_COMM(comm, 
		     MPI_Comm_remote_size(comm,&size) );
    
    /* return size */
    return PyInt_FromLong((long)size);
    
  }

  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
intercomm_create(PyObject *self, PyObject *args)
{
  MPI_Comm local_comm; 
  MPI_Comm bridge_comm;
  int local_leader, remote_leader, tag;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&iO&ii",
			 PyO2Cm, &local_comm,  &local_leader,
			 PyO2Cm, &bridge_comm, &remote_leader,
			 &tag) ) return NULL;

  MPI4PY_CHECK_NULL(local_comm,  MPI_COMM_NULL, MPI_ERR_COMM);
  MPI4PY_CHECK_NULL(bridge_comm, MPI_COMM_NULL, MPI_ERR_COMM);

  TRY {
    
    /* create intercommunicator */
    MPI_Comm newintercomm;
    MPI4PY_CALL_COMM_NEW(local_comm, newintercomm,
			 MPI_Intercomm_create(local_comm, local_leader,
					      bridge_comm, remote_leader, 
					      tag, &newintercomm) );

    /* return new communicator */
    return PyMPIComm_FromComm(newintercomm);
    
  }
  
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
intercomm_merge(PyObject *self, PyObject *args)
{  
  MPI_Comm comm; 
  int high;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&i",
			 PyO2Cm, &comm,
			 &high) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    
    /* merge intercommunicator */
    MPI_Comm newintracomm;
    MPI4PY_CALL_COMM_NEW(comm, newintracomm,
			 MPI_Intercomm_merge(comm, high, 
					     &newintracomm) );

    /* return new communicator */
    return PyMPIComm_FromComm(newintracomm);
    
  }
  
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/





/*------------------------------------------------------------------*/
/* Virtual Topologies                                               */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static PyObject *
topo_test(PyObject *self, PyObject *args)
{  
  MPI_Comm comm;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&", 
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    
    /* test communicator */
    int status;
    MPI4PY_CALL_COMM(comm, 
		     MPI_Topo_test(comm, &status) );
    
    /* return result */
    return PyInt_FromLong((long)status);
  }
  
  EXCEPT {
    return NULL;
  }

}


/*------------------------------------------------------------------*/
/* Cartcomm                                                         */
/*------------------------------------------------------------------*/


/*------------------------------------------------------------------*/
static PyObject *
cart_create(PyObject *self, PyObject *args)
{
  MPI_Comm comm;

  PyObject *o_dims = NULL;
  PyObject *o_pers = NULL;
  PyObject *o_reor = NULL;

  int  ndims   = 0;
  int* dims    = NULL;
  int* periods = NULL;
  int  reorder = 0;

  MPI_Comm comm_cart;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&OOO",
			 PyO2Cm, &comm,
			 &o_dims, &o_pers, &o_reor) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {

    /* check input objects */
    o_dims = PySequence_Fast(o_dims,"sequence expected");
    if ( !o_dims ) RAISE;
    o_pers = PySequence_Fast(o_pers,"sequence expected");
    if ( !o_pers ) RAISE;
    reorder = PyObject_IsTrue(o_reor);
    if ( reorder == -1 )
      PY_RAISE(PyExc_TypeError, "boolean expected");

    /* allocate arrays       */
    ndims = PySequence_Fast_GET_SIZE(o_dims);
    if ( ndims != PySequence_Fast_GET_SIZE(o_pers) )
      PY_RAISE(PyExc_ValueError, "incompatible sequence sizes");
    dims    = PyMem_New(int, ndims);
    periods = PyMem_New(int, ndims);
    if ( !dims || !periods) PY_RAISE(PyExc_MemoryError, "");

    /* fill arrays */
    { int i; for (i = 0; i < ndims; i++) {
      PyObject *d = PySequence_Fast_GET_ITEM(o_dims,i);
      PyObject *p = PySequence_Fast_GET_ITEM(o_pers,i);
      if ( !PyInt_Check(d) )
	PY_RAISE(PyExc_TypeError, "sequence item: integer expected");
      dims[i] = (int) PyInt_AS_LONG(d);
      periods[i] = PyObject_IsTrue(p);
      if ( periods[i] == -1 )
	PY_RAISE(PyExc_TypeError, "sequence item: boolean expected");
    }}

    /* create cartesian communicator */
    MPI4PY_CALL_COMM_NEW(comm, comm_cart,
			 MPI_Cart_create(comm, ndims, dims, periods, 
					 reorder, &comm_cart) );

    /* deallocate objects */
    PyMem_Del(dims);
    PyMem_Del(periods);
    Py_DECREF(o_dims);
    Py_DECREF(o_pers);

    /* return new communicator */
    return PyMPIComm_FromComm(comm_cart);

  }

  EXCEPT {
    PyMem_Del(dims);
    PyMem_Del(periods);
    Py_XDECREF(o_dims);
    Py_XDECREF(o_pers);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
dims_create(PyObject *self, PyObject *args)
{  
  int nnodes;
  PyObject *input  = NULL;
  PyObject *result = NULL;

  int ndims = 0;
  int *dims   = NULL;
  PyObject *s_dims = NULL;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "iO",
			 &nnodes, &input) ) return NULL;

  TRY {
    /* check input objects */
    if ( PyInt_Check(input) ) {
      ndims = (int) PyInt_AS_LONG(input);
    }
    else {
      s_dims = PySequence_Fast(input,"integer or sequence expected");
      if ( !s_dims ) return NULL;
      ndims = PySequence_Fast_GET_SIZE(s_dims);
    }

    /* allocate dims array */
    dims  = PyMem_New(int, ndims);
    if ( !dims ) PY_RAISE(PyExc_MemoryError, "");

    /* fill dims array */
    if (!s_dims)
      { int i; for (i = 0; i < ndims; i++) 
	dims[i] = 0;
      }
    else
      { int i; for (i = 0; i < ndims; i++) {
	PyObject *item = PySequence_Fast_GET_ITEM(s_dims,i);
	if ( !PyInt_Check(item) )
	  PY_RAISE(PyExc_TypeError,
		   "sequence item: integer expected");
	dims[i] = (int) PyInt_AS_LONG(item);
      }}

    /* obtain dims array */
    MPI4PY_CALL( MPI_Dims_create(nnodes, ndims, dims) );
    
    /* construct result */
    result = PyTuple_New((Py_ssize_t)ndims);
    if ( !result ) RAISE;
    { int i; for (i = 0; i < ndims; i++) {
      PyObject *o = PyInt_FromLong((long)dims[i]);
      if ( !o ) RAISE;
      PyTuple_SET_ITEM(result, i, o);
    }}

    /* deallocate objects */
    Py_XDECREF(s_dims);
    PyMem_Del(dims); dims = NULL;
    
    /* return result */
    return result;

  }

  EXCEPT {
    PyMem_Del(dims);
    Py_XDECREF(s_dims);
    Py_XDECREF(result);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
cartdim_get(PyObject *self, PyObject *args)
{  
  MPI_Comm comm;
  int  ndims;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    /* obtain ndims */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Cartdim_get(comm, &ndims) );
    /* return result */
    return PyInt_FromLong((long)ndims);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
cart_get(PyObject *self, PyObject *args)
{
  MPI_Comm comm;

  PyObject *o_dims    = NULL;
  PyObject *o_periods = NULL;
  PyObject *o_coords  = NULL;
  
  int  maxdims;
  int* dims    = NULL;
  int* periods = NULL;
  int* coords  = NULL;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);

  TRY {
    
    /* obtain maxdims */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Cartdim_get(comm, &maxdims) );

    /* allocate arrays */
    dims    = PyMem_New(int, maxdims);
    periods = PyMem_New(int, maxdims);
    coords  = PyMem_New(int, maxdims);
    if ( !dims || !periods || !coords ) 
      PY_RAISE(PyExc_MemoryError, "");
    
    /* get array's data */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Cart_get(comm, maxdims, 
				  dims, periods, coords) );

    /* allocate result lists */
    o_dims    = PyTuple_New((Py_ssize_t)maxdims);
    o_periods = PyTuple_New((Py_ssize_t)maxdims);
    o_coords  = PyTuple_New((Py_ssize_t)maxdims);
    if ( !o_dims || !o_periods || !o_coords ) RAISE;

    /* fill result lists */
    { int i; for (i = 0; i < maxdims; i++){
      PyObject *item;
      /* dims */
      item = PyInt_FromLong((long)dims[i]);
      if (!item) RAISE;
      PyTuple_SET_ITEM(o_dims,i,item);
      /* periods */
      item = PyInt_FromLong((long)periods[i]);
      if (!item) RAISE;
      PyTuple_SET_ITEM(o_periods,i,item);
      /* coords */
      item = PyInt_FromLong((long)coords[i]);
      if (!item) RAISE;
      PyTuple_SET_ITEM(o_coords,i,item);
    }}

    /* deallocate arrays */
    PyMem_Del(dims);
    PyMem_Del(periods);
    PyMem_Del(coords);

    /* return result */
    return Py_BuildValue("NNN", o_dims, o_periods, o_coords);
    
  }

  EXCEPT {
    PyMem_Del(dims);  
    PyMem_Del(periods);
    PyMem_Del(coords);
    Py_XDECREF(o_dims);
    Py_XDECREF(o_periods);
    Py_XDECREF(o_coords);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
cart_rank(PyObject *self, PyObject *args)
{  
  MPI_Comm comm;
  PyObject *o_coords = NULL;

  int* coords = NULL;
  int  rank;
  int  ndims;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O",
			 PyO2Cm, &comm, &o_coords) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    
    /* check input arguments */
    o_coords = PySequence_Fast(o_coords,"sequence expected");
    if ( !o_coords ) RAISE;

    /* allocate coords array */
    ndims = PySequence_Fast_GET_SIZE(o_coords);
    coords = PyMem_New(int, ndims);
    if ( !coords ) PY_RAISE(PyExc_MemoryError, "");
    
    /* fill coords array */
    { int i; for (i = 0; i < ndims; i++) {
      PyObject *item = PySequence_Fast_GET_ITEM(o_coords,i);
      if ( !PyInt_Check(item) )
	PY_RAISE(PyExc_TypeError, "sequence item: integer expected");
      coords[i] = (int) PyInt_AS_LONG(item);
    }}
    
    /* obtain rank */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Cart_rank(comm, coords, &rank) );

    /* deallocate objects */
    PyMem_Del(coords);
    Py_DECREF(o_coords);
    /* return result */
    return PyInt_FromLong((long)rank);
  }

  EXCEPT {
    PyMem_Del(coords);
    Py_XDECREF(o_coords);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
cart_coords(PyObject *self, PyObject *args)
{  
  MPI_Comm comm;

  int  rank;
  int  maxdims;
  int* coords = NULL;

  PyObject *result = NULL;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&i",
			 PyO2Cm, &comm, &rank) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);

  TRY {
    
    /* allocate coords array */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Cartdim_get(comm,&maxdims) );
    coords = PyMem_New(int, maxdims);
    if ( !coords ) PY_RAISE(PyExc_MemoryError, "");

    /* obtain coords */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Cart_coords(comm, rank, maxdims, coords) );
    
    /* construct result */
    result = PyTuple_New((Py_ssize_t)maxdims);
    if ( !result ) RAISE;
    { int i; for (i = 0; i < maxdims; i++) {
      PyObject *item = PyInt_FromLong((long) coords[i]);
      if ( !item ) PY_RAISE(PyExc_MemoryError, "");
      PyTuple_SET_ITEM(result, i, item);
    }}

    /* deallocate objects */
    PyMem_Del(coords);
    
    /* return result */
    return result;

  }

  EXCEPT {
    PyMem_Del(coords);
    Py_XDECREF(result);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
cart_shift(PyObject *self, PyObject *args)
{
  MPI_Comm comm;
  int direction;
  int disp;
  int rank_source;
  int rank_dest;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&ii",
			 PyO2Cm, &comm,
			 &direction, &disp) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    /* obtain source and dest */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Cart_shift(comm, direction, disp,
				    &rank_source, &rank_dest) );
    /* return result */
    return Py_BuildValue("ii", rank_source, rank_dest);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
cart_sub(PyObject *self, PyObject *args)
{
  MPI_Comm comm;

  PyObject *o_rem_dims = NULL;

  int  ndims;
  int* remain_dims = NULL;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O",
			 PyO2Cm, &comm, &o_rem_dims) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
    
  TRY {

    MPI_Comm newcomm;

    /* check input objects */
    o_rem_dims = PySequence_Fast(o_rem_dims, "sequence expected");
    if ( !o_rem_dims ) RAISE;
    MPI4PY_CALL_COMM(comm, 
		     MPI_Cartdim_get(comm,&ndims) );
    if ( ndims != PySequence_Fast_GET_SIZE(o_rem_dims) )
      PY_RAISE(PyExc_ValueError, "invalid sequence size");

    /* allocate array */
    remain_dims = PyMem_New(int, ndims);
    if ( !remain_dims ) PY_RAISE(PyExc_MemoryError, "");
    
    /* fill array */
    { int i; for (i = 0; i < ndims; i++) {
      PyObject *item = PySequence_Fast_GET_ITEM(o_rem_dims,i);
      remain_dims[i] = PyObject_IsTrue(item);
      if ( remain_dims[i] == -1 )
	PY_RAISE(PyExc_TypeError, "sequence item: boolean expected");
    }}

    /* create cartesian communicator */
    MPI4PY_CALL_COMM_NEW(comm, newcomm,
			 MPI_Cart_sub(comm, remain_dims, &newcomm) );
    
    /* deallocate objects */
    PyMem_Del(remain_dims);
    Py_DECREF(o_rem_dims);
    
    /* return new communicator */
    return PyMPIComm_FromComm(newcomm);

  }

   EXCEPT {
    PyMem_Del(remain_dims);
    Py_XDECREF(o_rem_dims);
    return NULL;
  }
  
}
/*------------------------------------------------------------------*/
static PyObject *
cart_map(PyObject *self, PyObject *args)
{
  MPI_Comm comm;

  PyObject *o_dims = NULL;
  PyObject *o_pers = NULL;

  int  ndims   = 0;
  int *dims    = NULL;
  int *periods = NULL;
  int newrank;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&OO",
			 PyO2Cm, &comm,
			 &o_dims, &o_pers) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {

    /* check input objects */
    o_dims = PySequence_Fast(o_dims,"sequence expected");
    if ( !o_dims ) RAISE;
    o_pers = PySequence_Fast(o_pers,"sequence expected");
    if ( !o_pers ) RAISE;

    /* allocate arrays       */
    ndims = PySequence_Fast_GET_SIZE(o_dims);
    if ( ndims != PySequence_Fast_GET_SIZE(o_pers) )
      PY_RAISE(PyExc_ValueError, "incompatible sequence sizes");
    dims    = PyMem_New(int, ndims);
    periods = PyMem_New(int, ndims);
    if ( !dims || !periods) PY_RAISE(PyExc_MemoryError, "");

    /* fill arrays */
    { int i; for (i = 0; i < ndims; i++) {
      PyObject *d = PySequence_Fast_GET_ITEM(o_dims,i);
      PyObject *p = PySequence_Fast_GET_ITEM(o_pers,i);
      if ( !PyInt_Check(d) )
	PY_RAISE(PyExc_TypeError, "sequence item: integer expected");
      dims[i] = (int) PyInt_AS_LONG(d);
      periods[i] = PyObject_IsTrue(p);
      if ( periods[i] == -1 )
	PY_RAISE(PyExc_TypeError, "sequence item: boolean expected");
    }}

    /* create cartesian communicator */
    MPI4PY_CALL_COMM(comm,
		     MPI_Cart_map(comm,
				  ndims, dims, periods,
				  &newrank) );

    /* deallocate objects */
    PyMem_Del(dims);
    PyMem_Del(periods);
    Py_DECREF(o_dims);
    Py_DECREF(o_pers);

    /* return reordered rank */
    return Py_BuildValue("i",newrank);

  }

  EXCEPT {
    PyMem_Del(dims);
    PyMem_Del(periods);
    Py_XDECREF(o_dims);
    Py_XDECREF(o_pers);
    return NULL;
  }

}
/*------------------------------------------------------------------*/


/*------------------------------------------------------------------*/
/* Graphcomm                                                        */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static PyObject *
graph_create(PyObject *self, PyObject *args)
{
  MPI_Comm comm;

  PyObject *o_index = NULL;
  PyObject *o_edges = NULL;
  PyObject *o_reord = NULL;

  int  nnodes;
  int* index = NULL;
  int  nedges;
  int* edges = NULL;
  int  reorder;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&OOO",
			 PyO2Cm, &comm,
			 &o_index, &o_edges, &o_reord) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {

    MPI_Comm comm_graph;

    /* check input objects */
    o_index = PySequence_Fast(o_index, "sequence expected");
    if ( !o_index ) RAISE;
    o_edges = PySequence_Fast(o_edges, "sequence expected");
    if ( !o_edges ) RAISE;
    reorder = PyObject_IsTrue(o_reord);
    if ( reorder == -1 )
      PY_RAISE(PyExc_TypeError, "boolean expected");

    /* allocate arrays */
    nnodes = PySequence_Fast_GET_SIZE(o_index);
    nedges = PySequence_Fast_GET_SIZE(o_edges);
    index = PyMem_New(int, nnodes);
    edges = PyMem_New(int, nedges);
    if ( !index || !edges ) PY_RAISE(PyExc_MemoryError, "");

    /* fill arrays */
    { int i; for (i = 0; i < nnodes; i++) {
      PyObject *item = PySequence_Fast_GET_ITEM(o_index,i);
      if ( !PyInt_Check(item) )
	PY_RAISE(PyExc_TypeError, "sequence item: integer expected");
      index[i] = (int) PyInt_AS_LONG(item);
    }}
    { int i; for (i = 0; i < nedges; i++) {
      PyObject *item = PySequence_Fast_GET_ITEM(o_edges,i);
      if ( !PyInt_Check(item) )
	PY_RAISE(PyExc_TypeError, "sequence item: integer expected");
      edges[i] = (int) PyInt_AS_LONG(item);
    }}

    /* create graph communicator */
    MPI4PY_CALL_COMM_NEW(comm, comm_graph,
			 MPI_Graph_create(comm,  nnodes, index, edges, 
					  reorder, &comm_graph) );

    /* deallocate objects */
    PyMem_Del(index);
    PyMem_Del(edges);
    Py_DECREF(o_index);
    Py_DECREF(o_edges);

    /* return new communicator */
    return PyMPIComm_FromComm(comm_graph);
  }

  EXCEPT {
    PyMem_Del(index);
    PyMem_Del(edges);
    Py_XDECREF(o_index);
    Py_XDECREF(o_edges);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
graphdims_get(PyObject *self, PyObject *args)
{
  MPI_Comm comm;
  int nnodes;
  int nedges;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    /* obtain nnodes and nedges */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Graphdims_get(comm, &nnodes, &nedges) );
    /* return result */
    return Py_BuildValue("ii", nnodes, nedges);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
graph_get(PyObject *self, PyObject *args)
{
  MPI_Comm comm;
  int  maxindex;
  int  maxedges;
  int* index = NULL;
  int* edges = NULL;
  
  PyObject *o_index = NULL;
  PyObject *o_edges = NULL;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
    

  TRY {

    /* obtain maxindex and maxedges */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Graphdims_get(comm, &maxindex, &maxedges) );

    /* allocate arrays */
    index = PyMem_New(int, maxindex);
    edges = PyMem_New(int, maxedges);
    if ( !index || !edges ) PY_RAISE(PyExc_MemoryError, "");
    
    /* get array's data */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Graph_get(comm, 
				   maxindex, maxedges,
				   index,    edges) );
    
    /* allocate result lists */
    o_index = PyList_New((Py_ssize_t)maxindex);
    o_edges = PyList_New((Py_ssize_t)maxedges);
    if ( !o_index || !o_edges ) RAISE;

    /* fill result lists */
    { int i; for (i = 0; i < maxindex; i++) {
      PyObject *item = PyInt_FromLong((long)index[i]);
      if (!item) RAISE;
      PyList_SET_ITEM(o_index,i,item);
    }}
    { int i; for (i = 0; i < maxedges; i++) {
      PyObject *item = PyInt_FromLong((long)edges[i]);
      if (!item) RAISE;
      PyList_SET_ITEM(o_edges,i,item);
    }}

    /* deallocate arrays */
    PyMem_Del(index);
    PyMem_Del(edges);

    /* return result */
    return Py_BuildValue("NN", o_index, o_edges);

  }
  
  EXCEPT {
    PyMem_Del(index);
    PyMem_Del(edges);
    Py_XDECREF(o_index);
    Py_XDECREF(o_edges);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
graph_neigh_count(PyObject *self, PyObject *args)
{
  MPI_Comm comm;
  int rank;
  int nneighbors;
   
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&i",
			 PyO2Cm, &comm, &rank) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    /* obtain nnodes and nedges */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Graph_neighbors_count(comm, rank,
					       &nneighbors) );
    /* return result */
    return PyInt_FromLong((long)nneighbors);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
graph_neigh(PyObject *self, PyObject *args)
{
  MPI_Comm comm;
  int rank;
  int  maxneighbors;
  int* neighbors = NULL;

  PyObject *o_neighbors = NULL;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&i",
			 PyO2Cm, &comm, &rank) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    
    /* obtain maxneighbors */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Graph_neighbors_count(comm, rank, &maxneighbors) );
    
    /* allocate array */
    neighbors = PyMem_New(int, maxneighbors);
    if ( !neighbors ) PY_RAISE(PyExc_MemoryError, "");
    
    /* get array data */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Graph_neighbors(comm, rank,
					 maxneighbors, neighbors) );
    
    /* allocate result list */
    o_neighbors = PyList_New((Py_ssize_t)maxneighbors);
    if ( !o_neighbors ) RAISE;
    
    /* fill list */
    { int i; for (i = 0; i < maxneighbors; i++){
      PyObject *item = PyInt_FromLong((long)neighbors[i]);
      if (!item) PY_RAISE(PyExc_MemoryError, "");
      PyList_SET_ITEM(o_neighbors,i,item);
    }}

    /* deallocate array */
    PyMem_Del(neighbors);

    /* return result */
    return o_neighbors;
  }

  EXCEPT {
    PyMem_Del(neighbors);
    Py_XDECREF(o_neighbors);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
graph_map(PyObject *self, PyObject *args)
{
  MPI_Comm comm;

  PyObject *o_index = NULL;
  PyObject *o_edges = NULL;

  int  nnodes;
  int* index = NULL;
  int  nedges;
  int* edges = NULL;
  int  newrank;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&OO",
			 PyO2Cm, &comm,
			 &o_index, &o_edges) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {

    /* check input objects */
    o_index = PySequence_Fast(o_index, "sequence expected");
    if ( !o_index ) RAISE;
    o_edges = PySequence_Fast(o_edges, "sequence expected");
    if ( !o_edges ) RAISE;

    /* allocate arrays */
    nnodes = PySequence_Fast_GET_SIZE(o_index);
    nedges = PySequence_Fast_GET_SIZE(o_edges);
    index = PyMem_New(int, nnodes);
    edges = PyMem_New(int, nedges);
    if ( !index || !edges ) PY_RAISE(PyExc_MemoryError, "");

    /* fill arrays */
    { int i; for (i = 0; i < nnodes; i++) {
      PyObject *item = PySequence_Fast_GET_ITEM(o_index,i);
      if ( !PyInt_Check(item) )
	PY_RAISE(PyExc_TypeError, "sequence item: integer expected");
      index[i] = (int) PyInt_AS_LONG(item);
    }}
    { int i; for (i = 0; i < nedges; i++) {
      PyObject *item = PySequence_Fast_GET_ITEM(o_edges,i);
      if ( !PyInt_Check(item) )
	PY_RAISE(PyExc_TypeError, "sequence item: integer expected");
      edges[i] = (int) PyInt_AS_LONG(item);
    }}

    /* create graph communicator */
    MPI4PY_CALL_COMM(comm,
		     MPI_Graph_map(comm,
				   nnodes, index, edges,
				   &newrank) );

    /* deallocate objects */
    PyMem_Del(index);
    PyMem_Del(edges);
    Py_DECREF(o_index);
    Py_DECREF(o_edges);

    /* return reorder rank */
    return Py_BuildValue("i",newrank);
  }

  EXCEPT {
    PyMem_Del(index);
    PyMem_Del(edges);
    Py_XDECREF(o_index);
    Py_XDECREF(o_edges);
    return NULL;
  }

}
/*------------------------------------------------------------------*/


/*------------------------------------------------------------------*/
/* Ports and Services                                               */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static PyObject *
open_port(PyObject *self, PyObject *args)
{  
  MPI_Info info;
  char *port_name;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2IN, &info) ) return NULL;
  
  TRY {
    /* open port */
#if HAVE_MPI_OPEN_PORT
    char buf[MPI_MAX_PORT_NAME] = { '\0' };
    port_name = buf;
    MPI4PY_CALL( MPI_Open_port(info, port_name) );
#else
    port_name = "";
    MPI4PY_RAISE_MPINOIMPL("MPI_OPEN_PORT");
#endif
    /* return port name */
    return Py_BuildValue("s", port_name);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
close_port(PyObject *self, PyObject *args)
{  
  char *port_name;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "s",
			 &port_name) ) return NULL;
  
  TRY {
    /* close port */
#if HAVE_MPI_CLOSE_PORT
    MPI4PY_CALL( MPI_Close_port(port_name) );
#else
    MPI4PY_RAISE_MPINOIMPL("MPI_CLOSE_PORT");
#endif
    /* return  */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
publish_name(PyObject *self, PyObject *args)
{  
  char *service_name;
  MPI_Info info;
  char *port_name;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "sO&s",
			 &service_name,
			 PyO2IN, &info,
			 &port_name) ) return NULL;
  
  TRY {
    /* publish service */
#if HAVE_MPI_PUBLISH_NAME
    MPI4PY_CALL( MPI_Publish_name(service_name, info, port_name) );
#else
    MPI4PY_RAISE_MPINOIMPL("MPI_PUBLISH_NAME");
#endif
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
unpublish_name(PyObject *self, PyObject *args)
{  
  char *service_name;
  MPI_Info info;
  char *port_name;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "sO&s",
			 &service_name,
			 PyO2IN, &info,
			 &port_name) ) return NULL;
  
  TRY {
    /* unpublish service */
#if HAVE_MPI_UNPUBLISH_NAME
    MPI4PY_CALL( MPI_Unpublish_name(service_name, info, port_name) );
#else
    MPI4PY_RAISE_MPINOIMPL("MPI_UNPUBLISH_NAME");
#endif
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
lookup_name(PyObject *self, PyObject *args)
{  
  char *service_name;
  MPI_Info info;
  char *port_name;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "sO&",
			 &service_name,
			 PyO2IN, &info) ) return NULL;
  
  TRY {
    /* lookup service */
#if HAVE_MPI_LOOKUP_NAME
    char buf[MPI_MAX_PORT_NAME] = { '\0' };
    port_name = buf;
    MPI4PY_CALL( MPI_Lookup_name(service_name, info, port_name) );
#else
    port_name = "";
    MPI4PY_RAISE_MPINOIMPL("MPI_LOOKUP_NAME");
#endif
    /* return port name */
    return Py_BuildValue("s", port_name);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_accept(PyObject *self, PyObject *args)
{  
  char *port_name;
  MPI_Info info;
  int root;
  MPI_Comm comm;
  MPI_Comm newcomm;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "zO&iO&",
			 &port_name,
			 PyO2IN, &info,
			 &root,
			 PyO2Cm, &comm) ) return NULL;
  
  TRY {
    /* accept */
#if HAVE_MPI_COMM_ACCEPT
    MPI4PY_CALL_COMM_NEW(comm, newcomm,
			 MPI_Comm_accept(port_name, info, root, 
					 comm, &newcomm) );
#else
    newcomm = MPI_COMM_NULL;
    MPI4PY_RAISE_MPINOIMPL("MPI_COMM_ACCEPT");
#endif
    /* return new communicator */
    return PyMPIComm_FromComm(newcomm);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_connect(PyObject *self, PyObject *args)
{  
  char *port_name;
  MPI_Info info;
  int root;
  MPI_Comm comm;
  MPI_Comm newcomm;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "zO&iO&",
			 &port_name,
			 PyO2IN, &info,
			 &root,
			 PyO2Cm, &comm) ) return NULL;
  
  TRY {
    /* connect */
#if HAVE_MPI_COMM_CONNECT
    MPI4PY_CALL_COMM_NEW(comm, newcomm,
			 MPI_Comm_connect(port_name, info, root, 
					  comm, &newcomm) );
#else
    newcomm = MPI_COMM_NULL;
    MPI4PY_RAISE_MPINOIMPL("MPI_COMM_CONNECT");
#endif
    /* return new communicator */
    return PyMPIComm_FromComm(newcomm);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_disconnect(PyObject *self, PyObject *args)
{  
  MPI_Comm *comm;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&", 
			 PyO2CmP, &comm) ) return NULL;
  
  MPI4PY_CHECK_NULL(*comm, MPI_COMM_NULL, MPI_ERR_COMM);

  TRY {
    /* disconnect communicator */
#if HAVE_MPI_COMM_DISCONNECT
    MPI4PY_CHECK_NOT_EQ(*comm, MPI_COMM_SELF,  MPI_ERR_COMM);
    MPI4PY_CHECK_NOT_EQ(*comm, MPI_COMM_WORLD, MPI_ERR_COMM);
    MPI4PY_CALL_COMM_FREE(*comm,  MPI_Comm_disconnect(comm) );
#else
    MPI4PY_RAISE_MPINOIMPL("MPI_COMM_DISCONNECT");
#endif
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static MPI4PYUNUSED int
comm_spawn_argv_new(PyObject *seq, char*** args)
{
  int argc = 0; 
  char **argv = NULL;

  if (seq == Py_None) return 1;
  if (!seq_as_array_string(seq, -1, &argv, &argc)) RAISE;
  argv[argc] = NULL;
  *args = argv;
  return 1;
 fail:
  PyMem_Del(argv);
  return 0;
}
static MPI4PYUNUSED int
comm_spawn_argv_del(PyObject *seq, char*** args)
{
  if (args  == NULL) return 1;
  if (*args == NULL) return 1;
  PyMem_Del(*args);
  *args = NULL;
  return 1;
}
#if HAVE_MPI_ERRCODES_IGNORE
#define PyMPI_ERRCODES_IGNORE MPI_ERRCODES_IGNORE
#else
#define PyMPI_ERRCODES_IGNORE ((int*)0)
#endif
static MPI4PYUNUSED int
comm_spawn_errc_new(PyObject *oerrc, int **errcodes, int maxprocs) 
{
  int *array_of_errcodes = NULL;
  *errcodes = PyMPI_ERRCODES_IGNORE;
  if (oerrc == Py_None) return 1;
  if (maxprocs < 0)     return 1;
  if (!PyList_Check(oerrc))
    PY_RAISE(PyExc_TypeError, "errcodes argument must be list");
  array_of_errcodes = PyMem_New(int, maxprocs);
  if (array_of_errcodes == NULL)
    PY_RAISE(PyExc_MemoryError, "allocating memory for errcodes");
  *errcodes = array_of_errcodes;
  return 1;
 fail:
  PyMem_Del(array_of_errcodes);
  return 0;
}
static MPI4PYUNUSED int
comm_spawn_errc_del(PyObject *oerrc, int **errcodes, int maxprocs) {
  int *skip = PyMPI_ERRCODES_IGNORE;
  if (errcodes  == NULL) return 1;
  if (*errcodes == skip) return 1;
  if (*errcodes == NULL) return 1;
  PyMem_Del(*errcodes);
  *errcodes = NULL;
  return 1;
}
static MPI4PYUNUSED int
comm_spawn_errc_set(int maxprocs, int *errcodes, PyObject *oerrc)
{
  int *skip = PyMPI_ERRCODES_IGNORE;
  PyObject *list = NULL;
  if (oerrc == Py_None) return 1;
  if (maxprocs < 0)     return 1;
  if (errcodes == skip) return 1;
  if (errcodes == NULL) return 1;
  /* XXX should check errors and issue a warning */
  list = list_from_array_int(errcodes, maxprocs);
  PyList_SetSlice(oerrc, 0, (Py_ssize_t)maxprocs, list);
  Py_XDECREF(list);
  return 1;
}
/*------------------------------------------------------------------*/
static PyObject *
comm_spawn(PyObject *self, PyObject *args)
{
  PyObject *oargv;
  PyObject *oerrc;
  
  char *command;
  char **argv;
  int maxprocs;
  MPI_Info info;
  int root;
  MPI_Comm comm;
  MPI_Comm intercomm;
  int *errcodes;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "zOiO&iO&O",
			 &command,
			 &oargv,
			 &maxprocs,
			 PyO2IN, &info,
			 &root,
			 PyO2Cm, &comm,
                         &oerrc        ) ) return NULL;
  
  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);

  TRY {
#if HAVE_MPI_COMM_SPAWN
    int rank;
    argv = MPI_ARGV_NULL;
    errcodes = MPI_ERRCODES_IGNORE;
    MPI4PY_CALL_COMM(comm, MPI_Comm_rank(comm, &rank) );
    if (rank == root) {
      if (!comm_spawn_argv_new(oargv, &argv)) RAISE;
      if (!comm_spawn_errc_new(oerrc, &errcodes, maxprocs)) RAISE;
    }
    /* spawn */
    MPI4PY_CALL_COMM_NEW(comm, intercomm,
			 MPI_Comm_spawn(command, argv, maxprocs,
					info, root,
					comm, &intercomm,
					errcodes) );
    if (rank == root) {
      comm_spawn_errc_set(maxprocs, errcodes, oerrc);
    }
#else
    argv = NULL; errcodes = NULL;
    intercomm = MPI_COMM_NULL;
    MPI4PY_RAISE_MPINOIMPL("MPI_COMM_SPAWN");
#endif
    comm_spawn_argv_del(oargv, &argv);
    comm_spawn_errc_del(oerrc, &errcodes, maxprocs);
    /* return result */
    return PyMPIComm_FromComm(intercomm);
  }
  EXCEPT {
    comm_spawn_argv_del(oargv, &argv);
    comm_spawn_errc_del(oerrc, &errcodes, maxprocs);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_get_parent(PyObject *self, PyObject *args)
{  
  MPI_Comm parent;

  TRY {
    /* get_parent */
#if HAVE_MPI_COMM_GET_PARENT
    MPI4PY_CALL_COMM_NEW(MPI_COMM_WORLD, parent,
                         MPI_Comm_get_parent(&parent) );
#else
    parent = MPI_COMM_NULL;
    MPI4PY_RAISE_MPINOIMPL("MPI_COMM_GET_PARENT");
#endif
    /* return result */
    return PyMPIComm_FromComm(parent);
  }
  EXCEPT {
    return NULL;
  }

}

/*------------------------------------------------------------------*/
static PyObject *
comm_join(PyObject *self, PyObject *args)
{  
  int fd;
  MPI_Comm newcomm;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "i", &fd) ) return NULL;
  
  TRY {
    /* join */
#if HAVE_MPI_COMM_JOIN
    MPI4PY_CALL_COMM_NEW(MPI_COMM_WORLD, newcomm,
                         MPI_Comm_join(fd, &newcomm) );
#else
    newcomm = MPI_COMM_NULL;
    MPI4PY_RAISE_MPINOIMPL("MPI_COMM_JOIN");
#endif
    /* return result */
    return PyMPIComm_FromComm(newcomm);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/



/*------------------------------------------------------------------*/

static PyObject *
make_buf(PyObject *self, PyObject *args)
{  
  PyObject *object;
  PyTypeObject *buftype = NULL;

  if (!PyArg_UnpackTuple(args, "", 1, 2,  &object, &buftype)) 
    return NULL;
  
  /* object is a tuple or list */
  if (PyTuple_Check(object) || PyList_Check(object)) {
    /* get tuple items (data, count, datatype) */
    Py_ssize_t ssize = PySequence_Fast_GET_SIZE(object);
    PyObject **items = PySequence_Fast_ITEMS(object);
    PyObject *data, *count, *dtype;
    switch (ssize) {
    case 2:
      data = items[0]; count = Py_None;  dtype = items[1]; break;
    case 3:
      data = items[0]; count = items[1]; dtype = items[2]; break;
    default:
      goto exit;
    }
    /* check tuple items */
    if (!PyMPIDatatype_Check(dtype))
      goto exit;
    if (!(count == Py_None ||
	  PyInt_Check(count)))
      goto exit;
    if (!(data == Py_None ||
	  PyObject_CheckReadBuffer(data) ||
	  (PyTuple_Check(data) && PyTuple_GET_SIZE(data) == 2)))
      goto exit;
    /* return tuple (data, count, dtype) and true fastflag  */
    return Py_BuildValue("(OOO)O", data, count, dtype, Py_True);
  }

exit:
  /* return original object and false fastflag */
  return Py_BuildValue("OO", object, Py_False);
}

/*------------------------------------------------------------------*/

typedef struct buffer_t {
  void *buf;
  int count;
  MPI_Datatype dtype;
} buffer_t;

static int
as_buffer(int writable, PyObject *obj, buffer_t *buf, int segments)
{
  PyObject *args = NULL;

  PyObject *odata = NULL;
  PyObject *odtype = NULL;
  PyObject *ocount = NULL;

  char *data = NULL;
  int count = -1;
  MPI_Datatype datatype = MPI_DATATYPE_NULL;
  
  /* initialization */
  buf->buf   = NULL;
  buf->count = 0;
  buf->dtype = MPI_DATATYPE_NULL;

  if (obj == NULL) 
    return 1;
  if ((args = PySequence_Tuple(obj)) == NULL) 
    return 0;
  if (!PyArg_UnpackTuple(args, "", 2, 3, &odata, &ocount, &odtype)) 
    return 0;
  
  if (odtype == NULL) {
    odtype = ocount;
    ocount = Py_None;
  }

  /* get count (if provided) */
  if (ocount != Py_None) {
    if (!PyInt_Check(ocount)) {
      PyErr_Format(PyExc_TypeError, 
		   "count must be integer, got %.200s",
		   ocount->ob_type->tp_name);
      goto fail;
    }
    count = (int) PyInt_AS_LONG(ocount);
    if (count < 0) {
      PyErr_Format(PyExc_ValueError, 
		   "count must be nonnegative, got %d",
		   count);
      goto fail;
    }
  }

  /* get datatype */
  datatype = PyMPIDatatype_AsDatatype(odtype);
  if (datatype == MPI_DATATYPE_NULL) {
    if (!PyErr_Occurred())
      PyErr_SetString(PyExc_ValueError,
                      "datatype is not valid");
    goto fail;
  }

  /* get buffer pointer (and count if not provided) */
  if (odata == Py_None) { /* XXX Make it equivalet to MPI_BOTTOM ? */
    data = NULL;
    count = 0;
  }
  else if (PyTuple_CheckExact(odata)) {
    PyObject *dataptr;
    if (PyTuple_GET_SIZE(odata) != 2) {
      PyErr_SetString(PyExc_TypeError,
		      "data must be a 2-tuple with "
		      "(data pointer integer|string, read-only flag)");
      goto fail;
    }
    /* get pointer */
    dataptr = PyTuple_GET_ITEM(odata, 0);
    if (PyInt_Check(dataptr) || PyLong_Check(dataptr)) {
      data = (char *) PyLong_AsVoidPtr(dataptr);
      if (PyErr_Occurred()) goto fail;
    }
    else if (PyString_Check(dataptr)) {
      int res; void *ptr = NULL;
      const char *str = PyString_AsString(dataptr);
      /*res = sscanf(str, "%p", (void **)&data);*/
      res = sscanf(str, "%p", &ptr);
      if (res < 1) {
	PyErr_Format(PyExc_ValueError, 
		     "data pointer string '%.20s' cannot be "
		     "converted to pointer", str);
	goto fail;
      }
      data = (char*) ptr;
    }
    else {
      PyErr_Format(PyExc_TypeError, 
		   "first element of data tuple "
		   "must be integer or string, got %.200s",
		   dataptr->ob_type->tp_name); 
      goto fail;
    }
    /* check for read-only */
    if (writable && PyObject_IsTrue(PyTuple_GET_ITEM(odata, 1))) {
      PyErr_SetString(PyExc_ValueError,
		      "read-only flag in data tuple is True"); 
      goto fail;
    }
    /* check count */
    if (count < 0) {
      PyErr_SetString(PyExc_ValueError, 
		      "data tuple requires a "
		      "nonnegative integer count");
      goto fail;
    }
  }
  else {
    /* get buffer pointer and length */
    int res;
    void *bufptr = NULL;
    Py_ssize_t buflen = 0;
    if (writable)
      res = PyObject_AsWriteBuffer(odata, (void **)&bufptr, &buflen);
    else
      res = PyObject_AsReadBuffer(odata, (const void **)&bufptr, &buflen);
    if (res < 0) goto fail;
    data = (char *) bufptr;
    /* guess count if not provided or negative */
    if (count < 0) {
      Py_ssize_t numitems;
      int itemsize;
      MPI_Aint lb, extent;
      MPI4PY_CALL( MPI_Type_size(datatype, &itemsize) );
#if HAVE_MPI_TYPE_GET_EXTENT
      MPI_Type_get_extent(datatype, &lb, &extent);
#else
      MPI_Type_lb(datatype, &lb);
      MPI_Type_extent(datatype, &extent);
#endif
      if (lb != 0 || extent == 0 || itemsize == 0) {
	PyErr_Format(PyExc_ValueError,
		     "cannot guess count from a datatype "
		     "with lower bound %d, extent %d and size %d",
		     (int)lb, (int)extent, itemsize);
	goto fail;
      }
      else if (extent != (MPI_Aint)itemsize) {
	PyErr_Format(PyExc_ValueError,
		     "cannot guess count from a datatype "
		     "with different extent %d and size %d",
		     (int)extent, itemsize);
	goto fail;
      }
      numitems = buflen / (Py_ssize_t) itemsize;
      if (segments > 0) {
        Py_ssize_t numsegs = (Py_ssize_t) segments;
        if (numitems % numsegs) {
          PyErr_Format(PyExc_ValueError,
                       "number of items %d not divisible "
                       "by required number of segments %d",
                       (int) numitems, (int)numsegs);
          goto fail;
        }
        numitems /= numsegs;
      }
      count = Py_SAFE_DOWNCAST(numitems, Py_ssize_t, int);
    }
  }

  Py_DECREF(args);
  buf->buf   = (void *) data;
  buf->count = count;
  buf->dtype = datatype;
  return 1;

 fail:
  Py_XDECREF(args);
  buf->buf   = NULL;
  buf->count = 0;
  buf->dtype = MPI_DATATYPE_NULL;
  return 0;
}

#define as_buffer_read(o,b)      as_buffer(0,o,b,0)
#define as_buffer_write(o,b)     as_buffer(1,o,b,0)
#define as_buffer_read_b(o,b,s)  as_buffer(0,o,b,s)
#define as_buffer_write_b(o,b,s) as_buffer(1,o,b,s)

static int
as_buffer_ignore(PyObject *obj, buffer_t *buf)
{
  /* MPI_DATATYPE_NULL can cause problems in sends/receives with
     dest/source = MPI_PROC_NULL in some implementations, eg. MPICH2.
     The standard does not explicitly mandate that usage of
     MPI_PROC_NULL implies that 'datatype' must be ignored :(.
     We set 'datatype' to MPI_BYTE, I hope it will not fail.
  */
  buf->buf   = NULL;
  buf->count = 0;
  buf->dtype = MPI_BYTE;
  return 1;
}

#define TestForInPlace(o) \
        (((o) == Py_None) || PyMPIAint_Check(o))

static int
as_inplace(PyObject *obj, void **ptr)
{
  TRY {
#if HAVE_MPI_IN_PLACE
    if (obj != Py_None) {
      MPI_Aint inplace = PyMPIAint_AsAint(obj);
      if (PyErr_Occurred()) RAISE;
      if (MPI_IN_PLACE != (void *) inplace)
	PY_RAISE(PyExc_ValueError,
		 "expecting MPI_IN_PLACE");
    }
    if (ptr) *ptr = MPI_IN_PLACE;
    return 1;
#else
    MPI4PY_RAISE_MPINOIMPL("MPI_IN_PLACE");
#endif
  }
  EXCEPT {
    return 0;
  }
}

static int
as_buffer_inplace(PyObject *obj, buffer_t *buf)
{
  /* initialization */
  buf->buf = NULL;
  buf->count = 0;
  buf->dtype = MPI_DATATYPE_NULL;
  /* set buffer pointer */
  return as_inplace(obj, &buf->buf);
}

/*------------------------------------------------------------------*/


/*------------------------------------------------------------------*/


#if !defined(MPI4PY_PICKLED)
#define MPI4PY_PICKLED MPI_BYTE
#endif

#if PY_VERSION_HEX < 0x03000000
#define PyMPIPickled_Check             PyString_Check
#define PyMPIPickled_AS_STRING         PyString_AS_STRING
#define PyMPIPickled_GET_SIZE          PyString_GET_SIZE
#define PyMPIPickled_FromStringAndSize PyString_FromStringAndSize
#else
#define PyMPIPickled_Check             PyBytes_Check
#define PyMPIPickled_AS_STRING         PyBytes_AS_STRING
#define PyMPIPickled_GET_SIZE          PyBytes_GET_SIZE
#define PyMPIPickled_FromStringAndSize PyBytes_FromStringAndSize
#endif

static int
arg_pickled_get(PyObject *arg, void** buf, int* len)
{
  if (PyMPIPickled_Check(arg)) {
    char *bptr = PyMPIPickled_AS_STRING(arg);
    Py_ssize_t size = PyMPIPickled_GET_SIZE(arg);
    *buf = (void*) bptr;
    *len = Py_SAFE_DOWNCAST(size, Py_ssize_t, int);
    return 1;
  }
  PyErr_SetString(PyExc_TypeError, "expecting a string or bytes");
  return 0;
}

static PyObject *
arg_pickled_out(PyObject *arg, int size, void** buf, int* len)
{
  PyObject *result = NULL;
  result = PyMPIPickled_FromStringAndSize(NULL, (Py_ssize_t)size);
  if (result != NULL) {
    char *bptr = PyMPIPickled_AS_STRING(result);
    Py_ssize_t size = PyMPIPickled_GET_SIZE(result);
    *buf = (void*) bptr;
    *len = Py_SAFE_DOWNCAST(size, Py_ssize_t, int);
  }
  return result;
}

/*------------------------------------------------------------------*/


/*------------------------------------------------------------------*/
/* Point-to-Point Communications                                    */
/*------------------------------------------------------------------*/


/*------------------------------------------------------------------*/
/* Send, Recv, Sendrecv                                             */
/*------------------------------------------------------------------*/
static PyObject *
comm_send_pickled(PyObject *self, PyObject *args)
{  
  PyObject *sendobj  = NULL;
  
  void *buf = NULL;
  int count = 0;
  int dest, tag;
  MPI_Comm comm;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OiiO&", 
			 &sendobj,
			 &dest, 
                         &tag,
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    /* get send buffer */
    if (dest != MPI_PROC_NULL) {
      if(!arg_pickled_get(sendobj, &buf, &count)) RAISE;
    }
    /* send string length */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Send(&count, 1, MPI_INT, dest, tag, comm) );
    /* send string buffer */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Send(buf, count, MPI4PY_PICKLED, 
			      dest, tag, comm) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_recv_pickled(PyObject *self, PyObject *args)
{
  PyObject *recvobj = NULL;
  PyObject *result = NULL;

  void *buf = NULL;
  int count = 0;
  int source, tag;
  MPI_Comm comm;
  MPI_Status *status;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OiiO&O&",
			 &recvobj,
			 &source,
			 &tag,
			 PyO2Cm, &comm,
			 PyO2SN, &status) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    MPI_Status recvsts;
    /* recv string size */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Recv(&count, 1, MPI_INT,
			      source, tag, comm, &recvsts) );
    /* allocate/reuse receive buffer */
    if (source != MPI_PROC_NULL) {
      result = arg_pickled_out(NULL, count, &buf, &count);
      if (result == NULL) RAISE;
      source = recvsts.MPI_SOURCE;
      tag = recvsts.MPI_TAG;
    }
    else {
      result = (Py_INCREF(Py_None), Py_None);
    }
    /* recv string buffer */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Recv(buf, count, MPI4PY_PICKLED,
			      source, tag, comm, status) );
    /* return result */
    return result;
  }
  EXCEPT {
    Py_XDECREF(result);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_sendrecv_pickled(PyObject *self, PyObject *args)
{  
  PyObject *sendobj  = NULL;
  PyObject *recvobj = NULL;
  PyObject *result = NULL;

  void *sendbuf = NULL;
  int sendcount = 0;
  int dest, sendtag;
  void *recvbuf = NULL;
  int recvcount = 0;
  int source, recvtag;
  MPI_Comm comm;
  MPI_Status *status;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OiiOiiO&O&", 
			 &sendobj, &dest,   &sendtag,
			 &recvobj, &source, &recvtag,
			 PyO2Cm, &comm,
			 PyO2SN, &status) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
    
  TRY {
    MPI_Status recvsts;
    /* get send buffer */
    if (dest != MPI_PROC_NULL) {
      if(!arg_pickled_get(sendobj, &sendbuf, &sendcount)) RAISE;
    }
    /* sendrecv string lengths */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Sendrecv(&sendcount, 1, MPI_INT, dest,   sendtag,
				  &recvcount, 1, MPI_INT, source, recvtag,
				  comm, &recvsts) );
    /* allocate receive buffer */
    if (source != MPI_PROC_NULL) {
      result = arg_pickled_out(NULL, recvcount, &recvbuf, &recvcount);
      if (result == NULL) RAISE;
      source = recvsts.MPI_SOURCE;
      recvtag = recvsts.MPI_TAG;
    }
    else {
      result = (Py_INCREF(Py_None), Py_None);
    }
    /* sendrecv string buffers */
    MPI4PY_CALL_COMM(comm,
		     MPI_Sendrecv(sendbuf, sendcount, MPI4PY_PICKLED, 
				  dest,  sendtag,
				  recvbuf, recvcount, MPI4PY_PICKLED,
				  source, recvtag,
				  comm, status) );
    /* return result */
    return result;
  }
  EXCEPT {
    Py_XDECREF(result);
    return NULL;
  }
  
}
/*------------------------------------------------------------------*/
static PyObject *
comm_send_buffer(PyObject *self, PyObject *args)
{  
  PyObject *sendobj;
  int dest, tag;
  MPI_Comm comm;

  char mode = '\0';

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OiiO&|c", 
			 &sendobj, &dest, &tag, 
			 PyO2Cm, &comm,
			 &mode) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    buffer_t sbuf;
    /* get buffer data */
    if (dest != MPI_PROC_NULL) {
      if (!as_buffer_read(sendobj, &sbuf)) RAISE;
    }
    else {
      if (!as_buffer_ignore(sendobj, &sbuf)) RAISE;
    }
    /* send buffer data */
    switch (mode) {
    case '\0':
      MPI4PY_CALL_COMM(comm,
		       MPI_Send(sbuf.buf, sbuf.count, sbuf.dtype,
				dest, tag, comm) ); break;
    case 'b': case 'B':
      MPI4PY_CALL_COMM(comm,
		       MPI_Bsend(sbuf.buf, sbuf.count, sbuf.dtype,
				 dest, tag, comm) ); break;
    case 's': case 'S':
      MPI4PY_CALL_COMM(comm,
		       MPI_Ssend(sbuf.buf, sbuf.count, sbuf.dtype,
				 dest, tag, comm) ); break;
    case 'r': case 'R':
      MPI4PY_CALL_COMM(comm,
		       MPI_Rsend(sbuf.buf, sbuf.count, sbuf.dtype,
				 dest, tag, comm) ); break;
    default:
      PY_RAISE(PyExc_ValueError, "mode switch not understood");
    }
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_recv_buffer(PyObject *self, PyObject *args)
{
  PyObject *recvobj;
  int source, tag;
  MPI_Comm comm;
  MPI_Status *status;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OiiO&O&",
			 &recvobj, &source, &tag,
			 PyO2Cm, &comm,
			 PyO2SN, &status) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    buffer_t rbuf;
    /* get buffer data */
    if (source != MPI_PROC_NULL) {
      if (!as_buffer_write(recvobj, &rbuf)) RAISE;
    }
    else {
      if (!as_buffer_ignore(recvobj, &rbuf)) RAISE;
    }
    /* receive buffer data */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Recv(rbuf.buf, rbuf.count, rbuf.dtype,
			      source, tag, comm, status) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_sendrecv_buffer(PyObject *self, PyObject *args)
{  
  PyObject *sendobj;
  int dest, sendtag;
  PyObject *recvobj;
  int source, recvtag;
  MPI_Comm comm;
  MPI_Status *status;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OiiOiiO&O&",
			 &sendobj,  &dest,   &sendtag,
			 &recvobj, &source, &recvtag,
			 PyO2Cm, &comm,
			 PyO2SN, &status) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
    
  TRY {
    buffer_t sbuf;
    buffer_t rbuf;
    /* get buffer data*/
    if (dest != MPI_PROC_NULL) {
      if (!as_buffer_read(sendobj, &sbuf)) RAISE;
    }
    else {
      if (!as_buffer_ignore(sendobj, &sbuf)) RAISE;
    }
    if (source != MPI_PROC_NULL) {
      if (!as_buffer_write(recvobj, &rbuf)) RAISE;
    }
    else {
      if (!as_buffer_ignore(sendobj, &rbuf)) RAISE;
    }
    /* sendrecv buffer data */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Sendrecv(sbuf.buf, sbuf.count, sbuf.dtype,
				  dest, sendtag,
				  rbuf.buf, rbuf.count, rbuf.dtype,
				  source, recvtag,
				  comm, status) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }
  
}
/*------------------------------------------------------------------*/
static PyObject *
comm_sendrecv_replace_buffer(PyObject *self, PyObject *args)
{  
  PyObject *object;
  int dest, sendtag;
  int source, recvtag;
  MPI_Comm comm;
  MPI_Status *status;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OiiiiO&O&",
			 &object,  
			 &dest,   &sendtag, 
			 &source, &recvtag,
			 PyO2Cm, &comm,
			 PyO2SN, &status) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
    
  TRY {
    buffer_t buf;
    /* get buffer data*/
    if (dest == MPI_PROC_NULL && source == MPI_PROC_NULL) {
      if (!as_buffer_ignore(object, &buf)) RAISE;
    } else {
      if (!as_buffer_write(object, &buf)) RAISE;
    }
    /* sendrecv buffer data */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Sendrecv_replace(buf.buf,
					  buf.count,
					  buf.dtype,
					  dest, sendtag,
					  source, recvtag,
					  comm, status) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }
  
}
/*------------------------------------------------------------------*/


/*------------------------------------------------------------------*/
/* Isend, Irecv                                                     */
/*------------------------------------------------------------------*/
static PyObject *
comm_isend_buffer(PyObject *self, PyObject *args)
{  
  PyObject *input;
  int dest, tag;
  MPI_Comm comm;
  MPI_Request request;

  char mode = '\0';

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OiiO&|c",
			 &input, &dest, &tag,
			 PyO2Cm, &comm,
			 &mode) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    buffer_t sbuf;
    /* get buffer data */
    if (dest != MPI_PROC_NULL) {
      if (!as_buffer_read(input, &sbuf)) RAISE;
    }
    else {
      if (!as_buffer_ignore(input, &sbuf)) RAISE;
    }
    /* send buffer data */
    switch (mode) {
    case '\0':
      MPI4PY_CALL_COMM(comm, 
		       MPI_Isend(sbuf.buf, sbuf.count, sbuf.dtype,
				 dest, tag, comm, &request) ); break;
    case 'b': case 'B':
      MPI4PY_CALL_COMM(comm, 
		       MPI_Ibsend(sbuf.buf, sbuf.count, sbuf.dtype,
				  dest, tag, comm, &request) ); break;
    case 's': case 'S':
      MPI4PY_CALL_COMM(comm, 
		       MPI_Issend(sbuf.buf, sbuf.count, sbuf.dtype,
				  dest, tag, comm, &request) ); break;
    case 'r': case 'R':
      MPI4PY_CALL_COMM(comm, 
		       MPI_Irsend(sbuf.buf, sbuf.count, sbuf.dtype,
				  dest, tag, comm, &request) ); break;
    default:
      PY_RAISE(PyExc_ValueError, "mode switch not understood");

    }
    /* return request */
    return PyMPIRequest_FromRequest(request);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_irecv_buffer(PyObject *self, PyObject *args)
{
  PyObject *recvobj;
  int source, tag;
  MPI_Comm comm;
  MPI_Request request;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OiiO&",
			 &recvobj, &source, &tag,
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    buffer_t rbuf;
    /* get buffer data */
    if (source != MPI_PROC_NULL) {
      if (!as_buffer_write(recvobj, &rbuf)) RAISE;
    }
    else {
      if (!as_buffer_ignore(recvobj, &rbuf)) RAISE;
    }
    /* receive buffer data */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Irecv(rbuf.buf, rbuf.count, rbuf.dtype,
			       source, tag, comm, &request) );
    /* return request */
    return PyMPIRequest_FromRequest(request);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/



/*------------------------------------------------------------------*/
/* Send_init, Recv_init                                             */
/*------------------------------------------------------------------*/
static PyObject *
comm_send_init_buffer(PyObject *self, PyObject *args)
{  
  PyObject *input;
  int dest, tag;
  MPI_Comm comm;
  MPI_Request request;
  
  char mode = '\0';

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OiiO&|c",
			 &input, &dest, &tag,
			 PyO2Cm, &comm,
			 &mode) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    buffer_t sbuf;
    /* get buffer data */
    if (dest != MPI_PROC_NULL) {
      if (!as_buffer_read(input, &sbuf)) RAISE;
    }
    else {
      if (!as_buffer_ignore(input, &sbuf)) RAISE;
    }
    /*  send init buffer data */
    switch (mode) {
    case '\0':
      MPI4PY_CALL_COMM(comm,
		       MPI_Send_init(sbuf.buf, sbuf.count, sbuf.dtype,
				     dest, tag, comm, &request) ); break;
    case 'b': case 'B':
      MPI4PY_CALL_COMM(comm,
		       MPI_Bsend_init(sbuf.buf, sbuf.count, sbuf.dtype,
				      dest, tag, comm, &request) ); break;
      
    case 's': case 'S':
      MPI4PY_CALL_COMM(comm,
		       MPI_Ssend_init(sbuf.buf, sbuf.count, sbuf.dtype,
				      dest, tag, comm, &request) ); break;
    case 'r': case 'R':
      MPI4PY_CALL_COMM(comm,
		       MPI_Rsend_init(sbuf.buf, sbuf.count, sbuf.dtype,
				      dest, tag, comm, &request) ); break;
    default:
      PY_RAISE(PyExc_ValueError, "mode switch not understood");
    }
    /* return request */
    return PyMPIRequest_FromRequest(request);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_recv_init_buffer(PyObject *self, PyObject *args)
{
  PyObject *recvobj;
  int source, tag;
  MPI_Comm comm;
  MPI_Request request;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OiiO&",
			 &recvobj, &source, &tag,
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    buffer_t rbuf;
    /* get buffer data */
    if (source != MPI_PROC_NULL) {
      if (!as_buffer_write(recvobj, &rbuf)) RAISE;
    }
    else {
      if (!as_buffer_ignore(recvobj, &rbuf)) RAISE;
    }
    /* receive buffer data */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Recv_init(rbuf.buf, rbuf.count, rbuf.dtype,
				   source, tag, comm, &request) );
    /* return request */
    return PyMPIRequest_FromRequest(request);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/




/*------------------------------------------------------------------*/
/* Probe, Iprobe                                                    */
/*------------------------------------------------------------------*/
static PyObject *
comm_probe(PyObject *self, PyObject *args)
{
  int source, tag;
  MPI_Comm    comm;
  MPI_Status *status;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "iiO&O&",
			 &source, &tag,
			 PyO2Cm,  &comm,
			 PyO2SN, &status) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    /* probe */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Probe(source, tag, comm, status) );
    /* return */
    Py_RETURN_NONE;
  }  
  EXCEPT {
    return NULL;
  }
 
}
/*------------------------------------------------------------------*/
static PyObject *
comm_iprobe(PyObject *self, PyObject *args)
{
  int source, tag;
  MPI_Comm comm;
  int flag;
  MPI_Status *status;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "iiO&O&",
			 &source, &tag,
			 PyO2Cm,  &comm,
			 PyO2SN, &status) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    /* iprobe */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Iprobe(source, tag, comm, &flag, status) );
    /* return flag */
    return PyBool_FromLong((long)flag);
  }  
  EXCEPT {
    return NULL;
  }
 
}
/*------------------------------------------------------------------*/





/*------------------------------------------------------------------*/
/* Collective Communications                                        */
/*------------------------------------------------------------------*/


/*------------------------------------------------------------------*/
/* Barrier                                                          */
/*------------------------------------------------------------------*/
static PyObject *
comm_barrier(PyObject *self, PyObject *args)
{  
  MPI_Comm comm;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
#if !HAVE_MPI_ROOT
    int flag; MPI4PY_CHECK( MPI_Comm_test_inter(comm, &flag) );
    if (flag) MPI4PY_RAISE_MPINOIMPL("extended collective communications");
#endif
    /* barrier synchronization */
    MPI4PY_CALL_COMM(comm,
		     MPI_Barrier(comm) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/


/*------------------------------------------------------------------*/
/* Bcast                                                            */
/*------------------------------------------------------------------*/
static PyObject *
comm_bcast_pickled(PyObject *self, PyObject *args)
{
  PyObject *input  = NULL;
  PyObject *result = NULL;
  
  void *buffer = NULL;
  int count = 0;
  int root; 
  MPI_Comm comm;


  /* parse input arguments */
  if (!PyArg_ParseTuple(args, "OiO&",
			&input, &root, 
			PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    int flag;
    MPI4PY_CHECK( MPI_Comm_test_inter(comm, &flag) );
    if (!flag) { /* intra-communication */
      int rank;
      MPI4PY_CHECK( MPI_Comm_rank(comm, &rank) );
      if (root == rank) /* root process sends */
	result = (Py_INCREF(input), input); /* bcast is in-place */
      else /* other processes receive */
	input = NULL;
    } 
    else { /* inter-communication */
#if HAVE_MPI_ROOT
      if (root == MPI_ROOT) { 
	/* this process sends, but does not receive */
	result = (Py_INCREF(Py_None), Py_None);
      } 
      else if (root == MPI_PROC_NULL) {
	/* this process does not send, nor receives */
	input = NULL;
	result = (Py_INCREF(Py_None), Py_None);
      } 
      else {
	/* this process does not send, but receives */
	input = NULL;
      }
#else
      MPI4PY_RAISE_MPINOIMPL("extended collective communications");
#endif
    }

    /* get input buffer at sending process */
    if (input != NULL) {
      if(!arg_pickled_get(input, &buffer, &count)) RAISE;
    }
    /* bcast string length */
    MPI4PY_CALL_COMM(comm,
		     MPI_Bcast(&count, 1, MPI_INT, root, comm) );
    /* allocate result buffer at receiving process */
    if (result == NULL) {
      result = arg_pickled_out(NULL, count, &buffer, &count);
      if (result == NULL) RAISE;
    }
    /* bcast string buffer */
    MPI4PY_CALL_COMM(comm,
		     MPI_Bcast(buffer, count, MPI4PY_PICKLED,
			       root, comm) );
    /* return result string */
    return result;
  }
  EXCEPT {
    Py_XDECREF(result);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_bcast_buffer(PyObject *self, PyObject *args)
{
  PyObject *input;

  int root; 
  MPI_Comm comm;

  /* parse input arguments */
  if (!PyArg_ParseTuple(args, "OiO&",
			&input, &root,
			PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    int flag;
    buffer_t buf;
    MPI4PY_CHECK( MPI_Comm_test_inter(comm, &flag) );
    if (!flag) { /* intra-communication */
      int rank; 
      MPI4PY_CHECK( MPI_Comm_rank(comm, &rank) );
      /* get buffer data */
      if (root == rank) {
	if (!as_buffer_read (input, &buf)) RAISE;
      } else { 
	if (!as_buffer_write(input, &buf)) RAISE;
      }
    } else { /* inter-communication */
#if HAVE_MPI_ROOT
      if (root == MPI_ROOT) {
	if (!as_buffer_read(input, &buf)) RAISE;
      } 
      else if (root == MPI_PROC_NULL) {
	if (!as_buffer_ignore(input, &buf)) RAISE;
      } 
      else {
	if (!as_buffer_write(input, &buf)) RAISE;
      }
#else
      MPI4PY_RAISE_MPINOIMPL("extended collective communications");
#endif
    }
    /* bcast buffer data */
    MPI4PY_CALL_COMM(comm,
		     MPI_Bcast(buf.buf, buf.count, buf.dtype,
			       root, comm) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/


/*------------------------------------------------------------------*/
/* Gather                                                           */
/*------------------------------------------------------------------*/
static PyObject *
comm_gather_pickled(PyObject *self, PyObject *args)
{  
  PyObject *input  = NULL;
  PyObject *output = NULL;
  PyObject *result = NULL;

  void *sbuff = NULL;
  int scount = 0;
  void *rbuff = NULL;
  int *rcounts = NULL;
  int *displs = NULL;
  int root;
  MPI_Comm comm;
  

  /* parse input arguments */
  if (!PyArg_ParseTuple(args, "OOiO&",
			&input, &output,
			&root, 
			PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
    
  TRY {

    int flag, size;
    MPI4PY_CHECK( MPI_Comm_test_inter(comm, &flag) );
    if (!flag) { /* intra-communication */
      int rank;
      /* get comm size & rank */
      MPI4PY_CHECK( MPI_Comm_size(comm, &size) );
      MPI4PY_CHECK( MPI_Comm_rank(comm, &rank) );
      if (root == rank) {
	/* this process sends and receives */
	result = NULL;
      }
      else {
	/* this process send, but does not receive */
	result = (Py_INCREF(Py_None), Py_None);
      }
    }
    else { /* inter-communication */
#if HAVE_MPI_ROOT
      /* get remote comm size */
      MPI4PY_CHECK( MPI_Comm_remote_size(comm, &size) );
      if (root == MPI_ROOT) {
	/* this process does not send, but receive */
	input = NULL;
	result = NULL;
      }
      else if (root == MPI_PROC_NULL) {
	/* this process does not send, nor receives */
	input = NULL;
	result = (Py_INCREF(Py_None), Py_None);
      }
      else {
	/* this process send, but does not receive */
	result = (Py_INCREF(Py_None), Py_None);
      }
#else
      MPI4PY_RAISE_MPINOIMPL("extended collective communications");
#endif
    }

    /* get input buffer at sending processes */
    if (input != NULL) {
      if(!arg_pickled_get(input, &sbuff, &scount)) RAISE;
    }
    /* allocate arrays 'rcounts' and 'displs' at receiving process */
    if (result == NULL) {
      rcounts = PyMem_New(int, size);
      if (rcounts == NULL) PY_RAISE(PyExc_MemoryError, "");
      displs  = PyMem_New(int, size);
      if (displs == NULL ) PY_RAISE(PyExc_MemoryError, "");
    }
    /* gather string lengths */
    MPI4PY_CALL_COMM(comm,
		     MPI_Gather(&scount, 1, MPI_INT,
				rcounts, 1, MPI_INT,
				root, comm) );
    /* create and fill result list at receiving process */
    if (result == NULL) {
      int i;
      result = PyList_New((Py_ssize_t)size);
      if (result == NULL) RAISE;
      for (i = 0; i < size; i++) {
        PyObject  *item = NULL;
        void      *bptr = NULL;
        int        blen = 0;
        Py_ssize_t bdsp = 0;
	/* create output object */
        item = arg_pickled_out(NULL, rcounts[i], &bptr, &blen);
	if (item == NULL) RAISE;
	/* set buffer in result list */
	PyList_SET_ITEM(result, i, item);
	/* calculate 'rbuff' pointer and 'displs' array */
	if (i == 0) rbuff = bptr;
        bdsp = ((char*)bptr) - ((char*)rbuff);
	displs[i] = Py_SAFE_DOWNCAST(bdsp, Py_ssize_t, int);
      }
    }
    /* gather string buffers */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Gatherv(sbuff, scount,          MPI4PY_PICKLED,
				 rbuff, rcounts, displs, MPI4PY_PICKLED,
				 root, comm) );
    /* deallocate memory */
    PyMem_Del(rcounts);
    PyMem_Del(displs);
    /* return result */
    return  result;
  }
  EXCEPT {
    PyMem_Del(rcounts);
    PyMem_Del(displs);
    Py_XDECREF(result);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_gather_buffer(PyObject *self, PyObject *args)
{
  PyObject *input;
  PyObject *output;
  int root;
  MPI_Comm comm;

  /* parse input arguments */
  if (!PyArg_ParseTuple(args, "OOiO&",
			&input, 
			&output,
			&root,
			PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    int flag;
    buffer_t sbuf;
    buffer_t rbuf;
    MPI4PY_CHECK( MPI_Comm_test_inter(comm, &flag) );
    if (!flag) { /* intra-communication */
      int size, rank;
      /* get comm size and rank */
      MPI4PY_CHECK( MPI_Comm_size(comm, &size) );
      MPI4PY_CHECK( MPI_Comm_rank(comm, &rank) );
      /* get buffer data */
      if (rank == root) {
	if (TestForInPlace(input)) {
	  if (!as_buffer_inplace(input, &sbuf)) RAISE;
	} else {
	  if (!as_buffer_read(input, &sbuf)) RAISE;
	}
	if (!as_buffer_write_b(output, &rbuf, size)) RAISE;
#       ifdef OPEN_MPI
	if (sbuf.buf == MPI_IN_PLACE) sbuf.count = rbuf.count;
#       endif
      } else {
	if (!as_buffer_ignore(output, &rbuf)) RAISE;
	if (!as_buffer_read  (input,  &sbuf)) RAISE;
      }
    } else { /* inter-communication */
#if HAVE_MPI_ROOT
      int size;
      /* get remote comm size */
      MPI4PY_CHECK( MPI_Comm_remote_size(comm, &size) );
      if (root == MPI_ROOT) {
	if (!as_buffer_ignore (input,  &sbuf      )) RAISE;
	if (!as_buffer_write_b(output, &rbuf, size)) RAISE;
      } 
      else if (root == MPI_PROC_NULL) {
	if (!as_buffer_ignore(input,  &sbuf)) RAISE;
	if (!as_buffer_ignore(output, &rbuf)) RAISE;
      } 
      else {
	if (!as_buffer_read  (input,  &sbuf)) RAISE;
	if (!as_buffer_ignore(output, &rbuf)) RAISE;
      }
#else
      MPI4PY_RAISE_MPINOIMPL("extended collective communications");
#endif
    }
    /* gather buffer data */
    MPI4PY_CALL_COMM(comm,
		     MPI_Gather(sbuf.buf, sbuf.count, sbuf.dtype,
				rbuf.buf, rbuf.count, rbuf.dtype,
				root, comm) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/


/*------------------------------------------------------------------*/
/* Scatter                                                          */
/*------------------------------------------------------------------*/
static PyObject *
comm_scatter_pickled(PyObject *self, PyObject *args)
{
  PyObject *input  = NULL;
  PyObject *output = NULL;
  PyObject *result = NULL;

  void *sbuff = NULL;
  int *scounts = NULL;
  int *displs = NULL;
  void *rbuff = NULL;
  int rcount  = 0;
  int root; 
  MPI_Comm comm;
  

  /* parse input arguments */
  if (!PyArg_ParseTuple(args, "OOiO&",
			&input, &output,
			&root, 
			PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);

  TRY {
    int flag, size;
    MPI4PY_CHECK( MPI_Comm_test_inter(comm, &flag) );
    if (!flag) { /* intra-communication */
      int rank;
      MPI4PY_CHECK( MPI_Comm_size(comm, &size) );
      MPI4PY_CHECK( MPI_Comm_rank(comm, &rank) );
      /* only root process sends, all receive */
      if (root != rank) input = NULL;
    } 
    else { /* inter-communication */
#if HAVE_MPI_ROOT
      MPI4PY_CHECK( MPI_Comm_remote_size(comm, &size) );
      if (root == MPI_ROOT) { 
	/* this process sends, but does not receive */
	result = (Py_INCREF(Py_None), Py_None);
      } 
      else if (root == MPI_PROC_NULL) {
	/* this process does not send, nor receives */
	input = NULL;
	result = (Py_INCREF(Py_None), Py_None);
      } 
      else {
	/* this process does not send, but receives */
	input = NULL;
      }
#else
      MPI4PY_RAISE_MPINOIMPL("extended collective communications");
#endif
    }
    
    /* get input strings at sending process */
    if (input != NULL) {
      int i;
      /* check input object */
      if (!PyList_Check(input))
	PY_RAISE(PyExc_TypeError, "expecting a list.");
      if (size != PyList_GET_SIZE(input) )
	PY_RAISE(PyExc_ValueError, "invalid list length.");
      /* allocate arrays 'scounts' and 'displs' */
      scounts = PyMem_New(int, size);
      if (scounts == NULL) PY_RAISE(PyExc_MemoryError, "");
      displs  = PyMem_New(int, size);
      if (displs == NULL) PY_RAISE(PyExc_MemoryError, "");
      /* calculate 'sbuff', 'scounts' and 'displs' */
      for (i = 0; i < size; i++) {
	PyObject  *item = NULL;
        void      *bptr = NULL;
        int        blen = 0;
        Py_ssize_t bdsp = 0;
        item = PyList_GET_ITEM(input, i);
        if(!arg_pickled_get(item, &bptr, &blen)) RAISE;
        if (i == 0) sbuff = bptr;
        bdsp = ((char*)bptr) - ((char*)sbuff);
	scounts[i] = blen;
	displs[i]  = Py_SAFE_DOWNCAST(bdsp, Py_ssize_t, int);
      }
    }
    /* scatter string lengths */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Scatter(scounts, 1, MPI_INT,
				 &rcount, 1, MPI_INT,
				 root, comm) );
    /* allocate result string at receiving processes */
    if (result == NULL) {
      result = arg_pickled_out(NULL, rcount, &rbuff, &rcount);
      if (result == NULL) RAISE;
    }
    /* scatter string buffers */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Scatterv(sbuff, scounts, displs, MPI4PY_PICKLED,
				  rbuff, rcount,          MPI4PY_PICKLED,
				  root, comm) );
    /* deallocate memory */
    PyMem_Del(scounts);
    PyMem_Del(displs);
    /* return result */
    return result;
  }
  EXCEPT {
    PyMem_Del(scounts);
    PyMem_Del(displs);
    Py_XDECREF(result);
    return NULL;
  }    
  
}
/*------------------------------------------------------------------*/
static PyObject *
comm_scatter_buffer(PyObject *self, PyObject *args)
{
  PyObject *input;
  PyObject *output;
  int root;
  MPI_Comm comm;

  /* parse input arguments */
  if (!PyArg_ParseTuple(args, "OOiO&",
			&input, &output,
			&root,
			PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    int flag;
    buffer_t sbuf;
    buffer_t rbuf;
    MPI4PY_CHECK( MPI_Comm_test_inter(comm, &flag) );
    if (!flag) { /* intra-communication */
      int size, rank;
      /* get comm size and rank */
      MPI4PY_CHECK( MPI_Comm_size(comm, &size) );
      MPI4PY_CHECK( MPI_Comm_rank(comm, &rank) );
      /* get buffer data */
      if (rank == root) {
	if (!as_buffer_read_b(input, &sbuf, size)) RAISE;
	if (TestForInPlace(output)) {
	  if (!as_buffer_inplace(output, &rbuf)) RAISE;
	} else {
	  if (!as_buffer_write  (output, &rbuf)) RAISE;
	}
#       ifdef OPEN_MPI
	if (rbuf.buf == MPI_IN_PLACE) rbuf.count = sbuf.count;
#       endif
      } else {
	if (!as_buffer_ignore(input,  &sbuf)) RAISE;
	if (!as_buffer_write (output, &rbuf)) RAISE;
      }
    } else { /* inter-communication */
#if HAVE_MPI_ROOT
      int size;
      /* get remote comm size */
      MPI4PY_CHECK( MPI_Comm_remote_size(comm, &size) );
      if (root == MPI_ROOT) {
	if (!as_buffer_read_b(input,  &sbuf, size)) RAISE;
	if (!as_buffer_ignore(output, &rbuf)) RAISE;
      } 
      else if (root == MPI_PROC_NULL) {
	if (!as_buffer_ignore(input,  &sbuf)) RAISE;
	if (!as_buffer_ignore(output, &rbuf)) RAISE;
      } 
      else {
	if (!as_buffer_ignore(input,  &sbuf)) RAISE;
	if (!as_buffer_read  (output, &rbuf)) RAISE;
      }
#else
      MPI4PY_RAISE_MPINOIMPL("extended collective communications");
#endif
    }
    /* scatter buffer data */
    MPI4PY_CALL_COMM(comm,
		     MPI_Scatter(sbuf.buf, sbuf.count, sbuf.dtype,
				 rbuf.buf, rbuf.count, rbuf.dtype,
				 root, comm) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/




/*------------------------------------------------------------------*/
/* Allgather                                                        */
/*------------------------------------------------------------------*/
static PyObject *
comm_allgather_pickled(PyObject *self, PyObject *args)
{  
  PyObject *input  = NULL;
  PyObject *output = NULL;
  PyObject *result = NULL;
  
  MPI_Comm comm;
  
  void* sbuff   = NULL;
  int   scount  = 0;
  void* rbuff   = NULL;
  int*  rcounts = NULL;
  int*  displs  = NULL; 

  /* parse input arguments */
  if (!PyArg_ParseTuple(args, "OOO&",
			&input, &output,
			PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
 
  TRY {

    int flag, size;
    MPI4PY_CHECK( MPI_Comm_test_inter(comm, &flag) );
    if (!flag) { /* intra-communication */
      /* get local comm size */
      MPI4PY_CHECK( MPI_Comm_size(comm, &size) );
    } 
    else { /* inter-communication */
      /* get remote comm size */
      MPI4PY_CHECK( MPI_Comm_remote_size(comm, &size) );
#if !HAVE_MPI_ROOT
      MPI4PY_RAISE_MPINOIMPL("extended collective communications");
#endif
    }

    /* allocate arrays 'rcounts' and 'displs' */
    rcounts = PyMem_New(int, size);
    displs  = PyMem_New(int, size);
    if (rcounts == NULL || displs == NULL )
      PY_RAISE(PyExc_MemoryError, "");
    
    /* get input object */
    if(!arg_pickled_get(input, &sbuff, &scount)) RAISE;
    
    /* gather string sizes */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Allgather(&scount, 1, MPI_INT,
				   rcounts, 1, MPI_INT,
				   comm) );
    
    /* create result list */
    result = PyList_New((Py_ssize_t)size);
    if (result == NULL) RAISE;
    /* fill result list */
    { int i; 
      for (i = 0; i < size; i++) {
        PyObject  *item = NULL;
        void      *bptr = NULL;
        int        blen = 0;
        Py_ssize_t bdsp = 0;
        /* create output object */
        item = arg_pickled_out(NULL, rcounts[i], &bptr, &blen);
        if (item == NULL) RAISE;
        /* set output in result list */
        PyList_SET_ITEM(result, i, item);
        /* calculate 'rbuff' pointer and 'displs' array */
        if (i == 0) rbuff = bptr;
        bdsp = ((char*)bptr) - ((char*)rbuff);
        displs[i] = Py_SAFE_DOWNCAST(bdsp, Py_ssize_t, int);
      }
    }
    
    /* gather string buffers */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Allgatherv(sbuff, scount,          MPI4PY_PICKLED,
				    rbuff, rcounts, displs, MPI4PY_PICKLED,
				    comm) );
    /* deallocate memory */
    PyMem_Del(rcounts);
    PyMem_Del(displs);
    
    /* return result */
    return  result;
    
  }
  
  EXCEPT {
    PyMem_Del(rcounts);
    PyMem_Del(displs);
    Py_XDECREF(result);
    return NULL;
  }
  
}
/*------------------------------------------------------------------*/
static PyObject *
comm_allgather_buffer(PyObject *self, PyObject *args)
{
  PyObject *input;
  PyObject *output;
  MPI_Comm comm;

  /* parse input arguments */
  if (!PyArg_ParseTuple(args, "OOO&",
			&input, &output,
			PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    int flag;
    buffer_t sbuf;
    buffer_t rbuf;
    MPI4PY_CHECK( MPI_Comm_test_inter(comm, &flag) );
    if (!flag) { /* intra-communication */
      int size;
      /* get comm size */
      MPI4PY_CHECK( MPI_Comm_size(comm, &size) );
      /* get buffer data */
      if (TestForInPlace(input)) {
	if (!as_buffer_inplace(input, &sbuf)) RAISE;
      } else {
	if (!as_buffer_read   (input, &sbuf)) RAISE;
      }
      if (!as_buffer_write_b(output, &rbuf, size)) RAISE;
#     ifdef OPEN_MPI
      if (sbuf.buf == MPI_IN_PLACE) sbuf.count = rbuf.count;
#     endif
    } 
    else { /* inter-communication */
#if HAVE_MPI_ROOT
      int size;
      /* get remote comm size */
      MPI4PY_CHECK( MPI_Comm_remote_size(comm, &size) );
      if (!as_buffer_read   (input,  &sbuf      )) RAISE;
      if (!as_buffer_write_b(output, &rbuf, size)) RAISE;
#else
      MPI4PY_RAISE_MPINOIMPL("extended collective communications");
#endif
    }
    /* gather buffer data */
    MPI4PY_CALL_COMM(comm,
		     MPI_Allgather(sbuf.buf, sbuf.count, sbuf.dtype,
				   rbuf.buf, rbuf.count, rbuf.dtype, 
				   comm) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/



/*------------------------------------------------------------------*/
/* Alltoall                                                         */
/*------------------------------------------------------------------*/
static PyObject *
comm_alltoall_pickled(PyObject *self, PyObject *args)
{
  PyObject *input  = NULL;
  PyObject *output = NULL;
  PyObject *result = NULL;
  
  MPI_Comm comm;
 
  void* sbuff   = NULL;
  int*  scounts = NULL;
  int*  sdispls = NULL;
  
  void* rbuff   = NULL;
  int*  rcounts = NULL;
  int*  rdispls = NULL; 

  /* parse input arguments */
  if (!PyArg_ParseTuple(args, "OOO&",
			&input, &output,
			PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {

    int flag, size;
    MPI4PY_CHECK( MPI_Comm_test_inter(comm, &flag) );
    if (!flag) { /* intra-communication */
      /* get comm size */
      MPI4PY_CHECK( MPI_Comm_size(comm, &size) );
    } 
    else { /* inter-communication */
      /* get remote comm size */
      MPI4PY_CHECK( MPI_Comm_remote_size(comm, &size) );
#if !HAVE_MPI_ROOT
      MPI4PY_RAISE_MPINOIMPL("extended collective communications");
#endif
    }

    /* allocate arrays 'scounts' and 'sdispls' */
    scounts = PyMem_New(int, size);
    sdispls = PyMem_New(int, size);
    if ( !scounts || !sdispls )
      PY_RAISE(PyExc_MemoryError, "");
    
    /* check input object */
    if ( !PyList_Check(input) )
      PY_RAISE(PyExc_TypeError, "expecting a list");
    /* check list length & comm size */
    if (size != PyList_GET_SIZE(input) )
      PY_RAISE(PyExc_ValueError, "invalid list length");
    
    /* calculate 'sbuff', 'scounts' and 'sdispls' */
    { 
      int i; 
      for (i = 0; i < size; i++) {
	PyObject  *item = NULL;
        void      *bptr = NULL;
        int        blen = 0;
        Py_ssize_t bdsp = 0;
        item = PyList_GET_ITEM(input, i);
        if(!arg_pickled_get(item, &bptr, &blen)) RAISE;
        if (i == 0) sbuff = bptr;
        bdsp = ((char*)bptr) - ((char*)sbuff);
	scounts[i] = blen;
	sdispls[i]  = Py_SAFE_DOWNCAST(bdsp, Py_ssize_t, int);
      }
    }
    
    /* allocate arrays 'rcounts' and 'rdispls' */
    rcounts = PyMem_New(int, size);
    rdispls = PyMem_New(int, size);
    if ( !rcounts || !rdispls )
      PY_RAISE(PyExc_MemoryError, "");

    /* alltoall string sizes */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Alltoall(scounts, 1, MPI_INT,
				  rcounts, 1, MPI_INT,
				  comm) );

    /* create and fill result list */
    result = PyList_New((Py_ssize_t)size);
    if (!result) RAISE;
    { 
      int i;
      for (i = 0; i < size; i++) {
        PyObject  *item = NULL;
        void      *bptr = NULL;
        int        blen = 0;
        Py_ssize_t bdsp = 0;
        /* create output object */
        item = arg_pickled_out(NULL, rcounts[i], &bptr, &blen);
        if (item == NULL) RAISE;
        /* set output in result list */
        PyList_SET_ITEM(result, i, item);
        /* calculate 'rbuff' pointer and 'displs' array */
        if (i == 0) rbuff = bptr;
        bdsp = ((char*)bptr) - ((char*)rbuff);
        rdispls[i] = Py_SAFE_DOWNCAST(bdsp, Py_ssize_t, int);
      }
    }
    
    /* alltoall string buffers */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Alltoallv(sbuff, scounts, sdispls, MPI4PY_PICKLED,
				   rbuff, rcounts, rdispls, MPI4PY_PICKLED,
				   comm) );

    /* deallocate memory */
    PyMem_Del(scounts);
    PyMem_Del(sdispls);
    PyMem_Del(rcounts);
    PyMem_Del(rdispls);
    
    /* return result */
    return  result;
    
  }

  EXCEPT {
    PyMem_Del(scounts);
    PyMem_Del(sdispls);
    PyMem_Del(rcounts);
    PyMem_Del(rdispls);
    Py_XDECREF(result);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_alltoall_buffer(PyObject *self, PyObject *args)
{
  PyObject *input;
  PyObject *output;
  MPI_Comm comm;

  /* parse input arguments */
  if (!PyArg_ParseTuple(args, "OOO&",
			&input, &output,
			PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    int flag;
    buffer_t sbuf;
    buffer_t rbuf;
    MPI4PY_CHECK( MPI_Comm_test_inter(comm, &flag) );
    if (!flag) { /* intra-communication */
      int size;
      /* get comm size */
      MPI4PY_CHECK( MPI_Comm_size(comm, &size) );
      /* get buffer data */
      if (!as_buffer_read_b (input,  &sbuf, size)) RAISE;
      if (!as_buffer_write_b(output, &rbuf, size)) RAISE;
    } 
    else { /* inter-communication */
#if HAVE_MPI_ROOT
      int size;
      /* get remote comm size */
      MPI4PY_CHECK( MPI_Comm_remote_size(comm, &size) );
      /* get buffer data */
      if (!as_buffer_read_b (input,  &sbuf, size)) RAISE;
      if (!as_buffer_write_b(output, &rbuf, size)) RAISE;
#else
      MPI4PY_RAISE_MPINOIMPL("extended collective communications");
#endif
    }
    /* gather buffer data */
    MPI4PY_CALL_COMM(comm,
		     MPI_Alltoall(sbuf.buf, sbuf.count, sbuf.dtype,
				  rbuf.buf, rbuf.count, rbuf.dtype, 
				  comm) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/


/*------------------------------------------------------------------*/
/* Global Reduction Operations                                      */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static PyObject *
op_create(PyObject *self, PyObject *args)
{
  MPI_User_function *function;
  PyObject *ofunction;
  int commute;
  MPI_Op *op;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OiO&",
			 &ofunction,
			 &commute,
			 PyO2OpP, &op) ) return NULL;
  
  TRY {
    void *voidptr = NULL;
    if (PyInt_Check(ofunction) || PyLong_Check(ofunction)) {
      voidptr = PyLong_AsVoidPtr(ofunction);
      if (PyErr_Occurred()) RAISE;
    } else if (PyCObject_Check(ofunction)) {
      voidptr = PyCObject_AsVoidPtr(ofunction);
      if (PyErr_Occurred()) RAISE;
    } else {
      PyErr_Format(PyExc_TypeError, 
		   "pointer or C object expected, %.200s found",
		   ofunction->ob_type->tp_name); RAISE;
    }
    function = (MPI_User_function *) ((Py_ssize_t)voidptr);
    /* create */
    MPI4PY_CALL( MPI_Op_create(function, commute, op) );
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
op_free(PyObject *self, PyObject *args)
{
  MPI_Op* op;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2OpP, &op) ) return NULL;
  
  MPI4PY_CHECK_NULL(*op, MPI_OP_NULL, MPI_ERR_OP);

  TRY {
    /* free */
    MPI4PY_CALL_FREE( MPI_Op_free(op) );
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_reduce_buffer(PyObject *self, PyObject *args)
{
  PyObject *input;
  PyObject *output;

  MPI_Op op;
  int root; 
  MPI_Comm comm;

  /* parse input arguments */
  if (!PyArg_ParseTuple(args, "OOO&iO&",
			&input, &output,
			PyO2Op, &op,
			&root,
			PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    int flag;
    buffer_t sbuf;
    buffer_t rbuf;
    MPI4PY_CHECK( MPI_Comm_test_inter(comm, &flag) );
    if (!flag) { /* intra-communication */
      int rank;
      /* get rank */
      MPI4PY_CHECK( MPI_Comm_rank(comm, &rank) );
      /* get buffer */
      if (rank == root) {
	if (!as_buffer_write(output, &rbuf)) RAISE;
	if (TestForInPlace(input)) {
	  if (!as_inplace(input, &sbuf.buf)) RAISE;
	  sbuf.count = rbuf.count;
	  sbuf.dtype = rbuf.dtype;
	} else {
	  if (!as_buffer_read(input, &sbuf)) RAISE;
	}
      } else {
	/*if (!as_buffer_ignore(output, &rbuf)) RAISE;*/
	if (!as_buffer_write(output, &rbuf)) RAISE;
	if (!as_buffer_read (input,  &sbuf)) RAISE;
      }
    } else { /* inter-communication */
#if HAVE_MPI_ROOT
      if (root == MPI_ROOT) {
	if (!as_buffer_ignore(input,  &sbuf)) RAISE;
	if (!as_buffer_write (output, &rbuf)) RAISE;
      } else if (root != MPI_PROC_NULL) {
	if (!as_buffer_read  (input,  &sbuf)) RAISE;
	if (!as_buffer_ignore(output, &rbuf)) RAISE;
      } else {
	if (!as_buffer_ignore(input,  &sbuf)) RAISE;
	if (!as_buffer_ignore(output, &rbuf)) RAISE;
      }
#else
      MPI4PY_RAISE_MPINOIMPL("extended collective communications");
#endif
    }
    /* reduce */
    MPI4PY_CALL_COMM(comm,
		     MPI_Reduce(sbuf.buf, rbuf.buf, 
				sbuf.count,  sbuf.dtype,
				op, root, comm) );
    /* return  */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_allreduce_buffer(PyObject *self, PyObject *args)
{
  PyObject *input;
  PyObject *output;

  MPI_Op op;
  MPI_Comm comm;

  /* parse input arguments */
  if (!PyArg_ParseTuple(args, "OOO&O&",
			&input, &output,
			PyO2Op, &op,
			PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    int flag;
    buffer_t sbuf;
    buffer_t rbuf;
    MPI4PY_CHECK( MPI_Comm_test_inter(comm, &flag) );
    if (!flag) { /* intra-communication */
      /* get buffers */
      if (!as_buffer_write(output, &rbuf)) RAISE;
      if (TestForInPlace(input)) {
	if (!as_inplace(input, &sbuf.buf)) RAISE;
	sbuf.count = rbuf.count;
	sbuf.dtype = rbuf.dtype;
      } else {
	if (!as_buffer_read   (input, &sbuf)) RAISE;
      }
    } else { /* inter-communication */
#if HAVE_MPI_ROOT
      if (!as_buffer_write(output, &rbuf)) RAISE;
      if (!as_buffer_read (input,  &sbuf)) RAISE;
#else
      MPI4PY_RAISE_MPINOIMPL("extended collective communications");
#endif
    }
    /* allreduce */
    MPI4PY_CALL_COMM(comm,
		     MPI_Allreduce(sbuf.buf, rbuf.buf,
				   sbuf.count, sbuf.dtype,
				   op, comm) );
    /* return  */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_scan_buffer(PyObject *self, PyObject *args)
{
  PyObject *input;
  PyObject *output;

  MPI_Op op;
  MPI_Comm comm;

  /* parse input arguments */
  if (!PyArg_ParseTuple(args, "OOO&O&",
			&input, &output,
			PyO2Op, &op,
			PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    buffer_t sbuf;
    buffer_t rbuf;
    /* get buffers */
    if (!as_buffer_write(output, &rbuf)) RAISE;
    if (TestForInPlace(input)) {
      if (!as_inplace(input, &sbuf.buf)) RAISE;
      sbuf.count = rbuf.count;
      sbuf.dtype = rbuf.dtype;
    } else {
      if (!as_buffer_read(input, &sbuf)) RAISE;
    }
    /* scan */
    MPI4PY_CALL_COMM(comm,
		     MPI_Scan(sbuf.buf, rbuf.buf,
			      sbuf.count, sbuf.dtype,
			      op, comm) );
    /* return  */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_exscan_buffer(PyObject *self, PyObject *args)
{
  PyObject *input;
  PyObject *output;

  MPI_Op op;
  MPI_Comm comm;

  /* parse input arguments */
  if (!PyArg_ParseTuple(args, "OOO&O&",
			&input, &output,
			PyO2Op, &op,
			PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
#if HAVE_MPI_EXSCAN
    buffer_t sbuf;
    buffer_t rbuf;
    /* get buffers */
    if (!as_buffer_write(output, &rbuf)) RAISE;
    if (TestForInPlace(input)) {
      if (!as_inplace(input, &sbuf.buf)) RAISE;
      sbuf.count = rbuf.count;
      sbuf.dtype = rbuf.dtype;
    } else {
      if (!as_buffer_read   (input, &sbuf)) RAISE;
    }
    /* exscan */
    MPI4PY_CALL_COMM(comm,
		     MPI_Exscan(sbuf.buf, rbuf.buf,
				sbuf.count, sbuf.dtype,
				op, comm) );
#else
    MPI4PY_RAISE_MPINOIMPL("MPI_EXSCAN");
#endif
    /* return  */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static PyObject *
comm_get_errhandler(PyObject *self, PyObject *args)
{
  MPI_Comm comm;
  MPI_Errhandler errhandler;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Cm,  &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
  
  TRY {
    /* get errhandler */
#if HAVE_MPI_COMM_GET_ERRHANDLER
    MPI4PY_CHECK( MPI_Comm_get_errhandler(comm, &errhandler) );
#else
    MPI4PY_CHECK( MPI_Errhandler_get(comm, &errhandler) );
#endif
#if defined(LAM_MPI)
    if (errhandler == MPI_ERRORS_RETURN ||
	errhandler == MPI_ERRORS_ARE_FATAL)
      errhandler->eh_refcount++;
#endif
    /* return errhandler */
    return PyMPIErrhandler_FromErrhandler(errhandler);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_set_errhandler(PyObject *self, PyObject *args)
{
  MPI_Comm comm;
  MPI_Errhandler errhandler;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O&",
			 PyO2Cm, &comm,
			 PyO2Eh, &errhandler) ) return NULL;

  MPI4PY_CHECK_NULL(comm,       MPI_COMM_NULL,       MPI_ERR_COMM);
  MPI4PY_CHECK_NULL(errhandler, MPI_ERRHANDLER_NULL, MPI_ERR_ARG);

  TRY {
    /* set errhandler */
#if HAVE_MPI_COMM_SET_ERRHANDLER
    MPI4PY_CHECK( MPI_Comm_set_errhandler(comm, errhandler) );
#else
    MPI4PY_CHECK( MPI_Errhandler_set(comm, errhandler) );
#endif
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/




/*------------------------------------------------------------------*/
/* Window                                                           */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static PyObject *
win_create(PyObject *self, PyObject *args)
{
  PyObject *memory;
  
  void *base;
  MPI_Aint size;
  int disp_unit;
  MPI_Info info;
  MPI_Comm comm;
  MPI_Win win;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OiO&O&",
			 &memory,
			 &disp_unit,
			 PyO2IN, &info,
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);

  TRY {
    /* get memory base pointer and size */
    if (memory == Py_None) { base = MPI_BOTTOM; size = 0; }
    else if (PyMPIMemory_AsMemory(memory, &base, &size) < 0) RAISE;
    /* create new window */
#if HAVE_MPI_WIN_CREATE
    MPI4PY_CALL_WIN_NEW(comm, win,
			MPI_Win_create(base, size, disp_unit,
				       info, comm, &win) );
#else
    win = MPI_WIN_NULL;
    MPI4PY_RAISE_MPINOIMPL("MPI_WIN");
#endif
    /* return new window */
    return PyMPIWin_FromWin(win);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
win_free(PyObject *self, PyObject *args)
{  
  MPI_Win *win;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&", 
			 PyO2WnP, &win) ) return NULL;
  
  MPI4PY_CHECK_NULL(*win, MPI_WIN_NULL, MPI_ERR_WIN);

  TRY {
    /* free window */
#if HAVE_MPI_WIN_FREE
    MPI4PY_CALL_WIN_FREE(*win, MPI_Win_free(win) );
#else
    *win = MPI_WIN_NULL;
    MPI4PY_RAISE_MPINOIMPL("MPI_WIN");
#endif
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/

#if HAVE_MPI_WIN_CREATE && HAVE_MPI_WIN_FREE

/*------------------------------------------------------------------*/
static PyObject *
win_get_memory(PyObject *self, PyObject *args)
{
  MPI_Win win;
  void *base = NULL;
  MPI_Aint *size = NULL;
  int flag = 0;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&", 
			 PyO2Wn, &win) ) return NULL;

  MPI4PY_CHECK_NULL(win, MPI_WIN_NULL, MPI_ERR_WIN);

  TRY {
    /* get win base pointer (attribute) */
    MPI4PY_CALL_WIN(win, MPI_Win_get_attr(win, MPI_WIN_BASE, &base, &flag) );
    if (!flag) PY_RAISE(PyExc_RuntimeError, 
			"attribute 'MPI_WIN_BASE' not found");
    /* get win memory size (attribute) */
    MPI4PY_CALL_WIN(win, MPI_Win_get_attr(win, MPI_WIN_SIZE,
					  &size, &flag) );
    if (!flag) PY_RAISE(PyExc_RuntimeError,
			"attribute 'MPI_WIN_SIZE' not found");
    if (!size) PY_RAISE(PyExc_RuntimeError,
			"attribute 'MPI_WIN_SIZE' returned NULL");
    return PyMPIMemory_FromMemory(base, *size);
  }
  EXCEPT {
    return NULL;
  }

}
static PyObject *
win_get_attrs(PyObject *self, PyObject *args)
{
  MPI_Win win;
  void *base;
  MPI_Aint *size;
  int *disp;
  int flag;
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&", 
			 PyO2Wn, &win) ) return NULL;

  MPI4PY_CHECK_NULL(win, MPI_WIN_NULL, MPI_ERR_WIN);

  TRY {
    /* get window base pointer (attribute) */
    base = NULL; flag = 0;
    MPI4PY_CALL_WIN(win, 
                    MPI_Win_get_attr(win, MPI_WIN_BASE, &base, &flag) );
    if (!flag) PY_RAISE(PyExc_RuntimeError, 
			"attribute 'MPI_WIN_BASE' not found");
    /* get window size (attribute) */
    size = NULL; flag = 0;
    MPI4PY_CALL_WIN(win, 
                    MPI_Win_get_attr(win, MPI_WIN_SIZE, &size, &flag) );
    if (!flag) PY_RAISE(PyExc_RuntimeError,
			"attribute 'MPI_WIN_SIZE' not found");
    if (!size) PY_RAISE(PyExc_RuntimeError,
			"attribute 'MPI_WIN_SIZE' returned NULL");
    /* get window displacement unit (attribute) */
    disp = NULL; flag = 0;
    MPI4PY_CALL_WIN(win, 
                    MPI_Win_get_attr(win, MPI_WIN_DISP_UNIT,  &disp, &flag) );
    if (!flag) PY_RAISE(PyExc_RuntimeError, 
                        "attribute 'MPI_WIN_DISP_UNIT' not found");
    if (!disp) PY_RAISE(PyExc_RuntimeError,
                        "attribute 'MPI_WIN_DISP_UNIT' returned NULL");
    
    return Py_BuildValue("NNN",
                         PyMPIAint_FromAint((MPI_Aint)base),
                         PyMPIAint_FromAint((MPI_Aint)(*size)),
                         PyInt_FromLong((long)(*disp)));
  }
  EXCEPT {
    return NULL;
  }

}
static PyObject *
win_get_group(PyObject *self, PyObject *args)
{
  MPI_Win win;
  MPI_Group group;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&", 
			 PyO2Wn, &win) ) return NULL;

  MPI4PY_CHECK_NULL(win, MPI_WIN_NULL, MPI_ERR_WIN);

  TRY {
    /* get win group */
    MPI4PY_CALL_WIN(win, MPI_Win_get_group(win, &group) );
    /* return group  */
    return PyMPIGroup_FromGroup(group);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static int
win_args_parser(int writable,
                PyObject *origin, int target_rank, PyObject *target,
                void    **o_addr, int *o_count, MPI_Datatype *o_dtype,
                MPI_Aint *t_disp, int *t_count, MPI_Datatype *t_dtype)
{
  int ok;
  buffer_t buf;
  PyObject *args;

  /* parse origin arguments */
  if (target_rank == MPI_PROC_NULL)
    ok = as_buffer_ignore(origin, &buf);
  else if (writable)
    ok = as_buffer_write(origin, &buf);
  else 
    ok = as_buffer_read(origin, &buf);
  if (!ok) goto fail;
  *o_addr  = buf.buf;
  *o_count = buf.count;
  *o_dtype = buf.dtype;

  /* parse target arguments */
  *t_disp  = 0;
  *t_count = *o_count;
  *t_dtype = *o_dtype;
  if (target == Py_None) goto exit;
  args = PySequence_Tuple(target);
  if (args == NULL) goto fail;
  ok = PyArg_ParseTuple(args, "|O&iO&",
                        PyO2Ad, t_disp,
                        /*  */  t_count,
                        PyO2Dt, t_dtype);
  Py_DECREF(args);
  if (!ok) goto fail;
 
 exit:
  return 1;

 fail:
  return 0;
}
/*------------------------------------------------------------------*/
static PyObject *
win_put(PyObject *self, PyObject *args)
{
  PyObject *origin;
  PyObject *target;

  void *origin_addr;
  int origin_count;
  MPI_Datatype origin_dtype;
  int target_rank;
  MPI_Aint target_disp;
  int target_count;
  MPI_Datatype target_dtype;
  MPI_Win win;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OiOO&",
			 &origin,
			 &target_rank,
                         &target,
			 PyO2Wn, &win) ) return NULL;

  MPI4PY_CHECK_NULL(win, MPI_WIN_NULL, MPI_ERR_WIN);

  TRY {
    /* origin and target arguments */
    if (!win_args_parser(0, origin, target_rank, target,
                         &origin_addr, &origin_count, &origin_dtype,
                         &target_disp, &target_count, &target_dtype)) RAISE;
    /* put */
    MPI4PY_CALL_WIN(win, 
		    MPI_Put(origin_addr, origin_count, origin_dtype,
			    target_rank, 
			    target_disp, target_count, target_dtype, 
			    win) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
win_get(PyObject *self, PyObject *args)
{
  PyObject *origin;
  PyObject *target;

  void *origin_addr;
  int origin_count;
  MPI_Datatype origin_dtype;
  int target_rank;
  MPI_Aint target_disp;
  int target_count;
  MPI_Datatype target_dtype;
  MPI_Win win;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OiOO&",
			 &origin,
			 &target_rank,
                         &target,
			 PyO2Wn, &win) ) return NULL;

  MPI4PY_CHECK_NULL(win, MPI_WIN_NULL, MPI_ERR_WIN);

  TRY {
    /* origin and target arguments */
    if (!win_args_parser(1, origin, target_rank, target,
                         &origin_addr, &origin_count, &origin_dtype,
                         &target_disp, &target_count, &target_dtype)) RAISE;
    /* get */
    MPI4PY_CALL_WIN(win, 
		    MPI_Get(origin_addr, origin_count, origin_dtype,
			    target_rank, 
			    target_disp, target_count, target_dtype, 
			    win) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
win_accumulate(PyObject *self, PyObject *args)
{
  PyObject *origin;
  PyObject *target;

  void *origin_addr;
  int origin_count;
  MPI_Datatype origin_dtype;
  int target_rank;
  MPI_Aint target_disp;
  int target_count;
  MPI_Datatype target_dtype;
  MPI_Op op;
  MPI_Win win;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "OiOO&O&",
			 &origin,
			 &target_rank,
                         &target,
			 PyO2Op, &op,
			 PyO2Wn, &win) ) return NULL;

  MPI4PY_CHECK_NULL(win, MPI_WIN_NULL, MPI_ERR_WIN);

  TRY {
    /* origin and target arguments */
    if (!win_args_parser(0, origin, target_rank, target,
                         &origin_addr, &origin_count, &origin_dtype,
                         &target_disp, &target_count, &target_dtype)) RAISE;
    /* get */
    MPI4PY_CALL_WIN(win, 
		    MPI_Accumulate(origin_addr, origin_count, origin_dtype,
				   target_rank, 
				   target_disp, target_count, target_dtype, 
				   op, win) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
win_fence(PyObject *self, PyObject *args)
{
  int assertion;
  MPI_Win win;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "iO&",
			 &assertion,
			 PyO2Wn, &win) ) return NULL;

  MPI4PY_CHECK_NULL(win, MPI_WIN_NULL, MPI_ERR_WIN);

  TRY {
    /* fence */
    MPI4PY_CALL_WIN(win,
		    MPI_Win_fence(assertion, win) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
win_start(PyObject *self, PyObject *args)
{
  MPI_Group group;
  int assertion;
  MPI_Win win;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&iO&",
			 PyO2Gr, &group,
			 &assertion,
			 PyO2Wn, &win) ) return NULL;

  MPI4PY_CHECK_NULL(win, MPI_WIN_NULL, MPI_ERR_WIN);

  TRY {
    /* fence */
    MPI4PY_CALL_WIN(win,
		    MPI_Win_start(group, assertion, win) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
win_complete(PyObject *self, PyObject *args)
{
  MPI_Win win;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Wn, &win) ) return NULL;

  MPI4PY_CHECK_NULL(win, MPI_WIN_NULL, MPI_ERR_WIN);

  TRY {
    /* fence */
    MPI4PY_CALL_WIN(win,
		    MPI_Win_complete(win) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
win_post(PyObject *self, PyObject *args)
{
  MPI_Group group;
  int assertion;
  MPI_Win win;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&iO&",
			 PyO2Gr, &group,
			 &assertion,
			 PyO2Wn, &win) ) return NULL;

  MPI4PY_CHECK_NULL(win, MPI_WIN_NULL, MPI_ERR_WIN);

  TRY {
    /* fence */
    MPI4PY_CALL_WIN(win,
		    MPI_Win_post(group, assertion, win) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
win_wait(PyObject *self, PyObject *args)
{
  MPI_Win win;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Wn, &win) ) return NULL;
  
  TRY {
    /* wait */
    MPI4PY_CALL_WIN(win, 
		    MPI_Win_wait(win) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
win_test(PyObject *self, PyObject *args)
{
  MPI_Win win;
  int flag;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Wn, &win) ) return NULL;

  TRY {
#if HAVE_MPI_WIN_TEST
    /* test */
    MPI4PY_CALL_WIN(win,
		    MPI_Win_test(win, &flag) );
    /* return flag */
    return PyBool_FromLong((long)flag);
#else
    flag = 0;
    MPI4PY_RAISE_MPINOIMPL("MPI_WIN_TEST");
#endif
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
win_lock(PyObject *self, PyObject *args)
{
  int lock_type; 
  int rank;
  int assertion;
  MPI_Win win;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "iiiO&",
			 &lock_type,
			 &rank,
			 &assertion,
			 PyO2Wn, &win) ) return NULL;

  MPI4PY_CHECK_NULL(win, MPI_WIN_NULL, MPI_ERR_WIN);

  TRY {
#if HAVE_MPI_WIN_LOCK
    /* lock */
    MPI4PY_CALL_WIN(win,
		    MPI_Win_lock(lock_type, rank, assertion, win) );
    /* return */
    Py_RETURN_NONE;
#else
    MPI4PY_RAISE_MPINOIMPL("MPI_WIN_LOCK");
#endif
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
win_unlock(PyObject *self, PyObject *args)
{
  int rank;
  MPI_Win win;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "iO&",
			 &rank,
			 PyO2Wn, &win) ) return NULL;

  MPI4PY_CHECK_NULL(win, MPI_WIN_NULL, MPI_ERR_WIN);

  TRY {
#if HAVE_MPI_WIN_UNLOCK
    /* unlock */
    MPI4PY_CALL_WIN(win,
		    MPI_Win_unlock(rank, win) );
    /* return */
    Py_RETURN_NONE;
#else
    MPI4PY_RAISE_MPINOIMPL("MPI_WIN_UNLOCK");
#endif
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static PyObject *
win_get_errhandler(PyObject *self, PyObject *args)
{
  MPI_Win win;
  MPI_Errhandler errhandler;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Wn,  &win) ) return NULL;

  MPI4PY_CHECK_NULL(win, MPI_WIN_NULL, MPI_ERR_WIN);
  
  TRY {
    /* get errhandler */
    MPI4PY_CHECK( MPI_Win_get_errhandler(win, &errhandler) );
#if defined(LAM_MPI)
    if (errhandler == MPI_ERRORS_RETURN ||
	errhandler == MPI_ERRORS_ARE_FATAL)
      errhandler->eh_refcount++;
#endif
    /* return errhandler */
    return PyMPIErrhandler_FromErrhandler(errhandler);
  }
  EXCEPT {
    return NULL;
  }

}
static PyObject *
win_set_errhandler(PyObject *self, PyObject *args)
{
  MPI_Win win;
  MPI_Errhandler errhandler;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O&",
			 PyO2Wn, &win,
			 PyO2Eh, &errhandler) ) return NULL;

  MPI4PY_CHECK_NULL(win,        MPI_WIN_NULL,        MPI_ERR_WIN);
  MPI4PY_CHECK_NULL(errhandler, MPI_ERRHANDLER_NULL, MPI_ERR_ARG);

  TRY {
    /* set errhandler */
    MPI4PY_CHECK( MPI_Win_set_errhandler(win, errhandler) );
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/

#else /* !HAVE_MPI_WIN_CREATE || !HAVE_MPI_WIN_FREE */

static PyObject *
win_mpinoimpl(PyObject *self, PyObject *args)
{
  TRY {
    MPI4PY_RAISE_MPINOIMPL("MPI_WIN");
  }
  EXCEPT {
    return NULL;
  }
}

#define win_get_memory win_mpinoimpl
#define win_get_attrs  win_mpinoimpl
#define win_get_group  win_mpinoimpl
#define win_put	       win_mpinoimpl
#define win_get        win_mpinoimpl
#define win_accumulate win_mpinoimpl
#define win_fence      win_mpinoimpl
#define win_start      win_mpinoimpl
#define win_complete   win_mpinoimpl
#define win_post       win_mpinoimpl
#define win_wait       win_mpinoimpl
#define win_test       win_mpinoimpl
#define win_lock       win_mpinoimpl
#define win_unlock     win_mpinoimpl

#define win_get_errhandler  win_mpinoimpl
#define win_set_errhandler  win_mpinoimpl

#endif /* HAVE_MPI_WIN_CREATE && HAVE_MPI_WIN_FREE */



/*------------------------------------------------------------------*/
/* File                                                             */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static MPI4PYUNUSED int
PyO2Am(PyObject *obj, int *amode) 
{
  if (obj == Py_None)
    *amode = MPI_MODE_RDONLY;
  else {
    *amode = (int) PyInt_AsLong(obj);
    if (*amode == -1 && PyErr_Occurred()) return 0;
  }
  return 1;
}

static PyObject *
file_open(PyObject *self, PyObject *args)
{
  MPI_Comm comm; 
  char *filename; 
  int amode;
  MPI_Info info; 
  MPI_File file;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&sO&O&",
			 PyO2Cm, &comm,
			 &filename,
			 PyO2Am, &amode,
			 PyO2IN, &info) ) return NULL;

  TRY {
    /* open file */
#if HAVE_MPI_FILE_OPEN
    MPI4PY_CALL_FILE_NEW(MPI_FILE_NULL, file,
			 MPI_File_open(comm, filename, amode,
				       info, &file) );
#else
    file = MPI_FILE_NULL;
    MPI4PY_RAISE_MPINOIMPL("MPI_FILE");
#endif
    /* return file */
    return PyMPIFile_FromFile(file);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
file_close(PyObject *self, PyObject *args)
{
  MPI_File *file;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2FlP,  &file) ) return NULL;

  MPI4PY_CHECK_NULL(*file, MPI_FILE_NULL, MPI_ERR_FILE);
  
  TRY {
    /* close file */
#if HAVE_MPI_FILE_CLOSE
    MPI4PY_CALL_FILE_FREE(*file,
			  MPI_File_close(file) );
#else
    *file = MPI_FILE_NULL;
    MPI4PY_RAISE_MPINOIMPL("MPI_FILE");
#endif
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}

#if HAVE_MPI_FILE_OPEN

/*------------------------------------------------------------------*/
static PyObject *
file_delete(PyObject *self, PyObject *args)
{
  char *filename; 
  MPI_Info info; 

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "sO&",
			 &filename,
			 PyO2IN, &info) ) return NULL;

  TRY {
    /* delete file */
    MPI4PY_CALL_FILE_NULL( MPI_File_delete(filename, info) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
file_preallocate(PyObject *self, PyObject *args)
{
  MPI_File file;
  MPI_Offset size;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O&",
			 PyO2Fl, &file,
			 PyO2Of, &size) ) return NULL;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);
  
  TRY {
    /* preallocate file */
    MPI4PY_CALL_FILE(file,
		     MPI_File_preallocate(file, size));
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
file_set_size(PyObject *self, PyObject *args)
{
  MPI_File file;
  MPI_Offset size;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O&",
			 PyO2Fl, &file,
			 PyO2Of, &size) ) return NULL;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);
  
  TRY {
    /* set file size */
    MPI4PY_CALL_FILE(file,
		     MPI_File_set_size(file, size));
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
file_get_size(PyObject *self, PyObject *args)
{
  MPI_File file;
  MPI_Offset size;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Fl, &file) ) return NULL;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);
  
  TRY {
    /* get file size */
    MPI4PY_CALL_FILE(file,
		     MPI_File_get_size(file, &size));
    /* return size */
    return PyMPIOffset_FromOffset(size);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
file_get_group(PyObject *self, PyObject *args)
{
  MPI_File file;
  MPI_Group group;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&", 
			 PyO2Fl, &file) ) return NULL;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);

  TRY {
    /* get file group */
    MPI4PY_CALL_FILE(file, MPI_File_get_group(file, &group) );
    /* return group  */
    return PyMPIGroup_FromGroup(group);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
file_get_amode(PyObject *self, PyObject *args)
{
  MPI_File file;
  int amode;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Fl, &file) ) return NULL;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);
  
  TRY {
    /* get file amode */
    MPI4PY_CALL_FILE(file,
		     MPI_File_get_amode(file, &amode));
    /* return */
    return PyInt_FromLong((long)amode);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
file_set_info(PyObject *self, PyObject *args)
{
  MPI_File file;
  MPI_Info info;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O&",
			 PyO2Fl, &file,
			 PyO2In, &info) ) return NULL;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);
  
  TRY {
    /* set file info */
    MPI4PY_CALL_FILE(file,
		     MPI_File_set_info(file, info));
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
file_get_info(PyObject *self, PyObject *args)
{
  MPI_File file;
  MPI_Info info;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Fl, &file) ) return NULL;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);
  
  TRY {
    /* get file info */
    MPI4PY_CALL_FILE(file,
		     MPI_File_get_info(file, &info));
    /* return */
    return PyMPIInfo_FromInfo(info);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
file_set_view(PyObject *self, PyObject *args)
{
  MPI_File file;
  MPI_Offset disp;
  MPI_Datatype etype;
  MPI_Datatype filetype;
  char *datarep;
  MPI_Info info;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O&O&O&sO&",
			 PyO2Fl,  &file,
			 PyO2Of, &disp,
			 PyO2Dt,  &etype,
			 PyO2Dt,  &filetype,
			 &datarep,
			 PyO2IN, &info) ) return NULL;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);
  
  TRY {
    /* set file view */
    MPI4PY_CALL_FILE(file,
		     MPI_File_set_view(file, disp, etype, 
				       filetype, datarep, info) );
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
/* XXX What to do here ?? */
static PyObject *
PyMPI_FILE_DATATYPE(MPI_Datatype datatype)
{
  PyObject *ob = PyMPIDatatype_FromDatatype(datatype);
  if (ob == NULL) return NULL;
  if (datatype != MPI_DATATYPE_NULL) { /* just in case */
    int ierr, ni = 0, na = 0, ndt = 0;
    int combiner = MPI_COMBINER_NAMED;
    ierr = MPI_Type_get_envelope(datatype, &ni, &na, &ndt, &combiner);
    if (combiner == MPI_COMBINER_NAMED)
      PyMPIDatatype_Inner(ob)->isref = 1;
  }
  return ob;
}
static PyObject *
file_get_view(PyObject *self, PyObject *args)
{
  MPI_File file;
  MPI_Offset disp;
  MPI_Datatype etype;
  MPI_Datatype filetype;
  char *datarep = NULL;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Fl,  &file) ) return NULL;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);
  
  TRY {
    char buf[MPI_MAX_DATAREP_STRING] = { '\0' };
    datarep = buf;
    /* get file view */
    MPI4PY_CALL_FILE(file,
		     MPI_File_get_view(file, &disp, &etype, 
				       &filetype, datarep) );
    /* return file view */
    return Py_BuildValue("NNNs",
			 PyMPIOffset_FromOffset(disp),
                         PyMPI_FILE_DATATYPE(etype),
                         PyMPI_FILE_DATATYPE(filetype),
                         datarep);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/

typedef struct buffer_io_t {
  PyObject *input;
  PyObject *output;
  void *buf;
  int count;
  MPI_Datatype dtype;
} buffer_io_t;

static MPI4PYUNUSED int
PyO2WB(PyObject *obj, buffer_io_t *buf) 
{
  /* initialization */
  buf->input  = NULL;
  buf->output = NULL;
  buf->buf    = NULL;
  buf->count  = 0;
  buf->dtype  = MPI_DATATYPE_NULL;
  /* */
  if (PyInt_Check(obj)) {
    PyObject *obuf; void *pbuf; int size;
    size = (int) PyInt_AS_LONG(obj);
    if (size < 0) size = 0; /* XXX or like file.read(), until EOF ? */
#if PY_VERSION_HEX < 0x03000000
    obuf = PyString_FromStringAndSize(NULL, (Py_ssize_t)size);
    if (obuf == NULL) return  0;
    pbuf = (void *) PyString_AS_STRING(obuf);
#else
    obuf = PyBytes_FromStringAndSize(NULL, (Py_ssize_t)size);
    if (obuf == NULL) return  0;
    pbuf = (void *) PyBytes_AS_STRING(obuf);
#endif
    buf->input  = NULL;
    buf->output = obuf;
    buf->buf    = pbuf;
    buf->count  = size;
    buf->dtype  = MPI_BYTE;
  } else {
    buffer_t wbuf;
    if (!as_buffer_write(obj, &wbuf)) return 0;
    buf->input  = NULL;
    buf->output = (Py_INCREF(Py_None), Py_None);
    buf->buf    = wbuf.buf;
    buf->count  = wbuf.count;
    buf->dtype  = wbuf.dtype;
  }
  return 1;
}

static int
PyO2RB(PyObject *obj, buffer_io_t *buf) 
{
  /* initialization */
  buf->input  = NULL;
  buf->output = NULL;
  buf->buf    = NULL;
  buf->count  = 0;
  buf->dtype  = MPI_DATATYPE_NULL;
  /* */
  if (PyString_CheckExact(obj)) {
    char *sbuf; Py_ssize_t slen;
    if(PyString_AsStringAndSize(obj, &sbuf, &slen) < 0)
      return 0;
    if (slen > (Py_ssize_t)INT_MAX) {
      PyErr_SetString(PyExc_ValueError,
		      "string too large"); 
      return 0;
    }
    buf->buf    = (void *) sbuf;
    buf->count  = Py_SAFE_DOWNCAST(slen, Py_ssize_t, int);
    buf->dtype  = MPI_CHAR;
  } else {
    buffer_t rbuf;
    if (!as_buffer_read(obj, &rbuf)) return 0;
    buf->buf    = rbuf.buf;
    buf->count  = rbuf.count;
    buf->dtype  = rbuf.dtype;
  }
  return 1;
}

/*------------------------------------------------------------------*/
static PyObject *
file_read(PyObject *self, PyObject *args)
{
  MPI_File file;
  MPI_Offset offset;
  PyObject *obuf; buffer_io_t buf;
  MPI_Status *status;

  int flag, prefix;
  
  /* parse input arguments */
  if (PyTuple_GET_SIZE(args) > 4)
    flag = (!PyArg_ParseTuple(args, "O&O&OO&i",
			      PyO2Fl, &file,
			      PyO2Of, &offset,
			      &obuf,
			      PyO2SN, &status,
			      &prefix)) ? 0 : 2; /* use explicit offset */
  else 
    flag = (!PyArg_ParseTuple(args, "O&OO&i",
			      PyO2Fl, &file,
			      &obuf,
			      PyO2SN, &status,
			      &prefix)) ? 0 : 1; /* use file pointers */
  if(!flag) return NULL; 
  --flag;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);
  
  TRY {

    /* get output buffer */
    if (!PyO2WB(obuf, &buf)) RAISE;

    /* read from file */

    if (flag) /* use explicit offset */

      switch (prefix) {
      case 0: /* noncollective */
	MPI4PY_CALL_FILE(file,
			 MPI_File_read_at(file, offset,
					  buf.buf, buf.count, buf.dtype,
					  status)
			 ); break;
      case 1: /* collective */
	MPI4PY_CALL_FILE(file,
			 MPI_File_read_at_all(file, offset,
					      buf.buf, buf.count, buf.dtype,
					      status)
			 ); break;
      }

    else  /* use file pointers */

      switch (prefix) {
      case 0: /* 0 <- (0 | 0) */ /* noncollective | individual file pointer */
	MPI4PY_CALL_FILE(file,
			 MPI_File_read(file,
				       buf.buf, buf.count, buf.dtype,
				       status)
			 ); break;
      case 1: /* 1 <- (1 | 0) */ /* collective | individual file pointer */
	MPI4PY_CALL_FILE(file,
			 MPI_File_read_all(file,
					   buf.buf, buf.count, buf.dtype,
					   status)
			 ); break;
      case 2: /* 2 <- (0 | 2) */ /* noncollective | shared file pointer */
	MPI4PY_CALL_FILE(file,
			 MPI_File_read_shared(file,
					      buf.buf, buf.count, buf.dtype,
					      status)
			 ); break;
      case 3: /* 3 <- (1 | 2) */ /* collective | shared file pointer*/
	MPI4PY_CALL_FILE(file,
			 MPI_File_read_ordered(file,
					       buf.buf, buf.count, buf.dtype,
					       status)
			 ); break;
      }

    /* release resources */
    Py_XDECREF(buf.input);
    /* return */
    return buf.output;
  }
  EXCEPT {
    Py_XDECREF(buf.input);
    Py_XDECREF(buf.output);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
file_write(PyObject *self, PyObject *args)
{
  MPI_File file;
  MPI_Offset offset;
  PyObject *ibuf; buffer_io_t buf;
  MPI_Status *status;

  int flag, prefix;
  
  /* parse input arguments */
  if (PyTuple_GET_SIZE(args) > 4)
    flag = (!PyArg_ParseTuple(args, "O&O&OO&i",
			      PyO2Fl, &file,
			      PyO2Of, &offset,
			      &ibuf,
			      PyO2SN, &status,
			      &prefix)) ? 0 : 2; /* use explicit offset */
  else 
    flag = (!PyArg_ParseTuple(args, "O&OO&i",
			      PyO2Fl, &file,
			      &ibuf,
			      PyO2SN, &status,
			      &prefix)) ? 0 : 1; /* use file pointers */
  if(!flag) return NULL;
  --flag;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);
  
  TRY {
    
    /* get input buffer */
    if (!PyO2RB(ibuf, &buf)) RAISE;

    /* write in file */

    if (flag) /* use explicit offset */

      switch (prefix) {
      case 0: /* noncollective */
	MPI4PY_CALL_FILE(file,
		       MPI_File_write_at(file, offset,
					 buf.buf, buf.count, buf.dtype,
					 status)
		       ); break;
      case 1: /* collective */
	MPI4PY_CALL_FILE(file,
			 MPI_File_write_at_all(file, offset,
					       buf.buf, buf.count, buf.dtype,
					       status)
			 ); break;
      }

    else /* use file pointers */

      switch (prefix) {
      case 0: /* 0 <- (0 | 0) */ /* noncollective | individual file pointer */
	MPI4PY_CALL_FILE(file,
			 MPI_File_write(file,
					buf.buf, buf.count, buf.dtype,
					status)
			 ); break;
      case 1: /* 1 <- (1 | 0) */ /* collective | individual file pointer */
	MPI4PY_CALL_FILE(file,
			 MPI_File_write_all(file,
					    buf.buf, buf.count, buf.dtype,
					    status)
			 ); break;
      case 2: /* 2 <- (0 | 2) */ /* noncollective | shared file pointer */
	MPI4PY_CALL_FILE(file,
			 MPI_File_write_shared(file,
					       buf.buf, buf.count, buf.dtype,
					       status)
			 ); break;
      case 3: /* 3 <- (1 | 2) */ /* collective | shared file pointer */
	MPI4PY_CALL_FILE(file,
			 MPI_File_write_ordered(file,
						buf.buf, buf.count, buf.dtype,
						status)
			 ); break;
      }
    
    /* release resources */
    Py_XDECREF(buf.input);
    Py_XDECREF(buf.output);
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    Py_XDECREF(buf.input);
    Py_XDECREF(buf.output);
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
file_iread(PyObject *self, PyObject *args)
{
  MPI_File file;
  MPI_Offset offset;
  PyObject *obuf; buffer_t buf;
  MPI_Request request;

  int flag, prefix;
  
  /* parse input arguments */
  if (PyTuple_GET_SIZE(args) > 3)
    flag = (!PyArg_ParseTuple(args, "O&O&Oi",
			      PyO2Fl, &file,
			      PyO2Of, &offset,
			      &obuf,
			      &prefix)) ? 0 : 2; /* use explicit offset */
  else 
    flag = (!PyArg_ParseTuple(args, "O&Oi",
			      PyO2Fl, &file,
			      &obuf,
			      &prefix)) ? 0 : 1; /* use file pointers */
  if(!flag) return NULL; 
  --flag;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);
  
  TRY {
    
    /* get output buffer */
    if (!as_buffer_write(obuf, &buf)) RAISE;


    if (flag) /* use explicit offset */

      MPI4PY_CALL_FILE(file,
		       MPI_File_iread_at(file, offset,
					 buf.buf, buf.count, buf.dtype,
					 &request)
		       );

    else /* use file pointers */
    
      switch (prefix) {
      case 0: /* individual file pointer */
	MPI4PY_CALL_FILE(file,
			 MPI_File_iread(file,
					buf.buf, buf.count, buf.dtype,
					&request)
			 ); break;
      case 2: /* shared file pointer */
	MPI4PY_CALL_FILE(file,
			 MPI_File_iread_shared(file,
					       buf.buf, buf.count, buf.dtype,
					       &request)
			 ); break;
      }

    /* return request */
    return PyMPIRequest_FromRequest(request);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
file_iwrite(PyObject *self, PyObject *args)
{
  MPI_File file;
  MPI_Offset offset;
  PyObject *ibuf; buffer_t buf;
  MPI_Request request;

  int flag, prefix;
  
  /* parse input arguments */
  if (PyTuple_GET_SIZE(args) > 3)
    flag = (!PyArg_ParseTuple(args, "O&O&Oi",
			      PyO2Fl, &file,
			      PyO2Of, &offset,
			      &ibuf,
			      &prefix)) ? 0 : 2; /* use explicit offset */
  else 
    flag = (!PyArg_ParseTuple(args, "O&Oi",
			      PyO2Fl, &file,
			      &ibuf,
			      &prefix)) ? 0 : 1; /* use file pointers */
  if(!flag) return NULL;
  --flag;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);
  
  TRY {
    
    /* get input buffer */
    if (!as_buffer_read(ibuf, &buf)) RAISE;

    /* write in file */
    if (flag) /* use explicit offset */

      MPI4PY_CALL_FILE(file,
		       MPI_File_iwrite_at(file, offset,
					  buf.buf, buf.count, buf.dtype,
					  &request)
		       );

    else /* use file pointers */
    
      switch (prefix) {
      case 0: /* individual file pointer */
	MPI4PY_CALL_FILE(file,
			 MPI_File_iwrite(file,
					 buf.buf, buf.count, buf.dtype,
					 &request)
			 ); break;
      case 2: /* shared file pointer */
	MPI4PY_CALL_FILE(file,
			 MPI_File_iwrite_shared(file,
						buf.buf, buf.count, buf.dtype,
						&request)
			 ); break;
      }

    /* return request */
    return PyMPIRequest_FromRequest(request);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
file_read_split(PyObject *self, PyObject *args)
{
  MPI_File file;
  MPI_Offset offset;
  PyObject *obuf; buffer_t buf;
  MPI_Status* status;
  
  int flag, prefix, beflag;

  /* parse input arguments */
  if (PyTuple_GET_SIZE(args) > 5)
    flag = (!PyArg_ParseTuple(args, "O&O&OO&ii",
			      PyO2Fl, &file,
			      PyO2Of, &offset,
			      &obuf,
			      PyO2SN, &status,
			      &prefix,
			      &beflag)) ? 0 : 2; /* use explicit offset */
  else 
    flag = (!PyArg_ParseTuple(args, "O&OO&ii",
			      PyO2Fl, &file,
			      &obuf,
			      PyO2SN, &status,
			      &prefix,
			      &beflag)) ? 0 : 1; /* use file pointers */
  if(!flag) return NULL;
  --flag;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);

  TRY {
    
    /* get output buffer */
    if (!as_buffer_write(obuf, &buf)) RAISE;

    if (flag) /* use explicit offset */

      switch (beflag) {
      case 0: /* begin */
	MPI4PY_CALL_FILE(file, 
			 MPI_File_read_at_all_begin(file, offset, buf.buf, buf.count, buf.dtype)
			 ); break;
      case 1: /* end */
	MPI4PY_CALL_FILE(file, 
			 MPI_File_read_at_all_end(file, buf.buf, status)
			 ); break;
      }

    else  /* use file pointers */

      switch (prefix) {
      case 1: /* 1 <- (1 | 0) */ /* collective | individual file pointer */
	switch (beflag) {
	case 0: /* begin */
	  MPI4PY_CALL_FILE(file, 
			   MPI_File_read_all_begin(file, buf.buf, buf.count, buf.dtype)
			   ); break;
	case 1: /* end */
	  MPI4PY_CALL_FILE(file, 
			   MPI_File_read_all_end(file, buf.buf, status)
			   ); break;
	} 
	break;
      case 3: /* 3 <- (1 | 2) */ /* collective | shared file pointer */
	switch (beflag) {
	case 0: /* begin */
	  MPI4PY_CALL_FILE(file, 
			   MPI_File_read_ordered_begin(file, buf.buf, buf.count, buf.dtype)
			   ); break;
	case 1: /* end */
	  MPI4PY_CALL_FILE(file, 
			   MPI_File_read_ordered_end(file, buf.buf, status)
			   ); break;
	} 
	break;
      }

    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
file_write_split(PyObject *self, PyObject *args)
{
  MPI_File file;
  MPI_Offset offset;
  PyObject *obuf; buffer_t buf;
  MPI_Status* status;
  
  int flag, prefix, beflag;

  /* parse input arguments */
  if (PyTuple_GET_SIZE(args) > 5)
    flag = (!PyArg_ParseTuple(args, "O&O&OO&ii",
			      PyO2Fl, &file,
			      PyO2Of, &offset,
			      &obuf,
			      PyO2SN, &status,
			      &prefix,
			      &beflag)) ? 0 : 2; /* use explicit offset */
  else 
    flag = (!PyArg_ParseTuple(args, "O&OO&ii",
			      PyO2Fl, &file,
			      &obuf,
			      PyO2SN, &status,
			      &prefix,
			      &beflag)) ? 0 : 1; /* use file pointers */
  if(!flag) return NULL;
  --flag;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);

  TRY {
    
    /* get input  buffer */
    if (!as_buffer_read(obuf, &buf)) RAISE;

    if (flag) /* use explicit offset */

      switch (beflag) {
      case 0: /* begin */
	MPI4PY_CALL_FILE(file, 
			 MPI_File_write_at_all_begin(file, offset, buf.buf, buf.count, buf.dtype)
			 ); break;
      case 1: /* end */
	MPI4PY_CALL_FILE(file, 
			 MPI_File_write_at_all_end(file, buf.buf, status)
			 ); break;
      }

    else  /* use file pointers */

      switch (prefix) {
      case 1: /* 1 <- (1 | 0) */ /* collective | individual file pointer */
	switch (beflag) {
	case 0: /* begin */
	  MPI4PY_CALL_FILE(file, 
			   MPI_File_write_all_begin(file, buf.buf, buf.count, buf.dtype)
			   ); break;
	case 1: /* end */
	  MPI4PY_CALL_FILE(file, 
			   MPI_File_write_all_end(file, buf.buf, status)
			   ); break;
	} 
	break;
      case 3: /* 3 <- (1 | 2) */ /* collective | shared file pointer */
	switch (beflag) {
	case 0: /* begin */
	  MPI4PY_CALL_FILE(file,
			   MPI_File_write_ordered_begin(file, buf.buf, buf.count, buf.dtype)
			   ); break;
	case 1: /* end */
	  MPI4PY_CALL_FILE(file, 
			   MPI_File_write_ordered_end(file, buf.buf, status)
			   ); break;
	} 
	break;
      }

    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static int
PyO2Wh(PyObject *obj, int *whence) 
{
  if (obj == Py_None)
    *whence = MPI_SEEK_SET;
  else {
    *whence = (int) PyInt_AsLong(obj);
    if (*whence == -1 && PyErr_Occurred()) return 0;
  }
  return 1;
}

static PyObject *
file_seek(PyObject *self, PyObject *args)
{
  MPI_File file;
  MPI_Offset offset;
  int whence;

  int shared = 0;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O&O&|i",
			 PyO2Fl, &file,
			 PyO2Of, &offset,
			 PyO2Wh, &whence,
			 &shared) ) return NULL;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);
  
  TRY {
    switch (shared) {
    case 0: /* individual file pointer */
      MPI4PY_CALL_FILE(file,
		       MPI_File_seek(file, offset, whence)
		       ); break;
    case 2: /* shared file pointer */
      MPI4PY_CALL_FILE(file,
		       MPI_File_seek_shared(file, offset, whence)
		       ); break;
    }
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
file_get_position(PyObject *self, PyObject *args)
{
  MPI_File file;
  MPI_Offset offset;
  
  int shared = 0;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&|i",
			 PyO2Fl, &file,
			 &shared) ) return NULL;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);
  
  TRY {
    /* get file position */
    switch (shared) {
    case 0:
      MPI4PY_CALL_FILE(file,
		       MPI_File_get_position(file, &offset)
		       ); break;
    case 2:
      MPI4PY_CALL_FILE(file,
		       MPI_File_get_position_shared(file, &offset)
		       ); break;
    }
    /* return file position */
    return PyMPIOffset_FromOffset(offset);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
file_set_atomicity(PyObject *self, PyObject *args)
{
  MPI_File file;
  int flag;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&i",
			 PyO2Fl, &file,
			 &flag) ) return NULL;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);
  
  TRY {
    /* set file atomicity */
    MPI4PY_CALL_FILE(file,
		     MPI_File_set_atomicity(file, flag));
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
file_get_atomicity(PyObject *self, PyObject *args)
{
  MPI_File file;
  int flag;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Fl, &file) ) return NULL;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);
  
  TRY {
    /* get file atomicity */
    MPI4PY_CALL_FILE(file,
		     MPI_File_get_atomicity(file, &flag));
    /* return */
    return PyBool_FromLong((long)flag);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
file_sync(PyObject *self, PyObject *args)
{
  MPI_File file;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Fl, &file) ) return NULL;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);
  
  TRY {
    /* sync file */
    MPI4PY_CALL_FILE(file,
		     MPI_File_sync(file));
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
file_get_byte_offset(PyObject *self, PyObject *args)
{
  MPI_File file;
  MPI_Offset offset;
  MPI_Offset disp;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O&",
			 PyO2Fl, &file,
			 PyO2Of, &offset) ) return NULL;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);
  
  TRY {
    /* get file byte offset */
    MPI4PY_CALL_FILE(file,
		     MPI_File_get_byte_offset(file, offset, &disp) );
    /* return byte offset */
    return PyMPIOffset_FromOffset(disp);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
file_get_type_extent(PyObject *self, PyObject *args)
{
  MPI_File file;
  MPI_Datatype datatype;
  MPI_Aint extent;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O&",
			 PyO2Fl, &file,
			 PyO2Dt, &datatype) ) return NULL;

  MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);
  
  TRY {
    /* get file datatype extent */
    MPI4PY_CALL_FILE(file,
		     MPI_File_get_type_extent(file, datatype, 
					      &extent) );
    /* return extent */
    return PyMPIAint_FromAint(extent);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static PyObject *
file_get_errhandler(PyObject *self, PyObject *args)
{
  MPI_File file;
  MPI_Errhandler errhandler;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Fl,  &file) ) return NULL;

  /*MPI4PY_CHECK_NULL(file, MPI_FILE_NULL, MPI_ERR_FILE);*/
  
  TRY {
    /* get errhandler */
    MPI4PY_CHECK( MPI_File_get_errhandler(file, &errhandler) );
#if defined(LAM_MPI)
    if (errhandler == MPI_ERRORS_RETURN ||
	errhandler == MPI_ERRORS_ARE_FATAL)
      errhandler->eh_refcount++;
#endif
    /* return errhandler */
    return PyMPIErrhandler_FromErrhandler(errhandler);
  }
  EXCEPT {
    return NULL;
  }

}
static PyObject *
file_set_errhandler(PyObject *self, PyObject *args)
{
  MPI_File file;
  MPI_Errhandler errhandler;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&O&",
			 PyO2Fl, &file,
			 PyO2Eh, &errhandler) ) return NULL;

  /*MPI4PY_CHECK_NULL(file,       MPI_FILE_NULL,       MPI_ERR_FILE);*/
  MPI4PY_CHECK_NULL(errhandler, MPI_ERRHANDLER_NULL, MPI_ERR_ARG);

  TRY {
    /* set errhandler */
    MPI4PY_CHECK( MPI_File_set_errhandler(file, errhandler) );
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/

#else /* !HAVE_MPI_FILE_OPEN */

static PyObject *
file_mpinoimpl(PyObject *self, PyObject *args)
{
  MPI4PY_RAISE_MPINOIMPL("MPI_FILE");
 fail:
  return NULL;
}

#define file_delete            file_mpinoimpl
#define file_set_size          file_mpinoimpl
#define file_preallocate       file_mpinoimpl
#define file_get_size          file_mpinoimpl
#define file_get_group         file_mpinoimpl
#define file_get_amode         file_mpinoimpl
#define file_set_info          file_mpinoimpl
#define file_get_info          file_mpinoimpl
#define file_set_view          file_mpinoimpl
#define file_get_view          file_mpinoimpl

#define file_read              file_mpinoimpl
#define file_write             file_mpinoimpl
#define file_iread             file_mpinoimpl
#define file_iwrite            file_mpinoimpl
#define file_read_split        file_mpinoimpl
#define file_write_split       file_mpinoimpl
#define file_seek              file_mpinoimpl
#define file_get_position      file_mpinoimpl

#define file_set_atomicity     file_mpinoimpl
#define file_get_atomicity     file_mpinoimpl
#define file_sync              file_mpinoimpl

#define file_get_byte_offset   file_mpinoimpl

#define file_get_type_extent   file_mpinoimpl

#define file_get_errhandler    file_mpinoimpl
#define file_set_errhandler    file_mpinoimpl


#endif /* HAVE_MPI_FILE_OPEN */


/*------------------------------------------------------------------*/


/*------------------------------------------------------------------*/
/* Naming Objects                                                   */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static PyObject *
comm_get_name(PyObject *self, PyObject *args)
{  
  MPI_Comm comm;
  char *comm_name;
  int resultlen;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Cm, &comm) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
 
  TRY {
    /* get comm name */
#if HAVE_MPI_COMM_GET_NAME
    char buf[MPI_MAX_OBJECT_NAME+1] = { '\0' };
    comm_name = buf; resultlen = 0;
    MPI4PY_CALL_COMM(comm,  
		     MPI_Comm_get_name(comm, comm_name, &resultlen) );
#   if (defined(MPICH_NAME) && MPICH_NAME == 1)
    resultlen = strlen(comm_name);
#   endif
#else
    comm_name = NULL; resultlen = 0;
    MPI4PY_RAISE_MPINOIMPL("MPI_COMM_GET_NAME");
#endif
    /* return comm name */
    return Py_BuildValue("s#", comm_name, resultlen);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_set_name(PyObject *self, PyObject *args)
{  
  MPI_Comm comm;
  char *comm_name;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&s",
			 PyO2Cm, &comm,
			 &comm_name) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);
 
  TRY {
    /* set comm name */
#if HAVE_MPI_COMM_SET_NAME
#   if (defined(MPICH_NAME) && MPICH_NAME == 1)
    char buf[MPI_MAX_OBJECT_NAME] = { '0' };
    PyOS_snprintf(buf, MPI_MAX_OBJECT_NAME, "%s", comm_name);
    comm_name = buf;
#   endif
    MPI4PY_CALL_COMM(comm,
		     MPI_Comm_set_name(comm, comm_name) );
#else
    MPI4PY_RAISE_MPINOIMPL("MPI_COMM_SET_NAME");
#endif
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static PyObject *
type_get_name(PyObject *self, PyObject *args)
{  
  MPI_Datatype datatype;
  char *type_name;
  int resultlen;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Dt, &datatype) ) return NULL;

  MPI4PY_CHECK_NULL(datatype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);
 
  TRY {
    /* get type name */
#if HAVE_MPI_TYPE_GET_NAME
    char buf[MPI_MAX_OBJECT_NAME] = { '0' };
    type_name = buf; resultlen = 0;
    MPI4PY_CALL( MPI_Type_get_name(datatype, type_name, &resultlen) );
#else
    type_name = NULL; resultlen = 0;
    MPI4PY_RAISE_MPINOIMPL("MPI_TYPE_GET_NAME");
#endif
    /* return type name */
    return Py_BuildValue("s#", type_name, resultlen);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
type_set_name(PyObject *self, PyObject *args)
{  
  MPI_Datatype datatype;
  char *type_name;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&s",
			 PyO2Dt, &datatype,
			 &type_name) ) return NULL;

  MPI4PY_CHECK_NULL(datatype, MPI_DATATYPE_NULL, MPI_ERR_TYPE);
 
  TRY {
    /* set type name */
#if HAVE_MPI_TYPE_SET_NAME
    MPI4PY_CALL( MPI_Type_set_name(datatype, type_name) );
#else
    MPI4PY_RAISE_MPINOIMPL("MPI_TYPE_SET_NAME");
#endif
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static PyObject *
win_get_name(PyObject *self, PyObject *args)
{  
  MPI_Win win;
  char *win_name;
  int resultlen;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&",
			 PyO2Wn, &win) ) return NULL;

  MPI4PY_CHECK_NULL(win, MPI_WIN_NULL, MPI_ERR_WIN);
 
  TRY {
    /* get win name */
#if HAVE_MPI_WIN_GET_NAME
    char buf[MPI_MAX_OBJECT_NAME] = { '0' };
    win_name = buf; resultlen = 0;
    MPI4PY_CALL_WIN(win, 
                    MPI_Win_get_name(win, win_name, &resultlen) );
#else
    win_name = NULL; resultlen = 0;
    MPI4PY_RAISE_MPINOIMPL("MPI_WIN_GET_NAME");
#endif
    /* return win name */
    return Py_BuildValue("s#", win_name, resultlen);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
win_set_name(PyObject *self, PyObject *args)
{  
  MPI_Win win;
  char *win_name;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&s",
			 PyO2Wn, &win,
			 &win_name) ) return NULL;

  MPI4PY_CHECK_NULL(win, MPI_WIN_NULL, MPI_ERR_WIN);
 
  TRY {
    /* set win name */
#if HAVE_MPI_WIN_SET_NAME
    MPI4PY_CALL_WIN(win, 
                    MPI_Win_set_name(win, win_name) );
#else
    MPI4PY_RAISE_MPINOIMPL("MPI_WIN_SET_NAME");
#endif
    /* return */
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/





/*------------------------------------------------------------------*/
/* Environmental Management                                         */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/

/* module initialization flag */
static int module_called_init = 0;
/* module finalization flag */
static int module_called_finalize = 0;

static int
env_mpi_finalize(void) {
  int ierr, flag_i, flag_f;
  MPI_Initialized(&flag_i);
  if (!flag_i) {
    PyErr_SetString(PyExc_RuntimeError,
		    "MPI_Init() has not been called");
    return -1;
  }
  ierr = MPI_Finalized(&flag_f);
  if (!flag_f){ 
    if (PyMPI_COMM_SELF  != MPI_COMM_NULL)
      ierr = MPI_Comm_free(&PyMPI_COMM_SELF);
    if (PyMPI_COMM_WORLD != MPI_COMM_NULL)
      ierr = MPI_Comm_free(&PyMPI_COMM_WORLD);
    if (module_called_init) {
      ierr = MPI_Finalize();
      if (ierr != MPI_SUCCESS ) {
        PyErr_Format(PyExc_RuntimeError,
                     "MPI_Finalize() failed [error code: %d]", ierr);
        return -1;
      }
      module_called_finalize = 1;
    }
  }
  return 0;
}

#define mpi_finalize env_mpi_finalize

static void
env_mpi_atexit(void) {
  int ierr, flag_i, flag_f;
  ierr = MPI_Initialized(&flag_i);
  ierr = MPI_Finalized(&flag_f);
  if (!flag_i || flag_f) return;
  if (PyMPI_COMM_SELF != MPI_COMM_NULL)
    ierr = MPI_Comm_free(&PyMPI_COMM_SELF);
  if (PyMPI_COMM_WORLD != MPI_COMM_NULL)
    ierr = MPI_Comm_free(&PyMPI_COMM_WORLD);
  if (!module_called_init || module_called_finalize) return;
  ierr = MPI_Finalize();
  module_called_finalize = 1;
  if (ierr != MPI_SUCCESS) {
    fflush(stderr);
    fprintf(stderr,"MPI_Finalize() failed [error code: %d]", ierr);
    fflush(stderr);
  }
}

static int
env_mpi_init_any(int threaded, int required, int *provided) {
  int ierr, flag_i=0, flag_f=0, level=0;
  ierr = MPI_Initialized(&flag_i);
  ierr = MPI_Finalized(&flag_f);
  if (flag_f) {
    PyErr_SetString(PyExc_RuntimeError,
		    "MPI_Finalize() has been already called");
    return -1;
  }
  if (!flag_i) {
    int argc = 0; char **argv = NULL;
#if defined(MPICH_NAME) && MPICH_NAME==1
    /* this does not actually work with MPICH 1, */
    /* but avoids a nasty segfault               */
    char *args[2];
    argc = 1; argv = args;
    argv[0] = Py_GetProgramName();
    argv[1] = NULL;
#endif
    /* initialize MPI */
#if HAVE_MPI_INIT_THREAD
    if (threaded) {
      ierr = MPI_Init_thread(&argc, &argv, required, &level);
      if (provided) *provided = level;
    }
    else {
      ierr = MPI_Init(&argc, &argv);
    }
#else
    ierr = MPI_Init(&argc, &argv);
#endif
    if (ierr != MPI_SUCCESS) {
      PyErr_Format(PyExc_RuntimeError,
		   "MPI_Init() failed [error code: %d]\n", ierr);
      return -1;
    }
    module_called_init = 1; /* set init flag */
    if (Py_AtExit(env_mpi_atexit) < 0)
      if(PyErr_WarnEx(PyExc_RuntimeWarning,
		      "cannot register MPI_Finalize() "
		      "with Py_AtExit()", 1) < 0) return -1;
  } 
  else if (threaded) {
#if HAVE_MPI_QUERY_THREAD
    ierr = MPI_Query_thread(&level);
#endif    
    if (provided) *provided = level;
  }

#if MPI4PY_ENABLE_FAST_ERRCHECK
  {
    ierr = MPI4PY_COMM_SET_EH(MPI_COMM_SELF,  MPI_ERRORS_RETURN);
    ierr = MPI4PY_COMM_SET_EH(MPI_COMM_WORLD, MPI_ERRORS_RETURN);
#if HAVE_MPI_FILE_OPEN
    ierr = MPI4PY_FILE_SET_EH(MPI_FILE_NULL,  MPI_ERRORS_RETURN);
#endif
  }
#endif
  return 0;
}

#define mpi_init() \
        env_mpi_init_any(0, 0, NULL)

#define mpi_init_thread(required, provided) \
         env_mpi_init_any(1, required, provided)


/*------------------------------------------------------------------*/
static PyObject *
env_init(PyObject *self, PyObject *args)
{
  /* try to initialize MPI */
  if (mpi_init() < 0) 
    return NULL;
  else
    Py_RETURN_NONE;

}
/*------------------------------------------------------------------*/
static PyObject *
env_init_thread(PyObject *self, PyObject *args)
{
  int required;
  int provided=0;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "i",
                         &required) ) return NULL;
  TRY {
#if HAVE_MPI_INIT_THREAD
    if (mpi_init_thread(required, &provided) < 0) RAISE;
    return PyInt_FromLong((long)provided);
#else
    required = provided = 0;
    MPI4PY_RAISE_MPINOIMPL("MPI_INIT_THREAD");
#endif
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
env_initialized(PyObject *self, PyObject *args)
{
  int flag;
  MPI_Initialized(&flag);
  return PyBool_FromLong((long)flag);
  
}
/*------------------------------------------------------------------*/

static PyObject *
env_finalize(PyObject *self, PyObject *args)
{
  /* try to finalize MPI */
  if (mpi_finalize() < 0) 
    return NULL;
  else
    Py_RETURN_NONE;

}
/*------------------------------------------------------------------*/
static PyObject *
env_finalized(PyObject *self, PyObject *args)
{  
  int flag;
  MPI_Finalized(&flag);
  return PyBool_FromLong((long)flag);

}
/*------------------------------------------------------------------*/
static PyObject *
env_query_thread(PyObject *self, PyObject *args)
{
  int provided;

  TRY {
#if HAVE_MPI_QUERY_THREAD
    MPI4PY_CHECK( MPI_Query_thread(&provided) );
    return PyInt_FromLong((long)provided);
#else
    provided = 0;
    MPI4PY_RAISE_MPINOIMPL("MPI_QUERY_THREAD");
#endif
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
env_is_thread_main(PyObject *self, PyObject *args)
{
  int flag;

  TRY {
#if HAVE_MPI_IS_THREAD_MAIN
    MPI4PY_CHECK( MPI_Is_thread_main(&flag) );
    return PyBool_FromLong((long)flag);
#else
    flag = 0;
    MPI4PY_RAISE_MPINOIMPL("MPI_IS_THREAD_MAIN");
#endif
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
comm_abort(PyObject *self, PyObject *args)
{
  MPI_Comm comm;
  int errorcode;
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&i",
			 PyO2Cm, &comm,
			 &errorcode) ) return NULL;

  MPI4PY_CHECK_NULL(comm, MPI_COMM_NULL, MPI_ERR_COMM);

  TRY {
    /* abort */
    MPI4PY_CALL_COMM(comm, 
		     MPI_Abort(comm, errorcode) );
    Py_RETURN_NONE;
  }
  
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/
static PyObject *
env_get_version(PyObject *self, PyObject *args)
{
  int version = 1;
  int subversion = 0;

#if HAVE_MPI_GET_VERSION
  MPI_Get_version(&version, &subversion);
#else
#if defined(MPI_VERSION)
  version = MPI_VERSION;
#endif
#if defined(MPI_SUBVERSION)
  subversion = MPI_SUBVERSION;
#endif
#endif
  return Py_BuildValue("ii", version, subversion);
}
/*------------------------------------------------------------------*/
static PyObject *
env_get_proc_name(PyObject *self, PyObject *args)
{
  char name[MPI_MAX_PROCESSOR_NAME] = { '0' };
  int  len = 0;
  
  TRY {
    MPI4PY_CALL( MPI_Get_processor_name(name, &len) );
    return Py_BuildValue("s#", name, len);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static PyObject *
env_wtime(PyObject *self, PyObject *args)
{
  return PyFloat_FromDouble( MPI_Wtime() );

}
/*------------------------------------------------------------------*/
static PyObject *
env_wtick(PyObject *self, PyObject *args)
{  
  return PyFloat_FromDouble( MPI_Wtick() );

}
/*------------------------------------------------------------------*/


/*------------------------------------------------------------------*/
static PyObject *
errhandler_free(PyObject *self, PyObject *args)
{
  MPI_Errhandler* errhandler;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O&", 
			 PyO2EhP, &errhandler) ) return NULL;
  
  MPI4PY_CHECK_NULL(*errhandler, MPI_ERRHANDLER_NULL, MPI_ERR_ARG);

  TRY {
    /* free errhandler */
#if defined(MPICH) || defined(MPICH2) || defined(MPICH_NAME) 
    MPI4PY_CALL_FREE( MPI_Errhandler_free(errhandler) );
#elif defined(LAM_MPI)
    MPI_Errhandler errhdl = *errhandler;
    if (errhdl != MPI_ERRORS_RETURN &&
	errhdl != MPI_ERRORS_ARE_FATAL) {
      MPI4PY_CALL_FREE( MPI_Errhandler_free(&errhdl) );
    } else {
      errhdl->eh_refcount--;
      errhdl = MPI_ERRHANDLER_NULL;
    }
    *errhandler = errhdl; 
#elif defined(OPEN_MPI)
    MPI_Errhandler errhdl = *errhandler;
    if (errhdl != MPI_ERRORS_RETURN &&
	errhdl != MPI_ERRORS_ARE_FATAL) {
      MPI4PY_CALL_FREE( MPI_Errhandler_free(&errhdl) );
    } else {
      errhdl = MPI_ERRHANDLER_NULL;
    }
    *errhandler = errhdl; 
#else
    MPI4PY_CALL_FREE( MPI_Errhandler_free(errhandler) );
#endif
    Py_RETURN_NONE;
  }

  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/



/*------------------------------------------------------------------*/
static PyObject *
env_error_string(PyObject *self, PyObject *args)
{
  int errorcode;
  char buf[MPI_MAX_ERROR_STRING+1] = { '0' };
  int  len;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "i", 
			 &errorcode) ) return NULL;
  
  TRY {
    /* get error string */
    if (errorcode >= MPI_SUCCESS && errorcode < MPI_ERR_LASTCODE) {
      MPI4PY_CALL( MPI_Error_string(errorcode, buf, &len) );
    } else {
      len = PyOS_snprintf(buf, MPI_MAX_ERROR_STRING,
                          "ierr:%d, unable to retrieve error string, "
                          "out of range [MPI_SUCCESS=%d, MPI_ERR_LASTCODE=%d)",
                          errorcode, MPI_SUCCESS, MPI_ERR_LASTCODE);
    }
    /* return result */
#if defined(LAM_MPI)
    if (errorcode >= MPI_SUCCESS && errorcode < MPI_ERR_LASTCODE)
      if (len > 18)
	return Py_BuildValue("s#", buf+18, len-18);
#endif
    return Py_BuildValue("s#", buf, len);
  }
  EXCEPT {
    return NULL;
  }
  
}
/*------------------------------------------------------------------*/
static PyObject *
env_error_class(PyObject *self, PyObject *args)
{
  int errorcode;
  int errorclass;

  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "i", 
			 &errorcode) ) return NULL;

  if (errorcode < MPI_SUCCESS || errorcode >= MPI_ERR_LASTCODE) {
    PyErr_Format(PyExc_ValueError, 
		 "invalid error code %d, out of range "
		 "[MPI_SUCCESS=%d, MPI_ERR_LASTCODE=%d)",
		 errorcode, MPI_SUCCESS, MPI_ERR_LASTCODE);
    return NULL;
  }
  
  TRY {
    /* get error class */
    MPI4PY_CALL( MPI_Error_class(errorcode, &errorclass) );
    /* return error class */
    return PyInt_FromLong((long)errorclass);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/



/*------------------------------------------------------------------*/
/* Helpers                                                          */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static PyObject *
_set_exception(PyObject *self, PyObject *args)
{
  PyObject *exception = PyExc_RuntimeError;
  PyObject *warning = PyExc_UserWarning;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "O|O",
			 &exception, 
			 &warning) ) return NULL;
  
  /* set exception class */
  if (exception != PyExc_MPIError) {
    if (exception != PyExc_RuntimeError) 
      { Py_INCREF(exception); }
    if (PyExc_MPIError != PyExc_RuntimeError) 
      { Py_XDECREF(PyExc_MPIError); }
    PyExc_MPIError = exception;
  }

  /* set warning class */
  if (warning != PyExc_MPIWarning) {
    if (warning != PyExc_UserWarning)
      { Py_INCREF(warning); }
    if (PyExc_MPIWarning != PyExc_UserWarning) 
      { Py_XDECREF(PyExc_MPIWarning); }
    PyExc_MPIWarning = warning;
  }

  /* return */
  Py_RETURN_NONE;
  
}
static PyObject *
_del_exception(PyObject *self, PyObject *args)
{
  /* clear exception class */
  if (PyExc_MPIError != PyExc_RuntimeError) {
    Py_XDECREF(PyExc_MPIError);
    PyExc_MPIError = PyExc_RuntimeError;
  }

  /* clear warning class */
  if (PyExc_MPIWarning != PyExc_UserWarning) {
    Py_XDECREF(PyExc_MPIWarning);
    PyExc_MPIError = PyExc_UserWarning;
  }

  /* return */
  Py_RETURN_NONE;
  
}
/*------------------------------------------------------------------*/
static PyObject *
_raise(PyObject *self, PyObject *args)
{
  int errorcode;
  
  /* parse input arguments */
  if ( !PyArg_ParseTuple(args, "i", 
			 &errorcode) ) return NULL;
  
  if (errorcode < MPI_SUCCESS || errorcode >= MPI_ERR_LASTCODE)
    errorcode = MPI_ERR_UNKNOWN;

  TRY {
    MPI4PY_CHECK( errorcode );
    Py_RETURN_NONE;
  }
  EXCEPT {
    return NULL;
  }
  
}
/*------------------------------------------------------------------*/
/* from: Python-2.5.1/Python/bltinmodule.c */
static PyObject *
py2_reduce(PyObject *self, PyObject *args)
{
	PyObject *seq, *func, *result = NULL, *it;

	if (!PyArg_UnpackTuple(args, "reduce", 2, 3, &func, &seq, &result))
		return NULL;
	if (result != NULL)
		Py_INCREF(result);

	it = PyObject_GetIter(seq);
	if (it == NULL) {
		PyErr_SetString(PyExc_TypeError,
		    "reduce() arg 2 must support iteration");
		Py_XDECREF(result);
		return NULL;
	}

	if ((args = PyTuple_New(2)) == NULL)
		goto Fail;

	for (;;) {
		PyObject *op2;

		if (args->ob_refcnt > 1) {
			Py_DECREF(args);
			if ((args = PyTuple_New(2)) == NULL)
				goto Fail;
		}

		op2 = PyIter_Next(it);
		if (op2 == NULL) {
			if (PyErr_Occurred())
				goto Fail;
 			break;
		}

		if (result == NULL)
			result = op2;
		else {
			PyTuple_SetItem(args, 0, result);
			PyTuple_SetItem(args, 1, op2);
			if ((result = PyEval_CallObject(func, args)) == NULL)
				goto Fail;
		}
	}

	Py_DECREF(args);

	if (result == NULL)
		PyErr_SetString(PyExc_TypeError,
			   "reduce() of empty sequence with no initial value");

	Py_DECREF(it);
	return result;

Fail:
	Py_XDECREF(args);
	Py_XDECREF(result);
	Py_DECREF(it);
	return NULL;
}
/*------------------------------------------------------------------*/



/*------------------------------------------------------------------*/
/* Misc                                                             */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/
static PyObject *
get_mpi_info(PyObject *self, PyObject *args)
{
  const char* name = "unknown";
  int major = 0;
  int minor = 0;
  int micro = 0;

  /* MPICH2 */
#if defined(MPICH_NAME) && MPICH_NAME==2
  name = "MPICH2";
  #if defined(MPICH2_VERSION)
  sscanf(MPICH2_VERSION,"%d.%d.%d",&major,&minor,&micro);
  #endif
#endif

  /* OpenMPI */
#if defined(OPEN_MPI)
  name = "OpenMPI";
  #if defined(OMPI_MAJOR_VERSION)
  major = OMPI_MAJOR_VERSION;
  #endif
  #if defined(OMPI_MINOR_VERSION)
  minor = OMPI_MINOR_VERSION;
  #endif
  #if defined(OMPI_RELEASE_VERSION)
  micro = OMPI_RELEASE_VERSION;
  #endif
#endif

  /* DeinoMPI */
#if defined(DEINO_MPI)
  name = "DeinoMPI";
#endif

  /* MPICH */
#if defined(MPICH_NAME) && MPICH_NAME==1
  name = "MPICH";
  #if defined(MPICH_VERSION)
  sscanf(MPICH_VERSION,"%d.%d.%d",&major,&minor,&micro);
  #endif
#endif

  /* LAM/MPI */
#if defined(LAM_MPI)  
  name = "LAM/MPI";
  #if defined(LAM_MAJOR_VERSION)
  major = LAM_MAJOR_VERSION;
  #endif
  #if defined(LAM_MINOR_VERSION)
  minor = LAM_MINOR_VERSION;
  #endif
  #if defined(LAM_RELEASE_VERSION)
  micro = LAM_RELEASE_VERSION;
  #endif
#endif
  
  return Py_BuildValue("s(iii)", name, major, minor, micro);
  
}

/*------------------------------------------------------------------*/
static PyObject *
distribute(PyObject *self, PyObject *args)
{  
  int N,B,i;
  
  /* parse input arguments */
  if (!PyArg_ParseTuple(args, "iii",
			&N,&B,&i) ) return NULL;
  
  TRY {

    int low,high;
    
    if ( N < 0 )
      PY_RAISE(PyExc_ValueError,"argument 1 less than zero");
    if ( B < 1 )
      PY_RAISE(PyExc_ValueError,"argument 2 less than one");
    if ( i < 0 || i >= B )
      PY_RAISE(PyExc_ValueError,"argument 3 out of range");

    {    
      int C = N / B; /* integer division */
      int R = N % B; /* remainder */
      if (i < R) {
	low  = i*C + i;
	high = low + C + 1;
      }
      else {
	low  = i*C + R;
	high = low + C;
      }
    }
    return Py_BuildValue("ii", low, high);
  }
  EXCEPT {
    return NULL;
  }

}
/*------------------------------------------------------------------*/






/*------------------------------------------------------------------*/
/* Python Module                                                    */
/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/

PyDoc_STRVAR(mpi_doc, "MPI extension module");

/*------------------------------------------------------------------*/

static struct PyMethodDef mpi_methods[] = 
{
  /* initializers */
  {"op_init"          , op_init         , METH_VARARGS},
  {"comm_init"        , comm_init       , METH_VARARGS},
  {"group_init"       , group_init      , METH_VARARGS},
  {"status_init"      , status_init     , METH_VARARGS},
  {"request_init"     , request_init    , METH_VARARGS},
  {"datatype_init"    , datatype_init   , METH_VARARGS},
  {"errhandler_init"  , errhandler_init , METH_VARARGS},
  {"win_init"         , win_init        , METH_VARARGS},
  {"info_init"        , info_init       , METH_VARARGS},
  {"file_init"        , file_init       , METH_VARARGS},

  /* clear handles (set them to MPI_XXX_NULL) */
  {"op_clear"          , op_clear         , METH_VARARGS},
  {"comm_clear"        , comm_clear       , METH_VARARGS},
  {"group_clear"       , group_clear      , METH_VARARGS},
  /*{"status_clear"      , status_clear     , METH_VARARGS},*/
  {"request_clear"     , request_clear    , METH_VARARGS},
  {"datatype_clear"    , datatype_clear   , METH_VARARGS},
  {"errhandler_clear"  , errhandler_clear , METH_VARARGS},
  {"win_clear"         , win_clear        , METH_VARARGS},
  {"info_clear"        , info_clear       , METH_VARARGS},
  {"file_clear"        , file_clear       , METH_VARARGS},

  /* Datatype */
  {"type_get_extent",    type_get_extent,    METH_VARARGS},
  {"type_ex",            type_ex,            METH_VARARGS},
  {"type_lb",            type_lb,            METH_VARARGS},
  {"type_ub",            type_ub,            METH_VARARGS},
  {"type_size",          type_size,          METH_VARARGS},
  {"type_dup",           type_dup,           METH_VARARGS},
  {"type_contiguous",    type_contiguous,    METH_VARARGS},
  {"type_vector",        type_vector,        METH_VARARGS},
  {"type_hvector",       type_hvector,       METH_VARARGS},
  {"type_indexed",       type_indexed,       METH_VARARGS},
  {"type_indexed_block", type_indexed_block, METH_VARARGS},
  {"type_hindexed",      type_hindexed,      METH_VARARGS},
  {"type_subarray",      type_subarray,      METH_VARARGS},
  {"type_darray",        type_darray,        METH_VARARGS},
  {"type_struct",        type_struct,        METH_VARARGS},
  {"type_resized",       type_resized,       METH_VARARGS},
  {"type_true_extent",   type_true_extent,   METH_VARARGS},
  {"type_commit",        type_commit,        METH_VARARGS},
  {"type_free",          type_free,          METH_VARARGS},

  {"pack",               type_pack,               METH_VARARGS},
  {"unpack",             type_unpack,             METH_VARARGS},
  {"pack_size",          type_pack_size,          METH_VARARGS},
  {"pack_external",      type_pack_external,      METH_VARARGS},
  {"unpack_external",    type_unpack_external,    METH_VARARGS},
  {"pack_external_size", type_pack_external_size, METH_VARARGS},

  /* Group */
  {"group_size",        group_size,        METH_VARARGS},
  {"group_rank",        group_rank,        METH_VARARGS},
  {"group_transl_rank", group_transl_rank, METH_VARARGS},
  {"group_compare",     group_compare,     METH_VARARGS},
  {"comm_group",        comm_group,        METH_VARARGS},
  {"group_union",       group_union,       METH_VARARGS},
  {"group_intersection",group_intersection,METH_VARARGS},
  {"group_difference",  group_difference,  METH_VARARGS},
  {"group_incl",        group_incl,        METH_VARARGS},
  {"group_excl",        group_excl,        METH_VARARGS},
  {"group_range_incl",  group_range_incl,  METH_VARARGS},
  {"group_range_excl",  group_range_excl,  METH_VARARGS},
  {"group_free",        group_free,        METH_VARARGS},

  /* Status */
  {"get_count",      status_get_count, METH_VARARGS},
  {"get_elements",   status_get_elems, METH_VARARGS},
  {"test_cancelled", test_cancelled,   METH_VARARGS},

  /* Request */
  {"request_free" ,      request_free,       METH_VARARGS},
  {"request_get_status", request_get_status, METH_VARARGS},
  {"wait",               request_wait,       METH_VARARGS},
  {"test",               request_test,       METH_VARARGS},
  {"cancel",             request_cancel,     METH_VARARGS},
  {"waitany",            request_waitany,    METH_VARARGS},
  {"testany",            request_testany,    METH_VARARGS},
  {"waitall",            request_waitall,    METH_VARARGS},
  {"testall",            request_testall,    METH_VARARGS},
  {"waitsome",           request_waitsome,   METH_VARARGS},
  {"testsome",           request_testsome,   METH_VARARGS},
  {"start",              request_start,      METH_VARARGS},
  {"startall",           request_startall,   METH_VARARGS},

  /* Comm - Intracomm */
  {"comm_check_any"    , comm_check_any   , METH_VARARGS},
  {"comm_check_intra"  , comm_check_intra , METH_VARARGS},
  {"comm_check_inter"  , comm_check_inter , METH_VARARGS},
  {"comm_check_cart"   , comm_check_cart  , METH_VARARGS},
  {"comm_check_graph"  , comm_check_graph , METH_VARARGS},
  
  {"comm_world", comm_world, METH_VARARGS},
  {"comm_self",  comm_self,  METH_VARARGS},

  {"comm_size",       comm_size,       METH_VARARGS},
  {"comm_rank",       comm_rank,       METH_VARARGS},
  {"comm_compare",    comm_compare,    METH_VARARGS},
  {"comm_dup",        comm_dup,        METH_VARARGS},
  {"comm_create",     comm_create,     METH_VARARGS},
  {"comm_split",      comm_split,      METH_VARARGS},
  {"comm_free",       comm_free,       METH_VARARGS},
  {"comm_test_intra", comm_test_intra, METH_VARARGS},

  {"open_port",       open_port,       METH_VARARGS},
  {"close_port",      close_port,      METH_VARARGS},
  {"publish_name",    publish_name,    METH_VARARGS},
  {"unpublish_name",  unpublish_name,  METH_VARARGS},
  {"lookup_name",     lookup_name,     METH_VARARGS},
  {"comm_accept",     comm_accept,     METH_VARARGS},
  {"comm_connect",    comm_connect,    METH_VARARGS},
  {"comm_disconnect", comm_disconnect, METH_VARARGS},
  {"comm_spawn",      comm_spawn,      METH_VARARGS},
  {"comm_get_parent", comm_get_parent, METH_NOARGS},
  {"comm_join",       comm_join,       METH_VARARGS},

  /* Helper */
  {"make_buf",  make_buf, METH_VARARGS},

  /* Point-to-Point Communications */
  {"send_pickled",       comm_send_pickled,            METH_VARARGS},
  {"recv_pickled",       comm_recv_pickled,            METH_VARARGS},
  {"sendrecv_pickled",   comm_sendrecv_pickled,        METH_VARARGS},

  {"send",              comm_send_buffer,             METH_VARARGS},
  {"recv",              comm_recv_buffer,             METH_VARARGS},
  {"sendrecv",          comm_sendrecv_buffer,         METH_VARARGS},
  {"sendrecv_replace",  comm_sendrecv_replace_buffer, METH_VARARGS},
  {"isend",             comm_isend_buffer,            METH_VARARGS},
  {"irecv",             comm_irecv_buffer,            METH_VARARGS},
  {"send_init",         comm_send_init_buffer,        METH_VARARGS},
  {"recv_init",         comm_recv_init_buffer,        METH_VARARGS},
  {"probe",             comm_probe,                   METH_VARARGS},
  {"iprobe",            comm_iprobe,                  METH_VARARGS},

  /* Collective Communications */
  {"barrier",          comm_barrier,           METH_VARARGS},
		       		   
  {"bcast_pickled",     comm_bcast_pickled,     METH_VARARGS},
  {"gather_pickled",    comm_gather_pickled,    METH_VARARGS},
  {"scatter_pickled",   comm_scatter_pickled,   METH_VARARGS},
  {"allgather_pickled", comm_allgather_pickled, METH_VARARGS},
  {"alltoall_pickled",  comm_alltoall_pickled,   METH_VARARGS},

  {"bcast",            comm_bcast_buffer,      METH_VARARGS},
  {"gather",           comm_gather_buffer,     METH_VARARGS},
  {"scatter",          comm_scatter_buffer,    METH_VARARGS},
  {"allgather",        comm_allgather_buffer,  METH_VARARGS},
  {"alltoall",         comm_alltoall_buffer,   METH_VARARGS},

  /* Operations */
  {"op_create", op_create, METH_VARARGS},
  {"op_free",   op_free,   METH_VARARGS},

  /* Global Reduction Operations */
  {"reduce",    comm_reduce_buffer,    METH_VARARGS},
  {"allreduce", comm_allreduce_buffer, METH_VARARGS},
  {"scan",      comm_scan_buffer,      METH_VARARGS},
  {"exscan",    comm_exscan_buffer,    METH_VARARGS},

  /* Intercomm */
  {"comm_test_inter",   comm_test_inter,   METH_VARARGS},
  {"comm_remote_size",  comm_remote_size,  METH_VARARGS},
  {"comm_remote_group", comm_remote_group, METH_VARARGS},
  {"intercomm_create",  intercomm_create,  METH_VARARGS},
  {"intercomm_merge",   intercomm_merge,   METH_VARARGS},

  /* Topologies */
  {"topo_test",         topo_test,         METH_VARARGS},
  {"cart_create",       cart_create,       METH_VARARGS},
  {"dims_create",       dims_create,       METH_VARARGS},
  {"cartdim_get",       cartdim_get,       METH_VARARGS},
  {"cart_get",          cart_get,          METH_VARARGS},
  {"cart_rank",         cart_rank,         METH_VARARGS},
  {"cart_coords",       cart_coords,       METH_VARARGS},
  {"cart_shift",        cart_shift,        METH_VARARGS},
  {"cart_sub",          cart_sub,          METH_VARARGS},
  {"cart_map",          cart_map,          METH_VARARGS},

  {"graph_create",      graph_create,      METH_VARARGS},
  {"graphdims_get",     graphdims_get,     METH_VARARGS},
  {"graph_get",         graph_get,         METH_VARARGS},
  {"graph_neigh_count", graph_neigh_count, METH_VARARGS},
  {"graph_neigh",       graph_neigh,       METH_VARARGS},
  {"graph_map",         graph_map,         METH_VARARGS},

  /* Info */
  {"info_create",       info_create,     METH_VARARGS},
  {"info_dup",          info_dup,        METH_VARARGS},
  {"info_free",         info_free,       METH_VARARGS},
  {"info_get",          info_get,        METH_VARARGS},
  {"info_set",          info_set,        METH_VARARGS},
  {"info_delete",       info_delete,     METH_VARARGS},
  {"info_get_nkeys",    info_get_nkeys,  METH_VARARGS},
  {"info_get_nthkey",   info_get_nthkey, METH_VARARGS},
  {"info_get_valuelen", info_get_vlen,   METH_VARARGS},

  /* Window */
  {"win_create",     win_create,     METH_VARARGS},
  {"win_free",       win_free,       METH_VARARGS},
  {"win_get_memory", win_get_memory, METH_VARARGS},
  {"win_get_attrs",  win_get_attrs,  METH_VARARGS},
  {"win_get_group",  win_get_group,  METH_VARARGS},
  {"win_put",        win_put,        METH_VARARGS},
  {"win_get",        win_get,        METH_VARARGS},
  {"win_accumulate", win_accumulate, METH_VARARGS},
  {"win_fence",      win_fence,      METH_VARARGS},
  {"win_start",      win_start,      METH_VARARGS},
  {"win_complete",   win_complete,   METH_VARARGS},
  {"win_post",       win_post,       METH_VARARGS},
  {"win_wait",       win_wait,       METH_VARARGS},
  {"win_test",       win_test,       METH_VARARGS},
  {"win_lock",       win_lock,       METH_VARARGS},
  {"win_unlock",     win_unlock,     METH_VARARGS},

  /* File */
  {"file_open"            , file_open            , METH_VARARGS},
  {"file_close"           , file_close           , METH_VARARGS},
  {"file_delete"          , file_delete          , METH_VARARGS},
  {"file_set_size"        , file_set_size        , METH_VARARGS},
  {"file_preallocate"     , file_preallocate     , METH_VARARGS},
  {"file_get_size"        , file_get_size        , METH_VARARGS},
  {"file_get_group"       , file_get_group       , METH_VARARGS},
  {"file_get_amode"       , file_get_amode       , METH_VARARGS},
  {"file_set_info"        , file_set_info        , METH_VARARGS},
  {"file_get_info"        , file_get_info        , METH_VARARGS},
  {"file_set_view"        , file_set_view        , METH_VARARGS},
  {"file_get_view"        , file_get_view        , METH_VARARGS},
  {"file_read"            , file_read            , METH_VARARGS},
  {"file_write"           , file_write           , METH_VARARGS},
  {"file_iread"           , file_iread           , METH_VARARGS},
  {"file_iwrite"          , file_iwrite          , METH_VARARGS},
  {"file_seek"            , file_seek            , METH_VARARGS},
  {"file_get_position"    , file_get_position    , METH_VARARGS},
  {"file_read_split"      , file_read_split      , METH_VARARGS},
  {"file_write_split"     , file_write_split     , METH_VARARGS},
  {"file_get_byte_offset" , file_get_byte_offset , METH_VARARGS},
  {"file_get_type_extent" , file_get_type_extent , METH_VARARGS},
  {"file_set_atomicity"   , file_set_atomicity   , METH_VARARGS},
  {"file_get_atomicity"   , file_get_atomicity   , METH_VARARGS},
  {"file_sync"            , file_sync            , METH_VARARGS},

  /* Memory */
  {"get_address",   get_address,   METH_VARARGS},

  {"alloc_mem",     alloc_mem,     METH_VARARGS},
  {"free_mem",      free_mem,      METH_VARARGS},

  {"buffer_attach", buffer_attach, METH_VARARGS},
  {"buffer_detach", buffer_detach, METH_NOARGS},


  /* Naming Objects */
  {"comm_get_name", comm_get_name, METH_VARARGS},
  {"comm_set_name", comm_set_name, METH_VARARGS},
  {"type_get_name", type_get_name, METH_VARARGS},
  {"type_set_name", type_set_name, METH_VARARGS},
  {"win_get_name",  win_get_name,  METH_VARARGS},
  {"win_set_name",  win_set_name,  METH_VARARGS},
  

  /* Environmental Management */
  {"init",               env_init,              METH_NOARGS},
  {"init_thread",        env_init_thread,       METH_VARARGS},
  {"initialized",        env_initialized,       METH_NOARGS},
  {"finalize",           env_finalize,          METH_NOARGS},
  {"finalized",          env_finalized,         METH_NOARGS},
  {"query_thread",       env_query_thread,      METH_NOARGS},
  {"is_thread_main",     env_is_thread_main,    METH_NOARGS},

  {"comm_abort",         comm_abort,            METH_VARARGS},
  
  {"get_version",        env_get_version,       METH_NOARGS},
  {"get_processor_name", env_get_proc_name,     METH_NOARGS},
  
  /* Timer */
  {"wtime", env_wtime, METH_NOARGS},
  {"wtick", env_wtick, METH_NOARGS},
  
  {"comm_get_errhandler", comm_get_errhandler, METH_VARARGS},
  {"comm_set_errhandler", comm_set_errhandler, METH_VARARGS},
  {"win_get_errhandler",  win_get_errhandler,  METH_VARARGS},
  {"win_set_errhandler",  win_set_errhandler,  METH_VARARGS},
  {"file_get_errhandler", file_get_errhandler, METH_VARARGS},
  {"file_set_errhandler", file_set_errhandler, METH_VARARGS},
  {"errhandler_free",     errhandler_free,     METH_VARARGS},

  {"error_class",  env_error_class,  METH_VARARGS},
  {"error_string", env_error_string, METH_VARARGS},

  /* Helpers */
  {"_del_exception", _del_exception, METH_NOARGS},
  {"_set_exception", _set_exception, METH_VARARGS},
  {"_raise",         _raise,         METH_VARARGS},
  {"_reduce",        py2_reduce,     METH_VARARGS},

  /* Misc */
  {"_mpi_info",  get_mpi_info, METH_NOARGS},
  {"distribute", distribute,   METH_VARARGS},

  {NULL, NULL} /* Sentinel */
};

/*------------------------------------------------------------------*/

/*------------------------------------------------------------------*/

#define MPI4PY_ADD_INTVAL(m, NAME, VALUE) \
  { if (PyModule_AddIntConstant(m, NAME, (VALUE)) < 0) return; }

#define MPI4PY_ADD_PYNONE(m, NAME)                         \
  { Py_INCREF(Py_None);                                    \
    if (PyModule_AddObject(m, #NAME, Py_None) < 0) return; }

#define MPI4PY_ADD_PYTYPE(m, Type)                   \
  { Py_INCREF(PyMPI##Type##_Type);                   \
    if (PyModule_AddObject(m, #Type,                 \
       (PyObject *) PyMPI##Type##_Type) < 0) return; }

#define MPI4PY_ADD_HANDLE(m, HANDLE, Type)                 \
  { PyObject *ob = PyMPI##Type##_From##Type(MPI_##HANDLE); \
    if (ob != NULL) PyMPI##Type##_Inner(ob)->isref = 1;    \
    if (PyModule_AddObject(m, #HANDLE, ob) < 0) return; }

#define MPI4PY_ADD_MCONST(m, HANDLE, Type)                 \
  { PyObject *ob = PyMPI##Type##_From##Type(MPI_##HANDLE); \
    if (PyModule_AddObject(m, #HANDLE, ob) < 0) return; }

/*------------------------------------------------------------------*/

#define MPI4PY_ADD_ICONST(m, INTVAL)\
        MPI4PY_ADD_INTVAL(m, #INTVAL, MPI_ ## INTVAL)

#define MPI4PY_ADD_ISTATE(m, STATE)\
        MPI4PY_ADD_INTVAL(m, #STATE, MPI_ ## STATE)

#define MPI4PY_ADD_ASSERT(m, ASSERT)\
        MPI4PY_ADD_INTVAL(m, #ASSERT, MPI_ ## ASSERT)

#define MPI4PY_ADD_LOCKTP(m, LOCKTP)\
        MPI4PY_ADD_INTVAL(m, #LOCKTP, MPI_ ## LOCKTP)

#define MPI4PY_ADD_FAMODE(m, AMODE)\
        MPI4PY_ADD_INTVAL(m, #AMODE, MPI_ ## AMODE)

#define MPI4PY_ADD_FPSEEK(m,  FSEEK)\
        MPI4PY_ADD_INTVAL(m, #FSEEK, MPI_ ## FSEEK)

#define MPI4PY_ADD_ERRCLS(m, ERRCLS)\
        MPI4PY_ADD_INTVAL(m, #ERRCLS, MPI_ ## ERRCLS)

#define MPI4PY_ADD_MAXSTR(m, MAXSTR)\
        MPI4PY_ADD_INTVAL(m, #MAXSTR, MPI_ ## MAXSTR)

/*------------------------------------------------------------------*/

PyMODINIT_FUNC
init_mpi(void) 
{
  PyObject *m;

  /* Default Exception and Warning */
  /* ----------------------------- */
  PyExc_MPIError = PyExc_RuntimeError;
  PyExc_MPIWarning = PyExc_UserWarning;

  /* Module Initialization */
  /* --------------------- */
  m = Py_InitModule4("_mpi", mpi_methods, mpi_doc,
		     NULL, PYTHON_API_VERSION);
  if (m == NULL) return;

  /* MPI Initialization */
  /* ------------------ */
  if (import_libmpi() < 0) return;
#if defined(MPI4PY_THREAD_LEVEL) && HAVE_MPI_INIT_THREAD
  {
    int index = MPI4PY_THREAD_LEVEL;
    int level = MPI_THREAD_SINGLE;
    switch (index) {
    case 0:  level = MPI_THREAD_SINGLE;     break;
    case 1:  level = MPI_THREAD_FUNNELED;   break;
    case 2:  level = MPI_THREAD_SERIALIZED; break;
    case 3:  level = MPI_THREAD_MULTIPLE;   break;
    default: level = MPI_THREAD_SINGLE;     break;
    }
    if (mpi_init_thread(level, NULL) < 0) return;
  }
#else
  if (mpi_init() < 0) return;
#endif

  /* Module Communicators */
  /* -------------------- */
  {
    int ierr;
    PyMPI_COMM_SELF = PyMPI_COMM_WORLD = MPI_COMM_NULL;
    ierr = MPI_Comm_dup(MPI_COMM_SELF, &PyMPI_COMM_SELF);
    if (ierr != MPI_SUCCESS) {
      PyMPI_COMM_SELF = MPI_COMM_NULL;
      PyErr_Format(PyExc_RuntimeError,
                   "MPI_Comm_dup() failed [error code: %d]\n", ierr);
      return;
    }
    ierr = MPI_Comm_dup(MPI_COMM_WORLD, &PyMPI_COMM_WORLD);
    if (ierr != MPI_SUCCESS) {
      PyMPI_COMM_WORLD = MPI_COMM_NULL;
      PyErr_Format(PyExc_RuntimeError,
                   "MPI_Comm_dup() failed [error code: %d]\n", ierr);
      return;
    }
  }
  
  {
#if HAVE_MPI_INIT_THREAD
    MPI4PY_ADD_ICONST(m, /*MPI_*/THREAD_SINGLE     );
    MPI4PY_ADD_ICONST(m, /*MPI_*/THREAD_FUNNELED   );
    MPI4PY_ADD_ICONST(m, /*MPI_*/THREAD_SERIALIZED );
    MPI4PY_ADD_ICONST(m, /*MPI_*/THREAD_MULTIPLE   );
#else
    MPI4PY_ADD_PYNONE(m, /*MPI_*/THREAD_SINGLE     );
    MPI4PY_ADD_PYNONE(m, /*MPI_*/THREAD_FUNNELED   );
    MPI4PY_ADD_PYNONE(m, /*MPI_*/THREAD_SERIALIZED );
    MPI4PY_ADD_PYNONE(m, /*MPI_*/THREAD_MULTIPLE   );
#endif
  }


  /* Assorted constants */
  /* ----------------- */  
  {
    /* MPI-1 */
    MPI4PY_ADD_ICONST(m, /*MPI_*/UNDEFINED               );
    MPI4PY_ADD_ICONST(m, /*MPI_*/PROC_NULL               );
    MPI4PY_ADD_ICONST(m, /*MPI_*/ANY_TAG                 );
    MPI4PY_ADD_ICONST(m, /*MPI_*/ANY_SOURCE              );
    MPI4PY_ADD_ICONST(m, /*MPI_*/BSEND_OVERHEAD          );
    MPI4PY_ADD_MCONST(m, /*MPI_*/BOTTOM   , /*MPI_*/Aint );
    /* MPI-2 */
    MPI4PY_ADD_ICONST(m, /*MPI_*/ROOT                    );
    MPI4PY_ADD_MCONST(m, /*MPI_*/IN_PLACE , /*MPI_*/Aint );
  }
  

  /* Group */
  /* ----- */
  {
    /* type */
    MPI4PY_ADD_PYTYPE(m, /*MPI_*/Group );
    /* null handle */
    MPI4PY_ADD_HANDLE(m, /*MPI_*/GROUP_NULL , /*MPI_*/Group );
    /* predefined */		     	    
    MPI4PY_ADD_HANDLE(m, /*MPI_*/GROUP_EMPTY, /*MPI_*/Group );
  }
  

  /* Communicator */
  /* ------------ */
  {
    /* type */
    MPI4PY_ADD_PYTYPE(m, /*MPI_*/Comm );
    /* null handle */
    MPI4PY_ADD_HANDLE(m, /*MPI_*/COMM_NULL  , /*MPI_*/Comm );
    /* predefined */		   	 
    MPI4PY_ADD_HANDLE(m, /*MPI_*/COMM_SELF  , /*MPI_*/Comm );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/COMM_WORLD , /*MPI_*/Comm );
    /* duplicated */
    #define MPI_COMM_SELF_DUP  PyMPI_COMM_SELF
    #define MPI_COMM_WORLD_DUP PyMPI_COMM_WORLD
    MPI4PY_ADD_HANDLE(m, /*MPI_*/COMM_SELF_DUP  , /*MPI_*/Comm );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/COMM_WORLD_DUP , /*MPI_*/Comm );
    #undef MPI_COMM_SELF_DUP
    #undef MPI_COMM_WORLD_DUP
    /* comparisons */
    MPI4PY_ADD_ICONST(m, /*MPI_*/IDENT     );
    MPI4PY_ADD_ICONST(m, /*MPI_*/CONGRUENT );
    MPI4PY_ADD_ICONST(m, /*MPI_*/SIMILAR   );
    MPI4PY_ADD_ICONST(m, /*MPI_*/UNEQUAL   );
    /* topologies */
    MPI4PY_ADD_ICONST(m, /*MPI_*/CART      );
    MPI4PY_ADD_ICONST(m, /*MPI_*/GRAPH     );
  }

  /* Info */
  /* ---- */
  {
    /* type */
    MPI4PY_ADD_PYTYPE(m, /*MPI_*/Info );
    /* null handle */
    MPI4PY_ADD_HANDLE(m, /*MPI_*/INFO_NULL, /*MPI_*/Info );
  }


  /* Status */
  /* ------ */
  {
    /* type */
    MPI4PY_ADD_PYTYPE(m, /*MPI_*/Status );
  }


  /* Request */
  /* ------- */
  {
    /* type */
    MPI4PY_ADD_PYTYPE(m, /*MPI_*/Request );
    /* null handle */
    MPI4PY_ADD_HANDLE(m, /*MPI_*/REQUEST_NULL, Request );
  }


  /* Datatype */
  /* ------ */
  {
    /* type */
    MPI4PY_ADD_PYTYPE(m, /*MPI_*/Datatype );
    /* null handle */
    MPI4PY_ADD_HANDLE(m, /*MPI_*/DATATYPE_NULL      , /*MPI_*/Datatype );
    /* elementary */                              
    MPI4PY_ADD_HANDLE(m, /*MPI_*/CHAR               , /*MPI_*/Datatype );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/SHORT              , /*MPI_*/Datatype );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/INT                , /*MPI_*/Datatype );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/LONG               , /*MPI_*/Datatype );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/UNSIGNED_CHAR      , /*MPI_*/Datatype );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/UNSIGNED_SHORT     , /*MPI_*/Datatype );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/UNSIGNED           , /*MPI_*/Datatype );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/UNSIGNED_LONG      , /*MPI_*/Datatype );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/FLOAT              , /*MPI_*/Datatype );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/DOUBLE             , /*MPI_*/Datatype );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/LONG_DOUBLE        , /*MPI_*/Datatype );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/BYTE               , /*MPI_*/Datatype );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/PACKED             , /*MPI_*/Datatype );
    /* reductions */                                      
#   define MPI_TWOINT  MPI_2INT
    MPI4PY_ADD_HANDLE(m, /*MPI_*/SHORT_INT          , /*MPI_*/Datatype );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/TWOINT             , /*MPI_*/Datatype );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/LONG_INT           , /*MPI_*/Datatype );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/FLOAT_INT          , /*MPI_*/Datatype );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/DOUBLE_INT         , /*MPI_*/Datatype );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/LONG_DOUBLE_INT    , /*MPI_*/Datatype );
    /* optional */
    MPI4PY_ADD_HANDLE(m, /*MPI_*/LONG_LONG          , /*MPI_*/Datatype );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/UNSIGNED_LONG_LONG , /*MPI_*/Datatype );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/LONG_LONG_INT      , /*MPI_*/Datatype );
    /* special (deprecated in MPI-2)*/
    MPI4PY_ADD_HANDLE(m, /*MPI_*/UB                 , /*MPI_*/Datatype ); 
    MPI4PY_ADD_HANDLE(m, /*MPI_*/LB                 , /*MPI_*/Datatype );
    /* MPI-2 */
    MPI4PY_ADD_HANDLE(m, /*MPI_*/SIGNED_CHAR        , /*MPI_*/Datatype );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/WCHAR              , /*MPI_*/Datatype );
    /* constants for subarray/darray constructors */
    MPI4PY_ADD_ISTATE(m, /*MPI_*/ORDER_C              );
    MPI4PY_ADD_ISTATE(m, /*MPI_*/ORDER_FORTRAN        );
    MPI4PY_ADD_ISTATE(m, /*MPI_*/DISTRIBUTE_NONE      );
    MPI4PY_ADD_ISTATE(m, /*MPI_*/DISTRIBUTE_BLOCK     );
    MPI4PY_ADD_ISTATE(m, /*MPI_*/DISTRIBUTE_CYCLIC    );
    MPI4PY_ADD_ISTATE(m, /*MPI_*/DISTRIBUTE_DFLT_DARG );
  }


  /* Op */
  /* -- */
  {
    /* type */
    MPI4PY_ADD_PYTYPE(m, /*MPI_*/Op );
    /* null handle */
    MPI4PY_ADD_HANDLE(m, /*MPI_*/OP_NULL , /*MPI_*/Op );
    /* predefined */
    MPI4PY_ADD_HANDLE(m, /*MPI_*/MAX     , /*MPI_*/Op );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/MIN     , /*MPI_*/Op );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/SUM     , /*MPI_*/Op );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/PROD    , /*MPI_*/Op );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/LAND    , /*MPI_*/Op );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/BAND    , /*MPI_*/Op );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/LOR     , /*MPI_*/Op );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/BOR     , /*MPI_*/Op );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/LXOR    , /*MPI_*/Op );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/BXOR    , /*MPI_*/Op );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/MAXLOC  , /*MPI_*/Op );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/MINLOC  , /*MPI_*/Op );
    /* MPI-2, one-sided communication */
    MPI4PY_ADD_HANDLE(m, /*MPI_*/REPLACE , /*MPI_*/Op );
  }

  /* Win */
  /* --- */
  {
    /* type */
    MPI4PY_ADD_PYTYPE(m, /*MPI_*/Win );
    /* null handle */
    MPI4PY_ADD_HANDLE(m, /*MPI_*/WIN_NULL, /*MPI_*/Win );
    /* asserts for one-sided communication */
    MPI4PY_ADD_ASSERT(m, /*MPI_*/MODE_NOCHECK    );
    MPI4PY_ADD_ASSERT(m, /*MPI_*/MODE_NOSTORE    );
    MPI4PY_ADD_ASSERT(m, /*MPI_*/MODE_NOPUT      );
    MPI4PY_ADD_ASSERT(m, /*MPI_*/MODE_NOPRECEDE  ); 
    MPI4PY_ADD_ASSERT(m, /*MPI_*/MODE_NOSUCCEED  );
    /* lock types for one-sided communication */
    MPI4PY_ADD_LOCKTP(m, /*MPI_*/LOCK_EXCLUSIVE  ); 
    MPI4PY_ADD_LOCKTP(m, /*MPI_*/LOCK_SHARED     ); 
  }

  /* File */
  /* ---- */
  {
    /* type */
    MPI4PY_ADD_PYTYPE(m, /*MPI_*/File );
    /* null handle */
    MPI4PY_ADD_HANDLE(m, /*MPI_*/FILE_NULL, /*MPI_*/File );
    /* access mode */
    MPI4PY_ADD_FAMODE(m, /*MPI_*/MODE_RDONLY          );
    MPI4PY_ADD_FAMODE(m, /*MPI_*/MODE_RDWR            );
    MPI4PY_ADD_FAMODE(m, /*MPI_*/MODE_WRONLY          );
    MPI4PY_ADD_FAMODE(m, /*MPI_*/MODE_CREATE          );
    MPI4PY_ADD_FAMODE(m, /*MPI_*/MODE_EXCL            );
    MPI4PY_ADD_FAMODE(m, /*MPI_*/MODE_DELETE_ON_CLOSE );
    MPI4PY_ADD_FAMODE(m, /*MPI_*/MODE_UNIQUE_OPEN     );
    MPI4PY_ADD_FAMODE(m, /*MPI_*/MODE_SEQUENTIAL      );
    MPI4PY_ADD_FAMODE(m, /*MPI_*/MODE_APPEND          );
    /* seek flag */
    MPI4PY_ADD_FPSEEK(m, /*MPI_*/SEEK_SET );
    MPI4PY_ADD_FPSEEK(m, /*MPI_*/SEEK_CUR );
    MPI4PY_ADD_FPSEEK(m, /*MPI_*/SEEK_END );
    /* special displacement argument */
    MPI4PY_ADD_MCONST(m, /*MPI_*/DISPLACEMENT_CURRENT, /*MPI_*/Offset );
  }
  
  /* Errhandler */
  /* ---------- */
  {
    /* type */
    MPI4PY_ADD_PYTYPE(m, /*MPI_*/Errhandler );
    /* null handle */
    MPI4PY_ADD_HANDLE(m, /*MPI_*/ERRHANDLER_NULL , /*MPI_*/Errhandler );
    /* predefined */
    MPI4PY_ADD_HANDLE(m, /*MPI_*/ERRORS_ARE_FATAL, /*MPI_*/Errhandler );
    MPI4PY_ADD_HANDLE(m, /*MPI_*/ERRORS_RETURN   , /*MPI_*/Errhandler );
  }
  
  /* Error Codes */
  /* ----------- */
  {
    /* MPI-1 */
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/SUCCESS                   );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_ARG                   );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_BUFFER                );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_COMM                  );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_COUNT                 );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_DIMS                  );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_GROUP                 );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_IN_STATUS             );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_INTERN                );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_LASTCODE              );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_OP                    );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_OTHER                 );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_PENDING               );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_RANK                  );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_REQUEST               );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_ROOT                  );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_TAG                   );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_TOPOLOGY              );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_TRUNCATE              );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_TYPE                  );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_UNKNOWN               );
    /* MPI-2 */
    /* invalid key value */
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_KEYVAL                );
    /* memory allocation error*/
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_NO_MEM                );
    /* port/service and spawn errors */
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_NAME                  );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_PORT                  );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_SERVICE               );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_SPAWN                 );
    /* info errors */
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_INFO                  );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_INFO_KEY              );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_INFO_VALUE            );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_INFO_NOKEY            );
    /* window errors */
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_WIN                   );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_BASE                  );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_SIZE                  );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_DISP                  );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_LOCKTYPE              );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_ASSERT                );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_RMA_CONFLICT          );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_RMA_SYNC              );
    /* file errors */
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_FILE                  );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_NOT_SAME              );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_AMODE                 );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_UNSUPPORTED_DATAREP   );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_UNSUPPORTED_OPERATION );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_NO_SUCH_FILE          );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_FILE_EXISTS           );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_BAD_FILE              );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_ACCESS                );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_NO_SPACE              );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_QUOTA                 );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_READ_ONLY             );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_FILE_IN_USE           );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_DUP_DATAREP           );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_CONVERSION            );
    MPI4PY_ADD_ERRCLS(m, /*MPI_*/ERR_IO                    );
  }

  /* More Constants */ /* (just in case someone need them) */
  /* -------------- */
  {
    /* MPI-1 */
    MPI4PY_ADD_MAXSTR(m, /*MPI_*/MAX_PROCESSOR_NAME );
    MPI4PY_ADD_MAXSTR(m, /*MPI_*/MAX_ERROR_STRING   );
    /* MPI-2 */
    MPI4PY_ADD_MAXSTR(m, /*MPI_*/MAX_PORT_NAME      );
    MPI4PY_ADD_MAXSTR(m, /*MPI_*/MAX_INFO_KEY       );
    MPI4PY_ADD_MAXSTR(m, /*MPI_*/MAX_INFO_VAL       );
    MPI4PY_ADD_MAXSTR(m, /*MPI_*/MAX_OBJECT_NAME    );
    MPI4PY_ADD_MAXSTR(m, /*MPI_*/MAX_DATAREP_STRING );
  }
  

  /* Environmental inquires */
  /* ---------------------- */
  {
    int tagub  = 32767;
    int host   = MPI_PROC_NULL;
    int io     = MPI_PROC_NULL;
    int wtglb  = 0;
    int appnum = MPI_UNDEFINED;
    int univsz = MPI_UNDEFINED;
    int flag, *attr;
    
    flag = 0; attr = NULL;
    MPI_Attr_get(MPI_COMM_WORLD, MPI_TAG_UB, &attr, &flag);
    if (flag && attr) tagub = *attr;

    flag = 0; attr = NULL;
    MPI_Attr_get(MPI_COMM_WORLD, MPI_HOST, &attr, &flag);
    if (flag && attr) host = *attr;

    flag = 0; attr = NULL;
    MPI_Attr_get(MPI_COMM_WORLD, MPI_IO, &attr, &flag);
    if (flag && attr) io = *attr;

    flag = 0; attr = NULL;
    MPI_Attr_get(MPI_COMM_WORLD, MPI_WTIME_IS_GLOBAL, &attr, &flag);
    if (flag && attr) wtglb = *attr;

#if !MPI4PY_SKIP_GET_MPI2_ATTRS

#   if HAVE_MPI_APPNUM
    flag = 0; attr = NULL;
    MPI_Attr_get(MPI_COMM_WORLD, MPI_APPNUM, &attr, &flag);
    if (flag && attr) appnum = *attr;
#   if defined(OPEN_MPI) /* could return -1 */
    if (appnum == -1) appnum = MPI_UNDEFINED;
#   endif
#   endif

#   if HAVE_MPI_UNIVERSE_SIZE
    flag = 0; attr = NULL;
#   if defined(MPICH2)  /* could hang at import time */
    if (appnum != MPI_UNDEFINED)
#   endif
    MPI_Attr_get(MPI_COMM_WORLD, MPI_UNIVERSE_SIZE, &attr, &flag);
    if (flag && attr) univsz = *attr;
#   endif

#endif
    
    MPI4PY_ADD_INTVAL(m, "TAG_UB"          , tagub  );
    MPI4PY_ADD_INTVAL(m, "HOST"            , host   );
    MPI4PY_ADD_INTVAL(m, "IO"              , io     );
    MPI4PY_ADD_INTVAL(m, "WTIME_IS_GLOBAL" , wtglb  );
    MPI4PY_ADD_INTVAL(m, "UNIVERSE_SIZE"   , univsz );
    MPI4PY_ADD_INTVAL(m, "APPNUM"          , appnum );
  }

  /* Convenience Constants */
  /* --------------------- */
  {
    int size, rank;
    MPI_Comm_size(MPI_COMM_WORLD, &size);
    MPI_Comm_rank(MPI_COMM_WORLD, &rank);

    MPI4PY_ADD_INTVAL(m, "WORLD_SIZE" , size );
    MPI4PY_ADD_INTVAL(m, "WORLD_RANK" , rank );
  }
  
}

/*------------------------------------------------------------------*/


/* 
   Local variables:
   c-basic-offset: 2
   indent-tabs-mode: nil
   End:
*/
