"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
// tslint:disable-next-line:no-var-requires
const crypto = require("crypto");
const encoding_1 = require("./encoding");
/**
 * Resources with this ID are hidden from humans
 *
 * They do not appear in the human-readable part of the logical ID,
 * but they are included in the hash calculation.
 */
const HIDDEN_FROM_HUMAN_ID = 'Resource';
/**
 * Resources with this ID are complete hidden from the logical ID calculation.
 */
const HIDDEN_ID = 'Default';
const PATH_SEP = '/';
const HASH_LEN = 8;
const MAX_HUMAN_LEN = 240; // max ID len is 255
/**
 * Calculates a unique ID for a set of textual components.
 *
 * This is done by calculating a hash on the full path and using it as a suffix
 * of a length-limited "human" rendition of the path components.
 *
 * @param components The path components
 * @returns a unique alpha-numeric identifier with a maximum length of 255
 */
function makeUniqueId(components) {
    components = components.filter(x => x !== HIDDEN_ID);
    if (components.length === 0) {
        throw new Error('Unable to calculate a unique id for an empty set of components');
    }
    // Lazy require in order to break a module dependency cycle
    const unresolvedTokens = components.filter(c => encoding_1.unresolved(c));
    if (unresolvedTokens.length > 0) {
        throw new Error(`ID components may not include unresolved tokens: ${unresolvedTokens.join(',')}`);
    }
    // top-level resources will simply use the `name` as-is in order to support
    // transparent migration of cloudformation templates to the CDK without the
    // need to rename all resources.
    if (components.length === 1) {
        return removeNonAlphanumeric(components[0]);
    }
    const hash = pathHash(components);
    const human = removeDupes(components)
        .filter(x => x !== HIDDEN_FROM_HUMAN_ID)
        .map(removeNonAlphanumeric)
        .join('')
        .slice(0, MAX_HUMAN_LEN);
    return human + hash;
}
exports.makeUniqueId = makeUniqueId;
/**
 * Take a hash of the given path.
 *
 * The hash is limited in size.
 */
function pathHash(path) {
    const md5 = crypto.createHash('md5').update(path.join(PATH_SEP)).digest("hex");
    return md5.slice(0, HASH_LEN).toUpperCase();
}
/**
 * Removes all non-alphanumeric characters in a string.
 */
function removeNonAlphanumeric(s) {
    return s.replace(/[^A-Za-z0-9]/g, '');
}
/**
 * Remove duplicate "terms" from the path list
 *
 * If the previous path component name ends with this component name, skip the
 * current component.
 */
function removeDupes(path) {
    const ret = new Array();
    for (const component of path) {
        if (ret.length === 0 || !ret[ret.length - 1].endsWith(component)) {
            ret.push(component);
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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