"use strict";
const nodeunit = require("nodeunit");
const lib_1 = require("../lib");
function floatEqual(test, actual, expected) {
    test.ok(
    // Floats are subject to rounding errors up to Number.ESPILON
    actual >= expected - Number.EPSILON && actual <= expected + Number.EPSILON, `${actual} == ${expected}`);
}
module.exports = nodeunit.testCase({
    'negative amount'(test) {
        test.throws(() => lib_1.Duration.seconds(-1), /negative/);
        test.done();
    },
    'unresolved amount'(test) {
        const stack = new lib_1.Stack();
        const lazyDuration = lib_1.Duration.seconds(lib_1.Token.asNumber({ resolve: () => 1337 }));
        test.equals(stack.resolve(lazyDuration.toSeconds()), 1337);
        test.throws(() => stack.resolve(lazyDuration.toMinutes()), /Unable to perform time unit conversion on un-resolved token/);
        test.done();
    },
    'Duration in seconds'(test) {
        const duration = lib_1.Duration.seconds(300);
        test.equal(duration.toSeconds(), 300);
        test.equal(duration.toMinutes(), 5);
        test.throws(() => duration.toDays(), /'300 seconds' cannot be converted into a whole number of days/);
        floatEqual(test, duration.toDays({ integral: false }), 300 / 86400);
        test.equal(lib_1.Duration.seconds(60 * 60 * 24).toDays(), 1);
        test.done();
    },
    'Duration in minutes'(test) {
        const duration = lib_1.Duration.minutes(5);
        test.equal(duration.toSeconds(), 300);
        test.equal(duration.toMinutes(), 5);
        test.throws(() => duration.toDays(), /'5 minutes' cannot be converted into a whole number of days/);
        floatEqual(test, duration.toDays({ integral: false }), 300 / 86400);
        test.equal(lib_1.Duration.minutes(60 * 24).toDays(), 1);
        test.done();
    },
    'Duration in hours'(test) {
        const duration = lib_1.Duration.hours(5);
        test.equal(duration.toSeconds(), 18000);
        test.equal(duration.toMinutes(), 300);
        test.throws(() => duration.toDays(), /'5 hours' cannot be converted into a whole number of days/);
        floatEqual(test, duration.toDays({ integral: false }), 5 / 24);
        test.equal(lib_1.Duration.hours(24).toDays(), 1);
        test.done();
    },
    'seconds to milliseconds'(test) {
        const duration = lib_1.Duration.seconds(5);
        test.equal(duration.toMilliseconds(), 5000);
        test.done();
    },
    'Duration in days'(test) {
        const duration = lib_1.Duration.days(1);
        test.equal(duration.toSeconds(), 86400);
        test.equal(duration.toMinutes(), 1440);
        test.equal(duration.toDays(), 1);
        test.done();
    },
    'toISOString'(test) {
        test.equal(lib_1.Duration.seconds(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.minutes(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.hours(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.days(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.seconds(5).toISOString(), 'PT5S');
        test.equal(lib_1.Duration.minutes(5).toISOString(), 'PT5M');
        test.equal(lib_1.Duration.hours(5).toISOString(), 'PT5H');
        test.equal(lib_1.Duration.days(5).toISOString(), 'PT5D');
        test.equal(lib_1.Duration.seconds(1 + 60 * (1 + 60 * (1 + 24))).toISOString(), 'PT1D1H1M1S');
        test.done();
    },
    'parse'(test) {
        test.equal(lib_1.Duration.parse('PT0S').toSeconds(), 0);
        test.equal(lib_1.Duration.parse('PT0M').toSeconds(), 0);
        test.equal(lib_1.Duration.parse('PT0H').toSeconds(), 0);
        test.equal(lib_1.Duration.parse('PT0D').toSeconds(), 0);
        test.equal(lib_1.Duration.parse('PT5S').toSeconds(), 5);
        test.equal(lib_1.Duration.parse('PT5M').toSeconds(), 300);
        test.equal(lib_1.Duration.parse('PT5H').toSeconds(), 18000);
        test.equal(lib_1.Duration.parse('PT5D').toSeconds(), 432000);
        test.equal(lib_1.Duration.parse('PT1D1H1M1S').toSeconds(), 1 + 60 * (1 + 60 * (1 + 24)));
        test.done();
    }
});
//# sourceMappingURL=data:application/json;base64,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